unit SDLUtils;
{******************************************************************************}
{                                                                              }
{       Borland Delphi SDL - Simple DirectMedia Layer                          }
{                SDL Utility functions                                         }
{                                                                              }
{                                                                              }
{ The initial developer of this Pascal code was :                              }
{ Tom Jones <tigertomjones@gmx.de>                                             }
{                                                                              }
{ Portions created by Tom Jones are                                            }
{ Copyright (C) 2000 - 2001 Tom Jones.                                         }
{                                                                              }
{                                                                              }
{ Contributor(s)                                                               }
{ --------------                                                               }
{ Dominique Louis <Dominique@SavageSoftware.com.au>                            }
{ Rbert Kisnmeth <mikrobi@freemail.hu>                                       }
{                                                                              }
{ Obtained through:                                                            }
{ Joint Endeavour of Delphi Innovators ( Project JEDI )                        }
{                                                                              }
{ You may retrieve the latest version of this file at the Project              }
{ JEDI home page, located at http://delphi-jedi.org                            }
{                                                                              }
{ The contents of this file are used with permission, subject to               }
{ the Mozilla Public License Version 1.1 (the "License"); you may              }
{ not use this file except in compliance with the License. You may             }
{ obtain a copy of the License at                                              }
{ http://www.mozilla.org/MPL/MPL-1.1.html                                      }
{                                                                              }
{ Software distributed under the License is distributed on an                  }
{ "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or               }
{ implied. See the License for the specific language governing                 }
{ rights and limitations under the License.                                    }
{                                                                              }
{ Description                                                                  }
{ -----------                                                                  }
{   Helper functions...                                                        }
{                                                                              }
{                                                                              }
{ Requires                                                                     }
{ --------                                                                     }
{   SDL.dll on Windows platforms                                               }
{   libSDL-1.1.so.0 on Linux platform                                          }
{                                                                              }
{ Programming Notes                                                            }
{ -----------------                                                            }
{                                                                              }
{                                                                              }
{                                                                              }
{                                                                              }
{ Revision History                                                             }
{ ----------------                                                             }
{               2000 - TJ : Initial creation                                   }
{   July   13   2001 - DL : Added PutPixel and GetPixel routines.              }
{   Sept   14   2001 - RK : Added flipping routines.                           }
{   Sept   19   2001 - RK : Added PutPixel & line drawing & blitting with ADD  }
{                           effect. Fixed a bug in SDL_PutPixel & SDL_GetPixel }
{                           Added PSDLRect()                                   }
{   Sept   22   2001 - DL : Removed need for Windows.pas by defining types here}
{                           Also removed by poor attempt or a dialog box       }
{   Sept   25   2001 - RK : Added PixelTest, NewPutPixel, SubPixel, SubLine,   }
{                           SubSurface, MonoSurface & TexturedSurface          }
{   Sept   26   2001 - DL : Made change so that it refers to native Pascal     }
{                           types rather that Windows types. This makes it more}
{                           portable to Linix.                                 }
{   Sept   27   2001 - RK : SDLUtils now can be compiled with FreePascal       }
{******************************************************************************}

interface

uses
{$IFDEF LINUX}
  Types,
  Xlib,
{$ENDIF}
  SysUtils,
  SDL;

// Types required so we don't need to use Windows.pas
type
  PInteger = ^Integer;
  PByte = ^Byte;
  PWord = ^Word;
  PLongWord = ^Longword;

  // General arrays
  PByteArray = ^TByteArray;
  TByteArray = array[0..32767] of byte;

  PWordArray = ^TWordArray;
  TWordArray = array[0..16383] of Word;

  PPoint = ^TPoint;
  TPoint = record
    x: Longint;
    y: Longint;
  end;

  PRect = ^TRect;
  TRect = record
    case Integer of
      0: (Left, Top, Right, Bottom: Integer);
      1: (TopLeft, BottomRight: TPoint);
  end;

  // Pixel procedures
function PixelTest(SrcSurface1: PSDL_Surface; SrcRect1: PSDL_Rect; SrcSurface2:
  PSDL_Surface; SrcRect2: PSDL_Rect; Left1, Top1, Left2, Top2: integer): Boolean;

function SDL_GetPixel(surface: PSDL_Surface; x: integer; y: integer): Uint32;

procedure SDL_PutPixel(surface_: PSDL_Surface; x: integer; y: integer; pixel:
  Uint32);

procedure SDL_NewPutPixel(Surface: PSDL_Surface; x: integer; y: integer; Color:
  cardinal);

procedure SDL_AddPixel(Surface: PSDL_Surface; x: integer; y: integer; Color:
  cardinal);

procedure SDL_SubPixel(Surface: PSDL_Surface; x: integer; y: integer; Color:
  cardinal);

// Line procedures
procedure SDL_DrawLine(Surface: PSDL_Surface; x1, y1, x2, y2: integer; Color:
  cardinal);

procedure SDL_AddLine(Surface: PSDL_Surface; x1, y1, x2, y2: integer; Color:
  cardinal);

procedure SDL_SubLine(Surface: PSDL_Surface; x1, y1, x2, y2: integer; Color:
  cardinal);

// Surface procedures
procedure SDL_AddSurface(SrcSurface: PSDL_Surface; SrcRect: PSDL_Rect;
  DestSurface: PSDL_Surface; DestRect: PSDL_Rect);

procedure SDL_SubSurface(SrcSurface: PSDL_Surface; SrcRect: PSDL_Rect;
  DestSurface: PSDL_Surface; DestRect: PSDL_Rect);

procedure SDL_MonoSurface(SrcSurface: PSDL_Surface; SrcRect: PSDL_Rect;
  DestSurface: PSDL_Surface; DestRect: PSDL_Rect; Color: cardinal);

procedure SDL_TexturedSurface(SrcSurface: PSDL_Surface; SrcRect: PSDL_Rect;
  DestSurface: PSDL_Surface; DestRect: PSDL_Rect; Texture: PSDL_Surface;
  TextureRect: PSDL_Rect);

// Flip procedures
procedure SDL_FlipRectH(Surface: PSDL_Surface; Rect: PSDL_Rect);

procedure SDL_FlipRectV(Surface: PSDL_Surface; Rect: PSDL_Rect);

function PSDLRect(aLeft, aTop, aWidth, aHeight: integer): PSDL_Rect;

function SDLRect(aLeft, aTop, aWidth, aHeight: integer): TSDL_Rect; overload;

function SDLRect(aRect: TRect): TSDL_Rect; overload;

function SDL_ScaleSurfaceRect(Surface: PSDL_Surface; SrcX1, SrcY1, SrcW, SrcH,
  Width, Height: integer): PSDL_Surface;

procedure SDL_ScrollX(Surface: PSDL_Surface; DifX: integer);

implementation

function PixelTest(SrcSurface1: PSDL_Surface; SrcRect1: PSDL_Rect; SrcSurface2:
  PSDL_Surface; SrcRect2: PSDL_Rect; Left1, Top1, Left2, Top2: integer): boolean;
var
  Src_Rect1, Src_Rect2: TSDL_Rect;
  right1, bottom1: integer;
  right2, bottom2: integer;
  Scan1Start, Scan2Start, ScanWidth, ScanHeight: cardinal;
  Mod1, Mod2: cardinal;
  Addr1, Addr2: cardinal;
  BPP: cardinal;
  Pitch1, Pitch2: cardinal;
  TransparentColor1, TransparentColor2: cardinal;
  tx, ty: cardinal;
  StartTick: cardinal;
  Color1, Color2: cardinal;
begin
  Result := false;
  if SrcRect1 = nil then
  begin
    with Src_Rect1 do
    begin
      x := 0;
      y := 0;
      w := SrcSurface1.w;
      h := SrcSurface1.h;
    end;
  end
  else
    Src_Rect1 := SrcRect1^;
  if SrcRect2 = nil then
  begin
    with Src_Rect2 do
    begin
      x := 0;
      y := 0;
      w := SrcSurface2.w;
      h := SrcSurface2.h;
    end;
  end
  else
    Src_Rect2 := SrcRect2^;
  with Src_Rect1 do
  begin
    Right1 := Left1 + w;
    Bottom1 := Top1 + h;
  end;
  with Src_Rect2 do
  begin
    Right2 := Left2 + w;
    Bottom2 := Top2 + h;
  end;
  if (Left1 >= Right2) or (Right1 <= Left2) or (Top1 >= Bottom2) or (Bottom1 <=
    Top2) then
    exit;
  if Left1 <= Left2 then
  begin
    // 1. left, 2. right
    Scan1Start := Src_Rect1.x + Left2 - Left1;
    Scan2Start := Src_Rect2.x;
    ScanWidth := Right1 - Left2;
    with Src_Rect2 do
      if ScanWidth > w then
        ScanWidth := w;
  end
  else
  begin
    // 1. right, 2. left
    Scan1Start := Src_Rect1.x;
    Scan2Start := Src_Rect2.x + Left1 - Left2;
    ScanWidth := Right2 - Left1;
    with Src_Rect1 do
      if ScanWidth > w then
        ScanWidth := w;
  end;
  with SrcSurface1^ do
  begin
    Pitch1 := Pitch;
    Addr1 := cardinal(Pixels);
    inc(Addr1, Pitch1 * UInt32(Src_Rect1.y));
    with format^ do
    begin
      BPP := BytesPerPixel;
      TransparentColor1 := colorkey;
    end;
  end;
  with SrcSurface2^ do
  begin
    TransparentColor2 := format.colorkey;
    Pitch2 := Pitch;
    Addr2 := cardinal(Pixels);
    inc(Addr2, Pitch2 * UInt32(Src_Rect2.y));
  end;
  Mod1 := Pitch1 - (ScanWidth * BPP);
  Mod2 := Pitch2 - (ScanWidth * BPP);
  inc(Addr1, BPP * Scan1Start);
  inc(Addr2, BPP * Scan2Start);
  if Top1 <= Top2 then
  begin
    // 1. up, 2. down
    ScanHeight := Bottom1 - Top2;
    if ScanHeight > Src_Rect2.h then
      ScanHeight := Src_Rect2.h;
    inc(Addr1, Pitch1 * UInt32(Top2 - Top1));
  end
  else
  begin
    // 1. down, 2. up
    ScanHeight := Bottom2 - Top1;
    if ScanHeight > Src_Rect1.h then
      ScanHeight := Src_Rect1.h;
    inc(Addr2, Pitch2 * UInt32(Top1 - Top2));
  end;
  case BPP of
    1: for ty := 1 to ScanHeight do
      begin
        for tx := 1 to ScanWidth do
        begin
          if (PByte(Addr1)^ <> TransparentColor1) and (PByte(Addr2)^ <>
            TransparentColor2) then
          begin
            Result := true;
            exit;
          end;
          inc(Addr1);
          inc(Addr2);
        end;
        inc(Addr1, Mod1);
        inc(Addr2, Mod2);
      end;
    2: for ty := 1 to ScanHeight do
      begin
        for tx := 1 to ScanWidth do
        begin
          if (PWord(Addr1)^ <> TransparentColor1) and (PWord(Addr2)^ <>
            TransparentColor2) then
          begin
            Result := true;
            exit;
          end;
          inc(Addr1, 2);
          inc(Addr2, 2);
        end;
        inc(Addr1, Mod1);
        inc(Addr2, Mod2);
      end;
    3: for ty := 1 to ScanHeight do
      begin
        for tx := 1 to ScanWidth do
        begin
          Color1 := PLongWord(Addr1)^ and $00FFFFFF;
          Color2 := PLongWord(Addr2)^ and $00FFFFFF;
          if (Color1 <> TransparentColor1) and (Color2 <> TransparentColor2)
            then
          begin
            Result := true;
            exit;
          end;
          inc(Addr1, 3);
          inc(Addr2, 3);
        end;
        inc(Addr1, Mod1);
        inc(Addr2, Mod2);
      end;
    4: for ty := 1 to ScanHeight do
      begin
        for tx := 1 to ScanWidth do
        begin
          if (PLongWord(Addr1)^ <> TransparentColor1) and (PLongWord(Addr2)^ <>
            TransparentColor2) then
          begin
            Result := true;
            exit;
          end;
          inc(Addr1, 4);
          inc(Addr2, 4);
        end;
        inc(Addr1, Mod1);
        inc(Addr2, Mod2);
      end;
  end;
end;

procedure SDL_NewPutPixel(Surface: PSDL_Surface; x: integer; y: integer; Color:
  cardinal);
var
  Addr, Pitch, BPP: cardinal;
begin
  Addr := cardinal(Surface.Pixels);
  Pitch := Surface.Pitch;
  BPP := Surface.format.BytesPerPixel;
  asm
  mov eax, y
  mul Pitch      // EAX := y * Pitch
  add Addr, eax  // Addr:= Addr + (y * Pitch)
  mov eax, x
  mov ecx, Color
  cmp BPP, 1
  jne @Not1BPP
  add eax, Addr  // Now:   EAX:= Addr + (y * Pitch) + x
  mov [eax], cl
  jmp @Quit
 @Not1BPP:
  cmp BPP, 2
  jne @Not2BPP
  mul BPP   // EAX := x * BPP
  add eax, Addr  // Now:   EAX:= Addr + (y * Pitch) + x * BPP
  mov [eax], cx
  jmp @Quit
 @Not2BPP:
  cmp BPP, 3
  jne @Not3BPP
  mul BPP   // EAX := x * BPP
  add eax, Addr  // Now:   EAX:= Addr + (y * Pitch) + x * BPP
  mov edx, [eax]
  and edx, $ff000000
  or edx, ecx
  mov [eax], edx
  jmp @Quit
 @Not3BPP:
  mul BPP   // EAX := x * BPP
  add eax, Addr  // Now:   EAX:= Addr + (y * Pitch) + x * BPP
  mov [eax], ecx
 @Quit:
  end;
end;

procedure SDL_AddPixel(Surface: PSDL_Surface; x: integer; y: integer; Color:
  cardinal);
var
  SrcColor, FinalColor: cardinal;
  Addr, Pitch, Bits: cardinal;
begin
  if Color = 0 then
    exit;
  Addr := cardinal(Surface.Pixels);
  Pitch := Surface.Pitch;
  Bits := Surface.format.BitsPerPixel;
  asm
    mov eax, y
    mul Pitch      // EAX := y * Pitch
    add Addr, eax  // Addr:= Addr + (y * Pitch)
    mov eax, x
    cmp Bits, 8
    jne @Not8bit
    add eax, Addr  // Now:   EAX:= Addr + (y * Pitch) + x
    mov cl, [eax]
    movzx ecx, cl
    mov SrcColor, ecx
    mov edx, Color
    and ecx, 3
    and edx, 3
    add ecx, edx
    cmp ecx, 3
    jbe @Skip1_8bit
    mov ecx, 3
   @Skip1_8bit:
    mov FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $1c
    and edx, $1c
    add ecx, edx
    cmp ecx, $1c
    jbe @Skip2_8bit
    mov ecx, $1c
   @Skip2_8bit:
    or FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $e0
    and edx, $e0
    add ecx, edx
    cmp ecx, $e0
    jbe @Skip3_8bit
    mov ecx, $e0
   @Skip3_8bit:
    or ecx, FinalColor
    mov [eax], cl
    jmp @Quit
   @Not8bit:
    cmp Bits, 15
    jne @Not15bit
    shl eax, 1
    add eax, Addr  // Now:   EAX:= Addr + (y * Pitch) + x * 2
    mov ecx, [eax]
    and ecx, $00007fff
    mov SrcColor, ecx
    mov edx, Color
    and ecx, $1f
    and edx, $1f
    add ecx, edx
    cmp ecx, $1f
    jbe @Skip1_15bit
    mov ecx, $1f
   @Skip1_15bit:
    mov FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $03e0
    and edx, $03e0
    add ecx, edx
    cmp ecx, $03e0
    jbe @Skip2_15bit
    mov ecx, $03e0
   @Skip2_15bit:
    or FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $7c00
    and edx, $7c00
    add ecx, edx
    cmp ecx, $7c00
    jbe @Skip3_15bit
    mov ecx, $7c00
   @Skip3_15bit:
    or ecx, FinalColor
    mov [eax], cx
    jmp @Quit
   @Not15Bit:
    cmp Bits, 16
    jne @Not16bit
    shl eax, 1
    add eax, Addr  // Now:   EAX:= Addr + (y * Pitch) + x * 2
    mov ecx, [eax]
    and ecx, $0000ffff
    mov SrcColor, ecx
    mov edx, Color
    and ecx, $1f
    and edx, $1f
    add ecx, edx
    cmp ecx, $1f
    jbe @Skip1_16bit
    mov ecx, $1f
   @Skip1_16bit:
    mov FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $07e0
    and edx, $07e0
    add ecx, edx
    cmp ecx, $07e0
    jbe @Skip2_16bit
    mov ecx, $07e0
   @Skip2_16bit:
    or FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $f800
    and edx, $f800
    add ecx, edx
    cmp ecx, $f800
    jbe @Skip3_16bit
    mov ecx, $f800
   @Skip3_16bit:
    or ecx, FinalColor
    mov [eax], cx
    jmp @Quit
   @Not16Bit:
    cmp Bits, 24
    jne @Not24bit
    mov ecx, 0
    add ecx, eax
    shl ecx, 1
    add ecx, eax
    mov eax, ecx
    jmp @32bit
   @Not24bit:
    shl eax, 2
   @32bit:
    add eax, Addr  // Now:   EAX:= Addr + (y * Pitch) + x * 2
    mov ecx, [eax]
    mov FinalColor, ecx
    and FinalColor, $ff000000
    and ecx, $00ffffff
    mov SrcColor, ecx
    mov edx, Color
    and ecx, $000000ff
    and edx, $000000ff
    add ecx, edx
    cmp ecx, $000000ff
    jbe @Skip1_32bit
    mov ecx, $000000ff
   @Skip1_32bit:
    or FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $0000ff00
    and edx, $0000ff00
    add ecx, edx
    cmp ecx, $0000ff00
    jbe @Skip2_32bit
    mov ecx, $0000ff00
   @Skip2_32bit:
    or FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $00ff0000
    and edx, $00ff0000
    add ecx, edx
    cmp ecx, $00ff0000
    jbe @Skip3_32bit
    mov ecx, $00ff0000
   @Skip3_32bit:
    or ecx, FinalColor
    mov [eax], ecx
   @Quit:
  end;
end;

procedure SDL_SubPixel(Surface: PSDL_Surface; x: integer; y: integer; Color:
  cardinal);
var
  SrcColor, FinalColor: cardinal;
  Addr, Pitch, Bits: cardinal;
begin
  if Color = 0 then
    exit;
  Addr := cardinal(Surface.Pixels);
  Pitch := Surface.Pitch;
  Bits := Surface.format.BitsPerPixel;
  asm
    mov eax, y
    mul Pitch      // EAX := y * Pitch
    add Addr, eax  // Addr:= Addr + (y * Pitch)
    mov eax, x
    cmp Bits, 8
    jne @Not8bit
    add eax, Addr  // Now:   EAX:= Addr + (y * Pitch) + x
    mov cl, [eax]
    movzx ecx, cl
    mov SrcColor, ecx
    mov edx, Color
    and ecx, 3
    and edx, 3
    sub ecx, edx
    jns @Skip1_8bit
    mov ecx, 0
   @Skip1_8bit:
    mov FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $1c
    and edx, $1c
    sub ecx, edx
    jns @Skip2_8bit
    mov ecx, 0
   @Skip2_8bit:
    or FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $e0
    and edx, $e0
    sub ecx, edx
    jns @Skip3_8bit
    mov ecx, 0
   @Skip3_8bit:
    or ecx, FinalColor
    mov [eax], cl
    jmp @Quit
   @Not8bit:
    cmp Bits, 15
    jne @Not15bit
    shl eax, 1
    add eax, Addr
    mov ecx, [eax]
    and ecx, $00007fff
    mov SrcColor, ecx
    mov edx, Color
    and ecx, $1f
    and edx, $1f
    sub ecx, edx
    jns @Skip1_15bit
    mov ecx, 0
   @Skip1_15bit:
    mov FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $03e0
    and edx, $03e0
    sub ecx, edx
    jns @Skip2_15bit
    mov ecx, 0
   @Skip2_15bit:
    or FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $7c00
    and edx, $7c00
    sub ecx, edx
    jns @Skip3_15bit
    mov ecx, 0
   @Skip3_15bit:
    or ecx, FinalColor
    mov [eax], cx
    jmp @Quit
   @Not15Bit:
    cmp Bits, 16
    jne @Not16bit
    shl eax, 1
    add eax, Addr
    mov ecx, [eax]
    and ecx, $0000ffff
    mov SrcColor, ecx
    mov edx, Color
    and ecx, $1f
    and edx, $1f
    sub ecx, edx
    jns @Skip1_16bit
    mov ecx, 0
   @Skip1_16bit:
    mov FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $07e0
    and edx, $07e0
    sub ecx, edx
    jns @Skip2_16bit
    mov ecx, 0
   @Skip2_16bit:
    or FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $f800
    and edx, $f800
    sub ecx, edx
    jns @Skip3_16bit
    mov ecx, 0
   @Skip3_16bit:
    or ecx, FinalColor
    mov [eax], cx
    jmp @Quit
   @Not16Bit:
    cmp Bits, 24
    jne @Not24bit
    mov ecx, 0
    add ecx, eax
    shl ecx, 1
    add ecx, eax
    mov eax, ecx
    jmp @32bit
   @Not24bit:
    shl eax, 2
   @32bit:
    add eax, Addr
    mov ecx, [eax]
    mov FinalColor, ecx
    and FinalColor, $ff000000
    and ecx, $00ffffff
    mov SrcColor, ecx
    mov edx, Color
    and ecx, $000000ff
    and edx, $000000ff
    sub ecx, edx
    jns @Skip1_32bit
    mov ecx, 0
   @Skip1_32bit:
    or FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $0000ff00
    and edx, $0000ff00
    sub ecx, edx
    jns @Skip2_32bit
    mov ecx, 0
   @Skip2_32bit:
    or FinalColor, ecx
    mov ecx, SrcColor
    mov edx, Color
    and ecx, $00ff0000
    and edx, $00ff0000
    sub ecx, edx
    jns @Skip3_32bit
    mov ecx, 0
   @Skip3_32bit:
    or ecx, FinalColor
    mov [eax], ecx
   @Quit:
  end;
end;

// This procedure works on 8, 15, 16, 24 and 32 bits color depth surfaces.
// In 8 bit color depth mode the procedure works with the default packed
//  palette (RRRGGGBB). It handles all clipping.

procedure SDL_AddSurface(SrcSurface: PSDL_Surface; SrcRect: PSDL_Rect;
  DestSurface: PSDL_Surface; DestRect: PSDL_Rect);
var
  Src, Dest: TSDL_Rect;
  Diff: integer;
  SrcAddr, DestAddr: cardinal;
  _ebx, _esi, _edi, _esp: cardinal;
  WorkX, WorkY: word;
  SrcMod, DestMod: cardinal;
  Bits: cardinal;
begin
  if (SrcSurface = nil) or (DestSurface = nil) then
    exit; // Remove this to make it faster
  if (SrcSurface.Format.BitsPerPixel <> DestSurface.Format.BitsPerPixel) then
    exit; // Remove this to make it faster
  if SrcRect = nil then
  begin
    with Src do
    begin
      x := 0;
      y := 0;
      w := SrcSurface.w;
      h := SrcSurface.h;
    end;
  end
  else
    Src := SrcRect^;
  if DestRect = nil then
  begin
    Dest.x := 0;
    Dest.y := 0;
  end
  else
    Dest := DestRect^;
  Dest.w := Src.w;
  Dest.h := Src.h;

  with DestSurface.Clip_Rect do
  begin
    // Source's right side is greater than the dest.cliprect
    if Dest.x + Src.w > x + w then
    begin
      smallint(Src.w) := x + w - Dest.x;
      smallint(Dest.w) := x + w - Dest.x;
      if smallint(Dest.w) < 1 then
        exit;
    end;

    // Source's bottom side is greater than the dest.clip
    if Dest.y + Src.h > y + h then
    begin
      smallint(Src.h) := y + h - Dest.y;
      smallint(Dest.h) := y + h - Dest.y;
      if smallint(Dest.h) < 1 then
        exit;
    end;

    // Source's left side is less than the dest.clip
    if Dest.x < x then
    begin
      Diff := x - Dest.x;
      Src.x := Src.x + Diff;
      smallint(Src.w) := smallint(Src.w) - Diff;
      Dest.x := x;
      smallint(Dest.w) := smallint(Dest.w) - Diff;
      if smallint(Dest.w) < 1 then
        exit;
    end;

    // Source's Top side is less than the dest.clip
    if Dest.y < y then
    begin
      Diff := y - Dest.y;
      Src.y := Src.y + Diff;
      smallint(Src.h) := smallint(Src.h) - Diff;
      Dest.y := y;
      smallint(Dest.h) := smallint(Dest.h) - Diff;
      if smallint(Dest.h) < 1 then
        exit;
    end;
  end;

  with SrcSurface^ do
  begin
    SrcAddr := cardinal(Pixels) + UInt32(Src.y) * Pitch + UInt32(Src.x) *
      Format.BytesPerPixel;
    SrcMod := Pitch - Src.w * Format.BytesPerPixel;
  end;
  with DestSurface^ do
  begin
    DestAddr := cardinal(Pixels) + UInt32(Dest.y) * Pitch + UInt32(Dest.x) *
      Format.BytesPerPixel;
    DestMod := Pitch - Dest.w * Format.BytesPerPixel;
    Bits := DestSurface.Format.BitsPerPixel;
  end;

  SDL_LockSurface(SrcSurface);
  SDL_LockSurface(DestSurface);

  case bits of
    8: asm
      mov _ebx, ebx
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov al, [esi]         // AL := source color
        cmp al, 0
        je @SkipColor         // if AL=0 then skip everything
        mov esp, eax          // ESP - source color
        mov bl, [edi]         // BL := destination color
        mov dl, bl            // DL := destination color
        and ax, $03           // Adding BLUE
        and bl, $03
        add al, bl
        cmp al, $03
        jbe @Skip1
        mov al, $03
       @Skip1:
        mov cl, al
        mov eax, esp          // Adding GREEN
        mov bl, dl
        and al, $1c
        and bl, $1c
        add al, bl
        cmp al, $1c
        jbe @Skip2
        mov al, $1c
       @Skip2:
        or cl, al
        mov eax, esp          // Adding RED
        mov bl, dl
        and ax, $e0
        and bx, $e0
        add ax, bx
        cmp ax, $e0
        jbe @Skip3
        mov al, $e0
       @Skip3:
        or cl, al
        mov [edi], cl
       @SkipColor:
        inc esi
        inc edi
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx,_ebx
      end;
    15: asm
      mov _ebx, ebx
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax          // WorkX := Src.w
       @Loopx:
        mov ax, [esi]         // AX := source color
        cmp ax, 0
        je @SkipColor         // if AX=0 then skip everything
        mov esp, eax          // ESP - source color
        mov bx, [edi]         // BX := destination color
        mov dx, bx            // DX := destination color
        and ax, $001F         // Adding BLUE
        and bx, $001F
        add ax, bx
        cmp ax, $001F
        jbe @Skip1
        mov ax, $001F
       @Skip1:
        mov cx, ax
        mov eax, esp          // Adding GREEN
        mov bx, dx
        and ax, $3E0
        and bx, $3E0
        add ax, bx
        cmp ax, $3E0
        jbe @Skip2
        mov ax, $3E0
       @Skip2:
        or cx, ax
        mov eax, esp          // Adding RED
        mov bx, dx
        and ax, $7C00
        and bx, $7C00
        add ax, bx
        cmp ax, $7C00
        jbe @Skip3
        mov ax, $7C00
       @Skip3:
        or cx, ax
        mov [edi], cx
       @SkipColor:
        add esi, 2
        add edi, 2
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx,_ebx
      end;
    16: asm
      mov _ebx, ebx
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov ax, [esi]         // AX := source color
        cmp ax, 0
        je @SkipColor         // if AX=0 then skip everything
        mov esp, eax          // ESP - source color
        mov bx, [edi]         // BX := destination color
        mov dx, bx            // DX := destination color
        and ax, $1F           // Adding BLUE
        and bx, $1F
        add ax, bx
        cmp ax, $1F
        jbe @Skip1
        mov ax, $1F
       @Skip1:
        mov cx, ax
        mov eax, esp          // Adding GREEN
        mov bx, dx
        and ax, $7E0
        and bx, $7E0
        add ax, bx
        cmp ax, $7E0
        jbe @Skip2
        mov ax, $7E0
       @Skip2:
        or cx, ax
        mov eax, esp          // Adding RED
        mov bx, dx
        and eax, $F800
        and ebx, $F800
        add eax, ebx
        cmp eax, $F800
        jbe @Skip3
        mov ax, $F800
       @Skip3:
        or cx, ax
        mov [edi], cx
       @SkipColor:
        add esi, 2
        add edi, 2
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx,_ebx
      end;
    24: asm
      mov _ebx, ebx
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       add WorkX, ax         // WorkX := Src.w * 2
       add WorkX, ax         // WorkX := Src.w * 3
       @Loopx:
        mov bl, [edi]         // BX := destination color
        mov al, [esi]         // AX := source color
        cmp al, 0
        je @Skip              // if AL=0 then skip COMPONENT
        mov ah, 0             // AX := COLOR COMPONENT
        mov bh, 0
        add bx, ax
        cmp bx, $00ff
        jb @Skip
        mov bl, $ff
       @Skip:
        mov [edi], bl

        inc esi
        inc edi
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx,_ebx
      end;
    32: asm
      mov _ebx, ebx
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       shl ax, 2
       mov WorkX, ax         // WorkX := Src.w * 4
       @Loopx:
        mov bl, [edi]         // BX := destination color
        mov al, [esi]         // AX := source color
        cmp al, 0
        je @Skip              // if AL=0 then skip COMPONENT
        mov ah, 0             // AX := COLOR COMPONENT
        mov bh, 0
        add bx, ax
        cmp bx, $00ff
        jb @Skip
        mov bl, $ff
       @Skip:
        mov [edi], bl

        inc esi
        inc edi
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx,_ebx
      end;
  end;
  SDL_UnlockSurface(SrcSurface);
  SDL_UnlockSurface(DestSurface);
end;

procedure SDL_SubSurface(SrcSurface: PSDL_Surface; SrcRect: PSDL_Rect;
  DestSurface: PSDL_Surface; DestRect: PSDL_Rect);
var
  Src, Dest: TSDL_Rect;
  Diff: integer;
  SrcAddr, DestAddr: cardinal;
  _ebx, _esi, _edi, _esp: cardinal;
  WorkX, WorkY: word;
  SrcMod, DestMod: cardinal;
  Bits: cardinal;
begin
  if (SrcSurface = nil) or (DestSurface = nil) then
    exit; // Remove this to make it faster
  if (SrcSurface.Format.BitsPerPixel <> DestSurface.Format.BitsPerPixel) then
    exit; // Remove this to make it faster
  if SrcRect = nil then
  begin
    with Src do
    begin
      x := 0;
      y := 0;
      w := SrcSurface.w;
      h := SrcSurface.h;
    end;
  end
  else
    Src := SrcRect^;
  if DestRect = nil then
  begin
    Dest.x := 0;
    Dest.y := 0;
  end
  else
    Dest := DestRect^;
  Dest.w := Src.w;
  Dest.h := Src.h;

  with DestSurface.Clip_Rect do
  begin
    // Source's right side is greater than the dest.cliprect
    if Dest.x + Src.w > x + w then
    begin
      smallint(Src.w) := x + w - Dest.x;
      smallint(Dest.w) := x + w - Dest.x;
      if smallint(Dest.w) < 1 then
        exit;
    end;

    // Source's bottom side is greater than the dest.clip
    if Dest.y + Src.h > y + h then
    begin
      smallint(Src.h) := y + h - Dest.y;
      smallint(Dest.h) := y + h - Dest.y;
      if smallint(Dest.h) < 1 then
        exit;
    end;

    // Source's left side is less than the dest.clip
    if Dest.x < x then
    begin
      Diff := x - Dest.x;
      Src.x := Src.x + Diff;
      smallint(Src.w) := smallint(Src.w) - Diff;
      Dest.x := x;
      smallint(Dest.w) := smallint(Dest.w) - Diff;
      if smallint(Dest.w) < 1 then
        exit;
    end;

    // Source's Top side is less than the dest.clip
    if Dest.y < y then
    begin
      Diff := y - Dest.y;
      Src.y := Src.y + Diff;
      smallint(Src.h) := smallint(Src.h) - Diff;
      Dest.y := y;
      smallint(Dest.h) := smallint(Dest.h) - Diff;
      if smallint(Dest.h) < 1 then
        exit;
    end;
  end;

  with SrcSurface^ do
  begin
    SrcAddr := cardinal(Pixels) + UInt32(Src.y) * Pitch + UInt32(Src.x) *
      Format.BytesPerPixel;
    SrcMod := Pitch - Src.w * Format.BytesPerPixel;
  end;
  with DestSurface^ do
  begin
    DestAddr := cardinal(Pixels) + UInt32(Dest.y) * Pitch + UInt32(Dest.x) *
      Format.BytesPerPixel;
    DestMod := Pitch - Dest.w * Format.BytesPerPixel;
    Bits := DestSurface.Format.BitsPerPixel;
  end;

  SDL_LockSurface(SrcSurface);
  SDL_LockSurface(DestSurface);

  case bits of
    8: asm
      mov _ebx, ebx
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov al, [esi]         // AL := source color
        cmp al, 0
        je @SkipColor         // if AL=0 then skip everything
        mov esp, eax          // ESP - source color
        mov bl, [edi]         // BL := destination color
        mov dl, bl            // DL := destination color
        and al, $03           // Subtract BLUE
        and bl, $03
        sub bl, al
        jns @Skip1
        mov bl, 0
       @Skip1:
        mov cl, bl
        mov eax, esp          // Subtract GREEN
        mov bl, dl
        and al, $1c
        and bl, $1c
        sub bl, al
        jns @Skip2
        mov bl, 0
       @Skip2:
        or cl, bl
        mov eax, esp          // Subtract RED
        mov bl, dl
        and ax, $e0
        and bx, $e0
        sub bx, ax
        jns @Skip3
        mov bl, 0
       @Skip3:
        or cl, bl
        mov [edi], cl
       @SkipColor:
        inc esi
        inc edi
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx,_ebx
      end;
    15: asm
      mov _ebx, ebx
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax          // WorkX := Src.w
       @Loopx:
        mov ax, [esi]         // AX := source color
        cmp ax, 0
        je @SkipColor         // if AX=0 then skip everything
        mov esp, eax          // ESP - source color
        mov bx, [edi]         // BX := destination color
        mov dx, bx            // DX := destination color
        and ax, $001F         // Subtract BLUE
        and bx, $001F
        sub bx, ax
        jns @Skip1
        mov bx, 0
       @Skip1:
        mov cx, bx
        mov eax, esp          // Subtract GREEN
        mov bx, dx
        and ax, $3E0
        and bx, $3E0
        sub bx, ax
        jns @Skip2
        mov bx, 0
       @Skip2:
        or cx, bx
        mov eax, esp          // Subtract RED
        mov bx, dx
        and ax, $7C00
        and bx, $7C00
        sub bx, ax
        jns @Skip3
        mov bx, 0
       @Skip3:
        or cx, bx
        mov [edi], cx
       @SkipColor:
        add esi, 2
        add edi, 2
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx,_ebx
      end;
    16: asm
      mov _ebx, ebx
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov ax, [esi]         // AX := source color
        cmp ax, 0
        je @SkipColor         // if AX=0 then skip everything
        mov esp, eax          // ESP - source color
        mov bx, [edi]         // BX := destination color
        mov dx, bx            // DX := destination color
        and ax, $1F           // Subtracting BLUE
        and bx, $1F
        sub bx, ax
        jns @Skip1
        mov bx, 0
       @Skip1:
        mov cx, bx
        mov eax, esp          // Adding GREEN
        mov bx, dx
        and ax, $7E0
        and bx, $7E0
        sub bx, ax
        jns @Skip2
        mov bx, 0
       @Skip2:
        or cx, bx
        mov eax, esp          // Adding RED
        mov bx, dx
        and eax, $F800
        and ebx, $F800
        sub ebx, eax
        jns @Skip3
        mov bx, 0
       @Skip3:
        or cx, bx
        mov [edi], cx
       @SkipColor:
        add esi, 2
        add edi, 2
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx,_ebx
      end;
    24: asm
      mov _ebx, ebx
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       add WorkX, ax         // WorkX := Src.w * 2
       add WorkX, ax         // WorkX := Src.w * 3
       @Loopx:
        mov bl, [edi]         // BX := destination color
        mov al, [esi]         // AX := source color
        cmp al, 0
        je @Skip              // if AL=0 then skip COMPONENT
        mov ah, 0             // AX := COLOR COMPONENT
        mov bh, 0
        sub bx, ax
        jns @Skip
        mov bl, 0
       @Skip:
        mov [edi], bl

        inc esi
        inc edi
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx,_ebx
      end;
    32: asm
      mov _ebx, ebx
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       shl ax, 2
       mov WorkX, ax         // WorkX := Src.w * 4
       @Loopx:
        mov bl, [edi]         // BX := destination color
        mov al, [esi]         // AX := source color
        cmp al, 0
        je @Skip              // if AL=0 then skip COMPONENT
        mov ah, 0             // AX := COLOR COMPONENT
        mov bh, 0
        sub bx, ax
        jns @Skip
        mov bl, 0
       @Skip:
        mov [edi], bl

        inc esi
        inc edi
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx,_ebx
      end;
  end;
  SDL_UnlockSurface(SrcSurface);
  SDL_UnlockSurface(DestSurface);
end;

procedure SDL_MonoSurface(SrcSurface: PSDL_Surface; SrcRect: PSDL_Rect;
  DestSurface: PSDL_Surface; DestRect: PSDL_Rect; Color: cardinal);
var
  Src, Dest: TSDL_Rect;
  Diff: integer;
  SrcAddr, DestAddr: cardinal;
  _ebx, _esi, _edi, _esp: cardinal;
  WorkX, WorkY: word;
  SrcMod, DestMod: cardinal;
  SrcTransparentColor: cardinal;
  Bits: cardinal;
begin
  if (SrcSurface = nil) or (DestSurface = nil) then
    exit; // Remove this to make it faster
  if (SrcSurface.Format.BitsPerPixel <> DestSurface.Format.BitsPerPixel) then
    exit; // Remove this to make it faster
  if SrcRect = nil then
  begin
    with Src do
    begin
      x := 0;
      y := 0;
      w := SrcSurface.w;
      h := SrcSurface.h;
    end;
  end
  else
    Src := SrcRect^;
  if DestRect = nil then
  begin
    Dest.x := 0;
    Dest.y := 0;
  end
  else
    Dest := DestRect^;
  Dest.w := Src.w;
  Dest.h := Src.h;

  with DestSurface.Clip_Rect do
  begin
    // Source's right side is greater than the dest.cliprect
    if Dest.x + Src.w > x + w then
    begin
      smallint(Src.w) := x + w - Dest.x;
      smallint(Dest.w) := x + w - Dest.x;
      if smallint(Dest.w) < 1 then
        exit;
    end;

    // Source's bottom side is greater than the dest.clip
    if Dest.y + Src.h > y + h then
    begin
      smallint(Src.h) := y + h - Dest.y;
      smallint(Dest.h) := y + h - Dest.y;
      if smallint(Dest.h) < 1 then
        exit;
    end;

    // Source's left side is less than the dest.clip
    if Dest.x < x then
    begin
      Diff := x - Dest.x;
      Src.x := Src.x + Diff;
      smallint(Src.w) := smallint(Src.w) - Diff;
      Dest.x := x;
      smallint(Dest.w) := smallint(Dest.w) - Diff;
      if smallint(Dest.w) < 1 then
        exit;
    end;

    // Source's Top side is less than the dest.clip
    if Dest.y < y then
    begin
      Diff := y - Dest.y;
      Src.y := Src.y + Diff;
      smallint(Src.h) := smallint(Src.h) - Diff;
      Dest.y := y;
      smallint(Dest.h) := smallint(Dest.h) - Diff;
      if smallint(Dest.h) < 1 then
        exit;
    end;
  end;

  with SrcSurface^ do
  begin
    SrcAddr := cardinal(Pixels) + UInt32(Src.y) * Pitch + UInt32(Src.x) *
      Format.BytesPerPixel;
    SrcMod := Pitch - Src.w * Format.BytesPerPixel;
    SrcTransparentColor := format.colorkey;
  end;
  with DestSurface^ do
  begin
    DestAddr := cardinal(Pixels) + UInt32(Dest.y) * Pitch + UInt32(Dest.x) *
      Format.BytesPerPixel;
    DestMod := Pitch - Dest.w * Format.BytesPerPixel;
    Bits := DestSurface.Format.BitsPerPixel;
  end;

  SDL_LockSurface(SrcSurface);
  SDL_LockSurface(DestSurface);

  case bits of
    8: asm
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      mov ecx, Color
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov al, [esi]         // AL := source color
        movzx eax, al
        cmp eax, SrcTransparentColor
        je @SkipColor         // if AL=Transparent color then skip everything
        mov [edi], cl
       @SkipColor:
        inc esi
        inc edi
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      end;
    15, 16: asm
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      mov ecx, Color
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov ax, [esi]         // AX := source color
        movzx eax, ax
        cmp eax, SrcTransparentColor
        je @SkipColor         // if AX=Transparent color then skip everything
        mov [edi], cx
       @SkipColor:
        inc esi
        inc esi
        inc edi
        inc edi
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      end;
    24: asm
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov _ebx, ebx
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      mov ecx, Color
      and ecx, $00ffffff
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov eax, [esi]         // EAX := source color
        and eax, $00ffffff
        cmp eax, SrcTransparentColor
        je @SkipColor         // if EAX=Transparent color then skip everything
        mov ebx, [edi]
        and ebx, $ff000000
        or ebx, ecx
        mov [edi], ecx
       @SkipColor:
        add esi, 3
        add edi, 3
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp, _esp
      mov edi, _edi
      mov esi, _esi
      mov ebx, _ebx
      end;
    32: asm
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      mov ecx, Color
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov eax, [esi]         // EAX := source color
        cmp eax, SrcTransparentColor
        je @SkipColor         // if EAX=Transparent color then skip everything
        mov [edi], ecx
       @SkipColor:
        add esi, 4
        add edi, 4
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       dec WorkY
      jnz @LoopY
      mov esp, _esp
      mov edi, _edi
      mov esi, _esi
      end;
  end;
  SDL_UnlockSurface(SrcSurface);
  SDL_UnlockSurface(DestSurface);
end;

// TextureRect.w and TextureRect.h are not used.
// The TextureSurface's size MUST larger than the drawing rectangle!!!

procedure SDL_TexturedSurface(SrcSurface: PSDL_Surface; SrcRect: PSDL_Rect;
  DestSurface: PSDL_Surface; DestRect: PSDL_Rect; Texture: PSDL_Surface;
  TextureRect: PSDL_Rect);
var
  Src, Dest: TSDL_Rect;
  Diff: integer;
  SrcAddr, DestAddr, TextAddr: cardinal;
  _ebx, _esi, _edi, _esp: cardinal;
  WorkX, WorkY: word;
  SrcMod, DestMod, TextMod: cardinal;
  SrcTransparentColor: cardinal;
  Bits: cardinal;
begin
  if (SrcSurface = nil) or (DestSurface = nil) then
    exit; // Remove this to make it faster
  if (SrcSurface.Format.BitsPerPixel <> DestSurface.Format.BitsPerPixel) then
    exit; // Remove this to make it faster
  if SrcRect = nil then
  begin
    with Src do
    begin
      x := 0;
      y := 0;
      w := SrcSurface.w;
      h := SrcSurface.h;
    end;
  end
  else
    Src := SrcRect^;
  if DestRect = nil then
  begin
    Dest.x := 0;
    Dest.y := 0;
  end
  else
    Dest := DestRect^;
  Dest.w := Src.w;
  Dest.h := Src.h;

  with DestSurface.Clip_Rect do
  begin
    // Source's right side is greater than the dest.cliprect
    if Dest.x + Src.w > x + w then
    begin
      smallint(Src.w) := x + w - Dest.x;
      smallint(Dest.w) := x + w - Dest.x;
      if smallint(Dest.w) < 1 then
        exit;
    end;

    // Source's bottom side is greater than the dest.clip
    if Dest.y + Src.h > y + h then
    begin
      smallint(Src.h) := y + h - Dest.y;
      smallint(Dest.h) := y + h - Dest.y;
      if smallint(Dest.h) < 1 then
        exit;
    end;

    // Source's left side is less than the dest.clip
    if Dest.x < x then
    begin
      Diff := x - Dest.x;
      Src.x := Src.x + Diff;
      smallint(Src.w) := smallint(Src.w) - Diff;
      Dest.x := x;
      smallint(Dest.w) := smallint(Dest.w) - Diff;
      if smallint(Dest.w) < 1 then
        exit;
    end;

    // Source's Top side is less than the dest.clip
    if Dest.y < y then
    begin
      Diff := y - Dest.y;
      Src.y := Src.y + Diff;
      smallint(Src.h) := smallint(Src.h) - Diff;
      Dest.y := y;
      smallint(Dest.h) := smallint(Dest.h) - Diff;
      if smallint(Dest.h) < 1 then
        exit;
    end;
  end;

  with SrcSurface^ do
  begin
    SrcAddr := cardinal(Pixels) + UInt32(Src.y) * Pitch + UInt32(Src.x) *
      Format.BytesPerPixel;
    SrcMod := Pitch - Src.w * Format.BytesPerPixel;
    SrcTransparentColor := format.colorkey;
  end;
  with DestSurface^ do
  begin
    DestAddr := cardinal(Pixels) + UInt32(Dest.y) * Pitch + UInt32(Dest.x) *
      Format.BytesPerPixel;
    DestMod := Pitch - Dest.w * Format.BytesPerPixel;
    Bits := DestSurface.Format.BitsPerPixel;
  end;
  with Texture^ do
  begin
    TextAddr := cardinal(Pixels) + UInt32(TextureRect.y) * Pitch +
      UInt32(TextureRect.x) * Format.BytesPerPixel;
    TextMod := Pitch - Src.w * Format.BytesPerPixel;
  end;

  SDL_LockSurface(SrcSurface);
  SDL_LockSurface(DestSurface);
  SDL_LockSurface(Texture);

  case bits of
    8: asm
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov _ebx, ebx
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ebx, TextAddr
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov al, [esi]         // AL := source color
        movzx eax, al
        cmp eax, SrcTransparentColor
        je @SkipColor         // if AL=Transparent color then skip everything
        mov al, [ebx]
        mov [edi], al
       @SkipColor:
        inc esi
        inc edi
        inc ebx
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       add ebx, TextMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx, _ebx
      end;
    15, 16: asm
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ecx, TextAddr
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov ax, [esi]         // AL := source color
        movzx eax, ax
        cmp eax, SrcTransparentColor
        je @SkipColor         // if AL=Transparent color then skip everything
        mov ax, [ecx]
        mov [edi], ax
       @SkipColor:
        inc esi
        inc esi
        inc edi
        inc edi
        inc ecx
        inc ecx
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       add ecx, TextMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      end;
    24: asm
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov _ebx, ebx
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ebx, TextAddr
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov eax, [esi]         // AL := source color
        and eax, $00ffffff
        cmp eax, SrcTransparentColor
        je @SkipColor         // if AL=Transparent color then skip everything
        mov eax, [ebx]
        and eax, $00ffffff
        mov ecx, [edi]
        and ecx, $ff000000
        or ecx, eax
        mov [edi], eax
       @SkipColor:
        add esi, 3
        add edi, 3
        add ebx, 3
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       add ebx, TextMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      mov ebx, _ebx
      end;
    32: asm
      mov _esi, esi
      mov _edi, edi
      mov _esp, esp
      mov esi, SrcAddr      // ESI - Source Offset
      mov edi, DestAddr     // EDI - Destination Offset
      mov ecx, TextAddr
      mov ax, Src.h         //  WorkY := Src.h
      mov WorkY, ax
      @LoopY:
       mov ax, Src.w
       mov WorkX, ax         // WorkX := Src.w
       @Loopx:
        mov eax, [esi]         // AL := source color
        cmp eax, SrcTransparentColor
        je @SkipColor         // if AL=Transparent color then skip everything
        mov eax, [ecx]
        mov [edi], eax
       @SkipColor:
        add esi, 4
        add edi, 4
        add ecx, 4
        dec WorkX
       jnz @LoopX
       add esi, SrcMod
       add edi, DestMod
       add ecx, TextMod
       dec WorkY
      jnz @LoopY
      mov esp,_esp
      mov edi,_edi
      mov esi,_esi
      end;
  end;
  SDL_UnlockSurface(SrcSurface);
  SDL_UnlockSurface(DestSurface);
  SDL_UnlockSurface(Texture);
end;

// Draw a line between x1,y1 and x2,y2 to the given surface
// NOTE: The surface must be locked before calling this!

procedure SDL_DrawLine(Surface: PSDL_Surface; x1, y1, x2, y2: integer; Color:
  cardinal);
var
  dx, dy, sdx, sdy, x, y, px, py: integer;
begin
  dx := x2 - x1;
  dy := y2 - y1;
  if dx < 0 then
    sdx := -1
  else
    sdx := 1;
  if dy < 0 then
    sdy := -1
  else
    sdy := 1;
  dx := sdx * dx + 1;
  dy := sdy * dy + 1;
  x := 0;
  y := 0;
  px := x1;
  py := y1;
  if dx >= dy then
  begin
    for x := 0 to dx - 1 do
    begin
      SDL_NewPutPixel(Surface, px, py, Color);
      y := y + dy;
      if y >= dx then
      begin
        y := y - dx;
        py := py + sdy;
      end;
      px := px + sdx;
    end;
  end
  else
  begin
    for y := 0 to dy - 1 do
    begin
      SDL_NewPutPixel(Surface, px, py, Color);
      x := x + dx;
      if x >= dy then
      begin
        x := x - dy;
        px := px + sdx;
      end;
      py := py + sdy;
    end;
  end;
end;

procedure SDL_AddLine(Surface: PSDL_Surface; x1, y1, x2, y2: integer; Color:
  cardinal);
var
  dx, dy, sdx, sdy, x, y, px, py: integer;
begin
  dx := x2 - x1;
  dy := y2 - y1;
  if dx < 0 then
    sdx := -1
  else
    sdx := 1;
  if dy < 0 then
    sdy := -1
  else
    sdy := 1;
  dx := sdx * dx + 1;
  dy := sdy * dy + 1;
  x := 0;
  y := 0;
  px := x1;
  py := y1;
  if dx >= dy then
  begin
    for x := 0 to dx - 1 do
    begin
      SDL_AddPixel(Surface, px, py, Color);
      y := y + dy;
      if y >= dx then
      begin
        y := y - dx;
        py := py + sdy;
      end;
      px := px + sdx;
    end;
  end
  else
  begin
    for y := 0 to dy - 1 do
    begin
      SDL_AddPixel(Surface, px, py, Color);
      x := x + dx;
      if x >= dy then
      begin
        x := x - dy;
        px := px + sdx;
      end;
      py := py + sdy;
    end;
  end;
end;

procedure SDL_SubLine(Surface: PSDL_Surface; x1, y1, x2, y2: integer; Color:
  cardinal);
var
  dx, dy, sdx, sdy, x, y, px, py: integer;
begin
  dx := x2 - x1;
  dy := y2 - y1;
  if dx < 0 then
    sdx := -1
  else
    sdx := 1;
  if dy < 0 then
    sdy := -1
  else
    sdy := 1;
  dx := sdx * dx + 1;
  dy := sdy * dy + 1;
  x := 0;
  y := 0;
  px := x1;
  py := y1;
  if dx >= dy then
  begin
    for x := 0 to dx - 1 do
    begin
      SDL_SubPixel(Surface, px, py, Color);
      y := y + dy;
      if y >= dx then
      begin
        y := y - dx;
        py := py + sdy;
      end;
      px := px + sdx;
    end;
  end
  else
  begin
    for y := 0 to dy - 1 do
    begin
      SDL_SubPixel(Surface, px, py, Color);
      x := x + dx;
      if x >= dy then
      begin
        x := x - dy;
        px := px + sdx;
      end;
      py := py + sdy;
    end;
  end;
end;

// flips a rectangle vertically on given surface

procedure SDL_FlipRectV(Surface: PSDL_Surface; Rect: PSDL_Rect);
var
  TmpRect: TSDL_Rect;
  Locked: boolean;
  y, FlipLength, RowLength: integer;
  Row1, Row2: Pointer;
  OneRow: TByteArray; // Optimize it if you wish
begin
  if Surface <> nil then
  begin
    if Rect = nil then
    begin // if Rect=nil then we flip the whole surface
      TmpRect := SDLRect(0, 0, Surface.w, Surface.h);
      Rect := @TmpRect;
    end;
    FlipLength := Rect^.h shr 1 - 1;
    RowLength := Rect^.w * Surface^.format.BytesPerPixel;
    if SDL_MustLock(Surface) then
    begin
      Locked := true;
      SDL_LockSurface(Surface);
    end
    else
      Locked := false;
    Row1 := pointer(cardinal(Surface^.Pixels) + UInt32(Rect^.y) *
      Surface^.Pitch);
    Row2 := pointer(cardinal(Surface^.Pixels) + (UInt32(Rect^.y) + Rect^.h - 1)
      * Surface^.Pitch);
    for y := 0 to FlipLength do
    begin
      Move(Row1^, OneRow, RowLength);
      Move(Row2^, Row1^, RowLength);
      Move(OneRow, Row2^, RowLength);
      inc(cardinal(Row1), Surface^.Pitch);
      dec(cardinal(Row2), Surface^.Pitch);
    end;
    if Locked then
      SDL_UnlockSurface(Surface);
  end;
end;

// flips a rectangle horizontally on given surface

procedure SDL_FlipRectH(Surface: PSDL_Surface; Rect: PSDL_Rect);
type
  T24bit = packed array[0..2] of byte;
  T24bitArray = packed array[0..8191] of T24bit;
  P24bitArray = ^T24bitArray;
  TLongWordArray = array[0..8191] of LongWord;
  PLongWordArray = ^TLongWordArray;
var
  TmpRect: TSDL_Rect;
  Row8bit: PByteArray;
  Row16bit: PWordArray;
  Row24bit: P24bitArray;
  Row32bit: PLongWordArray;
  y, x, RightSide, FlipLength: integer;
  Pixel: cardinal;
  Pixel24: T24bit;
  Locked: boolean;
begin
  if Surface <> nil then
  begin
    if Rect = nil then
    begin
      TmpRect := SDLRect(0, 0, Surface.w, Surface.h);
      Rect := @TmpRect;
    end;
    FlipLength := Rect^.w shr 1 - 1;
    if SDL_MustLock(Surface) then
    begin
      Locked := true;
      SDL_LockSurface(Surface);
    end
    else
      Locked := false;
    case Surface^.format.BytesPerPixel of
      1:
        begin
          Row8Bit := pointer(cardinal(Surface^.pixels) + UInt32(Rect^.y) *
            Surface^.pitch);
          for y := 1 to Rect^.h do
          begin
            RightSide := Rect^.w - 1;
            for x := 0 to FlipLength do
            begin
              Pixel := Row8Bit^[x];
              Row8Bit^[x] := Row8Bit^[RightSide];
              Row8Bit^[RightSide] := Pixel;
              dec(RightSide);
            end;
            inc(cardinal(Row8Bit), Surface^.pitch);
          end;
        end;
      2:
        begin
          Row16Bit := pointer(cardinal(Surface^.pixels) + UInt32(Rect^.y) *
            Surface^.pitch);
          for y := 1 to Rect^.h do
          begin
            RightSide := Rect^.w - 1;
            for x := 0 to FlipLength do
            begin
              Pixel := Row16Bit^[x];
              Row16Bit^[x] := Row16Bit^[RightSide];
              Row16Bit^[RightSide] := Pixel;
              dec(RightSide);
            end;
            inc(cardinal(Row16Bit), Surface^.pitch);
          end;
        end;
      3:
        begin
          Row24Bit := pointer(cardinal(Surface^.pixels) + UInt32(Rect^.y) *
            Surface^.pitch);
          for y := 1 to Rect^.h do
          begin
            RightSide := Rect^.w - 1;
            for x := 0 to FlipLength do
            begin
              Pixel24 := Row24Bit^[x];
              Row24Bit^[x] := Row24Bit^[RightSide];
              Row24Bit^[RightSide] := Pixel24;
              dec(RightSide);
            end;
            inc(cardinal(Row24Bit), Surface^.pitch);
          end;
        end;
      4:
        begin
          Row32Bit := pointer(cardinal(Surface^.pixels) + UInt32(Rect^.y) *
            Surface^.pitch);
          for y := 1 to Rect^.h do
          begin
            RightSide := Rect^.w - 1;
            for x := 0 to FlipLength do
            begin
              Pixel := Row32Bit^[x];
              Row32Bit^[x] := Row32Bit^[RightSide];
              Row32Bit^[RightSide] := Pixel;
              dec(RightSide);
            end;
            inc(cardinal(Row32Bit), Surface^.pitch);
          end;
        end;
    end;
    if Locked then
      SDL_UnlockSurface(Surface);
  end;
end;

// Use with caution! The procedure allocates memory for TSDL_Rect and return with its pointer.
// But you MUST free it after you don't need it anymore!!!

function PSDLRect(aLeft, aTop, aWidth, aHeight: integer): PSDL_Rect;
var
  Rect: PSDL_Rect;
begin
  New(Rect);
  with Rect^ do
  begin
    x := aLeft;
    y := aTop;
    w := aWidth;
    h := aHeight;
  end;
  Result := Rect;
end;

function SDLRect(aLeft, aTop, aWidth, aHeight: integer): TSDL_Rect;
begin
  with result do
  begin
    x := aLeft;
    y := aTop;
    w := aWidth;
    h := aHeight;
  end;
end;

function SDLRect(aRect: TRect): TSDL_Rect;
begin
  with aRect do
    result := SDLRect(Left, Top, Right - Left, Bottom - Top);
end;

procedure _Stretch8(Surface, Dst_Surface: PSDL_Surface; x1, x2, y1, y2, yr, yw,
  depth: integer);
var
  dx, dy, e, d, dx2: integer;
  src_pitch, dst_pitch: uint16;
  src_pixels, dst_pixels: PUint8;
begin
  if (yw >= dst_surface^.h) then
    exit;

  dx := (x2 - x1);
  dy := (y2 - y1);

  dy := dy shl 1;
  e := dy - dx;
  dx2 := dx shl 1;

  src_pitch := Surface^.pitch;
  dst_pitch := dst_surface^.pitch;

  src_pixels := PUint8(integer(Surface^.pixels) + yr * src_pitch + y1 * depth);
  dst_pixels := PUint8(integer(dst_surface^.pixels) + yw * dst_pitch + x1 *
    depth);

  for d := 0 to dx - 1 do
  begin
    move(src_pixels^, dst_pixels^, depth);
    while (e >= 0) do
    begin
      inc(src_pixels, depth);
      e := e - dx2;
    end;
    inc(dst_pixels, depth);
    e := e + dy;
  end;
end;

function sign(x: integer): integer;
begin
  if x > 0 then
    result := 1
  else
    result := -1;
end;

// Stretches a part of a surface

function SDL_ScaleSurfaceRect(Surface: PSDL_Surface; SrcX1, SrcY1, SrcW, SrcH,
  Width, Height: integer): PSDL_Surface;
var
  dst_surface: PSDL_Surface;
  dx, dy, e, d, dx2, srcx2, srcy2: integer;
  destx1, desty1: integer;
begin
  srcx2 := srcx1 + SrcW;
  srcy2 := srcy1 + SrcH;

  result := nil;
  destx1 := 0;
  desty1 := 0;
  dx := abs(integer(Height - desty1));
  dy := abs(integer(SrcY2 - SrcY1));
  e := (dy shl 1) - dx;
  dx2 := dx shl 1;
  dy := dy shl 1;

  dst_surface := SDL_CreateRGBSurface(SDL_HWPALETTE, width - destx1, Height -
    desty1,
    Surface^.Format^.BitsPerPixel, 0, 0, 0, 0);

  if (dst_surface^.format^.BytesPerPixel = 1) then
    SDL_SetColors(dst_surface, @Surface^.format^.palette^.colors^[0], 0, 256);
  SDL_SetColorKey(dst_surface, sdl_srccolorkey, surface^.format^.colorkey);

  if (SDL_MustLock(dst_surface)) then
    if (SDL_LockSurface(dst_surface) < 0) then
      exit;

  for d := 0 to dx - 1 do
  begin
    _Stretch8(Surface, dst_surface, destx1, Width, SrcX1, SrcX2, SrcY1, desty1,
      Surface^.format^.BytesPerPixel);
    while e >= 0 do
    begin
      inc(SrcY1);
      e := e - dx2;
    end;
    inc(desty1);
    e := e + dy;
  end;

  if SDL_MUSTLOCK(dst_surface) then
    SDL_UnlockSurface(dst_surface);

  result := dst_surface;
end;

procedure _MoveLine(Surface: PSDL_Surface; x1, x2, y1, xofs, depth: integer);
var
  src_pixels, dst_pixels: PUint8;
  i: integer;
begin
  src_pixels := PUint8(integer(Surface^.pixels) + Surface^.w * y1 * depth + x2 *
    depth);
  dst_pixels := PUint8(integer(Surface^.pixels) + Surface^.w * y1 * depth + (x2
    + xofs) * depth);
  for i := x2 downto x1 do
  begin
    move(src_pixels^, dst_pixels^, depth);
    dec(src_pixels);
    dec(dst_pixels);
  end;

end;

procedure SDL_ScrollX(Surface: PSDL_Surface; DifX: integer);
var
  r1, r2: TSDL_Rect;
  buffer: PSDL_Surface;
begin
  if (Surface <> nil) and (DifX <> 0) then
  begin
    buffer := SDL_CreateRGBSurface(SDL_HWSURFACE, (Surface^.w - DifX) * 2,
      Surface^.h * 2,
      Surface^.Format^.BitsPerPixel, 0, 0, 0, 0);
    if buffer <> nil then
    begin
      if (buffer^.format^.BytesPerPixel = 1) then
        SDL_SetColors(buffer, @Surface^.format^.palette^.colors^[0], 0, 256);
      r1 := SDLRect(DifX, 0, buffer^.w, buffer^.h);
      r2 := SDLRect(0, 0, buffer^.w, buffer^.h);
      SDL_BlitSurface(Surface, @r1, buffer, @r2);
      SDL_BlitSurface(buffer, @r2, Surface, @r2);
      SDL_FreeSurface(buffer);
    end;
  end;
end;

{ Return the pixel value at (x, y)
NOTE: The surface must be locked before calling this! }

function SDL_GetPixel(surface: PSDL_Surface; x: integer; y: integer): Uint32;
var
  bpp: UInt32;
  p: PInteger;
begin
  bpp := surface.format.BytesPerPixel;
  // Here p is the address to the pixel we want to retrieve
  p := Pointer(Uint32(surface.pixels) + UInt32(y) * surface.pitch + UInt32(x) *
    bpp);
  case bpp of
    1: result := PUint8(p)^;
    2: result := PUint16(p)^;
    3: if (SDL_BYTEORDER = SDL_BIG_ENDIAN) then
        result := PUInt8Array(p)[0] shl 16 or PUInt8Array(p)[1] shl 8 or
          PUInt8Array(p)[2]
      else
        result := PUInt8Array(p)[0] or PUInt8Array(p)[1] shl 8 or
          PUInt8Array(p)[2] shl 16;
    4: result := PUint32(p)^;
  else
    result := 0; // shouldn't happen, but avoids warnings
  end;
end;

{ Set the pixel at (x, y) to the given value
  NOTE: The surface must be locked before calling this! }

procedure SDL_PutPixel(surface_: PSDL_Surface; x: integer; y: integer; pixel:
  Uint32);
var
  bpp: UInt32;
  p: PInteger;
begin
  bpp := surface_.format.BytesPerPixel;
  p := Pointer(Uint32(surface_.pixels) + UInt32(y) * surface_.pitch + UInt32(x)
    * bpp);
  case bpp of
    1: PUint8(p)^ := pixel;
    2: PUint16(p)^ := pixel;
    3: if (SDL_BYTEORDER = SDL_BIG_ENDIAN) then
      begin
        PUInt8Array(p)[0] := (pixel shr 16) and $FF;
        PUInt8Array(p)[1] := (pixel shr 8) and $FF;
        PUInt8Array(p)[2] := pixel and $FF;
      end
      else
      begin
        PUInt8Array(p)[0] := pixel and $FF;
        PUInt8Array(p)[1] := (pixel shr 8) and $FF;
        PUInt8Array(p)[2] := (pixel shr 16) and $FF;
      end;
    4:
      PUint32(p)^ := pixel;
  end;
end;

end.

