(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                  AxKermit.pas 1.02                    *}
{*********************************************************}

{Global defines potentially affecting this unit}
{$I AxDefine.inc}

unit AxKermit;

interface

uses
  Classes,
  SysUtils,
  LibC,
  AxMisc,
  AxPort,
  AxPrtDrv,
  AxString;

const
  {Constants}
  ApxKermitDefMinRepeatCnt = 4;      {Minimum characters to use repeat prefix}
  ApxKermitFastAbort = False;        {Use Error packet for aborting}
  ApxKermitDefHibitPrefix = '&';     {Default char for hibit prefixing}
  ApxKermitCancelWait = 10000;         {Wait 10 seconds for cancel transmit}
  ApxKermitDiscardChar = 'D';        {For signaling an abort}
  ApxKermitMaxWindowSlots = 27;      {Avoids MS-Kermit bug}

  {For estimating protocol transfer times}
  ApxKermitOverhead = 20;      {Bytes of overhead for each block}
  ApxKermitTurnDelay = 1000;   {Msecs of turn around delay}
  ApxSWCKermitTurnDelay = 0;   {Msecs of turn around delay on SWC xfers}

  {#Z+}
  {Packet types}
  KBreak           = 'B';        {Break transmission (EOT)}
  KData            = 'D';        {Data packet}
  KError           = 'E';        {Error packet}
  KFile            = 'F';        {File header packet}
  KNak             = 'N';        {Negative acknowledge packet}
  KSendInit        = 'S';        {Initial packet (exchange options)}
  KDisplay         = 'X';        {Display text on screen packet}
  KAck             = 'Y';        {Acknowledge packet}
  KEndOfFile       = 'Z';        {End of file packet}
  {#Z-}

type
  TApxKermitOptions = record
    MaxPacketLen     : Byte;
    MaxTimeout       : Byte;
    PadCount         : Byte;
    PadChar          : Char;
    Terminator       : Char;
    CtlPrefix        : Char;
    HibitPrefix      : Char;
    Check            : Char;
    RepeatPrefix     : Char;
    CapabilitiesMask : Byte;
    WindowSize       : Byte;
    MaxLongPacketLen : Cardinal;
    SendInitSize     : Cardinal;
  end;
  
const
  {Default kermit options (from the Kermit Protocol Manual)}
  ApxDefKermitOptions : TApxKermitOptions =
    (MaxPacketLen : 80;                    {80 characters}
     MaxTimeout :  5;                      {5 seconds}
     PadCount : 0;                         {No pad chars}
     PadChar : #0;                         {Null pad char}
     Terminator : cCR;                     {Carriage return}
     CtlPrefix : '#';                      {'#' char}
     HibitPrefix : 'Y';                    {Space means no hibit prefixing}
     Check : '1';                          {1 byte chksum}
     RepeatPrefix : '~';                   {Default repeat prefix}
     CapabilitiesMask : 0;                 {No default extended caps}
     WindowSize : 0;                       {No default windows}
     MaxLongPacketLen : 0);                {No default long packets}

  {#Z+}
  {Default kermit options (from the Kermit Protocol Manual)}
  ApxMissingKermitOptions : TApxKermitOptions =
    (MaxPacketLen : 80;                    {80 characters}
     MaxTimeout :  5;                      {5 seconds}
     PadCount : 0;                         {No pad chars}
     PadChar : #0;                         {Null pad char}
     Terminator : cCR;                     {Carriage return}
     CtlPrefix : '#';                      {'#' char}
     HibitPrefix : ' ';                    {No hibit prefixing}
     Check : '1';                          {1 byte chksum}
     RepeatPrefix : ' ';                   {Default repeat prefix}
     CapabilitiesMask : 0;                 {No default extended caps}
     WindowSize : 0;                       {No default windows}
     MaxLongPacketLen : 0);                {No default long packets}
   {#Z-}

type
  TApxKermitState = (
    {Transmit states}
    tkInit,           {Send SendInit packet}
    tkInitReply,      {Wait for header reply to SendInit}
    tkCollectInit,    {Collect data packet for SendInit reply}
    tkOpenFile,       {Open next file to transmit}
    tkSendFile,       {Send File packet}
    tkFileReply,      {Wait for header reply to File}
    tkCollectFile,    {Collect data packet for File reply}
    tkCheckTable,     {Check table for space, escape next block if room}
    tkSendData,       {Send next Data packet}
    tkBlockReply,     {Wait for header reply to Data}
    tkCollectBlock,   {Collect data packet for Data reply}
    tkSendEof,        {Send Eof packet}
    tkEofReply,       {Wait for header reply to Eof}
    tkCollectEof,     {Collect data packet for Eof reply}
    tkSendBreak,      {Send Break packet}
    tkBreakReply,     {Wait for header reply to Break}
    tkCollectBreak,   {Collect data packet for Break reply}
    tkComplete,       {Send Complete packet}
    tkWaitCancel,     {Wait for cancel to go out}
    tkError,          {Done, log and clean up}
    tkDone,           {Signals end of protocol}

    {Receive states}
    rkInit,           {Set initial timer}
    rkGetInit,        {Wait for SendInit header}
    rkCollectInit,    {Collect SendInit data field}
    rkGetFile,        {Wait for File header}
    rkCollectFile,    {Collect File data field}
    rkGetData,        {Wait for Data header}
    rkCollectData,    {Collect Data data field}
    rkComplete,       {Normal completion}
    rkWaitCancel,     {Wait for cancel to go out}
    rkError,          {Error completion}
    rkDone);          {Signals end of protocolcompletion}

  TApxKermitHeaderState = (
    hskNone,           {No header collection in process}
    hskGotMark,        {Got mark}
    hskGotLen,         {Got length byte}
    hskGotSeq,         {Got sequence number}
    hskGotType,        {Got packet type}
    hskGotLong1,       {Got first byte of long length}
    hskGotLong2,       {Got second byte of long length}
    hskDone);          {Got everything}

  TApxKermitDataState = (
    dskData,           {Collecting data bytes}
    dskCheck1,         {Collecting check bytes}
    dskCheck2,         {Collecting check bytes}
    dskCheck3);        {Collecting check bytes}

  {Describes 4K internal input buffer for Kermit}
  PApxInBuffer = ^TApxInBuffer;
  TApxInBuffer = array[1..4096] of Char;

  {Holds info about Kermit data in Window slots}
  TApxSlotInfo = record
    Len     : Integer;
    Seq     : Integer;
    InUse   : Boolean;
    Acked   : Boolean;
    Retries : Cardinal;
  end;

  {Sliding window table, info}
  TApxInfoTable = array[1..ApxKermitMaxWindowSlots] of TApxSlotInfo;

  {Sliding window table, data}
  PApxDataTable = ^TApxDataTable;
  TApxDataTable = array[0..(ApxKermitMaxWindowSlots*1024)-1] of Char;

type
  TApxKermitDriver = class (TApxBaseProtocolDriver)
  
    private

      {General}
      
      FPacketType      : Char;            {Type of last packet}
      FKermitState     : TApxKermitState;    {Current state of machine}
      FKermitHeaderState: TApxKermitHeaderState; {Current header state}
      FKermitDataState : TApxKermitDataState;   {Current data state}
      FCheckKnown      : Boolean;         {True if we've agreed on check type}
      FLPInUse         : Boolean;         {True if we're using long packets}
      FUsingHibit      : Boolean;         {True if prefixing hibit chars}
      FUsingRepeat     : Boolean;         {True if using repeat cnt feature}
      FReceiveInProgress : Boolean;       {True if receiving a file}
      FTransmitInProgress : Boolean;      {True if transmitting a file}
      FDataLen         : Cardinal;        {Length of sent packet data field}
      FRecDataLen      : Cardinal;        {Length of recd packet data field}
      FActualDataLen   : Cardinal;        {Length decoded data bytes}
      FMinRepeatCnt    : Cardinal;        {Min threshold to use repeat feature}
      FRecBlockNum     : Cardinal;        {Blocknum of last received packet}
      FExpectedAck     : Cardinal;        {Blocknum of next expected Ack}
      FBlockCheck2     : Cardinal;        {For holding Crc check value}
      FSWCTurnDelay    : Cardinal;        {Turn delay to use for SWC mode}
      FKermitOptions   : TApxKermitOptions;  {Options for this transfer}
      FRmtKermitOptions: TApxKermitOptions;  {Options remote says to use}

      {Internal buffer management}
      FInBuff          : PApxInBuffer;       {Internal 4K input buffer}
      FInBuffHead      : Cardinal;        {Pointer to head of buffer}
      FInBuffTail      : Cardinal;        {Pointer to tail of buffer}

      {Transmitting...}
      FWorkEndPending  : Boolean;         {True if no more WorkBlocks}
      FWorkLen         : Cardinal;        {Count of bytes in temp pool}
      FLastWorkIndex   : Cardinal;        {For managing data pool}
      FWorkBlock       : PApxDataBlock;      {Holds quoted data block}

      {Table management}
      FTableSize       : Cardinal;        {Size of window table, 1-31}
      FTableHead       : Cardinal;        {Newest used slot}
      FTableTail       : Cardinal;        {Oldest used slot, rcv only}
      FBlockIndex      : Cardinal;        {Collects data field}
      FNext2Send       : Integer;         {Slot in table to send}
      FDataTable       : PApxDataTable;      {Window table data}
      FInfoTable       : TApxInfoTable;      {Window table info}

      {Temp variables used in state machine}
      FTempCheck       : Char;            {Used for collecting check chars}
      FC1              : Char;            {Used for collecting check chars}
      FC2              : Char;            {Used for collecting check chars}
      FC3              : Char;            {Used for collecting check chars}
      FSkipped         : Boolean;         {True if file was not accepted}
      FGetLong         : Boolean;         {True for long header}
      FLongCheck       : Integer;         {Long header checksum}
      FSaveCheck2      : Cardinal;        {Save incoming check between states}
      FSaveCheck       : LongInt;        {Save incoming check between states}
      
    protected
    
      procedure AllocateWindowTable;
      procedure DeallocateWindowTable;
      procedure RawInit;
      function CharReady : Boolean;
      function GetChar : Char;
      procedure CompactInBuff;
      procedure FillInBuff;
      procedure FlushInBuffer;
      procedure UpdateBlockCheck (CurByte: Byte);
      procedure SendBlockCheck;
      procedure PutToChar (C : Char);
      procedure PutHeader (HType : Char; Len : Cardinal);
      procedure TransmitBlock (var Block : TApxDataBlock;
                                   BLen  : Cardinal;
                                   BType : Char);
      procedure SendTerminator;
      procedure SendPacket (PT : Char);
      procedure SendError (Msg : String);
      procedure Cancel;   
      procedure ResetStatus;
      procedure GetDataChar (var C          : Char;
                             var TableIndex : Cardinal;
                             var RepeatCnt  : Cardinal);
      procedure CheckForHeader;
      function NextSeq (I : Integer) : Integer;
      function PrevSeq (I : Integer) : Integer;
      function TableFull : Boolean;
      function PacketsOutstanding : Boolean;
      function GetOldestSequence : Integer;
      function SeqInTable (CurSeq : Integer) : Integer;
      procedure GotAck (CurSeq : Cardinal);
      procedure WritePacket (Index : Byte);
      function SeqGreater (Seq1, Seq2 : Byte) : Boolean;
      function LoSeq : Byte;
      function HiSeq : Byte;
      function SeqDiff (Seq1, Seq2 : Byte) : Byte;
      procedure AddToTable (Seq : Byte);
      procedure SendNak;
      procedure SendAck (Seq : Byte);
      function DataCount (Index : Byte) : Cardinal;
      procedure ProcessDataPacket;
      function IncTableIndex (Index, Increment : Byte) : Byte;
      procedure FlushTableToDisk;
      procedure ReceiveBlock;
      procedure ExpandFileInfo;
      procedure ExtractFileInfo;
      procedure SendInitialize;
      procedure SendDataPacket (Slot : Cardinal);
      procedure ResendDataPacket (Seq : Integer);
      procedure SendFilePacket;
      procedure ProcessOptions;
      procedure SendOptions;
      function CheckRetries : Boolean;
      procedure LoadTransmitData;
      procedure OpenFile;

    public

      {Constructors/destructors}
      function Init (Options : Cardinal) : Integer; override;
      procedure Done; override;

      function Reinit : Integer; override;
      procedure DonePart ; override;
      {Options}
      function SetKermitOptions (KOptions : TApxKermitOptions) : Integer;
      function SetMaxPacketLen (MaxLen : Byte) : Integer;
      function SetMaxLongPacketLen (MaxLen : Cardinal) : Integer;
      function SetMaxWindows (MaxNum : Byte): Integer;
      function SetSWCTurnDelay (TrnDelay : Cardinal) : Integer;
      function SetMaxTimeoutSecs (MaxTimeout : Byte) : Integer;
      function SetPacketPadding (C : Char; Count : Byte) : Integer;
      function SetTerminator (C : Char) : Integer;
      function SetCtlPrefix (C : Char) : Integer;
      function SetHibitPrefix (C : Char) : Integer;
      function SetRepeatPrefix (C : Char) : Integer;
      function SetKermitCheck (CType : Byte) : Integer;
      function GetSWCSize : Byte;
      function GetLPStatus (var InUse : Boolean;
                      var PacketSize : Cardinal) : Integer;
      function WindowsUsed : Byte;

      {Control}
      procedure PrepareReceive; override;
      procedure Receive(Msg, wParam : Cardinal;
                       lParam : LongInt); override;
      procedure PrepareTransmit; override;
      procedure Transmit(Msg, wParam : Cardinal;
                       lParam : LongInt); override;

      procedure FinishWriting; override;
      procedure Assign (Source : TPersistent); override;
                       
    published

      property SWCTurnDelay    : Cardinal
               read FSWCTurnDelay write FSWCTurnDelay;
      property KermitOptions   : TApxKermitOptions
               read FKermitOptions write FKermitOptions;

  end;




implementation

uses
  AxProtcl;

{$IFDEF TRIALRUN}
  {$I TRIAL07.INC}
  {$I TRIAL03.INC}
  {$I TRIAL01.INC}
{$ENDIF}

const
  {'S' - SendInit packet option index}
  MaxL    = 1;     {Max packet length sender can receive (Def = none)}
  Time    = 2;     {Max seconds to wait before timing out (Def = none)}
  NPad    = 3;     {Number of padding chars before packets (Def = none)}
  PadC    = 4;     {Padding character (Def = Nul)}
  EOL     = 5;     {Packet terminator character (Def = CR)}
  QCtl    = 6;     {Prefix char for control-char encoding (Def = #)}
  QBin    = 7;     {Prefix char for hi-bit encoding (Def = ' ' none)}
  Chkt    = 8;     {1=chksum, 2=2 byte chksum, 3=CRC (Def = 1)}
  Rept    = 9;     {Prefix char for repeat-count encoding (Def = ' ' none)}
  Capa    = 10;    {Advanced capabilities bit masks}
  Windo   = 11;    {Size of the sliding window (in packets)}
  MaxLx1  = 12;    {long packet size div 95}
  MaxLx2  = 13;    {Long packet size mod 95}
  SendInitLen = 13; {Size of SendInit data block}
  MaxKermitOption = 13;

  {Advanced capability bit masks}
  LastMask       = $01;  {Set if more bit masks follow}
  LongPackets    = $02;  {Set if using long packets}
  SlidingWindows = $04;  {Set if using sliding windows}
  FileAttribute  = $08;  {Set if using Attribut packets, not supported}

  {Text strings for various error/abort conditions}
  eRecInitTO = 'Timeout waiting for RecInit packet';
  eFileTO = 'Timeout waiting for File packet';
  eDataTO = 'Timeout waiting for Data packet';
  eSync = 'Failed to syncronize protocol';
  eAsync = 'Blockcheck or other error';
  eCancel = 'Canceled';
  eFileExists = 'Not allowed to overwrite existing file';
  eFileError = 'Error opening or writing file';

  {Check to aCheckType conversion array}
  CheckVal : array[1..3] of Byte = (bcChecksum1, bcChecksum2, bcCrcK);

  {Used in ProtocolReceivePart/ProtocolTransmitPart}
  FirstDataState : array[Boolean] of TApxKermitDataState = (dskData, dskCheck1); 
  FreeMargin = 20;

  aDataTrigger = 0;

  LogKermitState : array[TApxKermitState] of TDispatchSubType = (
    dsttkInit, dsttkInitReply, dsttkCollectInit, dsttkOpenFile,
    dsttkSendFile, dsttkFileReply, dsttkCollectFile, dsttkCheckTable,
    dsttkSendData, dsttkBlockReply, dsttkCollectBlock, dsttkSendEof,
    dsttkEofReply, dsttkCollectEof, dsttkSendBreak, dsttkBreakReply,
    dsttkCollectBreak, dsttkComplete, dsttkWaitCancel, dsttkError,
    dsttkDone, dstrkInit, dstrkGetInit, dstrkCollectInit,
    dstrkGetFile, dstrkCollectFile, dstrkGetData, dstrkCollectData,
    dstrkComplete, dstrkWaitCancel, dstrkError, dstrkDone);

function ToChar(C : Char) : Char;
{-Returns C+$20}
asm
  add al,$20;
end;

function UnChar(C : Char) : Char;
{-Returns C-$20}
asm
  sub al,$20
end;

function Ctl(C : Char) : Char;
{-Returns C xor $40}
asm
  xor al,$40
end;

function Inc64(W : Cardinal) : Cardinal;
{-Returns (W+1) mod 64}
asm
  inc ax
  and ax,$3F
end;

function Dec64(W : Cardinal) : Cardinal;
{-Returns (W-1) or 63 if W=0}
asm
  dec ax
  jns @@done
  mov ax,63
  @@done:
end;

function IsCtl(C : Char) : Boolean;
begin
  IsCtl := (C <= #31) or (C = #127);
end;

function IsHiBit(C : Char) : Boolean;
begin
  IsHiBit := (Ord(C) and $80) <> 0;
end;

function HiBit(C : Char) : Char;
asm
  or ax,$80
end;

procedure TApxKermitDriver.FinishWriting;
{-Handle "discard" option}
begin
  if FileOpen then begin
    {Let parent close file}
    aapFinishWriting;

    {Discard the file if asked to do so}
    if (FActualDataLen >= 1) and
       (DataBlock^[1] = ApxKermitDiscardChar) then begin
      EraseWorkFile;
      if IOResult = 0 then ;
    end;
  end;
end;

procedure TApxKermitDriver.AllocateWindowTable;
{-Allocate the window table}
begin
  {Allocate sliding window data table}
  FDataTable := AllocMem (FTableSize * BlockLen);
end;

procedure TApxKermitDriver.DeallocateWindowTable;
{-Deallocate current window table}
begin
  FreeMem (FDataTable, FTableSize * BlockLen);
end;

procedure TApxKermitDriver.RawInit;
{-Do low-level initializations}
{$IFDEF TRIALRUN}
  {$I TRIAL04.INC}
{$ENDIF}
begin
{$IFDEF TRIALRUN}
  TC;
{$ENDIF}
  CurProtocol := Kermit;
  FileOfs := 0;
  BlockLen := ApxDefKermitOptions.MaxPacketLen;
  FileOpen := False;
  FUsingHibit := False;
  FUsingRepeat := False;
  FKermitOptions := ApxDefKermitOptions;
  FPacketType := ' ';
  FMinRepeatCnt := ApxKermitDefMinRepeatCnt;
  BatchProtocol := True;
  FLPInUse := False;
end;

function TApxKermitDriver.Init(Options : Cardinal) : Integer;
{-Allocates and initializes a protocol control block with options}
begin
 {Check for adequate output buffer size}
 if ComPort.OutBuffUsed + ComPort.OutBuffFree < 1024 then begin
   Init := ecOutputBufferTooSmall;
   Exit;
 end;
 {Allocate the protocol data record}
 if apInitProtocolData(ComPort, Options) <> 0 then begin
   Init := ecOutOfMemory;
   Exit;
 end;

  DataBlock := nil;
  FWorkBlock := nil;
  FDataTable := nil;

  RawInit;

  Overhead := ApxKermitOverhead;
  TurnDelay := ApxKermitTurnDelay;
  FSWCTurnDelay := ApxSWCKermitTurnDelay;

  FKermitOptions := ApxDefKermitOptions;
  with FKermitOptions do begin
    if MaxLongPacketLen = 0 then
      BlockLen := MaxPacketLen
    else
      BlockLen := MaxLongPacketLen;
    if WindowSize = 0 then
      FTableSize := 1
    else
      FTableSize := WindowSize;
    CheckType := CheckVal[Byte(Check)-$30];
  end;

  {Allocate data and work blocks}
  DataBlock := AllocMem (SizeOf (TApxDataBlock));
  FWorkBlock := AllocMem (SizeOf (TApxDataBlock));

  {Allocate table for data blocks}
  AllocateWindowTable;

  {All okay}
  Init := ecOK;
end;

function TApxKermitDriver.Reinit : Integer;
{-Allocates and initializes a protocol control block with options}
begin
  DataBlock := nil;
  FWorkBlock := nil;
  FDataTable := nil;

  RawInit;

  FKermitOptions := ApxDefKermitOptions;
  with FKermitOptions do begin
    if MaxLongPacketLen = 0 then
      BlockLen := MaxPacketLen
    else
      BlockLen := MaxLongPacketLen;
    if WindowSize = 0 then
      FTableSize := 1
    else
      FTableSize := WindowSize;
    CheckType := CheckVal[Byte(Check)-$30];
  end;

  {Allocate data and work blocks}
  DataBlock := AllocMem (SizeOf (TApxDataBlock));
  FWorkBlock := AllocMem (SizeOf (TApxDataBlock));

  {Allocate table for data blocks}
  AllocateWindowTable;

  {Allocate internal buffer }
  FInBuff := AllocMem (SizeOf (TApxInBuffer));
  FInBuffHead := 1;
  FInBuffTail := 1;

  {All okay}
  Reinit := ecOK;
end;

procedure TApxKermitDriver.DonePart;
{-Disposes of Kermit protocol record}
begin
  DeallocateWindowTable;
  FreeMem (DataBlock, SizeOf (TApxDataBlock));
  FreeMem (FWorkBlock, SizeOf (TApxDataBlock));
  if FInBuff <> nil then begin
    FreeMem (FInBuff, SizeOf(TApxInBuffer));
    FInBuff := nil;
  end;
end;

procedure TApxKermitDriver.Done;
{-Disposes of Kermit protocol record}
begin
  DonePart;
  apDoneProtocol;
end;

function TApxKermitDriver.SetKermitOptions(KOptions : TApxKermitOptions) : Integer;
{-Update the KermitProtocol object to use KOptions}
begin
  if CurProtocol <> Kermit then begin
    SetKermitOptions := ecBadProtocolFunction;
    Exit;
  end;

  FKermitOptions := KOptions;
  CheckType := CheckVal[Byte (FKermitOptions.Check) - $30];
  SetKermitOptions := ecOk;
  {Check for errors}
end;

function TApxKermitDriver.SetMaxPacketLen (MaxLen : Byte) : Integer;
{-Set the maximum packet length}
begin
  if CurProtocol <> Kermit then begin
    SetMaxPacketLen := ecBadProtocolFunction;
    Exit;
  end;

  if MaxLen > 94 then
    SetMaxPacketLen := ecBadArgument
  else begin
    SetMaxPacketLen := ecOk;
    FKermitOptions.MaxPacketLen := MaxLen;
  end;
end;

function TApxKermitDriver.SetMaxLongPacketLen (MaxLen : Cardinal) : Integer;
{-Set the maximum packet length}
var
  TempFlags : Cardinal;
begin
  if CurProtocol <> Kermit then begin
    SetMaxLongPacketLen := ecBadProtocolFunction;
    Exit;
  end;

  if MaxLen > 1024 then begin
    SetMaxLongPacketLen := ecBadArgument;
    Exit;
  end;

  {Assume success}
  SetMaxLongPacketLen := ecOK;

  {Deallocate current table}
  DeallocateWindowTable;

  if MaxLen > 0 then begin
    TempFlags := Flags;
    SetFlag (TempFlags, apKermitLongPackets);
    Flags := TempFlags;
    with FKermitOptions do begin
      CapabilitiesMask := CapabilitiesMask or LongPackets;
      MaxLongPacketLen := MaxLen;
      BlockLen := MaxLen;
      if FKermitOptions.Check = '1' then
        FKermitOptions.Check := '2';
    end;
  end else begin
    TempFlags := Flags;
    ClearFlag (TempFlags, apKermitLongPackets);
    Flags := TempFlags;
    with FKermitOptions do begin
      CapabilitiesMask := CapabilitiesMask and not LongPackets;
      MaxLongPacketLen := 0;
    end;
    BlockLen := 80;
  end;

  {Reallocate table}
  AllocateWindowTable;
end;

function TApxKermitDriver.SetMaxWindows (MaxNum : Byte) : Integer;
{-Set the number of windows for SWC}
var
  TempFlags : Cardinal;
begin
  if CurProtocol <> Kermit then begin
    SetMaxWindows := ecBadProtocolFunction;
    Exit;
  end;

  if MaxNum > ApxKermitMaxWindowSlots then begin
    SetMaxWindows := ecBadArgument;
    Exit;
  end;

  {Assume success}
  SetMaxWindows := ecOK;

  {Deallocate current table}
  DeallocateWindowTable;

  if MaxNum > 0 then begin
    TempFlags := Flags;
    SetFlag (TempFlags, apKermitSWC);
    Flags := TempFlags;
    with FKermitOptions do begin
      CapabilitiesMask := CapabilitiesMask or SlidingWindows;
      WindowSize := MaxNum and $1F;
      FTableSize := WindowSize;
    end;
  end else begin
    TempFlags := Flags;
    ClearFlag (TempFlags, apKermitSWC);
    Flags := TempFlags;
    with FKermitOptions do begin
      CapabilitiesMask := CapabilitiesMask and not SlidingWindows;
      WindowSize := 0;
    end;
    FTableSize := 1;
  end;

  {Reallocate current table}
  AllocateWindowTable;
end;

function TApxKermitDriver.SetSWCTurnDelay(TrnDelay : Cardinal) : Integer;
begin
  if CurProtocol <> Kermit then
    SetSWCTurnDelay := ecBadProtocolFunction
  else begin
    SetSWCTurnDelay := ecOK;
    FSWCTurnDelay := TrnDelay;
  end;
end;

function TApxKermitDriver.GetSWCSize : Byte;
{-Return size of current window (0 if not in use)}
begin
  if CurProtocol <> Kermit then
    GetSWCSize := 0
  else
    GetSWCSize := FKermitOptions.WindowSize;
end;

function TApxKermitDriver.GetLPStatus (var InUse      : Boolean;
                                       var PacketSize : Cardinal) : Integer;
{-Return status of long packet feature}
begin
  if CurProtocol <> Kermit then
    Result := ecBadProtocolFunction
  else begin
    Result := ecOK;
    InUse := FLPInUse;

    if FLPInUse then
      PacketSize := FKermitOptions.MaxLongPacketLen
    else
      PacketSize := 0;
  end;
end;

function TApxKermitDriver.SetMaxTimeoutSecs(MaxTimeout : Byte) : Integer;
{-Set the maximum time to wait for a packet}
begin
  if CurProtocol <> Kermit then
    SetMaxTimeoutSecs := ecBadProtocolFunction
  else begin
    SetMaxTimeoutSecs := ecOK;
    FKermitOptions.MaxTimeout := MaxTimeout;
  end;
end;

function TApxKermitDriver.SetPacketPadding (C : Char;
                                            Count : Byte) : Integer;
{-Set the pad character and count}
begin
  with FKermitOptions do begin
    if CurProtocol <> Kermit then
      SetPacketPadding := ecBadProtocolFunction
    else begin
      SetPacketPadding := ecOK;
      PadChar := C;
      PadCount := Count;
    end;
  end;
end;

function TApxKermitDriver.SetTerminator (C : Char) : Integer;
{-Set the packet terminator}
begin
  if CurProtocol <> Kermit then
    SetTerminator := ecBadProtocolFunction
  else begin
    SetTerminator := ecOK;
    FKermitOptions.Terminator := C;
  end;
end;

function TApxKermitDriver.SetCtlPrefix (C : Char) : Integer;
{-Set the control character quote prefix}
begin
  if CurProtocol <> Kermit then
    SetCtlPrefix := ecBadProtocolFunction
  else begin
    SetCtlPrefix := ecOK;
    FKermitOptions.CtlPrefix := C;
  end;
end;

function TApxKermitDriver.SetHibitPrefix (C : Char) : Integer;
{-Set the hibit quote prefix}
begin
  if CurProtocol <> Kermit then
    SetHibitPrefix := ecBadProtocolFunction
  else begin
    SetHibitPrefix := ecOK;
    FKermitOptions.HibitPrefix := C;
  end;
end;

function TApxKermitDriver.SetRepeatPrefix(C : Char) : Integer;
{-Set the repeat quote prefix}
begin
  if CurProtocol <> Kermit then
    SetRepeatPrefix := ecBadProtocolFunction
  else begin
    SetRepeatPrefix := ecOK;
    FKermitOptions.RepeatPrefix := C;
  end;
end;

function TApxKermitDriver.SetKermitCheck (CType : Byte) : Integer;
{-Set the block check type (bcCheckSum1 (default), bcCheckSum2, bcCrcK)}
begin
  if CurProtocol <> Kermit then begin
    SetKermitCheck := ecBadProtocolFunction;
    Exit;
  end;

  SetKermitCheck := ecOk;
  with FKermitOptions do begin
    case CType of
      bcCheckSum1 : Check := '1';
      bcCheckSum2 : Check := '2';
      bcCrcK      : Check := '3';
      else
        begin
          SetKermitCheck := ecBadArgument;
          Check := '1';
        end;
    end;
  end;
  CheckType := CheckVal[Byte (FKermitOptions.Check) - $30];
end;

{ Buffer management methods }
function TApxKermitDriver.CharReady : Boolean;
begin
  Result := FInBuffHead < FInBuffTail;
end;

function TApxKermitDriver.GetChar : Char;
begin
  inc(FInBuffHead);
  Result := FInBuff^[FInBuffHead];
  if FInBuffHead >= FInBuffTail then begin
    FInBuffHead := 1;
    FInBuffTail := 1;
  end;
end;

procedure TApxKermitDriver.CompactInBuff;
var
  TempBuffer : PApxInBuffer;
begin
  TempBuffer := AllocMem (SizeOf (TApxInBuffer));
  FillChar (TempBuffer^, SizeOf (TApxInBuffer), #0);
  Move (FInBuff^[FInBuffHead], TempBuffer^[1],
        FInBuffTail - FInBuffHead);
  Move (TempBuffer^[1], FInBuff^[1], SizeOf(TApxInBuffer));
  FInBuffTail := FInBuffTail - FInBuffHead + 1;
  FInBuffHead := 1;
  FreeMem (TempBuffer, SizeOf(TApxInBuffer));
end;

procedure TApxKermitDriver.FillInBuff;
begin
  while ComPort.ValidDispatcher.CharReady do begin
    inc(FInBuffTail);
    FInBuff^[FInBuffTail] := ComPort.GetChar;    
    if FInBuffHead > (SizeOf (FInBuff^) div 2) then
      CompactInBuff;
  end;
end;

procedure TApxKermitDriver.FlushInBuffer;
begin
  ComPort.FlushInBuffer;
  FInBuffHead := 1;
  FInBuffTail := 1;
end;

procedure TApxKermitDriver.UpdateBlockCheck (CurByte : Byte);
{-Updates the block check character (whatever it is)}
begin
  {Do checksums if requested or check type not known}
  BlockCheck := apUpdateCheckSum(CurByte, BlockCheck);

  {Do crc if requested or check type not known}
  FBlockCheck2 := apUpdateCrcKermit(CurByte, FBlockCheck2);
end;

procedure TApxKermitDriver.SendBlockCheck;
{-Makes final adjustment and sends the aBlockCheck character}
var
  Check : Cardinal;
  C : Char;
begin
  if FCheckKnown then
    FTempCheck := FKermitOptions.Check
  else
    FTempCheck := '1';

  case FTempCheck of
    '1' : {Standard 1 byte checksum}
      begin
        {Add bits 6,7 into 0-5}
        Check := Lo (BlockCheck);
        C := ToChar (Char ((Check + (Check shr 6)) and $3F));
        ComPort.PutChar (C);
      end;
    '2' : {2 byte checksum}
      begin
        {1st byte has bits 11-6, second has bits 5-0}
        Check := BlockCheck;
        C := ToChar (Char ((Check shr 6) and $3F));
        ComPort.PutChar (C);
        C := ToChar (Char (Check and $3F));
        ComPort.PutChar (C);
      end;
    '3' : {2 byte CRC}
      begin
        Check := FBlockCheck2;
        C := ToChar (Char ((Check shr 12) and $0F));
        ComPort.PutChar (C);
        C := ToChar (Char ((Check shr 6) and $3F));
        ComPort.PutChar (C);;
        C := ToChar (Char (Check and $3F));
        ComPort.PutChar (C);
      end;
  end;
end;

procedure TApxKermitDriver.PutToChar(C : Char);
{-Put a promoted character}
begin
  ComPort.PutChar (ToChar (C));
end;

procedure TApxKermitDriver.PutHeader (HType : Char; Len : Cardinal);
{-Start a header}
var
  I : Byte;
begin
  {Init the block check character}
  BlockCheck := 0;
  FBlockCheck2 := 0;

  {Send the Mark, Len, Seq and Type fields}
  ComPort.PutChar (cSoh);
  if Len <= 94 then begin
    PutToChar (Char (Len));
    PutToChar (Char (BlockNum));
    ComPort.PutChar(HType);
    UpdateBlockCheck (Byte (ToChar (Char (Len))));
    UpdateBlockCheck (Byte (ToChar (Char (BlockNum))));
    UpdateBlockCheck (Byte (HType));
  end else begin
    {Adjust Len to long packet specification}
    Dec (Len, 2);

    {Send Len, Seq and Type fields}
    PutToChar (#0);
    PutToChar (Char (BlockNum));
    ComPort.PutChar (HType);

    {Update header check}
    I := 32;
    Inc(I, Ord( ToChar( Char (BlockNum))));
    Inc(I, Ord( HType));

    {Send Lenx1 and Lenx2, update header checksum}
    PutToChar (Char (Len div 95));
    Inc (I, Ord (ToChar (Char (Len div 95))));
    PutToChar (Char (Len mod 95));
    Inc(I, Ord (ToChar (Char (Len mod 95))));
    I := (I + (I shr 6)) and $3F;

    {Send the header checksum}
    PutToChar(Char (I));

    {Update regular block check}
    UpdateBlockCheck (Byte (ToChar (#0)));
    UpdateBlockCheck (Byte (ToChar (Char (BlockNum))));
    UpdateBlockCheck (Byte (HType));
    UpdateBlockCheck (Byte (ToChar (Char (Len div 95))));
    UpdateBlockCheck (Byte (ToChar (Char (Len mod 95))));
    UpdateBlockCheck (Byte (ToChar (Char (I))));
  end;

  {Note what block number needs an Ack}
  FExpectedAck := BlockNum;
end;

procedure TApxKermitDriver.TransmitBlock(var Block : TApxDataBlock;
                          BLen : Cardinal;
                          BType : Char);
{-Transmits one data subpacket from Block}
var
    I : Cardinal;
begin
  if BLen = 0 then
    Exit;

  {Send the data field}
  ComPort.PutBlock(Block, BLen);
  for I := 1 to BLen do
    UpdateBlockCheck (Byte (Block [I]));
end;

procedure TApxKermitDriver.SendTerminator;
{-Send the terminator and padding chars}
begin
  ComPort.PutChar (FKermitOptions.Terminator);
end;

procedure TApxKermitDriver.SendPacket (PT : Char);
{-Send a packet of type PT}
const
    CheckLen : array[1..3] of Byte = (3, 4, 5);
var
    TotalLen : Cardinal;
    I : Byte;
begin
  {Put required padding}
  with FKermitOptions do
    for I := 1 to PadCount do
      ComPort.PutChar (PadChar);

  {Calc total length}
  TotalLen := FDataLen + CheckLen[(Byte (FKermitOptions.Check) - $30)];

  {Send the header...}
  PutHeader (PT, TotalLen);

  {Send the data field}
  TransmitBlock (DataBlock^, FDataLen, PT);

  {Finish up}
  SendBlockCheck;
  SendTerminator;
end;

procedure TApxKermitDriver.SendError (Msg : String);
{-Send error packet}
begin
  BlockNum := Inc64 (BlockNum);
  FDataLen := Length (Msg);
  Move (Msg[1], DataBlock^[1], FDataLen);
  SendPacket (KError);
end;

procedure TApxKermitDriver.Cancel;
{-Sends the cancel string}
const
  AckLen : array[1..3] of Byte = (3, 4, 5);
var
  B : Byte;
begin
  if ApxKermitFastAbort then
    {Abort by sending error packet (old method)}
    SendError(eCancel)

  else if FReceiveInProgress then begin
    {Abort by sending 'Z' in data field of Ack packet (new method)}
    B := AckLen[Byte (FKermitOptions.Check) - $30];
    DataBlock^[1] := 'Z';
    PutHeader(KAck, B+1);
    TransmitBlock(DataBlock^, 1, KAck);
    SendBlockCheck;
    SendTerminator;

  end else begin
    {Abort by sending EOF packet with 'D' in data field (new method)}
    FDataLen := 1;
    DataBlock^[1] := ApxKermitDiscardChar;
    BlockNum := Inc64 (BlockNum);
    SendPacket (KEndOfFile);
  end;

  {Show cancel to status}
  ProtocolStatus := psCancelRequested;
end;

procedure TApxKermitDriver.ResetStatus;
{-Typical reset but aBlockNum must _not_ be reset during protocol}
begin
  if InProgress = 0 then begin
    {New protocol, reset status vars}
    BytesRemaining := 0;
    BlockNum := 0;
  end;
  ProtocolError := ecOK;
  ProtocolStatus := psOK;
  SrcFileLen := 0;
  BytesTransferred := 0;
  ElapsedXfrTime := 0;
  BlockErrors := 0;
  TotalErrors := 0;
end;

procedure TApxKermitDriver.GetDataChar(var C : Char;
                          var TableIndex : Cardinal;
                          var RepeatCnt : Cardinal);
{-Get C from kDataTable handling all prefixing}
var
  Finished : Boolean;
  CtlChar : Boolean;
  HibitChar : Boolean;
  Repeating : Boolean;
begin
  Finished := False;
  CtlChar := False;
  HibitChar := False;
  Repeating := False;
  RepeatCnt := 1;

  with FKermitOptions do
    repeat
      C := FDataTable^[TableIndex];
      Inc (TableIndex);

      {Set flags according to the char received}
      if (C = HibitPrefix) and (FUsingHibit) and (not HibitChar) then begin
        if (CtlChar) then
          Exit;
        HibitChar := True;
      end else if C = CtlPrefix then begin
        if CtlChar then begin
          if HibitChar then
            C := Chr(Byte(C) or $80);
          Exit;
        end else
          {Note that the next char is Ctl escaped}
          CtlChar := True;
      end else if (C = RepeatPrefix) and (FUsingRepeat and not Repeating) then begin
        if CtlChar then begin
          {process as ctl char}
          if HibitChar then
            C := Chr(Byte(C) or $80);
          Exit;
        end else begin
          {Repeat flag set, get the count}
          C := FDataTable^[TableIndex];
          Inc(TableIndex);
          Repeating := True;
          RepeatCnt := Byte(UnChar(C));
        end;
      end else begin
        {Normal character}
        Finished := True;

        if (HibitChar and FUsingHibit) then
          C := Char(Byte(C) or $80);

        if CtlChar then
          {Don't escape normal or hibit Prefix characters}
          if (C = Char(Byte(CtlPrefix) or $80)) or
                 (FUsingRepeat and (C = Char (Byte (RepeatPrefix) or $80))) or
                 (FUsingHibit and (C = Char (Byte (HibitPrefix) or $80))) or
                 (C = RepeatPrefix) then
            {do nothing}
          else
            {Ok to Ctl it}
            C := Ctl(C);
      end;
    until Finished;
end;

procedure TApxKermitDriver.CheckForHeader;
{-Checks for a header}
const
  CheckLen : array[1..3] of Byte = (3, 4, 5);
var
  C : Char;
begin
  {Assume no header ready}
  ProtocolStatus := psNoHeader;

  {If continuing a previous header we need to restore aBlockCheck}
  if FKermitHeaderState <> hskNone then begin
    BlockCheck := FSaveCheck;
    FBlockCheck2 := FSaveCheck2;
  end;

  {Process potential header characters}
  while CharReady and (FKermitHeaderState <> hskDone) do begin
    C := GetChar;

    if C = cSoh then
      FKermitHeaderState := hskNone;

    case FKermitHeaderState of
      hskNone :
        if C = cSoh then begin
          FKermitHeaderState := hskGotMark;
          BlockCheck := 0;
          FBlockCheck2 := 0;
          FLongCheck := 32;
        end;
      hskGotMark :
        begin
          FKermitHeaderState := hskGotLen;
          UpdateBlockCheck(Byte(C));
          C := UnChar(C);
          FGetLong := (C = #0);
          FRecDataLen := Ord(C);
        end;
      hskGotLen :
        begin
          FKermitHeaderState := hskGotSeq;
          UpdateBlockCheck(Byte(C));
          Inc(FLongCheck, Byte(C));
          C := UnChar(C);
          FRecBlockNum := Ord(C);
        end;
      hskGotSeq :
        begin
          FPacketType := C;
          UpdateBlockCheck(Byte(C));
          Inc (FLongCheck, Byte(C));
          if FGetLong then
            FKermitHeaderState := hskGotType
          else
            FKermitHeaderState := hskDone;
        end;
      hskGotType :
        begin
          FKermitHeaderState := hskGotLong1;
          UpdateBlockCheck(Byte(C));
          Inc(FLongCheck, Byte(C));
          C := UnChar(C);
          FRecDataLen := Cardinal(C)*95;
        end;
      hskGotLong1 :
        begin
          FKermitHeaderState := hskGotLong2;
          UpdateBlockCheck(Byte(C));
          Inc(FLongCheck, Byte(C));
          C := UnChar(C);
          Inc(FRecDataLen, Byte(C));
        end;
      hskGotLong2 :
        begin
          FKermitHeaderState := hskDone;
          FLongCheck := (FLongCheck + (FLongCheck shr 6)) and $3F;
          UpdateBlockCheck(Byte(C));
          C := UnChar(C);
          if C = Char(FLongCheck) then
            ProtocolStatus := psBlockCheckError;
          Inc (FRecDataLen, 2);
        end;
    end;
  end;

  if FKermitHeaderState = hskDone then begin
    {Say we got a header}
    ProtocolStatus := psGotHeader;

    {Account for other extra bytes in length}
    if FCheckKnown then
      Dec(FRecDataLen, (CheckLen[Byte(FKermitOptions.Check)-$30]))
    else
      Dec(FRecDataLen, (CheckLen[1]));
    if Integer(FRecDataLen) < 0 then
      FRecDataLen := 0;
  end else begin
    {Say no header ready}
    ProtocolStatus := psNoHeader;
    FSaveCheck := BlockCheck;
    FSaveCheck2 := FBlockCheck2;
  end;
end;

function TApxKermitDriver.NextSeq(I : Integer) : Integer;
{-Increment I to next slot, accounting for current table size}
begin
  Inc(I);
  if I > Integer(FTableSize) then
    I := 1;
  NextSeq := I;
end;

function TApxKermitDriver.PrevSeq (I : Integer) : Integer;
{-Decrement I to previous slot, accounting for current table size}
begin
  Dec(I);
  if I = 0 then
    I := FTableSize;
  PrevSeq := I;
end;

function TApxKermitDriver.TableFull : Boolean;
{-Returns True if the send table is full}
begin
  TableFull := FInfoTable[NextSeq (FTableHead)].InUse;
end;

function TApxKermitDriver.PacketsOutstanding : Boolean;
{-True if there are unacked packets in the table}
var
  I : Integer;
begin
  PacketsOutstanding := True;
  for I := 1 to FTableSize do
    if FInfoTable[I].InUse then
      Exit;
  PacketsOutstanding := False;
end;

function TApxKermitDriver.GetOldestSequence : Integer;
var
  I, Oldest : Integer;
begin
  Oldest := MaxInt;
  for I := 1 to FTableSize do begin
    if FInfoTable[I].InUse and (FInfoTable[I].Seq < Oldest) then
      Oldest := I;
  end;
  if Oldest = MaxInt then
    Result := -1
  else
    Result := FInfoTable[Oldest].Seq;
end;

function TApxKermitDriver.SeqInTable (CurSeq : Integer) : Integer;
{-Return the position in the table of CurSeq, or -1 of not found}
var
  I : Integer;
begin
  SeqInTable := -1;
  for I := 1 to FTableSize do
    if FInfoTable[I].Seq = CurSeq then begin
      SeqInTable := I;
      Exit;
    end;
end;

procedure TApxKermitDriver.GotAck (CurSeq : Cardinal);
{-Note ACK for block number CurSeq}
var
  I : Integer;
begin
  I := SeqInTable(CurSeq);
  if I <> - 1 then
    FInfoTable[I].InUse := False;
end;

function TApxKermitDriver.WindowsUsed : Byte;
{-Return number of window slots in use}
var
  I : Integer;
  Cnt : Cardinal;
begin
  if not PacketsOutstanding then begin
    WindowsUsed := 0;
    Exit;
  end;

  Cnt := 0;
  for I := 1 to FTableSize do
    if FInfoTable[I].InUse then
      Inc(Cnt);

  WindowsUsed := Cnt;
end;

procedure TApxKermitDriver.WritePacket (Index : Byte);
{-Expand and write the packet from table slot Index}
var
  TIndex : Cardinal;
  WIndex : Cardinal;
  LastIndex : Cardinal;
  RepeatCnt : Cardinal;
  Free : Cardinal;
  Left : Cardinal;
  C : Char;
  Failed : Boolean;

  procedure WriteBlock;
  begin
    Failed := WriteProtocolBlock (FWorkBlock^, SizeOf(FWorkBlock^));
    FileOfs := FileOfs + SizeOf (FWorkBlock^);
    WIndex := 1;
    Free := SizeOf (FWorkBlock^);
  end;

begin
  {Set starting indexes}
  TIndex := Cardinal(Index - 1) * BlockLen;
  LastIndex := Integer(TIndex) + FInfoTable[Index].Len;
  WIndex := 1;

  {Loop through this block in kDataTable...}
  Failed := False;
  repeat
    {Get a character with escaping already translated}
    GetDataChar(C, TIndex, RepeatCnt);

    if RepeatCnt = 1 then begin
      {Single char, just add it to WorkBlock}
      FWorkBlock^[WIndex] := C;
      Inc(WIndex);
    end else begin
      {Repeating char, start filling aDataBlock(s)}
      Free := SizeOf(FWorkBlock^) - (WIndex - 1);
      Left := RepeatCnt;
      repeat
        if Free >= Left then begin
          FillChar (FWorkBlock^[WIndex], Left, C);
          Inc (WIndex, Left);
          Left := 0;
        end else begin
          FillChar(FWorkBlock^[WIndex], Free, C);
          Inc (WIndex, Free);
          Dec (Left, Free);
        end;

        {Flush WorkBlock if it fills}
        if WIndex = SizeOf (FWorkBlock^)+1 then
          WriteBlock;
      until (Left = 0) or Failed;
    end;

    {Flush WorkBlock if it fills}
    if WIndex = SizeOf (FWorkBlock^)+1 then
      WriteBlock;

  until (TIndex = LastIndex) or Failed;

  {Commit last, or only, block}
  if WIndex <> 1 then begin
    Failed := WriteProtocolBlock(FWorkBlock^, WIndex-1);
    FileOfs := FileOfs + LongInt (WIndex) - 1;
  end;
end;

function TApxKermitDriver.SeqGreater (Seq1, Seq2 : Byte) : Boolean;
{-Return True if Seq is greater than Seq2, accounting for wrap at 64}
var
  I : Integer;
begin
  I := Seq1 - Seq2;
  if I > 0 then
    SeqGreater := (I < 32)
  else
    SeqGreater := (Abs(I) > 32);
end;

function TApxKermitDriver.LoSeq : Byte;
{-Return sequence number of oldest possible sequence number}
{-Current Seq - (kTableSize)}
begin
  {Handle case of no windows}
  if FTableSize = 1 then begin
    LoSeq := FRecBlockNum;
    Exit;
  end;

  LoSeq := FInfoTable[FTableTail].Seq;
end;

function TApxKermitDriver.HiSeq : Byte;
    {-Return sequence number of highest acceptable sequence number}
var
    I     : Byte;
    Count : Byte;
begin
  {Handle case of no windows}
  if FTableSize = 1 then begin
    HiSeq := FRecBlockNum;
    Exit;
  end;

  {Search backwards counting free (acked) slots}
  I := PrevSeq (FTableHead);
  Count := 0;
  repeat
    with FInfoTable[I] do
      if Acked or not InUse then
        Inc (Count);
    I := PrevSeq(I);
  until (I = FTableHead);

  {HiSeq is current sequence number + Count}
  Inc (Count, FRecBlockNum);
  if Count > 64 then
    Dec (Count, 64);
  HiSeq := Count;
end;

function TApxKermitDriver.SeqDiff (Seq1, Seq2 : Byte) : Byte;
{-Assuming Seq1 > Seq2, return the difference}
begin
  if Seq1 > Seq2 then
    SeqDiff := Seq1 - Seq2
  else
    SeqDiff := (Seq1 + 64) - Seq2;
end;

procedure TApxKermitDriver.AddToTable (Seq : Byte);
{-Add Seq to proper location in table}
var
  CurSeq : Byte;
  HeadSeq : Byte;
  I : Cardinal;
  Diff : Cardinal;
begin
  {Calculate kTableHead value for Seq (range known to be OK)}
  HeadSeq := FInfoTable[FTableHead].Seq;

  if SeqGreater (Seq, HeadSeq) then begin
    {Incoming packet is new, rotate table, writing old slots as required}
    Diff := SeqDiff (Seq, HeadSeq);
    for I := 1 to Diff do begin
      FTableHead := NextSeq (FTableHead);
      if FTableHead = FTableTail then begin
        if FInfoTable[FTableTail].InUse then begin
          WritePacket(FTableTail);
          FInfoTable[FTableTail].InUse := False;
          FInfoTable[FTableTail].Acked := False;
        end;
        FTableTail := NextSeq (FTableTail);
      end;
    end;
    I := FTableHead;

  end else begin
    {Incoming packet is a retransmitted packet, find associated table index}
    CurSeq := HeadSeq;
    I := FTableHead;
    while CurSeq <> Seq do begin
      CurSeq := Dec64 (CurSeq);
      I := PrevSeq (I);
    end;
  end;

  {Stuff info table}
  FInfoTable[I].Seq   := Seq;
  FInfoTable[I].Acked := True;
  FInfoTable[I].Len   := FRecDataLen;
  FInfoTable[I].InUse := True;

  {Stuff data table}
  Move (DataBlock^, FDataTable^[(I-1)*BlockLen], FRecDataLen);
end;

procedure TApxKermitDriver.SendNak;
{-Send an nak packet for packet Seq}
const
  NakLen = 3;
begin
  PutHeader(KNak, NakLen);

  {Put checksum}
  SendBlockCheck;

  {Put terminator}
  SendTerminator;
end;

procedure TApxKermitDriver.SendAck (Seq : Byte);
{-Send an acknowledge packet for packet Seq}
const
  AckLen : array[1..3] of Byte = (3, 4, 5);
var
  B : Byte;
  Save : Byte;
begin
  B := AckLen[Byte (FKermitOptions.Check) - $30];

  {kpPutHeader uses aBlockNum so we'll need to change it temporarily}
  Save := BlockNum;
  BlockNum := Seq;

  PutHeader (KAck, B);

  {Put checksum}
  SendBlockCheck;

  {Put terminator}
  SendTerminator;

  BlockNum := Save;
end;

function TApxKermitDriver.DataCount (Index : Byte) : Cardinal;
{-Count actual data characters in slot Index}
var
  TIndex : Cardinal;
  DIndex : Cardinal;
  LastIndex : Cardinal;
  RepeatCnt : Cardinal;
  C : Char;
begin
  {Set starting indexes}
  TIndex := Cardinal (Index - 1) * BlockLen;
  LastIndex := Integer (TIndex) + FInfoTable[Index].Len;
  DIndex := 1;

  {Loop through this block in kDataTable...}
  repeat
    {Get a character with escaping already translated}
    GetDataChar (C, TIndex, RepeatCnt);
    Inc (DIndex, RepeatCnt);
  until (TIndex = LastIndex);

  {Commit last, or only, block}
  DataCount := DIndex-1;
end;

procedure TApxKermitDriver.ProcessDataPacket;
{-Process received data packet}
var
  I : Cardinal;
  Count : Cardinal;
begin
  ProtocolError := ecOK;

  if (SeqGreater (FRecBlockNum, LoSeq) or (FRecBlockNum = LoSeq)) and
     (SeqGreater (HiSeq, FRecBlockNum) or (FRecBlockNum = HiSeq)) then begin

    {Acceptable data packet}
    AddToTable (FRecBlockNum);

    {Exit on errors, will be handled by state machine}
    if ProtocolError <> ecOK then
      Exit;

    {Nak missing packets}
    if SeqGreater (FRecBlockNum, BlockNum) then begin
      I := BlockNum;
      repeat
        SendNak;
        I := Inc64(I);
      until I = FRecBlockNum;
    end else if FRecBlockNum = BlockNum then begin
      {Adjust status variables}
      Count := DataCount (FTableHead);
      BytesTransferred := BytesTransferred + LongInt (Count);
      BytesRemaining := BytesRemaining - LongInt (Count);
      ElapsedXfrTime := ElapsedTime (Timer);
    end;

    {Ack the packet we got}
    SendAck (FRecBlockNum);

    {Expect next highest sequence beyond highest table entry}
    BlockNum := Inc64 (FInfoTable[FTableHead].Seq);

  end else begin
    {Unacceptable block number, ignore it}
    BlockNum := BlockNum; 
  end;
end;

function TApxKermitDriver.IncTableIndex (Index, Increment : Byte) : Byte;
{-Increment table index, wrap at table size}
begin
  Inc (Index, Increment);
  if Index > FTableSize then
    Dec (Index, FTableSize);
  IncTableIndex := Index;
end;

procedure TApxKermitDriver.FlushTableToDisk;
{-Write all outstanding packets to disk}
var
  Last, I : Cardinal;
begin
  Last := IncTableIndex (FTableHead, 1);
  I := Last;
  repeat
    with FInfoTable[I] do begin
      if InUse then
        if Acked then
          WritePacket (I)
      else begin
        apProtocolError (ecTableFull);
        Exit;
      end;
    end;
    I := IncTableIndex (I, 1);
  until (I = Last);
end;


procedure TApxKermitDriver.ReceiveBlock;
{-Get the datafield of a Kermit packet}
var
  C : Char;
  Check1 : Cardinal;
  Check2 : Cardinal;
  Check3 : Cardinal;
label
    ExitPoint;
begin
  {Get the data block}
  if FRecDataLen > 1024 then
    FRecDataLen := 1024;
  FActualDataLen := FRecDataLen;

  {If continuing a previous block we need to restore aBlockCheck}
  if FBlockIndex <> 1 then begin
    BlockCheck := FSaveCheck;
    FBlockCheck2 := FSaveCheck2;
  end;

  {Set desired check type}
  if FCheckKnown then
    FTempCheck := FKermitOptions.Check
  else
    FTempCheck := '1';

  while CharReady do begin
    C := GetChar;

    case FKermitDataState of
      dskData :
        begin
          DataBlock^[FBlockIndex] := C;
          UpdateBlockCheck (Byte(C));
          Inc (FBlockIndex);
          if FBlockIndex > FRecDataLen then begin
            FKermitDataState := dskCheck1;
          end;
        end;
      dskCheck1 :
        begin
          FC1 := UnChar (C);
          if FTempCheck = '1' then begin
            Check1 := Lo (BlockCheck);
            Check1 := (Check1 + (Check1 shr 6)) and $3F;
            if Check1 <> Byte (FC1) then
              ProtocolStatus := psBlockCheckError
            else
              ProtocolStatus := psGotData;
            Exit;
          end else
            FKermitDataState := dskCheck2;
        end;
      dskCheck2 :
        begin
          FC2 := UnChar(C);
          if FTempCheck = '2' then begin
            {1st byte has bits 11-6}
            Check1 := (BlockCheck shr 6) and $3F;
            {Second byte has bits 5-0}
            Check2 := BlockCheck and $3F;
            if (Check1 <> Byte(FC1)) or (Check2 <> Byte(FC2)) then
              ProtocolStatus := psBlockCheckError
            else
              ProtocolStatus := psGotData;
            Exit;
          end else
            FKermitDataState := dskCheck3;
        end;
      dskCheck3 :
        begin
          FC3 := UnChar(C);
          Check1 := (FBlockCheck2 shr 12) and $0F;
          Check2 := (FBlockCheck2 shr 6) and $3F;
          Check3 := FBlockCheck2 and $3F;
          if (Check1 <> Byte(FC1)) or
             (Check2 <> Byte(FC2)) or
             (Check3 <> Byte(FC3)) then
            ProtocolStatus := psBlockCheckError
          else
            ProtocolStatus := psGotData;
          Exit;
        end;
    end;
  end;

  {If we exit this way we don't have a data block yet}
  ProtocolStatus := psNoData;
  FSaveCheck := BlockCheck;
  FSaveCheck2 := FBlockCheck2;
end;

procedure TApxKermitDriver.ExpandFileInfo;
{Un-escapes file info }
var
  ExName : PApxDataBlock;
  Index, NIndex : Cardinal;
  Repeating : Boolean;
  RepeatCount : Integer;
  C : Char;
begin
  ExName := AllocMem (SizeOf (TApxDataBlock));
  FillChar (ExName^[1], SizeOf (ExName^), #0);
  Repeating := False;
  RepeatCount := 0;
  Index := 1;
  NIndex := 1;
  repeat
    C := DataBlock^[Index];
    if Repeating then begin
      if RepeatCount = 0 then begin
        if C = FKermitOptions.CtlPrefix then begin
          { the repeat char is a literal char }
          ExName^[NIndex] := C;
          Inc(NIndex);
        end else
          { get the number of times to repeat the next char }
          RepeatCount := Ord(C) - 32
      end else begin
        { repeat the current char }
        FillChar (ExName^[NIndex], RepeatCount, C);
        inc (NIndex, RepeatCount);
        RepeatCount := 0;
        Repeating := False;
      end
    end else if C = FKermitOptions.RepeatPrefix then
      { see if this is a repeat char prefix }
      Repeating := True
    else begin
      { just a regular char }
      ExName^[NIndex] := C;
      inc(NIndex);
    end;
    inc(Index);
  until Index > FActualDataLen;
  { initialize aDataBlock }
  FillChar (DataBlock^[1], SizeOf (DataBlock^), #0);
  { mode the unescaped file info to aDataBlock }
  Move (ExName^[1], DataBlock^[1], NIndex);
  FActualDataLen := NIndex;
  FreeMem (ExName, SizeOf (TApxDataBlock));
end;

procedure TApxKermitDriver.ExtractFileInfo;
{-Extracts the file name from the aDatablock}
var
  S    : string[fsPathname];
  Name : string[fsName];
  NameExt : array[0..fsName] of Char;
begin
  ExpandFileInfo;
  if FActualDataLen <= 255 then begin
    Move (DataBlock^[1], FPathname[0], FActualDataLen);
    FPathname[FActualDataLen] := #0;
  end else begin
    Move (DataBlock^[1], FPathname[0], SizeOf (PathName));
    FPathname[fsPathName] := #0;
  end;

  {Should we use its directory or ours?}
  if not FlagIsSet (Flags, apHonorDirectory) then begin
    S := StrPas (Pathname);
    Name := ExtractFileName (S);
    StrPCopy (NameExt, Name);
    ApxIncludeTrailingPathDelimiterZ (PathName, DestDir);
    StrLCat (PathName, NameExt, SizeOf (PathName));
  end;
end;

procedure TApxKermitDriver.SendInitialize;
{-Send our SendInit packet and get a response}
const
  StdHdrLen = 13;
var
  kSaveCheckChar : Char;
begin
  {Send the header}
  PutHeader(KSendInit, StdHdrLen+3);

  with FKermitOptions do begin
    {Flush input buffer in preparation for reply}
    FlushInBuffer;

    WindowSize := WindowSize and $1F;
    {Send the data bytes for the Send Initialize packet}
    PutToChar (Char (MaxPacketLen));
    PutToChar (Char (MaxTimeout));
    PutToChar (Char (PadCount));
    ComPort.PutChar (Ctl (PadChar));
    PutToChar (Terminator);
    ComPort.PutChar (CtlPrefix);
    ComPort.PutChar (HibitPrefix);
    ComPort.PutChar (Check);
    ComPort.PutChar (RepeatPrefix);
    PutToChar (Char (CapabilitiesMask));
    PutToChar (Char (WindowSize));
    PutToChar (Char (MaxLongPacketLen div 95));
    PutToChar (Char (MaxLongPacketLen mod 95));

    {Always use 1-byte checksum for SendInit packets}
    kSaveCheckChar := Check;
    Check := '1';

    {Update the check value}
    UpdateBlockCheck (Byte (ToChar (Char (MaxPacketLen))));
    UpdateBlockCheck (Byte (ToChar (Char (MaxTimeout))));
    UpdateBlockCheck (Byte (ToChar (Char (PadCount))));
    UpdateBlockCheck (Byte (Ctl (PadChar)));
    UpdateBlockCheck (Byte (ToChar (Terminator)));
    UpdateBlockCheck (Byte (CtlPrefix));
    UpdateBlockCheck (Byte (HibitPrefix));
    UpdateBlockCheck (Byte (kSaveCheckChar));
    UpdateBlockCheck (Byte (RepeatPrefix));
    UpdateBlockCheck (Byte (ToChar (Char (CapabilitiesMask))));
    UpdateBlockCheck (Byte (ToChar (Char (WindowSize))));
    UpdateBlockCheck (Byte (ToChar (Char (MaxLongPacketLen div 95))));
    UpdateBlockCheck (Byte (ToChar (Char (MaxLongPacketLen mod 95))));

    {Send the check value and terminator}
    SendBlockCheck;
    SendTerminator;

    {Restore the desired check type}
    Check := kSaveCheckChar;
  end;
end;

procedure TApxKermitDriver.SendDataPacket (Slot : Cardinal);
{-Send the prepared data packet in kDataTable[Slot]}
var
  SaveBlockNum : Cardinal;
begin
  {Move data from table to aDataBlock}
  FDataLen := FInfoTable[Slot].Len;
  Move (FDataTable^[(Slot - 1) * BlockLen], DataBlock^, FDataLen);

  {Send the packet}
  SaveBlockNum := BlockNum;
  BlockNum := FInfoTable[Slot].Seq;
  SendPacket(KData);
  BlockNum := SaveBlockNum;
end;

procedure TApxKermitDriver.ResendDataPacket (Seq : Integer);
{-Resend a data packet}
var
  I : Cardinal;
  SaveBlockNum : Cardinal;
begin
  {Find our sequence in the table}
  for I := 1 to FTableSize do
    if FInfoTable[I].Seq = Seq then
      Break;
  {Move data from Table to a DataBlock}
  FDataLen := FInfoTable[I].Len;
  Move (FDataTable^[(I - 1) * BlockLen], DataBlock^, FDataLen);

  {Send the packet}
  SaveBlockNum := BlockNum;
  BlockNum := FINfoTable[I].Seq;
  SendPacket(kData);
  BlockNum := SaveBlockNum;
end;

procedure TApxKermitDriver.SendFilePacket;
{-Fill in the Data field with Pathname and send a file packet}
var
  S : TCharArray;
begin
  {Send the data field}
  if FlagIsSet (Flags, apIncludeDirectory) then
    StrCopy(S, Pathname)
  else
    JustFileNameZ (S, Pathname);
  FDataLen := StrLen (S);

  {Truncate if aPathname is a long filename greater than blocksize}
  if FDataLen > BlockLen then
    FDataLen := BlockLen;

  Move (S[0], DataBlock^[1], FDataLen);
  SendPacket (KFile);
end;

procedure TApxKermitDriver.ProcessOptions;
{-Save the just-received options}
var
  Tmp : Byte;
  LBLen : Cardinal;
  NewTableSize : Cardinal;
  NewaBlockLen : Cardinal;
begin
  ProtocolError := ecOK;

  {Move defaults in}
  FUsingRepeat := False;
  FUsingHibit := False;
  FRmtKermitOptions := ApxMissingKermitOptions;

  {Override the defaults where specified}
  Move (DataBlock^[1], FRmtKermitOptions, FRecDataLen);

  {Limit the block size, if requested}
  if FRmtKermitOptions.MaxPacketLen < FKermitOptions.MaxPacketLen then
    FKermitOptions.MaxPacketLen := FRmtKermitOptions.MaxPacketLen;

  {Set repeat option if both sides are asking for it}
  Tmp := Byte (FRmtKermitOptions.RepeatPrefix);
  if (Char(Tmp) = FKermitOptions.RepeatPrefix) and
     (((Tmp > 32) and (Tmp < 63)) or ((Tmp > 95) and (Tmp < 127))) then
    FUsingRepeat := True;

  {Set hibit quoting option if either side asks for it}
  Tmp := Byte(FRmtKermitOptions.HibitPrefix);
  if ((Tmp > 32) and (Tmp < 63)) or ((Tmp > 95) and (Tmp < 127)) then begin
    FUsingHibit := True;
    FKermitOptions.HibitPrefix := FRmtKermitOptions.HibitPrefix;
  end;
  if not FUsingHibit then begin
    Tmp := Byte (FKermitOptions.HibitPrefix);
    {if we want it, and the remote said he can do it if requested, turn it on}
    if ((Tmp > 32) and (Tmp < 63)) or ((Tmp > 95) and (Tmp < 127)) then
      if FRmtKermitOptions.HibitPrefix = 'Y' then
        FUsingHibit := True;
  end;

  {Set long packets if sender asks and we allow}
  if (Byte(FRmtKermitOptions.CapabilitiesMask) and LongPackets <> 0) and
     (FlagIsSet(Flags, apKermitLongPackets)) then begin
    FKermitOptions.CapabilitiesMask :=
          FKermitOptions.CapabilitiesMask or LongPackets;
    LBLen := Cardinal(Byte(UnChar(DataBlock^[MaxLx1])) * 95) +
                     (Byte(UnChar(DataBlock^[MaxLx2])));
    if LBLen = 0 then
      FKermitOptions.MaxLongPacketLen := FKermitOptions.MaxPacketLen
    else if (LBLen > 0) and (LBLen <= 1024) then
      FKermitOptions.MaxLongPacketLen := LBLen
    else
      FKermitOptions.MaxLongPacketLen := 500;
    FLPInUse := True;
  end;

  {Set SWC if sender asks and we allow}
  NewTableSize := FTableSize;
  if (Byte(FRmtKermitOptions.CapabilitiesMask) and SlidingWindows <> 0) and
     (FlagIsSet(Flags, apKermitSWC)) then begin
    FKermitOptions.CapabilitiesMask :=
          FKermitOptions.CapabilitiesMask or SlidingWindows;
    {If remote's window size is less than ours then use its size}
    Tmp := FRmtKermitOptions.WindowSize and $1F;
    if Tmp < FKermitOptions.WindowSize then begin
      FKermitOptions.WindowSize := Tmp;
      NewTableSize := Tmp;
    end;
  end else begin
    NewTableSize := 1;
    FKermitOptions.WindowSize := 1;
  end;

  if FKermitState = rkCollectInit then
    {We're receiving, use whatever block check type sender asks for}
    if (FRmtKermitOptions.Check >= '1') and
       (FRmtKermitOptions.Check <= '3') then
      FKermitOptions.Check := FRmtKermitOptions.Check
    else
      {We're transmitting, agree on check type or force '1'}
      if FKermitOptions.Check <> FRmtKermitOptions.Check then
        FKermitOptions.Check := '1';
  CheckType := CheckVal[Byte(FKermitOptions.Check)-$30];

  {Set status and other options}
  with FKermitOptions do begin
    if FLPInUse then
      NewaBlockLen := MaxLongPacketLen
    else
      NewaBlockLen := MaxPacketLen;
    if NewTableSize > 1 then
      TurnDelay := FSWCTurnDelay
    {else}
      {TurnDelay := KermitTurnDelay;}
  end;

  {Allocate new kDataTable to account for changes in aBlockLen/window count}
  if (NewTableSize <> FTableSize) or (NewaBlockLen <> BlockLen) then begin
    DeallocateWindowTable;
    FTableSize := NewTableSize;
    BlockLen := NewaBlockLen;
    AllocateWindowTable;
  end;
end;

procedure TApxKermitDriver.SendOptions;
{-Send our options}
var
  TotalLen : Byte;
begin
  Move (FKermitOptions, DataBlock^[1], MaxKermitOption);
  DataBlock^[12] := Char (FKermitOptions.MaxLongPacketLen div 95);
  DataBlock^[13] := Char (FKermitOptions.MaxLongPacketLen mod 95);
  TotalLen := MaxKermitOption + 3;

  {Can't use SendAck so we'll do everything here}
  PutHeader (KAck, TotalLen);

  {Put each option, transforming as required}
  PutToChar (DataBlock^[1]);                             {MaxL}
  UpdateBlockCheck (Byte (ToChar (DataBlock^[1])));
  PutToChar (DataBlock^[2]);                             {Time}
  UpdateBlockCheck (Byte (ToChar (DataBlock^[2])));
  PutToChar (DataBlock^[3]);                             {NPad}
  UpdateBlockCheck(Byte (ToChar (DataBlock^[3])));
  ComPort.PutChar (Ctl (DataBlock^[4]));                          {PadC}
  UpdateBlockCheck(Byte (Ctl (DataBlock^[4])));
  PutToChar (DataBlock^[5]);                             {EOL}
  UpdateBlockCheck (Byte (ToChar (DataBlock^[5])));
  ComPort.PutChar (DataBlock^[6]);                               {QCtl}
  UpdateBlockCheck (Byte (DataBlock^[6]));
  ComPort.PutChar (DataBlock^[7]);                               {QBin}
  UpdateBlockCheck (Byte (DataBlock^[7]));
  ComPort.PutChar (DataBlock^[8]);                               {Chkt}
  UpdateBlockCheck (Byte (DataBlock^[8]));
  ComPort.PutChar (DataBlock^[9]);                               {Rept}
  UpdateBlockCheck (Byte (DataBlock^[9]));
  PutToChar (DataBlock^[10]);                            {Capas}
  UpdateBlockCheck (Byte (ToChar (DataBlock^[10])));
  PutToChar (DataBlock^[11]);                            {Windo}
  UpdateBlockCheck (Byte (ToChar (DataBlock^[11])));
  PutToChar (DataBlock^[12]);                            {MaxLx1}
  UpdateBlockCheck (Byte (ToChar (DataBlock^[12])));
  PutToChar (DataBlock^[13]);                            {MaxLx2}
  UpdateBlockCheck (Byte (ToChar (DataBlock^[13])));

  {Put checksum and terminator}
  SendBlockCheck;
  SendTerminator;

  {Check type has been decided upon}
  FCheckKnown := True;
end;

function TApxKermitDriver.CheckRetries : Boolean;
{-Increments retry count, returns True if greater than aHandshakeRetry}
var
  Failed : Boolean;
begin
  ForceStatus := True;

  {Exit if an abort is pending}
  if ProtocolStatus = psCancelRequested then
    CheckRetries := True
  else begin
    BlockErrors := BlockErrors + 1;
    TotalErrors := TotalErrors + 1;
    Failed := BlockErrors > HandshakeRetry;
    if Failed then begin
      if ProtocolError = ecOK then
        ProtocolError := ecProtocolError;
      apProtocolError (ProtocolError);
    end;
    CheckRetries := Failed;
  end;
end;

procedure TApxKermitDriver.LoadTransmitData;
{-Escapes data from WorkBlock into DataBlock}
label
  Skip;
const
  SafetyMargin = 5;
var
  WIndex : Cardinal;
  DIndex : Cardinal;
  RIndex : Cardinal;
  RepeatCnt : Cardinal;
  C : Char;
  ByteCnt : Cardinal;

  function Repeating(C : Char; var Cnt : Cardinal) : Boolean;
  {Returns True (and new index) if repeat C's are found}
  const
    MaxRpt = 94;  {Per Kermit Protocol Manual}
  var
    Index : Cardinal;
  begin
    Index := WIndex;
    Cnt := 1;

    {Loop while next chars are the same as C}
    while (Index <= FWorkLen) and
          (FWorkBlock^[Index] = C) and
          (Cnt < MaxRpt) do begin
      Inc (Cnt);
      Inc (Index);
    end;

    {Set function result (Cnt already has repeat count)}
    Repeating := Cnt > FMinRepeatCnt;
  end;

  function ReloadWorkBlock : Boolean;
  {-Reloads WorkBlock if required -- Return False to Exit}
  begin
    ReloadWorkBlock := False;
    if FWorkEndPending and (WIndex > FWorkLen) then
      Exit;

    {Reload WorkBlock as needed}
    if (WIndex > SizeOf(FWorkBlock^)) then begin
      FWorkLen := SizeOf(FWorkBlock^);
      FWorkEndPending := ReadProtocolBlock (FWorkBlock^, FWorkLen);

      {Finished if no more bytes read}
      if FWorkEndPending and (FWorkLen = 0) then
        Exit;

      if ProtocolError = ecOK then begin
        WIndex := 1;
        FileOfs := FileOfs + LongInt (FWorkLen);
      end else begin
        Cancel;
        Exit;
      end;
    end;

    {If we get here, block was reloaded ok or didn't need reload}
    ReloadWorkBlock := True;
  end;

begin
  {Exit immediately if no more DataBlocks to send}
  if (FWorkEndPending) and (FLastWorkIndex > FWorkLen) then begin
    ProtocolStatus := psEndFile;
    Exit;
  end;

  with FKermitOptions do begin
    WIndex := FLastWorkIndex;
    DIndex := 1;
    ByteCnt := 0;

    if FLPInUse then
      RIndex := FKermitOptions.MaxLongPacketLen - SafetyMargin
    else
      RIndex := FKermitOptions.MaxPacketLen - SafetyMargin;

    while DIndex < RIndex do begin
      {C is the next character to move}
      C := FWorkBlock^[WIndex];
      Inc(WIndex);
      Inc(ByteCnt);

      {Look ahead for repeating char sequence}
      if FUsingRepeat then
        if Repeating (C, RepeatCnt) then begin
          {C is a repeating char, add repeat prefix and count}
          DataBlock^[DIndex] := RepeatPrefix;
          DataBlock^[DIndex+1] := ToChar(Char(RepeatCnt));
          Inc(DIndex, 2);
          Inc(WIndex, RepeatCnt-1);
          Inc(ByteCnt, RepeatCnt-1);
        end;

      {Process all escaping conditions}
      if FUsingHibit then begin
        if (C = HibitPrefix) or (C = Chr(Byte(HibitPrefix) or $80)) then begin
          if IsHibit(C) then begin
            DataBlock^[DIndex] := HibitPrefix;
            Inc(DIndex);
          end;
          DataBlock^[DIndex] := CtlPrefix;
          DataBlock^[DIndex+1] := HibitPrefix;
          Inc(Dindex,2);
          goto Skip;
        end else if IsHibit(C) then begin
          C := Chr(Byte(C) and $7F);
          DataBlock^[DIndex] := HibitPrefix;
          Inc(DIndex);
        end;
      end;

      if IsCtl(C) then begin
        {C is a control character, add prefix and modified C}
        DataBlock^[DIndex] := CtlPrefix;
        DataBlock^[DIndex+1] := Ctl(C);
        Inc(DIndex, 2);
      end else if (C = CtlPrefix) or (C = HiBit(CtlPrefix)) then begin
        {C is the prefix char, add prefix and normal CtlPrefix char}
        DataBlock^[DIndex] := CtlPrefix;
        DataBlock^[DIndex+1] := C;
        Inc(DIndex, 2);
      end else if FUsingRepeat and
            ((C = RepeatPrefix) or (C = Hibit(RepeatPrefix))) then begin
            {C is repeat prefix char, add prefix and normal RepeatPrefix char}
        DataBlock^[DIndex] := CtlPrefix;
        DataBlock^[Dindex+1] := C;
        Inc(DIndex, 2);
      end else begin
        {Normal, single, unescaped character}
        DataBlock^[DIndex] := C;
        Inc(DIndex);
      end;

Skip:
      {Check if WorkBlock should be reloaded}
      if not ReloadWorkBlock then begin
        FDataLen := DIndex - 1;
        BytesRemaining := BytesRemaining - LongInt (ByteCnt);
        BytesTransferred := BytesTransferred + LongInt (ByteCnt);
        ElapsedXfrTime := ElapsedTime(Timer);
        FLastWorkIndex := WIndex;
        Exit;
      end;
    end;

    FDataLen := DIndex - 1;
    BytesRemaining := BytesRemaining - LongInt (ByteCnt);
    BytesTransferred := BytesTransferred + LongInt (ByteCnt);
    ElapsedXfrTime := ElapsedTime(Timer);
    FLastWorkIndex := WIndex;
  end;
end;

procedure TApxKermitDriver.OpenFile;
{-Open file from data in just received file packet}
begin
  {Assume error}
  FKermitState := rkError;

  {Get info from file packet}
  ExtractFileInfo;

  {Send file name to user's LogFile procedure}
  LogFile (lfReceiveStart);

  {Accept this file?}
  FSkipped := False;
  if not AcceptFile(Pathname) then begin
    ProtocolStatus := psFileRejected;
    ForceStatus := True;
    Cancel;
    FSkipped := True;
    Exit;
  end;

  {Reset status stuff}
  FileOfs := 0;
  BlockNum := Inc64(FRecBlockNum);
  BlockErrors := 0;
  NewTimer(FTimer, 1);
  TimerStarted := False;
  BytesRemaining := 0;
  BytesTransferred := 0;
  TotalErrors := 0;
  SrcFileLen := 0;

  {Prepare to write to file}
  ProtocolError := ecOK;
  PrepareWriting;
  if (ProtocolError = ecOK) and
     (ProtocolStatus <> psCantWriteFile) then begin
    {File opened OK}
    FReceiveInProgress := True;
    SendAck(FRecBlockNum);

    {Init sequence}
    FTableHead := 1;
    FTableTail := 1;
    FillChar(FInfoTable, SizeOf(FInfoTable), 0);
    FInfoTable[1].Seq := FRecBlockNum;

    {Set next state}
    FKermitState := rkGetData;
    ComPort.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
  end else begin
    {Error opening/creating file, tell remote we're aborting}
    if ProtocolStatus <> psCantWriteFile then
      ProtocolError := ecCantWriteFile;
    SendError(eFileError);
    FKermitState := rkError;
  end;
end;

procedure TApxKermitDriver.PrepareReceive;
{-Prepare to start receiving}
begin
  SaveStatus := psOK;
  BlockNum := 0;
  FReceiveInProgress := False;
  FTransmitInProgress := False;
  BytesRemaining := 0;
  BytesTransferred := 0;
  TotalErrors := 0;
  ForceStatus := True;
  FKermitState := rkInit;
  FKermitHeaderState := hskNone;
  FillChar(FInfoTable, SizeOf(FInfoTable), 0);
  ElapsedXfrTime := 0;
  ResetStatus;
  ProtocolStatus := psProtocolHandshake;
  apShowFirstStatus;
  FTableHead := 1;
  FTableTail := 1;
  TimerStarted := False;
  FCheckKnown := False;
end;

procedure TApxKermitDriver.Receive(Msg, wParam : Cardinal;
                     lParam : LongInt);
{-Performs one increment of a Kermit receive}
label
  ExitPoint;
var
  TriggerID    : Cardinal absolute wParam;
  Finished     : Boolean;
  StatusTimeMS : Cardinal; 
begin
  EnterCriticalSection(FProtSection);

  {Exit if protocol was cancelled while waiting for crit section}
  if FKermitState = rkDone then begin
    LeaveCriticalSection(FProtSection);
    Exit;
  end;

  {Force TriggerID for TriggerAvail messages}
  if Msg = apx_TriggerAvail then begin
    FillInBuff;
    TriggerID := aDataTrigger;
  end;

  repeat

    ComPort.DebugLog.AddDebugEntry (TApxCustomProtocol,
                                Cardinal (AxdtKermit),
                                Cardinal (LogKermitState[FKermitState]),
                                0);

    {rkDone could arrive with the trailing padding and CR}
    if FKermitState = rkDone then begin
      while CharReady do
        GetChar;
          LeaveCriticalSection(FProtSection);  
        Exit;
      end;

    {Restore last status}
    ProtocolStatus := SaveStatus;

    {Check for user abort}
    if SaveStatus <> psCancelRequested then begin
      if Integer(TriggerID) = NoCarrierTrigger then begin
        ProtocolStatus := psAbortNoCarrier;
        FKermitState := rkError;
      end;
      if Msg = apx_ProtocolCancel then begin
        Cancel;
        FKermitState := rkError;
      end;
    end;

    {Show status at requested intervals and after significant events}
    if ForceStatus or (Integer(TriggerID) = StatusTrigger) then begin
      if FComPort.Dispatcher.TimerTimeRemaining (StatusTrigger,
                                                 StatusTimeMS) <> 0 then
        StatusTimeMS := 0;
      if LongInt (StatusTimeMS) <= 0 then begin
        ShowStatus(0);
        FComPort.Dispatcher.SetTimerTrigger(StatusTrigger, StatusInterval, True);
        ForceStatus := False;
      end;
      if Integer(TriggerID) = StatusTrigger then begin
        LeaveCriticalSection(FProtSection);
        Exit;
      end;
    end;

    {Preprocess incoming headers}
    case FKermitState of
      rkGetInit,
      rkGetFile,
      rkGetData :
            if TriggerID = aDataTrigger then begin
              {Header might be present, try to get one}
              CheckForHeader;
              case ProtocolStatus of
                psOK, psNoHeader, psGotHeader : ;
                else if CheckRetries then
                  FKermitState := rkError;
              end;
            end else if Integer(TriggerID) = TimeoutTrigger then begin
              {Timeout while waiting for header}
              if CheckRetries then
                {Fatal error if too many retries}
                FKermitState := rkError
              else
                {Let state machine take apropriate recovery action}
                ProtocolStatus := psTimeout;
            end else
              {Indicate that we don't have a header yet}
              ProtocolStatus := psNoHeader;
        end;

        {Preprocess incoming datapackets}
        case FKermitState of
          rkCollectInit,
          rkCollectFile,
          rkCollectData :
            if TriggerID = ApxProtocolDataTrigger then begin 
              ReceiveBlock;
              case ProtocolStatus of
                psOK, psNoData, psGotData : ;
                else begin
                  ForceStatus := True;
                  if CheckRetries then
                    FKermitState := rkError
                end;
              end;
            end else if Integer(TriggerID) = TimeoutTrigger then begin
              {Timeout waiting for datapacket}
              ForceStatus := True;
              if CheckRetries then
                {Fatal error if too many retries}
                FKermitState := rkError
              else
                {Let state machine take apropriate recovery action}
                ProtocolStatus := psTimeout;
            end else
              ProtocolStatus := psNoData;
        end;

        {Main state processor}
        case FKermitState of
          rkInit :
            begin
              BlockNum := 0;

              {Wait for SendInit packet}
              FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
              FKermitState := rkGetInit;
              FRecDataLen := 0;
            end;

          rkGetInit :
            case ProtocolStatus of
              psGotHeader :
                begin
                  FKermitState := rkCollectInit;
                  FKermitDataState := FirstDataState[FRecDataLen = 0];
                  FBlockIndex := 1;
                end;
              psNoheader :
                {Keep waiting};
              else
                FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
            end;

          rkCollectInit :
            case ProtocolStatus of
              psGotData :
                case FPacketType of
                  KSendInit :
                    begin
                      ProcessOptions;
                      if ProtocolError <> ecOK then
                        {if we have an error with the KSendInit packet, the}
                        {other side may be giving us a 1-byte checksum}
                        if BlockCheck = Ord(FC1) then
                          ProtocolError := ecOK;
                      if ProtocolError = ecOK then begin
                        SendOptions;
                        BlockErrors := 0;
                        FKermitState := rkGetFile;
                        FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
                      end else begin
                        FKermitState := rkError;
                        {force a 1-byte checksum since we got an error}
                        FCheckKnown := False;
                      end;
                    end;
                  KError :
                    FKermitState := rkError;
                  else begin
                    FKermitState := rkGetInit;
                    FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
                  end;
                end;
              psNoData :
                {Keep waiting for data};
              else begin
                {Timeout or other error, retry}
                FKermitState := rkGetInit;
                FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
              end;
            end;

          rkGetFile :
            case ProtocolStatus of
              psGotHeader :
                begin
                  FKermitState := rkCollectFile;
                  FKermitDataState := FirstDataState[FRecDataLen=0];
                  FBlockIndex := 1;
                end;
              psNoheader :
                {Keep waiting};
              else
                SendNak;
            end;

          rkCollectFile :
            case ProtocolStatus of
              psGotData :
                case FPacketType of
                  KFile :     {Open/create the file}
                    OpenFile;
                  KSendInit : {Got another SendInit, process again}
                    begin
                      ProcessOptions;
                      if ProtocolError = ecOK then begin
                        SendOptions;
                        BlockErrors := 0;
                        FKermitState := rkGetFile;
                      end else begin
                        FKermitState := rkError;
                        FCheckKnown := False;
                      end;                                          
                    end;
                  KDisplay : {Ignore}
                    ;
                  KBreak : {Got break, protocol transfer is finished}
                    begin
                      SendAck(FRecBlockNum);
                      FKermitState := rkComplete;
                    end;
                  KEndOfFile :  {Got out of place end of file header}
                    begin
                      SendAck (FRecBlockNum);
                      if CheckRetries then begin
                        SendError(eSync);
                        FKermitState := rkError;
                      end;
                      FKermitState := rkGetFile;
                    end;
                  else
                    FKermitState := rkError;
                end;
              psNoData :
                {Keep waiting for data};
              else
                {Timeout or other error, retry}
                FKermitState := rkGetFile;
                SendNak;
                FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
            end;

          rkGetData :
            case ProtocolStatus of
              psGotHeader :
                begin
                  FKermitState := rkCollectData;
                  FKermitDataState := FirstDataState[FRecDataLen=0];
                  FBlockIndex := 1;
                end;
              psNoheader :
                {Keep waiting};
              else
                FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
            end;

          rkCollectData :
            case ProtocolStatus of
              psGotData :
                case FPacketType of
                  KData : {Got data packet}
                    begin
                      ForceStatus := True;
                      ProcessDataPacket;
                      if ProtocolError = ecOK then begin
                        BlockErrors := 0;
                        FKermitState := rkGetData;
                        FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
                      end else begin
                        Cancel;
                        FKermitState := rkError;
                      end;
                    end;
                  KEndOfFile :
                    if (FActualDataLen > 1) and
                       (DataBlock^[1] = ApxKermitDiscardChar) then begin
                      FKermitState := rkError;
                      ProtocolStatus := psCancelRequested;
                    end else begin
                      FlushTableToDisk;
                      if ProtocolError = ecOK then begin
                        FinishWriting;
                        if ProtocolError = ecOK then begin
                          LogFile(lfReceiveOk);
                          FReceiveInProgress := False;
                          SendAck(FRecBlockNum);
                          BlockNum := NextSeq(FRecBlockNum);
                          BlockErrors := 0;
                          FKermitState := rkGetFile;
                          TimerStarted := False;
                        end else begin
                          Cancel;
                          FKermitState := rkError;
                        end;
                      end else begin
                        Cancel;
                        FKermitState := rkError;
                      end;
                    end;
                  KFile :
                    begin
                      SendAck(FRecBlockNum);
                      if CheckRetries then begin
                        SendError(eSync);
                        FKermitState := rkError;
                      end else
                        FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
                    end;
                end;
              psNoData :
                {Keep waiting for data};

              else begin
                {NAK if not using Windows (window logic will NAK later)}
                if FTableSize = 1 then
                  SendNak;
                FKermitState := rkGetData;
                FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
              end;
            end;

          rkWaitCancel :
            if (Integer(TriggerID) = TimeoutTrigger) or
               (Integer(TriggerID) = OutBuffUsedTrigger) then
             FKermitState := rkError;

          rkError :
            begin
              FinishWriting;
              if FSkipped then
                LogFile(lfReceiveSkip)
              else
                LogFile(lfReceiveFail);
              FlushInBuffer;
              FKermitState := rkComplete;
            end;

          rkComplete :
            begin
              FKermitState := rkDone;
              apShowLastStatus;

              {Remove our triggers, restore old triggers}
              FlushInBuffer;
              apSignalFinish (False);
            end;
        end;

        {Reset header state after complete headers}
        if ProtocolStatus = psGotHeader then
          FKermitHeaderState := hskNone;

        {Reset status for various conditions}
        case ProtocolStatus of
          psGotHeader, psNoHeader, psGotData, psNoData :
            ProtocolStatus := psOK;
        end;

        {Save last status value}
        SaveStatus := ProtocolStatus;

        {Stay in state machine or exit}
        case FKermitState of
          {Stay in state machine if more data ready}
          rkGetInit,
          rkCollectInit,
          rkGetFile,
          rkCollectFile,
          rkGetData,
          rkCollectData,
          rkWaitCancel  : Finished := not CharReady;           


          {Stay in state machine for these interim states}
          rkInit,
          rkComplete,
          rkError       : Finished := False;

          {Done state}
          rkDone        : Finished := True;
          else            Finished := True;
        end;

        {If staying in state machine force data ready}
        TriggerID := ApxProtocolDataTrigger; 
      until Finished;

      LeaveCriticalSection(FProtSection);
end;

procedure TApxKermitDriver.PrepareTransmit;
{-Prepare to start transmitting}
begin
  SaveStatus := psOK;
  BlockNum := 0;
  FReceiveInProgress := False;
  FTransmitInProgress := False;
  BytesRemaining := 0;
  BytesTransferred := 0;
  TotalErrors := 0;
  ForceStatus := True;
  FKermitState := tkInit;
  FKermitHeaderState := hskNone;
  FillChar(FInfoTable, SizeOf(FInfoTable), 0);
  ElapsedXfrTime := 0;
  ResetStatus;
  ProtocolStatus := psProtocolHandshake;
  apShowFirstStatus;
  TimerStarted := False;
  FCheckKnown := False;
end;

procedure TApxKermitDriver.Transmit(Msg, wParam : Cardinal;
                      lParam : LongInt);
{-Performs one increment of a Kermit transmit}
var
  TriggerID    : Cardinal absolute wParam;
  Finished     : Boolean;
  StatusTimeMS : Cardinal; 
begin
  EnterCriticalSection(FProtSection);

  {Exit if protocol was cancelled while waiting for crit section}
  if FKermitState = tkDone then begin
    LeaveCriticalSection(FProtSection); 
    Exit;
  end;

      {Force TriggerID for TriggerAvail messages}
      if Msg = apx_TriggerAvail then begin
        FillInBuff;
        TriggerID := ApxProtocolDataTrigger;
      end;

      repeat

        ComPort.DebugLog.AddDebugEntry (TApxCustomProtocol,
                                    Cardinal (AxdtKermit),
                                    Cardinal (LogKermitState[FKermitState]),
                                    0);
      
        {Nothing to do if state is tkDone}
        if FKermitState = tkDone then begin
          LeaveCriticalSection(FProtSection); 
          Exit;
        end;

        {Restore last status}
        ProtocolStatus := SaveStatus;

        {Check for user abort}
        if SaveStatus <> psCancelRequested then begin
          if Integer(TriggerID) = NoCarrierTrigger then begin
            ProtocolStatus := psAbortNoCarrier;
            FKermitState := rkError;
          end;
          if Msg = apx_ProtocolCancel then begin
            Cancel;
            FKermitState := tkError;
          end;
        end;

        {Show status at requested intervals and after significant events}
        if ForceStatus or (Integer(TriggerID) = StatusTrigger) then begin
          if FComPort.Dispatcher.TimerTimeRemaining(StatusTrigger, 
                                  StatusTimeMS) <> 0 then
            StatusTimeMS := 0;
          if LongInt (StatusTimeMS) <= 0 then begin
            ShowStatus(0);
            FComPort.Dispatcher.SetTimerTrigger(StatusTrigger, StatusInterval, True);
            ForceStatus := False;
          end;
          if Integer(TriggerID) = StatusTrigger then begin          
            LeaveCriticalSection(FProtSection); 
            Exit;
          end;
        end;

        {Preprocess incoming headers}
        case FKermitState of
          tkInitReply,
          tkFileReply,
          tkBlockReply,
          tkEofReply,
          tkBreakReply :
            if TriggerID = ApxProtocolDataTrigger then
              {Got data, see if it's a header}
              CheckForHeader
            else if Integer(TriggerID) = TimeoutTrigger then begin   
              {Timed out waiting for header...}
              ForceStatus := True;
              if CheckRetries then
                {Fatal error if too many retries}
                FKermitState := tkError
              else
                {Let state machine take apropriate recovery action}
                ProtocolStatus := psTimeout;
            end else
              {Indicate that we don't have a header yet}
              ProtocolStatus := psNoHeader;
        end;

        {Preprocess incoming datapackets}
        case FKermitState of
          tkCollectInit,
          tkCollectFile,
          tkCollectBlock,
          tkCollectEof,
          tkCollectBreak :
            if TriggerID = ApxProtocolDataTrigger then begin
              ReceiveBlock;
              case ProtocolStatus of
                psOK, psNoData, psGotData : ;
                else begin
                  ForceStatus := True;
                  if CheckRetries then
                    FKermitState := tkError
                end;
              end;
            end else if Integer(TriggerID) = TimeoutTrigger then begin
              {Timeout waiting for datapacket}
              ForceStatus := True;
              if CheckRetries then
                {Fatal error if too many retries}
                FKermitState := tkError
              else
                {Let state machine take apropriate recovery action}
                ProtocolStatus := psTimeout;
            end else
              {Indicate that we don't have any data yet}
              ProtocolStatus := psNoData;
        end;

        {Process current state}
        case FKermitState of
          tkInit :
            begin
              BlockNum := 0;

              {Send SendInit packet}
              SendInitialize;

              {Start waiting for reply}
              FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
              FKermitState := tkInitReply;
            end;

          tkInitReply :
            case ProtocolStatus of
              psGotHeader :
                begin
                  FKermitState := tkCollectInit;
                  FKermitDataState := FirstDataState[FRecDataLen=0];
                  FBlockIndex := 1;
                end;
              psNoheader :
                {Keep waiting};
              else
                {Timeout or block error, resend SendInit}
                if CheckRetries then
                  FKermitState := tkError
                else begin
                  FlushInBuffer;
                  FKermitState := tkInit;
                end;
            end;

          tkCollectInit :
            case ProtocolStatus of
              psGotData :
                case FPacketType of
                  KAck :
                    begin
                      ProcessOptions;
                      if ProtocolError = ecOK then begin
                        FKermitState := tkOpenFile;
                        BlockErrors := 0;
                        FCheckKnown := True;
                      end else
                        FKermitState := tkError;
                    end;
                  KError :
                    FKermitState := tkError;
                  else
                    if CheckRetries then
                      FKermitState := tkError
                    else
                      FKermitState := tkInit;
                end;
              psNoData :
                {Keep waiting for data};
              else
                {Timeout or other error, retry}
                FKermitState := tkInit;
            end;

          tkOpenFile :
            begin
              ForceStatus := True;
              ResetStatus;
              if not NextFile (FPathname) then begin
                {Error - no files to send (AsyncStatus already set)}
                FKermitState := tkError;
              end else begin
                ForceStatus := True;
                if UpcaseFileNames then
                  AnsiStrUpper(Pathname);
                PrepareReading;
                if ProtocolError = ecOK then begin
                  {Read the first protocol buffer}
                  FWorkLen := SizeOf(FWorkBlock^);
                  FileOfs := 0;
                  FWorkEndPending := ReadProtocolBlock(FWorkBlock^, FWorkLen);
                  if ProtocolError = ecOK then begin
                    FileOfs := FWorkLen;
                    FLastWorkIndex := 1;
                    FTransmitInProgress := True;
                    LogFile (lfTransmitStart);
                    FKermitState := tkSendFile;
                    BlockNum := Inc64(BlockNum);
                    NewTimer(FTimer, 1); 
                    TimerStarted := True;
                  end else
                    FKermitState := tkError;
                end else
                  FKermitState := tkError;
              end;
            end;

          tkSendFile :
            begin
              ForceStatus := True;
              {aBlockNum := Inc64(aBlockNum);}
              SendFilePacket;
              FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
              FKermitState := tkFileReply;
            end;

          tkFileReply :
            case ProtocolStatus of
              psGotHeader :
                begin
                  FKermitState := tkCollectFile;
                  FKermitDataState := FirstDataState[FRecDataLen=0];
                  FBlockIndex := 1;
                end;
              psNoheader :
                {Keep waiting};
              else if CheckRetries then
                FKermitState := tkError
              else
                FKermitState := tkSendFile;
            end;

          tkCollectFile :
            case ProtocolStatus of
              psGotData :
                case FPacketType of
                  KAck :
                    if FRecBlockNum = FExpectedAck then begin
                      BlockErrors := 0;
                      FTableHead := 1;
                      FKermitState := tkCheckTable;
                    end else begin
                      if CheckRetries then
                        FKermitState := tkError
                      else
                        FKermitState := tkSendFile;
                    end;
                  KEndOfFile,
                  KError :
                    FKermitState := tkError;
                  else
                    if CheckRetries then
                      FKermitState := tkError
                    else
                      FKermitState := tkSendFile;
                end;
              psNoData :
                {Keep waiting for data} ;
              else
                {Timeout or other error}
                FKermitState := tkSendFile;
            end;

          tkCheckTable :
            begin
              {See if there is room to load another buffer into table}
              if not TableFull then begin
                {Get next escaped block}
                LoadTransmitData;
                if ProtocolStatus = psEndFile then begin
                  {No more data to send, wait for acks or send eof}
                  if PacketsOutstanding then
                    FKermitState := tkBlockReply
                  else begin
                    BlockNum := Inc64(BlockNum);
                    FKermitState := tkSendEof;
                  end;
                end else begin
                  {Save in table}
                  BlockNum := Inc64(BlockNum);
                  FTableHead := NextSeq(FTableHead);
                  Move(DataBlock^, FDataTable^[(FTableHead-1)*BlockLen], FDataLen);
                  FInfoTable[FTableHead].Len := FDataLen;
                  FInfoTable[FTableHead].InUse := True;
                  FInfoTable[FTableHead].Seq := BlockNum;
                  FInfoTable[FTableHead].Retries := 0;
                  FNext2Send := FTableHead;
                  FKermitState := tkSendData;
                  FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, TransTimeout, True);
                  FComPort.Dispatcher.SetStatusTrigger(OutBuffUsedTrigger, 0, True);
                 end;
              end else begin
                {Table full, wait for Acks...}
                FKermitState := tkBlockReply;
                FComPort.Dispatcher.SetTimerTrigger (TimeoutTrigger,
                         FKermitOptions.MaxTimeout * 1000, True);
              end;
            end;

          tkSendData :
            if Integer(TriggerID) = OutBuffUsedTrigger then begin
              ForceStatus := True;
              SendDataPacket(FNext2Send);
              FKermitState := tkBlockReply;
              FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
            end else if Integer(TriggerID) = TimeoutTrigger then begin
              apProtocolError(ecTimeout);
              FKermitState := tkError;
            end;

          tkBlockReply :
            case ProtocolStatus of
              psGotHeader :
                begin
                  FKermitState := tkCollectBlock;
                  FKermitDataState := FirstDataState[FRecDataLen=0];
                  FBlockIndex := 1;
                end;
              psNoHeader :
                FKermitState := tkCheckTable;
              else begin
                FKermitState := tkSendData;
                ResendDataPacket(GetOldestSequence);
                FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, TransTimeout, True);
                FComPort.Dispatcher.SetStatusTrigger(OutBuffUsedTrigger, 0, True);
              end;
            end;

          tkCollectBlock :
            case ProtocolStatus of
              psGotData :
                case FPacketType of
                  KAck :
                    begin
                      BlockErrors := 0;
                      if (FRecDataLen > 0) and
                         (DataBlock^[1] in ['Z', 'X', 'D']) then begin
                        {Abort requested}
                        ProtocolStatus := psCancelRequested;
                        FKermitState := tkError;
                      end else begin
                        {Signal Ack, then go load next block}
                        GotAck(FRecBlockNum);
                        FKermitState := tkCheckTable;
                        FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
                      end;
                    end;
                  KError :
                    FKermitState := tkError;
                  else
                    if CheckRetries then
                      FKermitState := tkError
                    else begin
                      FNext2Send := SeqInTable(FRecBlockNum);
                      if FNext2Send <> -1 then begin
                        {Resend Nak'd packet}
                        Inc(FInfoTable[FNext2Send].Retries);
                        if FInfoTable[FNext2Send].Retries < HandshakeRetry then begin
                          FKermitState := tkSendData;
                          FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, TransTimeout, True);
                          FComPort.Dispatcher.SetStatusTrigger(OutBuffUsedTrigger, 0, True);
                        end else
                          FKermitState := tkError;
                      end else begin
                        {Nak outside of table...}
                        if FRecBlockNum = Inc64(BlockNum) then begin
                          {...was one past the table, treat as nak}
                          GotAck(BlockNum);
                          FKermitState := tkCheckTable;
                          FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger,
                                           HandshakeWait, True);
                        end else begin
                          {...was more than one past the table, ignore}
                          FNext2Send := FTableHead;
                          FKermitState := tkCheckTable;
                        end;
                      end;
                    end;
                end;
              psNoData :
                {Keep waiting};
              else begin
                FKermitState := tkSendData;
                FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, TransTimeout, True);
                FComPort.Dispatcher.SetStatusTrigger(OutBuffUsedTrigger, 0, True);
              end;                                               
            end;

          tkSendEof :
            begin
              ForceStatus := True;
              FinishReading;
              LogFile(lfTransmitOk);
              FTransmitInProgress := False;
              FDataLen := 0;
              {aBlockNum := Inc64(aBlockNum);}
              SendPacket(KEndOfFile);
              FKermitState := tkEofReply;
              FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
            end;

          tkEofReply :
            case ProtocolStatus of
              psGotHeader :
                begin
                  FKermitState := tkCollectEof;
                  FKermitDataState := FirstDataState[FRecDataLen=0];
                  FBlockIndex := 1;
                end;
              psNoHeader :
                {Keep waiting} ;
              else
                {Timeout or other error}
                FKermitState := tkSendEof;
            end;

          tkCollectEof :
            case ProtocolStatus of
              psGotData :
                case FPacketType of
                  KAck :
                    begin
                      BlockErrors := 0;
                      if not NextFile(FPathname) then begin
                        {No more files, terminate protocol}
                        FKermitState := tkSendBreak;
                        ProtocolError := ecOK;
                      end else begin
                        ResetStatus;
                        if UpcaseFileNames then
                          AnsiStrUpper(Pathname);
                        PrepareReading;
                        if ProtocolError = ecOK then begin
                          {Read the first protocol buffer}
                          FWorkLen := SizeOf(FWorkBlock^);
                          FileOfs := 0;
                          FWorkEndPending :=
                            ReadProtocolBlock(FWorkBlock^, FWorkLen);
                          if ProtocolError = ecOK then begin
                            FileOfs := FWorkLen;
                            FLastWorkIndex := 1;
                            FTransmitInProgress := True;
                            LogFile(lfTransmitStart);
                            FKermitState := tkSendFile;
                            BlockNum := Inc64(BlockNum);
                            NewTimer(FTimer, 1);
                            TimerStarted := True;
                          end else
                            FKermitState := tkError;
                        end else
                          FKermitState := tkError;
                      end;
                    end;
                  KError :
                     FKermitState := tkError;
                  else
                    if CheckRetries then
                      FKermitState := tkError
                    else
                      FKermitState := tkSendEof;
                end;
              psNoData :
                {Keep waiting for data};
              else
                {Timeout or other error}
                FKermitState := tkSendEof;
            end;

          tkSendBreak :
            begin
              ForceStatus := True;
              FDataLen := 0;
              BlockNum := Inc64(BlockNum);
              SendPacket(KBreak);
              FKermitState := tkBreakReply;
            end;

          tkBreakReply :
            case ProtocolStatus of
              psGotHeader :
                begin
                  FKermitState := tkCollectBreak;
                  FKermitDataState := FirstDataState[FRecDataLen=0];
                  FBlockIndex := 1;
                end;
              psNoHeader :
                {Keep waiting};
              else
                {Timeout or other error}
                FKermitState := tkSendBreak;
            end;

          tkCollectBreak :
            case ProtocolStatus of
              psGotData :
                case FPacketType of
                  KAck :
                    FKermitState := tkComplete
                  else
                    FKermitState := tkError;
                end;
              psNoData :
                {Keep waiting for data};
              else
                {Timeout or other error}
                FKermitState := tkSendBreak;
            end;

          tkWaitCancel :
            if (Integer(TriggerID) = TimeoutTrigger) or
               (Integer(TriggerID) = OutBuffUsedTrigger) then      
              FKermitState := tkError;

          tkError :
            begin
              FinishReading;
              LogFile(lfTransmitFail);
              FKermitState := tkComplete;
            end;

          tkComplete :
            begin
              apShowLastStatus;
              FKermitState := tkDone;
              apSignalFinish (True);
            end;
        end;

        {Reset header state after complete headers}
        if ProtocolStatus = psGotHeader then
          FKermitHeaderState := hskNone;

        {Reset aProtocolStatus for various conditions}
        case ProtocolStatus of
          psGotHeader, psNoHeader, psGotData, psNoData :
            ProtocolStatus := psOK;
        end;

        {Stay in state machine or exit}
        case FKermitState of
          {Stay in state machine if more data ready}
          tkInitReply,
          tkCollectInit,
          tkFileReply,
          tkCollectFile,
          tkCollectBlock,
          tkEofReply,
          tkCollectEof,
          tkBreakReply,
          tkCollectBreak,
          tkWaitCancel   : Finished := not CharReady;           

          {Stay in state machine if data ready or room in table}
          tkBlockReply : if not TableFull and
                            (ProtocolStatus <> psEndFile) then
                           Finished := False
                         else
                           Finished := not CharReady;

          {Stay in state machine, interim states}
          tkInit,
          tkOpenFile,
          tkSendFile,
          tkCheckTable,
          tkSendEof,
          tkSendBreak,
          tkComplete,
          tkError        : Finished := False;

          {Leave state machine, waiting for trigger}
          tkSendData     : Finished := True;

          {Done state}
          tkDone         : Finished := True;
          else             Finished := True;
        end;

        {Store protocol status}
        SaveStatus := ProtocolStatus;

        {If staying is state machine force data ready}
        TriggerID := ApxProtocolDataTrigger;
      until Finished;

    LeaveCriticalSection(FProtSection);  
  end;

procedure TApxKermitDriver.Assign (Source : TPersistent);
begin
  inherited Assign (Source);
  if Source is TApxKermitDriver then
    with Source as TApxKermitDriver do begin
      Self.FPacketType         := FPacketType;
      Self.FKermitState        := FKermitState;
      Self.FKermitHeaderState  := FKermitHeaderState;
      Self.FKermitDataState    := FKermitDataState;
      Self.FCheckKnown         := FCheckKnown;
      Self.FLPInUse            := FLPInUse;
      Self.FUsingHibit         := FUsingHibit;
      Self.FUsingRepeat        := FUsingRepeat;
      Self.FReceiveInProgress  := FReceiveInProgress;
      Self.FTransmitInProgress := FTransmitInProgress;
      Self.FDataLen            := FDataLen;
      Self.FRecDataLen         := FRecDataLen;
      Self.FActualDataLen      := FActualDataLen;
      Self.FMinRepeatCnt       := FMinRepeatCnt;
      Self.FRecBlockNum        := FRecBlockNum;
      Self.FExpectedAck        := FExpectedAck;
      Self.FBlockCheck2        := FBlockCheck2;
      Self.FSWCTurnDelay       := FSWCTurnDelay;
      Self.FKermitOptions      := FKermitOptions;
      Self.FRmtKermitOptions   := FRmtKermitOptions;
      Self.FInBuffHead         := FInBuffHead;
      Self.FInBuffTail         := FInBuffTail;
      FWorkEndPending          := FWorkEndPending;
      FWorkLen                 := FWorkLen;
      FLastWorkIndex           := FLastWorkIndex;
      FTableSize               := FTableSize;
      FTableHead               := FTableHead;
      FTableTail               := FTableTail;
      FBlockIndex              := FBlockIndex;
      FNext2Send               := FNext2Send;
      FTempCheck               := FTempCheck;
      FC1                      := FC1;
      FC2                      := FC2;
      FC3                      := FC3;
      FSkipped                 := FSkipped;
      FGetLong                 := FGetLong;
      FLongCheck               := FLongCheck;
      FSaveCheck2              := FSaveCheck2;
      FSaveCheck               := FSaveCheck;
    end;
end;

end.

