(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                   AxLinux.pas 1.02                    *}
{*********************************************************}

{Global defines potentially affecting this unit}
{$I AxDefine.inc}

{Options required for this unit}
{$X+,F+,K+,B-}

unit AxLinux;
  {-Device layer for standard Linux communications API}

{$IFNDEF Linux}
  !! This unit is intended for use on the Linux platform only.
{$ENDIF}

interface

uses
  Classes,
  SysUtils,
  Libc,
  Types,
  AxMisc,
  AxPort,
  AxExcept,
  AxString;

type

  TApxLinuxDispatcher = class(TApxBaseDispatcher)
  protected
    function GetBaud(Baud : LongInt) : LongInt;
    function GetBaudCode(Baud : LongInt) : LongInt;
  public
    function AvailableBytes : Integer; override;
    function CloseCom : Integer; override;
    function ConfigCom : Integer; override;
    function FlushBuffer(Buffer : TAxCommBuffer) : Integer; override;
    function GetComModemStatus(var Status : DWORD) : Integer; override;
    function GetLastErr : Integer; override;
    function OpenCom(const DevName : string) : Integer; override;
    function ProcessCommunications : Integer; override;
    function ReadCom(Buf : PAnsiChar; Size : Integer) : Integer; override;
    procedure RefreshStatus; override;
    function SendComBreak(mSec : DWORD) : Integer; override;
    procedure SetThreadBoost(Boost : TAxThreadBoost); override;
    procedure StartDispatcher; override;
    procedure StopDispatcher; override;
    function WaitComEvent(Timeout : DWORD) : Integer; override;
    function WriteCom(Buf : PAnsiChar; Size : Integer) : Integer; override;
  end;

implementation

const
  NCCS = 32;
  PriorityMultiplier = 25;

type
  { TAxTermIOS }
  PAxTermIOS = ^TAxTermIOS;
  TAxTermIOS = packed record
    c_iflag    : DWORD;                         { input mode flags }
    c_oflag    : DWORD;                         { output mode flags }
    c_cflag    : DWORD;                         { control mode flags }
    c_lflag    : DWORD;                         { local mode flags }
    c_line     : Byte;                          { line discipline }
    c_cc       : array[0..NCCS-1] of AnsiChar;  { control characters }
    c_ispeed   : DWORD;                         { input speed }
    c_ospeed   : DWORD;                         { output speed }
    guardbytes : array[0..6] of Byte;           { overflow detection / protection }
  end;

const
  { control characters (c_cc) }
  VINTR    = 0;
  VQUIT    = 1;
  VERASE   = 2;
  VKILL    = 3;
  VEOF     = 4;
  VTIME    = 5;
  VMIN     = 6;
  VSWTC    = 7;
  VSTART   = 8;
  VSTOP    = 9;
  VSUSP    = 10;
  VEOL     = 11;
  VREPRINT = 12;
  VDISCARD = 13;
  VWERASE  = 14;
  VLNEXT   = 15;
  VEOL2    = 16;

const
  { The following constants were octal in termios.h -- they've been converted }
  { to hex here, because we don't have anyone old enough to remember octal <g>}


  { input mode flag bits (c_iflag) }
  IGNBRK  = $00000001;  { Ignore a break condition }
  BRKINT  = $00000002;  { Send an 'INTR' to the processes associated with this terminal }
  IGNPAR  = $00000004;  { Ignore parity or parity errs, except for 'Break' }
  PARMRK  = $00000008;  { Mark the character as a result of a parity error }
  INPCK   = $00000010;  { Enable input parity check }
  ISTRIP  = $00000020;  { Strip high bit of incoming data }
  INLCR   = $00000040;  { Incoming LF translated to CR }
  IGNCR   = $00000080;  { Ignore incoming CR }
  ICRNL   = $00000100;  { Incoming CR translated to LF }
  IUCLC   = $00000200;  { Incoming uppercase chars converted to lowercase }
  IXON    = $00000400;  { Suspend sending when 'STOP' is received }
  IXANY   = $00000800;  { Resume sending when any char is received }
  IXOFF   = $00001000;  { Send 'STOP' when buffer is nearly full }
  IMAXBEL = $00002000;  { Echo 'BEL' when input overflows }

const
  { output mode flag bits (c_oflag) }
  OPOST   = $00000001;  { Enable post processing }
  OLCUC   = $00000002;  { Convert lowercase chars to uppercase }
  ONLCR   = $00000004;  { Convert LF to CRLF }
  OCRNL   = $00000008;  { Convert CR to LF }
  ONOCR   = $00000010;  { Don't send CR if column one }
  ONLRET  = $00000020;  { Assume LF means CR and add appropriate fill }
  OFILL   = $00000040;  { Use fill characters for delay }
  OFDEL   = $00000080;  { Use 'DEL' char for delay rather than 'NUL' }

  NLDLY   = $00000100;  { Mask used to detect LF delay flags }
  NL0     = $00000000;  { No delay }
  NL1     = $00000100;  { Delay about 0.1 seconds }
  CRDLY   = $00000600;  { Mask used to detect CR delay flags }
  CR0     = $00000000;  { No delay }
  CR1     = $00000200;  { Determine delay according to current column }
  CR2     = $00000400;  { Delay about 0.1 second }
  CR3     = $00000600;  { Delay about 0.15 seconds }
  TABDLY  = $00001800;  { Mask used to detect TAB delay flags }
  TAB0    = $00000000;  { No delay }
  TAB1    = $00000800;  { Determine delay according to current column }
  TAB2    = $00001000;  { Delay about 0.1 second }
  TAB3    = $00001800;  { Expand tabs into spaces }
  XTABS   = $00001800;  { Expand tabs into spaces }
  BSDLY   = $00002000;  { Mask used to detect BS delay flags }
  BS0     = $00000000;  { No delay }
  BS1     = $00002000;  { Delay about 0.05 seconds }
  VTDLY   = $00004000;  { Mask used to detect vert tab delay flags }
  VT0     = $00000000;  { No delay }
  VT1     = $00004000;  { Delay about 2 seconds }
  FFDLY   = $00008000;  { Mask used to detect form feed delay flags }
  FF0     = $00000000;  { No delay }
  FF1     = $00008000;  { Delay about 2 seconds }

const
  { control mode flags (c_cflag) }
  CBAUD    = $0000100F;  { Mask used to detect baud rate flag }
  B0       = $00000000;  { Hang up connection, do not assert DTR }
  B50      = $00000001;  { Use 50 baud - can be used to send a break }
  B75      = $00000002;  { Use 75 baud }
  B110     = $00000003;  { Use 110 baud }
  B134     = $00000004;  { Use 134 baud }
  B150     = $00000005;  { Use 150 baud }
  B200     = $00000006;  { Use 200 baud }
  B300     = $00000007;  { Use 300 baud }
  B600     = $00000008;  { Use 600 baud }
  B1200    = $00000009;  { Use 1200 baud }
  B1800    = $0000000A;  { Use 1800 baud }
  B2400    = $0000000B;  { Use 2400 baud }
  B4800    = $0000000C;  { Use 4800 baud }
  B9600    = $0000000D;  { Use 9600 baud }
  B19200   = $0000000E;  { Use 19200 baud }
  B38400   = $0000000F;  { Use 38400 baud }
  EXTA     = B19200;     { Deprecated - replaced by B19200 }
  EXTB     = B38400;     { Deprecated - replaced by B38400 }

  CSIZE    = $00000030;  { Mask to detect character size flags }
  CS5      = $00000000;  { Use char size of 5 bits }
  CS6      = $00000010;  { Use char size of 6 bits }
  CS7      = $00000020;  { Use char size of 7 bits }
  CS8      = $00000030;  { Use char size of 8 bits }
  CSTOPB   = $00000040;  { Send two stop bits, otherwise send one }
  CREAD    = $00000080;  { Enable receiver, otherwise don't recieve chars }
  PARENB   = $00000100;  { Enable parity generation and detection }
  PARODD   = $00000200;  { If set, odd parity -- otherwise even parity }
  HUPCL    = $00000400;  { Disconnect line when last process closes device - drop DTR }
  CLOCAL   = $00000800;  { Assume local control with no control from modem }
  CBAUDEX  = $00001000;  { Mask to detect extended baud rates }

  B57600   = $00001001;  { Use 57600 baud }
  B115200  = $00001002;  { Use 115200 baud }
  B230400  = $00001003;  { Use 230400 baud }
  B460800  = $00001004;  { Use 460800 baud }
  B500000  = $00001005;  { Use 500000 baud }
  B576000  = $00001006;  { Use 576000 baud }
  B921600  = $00001007;  { Use 921600 baud }
  B1000000 = $00001008;  { Use 1000000 baud }
  B1152000 = $00001009;  { Use 1152000 baud }
  B1500000 = $0000100A;  { Use 1500000 baud }
  B2000000 = $0000100B;  { Use 2000000 baud }
  B2500000 = $0000100C;  { Use 2500000 baud }
  B3000000 = $0000100D;  { Use 3000000 baud }
  B3500000 = $0000100E;  { Use 3500000 baud }
  B4000000 = $0000100F;  { Use 4000000 baud }

  CIBAUD   = $100F0000;  { input baud rate (not used) }
  CMSPAR   = $40000000;  { mark or space (stick) parity }
  CRTSCTS  = $80000000;  { flow control }

const
  { local mode flags }
  ISIG     = $00000001;  { Enable signals }
  ICANON   = $00000002;  { Perform canonical processing }
  XCASE    = $00000004;  { Perform weird processing to translate characters }
  ECHO     = $00000008;  { Enable echoing of characters as sent }
  ECHOE    = $00000010;  { Enable erase characters as BS-SP-BS }
  ECHOK    = $00000020;  { Echo a LF after a kill character }
  ECHONL   = $00000040;  { Echo a LF even if other echoing is suppressed }
  NOFLSH   = $00000080;  { Disable a flush for the INTR, QUIT, SUSP chars }
  TOSTOP   = $00000100;  { Send SIGTTOU to process running in background }
  ECHOCTL  = $00000200;  { Echo control characters in caret format }
  ECHOPRT  = $00000400;  { Echo erase character as character erased }
  ECHOKE   = $00000800;  { BS-SP-BS erase entire line on line kill }
  FLUSHO   = $00001000;  { Output is being flushed. Data written to terminal is discarded }
  PENDIN   = $00004000;  { Retype pending input at next read }
  IEXTEN   = $00008000;  { Enable extended functions }

const
  { tcFlow Action }
  TCOOFF    = 0;
  TCOON     = 1;
  TCIOFF    = 2;
  TCION     = 3;

  { tcFlush QueueSelector }
  TCIFLUSH  = 0;
  TCOFLUSH  = 1;
  TCIOFLUSH = 2;

  { tcSetAttr OptActions }
  TCSANOW   = 0;
  TCSADRAIN = 1;
  TCSAFLUSH = 2;

const
  { Modem lines }
  TIOCM_LE   = $0001;             { Line Enable }
  TIOCM_DTR  = $0002;             { Data Terminal Ready }
  TIOCM_RTS  = $0004;             { Request to Send }
  TIOCM_ST   = $0008;             { Secondary Transmit }
  TIOCM_SR   = $0010;             { Secondary Receive }
  TIOCM_CTS  = $0020;             { Clear to Send }
  TIOCM_CAR  = $0040;             { Carrier Detect }
  TIOCM_RNG  = $0080;             { Ring }
  TIOCM_DSR  = $0100;             { Data Set Ready }
  TIOCM_CD   = TIOCM_CAR;         { Carrier Detect }
  TIOCM_RI   = TIOCM_RNG;         { Ring }
  TIOCM_OUT1 = $2000;
  TIOCM_OUT2 = $4000;
  TIOCM_LOOP = $8000;

const
  { Line disciplines }
  N_TTY          = 0;
  N_SLIP         = 1;
  N_MOUSE        = 2;
  N_PPP          = 3;
  N_STRIP        = 4;
  N_AX25         = 5;
  N_X25          = 6;   { X.25 async }
  N_6PACK        = 7;
  N_MASC         = 8;   { Reserved for Mobitex module <kaz@cafe.net> }
  N_R3964        = 9;   { Reserved for Simatic R3964 module }
  N_PROFIBUS_FDL = 10;	{ Reserved for Profibus <Dave@mvhi.com> }
  N_IRDA         = 11;  { Linux IR - http://www.cs.uit.no/~dagb/irda/irda.html }
  N_SMSBLOCK     = 12;  { SMS block mode - for talking to GSM data cards about SMS messages }
  N_HDLC         = 13;  { synchronous HDLC }
  N_SYNC_PPP     = 14;  { synchronous PPP }

const
  { IOCTL Commands }
  TCGETS         = $5401;
  TCSETS         = $5402;
  TCSETSW        = $5403;
  TCSETSF        = $5404;
  TCGETA         = $5405;
  TCSETA         = $5406;
  TCSETAW        = $5407;
  TCSETAF        = $5408;
  TCSBRK         = $5409;
  TCXONC         = $540A;
  TCFLSH         = $540B;
  TIOCEXCL       = $540C;
  TIOCNXCL       = $540D;
  TIOCSCTTY      = $540E;
  TIOCGPGRP      = $540F;
  TIOCSPGRP      = $5410;
  TIOCOUTQ       = $5411;
  TIOCSTI        = $5412;
  TIOCGWINSZ     = $5413;
  TIOCSWINSZ     = $5414;
  TIOCMGET       = $5415;
  TIOCMBIS       = $5416;
  TIOCMBIC       = $5417;
  TIOCMSET       = $5418;
  TIOCGSOFTCAR   = $5419;
  TIOCSSOFTCAR   = $541A;
  FIONREAD       = $541B;
  TIOCINQ        = FIONREAD;
  TIOCLINUX      = $541C;
  TIOCCONS       = $541D;
  TIOCGSERIAL    = $541E;
  TIOCSSERIAL    = $541F;
  TIOCPKT        = $5420;
  FIONBIO        = $5421;
  TIOCNOTTY      = $5422;
  TIOCSETD       = $5423;
  TIOCGETD       = $5424;
  TCSBRKP        = $5425;  { Needed for POSIX tcsendbreak() }
  TIOCTTYGSTRUCT = $5426;  { For debugging only }
  TIOCSBRK       = $5427;  { BSD compatibility }
  TIOCCBRK       = $5428;  { BSD compatibility }
  TIOCGSID       = $5429;  { Return the session ID of FD }
//  TIOCGPTN  _IOR('T',0x30, unsigned int) /* Get Pty Number (of pty-mux device) */
//  TIOCSPTLCK  _IOW('T',0x31, int)  /* Lock/unlock Pty */

{ Return the output baud rate stored in TermIOS }
function cfGetOSpeed (const TermIOS : PAxTermIOS) : Integer; cdecl;
  external libcmodulename name 'cfgetospeed';
{ Return the input baud rate stored in TermIOS }
function cfGetISpeed (const TermIOS : PAxTermIOS) : Integer; cdecl;
  external libcmodulename name 'cfgetispeed';
{ Set the output baud rate stored in TermIOS to SPEED }
function cfSetOSpeed (TermIOS : PAxTermIOS; Speed : Integer) : Integer; cdecl;
  external libcmodulename name 'cfsetospeed';
{ Set the input baud rate stored in TermIOS to SPEED }
function cfSetISpeed (TermIOS : PAxTermIOS; Speed : Integer) : Integer; cdecl;
  external libcmodulename name 'cfsetispeed';
{ Put the state of FD into TermIOS }
function tcGetAttr (Handle : Integer; TermIOS : PAxTermIOS) : Integer; cdecl;
  external libcmodulename name 'tcgetattr';
{ Set the state of FD to TermIOS }
function tcSetAttr (Handle : Integer; OptActions : Integer;
  const TermIOS : PAxTermIOS) : Integer; cdecl;
  external libcmodulename name 'tcsetattr';
{ Send zero bits on Handle }
function tcSendBreak (Handle : Integer; Duration : Integer) : Integer; cdecl;
  external libcmodulename name 'tcsendbreak';
{ Wait for pending output to be written on FD }
function tcDrain (Handle : Integer) : Integer; cdecl;
  external libcmodulename name 'tcdrain';
{ Flush pending data on FD }
function tcFlush (Handle : Integer; QueueSelector : Integer) : Integer; cdecl;
  external libcmodulename name 'tcflush';
{ Suspend or restart transmission on FD }
function tcFlow (Handle : Integer; Action : Integer) : Integer; cdecl;
  external libcmodulename name 'tcflow';



{ Return number of bytes available from serial driver }
function TApxLinuxDispatcher.AvailableBytes : Integer;
var
  Res : Integer;
begin
  Result := ioctl(ComHandle, FIONREAD, @Res);
  if Result <> -1 then
    Result := Res;
end;

{ Close the COM port and cleanup }
function TApxLinuxDispatcher.CloseCom : Integer;
begin
  Result := __close(ComHandle);
end;

{ Configure line parameters }
function TApxLinuxDispatcher.ConfigCom : Integer;
var
  TIOS : TAxTermIOS;
  Stat : Integer;
begin
  FillChar(TIOS, SizeOf(TIOS) - SizeOf(TIOS.guardbytes), #0);
  FillChar(TIOS.guardbytes, SizeOf(TIOS.guardbytes), #204);

  { Get current TermIOS }
  Result := tcGetAttr(ComHandle, @TIOS);
  if Result <> 0 then Exit;

  { Simple check for overflow }
  Assert(TIOS.guardbytes[0] = $CC);

  { Initial setup for stuff we care about }

  { Line discipline }
  TIOS.c_line := N_TTY;

  { Control Options }
  TIOS.c_cflag := TIOS.c_cflag or (CLOCAL or CREAD);

  { Control Characters }
  TIOS.c_cc[VTIME] := #0;
  TIOS.c_cc[VMIN] := #1;

  { Select Non-Canonical (Raw) Mode }
  TIOS.c_lflag := TIOS.c_lflag and not (ICANON or ECHO or ECHOE or ISIG);

  { Set Parity Checking }
  TIOS.c_iflag := TIOS.c_iflag or (INPCK or ISTRIP);
  TIOS.c_iflag := TIOS.c_iflag and not(IGNPAR or PARMRK);

  { Set Break Flags }
  TIOS.c_iflag := TIOS.c_iflag and not IGNBRK;
  TIOS.c_iflag := TIOS.c_iflag or BRKINT;

  { Disable Input Character Conversions }
  TIOS.c_iflag := TIOS.c_iflag and not (ICRNL or INLCR);
  TIOS.c_iflag := TIOS.c_iflag and not (IUCLC or IMAXBEL or IGNCR);

  { Select Raw Output }
  TIOS.c_oflag := TIOS.c_oflag and not OPOST;

  { Set Baud }
  cfSetISpeed (@TIOS, GetBaudCode(Owner.Baud));
  cfSetOSpeed (@TIOS, GetBaudCode(Owner.Baud));

  { Databits }
  TIOS.c_cflag := TIOS.c_cflag and not(CSIZE);
  case Owner.DataBits of
    dbFive  : TIOS.c_cflag := TIOS.c_cflag or CS5;
    dbSix   : TIOS.c_cflag := TIOS.c_cflag or CS6;
    dbSeven : TIOS.c_cflag := TIOS.c_cflag or CS7;
    dbEight : begin
                TIOS.c_cflag := TIOS.c_cflag or CS8;
                TIOS.c_iflag := TIOS.c_iflag and not (ISTRIP);
              end;
  end;

  { Parity }
  case Owner.Parity of
    pNone  : begin
               TIOS.c_cflag := TIOS.c_cflag and not(PARENB);
             end;
    pOdd   : begin
               TIOS.c_cflag := TIOS.c_cflag or PARENB;
               TIOS.c_cflag := TIOS.c_cflag or PARODD;
             end;
    pEven  : begin
               TIOS.c_cflag := TIOS.c_cflag or PARENB;
               TIOS.c_cflag := TIOS.c_cflag and not(PARODD);
             end;
    pMark  : begin
               TIOS.c_cflag := TIOS.c_cflag or PARENB;
               TIOS.c_cflag := TIOS.c_cflag or CMSPAR;
               TIOS.c_cflag := TIOS.c_cflag or PARODD;
             end;
    pSpace : begin
               TIOS.c_cflag := TIOS.c_cflag or PARENB;
               TIOS.c_cflag := TIOS.c_cflag or CMSPAR;
               TIOS.c_cflag := TIOS.c_cflag and not(PARODD);
             end;
  end;

  { Stopbits }
  if Owner.StopBits = sbOne then
    TIOS.c_cflag := TIOS.c_cflag and not CSTOPB
  else
    TIOS.c_cflag := TIOS.c_cflag or CSTOPB;

  { HWFlowControl }

  { Get control line status }
  ioctl(ComHandle, TIOCMGET, @Stat);

  if Owner.HWFlowControl = True then begin
    TIOS.c_cflag := TIOS.c_cflag or CRTSCTS;
  end else begin
    TIOS.c_cflag := TIOS.c_cflag and not(CRTSCTS);

    { RTS }
    if Owner.RTS then begin
      Stat := Stat or TIOCM_RTS;
    end else begin
      Stat := Stat and not TIOCM_RTS;
    end;
  end;

  { DTR } 
  if Owner.DTR then begin
    Stat := Stat or TIOCM_DTR;
  end else begin
    Stat := Stat and not TIOCM_DTR;
  end;

  { Set control line status }
  ioctl(ComHandle, TIOCMSET, @Stat);

  { SWFlowControl }

  { Clear all flags first }
  TIOS.c_iflag := TIOS.c_iflag and not(IXON or IXOFF or IXANY);

  case Owner.SWFlowControl of
    swfNone :
      begin
        { Do nothing, leave flags cleared }
      end;

    swfReceive :
      begin
        { Set flags for receive flow control }
        TIOS.c_iflag := TIOS.c_iflag or (IXOFF or IXANY);
      end;

    swfTransmit :
      begin
        { Set flags for transmit flow control }
        TIOS.c_iflag := TIOS.c_iflag or (IXON);
      end;

    swfBoth :
      begin
        { Set flags for receive and transmit flow control }
        TIOS.c_iflag := TIOS.c_iflag or (IXON or IXOFF or IXANY);
      end;
  end;

  { XOffChar }
  TIOS.c_cc[VSTOP] := Owner.XOffChar;

  { XOnChar }
  TIOS.c_cc[VSTART] := Owner.XOnChar;

  Result := tcSetAttr(ComHandle, TCSADRAIN, @TIOS);
end;

{ Flush the input or output buffer }
function TApxLinuxDispatcher.FlushBuffer(Buffer : TAxCommBuffer) : Integer;
begin
  Result := inherited FlushBuffer(Buffer);
  case Buffer of
    cbInput : Result := tcflush(FComHandle, TCIFLUSH);
    cbOutput : Result := tcflush(FComHandle, TCOFLUSH);
  end;
end;

{ Return Baud from BaudCode }
function TApxLinuxDispatcher.GetBaud(Baud : LongInt) : LongInt;
begin
  if (Baud and CBAUDEX) = 0 then begin
    case Baud of
      B0     : Result := 0;
      B50    : Result := 50;
      B75    : Result := 75;
      B110   : Result := 110;
      B134   : Result := 134;
      B150   : Result := 150;
      B200   : Result := 200;
      B300   : Result := 300;
      B600   : Result := 600;
      B1200  : Result := 1200;
      B1800  : Result := 1800;
      B2400  : Result := 2400;
      B4800  : Result := 4800;
      B9600  : Result := 9600;
      B19200 : Result := 19200;
      B38400 : Result := 38400;
    else
      Result := -1;
    end;
  end else begin
    case Baud of
      B57600   : Result := 57600;
      B115200  : Result := 115200;
      B230400  : Result := 230400;
      B460800  : Result := 460800;
      B500000  : Result := 500000;
      B576000  : Result := 576000;
      B921600  : Result := 921600;
      B1000000 : Result := 1000000;
      B1152000 : Result := 1152000;
      B1500000 : Result := 1500000;
      B2000000 : Result := 2000000;
      B2500000 : Result := 2500000;
      B3000000 : Result := 3000000;
      B3500000 : Result := 3500000;
      B4000000 : Result := 4000000;
    else
      Result := -1;
    end;
  end;
end;

{ Return BaudCode for use in TermIOS structure }
function TApxLinuxDispatcher.GetBaudCode(Baud : LongInt) : LongInt;
begin
  if Baud > 460800 then begin
    { High range -- USB, maybe? }
    case Baud of
      3000001..3500000 : Result := B3500000;
      2500001..3000000 : Result := B3000000;
      2000001..2500000 : Result := B2500000;
      1500001..2000000 : Result := B2000000;
      1152001..1500000 : Result := B1500000;
      1000001..1152000 : Result := B1152000;
      921601..1000000  : Result := B1000000;
      576001..921600   : Result := B921600;
      500001..576000   : Result := B576000;
      460801..500000   : Result := B500000;
    else
      Result := B4000000;  { Highest constant currently available }
    end;
  end else begin
    { Most likely range for most situations }
    if Baud > 9600 then begin
      case Baud of
        230401..460800 : Result := B460800;
        115201..230400 : Result := B230400;
        57601..115200  : Result := B115200;
        38401..57600   : Result := B57600;
        19201..38400   : Result := B38400;
      else
        Result := B19200;
      end;
    end else begin
      { Low range, somewhat unlikely these days }
      case Baud of
        4801..9600 : Result := B9600;
        2401..4800 : Result := B4800;
        1801..2400 : Result := B2400;
        1201..1800 : Result := B1800;
        601..1200  : Result := B1200;
        301..600   : Result := B600;
        201..300   : Result := B300;
        151..200   : Result := B200;
        135..150   : Result := B150;
        111..134   : Result := B134;
        76..110    : Result := B110;
        51..75     : Result := B75;
        1..50      : Result := B50;
      else
        Result := B0;
      end;
    end;
  end;
end;

{ Get status of CTS, DSR, RI, DCD }
function TApxLinuxDispatcher.GetComModemStatus(var Status : DWORD) : Integer;
var
  Stat : Integer;
begin
  { Zero Status }
  Status := 0;

  { Get Status }
  Result := ioctl(ComHandle, TIOCMGET, @Stat);

  { Check CTS, translate to Windows flag }
  if (TIOCM_CTS and Stat) <> 0 then
    Status := Status or CTSMask;

  { Check DSR, translate to Windows flag }
  if (TIOCM_LE and Stat) <> 0 then
    Status := Status or DSRMask;

  { Check RI, translate to Windows flag }
  if (TIOCM_RNG and Stat) <> 0 then
    Status := Status or RIMask;

  { Check DCD, translate to Windows flag }
  if (TIOCM_CAR and Stat) <> 0 then
    Status := Status or DCDMask;
end;

{ Return the last error number }
function TApxLinuxDispatcher.GetLastErr : Integer;
begin
  Result := errno;
end;

{ Open the COM port specified by DevName }
function TApxLinuxDispatcher.OpenCom(const DevName : string): Integer;
begin
  Result := Open(PAnsiChar(DevName), O_RDWR or O_NOCTTY or O_NDELAY);
end;

{ Communications are running in separate threads -- give them a chance }
function TApxLinuxDispatcher.ProcessCommunications : Integer;
var
  Delay, Rem : TTimeSpec;
begin
  Result := ecOk;
  Delay.tv_sec := 0;
  Delay.tv_nsec := 10000000;           { 10,000,000 nSec delay (10 mSec) }
  FillChar(Rem, SizeOf(Rem), #0);
  nanosleep(Delay, @Rem);
end;

{ Read Size bytes from the COM port }
function TApxLinuxDispatcher.ReadCom(Buf : PAnsiChar; Size : Integer) : Integer;
begin
  Result := __read(ComHandle, Buf^, Size);
end;

{ Refresh COM status }
procedure TApxLinuxDispatcher.RefreshStatus;
begin
  { }
end;

{ Sends a break }
function TApxLinuxDispatcher.SendComBreak(mSec : DWORD) : Integer;
var
  Duration : Integer;
begin
  { Convert mSecs to jiffies }
  Duration := mSec div 10;
  if (mSec mod 10) >= 5 then Inc(Duration);

  { Send the break for the desired time }
  Result := tcSendBreak(ComHandle, Duration);
end;

{ Sets thread priority }
{ With the default schedule policy, this has no effect. This code is in }
{ place in case Borland provides a way to change schedule policy (which }
{ will require root in any case). }
procedure TApxLinuxDispatcher.SetThreadBoost(Boost : TAxThreadBoost);
begin
  if Boost <> ThreadBoost then begin
    ThreadBoost := Boost;

    if Assigned(ReadThread) then
      ReadThread.Priority := Ord(Boost) * PriorityMultiplier;

    if Assigned(WriteThread) then
      WriteThread.Priority := Ord(Boost) * PriorityMultiplier;
  end;
end;

{ Start dispatcher processing }
procedure TApxLinuxDispatcher.StartDispatcher;
begin
  DispActive := True;
  WriteThread := TWriteThread.Create(Self);
  ReadThread := TReadThread.Create(Self);
end;

{ Stop dispatcher processing }
procedure TApxLinuxDispatcher.StopDispatcher;
begin
  DispActive := False;
  if Assigned(WriteThread) then begin
    WriteThread.Terminate;
    OutputEvent.SetEvent;
  end;
  if Assigned(ReadThread) then begin
    ReadThread.Terminate;
  end;
end;

{ Block until a comms event or timeout occurs }
function TApxLinuxDispatcher.WaitComEvent(Timeout : DWORD) : Integer;
var
  PFD : TPollFD;
begin
  PFD.fd := ComHandle;
  PFD.events := POLLIN or POLLMSG;
  PFD.revents := 0;
  Result := poll(@PFD, 1, Timeout);
end;

{ Write data to the COM port }
function TApxLinuxDispatcher.WriteCom(Buf: PAnsiChar; Size: Integer): Integer;
begin
  Result := __write(ComHandle, Buf^, Size);
end;

end.
