(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                  AxModem.pas 1.02                     *}
{*********************************************************}

{Global defines potentially affecting this unit}
{$I AxDefine.inc}

unit AxModem;

interface

uses
  SysUtils, Types, Classes,
  Qt, QGraphics, QControls, QForms, QDialogs, QTypes,
  AxMisc, AxSystem, AxPort, AxPacket, AxLibMdm, AxExcept;

const
  ApxDefModemCapFolder = '{MODEMCAPINDEX}';
  ApxDefModemStatusCaption = 'Modem status';
  ApxDefOKResponse = 'OK'#13#10;
  ApxDefErrorResponse = 'ERROR'#13#10;
  ApxDefBusyResponse = 'BUSY'#13#10;
  ApxDefConnectResponse = 'CONNECT';
  ApxDefRingResponse = 'RING'#13#10;
  ApxDefModemEscape = '+++';
  ApxDefAnswerCommand = 'ATA'#13;
  ApxDefHangupCmd = 'ATH0'#13;
  ApxDefCommandTimeout = 30000;  { 30 second timeout waiting for modem to respond }
  ApxDefConnectTimeout = 60000;  { 60 second timeout waiting for modems to negotiate }
  ApxDefDTRTimeout = 1000;       { 1 second timeout for the modem to hangup after dropping DTR }
  ApxModemConfigVersion = '1.00';

type
  { predefine our class }
  TApxCustomModem = class;
  TApxAbstractModemStatus = class;

  TApxModemState = (
    msUnknown,               { Hasn't been initialized }
    msIdle,                  { Idle and ready }
    msInitializing,          { Starting initialization process }
    msAutoAnswerBackground,  { AutoAnswer, no rings received }
    msAutoAnswerWait,        { AutoAnswer, waiting for Nth ring }
    msAnswerWait,            { Answering call, waiting for connect }
    msDial,                  { Sending Dial command }
    msConnectWait,           { Sent the Dial command, wait for connect }
    msConnected,             { Done with connection process }
    msHangup,                { Starting hangup process }
    msCancel                 { Starting cancel process }
  );

  TApxModemLogCode = (
    mlNone,                  { None, nothing to log }
    mlDial,                  { Dialing }
    mlAutoAnswer,            { Initiated AutoAnswer }
    mlAnswer,                { Answering an incoming call }
    mlConnect,               { Connected }
    mlCancel,                { Call cancelled }
    mlBusy,                  { Called number was busy }
    mlConnectFail            { Connection attempt failed }
  );

  { used for the UpdateStatus method }
  TApxModemStatusAction = (
    msaStart,                { first time status display (clears everything) }
    msaClose,                { last time, cleans up }
    msaUpdate,               { normal updating }
    msaDetailReplace,        { replaces last line of details }
    msaClear                 { clears all details and adds DetailStr }
  );

  TApxModemSpeakerVolume = (svLow, svMed, svHigh);
  TApxModemSpeakerMode = (smOff, smOn, smDial);
  TApxModemFlowControl = (fcOff, fcHard, fcSoft);
  TApxModemErrorControl = (ecOff, ecOn, ecForced, ecCellular);
  TApxModemModulation = (smBell, smCCITT, smCCITT_V23);

  TApxModemConfig = record
    ConfigVersion : string[8];       { version tag to support future features }
    { port settings }
    AttachedTo : string[20];
    Manufacturer : string[100];
    ModemName : string[100];
    ModemModel : string[100];
    DataBits : TAxDataBits;
    Parity : TAxParity;
    StopBits : TAxStopBits;
    { speaker options }
    SpeakerVolume :  TApxModemSpeakerVolume;
    SpeakerMode : TApxModemSpeakerMode;
    { connection control }
    FlowControl : TApxModemFlowControl;
    ErrorControl : set of TApxModemErrorControl;
    Compression : Boolean;
    Modulation : TApxModemModulation;
    ToneDial : Boolean;
    BlindDial : Boolean;
    CallSetupFailTimeout : Integer;
    InactivityTimeout : Integer;
    { extra commands }
    ExtraSettings : string[50];
    Padding : Array[81..128] of Byte;  { Expansion room }
  end;

  TApxModemNameProp = class(TPersistent)
  private
    FManufacturer: string;
    FName: string;
    FModemFile: string;
    procedure SetManufacturer(const Value: string);
    procedure SetName(const Value: string);
    procedure SetModemFile(const Value: string);
  published
    property Manufacturer : string
      read FManufacturer write SetManufacturer;
    property Name : string
      read FName write SetName;
    property ModemFile : string
      read FModemFile write SetModemFile;
  end;

  TApxCallerIDInfo = record
    HasData : Boolean;
    Date   : string;
    Time   : string;
    Number : string;
    Name   : string;
    Msg    : string;
  end;

  { event types }
  TModemCallerIDEvent = procedure(Modem : TApxCustomModem;
    CallerID : TApxCallerIDInfo) of object;
  TModemNotifyEvent = procedure(Modem : TApxCustomModem) of object;
  TModemLogEvent = procedure(Modem : TApxCustomModem;
    LogCode : TApxModemLogCode) of object;
  TModemStatusEvent = procedure(Modem : TApxCustomModem;
    ModemState : TApxModemState) of object;

  TApxCustomModem = class(TApxBaseHandleComponent)
  private
    FAnswerOnRing: Integer;
    FBPSRate: DWORD;
    FComPort: TApxCustomComPort;
    FDialTimeout: Integer;
    FFailCode: Integer;
    FModemCapFolder: string;
    FRingWaitTimeout: DWORD;
    FRingCount: Integer;
    FStatusDisplay: TApxAbstractModemStatus;
    FSelectedDevice: TApxModemNameProp;
    FModemState: TApxModemState;
    FNegotiationResponses : TStringList;
    FOnModemCallerID: TModemCallerIDEvent;
    FOnModemLog: TModemLogEvent;
    FOnModemDisconnect: TModemNotifyEvent;
    FOnModemConnect: TModemNotifyEvent;
    FOnModemFail: TModemNotifyEvent;
    FOnModemStatus: TModemStatusEvent;
    FConnected: Boolean;
    FPhoneNumber: string;
    FStartTime : DWORD;
    FDeviceSelected: Boolean;
    FModemConfig : TApxModemConfig;
    FCallerIDInfo : TApxCallerIDInfo;

    function GetElapsedTime : DWORD;
    function GetNegotiationResponses: TStringList;

    procedure SetAnswerOnRing(const Value: Integer);
    procedure SetComPort(const Value: TApxCustomComPort);
    procedure SetDialTimeout(const Value: Integer);
    procedure SetModemCapFolder(const Value: string);
    procedure SetRingWaitTimeout(const Value: DWORD);
    procedure SetSelectedDevice(const Value: TApxModemNameProp);
    procedure SetStatusDisplay(const Value: TApxAbstractModemStatus);
  protected
    { Protected declarations }
    ResponsePacket : TApxDataPacket;
    Initialized : Boolean;
    PassthroughMode : Boolean;
    WaitingForResponse : Boolean;
    OKResponse : Boolean;
    ErrorResponse : Boolean;
    ConnectResponse : Boolean;
    TimedOut : Boolean;
    LastCommand : string;
    DcdTrigger : Word;
    StatusTimerTrigger : Word;
    FCallerIDProvided : Boolean;

    { opens port and ensures we are ready }
    procedure CheckReady;
    { generate the OnModemCallerID event }
    procedure DoCallerID;
    { generate the OnModemConnect event }
    procedure DoConnect;
    { generate the OnModemDisconnect event }
    procedure DoDisconnect;
    { generate the OnModemFail event }
    procedure DoFail(Failure : Integer);
    { generate the OnModemLog event }
    procedure DoLog(LogCode: TApxModemLogCode);
    { generate the OnModemStatus event }
    procedure DoStatus(NewModemState: TApxModemState);

    { initialize/configure the modem }
    procedure Initialize;

    { do stuff once we've been loaded }
    procedure Loaded; override;
    { do stuff when other components are added to the form }
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    { create the handle }
    procedure CreateWidget; override;
    { the AxModem message handler }
    procedure APXModemMessage(var Message : TAxMessage); message apx_CommandProcessed;
    { the AxModem event filter }
    function EventFilter(Sender : QObjectH; Event : QEventH) : Boolean; override;
    { add triggers to detect connection state }
    procedure PrepForConnect(EnableTriggers : Boolean);
    { detect modem responses }
    procedure ResponseStringPacket(Sender: TObject; Data: String);
    { detect timeouts }
    procedure ResponseTimeout(Sender : TObject);
    { status trigger notification event }
    procedure TriggerEvent(CP: TObject; TriggerHandle: Word);
    { send all commands in the list }
    function SendCommands(Commands : TList) : Boolean;
    { check the responses for the response }
    function CheckResponses(const Response, DefResponse : string;
      Responses : TList) : Boolean;
    { check the response for any errors }
    function CheckErrors(const Response : string) : Integer;
    { check for the CallerID tags }
    procedure CheckCallerID(const Response  : string);

  public
    { Public declarations }
    LmModem : TLmModem;
    LibModem : TApxLibModem;

    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    class function GetLogString(const D1, D2, D3 : DWORD) : string; override;

    property AnswerOnRing : Integer
      read FAnswerOnRing write SetAnswerOnRing
      default 2;
    property BPSRate : DWORD
      read FBPSRate;
    property CallerIDInfo : TApxCallerIDInfo
      read FCallerIDInfo;
    property ComPort : TApxCustomComPort
      read FComPort write SetComPort;
    property Connected : Boolean
      read FConnected;
    property DeviceSelected : Boolean
      read FDeviceSelected;
    property DialTimeout : Integer
      read FDialTimeout write SetDialTimeout
      default 60;
    property ElapsedTime : DWORD
      read GetElapsedTime;
    property FailureCode : Integer
      read FFailCode;
    property ModemCapFolder : string
      read FModemCapFolder write SetModemCapFolder;
    property ModemState : TApxModemState
      read FModemState;
    property NegotiationResponses : TStringList
      read GetNegotiationResponses;
    property PhoneNumber : string
      read FPhoneNumber;
    property RingCount : Integer
      read FRingCount;
    property RingWaitTimeout : DWORD
      read FRingWaitTimeout write SetRingWaitTimeout
      default 1200;
    property SelectedDevice : TApxModemNameProp
      read FSelectedDevice write SetSelectedDevice;
    property StatusDisplay : TApxAbstractModemStatus
      read FStatusDisplay write SetStatusDisplay;

    procedure AutoAnswer;
    procedure CancelCall;
    procedure ConfigAndOpen;
    function DefaultDeviceConfig : TApxModemConfig;
    procedure Dial(const ANumber : string);
    function FailureCodeMsg(const FailureCode : Integer) : string;
    function GetDevConfig : TApxModemConfig;
    function ModemLogToString(LogCode : TApxModemLogCode) : string;
    function ModemStatusMsg(Status : TApxModemState) : string;
    function SelectDevice : Boolean;
    function SendCommand(const Command : string) : Boolean;
    procedure SetDevConfig(const Config : TApxModemConfig);
    function ShowConfigDialog : Boolean;

    { undocumented }
    function ConvertXML(const S : string) : string;

    property OnModemCallerID : TModemCallerIDEvent
      read FOnModemCallerID write FOnModemCallerID;
    property OnModemConnect : TModemNotifyEvent
      read FOnModemConnect write FOnModemConnect;
    property OnModemDisconnect : TModemNotifyEvent
      read FOnModemDisconnect write FOnModemDisconnect;
    property OnModemFail : TModemNotifyEvent
      read FOnModemFail write FOnModemFail;
    property OnModemLog : TModemLogEvent
      read FOnModemLog write FOnModemLog;
    property OnModemStatus : TModemStatusEvent
      read FOnModemStatus write FOnModemStatus;
  end;

  TApxModem = class(TApxCustomModem)
  published
    property AnswerOnRing;
    property ComPort;
    property DialTimeout;
    property ModemCapFolder;
    property RingWaitTimeout;
    property SelectedDevice;
    property StatusDisplay;

    property OnModemCallerID;
    property OnModemConnect;
    property OnModemDisconnect;
    property OnModemFail;
    property OnModemLog;
    property OnModemStatus;
  end;

  TApxAbstractModemStatus = class(TApxBaseComponent)
  private
    FStatusDialog: TForm;
    FCaption: string;
    FStarted: Boolean;
    FModem: TApxCustomModem;
    procedure SetCaption(const Value: string);
    procedure SetStarted(Start : Boolean);
    procedure SetModem(const Value: TApxCustomModem);
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    property StatusDialog : TForm
      read FStatusDialog write FStatusDialog;
    property Caption : string
      read FCaption write SetCaption;
    property Modem : TApxCustomModem
      read FModem write SetModem;
    property Started : Boolean
      read FStarted;
    procedure UpdateDisplay(Modem : TApxCustomModem;
      const StatusStr, TimeStr, DetailStr : string;
      Action : TApxModemStatusAction);
  end;

  TApxModemStatus = class(TApxAbstractModemStatus)
  published
    property Caption;
    property Modem;
  end;


implementation

uses
  AxMdmCfg, AxMdmDlg;


const
  cclSelect                = DWORD($800000A1);
  cclSelectedMdmMfg        = DWORD($000000A2);
  cclSelectedMdmName       = DWORD($000000A3);
  cclSelectedMdmFile       = DWORD($000000A4);

  cclConfigAndOpen         = $00000003;
  cclDial                  = $00000004;
  cclAutoAnswer            = $00000005;
  cclCancelCall            = $00000006;
  cclRing                  = $00000007;
  cclAnswer                = $00000008;
  cclCallerID              = $00000009;
  cclConnected             = $00000010;
  cclDisconnected          = $00000011;
  cclFail                  = $00000012;
  cclConfigChange          = $00000013;
  cclInitialize            = $00000014;
  cclBusyResponse          = $00000015;

  cclEchoResponse          = $00000020;
  cclOKResponse            = $00000021;
  cclErrorResponse         = $00000022;
  cclTimeout               = $00000023;
  cclInformativeResponse   = $00000024;
  cclUnknownResponse       = $00000025;
  cclConnectResponse       = $00000026;

{ TApxModemNameProp }

procedure TApxModemNameProp.SetManufacturer(const Value: string);
  { write access method for Manufacturer property }
begin
  if FManufacturer <> Value then begin
    FManufacturer := Value;
  end;
end;

procedure TApxModemNameProp.SetModemFile(const Value: string);
begin
  FModemFile := Value;
end;

procedure TApxModemNameProp.SetName(const Value: string);
  { write access method for Name property }
begin
  FName := Value;
end;

{ TApxCustomModem }

procedure TApxCustomModem.APXModemMessage(var Message: TAxMessage);
begin
  case Message.WParam of
    ApxMessage_AutoAnswer :
      begin
        { got the message to answer the call... }
        PrepForConnect(True);
        FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclAnswer, 0, 0);
        if not SendCommands(LmModem.Answer) then
          DoFail(ecModemRejectedCommand);
      end;
    ApxMessage_CancelCall :
      begin
        CancelCall;
      end;
    ApxMessage_StartDial :
      begin
        ResponsePacket.Enabled := True;
        FComPort.Output := ConvertXML(LmModem.Settings.Prefix +
                                      LMModem.Settings.DialPrefix +
                                      FPhoneNumber +
                                      LmModem.Settings.Terminator);
        DoStatus(msConnectWait);
      end;
  end;
end;

procedure TApxCustomModem.AutoAnswer;
  { initiate auto answer mode }
begin
  CheckReady;
  FCallerIDProvided := False;
  if not Initialized then
    Initialize;
  FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclAutoAnswer, FAnswerOnRing, 0);
  { turn on the CallerID detection }
  SendCommands(LmModem.Voice.EnableCallerID);

  DoStatus(msAutoAnswerBackground);
  ResponsePacket.Timeout := 0;
  ResponsePacket.Enabled := True;
end;

procedure TApxCustomModem.CancelCall;
  { cancel whatever we're doing, we'll keep the port open }
begin
  FCallerIDProvided := False;
  DoStatus(msCancel);
  FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclCancelCall, 0, 0);
  if Connected then begin
    DoStatus(msHangup);
    { try lowering DTR first }
    WaitingForResponse := True;
    TimedOut := False;
    ResponsePacket.TimeOut := ApxDefDTRTimeout;
    ResponsePacket.Enabled := True;
    FComPort.DTR := False;

    { wait for the response }
    repeat
      Application.HandleMessage;
      if csDestroying in ComponentState then Exit;
    until not(WaitingForResponse) or (TimedOut);

    ResponsePacket.Enabled := False;
    if TimedOut then begin
      { lowering DTR didn't work, escape and send the hangup command }
      SendCommand(ApxDefModemEscape);
      if not SendCommands(LmModem.Hangup) then begin
        SendCommand(ApxDefModemEscape);
        SendCommand(ApxDefHangupCmd);
      end;
    end;
  end else if FModemState in [msAnswerWait, msConnectWait] then
    { we've started answering/dialing, send a #13 to terminate that }
    SendCommand('');
  PrepForConnect(False);
  DoDisconnect;
  FConnected := False;
  if Initialized then
    DoStatus(msIdle)
  else
    DoStatus(msUnknown);
end;

procedure TApxCustomModem.CheckCallerID(const Response: string);
  { check for the CallerID tags }
var
  I,
  Psn : Integer;
  S : string;

  function CheckIt : Boolean;
  begin
    Psn := Pos(S, Response);
    if Psn > 0 then begin
      Result := True;
      S := Copy(Response, Psn + Length(S), Length(Response));
      if Copy(S, Length(S) - 2, 2) = #13#10 then
        S := Copy(S, 1, Length(S) - 2);
    end else
      Result := False;
  end;

begin
  if LmModem.Responses.Date.Count > 0 then
    for I := 0 to pred(LmModem.Responses.Date.Count) do begin
      S := ConvertXML(PLmResponseData(LmModem.Responses.Date[I]).Response);
      if CheckIt then begin
        FCallerIDInfo.HasData := True;
        FCallerIDInfo.Date := S;
      end;
    end;

  if LmModem.Responses.Time.Count > 0 then
    for I := 0 to pred(LmModem.Responses.Time.Count) do begin
      S := ConvertXML(PLmResponseData(LmModem.Responses.Time[I]).Response);
      if CheckIt then begin
        FCallerIDInfo.HasData := True;
        FCallerIDInfo.Time := S;
      end;
    end;

  if LmModem.Responses.Number.Count > 0 then
    for I := 0 to pred(LmModem.Responses.Number.Count) do begin
      S := ConvertXML(PLmResponseData(LmModem.Responses.Number[I]).Response);
      if CheckIt then begin
        FCallerIDInfo.HasData := True;
        FCallerIDInfo.Number := S;
      end;
    end;

  if LmModem.Responses.Name.Count > 0 then
    for I := 0 to pred(LmModem.Responses.Name.Count) do begin
      S := ConvertXML(PLmResponseData(LmModem.Responses.Name[I]).Response);
      if CheckIt then begin
        FCallerIDInfo.HasData := True;
        FCallerIDInfo.Name := S;
      end;
    end;

  if LmModem.Responses.Msg.Count > 0 then
    for I := 0 to pred(LmModem.Responses.Msg.Count) do begin
      S := ConvertXML(PLmResponseData(LmModem.Responses.Msg[I]).Response);
      if CheckIt then begin
        FCallerIDInfo.HasData := True;
        FCallerIDInfo.Msg := S;
      end;
    end;
end;

function TApxCustomModem.CheckErrors(const Response: string): Integer;
begin
  if CheckResponses(Response, ApxDefErrorResponse, LmModem.Responses.Error) then
    Result := ecModemRejectedCommand
  else if CheckResponses(Response, ApxDefErrorResponse, LmModem.Responses.NoCarrier) then
    Result := ecNoCarrier
  else if CheckResponses(Response, ApxDefErrorResponse, LmModem.Responses.NoDialTone) then
    Result := ecNoDialTone
  else if CheckResponses(Response, ApxDefErrorResponse, LmModem.Responses.Busy) then
    Result := ecModemDetectedBusy
  else if CheckResponses(Response, ApxDefErrorResponse, LmModem.Responses.NoAnswer) then
    Result := ecNoAnswer
  else
    Result := ecOK;
end;

procedure TApxCustomModem.CheckReady;
begin
  if not Assigned(FComPort) then
    raise EPortNotAssigned.Create(ecPortNotAssigned, False);

  FComPort.OnTriggerStatus := TriggerEvent;
  FComPort.OnTriggerTimer := TriggerEvent;

  if not Assigned(ResponsePacket) then begin
    ResponsePacket := TApxDataPacket.Create(Self);
    ResponsePacket.Name := Name + '_ResponsePacket';
    ResponsePacket.Enabled := False;
    ResponsePacket.AutoEnable := False;
    ResponsePacket.Timeout := ApxDefCommandTimeout;
    ResponsePacket.OnStringPacket := ResponseStringPacket;
    ResponsePacket.OnTimeout := ResponseTimeout;
    ResponsePacket.ComPort := FComPort;
    ResponsePacket.StartCond := scAnyData;
    ResponsePacket.EndCond := [ecString];
    ResponsePacket.EndString := '?'#13#10;
    ResponsePacket.Enabled := True;
  end;
  if not FComPort.Open then
    FComPort.Open := True;
end;

function TApxCustomModem.CheckResponses(const Response, DefResponse: string;
  Responses: TList): Boolean;
  function StripCtrl(const S : string) : string;
    { strip out the CR/LF prefix and suffix }
  begin
    Result := S;
    while Pos(#13, Result) > 0 do
      Delete(Result, Pos(#13, Result), 1);
    while Pos(#10, Result) > 0 do
      Delete(Result, Pos(#10, Result), 1);
  end;
var
  I : Integer;
  S : string;
begin
  { assume it's not a response that we're looking for }
  Result := False;
  if Responses.Count > 0 then begin
    for I := 0 to pred(Responses.Count) do begin
      S := ConvertXML(PLmResponseData(Responses[I]).Response);
      if StripCtrl(S) = StripCtrl(Response) then begin
        Result := True;
        Break;
      end;
    end;
    if not Result then
      Result := Pos(DefResponse, Response) > 0;
  end else
    { see if the default response is at the beginning of the response }
    Result := Pos(DefResponse, Response) = 1;
end;

procedure TApxCustomModem.ConfigAndOpen;
  { open the port and configure the modem }
begin
  FCallerIDProvided := False;
  CheckReady;
  FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclConfigAndOpen, 0, 0);
  PassthroughMode := True;
  Initialize;
  DoStatus(msIdle);
  DoConnect;
end;

function TApxCustomModem.ConvertXML(const S: string): string;
  { converts the '<CR>' and '<LF>' from LibModem into #13 and #10 }
var
  Psn : Integer;
begin
  Result := S;
  while Pos('<CR>', AnsiUpperCase(Result)) > 0 do begin
    Psn := Pos('<CR>', AnsiUpperCase(Result));
    Delete(Result, Psn, Length('<CR>'));
    Insert(#13, Result, Psn);
  end;
  while Pos('<LF>', AnsiUpperCase(Result)) > 0 do begin
    Psn := Pos('<LF>', AnsiUpperCase(Result));
    Delete(Result, Psn, Length('<LF>'));
    Insert(#10, Result, Psn);
  end;
  { XML also doubles any '%' char, strip that }
  while Pos('%%', Result) > 0 do
    Delete(Result, Pos('%%', Result), 1);
end;

constructor TApxCustomModem.Create(AOwner: TComponent);
  { we're being created }
begin
  FSelectedDevice := TApxModemNameProp.Create;
  FStatusDisplay := nil;
  inherited;
  Initialized := False;
  PassthroughMode := False;
  ResponsePacket := nil;

  { property inits }
  FAnswerOnRing := 2;
  FBPSRate := 0;
  FConnected := False;
  FDialTimeout := 60;
  FFailCode := 0;
  FModemCapFolder := ApxDefModemCapFolder;
  FModemState := msUnknown;
  FNegotiationResponses := TStringList.Create;
  FRingCount := 0;
  FRingWaitTimeout := 1200;
  FSelectedDevice.Manufacturer := '';
  FSelectedDevice.Name := '';
  FStartTime := 0;
  LmModem.Manufacturer := 'Generic Hayes compatible';
  LmModem.Model := 'Generic modem';
  LmModem.FriendlyName := 'Generic modem';
  LibModem := TApxLibModem.Create(Self);
  FModemConfig := DefaultDeviceConfig;
  FCallerIDProvided := False;
  with CallerIDInfo do begin
    HasData := False;
    Date   := '';
    Time   := '';
    Number := '';
    Name   := '';
    Msg    := '';
  end;
end;

procedure TApxCustomModem.CreateWidget;
  { create our handle so we can do the message thing }
begin
  FHandle := QObject_create(nil, nil);
end;

procedure TApxCustomModem.TriggerEvent(CP: TObject;
  TriggerHandle: Word);
  { handle our DCD and timer triggers }
begin
  if TriggerHandle = DCDTrigger then begin
    if FComPort.DCD then
      DoConnect
    else
      DoDisconnect;
  end else if TriggerHandle = StatusTimerTrigger then begin
    DoStatus(FModemState);
    FComPort.SetTimerTrigger(StatusTimerTrigger, 1000, True);
    if (FModemState = msConnectWait) and
       (Integer(ElapsedTime div 1000) >= FDialTimeout) then begin
       { > DialTimeout elapsed, cancel }
      AxPostMessage(Handle, apx_CommandProcessed, nil, ApxMessage_CancelCall, 0);
      DoFail(ecNoAnswer);
    end;
  end;           
end;

function TApxCustomModem.DefaultDeviceConfig: TApxModemConfig;
begin
  with Result do begin
    ConfigVersion := ApxModemConfigVersion;
    { port settings }
    DataBits := dbEight;
    Parity := pNone;
    StopBits := sbOne;
    if Assigned(FComPort) then
      AttachedTo := FComPort.DeviceName
    else
      AttachedTo := 'unknown';

    Manufacturer := LmModem.Manufacturer;
    ModemName := LmModem.FriendlyName;
    ModemModel := LmModem.Model;
    { speaker options }
    SpeakerVolume :=  svMed;
    SpeakerMode := smDial;
    { connection control }
    FlowControl := fcHard;
    ErrorControl := [ecOn];
    Compression := True;;
    Modulation := smCCITT;;
    ToneDial := True;
    BlindDial := False;
    CallSetupFailTimeout := 60;
    InactivityTimeout := 0;
    { extra commands }
    ExtraSettings := '';
    FillChar(Padding, SizeOf(Padding), #0);
  end;
end;

destructor TApxCustomModem.Destroy;
  { we're being destroyed }
begin
  ResponsePacket.Free;
  FNegotiationResponses.Free;
  FSelectedDevice.Free;
  LibModem.Free;
  inherited Destroy;
end;

procedure TApxCustomModem.Dial(const ANumber: string);
  { initiate the dialing sequence }
begin
  FCallerIDProvided := False;
  CheckReady;
  FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclDial,
    DWORD(ANumber), Length(ANumber));
  PrepForConnect(True);
  FPhoneNumber := ANumber;
  PassthroughMode := False;
  if not Initialized then
    Initialize;
  FStartTime := AxTimeGetTime;
  DoStatus(msDial);
  AxPostmessage(Handle, apx_CommandProcessed, nil, ApxMessage_StartDial, 0);
end;

procedure TApxCustomModem.DoCallerID;
  { Generate the OnModemCallerID event }
begin
  FCallerIDProvided := True;
  FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclCallerID, 0, 0);
  if Assigned(FOnModemCallerID) then
    FOnModemCallerID(Self, FCallerIDInfo);
end;

procedure TApxCustomModem.DoConnect;
  { Generate the OnModemConnect event }
begin
  PrepForConnect(False);
  if not PassthroughMode then
    FConnected := True;
  if Assigned(FOnModemConnect) then
    FOnModemConnect(Self);
end;

procedure TApxCustomModem.DoDisconnect;
  { Generate the OnModemDisconnect event }
begin
  PrepForConnect(False);
  if Assigned(FOnModemDisconnect) then
    FOnModemDisconnect(Self);
end;

procedure TApxCustomModem.DoFail(Failure : Integer);
  { Generate the OnModemFail event }
begin
  FFailCode := Failure;
  FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclFail, FFailCode, 0);
  if Assigned(FOnModemFail) then
    FOnModemFail(Self)
  else
    case FFailCode of
      ecModemRejectedCommand :
        raise EModemRejectedCommand.CreateUnknown(Format('Modem rejected command:'#13#10'%s',[LastCommand]), 0);
      ecModemBusy :
        raise EModemBusy.CreateUnknown('Modem is doing something else', 0);
      ecDeviceNotSelected :
        raise EDeviceNotSelected.CreateUnknown('Device not selected', 0);
      ecModemNotResponding :
        raise EModemNotResponding.CreateUnknown('No response from modem', 0);
      ecModemDetectedBusy :
        raise EModemDetectedBusy.CreateUnknown('Called number was busy', 0);
      ecNoDialTone :
        raise ENoDialTone.CreateUnknown('No dialtone', 0);
      ecNoCarrier :
        raise ENoCarrier.CreateUnknown('No carrier', 0);
      ecNoAnswer :
        raise ENoAnswer.CreateUnknown('No answer', 0);
    end;
end;

procedure TApxCustomModem.DoLog(LogCode: TApxModemLogCode);
  { generate the OnModemLog event }
begin
  if Assigned(FOnModemLog) then
    FOnModemLog(Self, LogCode);
end;

procedure TApxCustomModem.DoStatus(NewModemState: TApxModemState);
  { change FModemState and generate the event }
var
  S : string;
  Action : TApxModemStatusAction;
  FirstState : Boolean;
begin
  if FModemState <> NewModemState then begin
    FirstState := True;
    { state changed, is it log-worthy? }
    case NewModemState of
      msIdle                  : if FModemState in [msAutoAnswerWait..msConnectWait] then
                                  DoLog(mlConnectFail)
                                else
                                  DoLog(mlNone);
      msAutoAnswerBackground  : DoLog(mlAutoAnswer);
      msAnswerWait            : DoLog(mlAnswer);
      msDial                  : DoLog(mlDial);
      msConnected             : DoLog(mlConnect);
      msHangup                : DoLog(mlCancel);
      msCancel                : DoLog(mlCancel);
    end;
  end else
    FirstState := False;

  FModemState := NewModemState;
  if Assigned(FOnModemStatus) then
    FOnModemStatus(Self, ModemState);

  if Assigned(FStatusDisplay) then begin
    { update the status display }
    if FStatusDisplay.Started then
      S := ModemStatusMsg(FModemState)
    else
      S := '';
    case FModemState of
      msConnectWait, msAutoAnswerWait  :
        if FirstState then
          Action := msaUpdate
        else
          Action := msaDetailReplace;
      msIdle, msConnected : Action := msaClose;
      else
        Action := msaUpdate;
    end;

    FStatusDisplay.UpdateDisplay(Self,
      ModemStatusMsg(FModemState),     { the status line }
      IntToStr(ElapsedTime div 1000),
      S,                               { for the detail list }
      Action);
  end;
end;

function TApxCustomModem.EventFilter(Sender: QObjectH;
  Event: QEventH): Boolean;
  procedure DoCustomDispatchEvent(Event : QCustomEventH; FreeData : Boolean);
  var
    DataReference : PAxMessage;
    Data          : TAxMessage;
  begin
    DataReference := QCustomEvent_data(Event);
    { Local copy of data }
    Data := DataReference^;
    if FreeData then
      Dispose(DataReference);
    try
      Self.Dispatch(Data);
    except
      raise;
    end;
    if not FreeData then
      DataReference^.Result := Data.Result;
  end;
begin
  Result := False;
  case QEvent_type(Event) of
    QEventType_AxPostDispatchMessage :
      begin
        DoCustomDispatchEvent(QCustomEventH(Event), True);
        Result := True;
      end;
    else inherited EventFilter(Sender, Event);
  end;
end;

function TApxCustomModem.FailureCodeMsg(
  const FailureCode: Integer): string;
  { convert a FailureCode into a string }
begin
  case FailureCode of
    ecDeviceNotSelected      : Result := 'Device not selected';
    ecModemRejectedCommand   : Result := 'Modem rejected command';
    ecModemBusy              : Result := 'Modem is doing something else';
    ecModemNotResponding     : Result := 'Modem not responding';
    ecModemDetectedBusy      : Result := 'Called number is busy';
    ecNoDialtone             : Result := 'No dialtone';
    ecNoCarrier              : Result := 'No carrier';
    ecNoAnswer               : Result := 'No answer';
  end;
end;

function TApxCustomModem.GetDevConfig: TApxModemConfig;
  { return the TApxModemConfig structure defining the selected modem }
begin
  Result := FModemConfig;
end;

function TApxCustomModem.GetElapsedTime: DWORD;
begin
  if FStartTime = 0 then
    Result := 0 { not timing }
  else
    Result := AxTimeGetTime - FStartTime;
end;

class function TApxCustomModem.GetLogString(const D1, D2,
  D3: DWORD): string;
  { returns a string for the debug log }
  function MakeStr : string;
  begin
    SetLength(Result, D3);
    Move(PByteArray(D2)[0], Result[1], D3);
  end;
var
  S : string;
begin
  case D1 of
    cclSelect :
      begin
        S := MakeStr;
        Result := 'Selecting device from ' + S + AxLineTerm;
      end;
    cclSelectedMdmFile :
      begin
        S := MakeStr;
        Result := 'Selected Device - file:' + S;
      end;
    cclSelectedMdmMfg :
      begin
        S := MakeStr;
        Result := 'Selected Device - mfg :' + S;
      end;
    cclSelectedMdmName :
      begin
        S := MakeStr;
        Result := 'Selected Device - name:' + S + AxLineTerm;
      end;
    cclConfigAndOpen :
      begin
        Result := 'ConfigAndOpen';
      end;
    cclDial :
      begin
        Result := 'Dialing ' + MakeStr;
      end;
    cclAutoAnswer :
      begin
        Result := Format('Enable AutoAnswer for %d ring(s)', [D2]);
      end;
    cclCancelCall :
      begin
        Result := 'CancelCall';
      end;
    cclRing :
      begin
        Result := Format('Ring %d detected', [D2]);
      end;
    cclAnswer :
      begin
        Result := 'Answering call';
      end;
    cclCallerID :
      begin
        Result := 'Interpreted response: CallerID';
      end;
    cclConnected :
      begin
        Result := 'Connected';
      end;
    cclDisconnected :
      begin
        Result := 'Disconnected';
      end;
    cclFail :
      begin
        Result := Format('Modem failed (%d)', [D2]);
      end;
    cclConfigChange :
      begin
        Result := 'Modem config changed';
      end;
    cclInitialize :
      begin
        Result := 'Initializing modem';
      end;
    cclBusyResponse :
      begin
        Result := 'Interpreted response: busy';
      end;
    cclEchoResponse :
      begin
        Result := 'Interpreted response: echo';
      end;
    cclOKResponse :
      begin
        Result := 'Interpreted response: OK';
      end;
    cclErrorResponse :
      begin
        Result := 'Interpreted response: ERROR';
      end;
    cclTimeout :
      begin
        Result := 'Command timed out';
      end;
    cclInformativeResponse :
      begin
        Result := 'Interpreted response: Informative';
      end;
    cclUnknownResponse :
      begin
        Result := 'Interpreted response: Unknown';
      end;
    cclConnectResponse :
      begin
        Result := 'Interpreted response: CONNECT';
      end;
    else
      Result := 'Undefined log string';
  end;
end;

function TApxCustomModem.GetNegotiationResponses: TStringList;
  { return the negotiation responses for this connection }
begin
  Result := FNegotiationResponses;
end;

procedure TApxCustomModem.Initialize;
  { initialize the modem }
  function PoundReplace(const str : string; Value : Integer) : string;
  { some modem init strings have variable params, replace them here }
  var
    I : Integer;
  begin
    Result := str;
    I := Pos('<#>', Result);
    { remove the '<#>' }
    Delete(Result, I, 3);
    { add the value }
    Insert(IntToStr(Value), Result, I);
  end;
var
  ConfigInit : string;
begin
  { set the msInitializing state }
  DoStatus(msInitializing);
  FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclInitialize, 0, 0);
  if not FDeviceSelected then
    raise EDeviceNotSelected.Create(ecDeviceNotSelected, False);
  if not SendCommands(LmModem.Init) then begin
    { fake it, using generic reset }
    SendCommand(LmModem.Reset);
  end;
  ConfigInit := LmModem.Settings.Prefix + ' ';
  with FModemConfig do begin
    { port settings }
    FComPort.DataBits := DataBits;
    FComPort.Parity := Parity;
    FComPort.StopBits := StopBits;
    { speaker options }
    case SpeakerVolume of
      svLow  : ConfigInit := ConfigInit + LmModem.Settings.SpeakerVolume_Low + ' ';
      svMed  : ConfigInit := ConfigInit + LmModem.Settings.SpeakerVolume_Med + ' ';
      svHigh : ConfigInit := ConfigInit + LmModem.Settings.SpeakerVolume_High + ' ';
    end;
    case SpeakerMode of
      smOff   : ConfigInit := ConfigInit + LmModem.Settings.SpeakerMode_Off + ' ';
      smOn    : ConfigInit := ConfigInit + LmModem.Settings.SpeakerMode_On + ' ';
      smDial  : ConfigInit := ConfigInit + LmModem.Settings.SpeakerMode_Dial + ' ';
    end;
    { connection control }
    case FlowControl of
      fcOff  : ConfigInit := ConfigInit + LmModem.Settings.FlowControl_Off + ' ';
      fcHard : ConfigInit := ConfigInit + LmModem.Settings.FlowControl_Hard + ' ';
      fcSoft : ConfigInit := ConfigInit + LmModem.Settings.FlowControl_Soft + ' ';
    end;
    if ecOff in ErrorControl then
      ConfigInit := ConfigInit + LmModem.Settings.ErrorControl_Off + ' '
    else begin
      ConfigInit := ConfigInit + LmModem.Settings.ErrorControl_On + ' ';
      if ecForced in ErrorControl then
       ConfigInit := ConfigInit + LmModem.Settings.ErrorControl_Forced + ' ';
      if ecCellular in ErrorControl then
        ConfigInit := ConfigInit + LmModem.Settings.ErrorControl_Cellular + ' ';
    end;
    if Compression then
      ConfigInit := ConfigInit + LmModem.Settings.Compression_On + ' '
    else
      ConfigInit := ConfigInit + LmModem.Settings.Compression_Off + ' ';
    case Modulation of
      smBell      : ConfigInit := ConfigInit + LmModem.Settings.Modulation_Bell + ' ';
      smCCITT     : ConfigInit := ConfigInit + LmModem.Settings.Modulation_CCITT + ' ';
      smCCITT_V23 : ConfigInit := ConfigInit + LmModem.Settings.Modulation_CCITT_V23 + ' ';
    end;
    if BlindDial then
      ConfigInit := ConfigInit + LmModem.Settings.Blind_On
    else
      ConfigInit := ConfigInit + LmModem.Settings.Blind_Off;
    ConfigInit := ConfigInit +
      PoundReplace(LmModem.Settings.CallSetupFailTimer, CallSetupFailTimeout) + ' ';
    ConfigInit := ConfigInit +
      PoundReplace(LmModem.Settings.InactivityTimeout, InactivityTimeout) + ' ';
    ConfigInit := ConfigInit + LmModem.Settings.Terminator;

    FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclInitialize, 1, 0);
    SendCommand(ConvertXML(ConfigInit));

    if ExtraSettings <> '' then begin
      FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclInitialize, 2, 0);
      SendCommand(ConvertXML(ExtraSettings + #13));
    end;
  end;
  Initialized := True;
end;

procedure TApxCustomModem.Loaded;
begin
  inherited;
  { get the ModemCap folder from the environment var }
  if not(csDesigning in ComponentState) then
    if FModemCapFolder = ApxDefModemCapFolder then
      FModemCapFolder := GetEnvironmentVariable('MODEMCAP');
end;

function TApxCustomModem.ModemLogToString(
  LogCode: TApxModemLogCode): string;
  { convert a LogCode into a string }
begin
  case LogCode of
    mlNone         : Result := 'None  nothing to log';
    mlDial         : Result := 'Dialing';
    mlAutoAnswer   : Result := 'Initiated AutoAnswer';
    mlAnswer       : Result := 'Answering an incoming call';
    mlConnect      : Result := 'Connected';
    mlCancel       : Result := 'Call cancelled';
    mlBusy         : Result := 'Called number was busy';
    mlConnectFail  : Result := 'Connection attempt failed';
    else             Result := 'Undefined modem log code';
  end;
end;

function TApxCustomModem.ModemStatusMsg(Status: TApxModemState): string;
  { convert a status code into a string }
var
  Plural : char;
begin
  case Status of
    msUnknown :
      Result := 'Hasn''t been initialized';
    msIdle :
      Result := 'Idle and ready';
    msInitializing :
      Result := 'Starting initialization process';
    msAutoAnswerBackground :
      Result := 'AutoAnswer no rings received';
    msAutoAnswerWait :
      begin
        if (FAnswerOnRing - FRingCount) > 1 then
          Plural := 's'
        else
          Plural := ' ';
        Result := Format('AutoAnswer waiting for %d more ring%s',
          [FAnswerOnRing - FRingCount, Plural]);
      end;
    msAnswerWait :
      Result := 'Answering call waiting for connect';
    msDial :
      Result := Format('Dialing %s', [FPhoneNumber]);
    msConnectWait :
      Result := 'Waiting for remote to answer';
    msConnected :
      Result := 'Connected';
    msHangup :
      Result := 'Starting hangup process';
    msCancel :
      Result := 'Starting cancel process';
    else
      Result := 'Undefined modem state';
  end;
end;

procedure TApxCustomModem.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);

  if not (csDesigning in ComponentState) then
    exit;
  if (Operation = opRemove) then begin
    { see if our com port is going away }
    if (AComponent = FComPort) then
      ComPort := nil;
    { see if our status dialog is going away }
    if (AComponent = FStatusDisplay) then
      StatusDisplay := nil;
  end else if (Operation = opInsert) then begin
    {Check for a com port being installed}
    if not Assigned(FComPort) and (AComponent is TApxCustomComPort) then
      ComPort := TApxCustomComPort(AComponent);
    if not Assigned(FStatusDisplay) and (AComponent is TApxAbstractModemStatus) then
      StatusDisplay := TApxAbstractModemStatus(AComponent);
  end;
end;

procedure TApxCustomModem.PrepForConnect(EnableTriggers: Boolean);
begin
  if EnableTriggers then begin
    { somebody set us up the trigger }
    if DCDTrigger > 0 then
      FComPort.RemoveTrigger(DCDTrigger);
    DCDTrigger := FComPort.AddStatusTrigger(stModem);
    FComPort.SetStatusTrigger(DCDTrigger, msDCDDelta, True);
    if StatusTimerTrigger > 0 then
      FComPort.RemoveTrigger(StatusTimerTrigger);
    StatusTimerTrigger := FComPort.AddTimerTrigger;
    FComPort.SetTimerTrigger(StatusTimerTrigger, 1000, True);
  end else begin
    if DCDTrigger > 0 then begin
      FComPort.RemoveTrigger(DCDTrigger);
      DCDTrigger := 0;
    end;
    if StatusTimerTrigger > 0 then begin
      FComPort.RemoveTrigger(StatusTimerTrigger);
      StatusTimerTrigger := 0;
    end;
  end;
  FNegotiationResponses.Clear;
end;

procedure TApxCustomModem.ResponseStringPacket(Sender: TObject;
  Data: String);
var
  Res : Integer;
begin
  { we've detected a string ending with #13#10, see if it is }
  { something we are looking for }
  { assume it's not }
  OKResponse := False;
  ErrorResponse := False;
  ConnectResponse := False;
  WaitingForResponse := True;

  { if we're waiting for the connection, add the response to the list }
  if FModemState in [msConnectWait, msAnswerWait] then begin
    if Data <> #13#10 then begin
      FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclInformativeResponse,
        0, 0);
      FNegotiationResponses.Add(Data);
    end;
  end;

  Res := CheckErrors(Data);
  if Res <> ecOK then begin
    ErrorResponse := True;
    WaitingForResponse := False;
    if (FModemState = msHangup) and (Res = ecNoCarrier) then
      { we've disconnected, handle it a little }
    else begin
      DoFail(Res);
      Exit;
    end;
  end;

  { check for caller ID tags }
  if FModemState in [msAutoAnswerBackground, msAutoAnswerWait, msAnswerWait] then
    CheckCallerID(Data);

  { interpret the response based on what state we're in }
  case FModemState of
    msUnknown,
    msIdle,
    msConnected : { anything here means that the packet wasn't disabled }
      begin
        ResponsePacket.Enabled := False;
        WaitingForResponse := False;
      end;
    msInitializing : { anything here should be a OK or ERROR response }
      begin
        if CheckResponses(Data, ApxDefOKResponse, LmModem.Responses.OK) then begin
          { it's an OK }
          FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclOKResponse, 0, 0);
          OKResponse := True;
          WaitingForResponse := False;
        end else
          if Pos(LastCommand, Data) > 0 then begin
            FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclEchoResponse, 0, 0);
            ResponsePacket.Enabled := True;
          end else
            FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclUnknownResponse, 0, 0);
      end;
    msAutoAnswerBackground :
      begin
        if CheckResponses(Data, ApxDefRingResponse, LmModem.Responses.Ring) then begin
          { it's the first RING }
          if not FCallerIDProvided and CallerIDInfo.HasData then begin
            DoCallerID;
          end;
          FRingCount := 1;
          FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclRing, FRingCount, 0);
          DoStatus(msAutoAnswerWait);
          ResponsePacket.TimeOut := FRingWaitTimeout;
          ResponsePacket.Enabled := True;
        end;
      end;
    msAutoAnswerWait : { looking for more RINGs }
      begin
        if CheckResponses(Data, ApxDefRingResponse, LmModem.Responses.Ring) then begin
          { it's another RING }
          inc(FRingCount);
          if not FCallerIDProvided and CallerIDInfo.HasData then begin
            DoCallerID;
          end;
          { see if we need to answer it now }
          if FRingCount >= FAnswerOnRing then begin
            DoStatus(msAnswerWait);
            WaitingForResponse := False;
            { send the ATA }
            AxPostmessage(Handle, apx_CommandProcessed, nil, ApxMessage_AutoAnswer, 0);
          end else begin
            { not enough rings }
            FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclRing, FRingCount, 0);
            DoStatus(msAutoAnswerWait);
            ResponsePacket.TimeOut := FRingWaitTimeout;
            ResponsePacket.Enabled := True;
          end;
        end;
      end;

    msAnswerWait,
    msDial,
    msConnectWait : { waiting for connect or error }
      begin
        if CheckResponses(Data, ApxDefConnectResponse, LmModem.Responses.Connect) then begin
          { it's a CONNECT }
          ConnectResponse := True;
          OKResponse := True;
          WaitingForResponse := False;
          FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclConnectResponse, 0, 0);
          if not FConnected then begin
            DoStatus(msConnected);
            DoConnect;
          end;
        end else
          ResponsePacket.Enabled := True;
      end;
    msHangup,
    msCancel : { starting hangup }
      begin
        WaitingForResponse :=False;
      end;
  end;
end;

procedure TApxCustomModem.ResponseTimeout(Sender: TObject);
begin
  { data packet timed out }
  TimedOut := True;
  if FModemState = msAutoAnswerWait then begin
    FRingCount := 0;
    DoStatus(msAutoAnswerBackground);
    ResponsePacket.Timeout := 0;
    ResponsePacket.Enabled := True;
  end;
end;

function TApxCustomModem.SelectDevice: Boolean;
  { display the modem selection dialog }
begin
  try
    if Assigned(FComPort) then
      FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclSelect,
        DWORD(FModemCapFolder), Length(FModemCapFolder));
    LibModem.LibModemPath := FModemCapFolder;
    Result := LibModem.SelectModem(
      FSelectedDevice.FModemFile,
      FSelectedDevice.FManufacturer,
      FSelectedDevice.FName, LmModem);
    FDeviceSelected := Result;
    if Result then begin
       FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclSelectedMdmFile,
        DWORD(FSelectedDevice.FModemFile), Length(FSelectedDevice.FModemFile));
       FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclSelectedMdmMfg,
        DWORD(FSelectedDevice.FManufacturer), Length(FSelectedDevice.FManufacturer));
       FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclSelectedMdmName,
        DWORD(FSelectedDevice.FName), Length(FSelectedDevice.FName));
    end;
  finally
    { eat the exeption here }
  end;
end;

function TApxCustomModem.SendCommand(const Command: string): Boolean;
  { send a command to the modem, returns when the response is received }
  { or on a timeout }
begin
  if WaitingForResponse then begin
    Result := False;
    DoFail(ecModemBusy);
    Exit;
  end;
  CheckReady;
  LastCommand := Command;

  Result := True;
  WaitingForResponse := True;
  OKResponse := False;
  ErrorResponse := False;
  ConnectResponse := False;
  TimedOut := False;

  ResponsePacket.Timeout := ApxDefCommandTimeout;
  ResponsePacket.Enabled := True;
  FComPort.Output := Command;
  { wait for the response }
  repeat
    Application.HandleMessage;
    if csDestroying in ComponentState then Exit;
  until not(WaitingForResponse) or (TimedOut);
  ResponsePacket.Enabled := False;
  if TimedOut then
    DoFail(ecModemNotResponding)
  else if ErrorResponse then
    DoFail(ecModemRejectedCommand);
  Result := not(TimedOut) and not(ErrorResponse);
end;

function TApxCustomModem.SendCommands(Commands: TList) : Boolean;
  { internal method to send all commands in the TLmCommands list }
var
  I : Integer;
begin
  Result := False;
  if Commands.Count > 0 then begin
    for I := 0 to pred(Commands.Count) do begin
      Result := SendCommand(ConvertXML(PLmModemCommand(Commands[I]).Command));
      if not Result then
        Break;
    end;
  end else
    { return False if no commands were available }
    Result := False;
end;

procedure TApxCustomModem.SetAnswerOnRing(const Value: Integer);
  { write access method for AnswerOnRing property }
begin
  FAnswerOnRing := Value;
end;

procedure TApxCustomModem.SetComPort(const Value: TApxCustomComPort);
  { write access method for ComPort property }
begin
  FComPort := Value;
end;

procedure TApxCustomModem.SetDevConfig(const Config: TApxModemConfig);
  { forces new configuration }
begin
  FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclConfigChange, 0, 0);
  FModemConfig := Config;
end;

procedure TApxCustomModem.SetDialTimeout(const Value: Integer);
  { write access method for DialTimeout property }
begin
  FDialTimeout := Value;
end;

procedure TApxCustomModem.SetModemCapFolder(const Value: string);
  { write access method for ModemCapFolder property }
begin
  FModemCapFolder := Value;
end;

procedure TApxCustomModem.SetRingWaitTimeout(const Value: DWORD);
  { write access method for RingWaitTimeout property }
begin
  FRingWaitTimeout := Value;
end;

procedure TApxCustomModem.SetSelectedDevice(
  const Value: TApxModemNameProp);
  { write access method for SelectedDevice property }
begin
  FSelectedDevice := Value;
  FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclSelectedMdmFile,
    DWORD(FSelectedDevice.FModemFile), Length(FSelectedDevice.FModemFile));
  FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclSelectedMdmMfg,
    DWORD(FSelectedDevice.FManufacturer), Length(FSelectedDevice.FManufacturer));
  FComport.DebugLog.AddDebugEntry(TApxCustomModem, cclSelectedMdmName,
    DWORD(FSelectedDevice.FName), Length(FSelectedDevice.FName));
end;


procedure TApxCustomModem.SetStatusDisplay(
  const Value: TApxAbstractModemStatus);
  { write access method for StatusDisplay property }
begin
  FStatusDisplay := Value;
end;

{ TApxAbstractModemStatus }

constructor TApxAbstractModemStatus.Create(AOwner: TComponent);
begin
  inherited;
  Caption := ApxDefModemStatusCaption;
  FStarted := False;
  FModem := nil;
  FStatusDialog := nil;
end;

destructor TApxAbstractModemStatus.Destroy;
begin
  FStatusDialog.Free;
  inherited;
end;

procedure TApxAbstractModemStatus.SetCaption(const Value: string);
begin
  if FCaption <> Value then begin
    FCaption := Value;
    if Assigned(FStatusDialog) then
      FStatusDialog.Caption := Value;
  end;
end;

procedure TApxAbstractModemStatus.SetModem(const Value: TApxCustomModem);
begin
  FModem := Value;
  if FStarted then begin
    SetStarted(False);
    SetStarted(True);
  end;
end;

procedure TApxAbstractModemStatus.SetStarted(Start: Boolean);
begin
  if Start = FStarted then exit;
  if Start then begin
    FStatusDialog := TApxModemStatusDialog.Create(self);
    FStatusDialog.Caption := Caption;
    TApxModemStatusDialog(FStatusDialog).Modem := FModem;

    FStatusDialog.Show;
  end else begin
    FStatusDialog.Free;
    FStatusDialog := nil;
  end;
  FStarted := Start;
end;

procedure TApxAbstractModemStatus.UpdateDisplay(Modem: TApxCustomModem;
  const StatusStr, TimeStr, DetailStr : string;
  Action : TApxModemStatusAction);
begin
  if Action = msaClose then begin
    SetStarted(False);
    Exit;
  end;
  if (not Started) then
    { create the dialog }
    SetStarted(True);

  TApxModemStatusDialog(FStatusDialog).UpdateDisplay(
    StatusStr,  { the status line }
    TimeStr,    { the 'Elapsed time' line }
    DetailStr,  { detail list }
    Action);    { how we're going to display it }

  if FModem.FModemState in [msUnknown, msIdle, msConnected] then
    SetStarted(False);
end;

function TApxCustomModem.ShowConfigDialog : Boolean;
var
  MdmCfgDlg : TApxModemConfigDialog;
begin
  MdmCfgDlg := nil;
  try
    MdmCfgDlg := TApxModemConfigDialog.Create(nil);
    MdmCfgDlg.LmModem := LmModem;
    if FModemConfig.AttachedTo = '' then
      FModemConfig.AttachedTo := FComPort.DeviceName;
    MdmCfgDlg.ModemConfig := DefaultDeviceConfig;
    Result := MdmCfgDlg.ShowModal = mrOK;
    if Result then begin
      FModemConfig := MdmCfgDlg.ModemConfig;
    end;
  finally
    MdmCfgDlg.Free;
  end;
end;

end.
