(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                   AxPBook.pas 1.02                    *}
{*********************************************************}

{Global defines potentially affecting this unit}
{$I AxDefine.inc}

{Options required for this unit}
{$G+,X+,F+}

unit AxPBook;

interface

uses
  SysUtils, Types, Classes, QGraphics, QControls, QForms, QDialogs, AxMisc, AxPort,
  QComCtrls, QStdCtrls, QExtCtrls, QGrids, IniFiles;

type
  TAxPBEntryType = (pbtSerial, pbtModem);

  TAxPBSettings = record
    DataBits : TAxDatabits;
    Parity   : TAxParity;
    StopBits : TAxStopBits;
  end;

  TAxPBEmulation =(pbeNone, pbeTTY, pbeVT100);

  { note, this must match the definition in AxProtcl.pas }
  TAxPBProtocol = (
    ptNoProtocol,
    ptXmodem,
    ptXmodemCRC,
    ptXmodem1K,
    ptXmodem1KG,
    ptYmodem,
    ptYmodemG,
    ptZmodem,
    ptKermit,
    ptAscii);

const
  { strings used to display the phone book fields }
  CAxPBEntryTypeString : array[low(TAxPBEntryType)..high(TAxPBEntryType)] of string =
    ('Serial', 'Modem');
  CAxPBSettingsDataBitsString : array[low(TAxDataBits)..high(TAxDataBits)] of char =
     ('5', '6', '7', '8');
  CAxPBSettingsParityString : array[low(TAxParity)..high(TAxParity)] of char =
     ('N', 'O', 'E', 'M', 'S');
  CAxPBSettingsParityLongString : array[low(TAxParity)..high(TAxParity)] of string =
     ('None', 'Odd', 'Even', 'Mark', 'Space');
  CAxPBSettingsStopBitsString  : array[low(TAxStopBits)..high(TAxStopBits)] of string =
     ('1', '2');
  CAxPBEmulationString : array[low(TAxPBEmulation)..high(TAxPBEmulation)] of string =
    ('None', 'TTY', 'VT100');
  CAxPBProtocolString : array[low(TAxPBProtocol)..high(TAxPBProtocol)] of string =
    ('None', 'XModem', 'XModemCRC', 'XModem1K', 'XModem1KG', 'YModem', 'YModemG',
     'ZModem', 'Kermit', 'Ascii');

type
  TAxPBookRecord = record
    EntryName : string[100];      { the name of the entry, must be unique }
    EntryType : TAxPBEntryType;   { the type of connection }
    Address   : string[100];      { the address (phone number) }
    Settings  : TAxPBSettings;    { DataBits, Parity, StopBits }
    Emulation : TAxPBEmulation;   { Default emulation }
    Protocol  : TAxPBProtocol;    { Default protocol }
    Custom1   : string[100];      { space for custom strings }
    Custom2   : string[100];      { can be used for just about anything }
    Custom3   : string[100];
    Custom4   : string[100];
  end;

  TApxCustomPhoneBook = class;

  { The phone book dialog }
  TApxPhoneBookDialog = class(TForm)
    pnlButton: TPanel;
    btnOK: TButton;
    btnCancel: TButton;
    grdEntries: TStringGrid;
    Label1: TLabel;
    edtPhoneBookLocation: TEdit;
    btnEdit: TButton;
    btnDelete: TButton;
    btnAdd: TButton;
    procedure FormResize(Sender: TObject);
    procedure btnOKClick(Sender: TObject);
    procedure btnAddClick(Sender: TObject);
    procedure btnEditClick(Sender: TObject);
    procedure grdEntriesSelectCell(Sender: TObject; ACol, ARow: Integer;
      var CanSelect: Boolean);
    procedure btnDeleteClick(Sender: TObject);
  private
    { Private declarations }
    IsDirty : Boolean;
    PhoneBook : TApxCustomPhoneBook;
    procedure UpdateGrid;
  public
    { Public declarations }
    SelectedEntry : TAxPBookRecord;

  end;

  TApxCustomPhoneBook = class(TApxBaseComponent)
  private
    FPhoneBookFile : string;
    FSelectedEntry: TAxPBookRecord;
    FPBIniFile : TIniFile;
    FPBEntries: TStringList;
    FEntryName: string;
    FEntryNames : TStringList;
    procedure SetPhoneBookFile(const Value: string);
    function GetEntryNames: TStringList;
    procedure SetEntryName(const Value: string);
  public
    { properties }
    property PBEntries : TStringList
      read FPBEntries;
    property PhoneBookFile : string               { path/name of the ini file }
      read FPhoneBookFile write SetPhoneBookFile;
    property EntryName : string                   { Name of the selected entry }
      read FEntryName write SetEntryName;
    property SelectedEntry : TAxPBookRecord       { The selected entry itself }
      read FSelectedEntry write FSelectedEntry;
    property EntryNames : TStringList             { Names of all entries in the INI file }
      read GetEntryNames;

    { methods }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;

    procedure AddEntry(Entry : TAxPBookRecord);
      {- Add an entry to the phone book }
    procedure ClosePhoneBook;
      {- Update and close the ini file }
    procedure DeleteEntry(EntryName : string);
      {- Deletes an entry from the phone book }
    function ShowEditDialog(EntryToEdit : TAxPBookRecord) : Boolean;
      {- Show the Edit entry dialog, True if they clicked OK }
    function ShowNewEntryDialog : Boolean;
      {- Show the New entry dialog, True if they clicked OK }
    function Execute : Boolean;
      {- Show the phone book selector dialog }
    procedure OpenPhoneBook;
      {- Open the ini file and load the values }
    function SettingsToStr(const Settings : TAxPBSettings) : string;
      {- Convert a TAxPBSettings to a string (8N1) - into INI file }
    function StrToSettings(const Str : string) : TAxPBSettings;
      {- Convert a string (8N1) to a TAxPBSettings - out of INI file }
  end;

  TApxPhoneBook = class(TApxCustomPhoneBook)
  published
    property PhoneBookFile;
  end;

implementation

{$R *.xfm}
uses AxPBEdit;

{ TApxCustomPhoneBook }

procedure TApxCustomPhoneBook.AddEntry(Entry: TAxPBookRecord);
  { Add an entry to the INI file }
begin
  FPBIniFile.WriteString(Entry.EntryName, 'Type', CAxPBEntryTypeString[Entry.EntryType]);
  FPBIniFile.WriteString(Entry.EntryName, 'Settings', SettingsToStr(Entry.Settings));
  FPBIniFile.WriteString(Entry.EntryName, 'Address', Entry.Address);
  FPBIniFile.WriteString(Entry.EntryName, 'Emulation', CAxPBEmulationString[Entry.Emulation]);
  FPBIniFile.WriteString(Entry.EntryName, 'Protocol', CAxPBProtocolString[Entry.Protocol]);
  FPBIniFile.WriteString(Entry.EntryName, 'Custom1', Entry.Custom1);
  FPBIniFile.WriteString(Entry.EntryName, 'Custom2', Entry.Custom2);
  FPBIniFile.WriteString(Entry.EntryName, 'Custom3', Entry.Custom3);
  FPBIniFile.WriteString(Entry.EntryName, 'Custom4', Entry.Custom4);
end;

procedure TApxCustomPhoneBook.ClosePhoneBook;
  { Update the INI file and close it }
begin
  FPBIniFile.UpdateFile;
  FPBIniFile.Free;
  FPBIniFile := nil;
end;

constructor TApxCustomPhoneBook.Create(AOwner: TComponent);
begin
  inherited;
  FPBIniFile := nil;
  FEntryNames := TStringList.Create;
  FPBEntries := TStringList.Create;
end;

procedure TApxCustomPhoneBook.DeleteEntry(EntryName: string);
begin
  FPBIniFile.EraseSection(EntryName);
end;

destructor TApxCustomPhoneBook.Destroy;
begin
  FPBIniFile.Free;
  FPBEntries.Free;
  FEntryNames.Free;
  inherited;
end;

function TApxCustomPhoneBook.Execute: Boolean;
  { Show the phone book dialog, returns True if they clicked OK }
var
  ApxPhoneBookDialog: TApxPhoneBookDialog;
begin
  ApxPhoneBookDialog := nil;
  try
    ApxPhoneBookDialog := TApxPhoneBookDialog.Create(nil);
    ApxPhoneBookDialog.edtPhoneBookLocation.Text := FPhoneBookFile;
    ApxPhoneBookDialog.PhoneBook := Self;
    ApxPhoneBookDialog.UpdateGrid;
    Result := ApxPhoneBookDialog.ShowModal = mrOK;
    if Result then begin
      FPhoneBookFile := ApxPhoneBookDialog.edtPhoneBookLocation.Text;
      FSelectedEntry := ApxPhoneBookDialog.SelectedEntry;
    end;
  finally
    ApxPhoneBookDialog.Free;
  end;
end;

function TApxCustomPhoneBook.GetEntryNames: TStringList;
  { get the names of the entries }
begin
  if not Assigned(FEntryNames) then begin
    FEntryNames := TStringList.Create;
  end;
  { refresh the list from the ini file }
  FPBIniFile.ReadSections(FEntryNames);
  Result := FEntryNames;
end;

procedure TApxCustomPhoneBook.OpenPhoneBook;
  { open the INI file, load the values }
begin
  if Assigned(FPBIniFile) then
    ClosePhoneBook;
  FPBIniFile := TIniFile.Create(FPhoneBookFile);
  FPBIniFile.GetStrings(FPBEntries);
  FEntryNames.Clear;
  FPBIniFile.ReadSections(FEntryNames);
end;

procedure TApxCustomPhoneBook.SetEntryName(const Value: string);
  { select a new entry }
  function IndexOfStr(S : string; Arr : Array of string) : Byte;
    { find the enumeration based on the array }
  begin
    Result := Low(Arr);
    while S <> Arr[Result] do begin
      inc(Result);
      if Result > High(Arr) then begin
        Result := 255;
        Break;
      end;
    end;
  end;
var
  EntryStrings : TStrings;
  Str : string;
begin        
  EntryStrings := nil;
  if Value = '' then begin
    { must be a new entry }
    FSelectedEntry.EntryName := '';
    FSelectedEntry.Address := '';
    FSelectedEntry.Settings.DataBits := dbEight;
    FSelectedEntry.Settings.Parity := pNone;
    FSelectedEntry.Settings.StopBits := sbOne;
    FSelectedEntry.Emulation := pbeTTY;
    FSelectedEntry.Protocol := ptZModem;
    FSelectedEntry.Custom1 := '';
    FSelectedEntry.Custom2 := '';
    FSelectedEntry.Custom3 := '';
    FSelectedEntry.Custom4 := '';
    FEntryName := Value;
  end else
    try
      EntryStrings := TStringList.Create;
      FPBIniFile.ReadSection(Value, EntryStrings);
      FSelectedEntry.EntryName := Value;
      Str :=FPBIniFile.ReadString(Value, 'EntryType', CAxPBEntryTypeString[pbtSerial]);
      FSelectedEntry.EntryType := TAxPBEntryType(IndexOfStr(Str, CAxPBEntryTypeString));
      FSelectedEntry.Address := FPBIniFile.ReadString(Value, 'Address', '');
      Str := FPBIniFile.ReadString(Value, 'Settings', '8N1');
      FSelectedEntry.Settings := StrToSettings(Str);
      Str := FPBIniFile.ReadString(Value, 'Emulation', CAxPBEmulationString[pbeTTY]);
      FSelectedEntry.Emulation := TAxPBEmulation(IndexOfStr(Str, CAxPBEmulationString));
      Str := FPBIniFile.ReadString(Value, 'Protocol', CAxPBProtocolString[ptZModem]);
      FSelectedEntry.Protocol := TAxPBProtocol(IndexOfStr(Str, CAxPBProtocolString));
      FSelectedEntry.Custom1 := FPBIniFile.ReadString(Value, 'Custom1', '');
      FSelectedEntry.Custom2 := FPBIniFile.ReadString(Value, 'Custom2', '');
      FSelectedEntry.Custom3 := FPBIniFile.ReadString(Value, 'Custom3', '');
      FSelectedEntry.Custom4 := FPBIniFile.ReadString(Value, 'Custom4', '');

      FEntryName := Value;
    finally
      EntryStrings.Free;
    end;
end;

procedure TApxCustomPhoneBook.SetPhoneBookFile(const Value: string);
  { select a new INI file, old one must be closed first }
begin
  if Value <> FPhoneBookFile then begin
    if Assigned(FPBIniFile) then
      raise Exception.Create(format('Phonebook file %s must be closed first', [FPhoneBookFile]));
    FPhoneBookFile := Value;
  end;
end;

function TApxCustomPhoneBook.SettingsToStr(
  const Settings: TAxPBSettings): string;
  { convert the record to a string (8N1) }
begin
  Result := CAxPBSettingsDataBitsString[Settings.DataBits];
  Result := Result + CAxPBSettingsParityString[Settings.Parity];
  Result := Result + CAxPBSettingsStopBitsString[Settings.StopBits];
end;

function TApxCustomPhoneBook.ShowEditDialog(
  EntryToEdit: TAxPBookRecord): Boolean;
  { Show the edit/add dialog to edit }
var
  AxPBEditor : TAxPBEditor;
  EntryName : string;
begin
  AxPBEditor := nil;
  try
    AxPBEditor := TAxPBEditor.Create(nil);
    EntryName := EntryToEdit.EntryName;
    Result := AxPBEditor.Execute(EntryToEdit);
    if Result then begin
      if EntryName <> EntryToEdit.EntryName then
        DeleteEntry(EntryToEdit.EntryName);
      AddEntry(EntryToEdit);
    end;
  finally
    AxPBEditor.Free;
  end;
end;

function TApxCustomPhoneBook.ShowNewEntryDialog: Boolean;
  { Show the edit/add dialog to add a new entry }
var
  AxPBEditor : TAxPBEditor;
begin
  AxPBEditor := nil;
  try
    AxPBEditor := TAxPBEditor.Create(nil);

    Result := AxPBEditor.Execute();
    if Result then begin
      AddEntry(AxPBEditor.PhoneBookEntry);
    end;
  finally
    AxPBEditor.Free;
  end;
end;

function TApxCustomPhoneBook.StrToSettings(
  const Str: string): TAxPBSettings;
  { convert a string (8N1) to the record }
begin
  if (Length(Str) < 3) or (Length(Str) > 5) then
    raise Exception.Create(format('Can''t determine data bits (%s)', [Str]));
  Result.DataBits := TAxDataBits(StrToIntDef(Str[1], ord(dbEight) - ord(dbFive)));
  case Upcase(Str[2]) of
    'N' : Result.Parity := pNone;
    'O' : Result.Parity := pOdd;
    'E' : Result.Parity := pEven;                    
    'M' : Result.Parity := pMark;
    'S' : Result.Parity := pSpace;
  else raise Exception.Create(Format('Can''t determine parity (%s)', [Str]));
  end;
  if Copy(Str, 3, Length(Str)) = '1' then
    Result.StopBits := sbOne
  else if Copy(Str, 3, Length(Str)) = '2' then
    Result.StopBits := sbTwo
  else raise Exception.Create(Format('Can''t determine stop bits (%s)', [Str]));
end;


{ TApxPhoneBookDialog }

procedure TApxPhoneBookDialog.FormResize(Sender: TObject);
var
  I, X, W : Integer;
begin
  { size the string grid columns }
  grdEntries.Left := 8;
  grdEntries.Width := ClientWidth - 16;
  I := grdEntries.ClientWidth - 4;

  grdEntries.Cells[0, 0] := 'Entry name';
  grdEntries.ColWidths[0] := Trunc(I * 0.30);
  grdEntries.Cells[1, 0] := 'Type';
  grdEntries.ColWidths[1] := Trunc(I * 0.10);
  grdEntries.Cells[2, 0] := 'Address';
  grdEntries.ColWidths[2] := Trunc(I * 0.30);
  grdEntries.Cells[3, 0] := 'Params';
  grdEntries.ColWidths[3] := Trunc(I * 0.10);
  grdEntries.Cells[4, 0] := 'Emulation';
  grdEntries.ColWidths[4] := Trunc(I * 0.10);
  grdEntries.Cells[5, 0] := 'Protocol';
  X := 0;
  for W := 0 to grdEntries.ColCount - 2 do
    X := X + grdEntries.ColWidths[W];
  grdEntries.ColWidths[5] := I - X - 2;
  edtPhoneBookLocation.Width := ClientWidth - 8 - edtPhoneBookLocation.Left;
  grdEntries.Height := pnlButton.Top - grdEntries.Top - 16;
end;

procedure TApxPhoneBookDialog.UpdateGrid;
var
  I : Integer;
begin
  for I := 1 to grdEntries.RowCount - 1 do
   grdEntries.Rows[I].Clear;
  grdEntries.RowCount := PhoneBook.EntryNames.Count + 2;
  for I := 0 to pred(PhoneBook.EntryNames.Count) do begin
    PhoneBook.EntryName := PhoneBook.EntryNames[I];
    grdEntries.Cells[0, succ(I)] := PhoneBook.SelectedEntry.EntryName;
    grdEntries.Cells[1, succ(I)] := CAxPBEntryTypeString[PhoneBook.SelectedEntry.EntryType];
    grdEntries.Cells[2, succ(I)] := PhoneBook.SelectedEntry.Address;
    grdEntries.Cells[3, succ(I)] := PhoneBook.SettingsToStr(PhoneBook.SelectedEntry.Settings);
    grdEntries.Cells[4, succ(I)] := CAxPBEmulationString[PhoneBook.SelectedEntry.Emulation];
    grdEntries.Cells[5, succ(I)] := CAxPBProtocolString[PhoneBook.SelectedEntry.Protocol];
  end;
  PhoneBook.EntryName := grdEntries.Cells[0, grdEntries.Row];
  SelectedEntry := PhoneBook.SelectedEntry;
end;

procedure TApxPhoneBookDialog.btnOKClick(Sender: TObject);
begin
  if IsDirty then begin
    PhoneBook.ClosePhoneBook;
    PhoneBook.OpenPhoneBook;
  end;
end;

procedure TApxPhoneBookDialog.btnAddClick(Sender: TObject);
begin
  PhoneBook.ShowNewEntryDialog;
  UpdateGrid;
end;

procedure TApxPhoneBookDialog.btnEditClick(Sender: TObject);
begin
  PhoneBook.ShowEditDialog(SelectedEntry);
  UpdateGrid;
end;

procedure TApxPhoneBookDialog.grdEntriesSelectCell(Sender: TObject; ACol,
  ARow: Integer; var CanSelect: Boolean);
begin
  PhoneBook.EntryName := grdEntries.Cells[0, ARow];
  SelectedEntry := PhoneBook.SelectedEntry;
end;

procedure TApxPhoneBookDialog.btnDeleteClick(Sender: TObject);
begin
  PhoneBook.DeleteEntry(SelectedEntry.EntryName);
  UpdateGrid;
end;

end.
