(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                  AxPacket.pas 1.02                    *}
{*********************************************************}

{ Global defines potentially affecting this unit }
{$I AxDefine.inc}

unit AxPacket;

interface

uses
  Types,
  SysUtils,
  Classes,
  QGraphics,
  QControls,
  QForms,
  QDialogs,
  AxMisc,
  AxExcept,
  AxPort,
  AxString;

type
  TPacketStartCond = (scString,scAnyData);
  TPacketEndCond = (ecString,ecPacketSize);
  TPacketEndSet = set of TPacketEndCond;

const
  EscapeCharacter = '\';   { Use \\ to specify an actual '\' in the match strings}
  WildCardCharacter = '?'; { Use \? to specify an actual '?' in the match strings} 
  adpDefEnabled = True;
  adpDefIgnoreCase = True;
  adpDefIncludeStrings = True;
  adpDefAutoEnable = True;
  adpDefStartCond = scString;
  adpDefTimeOut = 2184;

type
  TApxDataPacket = class;
  TApxDataPacketManager = class;
  TApxDataPacketManagerList = class
    {Maintains a list of packet managers so that a packet can
     locate the current packet manager for its comport.
     If no packet manager currently exists for the port, the
     packet will create one. When the last packet dis-connects
     itself from the packet manager, the packet manager self-
     destructs.}
  private
    ManagerList : TList;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Insert(Value : TApxDataPacketManager);
    procedure Remove(Value : TApxDataPacketManager);
    function GetPortManager(ComPort : TApxCustomComPort) : TApxDataPacketManager;
  end;

  TApxDataPacketManager = class
    {Packet manager. One instance of these exists per com port using
     packets. The packet manager does the actual data buffering for
     all packets attached to its port.}
  private
    PacketList : TList;
    fComPort : TApxCustomComPort;
    HandlerInstalled : Boolean;
    fEnabled : Boolean;
    BufferPtr : Integer;
    fDataBuffer : pChar;
    dpDataBufferSize : Integer;
    fCapture : TApxDataPacket;
    Timer : Integer;
    fInEvent : Boolean;
    NotifyPending : Boolean;
    NotifyStart : Integer;
    EnablePending : Boolean;                                         
  protected
    procedure DisposeBuffer;
     {- Get rid of any pending data and release any buffer space}
    procedure NotifyData(NewDataStart : Integer);
     {- Notify the attached packet(s) that new data is available}
    procedure EnablePackets;
     {- Initialize all enabled packets for data capture}
    procedure DisablePackets;
     {- Shut off data capture for all attached packets}
    procedure PacketTriggerHandler(Msg, wParam : Cardinal;
                                 lParam : Longint);
     {- process messages from dispatcher}
    procedure PortOpenClose(CP : TObject; Opening : Boolean);
     {- Event handler for the port open/close event}
    procedure SetInEvent(Value : Boolean);
     {- Property write method for the InEvent property}
    procedure SetEnabled(Value : Boolean);
     {- Proporty write method for the Enabled property}
  public
    constructor Create(ComPort : TApxCustomComPort);
    destructor Destroy; override;
    procedure Enable;
     {- Install com port event handlers}
    procedure EnableIfPending;                                       
     {- Enable after form load}
    procedure Disable;
     {- Remove com port event handlers}
    procedure Insert(Value : TApxDataPacket);
     {- Add a packet to the list}
    procedure Remove(Value : TApxDataPacket);
     {- Remove a packet to the list}
    procedure RemoveData(Start,Size : Integer);
     {- Remove packet data from the data buffer}
    procedure SetCapture(Value : TApxDataPacket; TimeOut : Integer);
     {- Set ownership of incoming data to a particular packet}
    procedure ReleaseCapture(Value : TApxDataPacket);
     {- Opposite of SetCapture, see above}
    property DataBuffer : pChar read fDataBuffer;
     {- The packet data buffer for the port. Only packets should access this}
    property ComPort : TApxCustomComPort read fComPort;
     {- The com port associated with this packet manager}
    property Enabled : Boolean read fEnabled write SetEnabled;
     {- Controls whether the packet manager is active
        set/reset when the com port is opened or closed}
    property InEvent : Boolean read fInEvent write SetInEvent;
     {- Event flag set by packets to prevent recursion issues}
  end;

  TPacketMode = (dpIdle,dpWaitStart,dpCollecting);
  TPacketNotifyEvent = procedure(Sender: TObject; Data : Pointer; Size : Integer) of object;
  TStringPacketNotifyEvent = procedure(Sender: TObject; Data : string) of object;
  TApxDataPacket = class(TApxBaseComponent)
  private
    fManager : TApxDataPacketManager;
    fStartCond : TPacketStartCond;
    fEndCond : TPacketEndSet;
    fStartString,fEndString : string;
    fComPort : TApxCustomComPort;
    fMode : TPacketMode;
    fPacketSize : Integer;
    fOnPacket : TPacketNotifyEvent;
    fOnStringPacket : TStringPacketNotifyEvent;
    fOnTimeOut : TNotifyEvent;
    fTimeOut : Integer;
    fDataSize : Integer;
    fBeginMatch : Integer;
    fAutoEnable : Boolean;
    fIgnoreCase : Boolean;
    fEnabled : Boolean;
    fIncludeStrings : Boolean;

    PacketBuffer : pChar;
    StartMatchPos,EndMatchPos,EndMatchStart : Integer;               
    LocalPacketSize : Integer;
    WildStartString,
    WildEndString,
    InternalStartString,
    InternalEndString : string;
    NameStr : string;                                                 
    WillCollect : Boolean;
    EnablePending : Boolean;
    HaveCapture : Boolean;
    FSyncEvents : Boolean;                                          
  protected
    procedure SetComPort(const NewComPort : TApxCustomComPort);
    procedure Notification(AComponent : TComponent; Operation : TOperation); override;
    procedure SetEnabled(Value : Boolean);
    procedure SetMode(Value : TPacketMode);
    procedure SetEndCond(const Value: TPacketEndSet);
    procedure SetEndString(Value : String);                                 
    procedure ProcessData(StartPtr : Integer);
     {- Processes incoming data, collecting and/or looking for a match}
    procedure Packet(Reason : TPacketEndCond);
     {- Set up parameters and call DoPacket to generate an event}
    procedure TimedOut;
     {- Set up parameters and call DoTimeout to generate an event}
    procedure DoTimeout;
     {- Generate an OnTimeOut event}
    procedure DoPacket;
     {- Generate an OnPacket event}
    procedure NotifyRemove(Data : Integer);
     {- Called by the packet manager to cancel any partial matches}
    procedure Resync;
     {- Look for a match starting beyond the first character.
        Called when a partial match fails, or when data has
        been removed by another packet.}
    procedure CancelMatch;
     {- Cancel any pending partial match. Called by the packet manager
        when another packet takes capture.}
    procedure Loaded; override;
    procedure LogPacketEvent(Event : TDispatchSubType;
      Data : Pointer; DataSize : Integer);                            
     {- add packet specific events to log file, if logging is requested}

    property Manager : TApxDataPacketManager read fManager write fManager;
     {- The packet manager controlling this packet}
    property BeginMatch : Integer read fBeginMatch;
     {- Beginning of the current match. -1 if no match yet}
    property Mode : TPacketMode read fMode write SetMode;
     {- Current mode. Can be either Idle = not currently enabled,
        WaitStart = trying to match the start string, or
        Collecting = start condition has been met; collecting data}
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    class function GetLogString(const D1, D2, D3 : DWORD) : string; override;
    procedure Enable;
     {- Enable the packet}
    procedure Disable;
     {- Disable the packet}
    procedure GetCollectedString(var Data : String);
     {- Returns data collected in OnStringPacket format}
    procedure GetCollectedData(var Data : Pointer; var Size : Integer);
     {- Returns data collected in OnPacket format}
    property SyncEvents : Boolean read FSyncEvents write FSyncEvents;  
     {- Controls whether packet events are synchronized to the main VCL thread.
        Default is True.}
    property PacketMode : TPacketMode read fMode;                      
     {- Read-only property to show if we are idle, waiting, or collecting }
  published
    property Enabled : Boolean read fEnabled write SetEnabled nodefault; 
     {- Is the packet enabled.}
    property AutoEnable : Boolean read fAutoEnable write fAutoEnable default adpDefAutoEnable;
     {- Fire only first time, or fire whenever the conditions are met.}
    property StartCond : TPacketStartCond read fStartCond write fStartCond default adpDefStartCond;
     {- Conditions for this packet to start collecting data}
    property EndCond : TPacketEndSet read fEndCond write SetEndCond default [];
     {- Conditions for this packet to stop collecting data}
    property StartString : string read fStartString write fStartString;
     {- Packet start string}
    property EndString : string read fEndString write SetEndString;   
     {- Packet end string}
    property IgnoreCase : Boolean read fIgnoreCase write fIgnoreCase default adpDefIgnoreCase;
     {- Ignore case when matching StartString and EndString}
    property ComPort : TApxCustomComPort read FComPort write SetComPort;
     {- The com port for which data is being read}
    property PacketSize : Integer read fPacketSize write fPacketSize;
     {- Size of a packet with packet size as part of the end conditions}
    property IncludeStrings : Boolean read fIncludeStrings write fIncludeStrings default adpDefIncludeStrings;
     {- Controls whether any start and end strings should be included in the
        data buffer passed to the event handler}
    property TimeOut : Integer read fTimeOut write fTimeOut default adpDefTimeOut;
     {- Number of ticks that can pass from when the packet goes into data
        collection mode until the packet is complete. 0 = no timeout}
    property OnPacket : TPacketNotifyEvent read fOnPacket write fOnPacket;
     {- Event fired when a complete packet is received}
    property OnStringPacket : TStringPacketNotifyEvent read fOnStringPacket write fOnStringPacket;
     {- Event fired when a complete packet is received}
    property OnTimeout : TNotifyEvent read fOnTimeout write fOnTimeout;
     {- Event fired when a packet times out}
  end;

implementation

{$IFDEF TRIALRUN}
  {$I TRIAL07.INC}
  {$I TRIAL03.INC}
  {$I TRIAL01.INC}
{$ENDIF}

var
  PacketManagerList : TApxDataPacketManagerList;

constructor TApxDataPacketManagerList.Create;
begin
  inherited Create;
  ManagerList := TList.Create;
end;

destructor TApxDataPacketManagerList.Destroy;
begin
  ManagerList.Free;
  inherited Destroy;
end;

procedure TApxDataPacketManagerList.Insert(Value : TApxDataPacketManager);
begin
  ManagerList.Add(Value);
end;

procedure TApxDataPacketManagerList.Remove(Value : TApxDataPacketManager);
begin
  ManagerList.Remove(Value);
end;

function TApxDataPacketManagerList.GetPortManager(ComPort : TApxCustomComPort) : TApxDataPacketManager;
var
  i : integer;
begin
  Result := nil;
  for i := 0 to pred(ManagerList.Count) do
    if TApxDataPacketManager(ManagerList[i]).ComPort = ComPort then begin
      Result := TApxDataPacketManager(ManagerList[i]);
      exit;
    end;
end;

constructor TApxDataPacketManager.Create(ComPort : TApxCustomComPort);
begin
  inherited Create;
  fComPort := ComPort;
  fComPort.RegisterUserCallback(PortOpenClose);
  PacketList := TList.Create;
  PacketManagerList.Insert(Self);
  Enabled := fComPort.Open
    and ([csDesigning, csLoading] * fComPort.ComponentState = []);   
  EnablePending :=
    not (csDesigning in fComPort.ComponentState) and
    not Enabled and fComPort.Open;                                   
end;

destructor TApxDataPacketManager.Destroy;
begin
  PacketManagerList.Remove(Self);
  Enabled := False;
  fComPort.DeregisterUserCallback(PortOpenClose);
  DisposeBuffer;
  PacketList.Free;
  inherited Destroy;
end;

procedure TApxDataPacketManager.EnableIfPending;                      
begin
  if EnablePending then begin
    Enabled := True;
    EnablePending := False;
  end;
end;

procedure TApxDataPacketManager.Insert(Value : TApxDataPacket);
begin
  PacketList.Add(Value);
  Value.Manager := Self;
end;

procedure TApxDataPacketManager.Remove(Value : TApxDataPacket);
begin
  Value.Manager := nil;
  PacketList.Remove(Value);
  if PacketList.Count = 0 then
    Free;
end;

procedure TApxDataPacketManager.RemoveData(Start,Size : Integer);
var
  NewStart,i : Integer;
begin
  NewStart := Start+Size;
  dec(BufferPtr,NewStart);
  if BufferPtr > 0 then begin
    move(fDataBuffer[NewStart],fDataBuffer[0],BufferPtr);
  end else
    DisposeBuffer;
  for i := 0 to pred(PacketList.Count) do
    TApxDataPacket(PacketList[i]).NotifyRemove(NewStart);
end;

procedure TApxDataPacketManager.SetCapture(Value : TApxDataPacket; TimeOut : Integer);
var
  i : integer;
begin
  fCapture := Value;
  if TimeOut <> 0 then
    fComPort.Dispatcher.SetTimerTrigger(Timer,TimeOut,True);
  Value.HaveCapture := True;
  for i := 0 to pred(PacketList.Count) do
    if PacketList[i] <> fCapture then
      TApxDataPacket(PacketList[i]).CancelMatch;
end;

procedure TApxDataPacketManager.ReleaseCapture(Value : TApxDataPacket);
var
  ErrorCode : Integer;
begin
  ErrorCode := fComPort.Dispatcher.SetTimerTrigger(Timer,0,False);
  if (ErrorCode < ecOk) and
      not (csLoading in fCapture.ComponentState) then
    raise EPacket.Create (ErrorCode, False);
  fCapture := nil;                                                    
  Value.HaveCapture := False;
  NotifyData(0);
end;

procedure TApxDataPacketManager.SetInEvent(Value : Boolean);
var
  i : Integer;
begin
  if Value <> fInEvent then begin
    fInEvent := Value;
    if Value then begin
      for i := 0 to pred(PacketList.Count) do
        with TApxDataPacket(PacketList[i]) do
          if fEnabled then
            Disable;
    end else begin
      for i := 0 to pred(PacketList.Count) do
        with TApxDataPacket(PacketList[i]) do
          if fEnabled then
            Enable;
      if NotifyPending then begin
        if assigned(fDataBuffer) then
          NotifyData(NotifyStart);
        NotifyPending := False;
      end;
    end;
  end;
end;

procedure TApxDataPacketManager.NotifyData(NewDataStart : Integer);
var
  i : integer;
  Interest : Boolean;
begin
  if InEvent then begin
    NotifyPending := True;
    NotifyStart := NewDataStart;
    exit;
  end;
  if BufferPtr > 0 then
    if assigned(fCapture) then
      fCapture.ProcessData(NewDataStart)
    else begin
      for i := 0 to pred(PacketList.Count) do begin
        TApxDataPacket(PacketList[i]).ProcessData(NewDataStart);
        if assigned(fCapture) then break;
        if not assigned(fDataBuffer) then
          exit;
      end;
      if not assigned(fCapture) then begin
        Interest := False;
        for i := 0 to pred(PacketList.Count) do
          with TApxDataPacket(PacketList[i]) do
            if Enabled and (Mode <> dpIdle) and (BeginMatch <> -1) then begin
              Interest := True;
              break;
            end;
        if not Interest then
          DisposeBuffer;
      end;
    end;
end;

procedure TApxDataPacketManager.EnablePackets;
var
  i : integer;
begin
  for i := 0 to pred(PacketList.Count) do
    with TApxDataPacket(PacketList[i]) do
      if Enabled then
        Enable;
end;

procedure TApxDataPacketManager.DisablePackets;
var
  i : integer;
begin
  for i := 0 to pred(PacketList.Count) do
    with TApxDataPacket(PacketList[i]) do
      Disable;
end;

procedure TApxDataPacketManager.PortOpenClose(CP : TObject; Opening : Boolean);
begin
  if Opening then begin
    Enabled := True;
    EnablePackets;
  end else begin
    DisablePackets;
    Enabled := False;
  end;
end;

procedure TApxDataPacketManager.PacketTriggerHandler(Msg, wParam : Cardinal;
                                 lParam : Longint);
var
  NewDataStart : Integer;
begin
  if Msg = apx_TriggerAvail then begin
    NewDataStart := BufferPtr;
    if (BufferPtr+Integer(wParam)) >= dpDataBufferSize then begin       
      inc(dpDataBufferSize,FComPort.InSize);
      ReAllocMem(fDataBuffer,dpDataBufferSize);
    end;
    wParam := fComPort.Dispatcher.GetBlock(pChar(@fDataBuffer[BufferPtr]),wParam);
    inc(BufferPtr,wParam);
    NotifyData(NewDataStart);
  end else if (Msg = apx_TriggerTimer) and
    (Integer(wParam) = Timer) and
    Assigned(fCapture) then                                          
      fCapture.TimedOut;
end;

procedure TApxDataPacketManager.DisposeBuffer;
begin
  if Assigned(fDataBuffer) then begin
    FreeMem(fDataBuffer,dpDataBufferSize);
    fDataBuffer := nil;
  end;
  dpDataBufferSize := 0;
  BufferPtr := 0;
end;

procedure TApxDataPacketManager.SetEnabled(Value : Boolean);
begin
  if Value <> fEnabled then begin
    if Value then
      Enable
    else
      Disable;
    fEnabled := Value;
  end;
end;

procedure TApxDataPacketManager.Enable;
begin
  if not HandlerInstalled then begin
    if Assigned(fComPort) then begin
      fComPort.Dispatcher.RegisterEventTriggerHandler(PacketTriggerHandler);
      HandlerInstalled := True;
      Timer := fComPort.Dispatcher.AddTimerTrigger;
    end;
  end;
end;

procedure TApxDataPacketManager.Disable;
begin
  if HandlerInstalled then begin
    fComPort.Dispatcher.RemoveTrigger(Timer);
    fComPort.Dispatcher.DeregisterEventTriggerHandler(PacketTriggerHandler);
    HandlerInstalled := False;
    DisposeBuffer;                                                  
  end;
end;

constructor TApxDataPacket.Create(AOwner : TComponent);
{$IFDEF TRIALRUN}
  {$I TRIAL04.INC}
{$ENDIF}
begin
{$IFDEF TRIALRUN}
  TC;
{$ENDIF}
  inherited Create(AOwner);

  FSyncEvents := True;                                               
  {Search for comport}
  if (csDesigning in ComponentState) then
    ComPort := SearchComPort(Owner);

  fIgnoreCase := adpDefIgnoreCase;
  if csDesigning in ComponentState then
    fEnabled := adpDefEnabled
  else
    fEnabled := False;                                              
  fIncludeStrings := adpDefIncludeStrings;
  fEndCond := [];
  fAutoEnable := adpDefAutoEnable;
  fStartCond := adpDefStartCond;
  fTimeOut := adpDefTimeOut;

  Mode := dpIdle;
end;

destructor TApxDataPacket.Destroy;
begin
  ComPort := nil;
  inherited Destroy;
end;

class function TApxDataPacket.GetLogString(const D1, D2, D3: DWORD): string;
var
  D2Tmp : DWORD;
begin
  if (D1 and $80000000) = $80000000 then begin
    D2Tmp := D2;
    Result := MessageNumberToString(AxdtPacket) + '  ' +
              MessageNumberToString(D1 and $7FFFFFFF) + '  ' + AxLineTerm +
              HexifyBlock(PByteArray(D2Tmp)[0], D3);
  end else begin
    Result := MessageNumberToString(AxdtPacket) + '  ' +
              MessageNumberToString(D1) + '  ' +
              IntToStr(D2) + '  ' +
              IntToStr(D3);                   
  end;
end;                         

procedure TApxDataPacket.SetMode(Value : TPacketMode);
begin
  if Value <> fMode then begin
    if Value = dpCollecting then
      Manager.SetCapture(Self,TimeOut)
    else if HaveCapture then
      Manager.ReleaseCapture(Self);
    fMode := Value;
    case fMode of
    dpIdle :
      LogPacketEvent(dstIdle,nil,0);
    dpWaitStart :
      LogPacketEvent(dstWaiting,nil,0);                              
    else
      LogPacketEvent(dstCollecting,nil,0);                           
    end;
  end;
end;

procedure TApxDataPacket.Notification(AComponent : TComponent;
                                        Operation : TOperation);
  {Link/unlink comport when dropped or removed from form}
begin
  inherited Notification(AComponent, Operation);

  if (Operation = opRemove) then begin
    {See if our com port is going away}
    if (AComponent = FComPort) then
      ComPort := nil;
  end else if (Operation = opInsert) then
    {Check for a com port being installed}
    if not Assigned(FComPort) and (AComponent is TApxCustomComPort) then
      ComPort := TApxCustomComPort(AComponent);
end;

procedure TApxDataPacket.SetComPort(const NewComPort : TApxCustomComPort);
var
  Manager : TApxDataPacketManager;
begin
  if NewComPort <> fComPort then begin
    if Assigned(fComPort) then
      PacketManagerList.GetPortManager(fComPort).Remove(Self);
    FComPort := NewComPort;
    if Assigned(fComPort) then begin
      Manager := PacketManagerList.GetPortManager(fComPort);
      if Manager = nil then
        Manager := TApxDataPacketManager.Create(fComPort);
      Manager.Insert(Self);
    end;
  end;
end;

procedure TApxDataPacket.SetEnabled(Value : Boolean);
begin
  if Value <> fEnabled then begin
    if Value then
      Enable
    else
      Disable;
    fEnabled := Value;
  end;
end;

procedure TApxDataPacket.Resync;
var
  Match : Boolean;
begin
  repeat
    inc(fBeginMatch);
    StartMatchPos := 1;
    Match := True;
    while Match and (BeginMatch <= Manager.BufferPtr - 1)
    and (StartMatchPos <= length(InternalStartString)) do begin
      if (WildStartString[StartMatchPos] = '1')
      or (not IgnoreCase
        and (Manager.DataBuffer[BeginMatch+StartMatchPos - 1]
          = InternalStartString[StartMatchPos]))
      or (IgnoreCase
        and (UpCase(Manager.DataBuffer[BeginMatch+StartMatchPos - 1])
          = InternalStartString[StartMatchPos])) then
        inc(StartMatchPos)
      else
        Match := False;
    end;
    if Match and (BeginMatch <= Manager.BufferPtr-1) then begin
      if StartMatchPos >= length(InternalStartString) then
        if (EndCond = []) then begin
          fDataSize := length(InternalStartString);
          Packet(ecPacketSize);
          exit;
        end else
          Mode := dpCollecting;
      break;
    end;
  until BeginMatch > Manager.BufferPtr - 1;
  if BeginMatch > Manager.BufferPtr - 1 then begin
    fBeginMatch := -1;
    StartMatchPos := 1;
  end;
end;

procedure TApxDataPacket.ProcessData(StartPtr : Integer);
var
  I,J : Integer;
  C : Char;
  Match : Boolean;
begin
  if Enabled then begin
    I := StartPtr;
    while I < Manager.BufferPtr do begin
      if Mode = dpIdle then
        if WillCollect then begin
          Mode := dpCollecting;
          WillCollect := False;
        end else
          break;
      C := Manager.DataBuffer[I];
      if Mode <> dpCollecting then
        begin
          if (WildStartString[StartMatchPos] = '1')
          or (not IgnoreCase and (C = InternalStartString[StartMatchPos]))
          or (IgnoreCase and (UpCase(C) = InternalStartString[StartMatchPos])) then begin
            if BeginMatch = -1 then
              fBeginMatch := I;
            if StartMatchPos = length(InternalStartString) then begin
              if (EndCond = []) then begin
                fDataSize := I - BeginMatch + 1;
                Packet(ecPacketSize);
                I := BeginMatch + 1;
                StartMatchPos := 1;
                continue;
              end else
                Mode := dpCollecting;
            end else
              inc(StartMatchPos);
          end else if BeginMatch <> -1 then begin
            I := BeginMatch + 1;
            StartMatchPos := 1;
            fBeginMatch := -1;
            continue;                                               
          end;
        end
      else
        begin
          if BeginMatch = -1 then
            fBeginMatch := I;
          if (ecPacketSize in EndCond)
          and ((I - BeginMatch) + 1 >= LocalPacketSize) then begin
            fDataSize := (I - BeginMatch) + 1;
            Packet(ecPacketSize);
            exit;
          end else
          if (ecString in EndCond) then begin
              if (WildEndString[EndMatchPos] = '1')
              or (not IgnoreCase and (C = InternalEndString[EndMatchPos]))
              or (IgnoreCase and (UpCase(C) = InternalEndString[EndMatchPos])) then begin
                if EndMatchPos = length(InternalEndString) then begin
                  fDataSize := I - BeginMatch + 1;
                  Packet(ecString);
                  exit;
                end else
                  inc(EndMatchPos);
              end else begin
                {No match here, but we may already have seen part of the string}
                if EndMatchPos > 1 then begin
                  Match := False;
                  EndMatchStart := I-1;                             
                  for j := 2 to EndMatchPos do begin
                    EndMatchPos := J - 1;
                    Match := True;
                    repeat
                      if (WildEndString[EndMatchPos] = '1')
                      or (not IgnoreCase
                          and (Manager.DataBuffer[EndMatchStart + EndMatchPos]
                            = InternalEndString[EndMatchPos]))
                      or (IgnoreCase
                          and (UpCase(Manager.DataBuffer[EndMatchStart + EndMatchPos])
                            = InternalEndString[EndMatchPos])) then
                        inc(EndMatchPos)
                      else
                        Match := False;
                      if Match and (EndMatchPos > length(InternalEndString)) then begin
                        fDataSize := (EndMatchStart + EndMatchPos) - BeginMatch;
                        Packet(ecString);
                        exit;
                      end;                                                              
                    until not Match
                      or (EndMatchPos > length(InternalEndString))
                      or ((EndMatchStart + EndMatchPos) > Manager.BufferPtr - 1);
                    if Match then begin
                      inc(EndMatchPos);
                      break;
                    end
                  end;
                  if not Match then begin
                    EndMatchPos := 1;
                    EndMatchStart := -1;
                  end;
                end else begin
                  EndMatchPos := 1;
                  EndMatchStart := -1;
                end;
              end;
            end;
        end;
      if Manager.DataBuffer = nil then
        break;
      inc(I);
    end;
  end;                                                              
end;

procedure TApxDataPacket.Loaded;
begin
  inherited Loaded;
  if assigned(fManager) then
    Manager.EnableIfPending;                                         
  if EnablePending then begin
    Enable;
    EnablePending := False;
  end;
  NameStr := 'Packet:'+Name;                                         
end;

procedure SetupWildMask(var MatchString,Mask : string);
var
  i,j : Integer;
  Esc : boolean;
  Ch : char;
begin
  Esc := False;
  j := 0;
  SetLength(Mask,length(MatchString));
  for i := 1 to length(MatchString) do
    if Esc then begin
      inc(j);
      MatchString[j] := MatchString[i];
      Mask[j] := '0';
      Esc := False;
    end else if MatchString[i] = EscapeCharacter then
      Esc := True
    else begin
      Ch := MatchString[i];
      inc(j);
      MatchString[j] := Ch;
      if Ch = WildCardCharacter then
        Mask[j] := '1'
      else
        Mask[j] := '0';
    end;
  SetLength(MatchString,j);
  SetLength(Mask,j);
end;

procedure TApxDataPacket.LogPacketEvent (Event    : TDispatchSubType;
                                         Data     : Pointer;
                                         DataSize : Integer);
begin
  if (Data <> nil) and (DataSize <> 0) then
    FManager.ComPort.DebugLog.AddDebugEntry (TApxDataPacket,
                                             DWORD (Event) or $80000000,
                                             Cardinal (Data),
                                             DataSize)
  else
    FManager.ComPort.DebugLog.AddDebugEntry (TApxDataPacket,
                                             DWORD (Event) or $80000000,
                                             Cardinal (@NameStr[1]),
                                             Length (NameStr));
end;

procedure TApxDataPacket.Enable;
begin
  if (csDesigning in ComponentState) then
    exit;                                                           
  if csLoading in ComponentState then begin
    EnablePending := True;
    exit;
  end;
  if assigned(fManager) and Manager.Enabled then begin
    if fManager.InEvent then begin
      EnablePending := True;
      exit;
    end;

    LogPacketEvent(dstEnable,nil,0);

    if (StartCond = scString) then begin
      LogPacketEvent(dstStartStr,@FStartString[1],length(StartString));
      if (StartString  = '') then
        raise EInvalidProperty.Create(ecStartStringEmpty, False);
      if (ecPacketSize in EndCond) and (PacketSize < length(StartString)) then
        raise EInvalidProperty.Create(ecPacketTooSmall, False);
      if not IncludeStrings then
        inc(LocalPacketSize,length(StartString));
      Mode := dpWaitStart;
      if IgnoreCase then
        InternalStartString := UpperCase(StartString)
      else
        InternalStartString := StartString;
      SetupWildMask(InternalStartString,WildStartString);         
    end else
      if (EndCond = []) then
        raise EInvalidProperty.Create(ecNoEndCharCount, False)
      else
        if Manager.fCapture = nil then
          Mode := dpCollecting
        else
          WillCollect := True;
    if (ecString in EndCond) then begin
      if (EndString  = '') then
        raise EInvalidProperty.Create(ecEmptyEndString, False);
      LogPacketEvent(dstEndStr,@FEndString[1],length(EndString));   
      if not IncludeStrings then
        inc(LocalPacketSize,length(EndString));
      if IgnoreCase then
        InternalEndString := UpperCase(EndString)
      else
        InternalEndString := EndString;
      SetupWildMask(InternalEndString,WildEndString);               
    end;
    if (ecPacketSize in EndCond) and (PacketSize = 0) then
      raise EInvalidProperty.Create(ecZeroSizePacket, False);
  end;
  LocalPacketSize := PacketSize;
  StartMatchPos := 1;
  fBeginMatch := -1;
  EndMatchPos := 1;
end;

procedure TApxDataPacket.Disable;
begin
  if not EnablePending and not WillCollect and (Mode = dpIdle) then 
    exit;                                                           
  EnablePending := False;
  WillCollect := False;
  if assigned(fManager) then begin
    Mode := dpIdle;
    LogPacketEvent(dstDisable, nil, 0);                             
  end;
end;

procedure TApxDataPacket.NotifyRemove(Data : Integer);
begin
  if Enabled and (BeginMatch <> -1) then
    if BeginMatch < Data then
      Enable
    else
      if BeginMatch <> -1 then
        Resync;
end;

procedure TApxDataPacket.CancelMatch;
begin
  if Enabled and assigned(fComPort) then begin
    Disable;
    Enable;
  end;
end;

procedure TApxDataPacket.DoPacket;
var
  S : string;
begin
  try
    if Assigned(fOnPacket) then
      fOnPacket(Self,Packetbuffer,fDataSize);
    if Assigned(fOnStringPacket) then begin
      {$IFOPT H-}
      if fDataSize > 255 then
        raise EStringSizeError.Create(ecPacketTooLong, False);
      {$ENDIF}
      SetLength(S, fDataSize);
      Move(PacketBuffer^, S[1], fDataSize);
      fOnStringPacket(Self,S);
    end;
  except                                                               
    Application.HandleException(Self);                                 
  end;
end;

procedure TApxDataPacket.Packet(Reason : TPacketEndCond);
var
  LocalSize : Integer;
begin
  fManager.InEvent := True;
  try
    Enabled := False;
    LocalSize := fDataSize;
    if (StartCond = scString) and not IncludeStrings then begin
      PacketBuffer := pChar(@Manager.DataBuffer[BeginMatch+length(InternalStartString)]);
      dec(fDataSize,length(InternalStartString));
    end else
      PacketBuffer := pChar(@Manager.DataBuffer[BeginMatch]);
    if not IncludeStrings and (Reason = ecString) then
      dec(fDataSize,length(InternalEndString));
    LogPacketEvent(dstStringPacket,nil,0);
    case Reason of
    ecString :
      LogPacketEvent(dstStringPacket,PacketBuffer,fDataSize);
    else
      LogPacketEvent(dstSizePacket,PacketBuffer,fDataSize);
    end;
    if SyncEvents and assigned(ComPort.Dispatcher.DispThread) then
      ComPort.Dispatcher.DispThread.Sync(DoPacket)
    else
      DoPacket;
    Manager.RemoveData(BeginMatch,LocalSize);
    if AutoEnable then
      Enabled := True;
  finally
    fManager.InEvent := False;
  end;
end;

procedure TApxDataPacket.DoTimeout;
begin
  try
    if Assigned(fOnTimeout) then
      fOnTimeout(Self);
  except
    Application.HandleException(Self);
  end;                                                              
end;

procedure TApxDataPacket.TimedOut;
begin
  fManager.InEvent := True;
  try
    LogPacketEvent(dstPacketTimeout,nil,0);
    Enabled := False;
    if SyncEvents and assigned(ComPort.Dispatcher.DispThread) then
      ComPort.Dispatcher.DispThread.Sync(DoTimeout)
    else
      DoTimeout;
  finally
    fManager.InEvent := False;
  end;
end;

procedure TApxDataPacket.SetEndString(Value: String);
var
  OldEnabled : Boolean;
begin
  OldEnabled := Enabled;
  Enabled := False;
  FEndString := Value;
  Enabled := OldEnabled;
end;

procedure TApxDataPacket.SetEndCond(const Value: TPacketEndSet);    
var
  OldEnabled : Boolean;
begin
  OldEnabled := Enabled;
  Enabled := False;
  fEndCond := Value;
  Enabled := OldEnabled;
end;

procedure TApxDataPacket.GetCollectedString(var Data: String);        
 {- Returns data collected in OnStringPacket format}
var
  SLength : Integer;
begin
  SLength := fDataSize;
  {$IFOPT H-}
  if SLength > 255 then
    SLength := 255;
  {$ENDIF}
  SetLength(Data, SLength);
  Move(PacketBuffer^, Data[1], SLength);
end;

procedure TApxDataPacket.GetCollectedData(var Data: Pointer;
  var Size: Integer);
 {- Returns data collected in OnPacket format}
begin
  Data := PacketBuffer;
  Size := fDataSize;
end;

initialization
  PacketManagerList := TApxDataPacketManagerList.Create;

finalization
  PacketManagerList.Free;

end.

