(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                   AxPort.pas 1.02                     *}
{*********************************************************}

{ Global defines potentially affecting this unit }
{$I AxDefine.inc}

{Options required for this unit}
{$G+,X+,F+,J+}
{$C MOVEABLE,DEMANDLOAD,DISCARDABLE}

unit AxPort;
  { Serial port component }

interface

uses
{$IFDEF Win32}
  Windows,
{$ENDIF}
{$IFDEF Linux}
  Libc,
{$ENDIF}
  Qt,
  Types,
  SysUtils,
  Classes,
  SyncObjs,
  QControls,
  QForms,
  AxMisc,
  AxSelCom,
  AxSystem,
  AxExcept;

const
  FirstTriggerCounter = 1;
  MaxTriggerHandle = 65536 shr 4;  { Highest trigger handle number }
  StatusTypeMask = $0007;

const
  { Debug log stuff }
  cclCloseCom            = $00000001;
  cclStatusTrigger       = $00000002;
  cclAvailTrigger        = $00000003;
  cclDataTrigger         = $00000004;
  cclTimerTrigger        = $00000005;
  cclAllocWndHandler     = $00000006;
  cclAllocProcHandler    = $00000007;
  cclAllocEventHandler   = $00000008;
  cclAllocTimerTrigger   = $00000009;
  cclAllocStatusTrigger  = $00000010;
  cclDispWndHandler      = $00000011;
  cclDispProcHandler     = $00000012;
  cclDispEventHandler    = $00000013;
  cclDispTimerTrigger    = $00000014;
  cclDispStatusTrigger   = $00000015;
  cclDispDataTrigger     = $00000016;
  cclTimerTriggerActive  = $00000017;
  cclTimerTriggerDeact   = $00000018;
  cclTimerTriggerExtend  = $00000019;
  cclStatusTriggerActive = $00000020;

  cclWriteCom         = DWORD($80000001);
  cclReadCom          = DWORD($80000002);
  cclOpenCom          = DWORD($80000003);
  cclAllocDataTrigger = DWORD($80000004);

const

  { For clearing modem status }
  ClearDelta    = $F0;
  ClearNone     = $FF;
  ClearDeltaCTS = Byte(not DeltaCTSMask);
  ClearDeltaDSR = Byte(not DeltaDSRMask);
  ClearDeltaRI  = Byte(not DeltaRIMask);
  ClearDeltaDCD = Byte(not DeltaDCDMask);

const
  { Modem status trigger options }
  msCTSDelta   = $0010;
  msDSRDelta   = $0020;
  msRingDelta  = $0004;
  msDCDDelta   = $0080;

const
  { Line status trigger options }
  lsOverrun    = $0001;
  lsParity     = $0002;
  lsFraming    = $0004;
  lsBreak      = $0008;

const
  { Line and driver errors }
  leNoError    = 0;   { No error, ordinal value matches ecOK }
  leBuffer     = 1;   { Buffer overrun in serial driver }
  leOverrun    = 2;   { UART receiver overrun }
  leParity     = 3;   { UART receiver parity error }
  leFraming    = 4;   { UART receiver framing error }
  leCTSTO      = 5;   { Transmit timeout waiting for CTS }
  leDSRTO      = 6;   { Transmit timeout waiting for DSR }
  leDCDTO      = 7;   { Transmit timeout waiting for RLSD }
  leTxFull     = 8;   { Transmit queue is full }
  leBreak      = 9;   { Break condition received }

const
  { Status trigger subtypes }
  stNotActive   = 0;  { not active }
  stModem       = 1;  { Trigger on modem status change }
  stLine        = 2;  { Trigger on line status change }
  stOutBuffFree = 3;  { Trigger on outbuff free level }
  stOutBuffUsed = 4;  { Trigger on outbuff used level }
  stOutSent     = 5;  { Trigger on any PutXxx call }

type
  TApHandlerFlagUpdate = (fuKeepPort, fuEnablePort, fuDisablePort);

  { Port characteristic enumerations }
  TAxDatabits = (dbFive, dbSix, dbSeven, dbEight);
  TAxParity = (pNone, pOdd, pEven, pMark, pSpace);
  TAxStopbits = (sbOne, sbTwo);
  TAxSWFlowControl = (swfNone, swfReceive, swfTransmit, swfBoth);

  { Port state }
  TPortState = (psClosed, psShuttingDown, psOpen);

  { Base device configuration class }
  TApxBaseDeviceParams = class(TPersistent)
  private
    { Properties / events }
    FOnChange : TNotifyEvent;
    { Internal fields }
    bdCS : TCriticalSection;
    bdUpdateCount : DWORD;
  protected
    procedure Changed; dynamic;
    procedure Lock;
    procedure Unlock;
  public
    constructor Create;
    destructor Destroy; override;
    procedure BeginUpdate;
    procedure EndUpdate;
    property OnChange : TNotifyEvent read FOnChange write FOnChange;
  end;

  { Base serial device configuration class }
  TApxSerialDeviceParams = class(TApxBaseDeviceParams)
  private
    { Line Parameters }
    FBaud         : LongInt;
    FBufferFull   : Byte;
    FBufferResume : Byte;
    FDatabits     : TAxDatabits;
    FDTR          : Boolean;
    FParity       : TAxParity;
    FRTS          : Boolean;
    FStopbits     : TAxStopbits;

    { Buffer Sizes }
    FInSize  : Word;
    FOutSize : Word;

    { Flow Control }
    FHWFlowControl : Boolean;
    FSWFlowControl : TAxSWFlowControl;
    FXOffChar      : AnsiChar;
    FXOnChar       : AnsiChar;

    { Misc Parameters }
    FBaseAddress  : Word;
    FRS485Mode    : Boolean;
    FUseEventWord : Boolean;

    { Property access methods }
    function GetBaseAddress : Word;
    function GetBaud : LongInt;
    function GetBufferFull : Byte;
    function GetBufferResume : Byte;
    function GetDatabits : TAxDatabits;
    function GetDTR : Boolean;
    function GetHWFlowControl : Boolean;
    function GetSWFlowControl : TAxSWFlowControl;
    function GetInSize : Word;
    function GetOutSize : Word;
    function GetParity : TAxParity;
    function GetRS485Mode : Boolean;
    function GetRTS : Boolean;
    function GetStopbits : TAxStopbits;
    function GetUseEventWord : Boolean;
    function GetXOffChar : AnsiChar;
    function GetXOnChar : AnsiChar;
    procedure SetBaseAddress(const Value : Word);
    procedure SetBaud(const Value : LongInt);
    procedure SetBufferFull(const Value : Byte);
    procedure SetBufferResume(const Value : Byte);
    procedure SetDatabits(const Value : TAxDatabits);
    procedure SetDTR(const Value : Boolean);
    procedure SetHWFlowControl(const Value : Boolean);
    procedure SetInSize(const Value : Word);
    procedure SetOutSize(const Value : Word);
    procedure SetParity(const Value : TAxParity);
    procedure SetRS485Mode(const Value : Boolean);
    procedure SetRTS(const Value : Boolean);
    procedure SetStopbits(const Value : TAxStopbits);
    procedure SetSWFlowControl(const Value : TAxSWFlowControl);
    procedure SetUseEventWord(const Value : Boolean);
    procedure SetXOffChar(const Value : AnsiChar);
    procedure SetXOnChar(const Value : AnsiChar);
  public
    property BaseAddress : Word
      read GetBaseAddress write SetBaseAddress;
    property Baud : LongInt
      read GetBaud write SetBaud;
    property BufferFull : Byte
      read GetBufferFull write SetBufferFull;
    property BufferResume : Byte
      read GetBufferResume write SetBufferResume;
    property Databits : TAxDatabits
      read GetDatabits write SetDatabits;
    property DTR : Boolean
      read GetDTR write SetDTR;
    property HWFlowControl : Boolean
      read GetHWFlowControl write SetHWFlowControl;
    property InSize : Word
      read GetInSize write SetInSize;
    property OutSize : Word
      read GetOutSize write SetOutSize;
    property Parity : TAxParity
      read GetParity write SetParity;
    property RS485Mode : Boolean
      read GetRS485Mode write SetRS485Mode;
    property RTS : Boolean
      read GetRTS write SetRTS;
    property Stopbits : TAxStopbits
      read GetStopbits write SetStopbits;
    property SWFlowControl : TAxSWFlowControl
      read GetSWFlowControl write SetSWFlowControl;
    property UseEventWord : Boolean
      read GetUseEventWord write SetUseEventWord;
    property XOffChar : AnsiChar
      read GetXOffChar write SetXOffChar;
    property XOnChar : AnsiChar
      read GetXOnChar write SetXOnChar;
  end;

  { General trigger event handler }
  TTriggerEvent = procedure(CP : TObject; Msg, TriggerHandle, Data : Word) of object;

  { Specific trigger event handlers }
  TTriggerAvailEvent = procedure(CP : TObject; Count : Word) of object;
  TTriggerDataEvent = procedure(CP : TObject; TriggerHandle : Word) of object;
  TTriggerStatusEvent = procedure(CP : TObject; TriggerHandle : Word) of object;
  TTriggerTimerEvent = procedure(CP : TObject; TriggerHandle : Word) of object;

  { Status event handlers }
  TTriggerLineErrorEvent = procedure(CP : TObject; Error : Word; LineBreak : Boolean) of object;

  { Port open/close callbacks }
  TPortCallback = procedure(CP : TObject; Opening : Boolean) of object;

  { For keeping track of port users }
  PUserListEntry = ^TUserListEntry;
  TUserListEntry = record
    Handle     : QObjectH;
    OpenClose  : TPortCallback;
  end;

  TAxCommBuffer = (cbInput, cbOutput);
  TAxThreadBoost = (tbNone, tbPlusOne, tbPlusTwo);

const
  { Debug log constants }
  deEnabled  = 1;
  deDisabled = 2;
  deString   = DWORD($80000000);

  { Property defaults }
  adpoDefComNumber     = 0;
  adpoDefAutoOpen      = True;
  adpoDefBaseAddress   = 0;
  adpoDefBaudRt        = 19200;
  adpoDefBufferFull    = 90;
  adpoDefBufferResume  = 10;
  adpoDefDatabits      = dbEight;
  adpoDefOpen          = False;
  adpoDefParity        = pNone;
  adpoDefStopbits      = sbOne;
  adpoDefInSize        = 4096;
  adpoDefOutSize       = 4096;
  adpoDefPromptForPort = True;
  adpoDefDTR           = True;
  adpoDefRTS           = True;
  adpoDefUseEventWord  = True;
  adpoDefSWFlowControl = swfNone;
  adpoDefXOnChar       = #17;
  adpoDefXOffChar      = #19;
  adpoDefRS485Mode     = False;

type
  { Forward declarations }
  TApxCustomComPort = class;
  TApxBaseDispatcher = class;

  TApxDispatcherThread = class(TThread)
  private
    H : TApxBaseDispatcher;
    pMsg, pTrigger : Cardinal;
    plParam : LongInt;
    pTriggerEvent : TApxNotifyEvent;
    procedure SyncEvent;
  public
    constructor Create(Disp : TApxBaseDispatcher);
    procedure SyncNotify(Msg, Trigger : Cardinal; lParam : LongInt; Event : TApxNotifyEvent);
    procedure Sync(Method : TThreadMethod);
  end;

  TReadThread = class(TApxDispatcherThread)
    procedure Execute; override;
  end;

  TWriteThread = class(TApxDispatcherThread)
    procedure Execute; override;
  end;

  { Standard COM port record }
  TApxDispatcherClass = class of TApxBaseDispatcher;
  TApxBaseDispatcher = class
  protected
    FOwner            : TApxCustomComPort;
    FComHandle        : Integer;    { COM or other device handle (returned from system) }
    LastError         : Integer;    { Last error from COM API }
    DBufSize          : Cardinal;   { Size of input buffer }
    OBufSize          : Cardinal;   { Size of output buffer }
    ModemStatus       : Cardinal;   { Modem status register }
    LastBaud          : LongInt;    { Last baud set }
    Flags             : Cardinal;   { Option flags }
    DTRAuto           : Boolean;    { True if in handshake mode }
    RTSState          : Boolean;    { Last set RTS state }
    RTSAuto           : Boolean;    { True if in handshake mode }
    LastModemStatus   : Cardinal;   { Last modem status read }
    LastLineErr       : Cardinal;   { Last line error read }

    { Trigger stuff }
    PortHandlerInstalled : Boolean; { True if any of the comport's trigger handlers <> nil }
    HandlerServiceNeeded : Boolean; { True if handlers need to be serviced }
    QtTriggerHandlers    : TList;
    ProcTriggerHandlers  : TList;
    EventTriggerHandlers : TList;
    TimerTriggers  : TList;         { Timer triggers }
    DataTriggers   : TList;         { Data triggers }
    StatusTriggers : TList;         { Status triggers }
    LastTailData   : Cardinal;      { Tail of last data checked for data }
    LastTailLen    : Cardinal;      { Tail of last data sent in len msg }
    GlobalStatHit  : Boolean;       { True if at least one status trigger hit }
    InAvailMessage : Boolean;       { True when within Avail msg }
    GetCount       : Cardinal;      { Chars looked at in Avail msg }
    MaxGetCount    : Cardinal;      { Max chars looked at in Avail }
    NotifyTail     : Cardinal;      { Position of last character notified }

    { Thread stuff }
    ThreadBoost     : TAxThreadBoost;
    ReadThread      : TReadThread;
    WriteThread     : TWriteThread;
    DataLock        : TCriticalSection;  { For all routines }
    OutputLock      : TCriticalSection;  { For output buffer and related data }
    DispLock        : TCriticalSection;  { For dispatcher buffer and related data }
    OutputEvent     : TEvent;            { Signals data ready to send }
    CurrentEvent    : DWORD;             { Current communications event }
    RingFlag        : Boolean;           { True when ringte event received }

    { Output buffer -- protected by OutputLock }
    OBuffer       : POBuffer;          { Output buffer }
    OBufHead      : Cardinal;          { Head offset in OBuffer }
    OBufTail      : Cardinal;          { Tail offset in OBuffer }
    OBufFull      : Boolean;           { Output buffer is full }

    { Dispatcher stuff -- protected by DispLock }
    DBuffer        : PDBuffer;         { Dispatch buffer }
    DBufHead       : Cardinal;         { Head offset in DBuffer }
    DBufTail       : Cardinal;         { Tail offset in DBuffer }
    DBufFull       : Boolean;          { Dispatch buffer full }

    FEventBusy     : Boolean;          { True if we're processing a COM event }
    DeletePending  : Boolean;          { True if an event handler was deleted during a busy state }
    ClosePending   : Boolean;          { True if close pending }
    OutSentPending : Boolean;          { True if stOutSent trigger pending }

    TimerID        : Cardinal;
    TriggerCounter : Cardinal;         { Last allocated trigger handle }
    DispActive     : Boolean;
    DoDonePortPrim : Boolean;
    ActiveThreads  : Integer;

    procedure ThreadGone(Sender: TObject);

    { Protected virtual dispatcher functions }
    function AvailableBytes : Integer; virtual; abstract;
    function CloseCom : Integer; virtual; abstract;
    function ConfigCom : Integer; virtual; abstract;
    function FlushBuffer(Buffer : TAxCommBuffer) : Integer; virtual;
    function GetComModemStatus(var Status : DWORD) : Integer; virtual; abstract;
    function GetLastErr : Integer; virtual; abstract;
    function InBuffCount(Head, Tail: Cardinal; Full : Boolean) : Cardinal;
    function OpenCom(const DevName : string) : Integer; virtual; abstract;
    function ReadCom(Buf : PAnsiChar; Size : Integer) : Integer; virtual; abstract;
    procedure RefreshStatus; virtual; abstract;
    function SendComBreak(mSec : DWORD) : Integer; virtual; abstract;
    procedure StartDispatcher; virtual; abstract;
    procedure StopDispatcher; virtual; abstract;
    function WaitComEvent(Timeout : DWORD) : Integer; virtual; abstract;
    function WriteCom(Buf : PAnsiChar; Size : Integer) : Integer; virtual; abstract;

    function CheckReceiveTriggers : Boolean;
    function CheckStatusTriggers : Boolean;
    function CheckTimerTriggers : Boolean;
    function CheckTriggers : Boolean;
    procedure DonePortPrim; virtual;

    function ExtractData : Boolean;
    function FindTriggerFromHandle(TriggerHandle : Cardinal; Delete : Boolean;
      var T : TTriggerType; var Trigger : Pointer) : Integer;
    function GetModemStatusPrim(ClearMask : Byte) : Byte;
    function GetTriggerHandle : Cardinal;

    procedure MapEventsToMS(Events : Integer);
    function PeekBlockPrim(Block : PAnsiChar; Offset : Cardinal; Len : Cardinal;
      var NewTail : Cardinal) : Integer;
    function PeekCharPrim(var C : AnsiChar; Count : Cardinal) : Integer;
    procedure ResetStatusHits;
    procedure ResetDataTriggers;
    function SendNotify(Msg, Trigger, Data : Cardinal) : Boolean;
  public
    DataPointers  : TDataPointerArray;            { Array of data pointers }

    constructor Create(Owner : TApxCustomComPort);
    destructor Destroy; override;

    property Active : Boolean read DispActive;
    property ComHandle : Integer read FComHandle;
    { Public virtual dispatcher functions }

    property EventBusy : Boolean read FEventBusy write FEventBusy;
    property Owner : TApxCustomComPort read FOwner;
    property DispThread : TReadThread read ReadThread;

    function AddDataTrigger(Data : PAnsiChar; IgnoreCase : Boolean) : Integer;
    function AddDataTriggerLen(Data : PAnsiChar; IgnoreCase : Boolean;
     Len : Cardinal) : Integer;
    function AddStatusTrigger(SType : Cardinal) : Integer;
    function AddTimerTrigger : Integer;
    function CheckCTS : Boolean;
    function CheckDCD : Boolean;
    function CheckDeltaCTS : Boolean;
    function CheckDeltaDSR : Boolean;
    function CheckDeltaRI : Boolean;
    function CheckDeltaDCD : Boolean;
    function CheckDSR : Boolean;
    function CheckLineBreak : Boolean;
    function CheckRI : Boolean;
    function ClassifyStatusTrigger(TriggerHandle : Cardinal) : Cardinal;
    class procedure ClearSaveBuffers(var Save : TTriggerSave);
    procedure DeregisterQtTriggerHandler(Handle : QObjectH);
    procedure DeregisterProcTriggerHandler(NP : TApxNotifyProc);
    procedure DeregisterEventTriggerHandler(NP : TApxNotifyEvent);
    procedure DonePort;
    function ExtendTimer(TriggerHandle : Cardinal;
      mSecs : DWORD) : Integer;
    function CharReady : Boolean;
    function GetBlock(Block : PAnsiChar; Len : Cardinal) : Integer;
    function GetChar(var C : AnsiChar) : Integer;
    function GetDataPointer(var P : Pointer; Index : Cardinal) : Integer;
    function GetLineError : Integer;
    function GetModemStatus : Byte;
    function InBuffUsed : Cardinal;
    function InBuffFree : Cardinal;
    function InitPort(const DeviceName : string) : Integer;
    function OutBuffUsed : Cardinal;
    function OutBuffFree : Cardinal;
    procedure ParamsChanged(Sender : TObject); dynamic;
    function PeekBlock(Block : PAnsiChar; Len : Cardinal) : Integer;
    function PeekChar(var C : AnsiChar; Count : Cardinal) : Integer;
    function ProcessCommunications : Integer; virtual; abstract;
    function PutBlock(const Block; Len : Cardinal) : Integer;
    function PutChar(C : AnsiChar) : Integer;
    function PutString(S : string) : Integer;
    procedure RegisterQtTriggerHandler(Handle : QObjectH);
    procedure RegisterProcTriggerHandler(NP : TApxNotifyProc);
    procedure RegisterSyncEventTriggerHandler(NP : TApxNotifyEvent);
    procedure RegisterEventTriggerHandler(NP : TApxNotifyEvent);
    procedure RemoveAllTriggers;
    function RemoveTrigger(TriggerHandle : Cardinal) : Integer;
    procedure RestoreTriggers(var Save : TTriggerSave);
    procedure SaveTriggers(var Save : TTriggerSave);
    procedure SetThreadBoost(Boost : TAxThreadBoost); virtual; abstract;
    function SetDataPointer(P : Pointer; Index : Cardinal) : Integer;
    procedure SetEventBusy(var WasOn : Boolean; SetOn : Boolean);
    function SetStatusTrigger(TriggerHandle : Cardinal;
      Value : Cardinal; Activate : Boolean) : Integer;
    function SetTimerTrigger(TriggerHandle : Cardinal;
      mSecs : DWORD; Activate : Boolean) : Integer;
    function TimerTimeRemaining(TriggerHandle : Cardinal;
      var mSecsRemaining : DWORD) : Longint;
    procedure UpdateHandlerFlags(FlagUpdate : TApHandlerFlagUpdate); virtual;
  end;

  TApxBaseLog = class(TPersistent)
  private
    { Property variables }
    FEnabled : Boolean;
    FFileName : TFileName;
    { Internal variables }
    blLogCS : TRTLCriticalSection;
    blOwner : TApxCustomComPort;
    { Property methods }
    function GetFileName : TFileName;
  protected
    procedure LockLog;
    procedure UnlockLog;
    function GetEnabled : Boolean;
    procedure SetEnabled(const Value : Boolean); virtual;
    procedure SetFileName(const Value : TFileName); virtual;
  public
    constructor Create(Owner : TApxCustomComPort); virtual;
    destructor Destroy; override;
    { Properties }
    property Enabled : Boolean
      read GetEnabled write SetEnabled;
    property FileName : TFileName
      read GetFileName write SetFileName;
  end;

  TApxWriteMode = (wmOverwrite, wmAppend);

  { Record for log entries }
  PApxDebugRec = ^TApxDebugRec;
  TApxDebugRec = record
    drClass : TApxComponentClass;
    drTime : DWORD;
    drData1 : DWORD;
    drData2 : DWORD;
    drData3 : DWORD;
  end;

  PApxDebugBuffer = ^TApxDebugBuffer;
  TApxDebugBuffer = array[0..MaxDebugLog] of Byte;

  TApxDebugLog = class(TApxBaseLog)
  private
    { Property variables }
    FBufferSize : DWORD;
    FWriteMode : TApxWriteMode;
    { Private variables }
    dlBuffer : PApxDebugBuffer;
    dlBufferHead : DWORD;
    dlBufferTail : DWORD;
    dlTempBuffer : PByteArray;
    dlTempSize : DWORD;
    dlTimeBase : DWORD;
  protected
    { Property access methods }
    function GetBufferFree : DWORD;
    function GetBufferSize : DWORD;
    function GetWriteMode : TApxWriteMode;
    procedure SetBufferSize(const Value : DWORD);
    procedure SetEnabled(const Value : Boolean); override;
    procedure SetWriteMode(const Value : TApxWriteMode);
    { Internal methods }
    procedure dlCheckTempSize(SizeReq : DWORD);
    function dlDoFileHeader : string;
    function dlPopDebugEntry(var DebugRec : TApxDebugRec) : Boolean;
    function dlTimeStamp(Mark : DWORD) : string;
  public
    { Public methods }
    constructor Create(Owner : TApxCustomComPort); override;
    destructor Destroy; override;
    procedure AddDebugEntry(const deClass : TApxComponentClass; const D1, D2, D3 : DWORD);
    procedure ClearBuffer;
    procedure DumpLog;
    procedure WriteDebugString(const DebugString : string);
    { Public properties }
    property BufferFree : DWORD read GetBufferFree;
  published
    { Published properties }
    property BufferSize : DWORD read GetBufferSize write SetBufferSize;
    property WriteMode : TApxWriteMode read GetWriteMode write SetWriteMode;
    { Inherited properties }
    property Enabled;
    property FileName;
  end;

  { Port component }
  TApxCustomComPort = class(TApxBaseHandleComponent)
  protected { private }
    { Internal stuff }
    CopyTriggers      : Boolean;               { Copy triggers on open }
    DeviceParams      : TApxSerialDeviceParams;{ Line configuration options }
    ForceOpen         : Boolean;               { Force open after loading }
    OpenPending       : Boolean;               { True if Open := True while shutting down }
    PortState         : TPortState;            { State of the physical port/dispatcher }
    SaveTriggerBuffer : TTriggerSave;          { Triggers to copy }
    UserList          : TList;                 { List of comport users }
    FMasterTerminal   : TWidgetControl;        { The terminal that replies to requests }

    { Port info }
    FComNumber        : Word;                  { Number corresponding to physical COM port }
    FDeviceLayer      : string;                { Device layer for this port }
    FDeviceName       : string;                { Device name }
    FDispatcher       : TApxBaseDispatcher;    { Handle to dispatcher object }

    { Parameters }
    FOpen             : Boolean;               { True if the port is open }
    FPromptForPort    : Boolean;               { True to display the comport selection dialog if no port is selected }
    FAutoOpen         : Boolean;               { True to do implicit opens }
    FThreadBoost      : TAxThreadBoost;        { Boost for dispatcher threads }

    { Debugging }
    FDebugLog         : TApxDebugLog;          { Debug log }

    { Options }
    FUseEventWord     : Boolean;               { True to use the EventWord }

    { Triggers }
    FOnTrigger        : TTriggerEvent;         { All-encompassing event handler }
    FOnTriggerAvail   : TTriggerAvailEvent;    { APX_TRIGGERAVAIL events }
    FOnTriggerData    : TTriggerDataEvent;     { APX_TRIGGERDATA events }
    FOnTriggerStatus  : TTriggerStatusEvent;   { APX_TRIGGERSTATUS events }
    FOnTriggerTimer   : TTriggerTimerEvent;    { APX_TRIGGERTIMER events }
    FOnTriggerLineError   : TTriggerLineErrorEvent;  { Got line error }
    FOnTriggerModemStatus : TNotifyEvent;      { Got modem status change }
    FOnTriggerOutbuffFree : TNotifyEvent;      { Outbuff free above mark }
    FOnTriggerOutbuffUsed : TNotifyEvent;      { Outbuff used above mark }
    FOnTriggerOutSent     : TNotifyEvent;      { Data was transmitted }

    FOnPortOpen       : TNotifyEvent;          { Port just opened }
    FOnPortClose      : TNotifyEvent;          { Port just closed }

    { Property read methods }
    function GetBaseAddress : Word;
    function GetBaud : LongInt;
    function GetBufferFull : Byte;
    function GetBufferResume : Byte;
    function GetCTS : Boolean;
    function GetDatabits : TAxDatabits;
    function GetDCD : Boolean;
    function GetDeltaCTS : Boolean;
    function GetDeltaDCD : Boolean;
    function GetDeltaDSR : Boolean;
    function GetDeltaRI : Boolean;
    function GetDispatcher : TApxBaseDispatcher;
    function GetDSR : Boolean;
    function GetDTR : Boolean;
    function GetHWFlowControl : Boolean;
    function GetInBuffFree : Word;
    function GetInBuffUsed : Word;
    function GetInSize : Word;
    function GetLineError : Word;
    function GetLineBreak : Boolean;
    function GetModemStatus : Byte;
    function GetOutBuffFree : Word;
    function GetOutBuffUsed : Word;
    function GetOutSize : Word;
    function GetParity : TAxParity;
    function GetRI : Boolean;
    function GetRS485Mode : Boolean;
    function GetRTS : Boolean;
    function GetStopbits : TAxStopbits;
    function GetSWFlowControl : TAxSWFlowControl;
    function GetXOffChar : AnsiChar;
    function GetXOnChar : AnsiChar;

    { Property write methods }
    procedure SetBaseAddress(const Value : Word);
    procedure SetBaud(const Value : LongInt);
    procedure SetBufferFull(const Value : Byte);
    procedure SetBufferResume(const Value : Byte);
    procedure SetComNumber(const Value : Word);
    procedure SetDatabits(const Value : TAxDatabits);
    procedure SetDeviceLayer(const Value : string);
    procedure SetDeviceName(const Value : string);
    procedure SetDTR(const Value : Boolean);
    procedure SetHWFlowControl(const Value : Boolean);
    procedure SetInSize(const Value : Word);
    procedure SetOpen(const Value : Boolean);
    procedure SetOutSize(const Value : Word);
    procedure SetParity(const Value : TAxParity);
    procedure SetRS485Mode(const Value : Boolean);
    procedure SetRTS(const Value : Boolean);
    procedure SetStopbits(const Value : TAxStopbits);
    procedure SetSWFlowControl(const Value : TAxSWFlowControl);
    procedure SetThreadBoost(const Value : TAxThreadBoost);
    procedure SetUseEventWord(const Value : Boolean);
    procedure SetXOffChar(const Value : AnsiChar);
    procedure SetXOnChar(const Value : AnsiChar);

    { Trigger write methods }
    procedure SetOnTrigger(const Value : TTriggerEvent);
    procedure SetOnTriggerAvail(const Value : TTriggerAvailEvent);
    procedure SetOnTriggerData(const Value : TTriggerDataEvent);
    procedure SetOnTriggerLineError(const Value : TTriggerLineErrorEvent);
    procedure SetOnTriggerModemStatus(const Value : TNotifyEvent);
    procedure SetOnTriggerOutbuffFree(const Value : TNotifyEvent);
    procedure SetOnTriggerOutbuffUsed(const Value : TNotifyEvent);
    procedure SetOnTriggerOutSent(const Value : TNotifyEvent);
    procedure SetOnTriggerStatus(const Value : TTriggerStatusEvent);
    procedure SetOnTriggerTimer(const Value : TTriggerTimerEvent);
  protected
    { Misc }
    procedure CreateWidget; override;
    procedure Loaded; override;
    procedure RegisterComPort(Enabling : Boolean); virtual;
    procedure ValidateComport; virtual;

    { Message handlers }
    procedure APXClosePending(var Message : TAxMessage); message APX_CLOSEPENDING;
    procedure APXTriggerAvail(var Message : TAxMessage); message APX_TRIGGERAVAIL;
    procedure APXTriggerData(var Message : TAxMessage); message APX_TRIGGERDATA;
    procedure APXTriggerStatus(var Message : TAxMessage); message APX_TRIGGERSTATUS;
    procedure APXTriggerTimer(var Message : TAxMessage); message APX_TRIGGERTIMER;

    { Trigger event methods }
    procedure Trigger(Msg, TriggerHandle, Data : Word); virtual;
    procedure TriggerAvail(Count : Word); virtual;
    procedure TriggerData(TriggerHandle : Word); virtual;
    procedure TriggerStatus(TriggerHandle : Word); virtual;
    procedure TriggerTimer(TriggerHandle : Word); virtual;
    procedure UpdateHandlerFlag; virtual;

    { Port open/close/change event methods }
    procedure PortOpen; dynamic;
    procedure PortClose; dynamic;

    { Status trigger methods }
    function EventFilter(Sender: QObjectH; Event: QEventH): Boolean; override;
    procedure TriggerLineError(const Error : Word; const LineBreak : Boolean); virtual;
    procedure TriggerModemStatus; virtual;
    procedure TriggerOutbuffFree; virtual;
    procedure TriggerOutbuffUsed; virtual;
    procedure TriggerOutSent; virtual;

  public
    { Creation/destruction }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;

    { Class methods }
    class function GetLogString(const D1, D2, D3 : DWORD) : string; override;
    class procedure RegisterDeviceLayer(const ATag : string; AClass : TApxDispatcherClass);
    class procedure UnRegisterDeviceLayerClass(AClass : TApxDispatcherClass);

    { General }
    procedure AddStringToLog(S : string);
    procedure Assign(Source: TPersistent); override;
    procedure DeregisterUser(const H : QObjectH);
    procedure DeregisterUserCallback(CallBack : TPortCallback);
    procedure DonePort; virtual;
    procedure FlushInBuffer;
    procedure FlushOutBuffer;
    procedure ForcePortOpen;
    procedure GetDeviceLayerList(const DevList : TStringList);
    procedure InitPort; dynamic;
    procedure ProcessCommunications; virtual;
    procedure RegisterUser(const H : QObjectH);
    procedure RegisterUserCallback(CallBack : TPortCallback);
    procedure SendBreak(mSecs : DWORD);
    function ValidDispatcher : TApxBaseDispatcher;

    { Trigger management }
    function AddDataTrigger(const Data : ShortString; const IgnoreCase : Boolean) : Word;
    function AddStatusTrigger(const SType : Word) : Word;
    function AddTimerTrigger : Word;
    procedure RemoveAllTriggers;
    procedure RemoveTrigger(const Handle : Word);
    procedure SetStatusTrigger(const Handle : Word; const Value : Word; const Activate : Boolean);
    procedure SetTimerTrigger(const Handle : Word; const mSecs : DWORD; const Activate : Boolean);

    { I/O }
    function CharReady : Boolean;
    procedure GetBlock(var Block; const Len : Word);
    function GetChar : AnsiChar;
    procedure PeekBlock(var Block; const Len : Word);
    function PeekChar(const Count : Word) : AnsiChar;
    function PutBlock(const Block; const Len : Word) : Integer;
    procedure PutChar(const C : AnsiChar);
    procedure PutString(const S : string);

    { Properties }
    property ComNumber : Word
      read FComNumber write SetComNumber default adpoDefComNumber;
    property DeviceLayer : string
      read FDeviceLayer write SetDeviceLayer;
    property DeviceName : string
      read FDeviceName write SetDeviceName;
    property Baud : LongInt
      read GetBaud write SetBaud default adpoDefBaudRt;
    property Parity : TAxParity
      read GetParity write SetParity default adpoDefParity;
    property PromptForPort : Boolean
      read FPromptForPort write FPromptForPort
      default adpoDefPromptForPort;
    property DataBits : TAxDatabits
      read GetDatabits write SetDatabits default adpoDefDatabits;
    property StopBits : TAxStopbits
      read GetStopbits write SetStopbits default adpoDefStopbits;

    { Miscellaneous port properties }
    property InSize : Word
      read GetInSize write SetInSize default adpoDefInSize;
    property OutSize : Word
      read GetOutSize write SetOutSize default adpoDefOutSize;
    property Open : Boolean
      read FOpen write SetOpen default adpoDefOpen;
    property AutoOpen : Boolean
      read FAutoOpen write FAutoOpen default adpoDefAutoOpen;
    property RS485Mode : Boolean
      read GetRS485Mode write SetRS485Mode default adpoDefRS485Mode;
    property BaseAddress : Word
      read GetBaseAddress write SetBaseAddress
      default adpoDefBaseAddress;
    property ThreadBoost : TAxThreadBoost
      read FThreadBoost write SetThreadBoost;
    property MasterTerminal : TWidgetControl
      read FMasterTerminal write FMasterTerminal;

    { Modem control/status }
    property DTR : Boolean
      read GetDTR write SetDTR default adpoDefDTR;
    property RTS : Boolean
      read GetRTS write SetRTS default adpoDefRTS;

    { Flow control properties }
    property HWFlowControl : Boolean
      read GetHWFlowControl write SetHWFlowControl default True;
    property SWFlowControl : TAxSWFlowControl
      read GetSWFlowControl write SetSWFlowControl default adpoDefSWFlowControl;
    property XOffChar : AnsiChar
      read GetXOffChar write SetXoffChar default adpoDefXOffChar;
    property XOnChar : AnsiChar
      read GetXonChar write SetXonChar default adpoDefXOnChar;
    property BufferFull : Byte
      read GetBufferFull write SetBufferFull default adpoDefBufferFull;
    property BufferResume : Byte
      read GetBufferResume write SetBufferResume default adpoDefBufferResume;

    { Debugging }
    property DebugLog : TApxDebugLog
      read FDebugLog write FDebugLog;

    { Options }
    property UseEventWord : Boolean
      read FUseEventWord write SetUseEventWord default adpoDefUseEventWord;

    { Trigger events }
    property OnTrigger : TTriggerEvent
      read FOnTrigger write SetOnTrigger;
    property OnTriggerAvail : TTriggerAvailEvent
      read FOnTriggerAvail write SetOnTriggerAvail;
    property OnTriggerData : TTriggerDataEvent
      read FOnTriggerData write SetOnTriggerData;
    property OnTriggerStatus : TTriggerStatusEvent
      read FOnTriggerStatus write SetOnTriggerStatus;
    property OnTriggerTimer : TTriggerTimerEvent
      read FOnTriggerTimer write SetOnTriggerTimer;

    { Port open/close/change events }
    property OnPortOpen : TNotifyEvent
      read FOnPortOpen write FOnPortOpen;
    property OnPortClose : TNotifyEvent
      read FOnPortClose write FOnPortClose;

    { Status events }
    property OnTriggerLineError : TTriggerLineErrorEvent
      read FOnTriggerLineError write SetOnTriggerLineError;
    property OnTriggerModemStatus : TNotifyEvent
      read FOnTriggerModemStatus write SetOnTriggerModemStatus;
    property OnTriggerOutbuffFree : TNotifyEvent
      read FOnTriggerOutbuffFree write SetOnTriggerOutbuffFree;
    property OnTriggerOutbuffUsed : TNotifyEvent
      read FOnTriggerOutbuffUsed write SetOnTriggerOutbuffUsed;
    property OnTriggerOutSent : TNotifyEvent
      read FOnTriggerOutSent write SetOnTriggerOutSent;

    { I/O properties }
    property Output : string
      write PutString;

    { TComHandle, read only }
    property Dispatcher : TApxBaseDispatcher
      read GetDispatcher;

    { Modem status, read only }
    property ModemStatus : Byte
      read GetModemStatus;
    property DSR : Boolean
      read GetDSR;
    property CTS : Boolean
      read GetCTS;
    property RI : Boolean
      read GetRI;
    property DCD : Boolean
      read GetDCD;
    property DeltaDSR : Boolean
      read GetDeltaDSR;
    property DeltaCTS : Boolean
      read GetDeltaCTS;
    property DeltaRI : Boolean
      read GetDeltaRI;
    property DeltaDCD : Boolean
      read GetDeltaDCD;

    { Line errors }
    property LineError : Word
      read GetLineError;
    property LineBreak : Boolean
      read GetLineBreak;

    { Buffer info, readonly }
    property InBuffUsed : Word
      read GetInBuffUsed;
    property InBuffFree : Word
      read GetInBuffFree;
    property OutBuffUsed : Word
      read GetOutBuffUsed;
    property OutBuffFree : Word
      read GetOutBuffFree;
  end;

  { Port component }
  TApxComPort = class(TApxCustomComPort)
  published
    { Properties }
    property AutoOpen;
    property Baud;
    property BufferFull;
    property BufferResume;
    property ComNumber;
    property DataBits;
    property DebugLog;
    property DeviceLayer;
    property DTR;
    property HWFlowControl;
    property InSize;
    property Open;
    property OutSize;
    property Parity;
    property PromptForPort;
    property RS485Mode;
    property RTS;
    property StopBits;
    property SWFlowControl;
    property XOffChar;
    property XOnChar;
    property UseEventWord;
    { Events }
    property OnTrigger;
    property OnTriggerAvail;
    property OnTriggerData;
    property OnTriggerStatus;
    property OnTriggerTimer;
    property OnTriggerLineError;
    property OnTriggerModemStatus;
    property OnTriggerOutbuffFree;
    property OnTriggerOutbuffUsed;
    property OnTriggerOutSent;
  end;

  function SearchComPort(const C : TComponent) : TApxCustomComPort;

implementation

uses
  {$IFDEF Win32}
  AxWin32;
  {$ENDIF}
  {$IFDEF Linux}
  AxLinux;
  {$ENDIF}

{$IFDEF Linux}
threadvar
  MsgEvent : TEvent;
{$ENDIF}

const
  LastCID : Integer = -1;
  LastDispatcher : TApxBaseDispatcher = nil;

{ TApxBaseDeviceParams }

{ Create instance }
constructor TApxBaseDeviceParams.Create;
begin
  inherited;
  bdCS := TCriticalSection.Create;
end;

{ Destroy instance }
destructor TApxBaseDeviceParams.Destroy;
begin
  bdCS.Free;
  inherited;
end;

{ Increments update counter }
procedure TApxBaseDeviceParams.BeginUpdate;
begin
  Lock;
  try
    Inc(bdUpdateCount);
  finally
    Unlock;
  end;
end;

{ Fires OnChange event if appropriate }
procedure TApxBaseDeviceParams.Changed;
begin
  Lock;
  try
    if Assigned(FOnChange) and (bdUpdateCount = 0) then
      FOnChange(Self);
  finally
    Unlock;
  end;
end;

{ Decrements update counter }
procedure TApxBaseDeviceParams.EndUpdate;
begin
  Lock;
  try
    Assert(bdUpdateCount > 0, 'Unbalanced BeginUpdate/EndUpdate');
    Dec(bdUpdateCount);
    Changed;
  finally
    Unlock;
  end;
end;

{ Enters critical section }
procedure TApxBaseDeviceParams.Lock;
begin
  bdCS.Enter;
end;

{ Leaves critical section }
procedure TApxBaseDeviceParams.Unlock;
begin
  bdCS.Leave;
end;

{ TApxSerialDeviceConfig }

{ Returns the BaseAddress (optionally used for RS485Mode) }
function TApxSerialDeviceParams.GetBaseAddress: Word;
begin
  Lock;
  try
    Result := FBaseAddress;
  finally
    Unlock;
  end;
end;

{ Returns the baud rate }
function TApxSerialDeviceParams.GetBaud: LongInt;
begin
  Lock;
  try
    Result := FBaud;
  finally
    Unlock;
  end;
end;

{ Returns the Buffer Full mark }
function TApxSerialDeviceParams.GetBufferFull : Byte;
begin
  Lock;
  try
    Result := FBufferFull;
  finally
    Unlock;
  end;
end;

{ Returns the Buffer Resume mark }
function TApxSerialDeviceParams.GetBufferResume : Byte;
begin
  Lock;
  try
    Result := FBufferResume;
  finally
    Unlock;
  end;
end;

{ Returns the line databits }
function TApxSerialDeviceParams.GetDatabits: TAxDatabits;
begin
  Lock;
  try
    Result := FDatabits;
  finally
    Unlock;
  end;
end;

{ Returns the DTR state }
function TApxSerialDeviceParams.GetDTR: Boolean;
begin
  Lock;
  try
    Result := FDTR;
  finally
    Unlock;
  end;
end;

{ Returns the HWFlowControl setting }
function TApxSerialDeviceParams.GetHWFlowControl : Boolean;
begin
  Lock;
  try
    Result := FHWFlowControl;
  finally
    Unlock;
  end;
end;

{ Returns the input buffer size }
function TApxSerialDeviceParams.GetInSize: Word;
begin
  Lock;
  try
    Result := FInSize;
  finally
    Unlock;
  end;
end;

{ Returns the output buffer size }
function TApxSerialDeviceParams.GetOutSize: Word;
begin
  Lock;
  try
    Result := FOutSize;
  finally
    Unlock;
  end;
end;

{ Returns the line parity }
function TApxSerialDeviceParams.GetParity: TAxParity;
begin
  Lock;
  try
    Result := FParity;
  finally
    Unlock;
  end;
end;

{ Returns RS485 mode setting }
function TApxSerialDeviceParams.GetRS485Mode: Boolean;
begin
  Lock;
  try
    Result := FRS485Mode;
  finally
    Unlock;
  end;
end;

{ Returns the RTS state }
function TApxSerialDeviceParams.GetRTS : Boolean;
begin
  Lock;
  try
    Result := FRTS;
  finally
    Unlock;
  end;
end;

{ Returns the line stopbits }
function TApxSerialDeviceParams.GetStopbits : TAxStopbits;
begin
  Lock;
  try
    Result := FStopbits;
  finally
    Unlock;
  end;
end;

{ Returns the SW flow control setting }
function TApxSerialDeviceParams.GetSWFlowControl : TAxSWFlowControl;
begin
  Lock;
  try
    Result := FSWFlowControl;
  finally
    Unlock;
  end;
end;

{ Returns whether the event word will be used }
function TApxSerialDeviceParams.GetUseEventWord : Boolean;
begin
  Lock;
  try
    Result := FUseEventWord;
  finally
    Unlock;
  end;
end;

{ Returns the XOff character used for SW flow control }
function TApxSerialDeviceParams.GetXoffChar : AnsiChar;
begin
  Lock;
  try
    Result := FXoffChar;
  finally
    Unlock;
  end;
end;

{ Returns the XOn character used for SW flow control }
function TApxSerialDeviceParams.GetXonChar : AnsiChar;
begin
  Lock;
  try
    Result := FXonChar;
  finally
    Unlock;
  end;
end;

{ Changes the BaseAddress (optionally used for RS485Mode) }
procedure TApxSerialDeviceParams.SetBaseAddress(const Value : Word);
begin
  Lock;
  try
    { Set new value if different }
    if FBaseAddress <> Value then begin
      FBaseAddress := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the baud rate }
procedure TApxSerialDeviceParams.SetBaud(const Value : LongInt);
begin
  Lock;
  try
    { Set new value if different }
    if FBaud <> Value then begin
      FBaud := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Set Buffer Full mark }
procedure TApxSerialDeviceParams.SetBufferFull(const Value : Byte);
var
  Temp : Byte;
begin
  { Clamp Value if necessary }
  if Value > 100 then
    Temp := 100
  else
    Temp := Value;

  Lock;
  try
    { Set new value if different }
    if FBufferFull <> Temp then begin
      FBufferFull := Temp;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Set Buffer Resume mark }
procedure TApxSerialDeviceParams.SetBufferResume(const Value : Byte);
var
  Temp : Byte;
begin
  { Clamp Value if necessary }
  if Value > 100 then
    Temp := 100
  else
    Temp := Value;

  Lock;
  try
    { Set new value if different }
    if FBufferResume <> Temp then begin
      FBufferResume := Temp;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the line databits }
procedure TApxSerialDeviceParams.SetDatabits(const Value : TAxDatabits);
begin
  Lock;
  try
    { Set new value if different }
    if FDatabits <> Value then begin
      FDatabits := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the DTR state }
procedure TApxSerialDeviceParams.SetDTR(const Value : Boolean);
begin
  Lock;
  try
    { Set new value if different }
    if FDTR <> Value then begin
      FDTR := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the HWFlowControl setting }
procedure TApxSerialDeviceParams.SetHWFlowControl(const Value : Boolean);
begin
  Lock;
  try
    { Set new value if different }
    if FHWFlowControl <> Value then begin
      FHWFlowControl := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the input buffer size }
procedure TApxSerialDeviceParams.SetInSize(const Value : Word);
begin
  Lock;
  try
    { Set new value if different }
    if FInSize <> Value then begin
      FInSize := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the output buffer size }
procedure TApxSerialDeviceParams.SetOutSize(const Value : Word);
begin
  Lock;
  try
    { Set new value if different }
    if FOutSize <> Value then begin
      FOutSize := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the line parity }
procedure TApxSerialDeviceParams.SetParity(const Value: TAxParity);
begin
  Lock;
  try
    { Set new value if different }
    if FParity <> Value then begin
      FParity := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes RS485 mode setting }
procedure TApxSerialDeviceParams.SetRS485Mode(const Value: Boolean);
begin
  Lock;
  try
    { Set new value if different }
    if FRS485Mode <> Value then begin
      FRS485Mode := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the RTS state }
procedure TApxSerialDeviceParams.SetRTS(const Value: Boolean);
begin
  Lock;
  try
    { Set new value if different }
    if FRTS <> Value then begin
      FRTS := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the line stopbits }
procedure TApxSerialDeviceParams.SetStopbits(const Value: TAxStopbits);
begin
  Lock;
  try
    { Set new value if different }
    if FStopbits <> Value then begin
      FStopbits := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the SW flow control setting }
procedure TApxSerialDeviceParams.SetSWFlowControl(const Value : TAxSWFlowControl);
begin
  Lock;
  try
    { Set new value if different }
    if FSWFlowControl <> Value then begin
      FSWFlowControl := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Controls whether the event word will be used }
procedure TApxSerialDeviceParams.SetUseEventWord(const Value: Boolean);
begin
  Lock;
  try
    { Set new value if different }
    if FUseEventWord <> Value then begin
      FUseEventWord := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the XOff character for SWFlowControl }
procedure TApxSerialDeviceParams.SetXOffChar(const Value : AnsiChar);
begin
  Lock;
  try
    { Set new value if different }
    if FXOffChar <> Value then begin
      FXOffChar := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Changes the XOn character for SWFlowControl }
procedure TApxSerialDeviceParams.SetXOnChar(const Value : AnsiChar);
begin
  Lock;
  try
    { Set new value if different }
    if FXOnChar <> Value then begin
      FXOnChar := Value;
      Changed;
    end;
  finally
    Unlock;
  end;
end;

{ Tracks thread inventory }
procedure TApxBaseDispatcher.ThreadGone(Sender: TObject);
begin
  if Sender = WriteThread then begin
    WriteThread := nil;
  end;

  if Sender = ReadThread then begin
    ReadThread := nil;
    if DoDonePortPrim then begin
      DonePortPrim;
      DoDonePortPrim := False;
    end;
  end;
end;

{ Create instance of dispatcher }
constructor TApxBaseDispatcher.Create(Owner : TApxCustomComPort);
begin
  inherited Create;
  FOwner := Owner;

  { Set our params hook }
  Owner.DeviceParams.OnChange := ParamsChanged;

  { Allocate critical section objects }
  DataLock := TCriticalSection.Create;
  OutputLock := TCriticalSection.Create;
  DispLock := TCriticalSection.Create;

  { Allocate events }
  OutputEvent := TEvent.Create(nil, False, False, 'AxOutputEvent');

  { Aoolocate trigger handler lists }
  QtTriggerHandlers := TList.Create;
  ProcTriggerHandlers := TList.Create;
  EventTriggerHandlers := TList.Create;
  TimerTriggers := TList.Create;
  DataTriggers  := TList.Create;
  StatusTriggers:= TList.Create;
  TriggerCounter := FirstTriggerCounter;
end;

{ Destroys instance of dispatcher }
destructor TApxBaseDispatcher.Destroy;
begin
  if ClosePending then
    DonePortPrim
  else
    DonePort;

  { Clear params hook }
  Owner.DeviceParams.OnChange := nil;

  { Delete timer triggers }
  while TimerTriggers.Count > 0 do begin
    Dispose(PTimerTrigger(TimerTriggers[0]));
    TimerTriggers.Delete(0);
  end;
  TimerTriggers.Free;

  { Delete data triggers }
  while DataTriggers.Count > 0 do begin
    Dispose(PDataTrigger(DataTriggers[0]));
    DataTriggers.Delete(0);
  end;
  DataTriggers.Free;

  { Delete status triggers }
  while StatusTriggers.Count > 0 do begin
    Dispose(PStatusTrigger(StatusTriggers[0]));
    StatusTriggers.Delete(0);
  end;
  StatusTriggers.Free;

  { Delete Qt trigger handlers }
  while QtTriggerHandlers.Count > 0 do begin
    Dispose(PQtTriggerHandler(QtTriggerHandlers[0]));
    QtTriggerHandlers.Delete(0);
  end;
  QtTriggerHandlers.Free;

  { Delete Proc trigger handlers }
  while ProcTriggerHandlers.Count > 0 do begin
    Dispose(PProcTriggerHandler(ProcTriggerHandlers[0]));
    ProcTriggerHandlers.Delete(0);
  end;
  ProcTriggerHandlers.Free;

  { Delete Event trigger handlers }
  while EventTriggerHandlers.Count > 0 do begin
    Dispose(PEventTriggerHandler(EventTriggerHandlers[0]));
    EventTriggerHandlers.Delete(0);
  end;
  EventTriggerHandlers.Free;

  { Free the critical sections }
  DataLock.Free;
  OutputLock.Free;
  DispLock.Free;

  { Free the Events }
  OutputEvent.Free;

  inherited Destroy;
end;

{ Return number of chars between Tail and Head }
function TApxBaseDispatcher.InBuffCount(Head, Tail : Cardinal; Full : Boolean) : Cardinal;
begin
  if Head = Tail then
    if Full then
      Result := DBufSize
    else
      Result := 0
  else if Head > Tail then
    Result := Head-Tail
  else
    Result := (Head+DBufSize)-Tail;
end;

{ Set bits in ModemStatus according to flags in Events }
procedure TApxBaseDispatcher.MapEventsToMS(Events : Integer);
var
  OldMS : Byte;
  Delta : Byte;
begin
  { Note old, get new }
  OldMS := ModemStatus;
  GetModemStatusPrim($FF);

  { Set delta bits }
  Delta := (OldMS xor ModemStatus) and $F0;
  ModemStatus := ModemStatus or (Delta shr 4);
end;

{ Routines used by constructor }

{ Remove all triggers }
procedure TApxBaseDispatcher.RemoveAllTriggers;
begin
  DataLock.Enter;
  try
    while TimerTriggers.Count > 0 do begin
      Dispose(PTimerTrigger(TimerTriggers[0]));
      TimerTriggers.Delete(0);
    end;
    while DataTriggers.Count > 0 do begin
      Dispose(PDataTrigger(DataTriggers[0]));
      DataTriggers.Delete(0);
    end;
    while StatusTriggers.Count > 0 do begin
      Dispose(PStatusTrigger(StatusTriggers[0]));
      StatusTriggers.Delete(0);
    end;

    TriggerCounter := FirstTriggerCounter;
  finally
    DataLock.Leave;
  end;
end;

procedure TApxBaseDispatcher.ResetStatusHits;
var
  i : Integer;
begin
  for i := pred(StatusTriggers.Count) downto 0 do
    PStatusTrigger(StatusTriggers[i])^.StatusHit := False;
  GlobalStatHit := False;
end;

procedure TApxBaseDispatcher.ResetDataTriggers;
var
  i : Integer;
begin
  for i := pred(DataTriggers.Count) downto 0 do
    with PDataTrigger(DataTriggers[i])^ do
      FillChar(tChkIndex, SizeOf(TCheckIndex), 0);
end;

function TApxBaseDispatcher.InitPort(const DeviceName : string) : Integer;
begin
  RingFlag := False;
  Result := ecOk;

  { Required inits in case pDonePort is called }
  DBuffer := nil;
  OBuffer := nil;
  FEventBusy := False;
  DeletePending := False;

  { Ask system to open the COM port }
  FComHandle := OpenCom(DeviceName);
  if FComHandle < 0 then begin
    Result := -Integer(GetLastErr);
    DonePort;
    Exit;
  end else begin
    { Log port open }
    FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclOpenCom,
      DWORD(DeviceName), Length(DeviceName));
  end;

  { Allocate dispatch buffer }
  DBuffer := AllocMem(Owner.InSize);
  DBufSize := Owner.InSize;

  DBufHead := 0;
  DBufTail := 0;
  DBufFull := False;

  { Allocate output buffer }
  OBuffer := AllocMem(Owner.OutSize);
  OBufSize := Owner.OutSize;

  OBufHead := 0;
  OBufTail := 0;
  OBufFull := False;

  { Initialize fields }
  LastError := 0;
  OutSentPending := False;
  ClosePending := False;
  GetCount := 0;
  LastLineErr := 0;
  LastModemStatus := 0;

  { Set all line parameters }
  ConfigCom;

  { Trigger inits }
  LastTailData := 0;
  LastTailLen := 1;
  RemoveAllTriggers;

  NotifyTail := 0;
  ResetStatusHits;

  InAvailMessage := False;

  ModemStatus := 0;
  GetModemStatusPrim($F0);

  { Set the requested line parameters }
  LastBaud := 115200;

  { Get initial status }
  RefreshStatus;

  { Start dispatcher }
  StartDispatcher;
end;

{ Close the port and free the handle }
procedure TApxBaseDispatcher.DonePortPrim;
begin
  { Stop dispatcher }
  if DispActive then
    StopDispatcher;

  if OBuffer <> nil then begin
    FreeMem(OBuffer);
    OBuffer := nil;
  end;

  { Free memory }
  if DBuffer <> nil then begin
    FreeMem(DBuffer);
    DBuffer := nil;
  end;
end;

{ Close the port and free the handle }
procedure TApxBaseDispatcher.DonePort;
begin
  { Always close the physical port... }
  if FComHandle >= 0 then begin
    { Flush the buffers }
    FlushBuffer(cbOutput);
    FlushBuffer(cbInput);

    CloseCom;
    { Log CloseCom }
    FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclCloseCom, 0, 0);
  end;

  { ...but destroy our object only if not within a notify }
  if FEventBusy then begin
    ClosePending := True;
  end else
    DonePortPrim;
end;

function ActualBaud(BaudCode : LongInt) : LongInt;
const
  BaudTable : array[0..23] of LongInt =
    (110,    300,    600,    1200,    2400,    4800,    9600,    14400,
     19200,  0,      0,      38400,   0,       0,       0,       56000,
     0,      0,      0,      128000,  0,       0,       0,       256000);
var
  Index : Cardinal;
  Baud : LongInt;
begin
  if BaudCode = $FEFF then
    { COMM.DRV's 115200 hack }
    Result := 115200
  else if BaudCode < $FF10 then
    { Must be a baud rate, return it }
    Result := BaudCode
  else begin
    { It's a code, look it up }
    Index := BaudCode - $FF10;
    if Index > 23 then
      { Unknown code, just return it }
      Result := BaudCode
    else begin
      Baud := BaudTable[Index];
      if Baud = 0 then
        { Unknown code, just return it }
        Result := BaudCode
      else
        Result := Baud;
    end;
  end;
end;

{ Primitive to return the modem status and clear mask }
function TApxBaseDispatcher.GetModemStatusPrim(ClearMask : Byte) : Byte;
var
  Data : DWORD;
begin
  DataLock.Enter;
  try

    { Get the new absolute values }
    GetComModemStatus(Data);
    ModemStatus := (ModemStatus and $0F) or Byte(Data);

    { Special case, transfer RI bit to TERI bit }
    if RingFlag then begin
      RingFlag := False;
      ModemStatus := ModemStatus or $04;
    end;

    { Return the current ModemStatus value }
    Result := Lo(ModemStatus);

    { Clear specified delta bits }
    ModemStatus := ModemStatus and Clearmask;

  finally
    DataLock.Leave;
  end;
end;

{ Return the modem status byte and clear the delta bits }
function TApxBaseDispatcher.GetModemStatus : Byte;
begin
  Result := GetModemStatusPrim(ClearDelta);
end;

{ Returns True if CTS is high }
function TApxBaseDispatcher.CheckCTS : Boolean;
begin
  Result := GetModemStatusPrim(ClearDeltaCTS) and CTSMask = CTSMask;
end;

{ Returns True if DSR is high }
function TApxBaseDispatcher.CheckDSR : Boolean;
begin
  Result := GetModemStatusPrim(ClearDeltaDSR) and DSRMask = DSRMask;
end;

{ Returns True if RI is high }
function TApxBaseDispatcher.CheckRI : Boolean;
begin
  Result := GetModemStatusPrim(ClearDeltaRI) and RIMask = RIMask;
end;

{ Returns True if DCD is high }
function TApxBaseDispatcher.CheckDCD : Boolean;
begin
  Result := GetModemStatusPrim(ClearDeltaDCD) and DCDMask = DCDMask;
end;

{ Returns True if DeltaCTS is high }
function TApxBaseDispatcher.CheckDeltaCTS : Boolean;
begin
  Result := GetModemStatusPrim(ClearDeltaCTS) and DeltaCTSMask = DeltaCTSMask;
end;

{ Returns True if DeltaDSR is high }
function TApxBaseDispatcher.CheckDeltaDSR : Boolean;
begin
  Result := GetModemStatusPrim(ClearDeltaDSR) and DeltaDSRMask = DeltaDSRMask;
end;

{ Returns True if DeltaRI is high }
function TApxBaseDispatcher.CheckDeltaRI : Boolean;
begin
  Result := GetModemStatusPrim(ClearDeltaRI) and DeltaRIMask = DeltaRIMask;
end;

{ Returns True if DeltaDCD is high }
function TApxBaseDispatcher.CheckDeltaDCD : Boolean;
begin
  Result := GetModemStatusPrim(ClearDeltaDCD) and DeltaDCDMask = DeltaDCDMask;
end;

{ Return current line errors }
function TApxBaseDispatcher.GetLineError : Integer;
const
  AllErrorMask = ce_RxOver + ce_Overrun + ce_RxParity + ce_Frame;
var
  GotError : Boolean;
begin
  DataLock.Enter;
  try

    GotError := True;
    if FlagIsSet(LastError, ce_RxOver) then
      Result := leBuffer
    else if FlagIsSet(LastError, ce_Overrun) then
      Result := leOverrun
    else if FlagIsSet(LastError, ce_RxParity) then
      Result := leParity
    else if FlagIsSet(LastError, ce_Frame) then
      Result := leFraming
    else if FlagIsSet(LastError, ce_Break) then
      Result := leBreak
    else begin
      GotError := False;
      Result := leNoError;
    end;

    { Clear all error flags }
    if GotError then
      LastError := LastError and not AllErrorMask;

  finally
    DataLock.Leave;
  end;
end;

function TApxBaseDispatcher.CheckLineBreak : Boolean;
begin
  DataLock.Enter;
  try
    Result := FlagIsSet(LastError, ce_Break);
    LastError := LastError and not ce_Break;
  finally
    DataLock.Leave;
  end;
end;

{ Return True if at least one character is ready at the device driver }
function TApxBaseDispatcher.CharReady : Boolean;
var
  NewTail : Cardinal;
begin
  DispLock.Enter;
  try

    if InAvailMessage then begin
      NewTail := DBufTail + GetCount;
      if NewTail >= DBufSize then
        Dec(NewTail, DBufSize);
      Result := (DBufHead <> NewTail)
        or (DBufFull and (GetCount < DBufSize));
    end else
      Result := (DBufHead <> DBufTail) or DBufFull;

  finally
    DispLock.Leave;
  end;
end;

{ Return the Count'th character but don't remove it from the buffer }
function TApxBaseDispatcher.PeekCharPrim(var C : AnsiChar; Count : Cardinal) : Integer;
var
  NewTail : Cardinal;
  InCount : Cardinal;
begin
  Result := ecOK;
  DispLock.Enter;
  try

    if DBufHead > DBufTail then
      InCount := DBufHead-DBufTail
    else if DBufHead <> DBufTail then
      InCount := ((DBufHead+DBufSize)-DBufTail)
    else if DBufFull then
      InCount := DBufSize
    else
      InCount := 0;

    if InCount >= Count then begin
      { Calculate index of requested character }
      NewTail := DBufTail + (Count - 1);
      if NewTail >= DBufSize then
        NewTail := (NewTail - DBufSize);
      C := DBuffer^[NewTail];
    end else
      Result := ecBufferIsEmpty;

  finally
    DispLock.Leave;
  end;
end;

{ Return the Count'th character but don't remove it from the buffer }
{ Account for GetCount }
function TApxBaseDispatcher.PeekChar(var C : AnsiChar; Count : Cardinal) : Integer;
begin
  DispLock.Enter;
  try

    if InAvailMessage then
      Inc(Count, GetCount);

    Result := PeekCharPrim(C, Count);

  finally
    DispLock.Leave;
  end;
end;

{ Return next char and remove it from buffer }
function TApxBaseDispatcher.GetChar(var C : AnsiChar) : Integer;
begin
  DispLock.Enter;
  try

    { If within an APX_TRIGGERAVAIL message then do not physically      }
    { extract the character. It will be removed by the dispatcher after }
    { all trigger handlers have seen it. If not within an               }
    { APX_TRIGGERAVAIL message then physically extract the character    }

    if InAvailMessage then begin
      Inc(GetCount);
      Result := PeekCharPrim(C, GetCount);
      if Result < ecOK then begin
        Dec(GetCount);
        Exit;
      end;
    end else begin
      Result := PeekCharPrim(C, 1);
      if Result >= ecOK then begin
        { Increment the tail index }
        Inc(DBufTail);
        if DBufTail = DBufSize then
          DBufTail := 0;
        DBufFull := False;
      end;
    end;

  finally
    DispLock.Leave;
  end;
end;

{ Parameters have changed, need updating }
procedure TApxBaseDispatcher.ParamsChanged(Sender : TObject);
begin
  ConfigCom;
end;

{ Return Block from port, return new tail value }
function TApxBaseDispatcher.PeekBlockPrim(Block : PAnsiChar;
  Offset : Cardinal; Len : Cardinal; var NewTail : Cardinal) : Integer;
var
  Count : Cardinal;
  EndCount : Cardinal;
  BeginCount : Cardinal;
begin
  DispLock.Enter;
  try

    { Get count }
    Count := InBuffCount(DBufHead, DBufTail, DBufFull);

    { Set new tail value }
    NewTail := DBufTail + Offset;
    if NewTail >= DBufSize then
      Dec(NewTail, DBufSize);

    if Count >= Len then begin
      { Set begin/end buffer counts }
      if NewTail+Len < DBufSize then begin
        EndCount := Len;
        BeginCount := 0;
      end else begin
        EndCount := (DBufSize-NewTail);
        BeginCount := Len-EndCount;
      end;

      if EndCount <> 0 then begin
        { Move data from end of dispatch buffer }
        Move(DBuffer^[NewTail], Pointer(Block)^, EndCount);
        Inc(NewTail, EndCount);
      end;

      if BeginCount <> 0 then begin
        { Move data from beginning of dispatch buffer }
        Move(DBuffer^[0],
             PByteBuffer(Block)^[EndCount+1],
             BeginCount);
        NewTail := BeginCount;
      end;

      { Wrap newtail }
      if NewTail = DBufSize then
        NewTail := 0;

      Result := Len;
    end else
      Result := ecBufferIsEmpty;

  finally
    DispLock.Leave;
  end;
end;

{ Return Block from ComPort but don't set new tail value }
function TApxBaseDispatcher.PeekBlock(Block : PAnsiChar; Len : Cardinal) : Integer;
var
  Tail : Cardinal;
  Offset : Cardinal;
begin
  DispLock.Enter;
  try

    { Get block }
    if InAvailMessage then
      Offset := GetCount
    else
      Offset := 0;
    Result := PeekBlockPrim(Block, Offset, Len, Tail);

  finally
    DispLock.Leave;
  end;
end;

{ Get Block from ComPort and set new tail }
function TApxBaseDispatcher.GetBlock(Block : PAnsiChar; Len : Cardinal) : Integer;
var
  Tail : Cardinal;
begin
  DispLock.Enter;
  try

    { If within an APX_TRIGGERAVAIL message then do not physically }
    { extract the data. It will be removed by the dispatcher after }
    { all trigger handlers have seen it. If not within an          }
    { APX_TRIGGERAVAIL message, then physically extract the data   }

    if InAvailMessage then begin
      Result := PeekBlockPrim(Block, GetCount, Len, Tail);
      if Result > 0 then
        Inc(GetCount, Result);
    end else begin
      Result := PeekBlockPrim(Block, 0, Len, Tail);
      if Result > 0 then begin
        DBufTail := Tail;
        DBufFull := False;
      end;
    end;

  finally
    DispLock.Leave;
  end;
end;

{ Route through PutBlock to transmit a single character }
function TApxBaseDispatcher.PutChar(C : AnsiChar) : Integer;
begin
  Result := PutBlock(C, 1);
end;

{ Send as a block }
function TApxBaseDispatcher.PutString(S : string) : Integer;
begin
  Result := PutBlock(S[1], Length(S));
end;

{ Place Block in Output buffer, notify Output thread }
function TApxBaseDispatcher.PutBlock(const Block; Len : Cardinal) : Integer;
var
  TempLen : Cardinal;
begin
  { Exit immediately if nothing to do }
  Result := ecOK;
  if Len = 0 then Exit;

  OutputLock.Enter;
  try

    { Is there enough free space in the outbuffer? }
    if OutBuffFree < Len then begin
      Result := ecOutputBufferTooSmall;
      Exit;
    end;

    { Add data to output buffer }
    if OBufTail >= OBufHead then begin

      { Check to see if we'll need to wrap }
      if (OBufTail + Len) < OBufSize then begin

        { No wrap required, move data }
        Move(Block, OBuffer^[OBufTail], Len);
        Inc(OBufTail, Len);
      end else begin

        { Wrap required, move data in two chunks }
        TempLen := OBufSize - OBufTail;
        Move(Block, OBuffer^[OBufTail], TempLen);

        { Calc size of second chunk, move, set new tail }
        OBufTail := Len-TempLen;
        Move(TOBuffer(Block)[TempLen], OBuffer^[0], OBufTail);
      end;

    end else begin
      { Already wrapped; move data }
      Move(Block, OBuffer^[OBufTail], Len);
      Inc(OBufTail, Len);
    end;

    { Test for full buffer }
    if OBufTail = OBufHead then
      OBufFull := True;

    { Set output event }
    OutputEvent.SetEvent;

    { Flag output trigger }
    OutSentPending := True;

  finally
    OutputLock.Leave;
  end;
end;

{ Return number of bytes currently in input buffer }
function TApxBaseDispatcher.InBuffUsed : Cardinal;
begin
  DispLock.Enter;
  try

    if DBufHead = DBufTail then
      if DBufFull then
        Result := DBufSize
      else
        Result := 0
    else if DBufHead > DBufTail then
      Result := DBufHead-DBufTail
    else
      Result := (DBufHead+DBufSize)-DBufTail;

    if InAvailMessage then
      { In APX_TRIGGERAVAIL message so reduce by retrieved chars }
      Dec(Result, GetCount);

  finally
    DispLock.Leave;
  end;
end;

{ Return number of bytes free in input buffer }
function TApxBaseDispatcher.InBuffFree : Cardinal;
begin
  DispLock.Enter;
  try

    if DBufHead = DBufTail then
      if DBufFull then
        Result := 0
      else
        Result := DBufSize
    else if DBufHead > DBufTail then
      Result := (DBufTail+DBufSize)-DBufHead
    else
      Result := DBufTail-DBufHead;

    if InAvailMessage then
      { In APX_TRIGGERAVAIL message so reduce by retrieved chars }
      Inc(Result, GetCount);

  finally
    DispLock.Leave;
  end;
end;

{ Return number of bytes currently in output buffer }
function TApxBaseDispatcher.OutBuffUsed : Cardinal;
begin
  OutputLock.Enter;
  try
    if OBufTail >= OBufHead then begin
      if OBufFull then
        Result := OBufSize
      else
        Result := OBufTail - OBufHead;
    end else begin
      Result := OBufTail + (OBufSize - OBufHead);
    end;
  finally
    OutputLock.Leave;
  end;
end;

{ Return number of bytes free in output buffer }
function TApxBaseDispatcher.OutBuffFree : Cardinal;
begin
  OutputLock.Enter;
  try
    Result := OBufSize - OutBuffUsed;
  finally
    OutputLock.Leave;
  end;
end;

{ Flush the appropriate buffer }
function TApxBaseDispatcher.FlushBuffer(Buffer : TAxCommBuffer) : Integer;
begin
  Result := 0;

  if Buffer = cbInput then begin
    DispLock.Enter;
    try
      { Flush the dispatcher's buffer }
      if InAvailMessage then
        MaxGetCount := InBuffCount(DBufHead, DBufTail, DBufFull)
      else begin
        DBufHead := 0;
        DBufTail := 0;
        GetCount := 0;
      end;
      DBufFull := False;

      { Reset data triggers }
      ResetDataTriggers;

    finally
      DispLock.Leave;
    end;
  end else begin
    OutputLock.Enter;
    try
      OBufHead := 0;
      OBufTail := 0;
      OBufFull := False;
    finally
      OutputLock.Leave;
    end;
  end;
end;

{ Send trigger messages, return False to stop checking triggers }
function TApxBaseDispatcher.SendNotify(Msg, Trigger, Data : Cardinal) : Boolean;
var
  I : Integer;
begin
  Result := True;

  if not HandlerServiceNeeded then Exit;

  { Don't let dispatcher change anything while sending messages }
  DataLock.Enter;
  try
    FEventBusy := True;
  finally
    DataLock.Leave;
  end;

  try
    MaxGetCount := 0;

    { Flag APX_TRIGGERAVAIL messages }
    InAvailMessage := (Msg = APX_TRIGGERAVAIL) or (Msg = APX_TRIGGERDATA);

    { Clear trigger handle modification flags }
    for i := 0 to pred(EventTriggerHandlers.Count) do
      with PEventTriggerHandler(EventTriggerHandlers[i])^ do begin
        GetCount := 0;

        if not thDeleted then
          if thSync then
            ReadThread.SyncNotify(Msg, Trigger, Data, thNotify)
          else
            thNotify(Msg, Trigger, Data);

        if ClosePending then begin
          { Port was closed by event handler, bail out }
          Result := False;
          Exit;
        end;

        { Note deepest look at input buffer }
        if GetCount > MaxGetCount then
          MaxGetCount := GetCount;
      end;

    for i := 0 to pred(ProcTriggerHandlers.Count) do
      with PProcTriggerHandler(ProcTriggerHandlers[i])^ do begin
        GetCount := 0;

        if not thDeleted and (@thNotify <> nil) then
          thNotify(Msg, Trigger, Data);

        if ClosePending then begin
          { Port was closed by event handler, bail out }
          Result := False;
          Exit;
        end;

        { Note deepest look at input buffer }
        if GetCount > MaxGetCount then
          MaxGetCount := GetCount;
      end;

    if (QtTriggerHandlers.Count > 1) or PortHandlerInstalled then
      for i := 0 to pred(QtTriggerHandlers.Count) do
        with PQtTriggerHandler(QtTriggerHandlers[i])^ do begin
          GetCount := 0;

          if not thDeleted then begin
            {$IFDEF MSWindows}
            SendMessage(thHandle, Msg, Trigger, Data);
            {$ENDIF}
            {$IFDEF Linux}
            Assert(MsgEvent <> nil);
            AxPostMessage(thHandle, Msg, MsgEvent, Trigger, Data);
            if Assigned(WakeMainThread) then WakeMainThread(nil);
            MsgEvent.WaitFor($FFFFFFFF);
            {$ENDIF}
          end;

          if ClosePending then begin
            { Port was closed by event handler, bail out }
            Result := False;
            Exit;
          end;

          { Note deepest look at input buffer }
          if GetCount > MaxGetCount then
            MaxGetCount := GetCount;
        end;

    { If in APX_TRIGGERAVAIL message then remove the data now. }
    if InAvailMessage then begin
      DispLock.Enter;
      try
        InAvailMessage := False;
        Inc(DBufTail, MaxGetCount);
        if DBufTail >= DBufSize then
          Dec(DBufTail, DBufSize);
        if MaxGetCount <> 0 then
          DBufFull := False;

        { Force CheckTriggers to exit if another avail msg is pending }
        { Note: for avail msgs, trigger is really the byte count      }
        if (Msg = APX_TRIGGERAVAIL) and (MaxGetCount <> Trigger) then
          Result := False;
      finally
        DispLock.Leave;
      end;
    end;
  finally

    DataLock.Enter;
    try
      FEventBusy := False;

    if DeletePending then begin

      for i := pred(QtTriggerHandlers.Count) downto 0 do
        if PQtTriggerHandler(QtTriggerHandlers[i])^.thDeleted then begin
          Dispose(PQtTriggerHandler(QtTriggerHandlers[i]));
          QtTriggerHandlers.Delete(i);
        end;

      for i := pred(ProcTriggerHandlers.Count) downto 0 do
        if PProcTriggerHandler(ProcTriggerHandlers[i])^.thDeleted then begin
          Dispose(PProcTriggerHandler(ProcTriggerHandlers[i]));
          ProcTriggerHandlers.Delete(i);
        end;

      for i := pred(EventTriggerHandlers.Count) downto 0 do
        if PEventTriggerHandler(EventTriggerHandlers[i])^.thDeleted then begin
          Dispose(PEventTriggerHandler(EventTriggerHandlers[i]));
          EventTriggerHandlers.Delete(i);
        end;

      DeletePending := False;
      UpdateHandlerFlags(fuKeepPort);
    end;
    finally
      DataLock.Leave;
    end;
  end;
  GetCount := 0;
end;

{ Checks for string P on consecutive calls, returns True when found }
function MatchString(var Indexes : TCheckIndex; const C : AnsiChar; Len : Cardinal;
  P : PAnsiChar; IgnoreCase : Boolean) : Boolean;
var
  I        : Cardinal;
  Check    : Boolean;
  GotFirst : Boolean;
  TempChar : array[0..1] of AnsiChar;
begin
  Result := False;

  TempChar[0] := C;
  TempChar[1] := #0;

  if IgnoreCase then
    AnsiStrUpper(TempChar);

  GotFirst := False;
  Check := True;
  for I := 0 to Len-1 do begin
    { Check another index? }
    if Check then begin
      { Compare this index... }
      if TempChar[0] = P[Indexes[I]] then
        { Got match, was it complete? }
        if Indexes[I] = Len-1 then begin
          Indexes[I] := 0;
          Result := True;

          { Clear all inprogress matches }
          FillChar(Indexes, SizeOf(Indexes), 0);
        end else
          Inc(Indexes[I])
      else
        { No match, reset index }
        if TempChar[0] = P[0] then begin
          GotFirst := True;
          Indexes[I] := 1
        end else
          Indexes[I] := 0;
    end;

    { See if last match was on first char }
    if Indexes[I] = 1 then
      GotFirst := True;

    { See if we should check the next index }
    if I <> Len-1 then
      if GotFirst then
        { Got a previous restart, don't allow more restarts }
        Check := Indexes[I+1] <> 0
      else
        { Not a restart, check next index if in progress or on first char }
        Check := (Indexes[I+1] <> 0) or (TempChar[0] = P[0])
    else
      Check := False;
  end;
end;

{ Check status triggers for H, send notification messages as required }
{ Return True if more checks remain }
function TApxBaseDispatcher.CheckStatusTriggers : Boolean;
var
  J : Integer;
  Hit : Cardinal;
  StatusLen : Cardinal;
  Res : Byte;
  BufCnt : Cardinal;
begin
  { Check status triggers }
  for J := 0 to pred(StatusTriggers.Count) do begin
    with PStatusTrigger(StatusTriggers[J])^ do begin
      if tSActive and not StatusHit then begin
        Hit := stNotActive;
        StatusLen := 0;
        case tSType of
          stLine :
            if LastError and tValue <> 0 then begin
              Hit := stLine;
              tValue := LastError;
            end;
          stModem :
            begin
              { Check for changed bits }
              Res := Lo(tValue) xor ModemStatus;

              { Skip bits not in our mask }
              Res := Res and Hi(tValue);

              { If anything is still set, it's a hit }
              if Res <> 0 then begin
                Hit := stModem;
              end;
            end;
          stOutBuffFree :
            begin
              BufCnt := OutBuffFree;
              if BufCnt >= tValue then begin
                StatusLen := BufCnt;
                Hit := stOutBuffFree;
              end;
            end;
          stOutBuffUsed :
            begin
              BufCnt := OutBuffUsed;
              if BufCnt <= tValue then begin
                StatusLen := BufCnt;
                Hit := stOutBuffUsed;
              end;
            end;
          stOutSent :
            if OutSentPending then begin
              OutSentPending := False;
              StatusLen := 0;
              Hit := stOutSent;
            end;
        end;
        if Hit <> stNotActive then begin
          { Clear the trigger and send the notification message }
          tSActive := False;

          { Prevent status trigger re-entrancy issues }
          GlobalStatHit := True;
          StatusHit := True;
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclStatusTrigger, tHandle, 0);
          Result := SendNotify(APX_TRIGGERSTATUS, tHandle, StatusLen);
          Exit;
        end;
      end;
    end;
    if J >= StatusTriggers.Count then break;
  end;
  { No more checks required }
  Result := False;
end;

{ Check all receive triggers for H, send notification messages as required }
{ Return True if more checks remain }
function TApxBaseDispatcher.CheckReceiveTriggers : Boolean;
type
  LH = record L,H : Byte; end;
var
  I : Cardinal;
  J : Integer;
  BufCnt : Cardinal;
  MatchSize : Cardinal;
  CC : Cardinal;
  AnyMatch : Boolean;
  C : AnsiChar;

  { Return the number of characters available between CurTail }
  { and DBufTail that haven't already been extracted. CurTail }
  { is first adjusted downward by Adjust, the size of the     }
  { current match string }
  function CharCount(CurTail, Adjust : Cardinal) : Cardinal;
  begin
    if Adjust <= CurTail then
      Dec(CurTail, Adjust)
    else
      CurTail := (CurTail + DBufSize) - Adjust;
    Result := InBuffCount(CurTail, DBufTail, DBufFull) + 1;
    if InAvailMessage then
      Dec(Result, GetCount);
  end;

begin
  { Assume triggers need to be re-checked }
  Result := True;

  I := LastTailData;
  { Check data triggers }
  if LastTailData <> DBufHead then begin
    { Prepare }

    { Loop through new data in dispatch buffer }
    while I <> DBufHead do begin
      C := DBuffer^[I];

      { Check each trigger for a match on this character }
      AnyMatch := False;
      MatchSize := 0;
      for J := 0 to pred(DataTriggers.Count) do
        with PDataTrigger(DataTriggers[J])^ do
          if tLen <> 0 then begin
            tMatched := tMatched or
                        MatchString(tChkIndex, C, tLen, tData, tIgnoreCase);
            if tMatched and (tLen > MatchSize) then
              MatchSize := tLen;
            if not AnyMatch then
              AnyMatch := tMatched;
          end;

      { Send len message if we have any matches }
      if AnyMatch then begin
        { Send len message up to first matching char }
        if (NotifyTail <> I) and ((CharCount(I, 0) - MatchSize) > 0) then begin

          { Generate len message for preceding data }
          CC := CharCount(I, MatchSize);
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclAvailTrigger, CC, 0);
          Result := SendNotify(APX_TRIGGERAVAIL, CC, 0);
          LastTailData := I;
          NotifyTail := I;
        end;

        { Process the matches }
        for J := pred(DataTriggers.Count) downto 0 do begin
          with PDataTrigger(DataTriggers[J])^ do
            if tMatched then begin
              { No preceding data or msg pending, send data msg }
              FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclDataTrigger, tHandle, 0);
              tMatched := False;
              Result := SendNotify(APX_TRIGGERDATA, tHandle, tLen);
            end;
          if J >= DataTriggers.Count then Break;
        end;

        { Exit after all data triggers that matched on this char }
        if I = DBufSize-1 then
          LastTailData := 0
        else
          LastTailData := I+1;
        Exit;
      end;

      { Next index for buffer }
      if I = DBufSize-1 then
        I := 0
      else
        inc(I);
    end;

    { Update last tail for data triggers }
    LastTailData := I;
  end;

  { Check for length trigger }

  BufCnt := InBuffUsed;

  if (NotifyTail <> I) and (BufCnt > 0) then begin
    FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclAvailTrigger, BufCnt, 0);
    Result := SendNotify(APX_TRIGGERAVAIL, BufCnt, 0);
    NotifyTail := I;
    Exit;
  end;

  { No more checks required }
  Result := False;
end;

{ Check timer triggers for H, send notification messages as required }
{ Return True if more checks remain }
function TApxBaseDispatcher.CheckTimerTriggers : Boolean;
var
  J : Integer;
begin
  { Check for timer triggers }
  for J := 0 to pred(TimerTriggers.Count) do begin
    with PTimerTrigger(TimerTriggers[J])^ do
      if tActive and TimerExpired(tET) then begin
        tActive := False;

        FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclTimerTrigger, tHandle, 0);

        Result := SendNotify(APX_TRIGGERTIMER, tHandle, 0);
        Exit;
      end;
    if J >= TimerTriggers.Count then break;
  end;
  { No more checks required }
  Result := False;
end;

{ Move data from communications driver to dispatch buffer }
{ Return True if data available, false otherwise }
function TApxBaseDispatcher.ExtractData : Boolean;
var
  BytesToRead : Cardinal;
  FreeSpace : Cardinal;
  BeginFree : Cardinal;
  EndFree : Cardinal;
  Len : Integer;
begin
  DispLock.Enter;
  try

    { Nothing to do if dispatch buffer is already full }
    if DBufFull then begin
      Result := True;
      Exit;
    end;

    if AvailableBytes > 0 then begin
      Result := True;

      if DBufHead = DBufTail then begin
        { Buffer is completely empty (DBufFull was checked above) }
        FreeSpace := DBufSize;
        EndFree := DBufSize-DBufHead;
      end else if DBufHead > DBufTail then begin
        { Buffer not wrapped }
        FreeSpace := (DBufTail+DBufSize)-DBufHead;
        EndFree := DBufSize-DBufHead;
      end else begin
        { Buffer is wrapped }
        FreeSpace := DBufTail-DBufHead;
        EndFree := DBufTail-DBufHead;
      end;

      { Figure out how much data to read }
      if DWORD(AvailableBytes) > FreeSpace then begin
        BytesToRead := FreeSpace;
      end else begin
        BytesToRead := AvailableBytes;
      end;

      { Figure where data fits (end and/or beginning of buffer) }
      if BytesToRead > EndFree then
        BeginFree := BytesToRead-EndFree
      else
        BeginFree := 0;

      { Move data to end of dispatch buffer }
      if EndFree <> 0 then begin
        Len := ReadCom(PAnsiChar(@DBuffer^[DBufHead]), EndFree);

        { Restore data count on errors }
        if Len < 0 then begin
          Len := 0;
        end;

        if Len = 0 then
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclReadCom, 0, 0)
        else
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclReadCom,
            DWORD(@DBuffer^[DBufHead]), Len);

        { Increment buffer head }
        Inc(DBufHead, Len);

        if Cardinal(Len) < EndFree then
          BeginFree := 0;

      end else
        Len := 0;

      { Handle buffer wrap }
      if DBufHead = DBufSize then
        DBufHead := 0;

      { Check for a full dispatch buffer }
      if Len <> 0 then
        DBufFull := DBufHead = DBufTail;

      { Move data to beginning of dispatch buffer }
      if BeginFree <> 0 then begin
        Len := ReadCom(PAnsiChar(@DBuffer^[DBufHead]), BeginFree);

        { Restore data count on errors }
        if Len < 0 then begin
          Len := Abs(Len);
        end;

        if Len = 0 then
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclReadCom, 0, 0)
        else
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclReadCom, DWORD(@DBuffer^[DBufHead]), Len);

        { Increment buffer head }
        Inc(DBufHead, Len);

        { Check for a full dispatch buffer }
        DBufFull := DBufHead = DBufTail;

      end;
    end else
      Result := False;

  finally
    DispLock.Leave;
  end;
end;

{ Check all triggers for H, send notification messages as required }
{ Return True if more checks remain }
function TApxBaseDispatcher.CheckTriggers : Boolean;
begin
  Result := True;

  { Check timers, exit true if any hit }
  if CheckTimerTriggers then
    Exit;

  { Check status triggers, exit true if any hit }
  if CheckStatusTriggers then
    Exit;

  { Check receive data triggers, exit true if any hit }
  if CheckReceiveTriggers then
    Exit;

  { No trigger hits, exit false }
  Result := False;
end;

{ Trigger functions }

procedure TApxBaseDispatcher.RegisterQtTriggerHandler(Handle : QObjectH);
var
  TH : PQtTriggerHandler;
begin
  DataLock.Enter;
  try

    { Allocate memory for TriggerHandler node }
    New(TH);

    { Fill in data }
    with TH^ do begin
      thHandle := Handle;
      thDeleted := False;
    end;

    QtTriggerHandlers.Add(TH);
    HandlerServiceNeeded := True;

    FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclAllocWndHandler, DWORD(Handle), 0);

  finally
    DataLock.Leave;
  end;
end;

procedure TApxBaseDispatcher.RegisterProcTriggerHandler(NP : TApxNotifyProc);
var
  TH : PProcTriggerHandler;
begin
  DataLock.Enter;
  try

    { Allocate memory for TriggerHandler node }
    New(TH);

    { Fill in data }
    with TH^ do begin
      thnotify := NP;
      thDeleted := False;
    end;

    ProcTriggerHandlers.Add(TH);
    HandlerServiceNeeded := True;

    FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclAllocProcHandler, 0, 0);

  finally
    DataLock.Leave;
  end;
end;

procedure TApxBaseDispatcher.RegisterSyncEventTriggerHandler(NP : TApxNotifyEvent);
var
  TH : PEventTriggerHandler;
begin
  DataLock.Enter;
  try

    { Allocate memory for TriggerHandler node }
    New(TH);

    { Fill in data }
    with TH^ do begin
      thNotify := NP;
      thSync := True;
      thDeleted := False;
    end;

    EventTriggerHandlers.Add(TH);
    HandlerServiceNeeded := True;

    FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclAllocEventHandler, 1, 0);

  finally
    DataLock.Leave;
  end;
end;

procedure TApxBaseDispatcher.RegisterEventTriggerHandler(NP : TApxNotifyEvent);
var
  TH : PEventTriggerHandler;
begin
  DataLock.Enter;
  try

    { Allocate memory for TriggerHandler node }
    New(TH);

    { Fill in data }
    with TH^ do begin
      thNotify := NP;
      thSync := False;
      thDeleted := False;
    end;

    EventTriggerHandlers.Add(TH);
    HandlerServiceNeeded := True;

    FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclAllocEventHandler, 0, 0);

  finally
    DataLock.Leave;
  end;
end;

procedure TApxBaseDispatcher.DeregisterQtTriggerHandler(Handle : QObjectH);
var
  i : Integer;
begin
  DataLock.Enter;
  try

    for i := 0 to pred(QtTriggerHandlers.Count) do
      with PQtTriggerHandler(QtTriggerHandlers[i])^ do
        if thHandle = Handle then begin
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclDispWndHandler, DWORD(Handle), 0);
          if FEventBusy then begin
            thDeleted := True;
            DeletePending := True;
          end else begin
            Dispose(PQtTriggerHandler(QtTriggerHandlers[i]));
            QtTriggerHandlers.Delete(i);
          end;
          exit;
        end;

    UpdateHandlerFlags(fuKeepPort);

  finally
    DataLock.Leave;
  end;
end;

procedure TApxBaseDispatcher.DeregisterProcTriggerHandler(NP : TApxNotifyProc);
var
  i : Integer;
begin
  DataLock.Enter;
  try

    for i := 0 to pred(ProcTriggerHandlers.Count) do
      with PProcTriggerHandler(ProcTriggerHandlers[i])^ do
        if @thNotify = @NP then begin
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclDispProcHandler, 0, 0);
          if FEventBusy then begin
            thDeleted := True;
            DeletePending := True;
          end else begin
            Dispose(PProcTriggerHandler(ProcTriggerHandlers[i]));
            ProcTriggerHandlers.Delete(i);
          end;
          exit;
        end;

    UpdateHandlerFlags(fuKeepPort);

  finally
    DataLock.Leave;
  end;
end;

procedure TApxBaseDispatcher.DeregisterEventTriggerHandler(NP : TApxNotifyEvent);
var
  i : Integer;
begin
  DataLock.Enter;
  try

    for i := 0 to pred(EventTriggerHandlers.Count) do
      with PEventTriggerHandler(EventTriggerHandlers[i])^ do
        if @thNotify = @NP then begin
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclDispEventHandler, 0, 0);
          if FEventBusy then begin
            thDeleted := True;
            DeletePending := True;
          end else begin
            Dispose(PEventTriggerHandler(EventTriggerHandlers[i]));
            EventTriggerHandlers.Delete(i);
          end;
          exit;
        end;

    UpdateHandlerFlags(fuKeepPort);

  finally
    DataLock.Leave;
  end;
end;

{ Find, allocate and return the first free trigger handle }
function TApxBaseDispatcher.GetTriggerHandle : Cardinal;
var
  I : Integer;
  Good : Boolean;
begin
  { Allocate a trigger handle. If we can, within the size of the handle's datatype, we }
  { just increment TriggerCounter to get a new handle. If not, we need to check the    }
  { existing handles to find a unique one. }
  if TriggerCounter < MaxTriggerHandle then begin
    Result := TriggerCounter shl 3; { low three bits reserved for trigger specific information }
    inc(TriggerCounter);
  end else begin
    Result := FirstTriggerCounter shl 3; { lowest possible handle value }
    repeat
      Good := True;  { Assume success }
      for i := 0 to pred(TimerTriggers.Count) do
        with PTimerTrigger(TimerTriggers[i])^ do
          if tHandle = Result then begin
            Good := False;
            Break;
          end;
      if Good then for i := 0 to pred(StatusTriggers.Count) do
        with PStatusTrigger(StatusTriggers[i])^ do
          if tHandle = Result then begin
            Good := False;
            Break;
          end;
      if Good then for i := 0 to pred(DataTriggers.Count) do
        with PDataTrigger(DataTriggers[i])^ do
          if tHandle = Result then begin
            Good := False;
            Break;
          end;
      if not Good then
        inc(Result,(1 shl 3));
    until Good;
  end;
end;

{ Find the trigger index }
function TApxBaseDispatcher.FindTriggerFromHandle(TriggerHandle : Cardinal;
  Delete : Boolean; var T : TTriggerType; var Trigger : Pointer) : Integer;
var
  i : Integer;
  b : Byte;
begin
  T := ttNone;
  Result := ecOk;
  if (TriggerHandle > 1) then begin
    for i := 0 to pred(TimerTriggers.Count) do begin
      Trigger := TimerTriggers[i];
      with PTimerTrigger(Trigger)^ do
        if tHandle = TriggerHandle then begin
          T := ttTimer;
          if Delete then begin
            TimerTriggers.Delete(i);
            Dispose(PTimerTrigger(Trigger));
            FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclDispTimerTrigger, TriggerHandle, 0);
            Trigger := nil;
          end;
          exit;
        end;
    end;
    for i := 0 to pred(StatusTriggers.Count) do begin
      Trigger := StatusTriggers[i];
      with PStatusTrigger(Trigger)^ do
        if tHandle = TriggerHandle then begin
          T := ttStatus;
          if Delete then begin
            StatusTriggers.Delete(i);
            Dispose(PStatusTrigger(Trigger));
            b := lo(TriggerHandle and (StatusTypeMask));
            FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclDispStatusTrigger, TriggerHandle, b);
            Trigger := nil;
          end;
          exit;
        end;
    end;
    for i := 0 to pred(DataTriggers.Count) do begin
      Trigger := DataTriggers[i];
      with PDataTrigger(Trigger)^ do
        if tHandle = TriggerHandle then begin
          T := ttData;
          if Delete then begin
            DataTriggers.Delete(i);
            Dispose(PDataTrigger(Trigger));
            FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclDispDataTrigger, TriggerHandle, 0);
            Trigger := nil;
          end;
          exit;
        end;
    end;
  end else begin
    T := ttNone;
    Trigger := nil;
  end;
  if T = ttNone then
    Result := ecBadTriggerHandle;
end;

{ Add a timer trigger }
function TApxBaseDispatcher.AddTimerTrigger : Integer;
var
  NewTimerTrigger : PTimerTrigger;
begin
  DataLock.Enter;
  try

  NewTimerTrigger := AllocMem(sizeof(TTimerTrigger));
  with NewTimerTrigger^ do begin
    tHandle := GetTriggerHandle;
    tmSecs  := 0;
    tActive := False;
    tValid  := True;
    Result  := tHandle;
  end;
  TimerTriggers.Add(NewTimerTrigger);
  FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclAllocTimerTrigger, Result, 0);
  finally
    DataLock.Leave;
  end;
end;

{ Add a data trigger, data is any ASCIIZ string so no embedded nulls }
function TApxBaseDispatcher.AddDataTriggerLen(Data : PAnsiChar;
  IgnoreCase : Boolean; Len : Cardinal) : Integer;
var
  NewDataTrigger : PDataTrigger;
  LogData : PByteArray;
begin
  DataLock.Enter;
  try
    if Len <= MaxTrigData then begin
      NewDataTrigger := AllocMem(sizeof(TDataTrigger));
      with NewDataTrigger^ do begin
        tHandle := GetTriggerHandle;
        tLen := Len;
        FillChar(tChkIndex, SizeOf(TCheckIndex), 0);
        tMatched := False;
        tIgnoreCase := IgnoreCase;
        if IgnoreCase and (Len <> 0) then
          AnsiStrUpper(Data);
        Move(Data^, tData, Len);
        Result := tHandle;
      end;
      DataTriggers.Add(NewDataTrigger);

      GetMem(LogData, Len+4);
      try
        { Handle goes in first four bytes }
        Move(Result, LogData[0], SizeOf(Result));
        { Data goes in the remainder }
        Move(Data[0], LogData[4], Len);
        FOwner.DebugLog.AddDebugEntry(TApxCustomComPort,
          cclAllocDataTrigger, DWORD(LogData), Len+4);
      finally
        FreeMem(LogData);
      end;
    end else
      Result := ecTriggerTooLong;
  finally
    DataLock.Leave;
  end;
end;

{ Add a data trigger, data is any ASCIIZ string so no embedded nulls }
function TApxBaseDispatcher.AddDataTrigger(Data : PAnsiChar;
  IgnoreCase : Boolean) : Integer;
begin
  Result := AddDataTriggerLen(Data, IgnoreCase, StrLen(Data));
end;

{ Add a status trigger of type SType }
function TApxBaseDispatcher.AddStatusTrigger(SType : Cardinal) : Integer;
var
  NewStatusTrigger : PStatusTrigger;
begin
  if (SType > stOutSent) then begin
    Result := ecBadArgument;
    Exit;
  end;

  DataLock.Enter;
  try

    NewStatusTrigger := AllocMem(sizeof(TStatusTrigger));
    with NewStatusTrigger^ do begin
      tHandle := GetTriggerHandle or SType;
      tSType := SType;
      tSActive := False;
      Result := tHandle;
    end;
    StatusTriggers.Add(NewStatusTrigger);
    FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclAllocStatusTrigger, Result, SType);

  finally
    DataLock.Leave;
  end;
end;

{ Remove the trigger for Index }
function TApxBaseDispatcher.RemoveTrigger(TriggerHandle : Cardinal) : Integer;
var
  Trigger : Pointer;
  T : TTriggerType;
begin
  DataLock.Enter;
  try

    if TriggerHandle = 1 then
      { Length trigger }
      begin
        Result := ecOk;
      end
    else
      { Other trigger }
      Result := FindTriggerFromHandle(TriggerHandle, True, T, Trigger);

  finally
    DataLock.Leave;
  end;
end;

function TApxBaseDispatcher.SetTimerTrigger(TriggerHandle : Cardinal;
  mSecs : DWORD; Activate : Boolean) : Integer;
var
  Trigger : PTimerTrigger;
  T : TTriggerType;
begin
  DataLock.Enter;
  try

    FindTriggerFromHandle(TriggerHandle, False, T, Pointer(Trigger));
    if (Trigger <> nil) and (T = ttTimer) then
      with Trigger^ do begin
        if Activate then begin
          if mSecs <> 0 then
            tmSecs := mSecs;
          NewTimer(tET, tmSecs);
        end;

        if Activate then
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort,
            cclTimerTriggerActive, TriggerHandle, mSecs)
        else
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort,
            cclTimerTriggerDeact, TriggerHandle, 0);

        tActive := Activate;
        Result := ecOk;
      end
    else
      Result := ecBadTriggerHandle;

  finally
    DataLock.Leave;
  end;
end;

function TApxBaseDispatcher.ExtendTimer(TriggerHandle : Cardinal;
  mSecs : DWORD) : Integer;
var
  Trigger : PTimerTrigger;
  T : TTriggerType;
begin
  DataLock.Enter;
  try

    FindTriggerFromHandle(TriggerHandle, False, T, Pointer(Trigger));
    if (Trigger <> nil) and (T = ttTimer) then
      with Trigger^ do begin
        Inc(tET.ExpireMS, mSecs);
        Result := ecOk;

        FOwner.DebugLog.AddDebugEntry(TApxCustomComPort,
          cclTimerTriggerExtend, TriggerHandle, mSecs)
      end
    else
      Result := ecBadTriggerHandle;

  finally
    DataLock.Leave;
  end;
end;

function TApxBaseDispatcher.TimerTimeRemaining(TriggerHandle : Cardinal;
                              var mSecsRemaining : DWORD) : Longint;
var
  Trigger : PTimerTrigger;
  T : TTriggerType;
begin
  mSecsRemaining := 0;
  DataLock.Enter;
  try

    FindTriggerFromHandle(TriggerHandle, False, T, Pointer(Trigger));
    if (Trigger <> nil) and (T = ttTimer) then
      with Trigger^ do begin
        mSecsRemaining := RemainingTime(tET);
        Result := ecOk;
      end
    else
      Result := ecBadTriggerHandle;

  finally
    DataLock.Leave;
  end;
end;

procedure TApxBaseDispatcher.UpdateHandlerFlags(FlagUpdate : TApHandlerFlagUpdate);
var
  HandlersInstalled : Boolean;
begin
  DataLock.Enter;
  try
    HandlersInstalled := (QtTriggerHandlers.Count > 1) or
      (ProcTriggerHandlers.Count > 0) or (EventTriggerHandlers.Count > 0);

    case FlagUpdate of
      fuKeepPort :
        HandlerServiceNeeded := (HandlersInstalled or PortHandlerInstalled);

      fuEnablePort :
        begin
          PortHandlerInstalled := True;
          HandlerServiceNeeded := True;
        end;

      fuDisablePort :
        begin
          PortHandlerInstalled := False;
          HandlerServiceNeeded := HandlersInstalled;
        end;
    end;
  finally
    DataLock.Leave;
  end;
end;

function TApxBaseDispatcher.SetStatusTrigger(TriggerHandle : Cardinal;
  Value : Cardinal; Activate : Boolean) : Integer;
type
  LH = record L,H : Byte; end;
var
  Trigger : PStatusTrigger;
  T : TTriggerType;

  { Return mask that can be checked against LastError later }
  function SetLineBits(Value : Cardinal) : Cardinal;
  begin
    Result := 0;
    if FlagIsSet(Value, lsOverrun) then
      Result := ce_Overrun;
    if FlagIsSet(Value, lsParity) then
      Result := Result or ce_RxParity;
    if FlagIsSet(Value, lsFraming) then
      Result := Result or ce_Frame;
    if FlagIsSet(Value, lsBreak) then
      Result := Result or ce_Break;
  end;

begin
  DataLock.Enter;
  try

    FindTriggerFromHandle(TriggerHandle, False, T, Pointer(Trigger));
    if (Trigger <> nil) and (T = ttStatus) then
      with Trigger^ do begin
        if Activate then begin
          case tSType of
            stLine  :
              tValue := SetLineBits(Value);
            stModem :
              begin
                { Hi tValue is delta mask, Lo is current modem status }
                LH(tValue).H := Value;
                LH(tValue).L := Value and ModemStatus;
              end;
            stOutBuffFree,
            stOutBuffUsed :
              tValue := Value;
          end;
          FOwner.DebugLog.AddDebugEntry(TApxCustomComPort,
            cclStatusTriggerActive, TriggerHandle, tValue);
        end;
        tSActive := Activate;
        Result := ecOK;
      end
    else
      Result := ecBadTriggerHandle;

  finally
    DataLock.Leave;
  end;
end;

class procedure TApxBaseDispatcher.ClearSaveBuffers(var Save : TTriggerSave);
begin
  with Save do begin
    if tsTimerTriggers <> nil then begin
      while tsTimerTriggers.Count > 0 do begin
        Dispose(PTimerTrigger(tsTimerTriggers[0]));
        tsTimerTriggers.Delete(0);
      end;
      tsTimerTriggers.Free;
      tsTimerTriggers := nil;
    end;
    if tsDataTriggers <> nil then begin
      while tsDataTriggers.Count > 0 do begin
        Dispose(PDataTrigger(tsDataTriggers[0]));
        tsDataTriggers.Delete(0);
      end;
      tsDataTriggers.Free;
      tsDataTriggers := nil;
    end;
    if tsStatusTriggers <> nil then begin
      while tsStatusTriggers.Count > 0 do begin
        Dispose(PStatusTrigger(tsStatusTriggers[0]));
        tsStatusTriggers.Delete(0);
      end;
      tsStatusTriggers.Free;
      tsStatusTriggers := nil;
    end;
  end;
end;

{ Saves all current triggers to Save }
procedure TApxBaseDispatcher.SaveTriggers(var Save : TTriggerSave);
var
  i : Integer;
  NewTimerTrigger : PTimerTrigger;
  NewDataTrigger : PDataTrigger;
  NewStatusTrigger : PStatusTrigger;
begin
  with Save do begin
    DataLock.Enter;
    try
      ClearSaveBuffers(Save);

      tsTimerTriggers := TList.Create;
      for i := 0 to pred(TimerTriggers.Count) do begin
        NewTimerTrigger := AllocMem(sizeof(TTimerTrigger));
        move(PTimerTrigger(TimerTriggers[i])^, NewTimerTrigger^,
          sizeof(TTimerTrigger));
        tsTimerTriggers.Add(NewTimerTrigger);
      end;
      tsDataTriggers := TList.Create;

      for i := 0 to pred(DataTriggers.Count) do begin
        NewDataTrigger := AllocMem(sizeof(TDataTrigger));
        move(PDataTrigger(DataTriggers[i])^, NewDataTrigger^,
          sizeof(TDataTrigger));
        tsDataTriggers.Add(NewDataTrigger);
      end;

      tsStatusTriggers := TList.Create;
      for i := 0 to pred(StatusTriggers.Count) do begin
        NewStatusTrigger := AllocMem(sizeof(TStatusTrigger));
        move(PStatusTrigger(StatusTriggers[i])^, NewStatusTrigger^,
          sizeof(TStatusTrigger));
        tsStatusTriggers.Add(NewStatusTrigger);
      end;

    finally
      DataLock.Leave;
    end;
  end;
end;

{ Restores previously saved triggers }
procedure TApxBaseDispatcher.RestoreTriggers(var Save : TTriggerSave);
var
  I : Integer;
  NewTimerTrigger : PTimerTrigger;
  NewDataTrigger : PDataTrigger;
  NewStatusTrigger : PStatusTrigger;
begin
  with Save do begin
    DataLock.Enter;
    try

      while TimerTriggers.Count > 0 do begin
        Dispose(PTimerTrigger(TimerTriggers[0]));
        TimerTriggers.Delete(0);
      end;

      while DataTriggers.Count > 0 do begin
        Dispose(PDataTrigger(DataTriggers[0]));
        DataTriggers.Delete(0);
      end;

      while StatusTriggers.Count > 0 do begin
        Dispose(PStatusTrigger(StatusTriggers[0]));
        StatusTriggers.Delete(0);
      end;

      if tsTimerTriggers <> nil then
        for I := 0 to pred(tsTimerTriggers.Count) do begin
          NewTimerTrigger := AllocMem(sizeof(TTimerTrigger));
          move(PTimerTrigger(tsTimerTriggers[i])^, NewTimerTrigger^,
            sizeof(TTimerTrigger));
          TimerTriggers.Add(NewTimerTrigger);
        end;

      if tsDataTriggers <> nil then
        for I := 0 to pred(tsDataTriggers.Count) do begin
          NewDataTrigger := AllocMem(sizeof(TDataTrigger));
          move(PDataTrigger(tsDataTriggers[i])^, NewDataTrigger^,
            sizeof(TDataTrigger));
          DataTriggers.Add(NewDataTrigger);
        end;

      if tsStatusTriggers <> nil then
        for I := 0 to pred(tsStatusTriggers.Count) do begin
          NewStatusTrigger := AllocMem(sizeof(TStatusTrigger));
          move(PStatusTrigger(tsStatusTriggers[i])^, NewStatusTrigger^,
            sizeof(TStatusTrigger));
          StatusTriggers.Add(NewStatusTrigger);
        end;

    finally
      DataLock.Leave;
    end;
  end;
end;

{ Set a data pointer }
function TApxBaseDispatcher.SetDataPointer(P : Pointer; Index : Cardinal) : Integer;
begin
  DataLock.Enter;
  try

    if (Index >= 1) and (Index <= MaxDataPointers) then begin
      DataPointers[Index] := P;
      Result := ecOK;
    end else
      Result := ecBadArgument;

  finally
    DataLock.Leave;
  end;
end;

{ Return a data pointer }
function TApxBaseDispatcher.GetDataPointer(var P : Pointer; Index : Cardinal) : Integer;
begin
  DataLock.Enter;
  try

    if (Index >= 1) and (Index <= MaxDataPointers) then begin
      P := DataPointers[Index];
      Result := ecOK;
    end else
      Result := ecBadArgument;

  finally
    DataLock.Leave;
  end;
end;

{ Return the type for TriggerHandle }
function TApxBaseDispatcher.ClassifyStatusTrigger(TriggerHandle : Cardinal) : Cardinal;
begin
  Result := TriggerHandle and StatusTypeMask;
end;

{ Set/Clear the event busy flag }
procedure TApxBaseDispatcher.SetEventBusy(var WasOn : Boolean; SetOn : Boolean);
begin
  DataLock.Enter;
  try
    WasOn := FEventBusy;
    FEventBusy := SetOn;
  finally
    DataLock.Leave;
  end;
end;

constructor TApxDispatcherThread.Create(Disp : TApxBaseDispatcher);
begin
  H := Disp;
  inherited Create(False);
  FreeOnTerminate := True;
end;

procedure TApxDispatcherThread.SyncEvent;
begin
  pTriggerEvent(pMsg,pTrigger,plParam);
end;

procedure TApxDispatcherThread.SyncNotify(Msg, Trigger : Cardinal;
  lParam : LongInt; Event : TApxNotifyEvent);
begin
  pMsg := Msg;
  pTrigger := Trigger;
  plParam := lParam;
  pTriggerEvent := Event;
  Synchronize(SyncEvent);
  if Assigned(WakeMainThread) then WakeMainThread(nil);
end;

{ public version of Synchronize }
procedure TApxDispatcherThread.Sync(Method: TThreadMethod);
begin
  Synchronize(Method);
  if Assigned(WakeMainThread) then WakeMainThread(nil);
end;

procedure TReadThread.Execute;
  { -Wait for and process communications events }
const
  ModemEvent = EV_CTS or EV_DSR or EV_RLSD or EV_RING or EV_RINGTE;
  LineEvent = EV_ERR or EV_BREAK;

  procedure ProcessComEvent(H : TApxBaseDispatcher);
  begin
    with H do begin
      { Check for modem events }
      if CurrentEvent and ModemEvent <> 0 then begin

        { A modem status event... }
        MapEventsToMS(CurrentEvent);

        { Check for status triggers }
        if not FEventBusy then begin
          while CheckStatusTriggers do
            if ClosePending then
              Exit;

          { Allow status triggers to hit again }
          if GlobalStatHit then
            ResetStatusHits;
        end;
      end;

      { Check for line events }
      if CurrentEvent and LineEvent <> 0 then begin
        { A line status/error event }
        RefreshStatus;

        { Check for status triggers }
        if not FEventBusy then begin
          while CheckStatusTriggers do
            if ClosePending then
              Exit;

          { Allow status triggers to hit again }
          if GlobalStatHit then
            ResetStatusHits;
        end;
      end;

      { Get any available data }
      ExtractData;

      { Check for received status & data triggers }
      if not FEventBusy then begin
        while CheckStatusTriggers do
          if ClosePending then
            Exit;
        while CheckReceiveTriggers do
          if ClosePending then
            Exit;
      end;
      if GlobalStatHit then
        ResetStatusHits;
    end;
  end;

  procedure ProcessTimer(H : TApxBaseDispatcher);
  begin
    with H do begin
      if ClosePending then
        Exit;

      if not FEventBusy then begin
        GlobalStatHit := False;

        { Issue all status and timer triggers }
        while (CheckStatusTriggers or CheckTimerTriggers) and not ClosePending do ;

        { Allow status triggers to hit again }
        if GlobalStatHit then
          ResetStatusHits;

      end;
    end;
  end;

  { Yes boys and girls, we're checking if there's any data in the  }
  { output buffer without the protection of a critical section.    }
  { This is intentional, and justified (IMO) because (1) We only   }
  { read, never write; (2) What we're doing here is fail-safe even }
  { if we hit a variable at a "bad time", the worse thing that     }
  { could happen is we'll set the output event unnecessarily -     }
  { which would be no deal deal; (3) We can't risk a thread        }
  { deadlock here. Don't try this at home.  :o)                    }
  procedure CheckOutBuf;
  begin
    if H.OBufFull or (H.OBufTail <> H.OBufHead) then
      H.OutputEvent.SetEvent;
  end;

begin
  { Hack to make exception handling behave }
  if AreOSExceptionsBlocked then ;

  { Increment thread counter }
  InterLockedIncrement(H.ActiveThreads);

  { Create MsgEvent for synchronizing PostEvents }
  MsgEvent := TEvent.Create(nil, False, False, '');

  try
    with H do begin
      try
        { Repeat until port is closed }
        while not Terminated do begin

          { Wait for either a com event or a timeout }
          WaitComEvent(50);

          { Exit immediately if thread was killed while waiting }
          if Terminated then begin
            { Finished here, okay to close the port }
            Exit;
          end;

          { Process it... }
          ProcessComEvent(H);
          ProcessTimer(H);

          { Wake up the output thread if necessary }
          CheckOutBuf;
        end;

      finally
        { Make sure DonePortPrim gets called }
        DoDonePortPrim := ClosePending;
        H.ThreadGone(Self);
        MsgEvent.Free;
        MsgEvent := nil;
      end;
    end;
  except
    ShowException(ExceptObject, ExceptAddr);
  end;
end;

{ Write data to port }
procedure TWriteThread.Execute;
var
  Err : Integer;
  WriteSize : DWORD;
begin
  { Hack to make exception handling behave }
  if AreOSExceptionsBlocked then ;

  { Increment thread counter }
  InterLockedIncrement(H.ActiveThreads);

  { Create MsgEvent for synchronizing PostEvents }
  MsgEvent := TEvent.Create(nil, False, False, '');

  { Main execution loop }
  while not Terminated do begin

    { Wait for data to send }
    H.OutputEvent.WaitFor($FFFFFFFF);  { Infinite wait }

    H.OutputLock.Enter;
    try
      { Is there data to send? }
      if H.OutBuffUsed > 0 then begin

        { Check for wrap }
        if (H.OBufTail < H.OBufHead) or H.OBufFull then begin
          WriteSize := H.OBufSize - H.OBufHead;
          Err := H.WriteCom(@H.OBuffer[H.OBufHead], WriteSize);
          if Err >= 0 then begin
            { Log successful writes }
            H.FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclWriteCom,
              DWORD(@H.OBuffer[H.OBufHead]), Err);
            { Reposition buffer head }
            Inc(H.OBufHead, Err);
            { Sent first half, unwrap the wrap }
            if (H.OBufHead = H.OBufSize) then
              H.OBufHead := 0;
          end;
        end else begin
          WriteSize := H.OBufTail - H.OBufHead;
          Err := H.WriteCom(@H.OBuffer[H.OBufHead], WriteSize);
          if Err >= 0 then begin
            { Log successful writes }
            H.FOwner.DebugLog.AddDebugEntry(TApxCustomComPort, cclWriteCom,
              DWORD(@H.OBuffer[H.OBufHead]), Err);
            { Reposition buffer head }
            Inc(H.OBufHead, Err);
          end;
        end;
      end;
    finally
      H.OutputLock.Leave;
      H.ThreadGone(Self);
      MsgEvent.Free;
      MsgEvent := nil;
    end;
  end;
end;

{ TApxBaseLog }

{ Create instance of TApxBaseLog }
constructor TApxBaseLog.Create(Owner : TApxCustomComPort);
begin
  inherited Create;
  blOwner := Owner;
  InitializeCriticalSection(blLogCS);
end;

{ Destroy instance of TApxBaseLog }
destructor TApxBaseLog.Destroy;
begin
  DeleteCriticalSection(blLogCS);
  inherited;
end;

{ Enters TApxBaseLog critical section }
procedure TApxBaseLog.LockLog;
begin
  EnterCriticalSection(blLogCS);
end;

{ Leaves TApxBaseLog critical section }
procedure TApxBaseLog.UnlockLog;
begin
  LeaveCriticalSection(blLogCS);
end;

{ Get Enabled property }
function TApxBaseLog.GetEnabled : Boolean;
begin
  LockLog;
  try
    Result := FEnabled;
  finally
    UnlockLog;
  end;
end;

{ Get FileName property }
function TApxBaseLog.GetFileName : TFileName;
begin
  LockLog;
  try
    Result := FFileName;
  finally
    UnlockLog;
  end;
end;

{ Set Enabled property }
procedure TApxBaseLog.SetEnabled(const Value : Boolean);
begin
  LockLog;
  try
    FEnabled := Value;
  finally
    UnlockLog;
  end;
end;

{ Set FileName property }
procedure TApxBaseLog.SetFileName(const Value : TFileName);
begin
  LockLog;
  try
    FFileName := Value;
  finally
    UnlockLog;
  end;
end;

{ TApxDebugLog }

constructor TApxDebugLog.Create(Owner : TApxCustomComPort);
begin
  inherited;
  FBufferSize := 65536;
  FFileName := 'debug.log';
  dlTimeBase := AxTimeGetTime;
end;

destructor TApxDebugLog.Destroy;
begin
  FreeMem(dlBuffer);
  FreeMem(dlTempBuffer);
  inherited;
end;

{ AddDebugEntry notes:                                                }
{                                                                     }
{ This is designed to be a flexible and extendable debug logging      }
{ facility that allows new components to add their own customized     }
{ logging. It works together with the GetLogString class method       }
{ introduced in TApxBaseComponent.                                    }
{                                                                     }
{ deClass -- Class of component adding the entry (Self.ClassType)     }
{ When dumping the debug log to disk, this class will be expected to  }
{ return a string for the log based on the data contained in D1,      }
{ D2 and D3.                                                          }
{                                                                     }
{ D1, D2, D3 are "info" fields to be used in deClass's overridden     }
{ GetLogString class method. deClass will then presumably use the     }
{ info to format a string and pass it back to the debug log.          }
{                                                                     }
{ Special case: If the high bit of D1 is set, D2 becomes a pointer    }
{ to data, and D3 is the size of the data. Make *sure* the high bit   }
{ isn't set unless you are using this special situation.              }
{                                                                     }
{ Obviously, every class that might get passed in for deClass needs   }
{ to have an overridden GetLogString method that understands the      }
{ values it's passing in for D1, D2 & D3                              }
{                                                                     }
{ If you just have a simple case for logging that probably won't get  }
{ used that often, consider adding entries with the WriteDebugString  }
{ method.                                                             }
procedure TApxDebugLog.AddDebugEntry(const deClass : TApxComponentClass; const D1, D2, D3 : DWORD);
var
  DebugEntry : TApxDebugRec;
  EntryPtr : PApxDebugRec;
  SizeReq, TimeMrk, ChunkSize : DWORD;
  HasData : Boolean;
begin
  LockLog;
  try
    { Bail if we're not logging }
    if not Enabled then Exit;

    TimeMrk := AxTimeGetTime;

    { Determine size needed }
    SizeReq := SizeOf(TApxDebugRec);
    if (D1 and $80000000) = $80000000 then begin
      HasData := True;
      Inc(SizeReq, D3);
    end else begin
      HasData := False;
    end;

    { Bail if SizeReq is bigger than the whole buffer }
    if SizeReq > FBufferSize then Exit;

    { Make more room in buffer if necessary }
    while (SizeReq > BufferFree) and dlPopDebugEntry(DebugEntry) do ;

    { Do we need to wrap this entry? }
    if (dlBufferTail + SizeReq) <= FBufferSize then begin

      { Wrap not required, write directly to dlBuffer }
      EntryPtr := @dlBuffer[dlBufferTail];
      EntryPtr.drClass := deClass;
      EntryPtr.drTime := TimeMrk;
      EntryPtr.drData1 := D1;
      EntryPtr.drData2 := D2;
      EntryPtr.drData3 := D3;

      { Write add'l data if necessary }
      if HasData then begin
        Move(Pointer(D2)^, dlBuffer[dlBufferTail + SizeOf(TApxDebugRec)], D3);
      end;
      Inc(dlBufferTail, SizeReq);

      { Fix tail if necessary }
      if dlBufferTail = FBufferSize then
        dlBufferTail := 0;

    end else begin

      { Wrap required, use temp buffer }
      dlCheckTempSize(SizeReq);

      EntryPtr := @dlTempBuffer[0];
      EntryPtr.drClass := deClass;
      EntryPtr.drTime := TimeMrk;
      EntryPtr.drData1 := D1;
      EntryPtr.drData2 := D2;
      EntryPtr.drData3 := D3;

      { Write add'l data if necessary }
      if HasData then begin
        Move(Pointer(D2)^, dlTempBuffer[SizeOf(TApxDebugRec)], D3);
      end;

      { Move first half }
      ChunkSize := FBufferSize - dlBufferTail;
      Move(dlTempBuffer[0], dlBuffer[dlBufferTail], ChunkSize);

      { Move second half }
      Move(dlTempBuffer[ChunkSize], dlBuffer[0], SizeReq - ChunkSize);

      { Set tail }
      dlBufferTail := SizeReq - ChunkSize;
    end;
  finally
    UnlockLog;
  end;
end;

{ Clears all data from buffer (does not write data to disk) }
procedure TApxDebugLog.ClearBuffer;
begin
  LockLog;
  try
    dlBufferHead := 0;
    dlBufferTail := 0;
  finally
    UnlockLog;
  end;
end;

{ Verifies the size of the temp buffer }
procedure TApxDebugLog.dlCheckTempSize(SizeReq : DWORD);
begin
  if (SizeReq > dlTempSize) then begin
    ReallocMem(dlTempBuffer, SizeReq);
    dlTempSize := SizeReq;
  end;
end;

{ Generates a file header for the log }
function TApxDebugLog.dlDoFileHeader : string;
var
  Cmp : string;
begin
  { Compiler version }
  Cmp := 'Unknown Compiler';

  {$IFDEF Linux}
    {$IFDEF VER140}
      Cmp := 'Kylix 1.0';
    {$ENDIF}
  {$ENDIF}

  { Create / format header string }
  Result := 'TurboPower Cross-Platform Async Professional ' + ApxVersionStr + AxLineTerm +
    'Compiler : ' + Cmp + AxLineTerm +
    'Operating System : ' + AxGetOSVersionString + AxLineTerm +
    '=============================================================================' +
    AxLineTerm + AxLineTerm;
end;

{ Pop debug record from log, return False if no record to return }
function TApxDebugLog.dlPopDebugEntry(var DebugRec : TApxDebugRec) : Boolean;
type
  BytesArray = array[0..SizeOf(TApxDebugRec)-1] of Byte;
var
  Bytes : BytesArray absolute DebugRec;
  ChunkSize : DWORD;
begin
  LockLog;
  try
    { Check for empty buffer }
    if (dlBufferHead = dlBufferTail) then begin
      Result := False;
      Exit;
    end else begin
      Result := True;
    end;

    { Check to see if debug record wraps }
    if (dlBufferHead + SizeOf(TApxDebugRec)) <= FBufferSize then begin

      { No wrap, copy directly over }
      Move(dlBuffer[dlBufferHead], DebugRec, SizeOf(DebugRec));
      Inc(dlBufferHead, SizeOf(DebugRec));

      { Fix head if needed }
      if (dlBufferHead = FBufferSize) then dlBufferHead := 0;
    end else begin

      { Need to deal with wrap -- copy first half }
      ChunkSize := (FBufferSize - dlBufferHead);
      Move(dlBuffer[dlBufferHead], Bytes[0], ChunkSize);

      { Copy second half }
      Move(dlBuffer[0], Bytes[ChunkSize], (SizeOf(DebugRec) - ChunkSize));
      dlBufferHead := SizeOf(DebugRec) - ChunkSize;
    end;

    { Do we have data? If so, deal with it }
    if (DebugRec.drData1 and $80000000) = $80000000 then begin

      { Check to see if debug data wraps }
      if (dlBufferHead + DebugRec.drData3) <= FBufferSize then begin

        { No wrap -- point D2 to buffer }
        DebugRec.drData2 := DWORD(@dlBuffer[dlBufferHead]);
        Inc(dlBufferHead, DebugRec.drData3);
      end else begin

        { Wrap -- copy first half to temp buffer }
        dlCheckTempSize(DebugRec.drData3);
        ChunkSize := (FBufferSize - dlBufferHead);
        Move(dlBuffer[dlBufferHead], dlTempBuffer[0], ChunkSize);

        { Copy second half }
        Move(dlBuffer[0], dlTempBuffer[ChunkSize], (DebugRec.drData3 - ChunkSize));
        DebugRec.drData2 := DWORD(@dlTempBuffer[0]);
        dlBufferHead := DebugRec.drData3 - ChunkSize;
      end;
    end

  finally
    UnlockLog;
  end;
end;

{ Return time stamp string }
function TApxDebugLog.dlTimeStamp(Mark : DWORD) : string;
begin
  Result := Format('%07.7d : ', [Mark - dlTimeBase]);
  Insert('.', Result, 5);
end;

{ Dumps log file to disk }
procedure TApxDebugLog.DumpLog;
var
  DR : TApxDebugRec;
  FS : TFileStream;
  S, T : string;
begin
  LockLog;

  try
    { Open file stream }
    if FileExists(FileName) and (WriteMode = wmAppend) then begin
      FS := TFileStream.Create(FileName, fmOpenReadWrite or fmShareDenyWrite);
      FS.Seek(soFromEnd, 0);
    end else begin
      FS := TFileStream.Create(FileName, fmCreate or fmShareDenyWrite);
    end;

    try
      { Do file header if appropriate }
      if (FS.Size = 0) then begin
        S := dlDoFileHeader;
        FS.Write(S[1], Length(S));
      end;

      { Cycle through all data }
      while dlPopDebugEntry(DR) do begin
        if DR.drClass <> nil then begin

          S := '!!! Unhandled log entry : ' + DR.drClass.ClassName +
            '[' + IntToStr(DR.drData1) + ']' + AxLineTerm;

          { It belongs to someone else, let them process it }
          if DR.drClass.InheritsFrom(TApxBaseComponent) then
            S := TApxBaseComponentClass(DR.drClass).GetLogString(DR.drData1, DR.drData2, DR.drData3);

          if DR.drClass.InheritsFrom(TApxBaseHandleComponent) then
            S := TApxBaseHandleComponentClass(DR.drClass).GetLogString(DR.drData1, DR.drData2, DR.drData3);

        end else begin

          { Something we're supposed to know about, deal with it }
          case DR.drData1 of

            { Logging enabled }
            deEnabled :
              begin
                S := '**** Logging Enabled' + AxLineTerm;
              end;

            { Logging disabled }
            deDisabled :
              begin
                S := '**** Logging Disabled' + AxLineTerm;
              end;

            { WriteDebugString entry }
            deString :
              begin
                SetLength(S, DR.drData3);
                Move(PByteArray(DR.drData2)[0], S[1], DR.drData3);
              end;

          { If it falls through, I guess we didn't know about it after all <g> }
          end;
        end;

        { Write time stamp }
        T := dlTimeStamp(DR.drTime);
        FS.Write(T[1], Length(T));

        { Write log string }
        FS.Write(S[1], Length(S));

        { Write trailing CRLF }
        FS.Write(AxLineTerm, Length(AxLineTerm));
      end;

    finally
      FS.Free;
    end;

  finally
    UnlockLog;
  end;
end;

{ Calculates free space in the buffer }
function TApxDebugLog.GetBufferFree : DWORD;
begin
  LockLog;
  try
    if (dlBufferHead <= dlBufferTail) then
      { One less than actual, since we always leave one byte free }
      Result := Pred(FBufferSize - (dlBufferTail - dlBufferHead))
    else
      Result := Pred(dlBufferHead - dlBufferTail);
  finally
    UnlockLog;
  end;
end;

{ Retrieves buffer size }
function TApxDebugLog.GetBufferSize : DWORD;
begin
  LockLog;
  try
    Result := FBufferSize;
  finally
    UnlockLog;
  end;
end;

{ Retrieves write mode }
function TApxDebugLog.GetWriteMode : TApxWriteMode;
begin
  LockLog;
  try
    Result := FWriteMode;
  finally
    UnlockLog;
  end;
end;

{ Sets the size of the logging buffer }
procedure TApxDebugLog.SetBufferSize(const Value : DWORD);
begin
  LockLog;
  try
    if Value <> FBufferSize then begin
      FBufferSize := Value;
      ReallocMem(dlBuffer, Value);
      ClearBuffer;
    end;
  finally
    UnlockLog;
  end;
end;

{ Enables (or disables) logging }
procedure TApxDebugLog.SetEnabled(const Value : Boolean);
begin
  inherited;
  LockLog;
  try
    if (Value = True) then begin

      { Allocate buffer if not already done }
      if (dlBuffer = nil) then begin
        GetMem(dlBuffer, FBufferSize);
      end;

      { Init temp buffer if not already done }
      if (dlTempBuffer = nil) then begin
        dlTempSize := 1024;
        GetMem(dlTempBuffer, dlTempSize);
      end;

      AddDebugEntry(nil, deEnabled, 0, 0);

    end else begin
      AddDebugEntry(nil, deDisabled, 0, 0);
    end;
  finally
    UnlockLog;
  end;
end;

{ Write debug string to log buffer }
procedure TApxDebugLog.WriteDebugString(const DebugString : string);
begin
  AddDebugEntry(nil, deString, DWORD(DebugString), Length(DebugString));
end;

procedure TApxDebugLog.SetWriteMode(const Value : TApxWriteMode);
begin
  LockLog;
  try
    FWriteMode := Value;
  finally
    UnlockLog;
  end;
end;

{ TAxDeviceLayerList }

var
  dllCS : TCriticalSection = nil;

type
  TApxDeviceLayerInfo = class
    DispatcherClass : TApxDispatcherClass;
  end;

  TApxDeviceLayerList = class(TStringList)
  protected
    procedure LockList;
    procedure UnlockList;
  public
    constructor Create;
    destructor Destroy; override;
    procedure AddDeviceLayer(const ATag : string; AClass : TApxDispatcherClass);
    function DeviceIsValid(const Device : string) : Boolean;
    function GetDefaultDevice : string;
    procedure Remove(AClass : TApxDispatcherClass);
  end;

constructor TApxDeviceLayerList.Create;
begin
  inherited Create;
  Sorted := False;
  Duplicates := dupError;
  CaseSensitive := False;

  { Set default devices }

  {$IFDEF Win32}
  AddDeviceLayer('dlWin32', TApxWin32Dispatcher);
  {$ENDIF}
  {$IFDEF Linux}
  AddDeviceLayer('dlLinux', TApxLinuxDispatcher);
  {$ENDIF}
end;

destructor TApxDeviceLayerList.Destroy;
var
  I: Integer;
begin
  LockList;
  try
    for I := 0 to Count-1 do
      TApxDeviceLayerInfo(Objects[I]).Free;
    inherited Destroy;
  finally
    UnlockList;
  end;
end;

procedure TApxDeviceLayerList.LockList;
begin
  dllCS.Enter;
end;

procedure TApxDeviceLayerList.UnlockList;
begin
  dllCS.Leave;
end;

procedure TApxDeviceLayerList.AddDeviceLayer(const ATag : string; AClass : TApxDispatcherClass);
var
  Temp : TApxDeviceLayerInfo;
begin
  LockList;
  try
    Temp := TApxDeviceLayerInfo.Create;
    with Temp do begin
      DispatcherClass := AClass;
    end;
    AddObject(ATag, Temp);
  finally
    UnlockList;
  end;
end;

function TApxDeviceLayerList.DeviceIsValid(const Device : string) : Boolean;
begin
  LockList;
  try
    Result := IndexOf(Device) <> -1;
  finally
    UnlockList;
  end;
end;

function TApxDeviceLayerList.GetDefaultDevice : string;
begin
  LockList;
  try
    Result := Self.Strings[0];
  finally
    UnlockList;
  end;
end;

procedure TApxDeviceLayerList.Remove(AClass : TApxDispatcherClass);
var
  I : Integer;
  Cvt : TApxDeviceLayerInfo;
begin
  LockList;
  try
    for I := Count-1 downto 0 do begin
      Cvt := TApxDeviceLayerInfo(Objects[I]);
      if Cvt.DispatcherClass.InheritsFrom(AClass) then begin
        Cvt.Free;
        Delete(I);
      end;
    end;
  finally
    UnlockList;
  end;
end;

var
  DeviceLayers : TApxDeviceLayerList = nil;

  function GetDeviceLayers : TApxDeviceLayerList;
  begin
    dllCS.Enter;
    try
      if DeviceLayers = nil then
        DeviceLayers := TApxDeviceLayerList.Create;
      Result := DeviceLayers;
    finally
      dllCS.Leave;
    end;
  end;

{ TApxCustomComPort }

{ Create the object instance }
constructor TApxCustomComPort.Create(AOwner : TComponent);
begin
  { Debugging }
  FDebugLog := TApxDebugLog.Create(Self);

  { Configuration }
  DeviceParams := TApxSerialDeviceParams.Create;

  { Create the registration list before notification events are sent }
  UserList := TList.Create;

  { This causes notification events for all other components }
  inherited Create(AOwner);

  { Private inits }
  PortState    := psClosed;
  ForceOpen    := False;
  CopyTriggers := False;

  { Data inits }
  FPromptForPort := adpoDefPromptForPort;
  FDeviceLayer   := GetDeviceLayers.GetDefaultDevice;
  FOpen          := adpoDefOpen;
  FAutoOpen      := adpoDefAutoOpen;

  { Contained within DeviceParams }
  DeviceParams.BaseAddress := adpoDefBaseAddress;
  DeviceParams.Baud := adpoDefBaudRt;
  DeviceParams.Databits := adpoDefDatabits;
  DeviceParams.DTR := adpoDefDTR;
  DeviceParams.HWFlowControl := True;
  DeviceParams.InSize := adpoDefInSize;
  DeviceParams.OutSize := adpoDefOutSize;
  DeviceParams.Parity := adpoDefParity;
  DeviceParams.RS485Mode := adpoDefRS485Mode;
  DeviceParams.RTS := adpoDefRTS;
  DeviceParams.Stopbits := adpoDefStopbits;
  DeviceParams.UseEventWord := adpoDefUseEventWord;
  DeviceParams.XOffChar := adpoDefXOffChar;
  DeviceParams.XOnChar := adpoDefXOnChar;
end;

{ Destroy the object instance }
destructor TApxCustomComPort.Destroy;
var
  I : Word;
  UL : PUserListEntry;
begin

  { Close the port }
  if (PortState = psOpen) then begin
    DonePort;
  end;

  { Dump debug log }
  if Assigned(FDebugLog) then
    FDebugLog.DumpLog;

  { Get rid of the user list }
  if UserList.Count > 0 then begin
    for I := UserList.Count-1 downto 0 do begin
      UL := UserList.Items[I];
      UserList.Remove(UL);
      Dispose(UL);
    end;
  end;
  UserList.Free;
  DeviceParams.Free;
  FDebugLog.Free;

  TApxBaseDispatcher.ClearSaveBuffers(SaveTriggerBuffer);

  inherited Destroy;
end;

class function TApxCustomComPort.GetLogString(const D1, D2, D3 : DWORD) : string;
const
  StatusTrigStr : array[0..5] of string =
    ('NotActive', 'Modem', 'Line', 'OutBuffFree', 'OutBuffUsed', 'OutSent');
var
  S : string;
  TempInt : Integer;
begin
  { Awww, what the heck }
  if D3 = 1 then
    S := 'byte'
  else
    S := 'bytes';

  { Format and return appropriate string }
  case D1 of
    cclOpenCom :
      begin
        SetLength(S, D3);
        Move(PByteArray(D2)[0], S[1], D3);
        Result := '+++ Opened ' + S + AxLineTerm;
      end;
    cclCloseCom :
      begin
        Result := '--- Close' + AxLineTerm;
      end;
    cclWriteCom :
      begin
        Result := Format('Write - %d %s written to device' + AxLineTerm, [D3, S]);
        Result := Result + HexifyBlock(PByteArray(D2)[0], D3);
      end;

    cclReadCom :
      begin
        Result := Format('Read - %d %s read from device' + AxLineTerm, [D3, S]);
        Result := Result + HexifyBlock(PByteArray(D2)[0], D3);
      end;

    cclStatusTrigger :
      begin
        Result := Format('Status Trigger Fired :: Handle = %d' + AxLineTerm, [D2]);
      end;

    cclAvailTrigger :
      begin
        Result := Format('Avail Trigger Fired :: %d bytes available' + AxLineTerm, [D2]);
      end;

    cclDataTrigger :
      begin
        Result := Format('Data Trigger Fired :: Handle = %d' + AxLineTerm, [D2]);
      end;

    cclTimerTrigger :
      begin
        Result := Format('Timer Trigger Fired :: Handle = %d' + AxLineTerm, [D2]);
      end;

    cclAllocWndHandler :
      begin
        Result := Format('Window Handler Allocated :: Handle = %d' + AxLineTerm, [D2]);
      end;

    cclAllocProcHandler :
      begin
        Result := 'Procedure Handler Allocated' + AxLineTerm;
      end;

    cclAllocEventHandler :
      begin
        if D2 = 0 then
          Result := 'Non-Sync Event Handler Allocated' + AxLineTerm
        else
          Result := 'Sync Event Handler Allocated' + AxLineTerm;
      end;

    cclAllocDataTrigger :
      begin
        TempInt := Integer(Pointer(D2)^);
        Result := Format('Data Trigger Allocated :: Handle = %d' + AxLineTerm, [TempInt]);
        Result := Result + HexifyBlock(PByteArray(D2)[4], D3-4);
      end;

    cclAllocTimerTrigger :
      begin
        Result := Format('Timer Trigger Allocated :: Handle = %d' + AxLineTerm, [D2]);
      end;

    cclAllocStatusTrigger :
      begin
        Result := Format('%s Status Trigger Allocated :: Handle = %d' +
          AxLineTerm, [StatusTrigStr[D3], D2]);
      end;

    cclDispWndHandler :
      begin
        Result := Format('Window Handler Disposed :: Handle = %d' + AxLineTerm, [D2]);
      end;

    cclDispProcHandler :
      begin
        Result := 'Procedure Handler Disposed' + AxLineTerm;
      end;

    cclDispEventHandler :
      begin
        Result := 'Event Handler Disposed' + AxLineTerm;
      end;

    cclDispTimerTrigger :
      begin
        Result := Format('Timer Trigger Disposed :: Handle = %d' + AxLineTerm, [D2]);
      end;

    cclDispStatusTrigger :
      begin
        Result := Format('Status Trigger Disposed :: Handle = %d' + AxLineTerm, [D2]);
      end;

    cclDispDataTrigger :
      begin
        Result := Format('Data Trigger Disposed :: Handle = %d' + AxLineTerm, [D2]);
      end;

    cclTimerTriggerActive :
      begin
        Result := Format('Timer Trigger Activated :: Handle = %d :: %d mSec' + AxLineTerm, [D2, D3]);
      end;

    cclTimerTriggerDeact :
      begin
        Result := Format('Timer Trigger Deactivated :: Handle = %d' + AxLineTerm, [D2]);
      end;

    cclTimerTriggerExtend :
      begin
        Result := Format('Timer Trigger Extended :: Handle = %d :: %d mSec' + AxLineTerm, [D2, D3]);
      end;

    cclStatusTriggerActive :
      begin
        Result := Format('Status Trigger Activated :: Handle = %d :: $%.8x' + AxLineTerm, [D2, D3]);
      end;

  end;
end;

class procedure TApxCustomComPort.RegisterDeviceLayer(const ATag : string;
  AClass : TApxDispatcherClass);
begin
  GetDeviceLayers.AddDeviceLayer(ATag, AClass);
end;

class procedure TApxCustomComPort.UnRegisterDeviceLayerClass(AClass : TApxDispatcherClass);
begin
  if Assigned(DeviceLayers) then DeviceLayers.Remove(AClass);
end;

{ Handle ClosePending messages }
procedure TApxCustomComPort.APXClosePending(var Message : TAxMessage);
begin
  if FDispatcher.Active then begin
    AxPostMessage(FHandle, APX_CLOSEPENDING, nil, 0, Message.LParam);
  end else begin
    { Get rid of the trigger handler }
    RegisterComPort(False);
    FDispatcher.Free;
    FDispatcher := nil;
    PortState := psClosed;
    if OpenPending then begin
      InitPort;
      OpenPending := False;
    end;
  end;
end;

{ Handle TriggerAvail messages }
procedure TApxCustomComPort.APXTriggerAvail(var Message : TAxMessage);
begin
  Trigger(Message.Msg, Message.WParam, Message.WParam);
  TriggerAvail(Message.WParam);
  {$IFDEF Linux}
  if Assigned(Message.Event) then
    Message.Event.SetEvent;
  {$ENDIF}
end;

{ Handle TriggerData messages }
procedure TApxCustomComPort.APXTriggerData(var Message : TAxMessage);
begin
  Trigger(Message.Msg, Message.WParam, Message.LParam);
  TriggerData(Message.WParam);
  {$IFDEF Linux}
  if Assigned(Message.Event) then
    Message.Event.SetEvent;
  {$ENDIF}
end;

{ Handle TriggerStatus messages }
procedure TApxCustomComPort.APXTriggerStatus(var Message : TAxMessage);
begin
  Trigger(Message.Msg, Message.WParam, Message.LParam);
  TriggerStatus(Message.WParam);
  case Dispatcher.ClassifyStatusTrigger(Message.WParam) of
    stModem       : TriggerModemStatus;
    stLine        : TriggerLineError(LineError, LineBreak);
    stOutBuffFree : TriggerOutbuffFree;
    stOutBuffUsed : TriggerOutbuffUsed;
    stOutSent     : TriggerOutSent;
  end;
  {$IFDEF Linux}
  if Assigned(Message.Event) then
    Message.Event.SetEvent;
  {$ENDIF}
end;

{ Handle TriggerTimer messages }
procedure TApxCustomComPort.APXTriggerTimer(var Message : TAxMessage);
begin
  Trigger(Message.Msg, Message.WParam, Message.LParam);
  TriggerTimer(Message.WParam);
  {$IFDEF Linux}
  if Assigned(Message.Event) then
    Message.Event.SetEvent;
  {$ENDIF}
end;

function TApxCustomComPort.EventFilter(Sender : QObjectH; Event : QEventH) : Boolean;

  procedure DoCustomDispatchEvent(Event : QCustomEventH; FreeData : Boolean);
  var
    DataReference : PAxMessage;
    Data          : TAxMessage;
  begin
    DataReference := QCustomEvent_data(Event);

    { Local copy of data }
    Data := DataReference^;

    if FreeData then
      Dispose(DataReference);

    try
      Self.Dispatch(Data);
    except
      raise;
    end;

    if not FreeData then
      DataReference^.Result := Data.Result;
  end;

begin

  case QEvent_type(Event) of

    QEventType_AxSendDispatchMessage :
      begin
        DoCustomDispatchEvent(QCustomEventH(Event), False);
        Result := True;
      end;

    QEventType_AxPostDispatchMessage :
      begin
        DoCustomDispatchEvent(QCustomEventH(Event), True);
        Result := True;
      end;

    else Result := inherited EventFilter(Sender, Event);
  end;
end;

{ Return the current dispatcher object. Raise an exception if nil }
function TApxCustomComPort.ValidDispatcher : TApxBaseDispatcher;
begin
  if (Dispatcher = nil) and not (csLoading in Self.ComponentState) then
    raise ECommNotOpen.Create (ecCommNotOpen, False);
  Result := Dispatcher;
end;

{ Set the databits }
procedure TApxCustomComPort.SetDatabits(const Value : TAxDatabits);
begin
  DeviceParams.Databits := Value;
end;

{ Set a new device layer, ignore if port is open }
procedure TApxCustomComPort.SetDeviceLayer(const Value : string);
begin
  if (not AnsiSameStr(Value, FDeviceLayer)) and (PortState = psClosed) then begin
    if GetDeviceLayers.DeviceIsValid(Value) then begin
      FDeviceLayer := Value;
    end;
  end;
end;

{ Set a new device name, close the old device if open }
procedure TApxCustomComPort.SetDeviceName(const Value : string);
var
  WasOpen : Boolean;
begin
  if AnsiCompareStr(Value, FDeviceName) <> 0 then begin
    WasOpen := (PortState = psOpen);
    if (PortState = psOpen) then begin
      Dispatcher.SaveTriggers(SaveTriggerBuffer);
      Open := False;
    end;
    FDeviceName := Value;
    if WasOpen then begin
      Open := True;
      Dispatcher.RestoreTriggers(SaveTriggerBuffer);
    end;
  end;
end;

{ Change the setting of the DTR bit }
procedure TApxCustomComPort.SetDTR(const Value : Boolean);
begin
  DeviceParams.DTR := Value;
end;

{ Return the current baud }
function TApxCustomComPort.GetBaud : LongInt;
begin
  Result := DeviceParams.Baud;
end;

{ Return the buffer full setting }
function TApxCustomComPort.GetBufferFull : Byte;
begin
  Result := DeviceParams.BufferFull;
end;

{ Return the buffer resume setting }
function TApxCustomComPort.GetBufferResume : Byte;
begin
  Result := DeviceParams.BufferResume;
end;

{ Set a new baud rate }
procedure TApxCustomComPort.SetBaud(const Value : LongInt);
begin
  DeviceParams.Baud := Value;
end;

{ Set new insize, requires re-opening port if port was open }
procedure TApxCustomComPort.SetInSize(const Value : Word);
begin
  DeviceParams.InSize := Value;
end;

{ Set new outsize, requires re-opening port if port was open }
procedure TApxCustomComPort.SetOutSize(const Value : Word);
begin
  DeviceParams.OutSize := Value;
end;

{ Open/close the port }
procedure TApxCustomComPort.SetOpen(const Value : Boolean);
begin
  if FOpen <> Value then begin
    if not (csDesigning in ComponentState) and
       not (csLoading in ComponentState) then begin
      if Value then begin
        if (PortState = psClosed) then
          InitPort
        else
          OpenPending := True;
      end else begin
        DonePort;
      end;
    end else begin
      FOpen := Value;
      if Value then
        ForceOpen := True;
    end;
  end;
end;

{ Set hardware flow control (RTS/CTS only) }
procedure TApxCustomComPort.SetHWFlowControl(const Value : Boolean);
begin
  DeviceParams.HWFlowControl := Value;
  { Force RS485 mode off if using RTS/CTS flow control }
  if DeviceParams.HWFlowControl = True then
    DeviceParams.RS485Mode := False;
end;

{ Set new SW flow control parameters }
procedure TApxCustomComPort.SetSWFlowControl(const Value : TAxSWFlowControl);
begin
  DeviceParams.SWFlowControl := Value;
end;

{ Set new XOff character }
procedure TApxCustomComPort.SetXOffChar(const Value : AnsiChar);
begin
  DeviceParams.XOffChar := Value;
end;

{ Set new XOn character }
procedure TApxCustomComPort.SetXOnChar(const Value : AnsiChar);
begin
  DeviceParams.XOnChar := Value;
end;

{ Set buffer full mark }
procedure TApxCustomComPort.SetBufferFull(const Value : Byte);
begin
  DeviceParams.BufferFull := Value;
end;

{ Change buffer resume mark }
procedure TApxCustomComPort.SetBufferResume(const Value : Byte);
begin
  DeviceParams.BufferResume := Value;
end;

{ Change the COM number }
procedure TApxCustomComPort.SetComNumber(const Value : Word);
begin
  if FComNumber <> Value then begin
    DeviceName := AxMakeComName(Value);
    FComNumber := Value;
  end;
end;

procedure TApxCustomComPort.SetOnTrigger(const Value : TTriggerEvent);
begin
  FOnTrigger := Value;
  UpdateHandlerFlag;
end;

procedure TApxCustomComPort.SetOnTriggerAvail(const Value : TTriggerAvailEvent);
begin
  FOnTriggerAvail := Value;
  UpdateHandlerFlag;
end;

procedure TApxCustomComPort.SetOnTriggerData(const Value : TTriggerDataEvent);
begin
  FOnTriggerData := Value;
  UpdateHandlerFlag;
end;

procedure TApxCustomComPort.SetOnTriggerStatus(const Value : TTriggerStatusEvent);
begin
  FOnTriggerStatus := Value;
  UpdateHandlerFlag;
end;

procedure TApxCustomComPort.SetOnTriggerTimer(const Value : TTriggerTimerEvent);
begin
  FOnTriggerTimer := Value;
  UpdateHandlerFlag;
end;

procedure TApxCustomComPort.SetOnTriggerLineError(const Value : TTriggerLineErrorEvent);
begin
  FOnTriggerLineError := Value;
  UpdateHandlerFlag;
end;

procedure TApxCustomComPort.SetOnTriggerModemStatus(const Value : TNotifyEvent);
begin
  FOnTriggerModemStatus := Value;
  UpdateHandlerFlag;
end;

procedure TApxCustomComPort.SetOnTriggerOutbuffFree(const Value : TNotifyEvent);
begin
  FOnTriggerOutbuffFree := Value;
  UpdateHandlerFlag;
end;

procedure TApxCustomComPort.SetOnTriggerOutbuffUsed(const Value : TNotifyEvent);
begin
  FOnTriggerOutbuffUsed := Value;
  UpdateHandlerFlag;
end;

procedure TApxCustomComPort.SetOnTriggerOutSent(const Value : TNotifyEvent);
begin
  FOnTriggerOutSent := Value;
  UpdateHandlerFlag;
end;

{ Return the current Dispatcher, opening the port if necessary }
function TApxCustomComPort.GetDispatcher : TApxBaseDispatcher;
begin
  if FDispatcher = nil then
    if not (csDesigning in ComponentState) then begin
      if (PortState <> psOpen) and
          (not (csLoading in ComponentState)) and
          AutoOpen then
        Open := True;
    end;
  Result := FDispatcher;
end;

{ Return the current modem status register value }
function TApxCustomComPort.GetModemStatus : Byte;
begin
  if (PortState = psShuttingDown) then
    Result := 0
  else
    Result := ValidDispatcher.GetModemStatus;
end;

{ Return CTS bit state }
function TApxCustomComPort.GetCTS : Boolean;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.CheckCTS
  else
    Result := False;
end;

{ Return the DSR bit state }
function TApxCustomComPort.GetDSR : Boolean;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.CheckDSR
  else
    Result := False;
end;

{ Return the DTR bit state }
function TApxCustomComPort.GetDTR : Boolean;
begin
  Result := DeviceParams.DTR;
end;

{ Return RI bit state }
function TApxCustomComPort.GetRI : Boolean;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.CheckRI
  else
    Result := False;
end;

{ Return DCD bit state }
function TApxCustomComPort.GetDCD : Boolean;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.CheckDCD
  else
    Result := False;
end;

{ Return the databits setting }
function TApxCustomComPort.GetDatabits : TAxDatabits;
begin
  Result := DeviceParams.Databits;
end;

{ Return delta CTS bit state }
function TApxCustomComPort.GetDeltaCTS : Boolean;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.CheckDeltaCTS
  else
    Result := False;
end;

{ Return delta DCD bit state }
function TApxCustomComPort.GetDeltaDCD : Boolean;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.CheckDeltaDCD
  else
    Result := False;
end;

{ Return delta DSR bit state }
function TApxCustomComPort.GetDeltaDSR : Boolean;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.CheckDeltaDSR
  else
    Result := False;
end;

{ Return delta RI bit state }
function TApxCustomComPort.GetDeltaRI : Boolean;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.CheckDeltaRI
  else
    Result := False;
end;

{ Return HW flow control setting }
function TApxCustomComPort.GetHWFlowControl : Boolean;
begin
  Result := DeviceParams.HWFlowControl;
end;

{ Return amount of freespace in input buffer }
function TApxCustomComPort.GetInBuffFree : Word;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.InBuffFree
  else
    Result := DeviceParams.InSize;
end;

{ Return the number of used bytes in the input buffer }
function TApxCustomComPort.GetInBuffUsed : Word;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.InBuffUsed
  else
    Result := 0;
end;

{ Return the desired size of the input buffer }
function TApxCustomComPort.GetInSize : Word;
begin
  Result := DeviceParams.InSize;
end;

{ Return most severe current line error }
function TApxCustomComPort.GetLineError : Word;
var
  ErrorCode : Integer;
begin
  if (PortState = psOpen) then begin
    ErrorCode := Dispatcher.GetLineError;
    if (ErrorCode < ecOk) and
        not (csLoading in Self.ComponentState) then
      raise EPortError.Create(ErrorCode, False)
    else
      Result := Word(ErrorCode);
  end else
    Result := leNoError;
end;

{ Return True if break received }
function TApxCustomComPort.GetLineBreak : Boolean;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.CheckLineBreak
  else
    Result := False;
end;

{ Return amount of free space in output buffer }
function TApxCustomComPort.GetOutBuffFree : Word;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.OutBuffFree
  else
    Result := DeviceParams.OutSize;
end;

{ Return number of used bytes in output buffer }
function TApxCustomComPort.GetOutBuffUsed : Word;
begin
  if (PortState = psOpen) then
    Result := Dispatcher.OutBuffUsed
  else
    Result := 0;
end;

{ Return the desired size of the output buffer }
function TApxCustomComPort.GetOutSize : Word;
begin
  Result := DeviceParams.OutSize;
end;

{ Change the UART base address (supports RS485Mode) }
procedure TApxCustomComPort.SetBaseAddress(const Value : Word);
begin
  DeviceParams.BaseAddress := Value;
end;

{ Return the parity }
function TApxCustomComport.GetParity : TAxParity;
begin
  Result := DeviceParams.Parity;
end;

{ Return the state of RS485Mode }
function TApxCustomComport.GetRS485Mode : Boolean;
begin
  Result := DeviceParams.RS485Mode;
end;

{ Return the value of RTS }
function TApxCustomComport.GetRTS : Boolean;
begin
  Result := DeviceParams.RTS;
end;

{ Return the Stopbits setting }
function TApxCustomComPort.GetStopbits : TAxStopbits;
begin
  Result := DeviceParams.Stopbits;
end;

{ Return the SWFlowControl setting }
function TApxCustomComPort.GetSWFlowControl : TAxSWFlowControl;
begin
  Result := DeviceParams.SWFlowControl;
end;

{ Return the Xoff char setting }
function TApxCustomComPort.GetXOffChar : AnsiChar;
begin
  Result := DeviceParams.XOffChar;
end;

{ Return the XOn char setting }
function TApxCustomComPort.GetXOnChar : AnsiChar;
begin
  Result := DeviceParams.XOnChar;
end;

{ Set the UseEventWord option }
procedure TApxCustomComPort.SetUseEventWord(const Value : Boolean);
begin
  DeviceParams.UseEventWord := Value;
end;

{ Set the line parity }
procedure TApxCustomComPort.SetParity(const Value : TAxParity);
begin
  DeviceParams.Parity := Value;
end;

{ Set the RS485 mode }
procedure TApxCustomComPort.SetRS485Mode(const Value : Boolean);
begin
  DeviceParams.BeginUpdate;
  try
    DeviceParams.RS485Mode := Value;
    if Value then begin
      { Force rts/cts flow control off }
      DeviceParams.HWFlowControl := False;
      { Force RTS off }
      DeviceParams.RTS := False;
    end;
  finally
    DeviceParams.EndUpdate;
  end;
end;

{ Change the line RTS setting }
procedure TApxCustomComPort.SetRTS(const Value : Boolean);
begin
  DeviceParams.RTS := Value;
end;

{ Change the line stop bit setting }
procedure TApxCustomComPort.SetStopbits(const Value : TAxStopbits);
begin
  DeviceParams.Stopbits := Value;
end;

{ Get the base address }
function TApxCustomComPort.GetBaseAddress : Word;
begin
  Result := DeviceParams.BaseAddress;
end;

{ TApxComPort protected }

procedure TApxCustomComPort.CreateWidget;
begin
  FHandle := QObject_create(nil, nil);
end;

{ Physically open the port if FOpen is True }
procedure TApxCustomComPort.Loaded;
begin
  inherited Loaded;

  if not (csDesigning in ComponentState) then begin
    if ForceOpen then
      FOpen := True;
    if FOpen then begin
      ForceOpen := False;
      try
        InitPort;
      except
        FOpen := False;
        Application.HandleException(nil);
      end;
    end;
  end;
end;

{ Set the thread boost }
procedure TApxCustomComPort.SetThreadBoost(const Value : TAxThreadBoost);
begin
  if (FThreadBoost <> Value) then begin
    FThreadBoost := Value;
    if (PortState = psOpen) then
      Dispatcher.SetThreadBoost(Value);
  end;
end;

{ For internal processing of all triggers }
procedure TApxCustomComPort.Trigger(Msg, TriggerHandle, Data : Word);
begin
  if Assigned(FOnTrigger) then
    FOnTrigger(Self, Msg, TriggerHandle, Data);
end;

{ For internal triggeravail processing }
procedure TApxCustomComPort.TriggerAvail(Count : Word);
begin
  if Assigned(FOnTriggerAvail) then
    FOnTriggerAvail(Self, Count);
end;

{ For internal triggerdata processing }
procedure TApxCustomComPort.TriggerData(TriggerHandle : Word);
begin
  if Assigned(FOnTriggerData) then
    FOnTriggerData(Self, TriggerHandle);
end;

{ For internal triggerstatus processing }
procedure TApxCustomComPort.TriggerStatus(TriggerHandle : Word);
begin
  if Assigned(FOnTriggerStatus) then
    FOnTriggerStatus(Self, TriggerHandle);
end;

{ For internal triggertimer processing }
procedure TApxCustomComPort.TriggerTimer(TriggerHandle : Word);
begin
  if Assigned(FOnTriggerTimer) then
    FOnTriggerTimer(Self, TriggerHandle);
end;

procedure TApxCustomComPort.UpdateHandlerFlag;
begin
  if (PortState <> psOpen) then Exit;
  if Assigned(FOnTrigger) or Assigned(FOnTriggerAvail) or
    Assigned(FOnTriggerData) or Assigned(FOnTriggerStatus) or
    Assigned(FOnTriggerTimer) or Assigned(FOnTriggerLineError) or
    Assigned(FOnTriggerModemStatus) or Assigned(FOnTriggerOutbuffFree) or
    Assigned(FOnTriggerOutbuffUsed) or Assigned(FOnTriggerOutSent) then
    FDispatcher.UpdateHandlerFlags(fuEnablePort)
  else
    FDispatcher.UpdateHandlerFlags(fuDisablePort);
end;

{ Port open processing }
procedure TApxCustomComPort.PortOpen;
var
  I : Word;
  UL : PUserListEntry;
begin
  { Tell all comport users that the port is now open }
  if UserList.Count > 0 then begin
    for I := UserList.Count-1 downto 0 do begin
      UL := UserList.Items[I];
      with UL^ do begin
        if Handle <> nil then
          AxSendMessage(Handle, APX_PORTOPEN, 0, 0)
        else
          UL^.OpenClose(Self, True);
      end;
    end;
  end;

  if Assigned(FOnPortOpen) then
    FOnPortOpen(Self);
end;

{ Port close processing }
procedure TApxCustomComPort.PortClose;
var
  I : Word;
  UL : PUserListEntry;
begin
  { Tell all COM port users that the port is now closed }
  if UserList.Count > 0 then begin
    for I := UserList.Count-1 downto 0 do begin
      UL := UserList.Items[I];
      with UL^ do begin
        if Handle <> nil then
          AxSendMessage(Handle, APX_PORTCLOSE, 0, 0)
        else
          UL^.OpenClose(Self, False);
      end;
    end;
  end;

  if Assigned(FOnPortClose) then
    FOnPortClose(Self);
end;

{ Received a line error }
procedure TApxCustomComPort.TriggerLineError(const Error : Word;
  const LineBreak : Boolean);
begin
  if Assigned(FOnTriggerLineError) then
    FOnTriggerLineError(Self, Error, LineBreak);
end;

{ Received a modem status change }
procedure TApxCustomComPort.TriggerModemStatus;
begin
  if Assigned(FOnTriggerModemStatus) then
    FOnTriggerModemStatus(Self);
end;

{ Received and outbuff free trigger }
procedure TApxCustomComPort.TriggerOutbuffFree;
begin
  if Assigned(FOnTriggerOutbuffFree) then
    FOnTriggerOutbuffFree(Self);
end;

{ Received and outbuff used trigger }
procedure TApxCustomComPort.TriggerOutbuffUsed;
begin
  if Assigned(FOnTriggerOutbuffUsed) then
    FOnTriggerOutbuffUsed(Self);
end;

{ Received an outsent trigger }
procedure TApxCustomComPort.TriggerOutSent;
begin
  if Assigned(FOnTriggerOutSent) then
    FOnTriggerOutSent(Self);
end;

{ Setup / destroy Qt event mechanism for triggers }
procedure TApxCustomComPort.RegisterComPort(Enabling : Boolean);
begin
  if Enabling then begin
    { Make sure event handler is created/registered }
    HandleNeeded;
    { Register it }
    FDispatcher.RegisterQtTriggerHandler(FHandle);
  end else begin
    { Deregister it }
    FDispatcher.DeregisterQtTriggerHandler(FHandle);
    { Destroy Handle and Hook }
    DestroyWidget;
  end;
end;

{ Validate comport, show device selection dialog if necessary }
procedure TApxCustomComPort.ValidateComport;
var
  ComSelDlg : TComSelectForm;
begin
  if (FComNumber = 0) and (DeviceName = '') then
    if (not FPromptForPort) then
      raise ENoPortSelected.Create(ecNoPortSelected, False)
    else begin
      ComSelDlg := TComSelectForm.Create(Application);
      try
        if (ComSelDlg.ShowModal = mrOk) then begin
          { Set DeviceName appropriately }
          DeviceName := ComSelDlg.SelectedDevice;
          { Set ComNumber if possible }
          ComNumber := AxExtractComNumber(DeviceName);
        end else
          raise ENoPortSelected.Create(ecNoPortSelected, False);
      finally
        ComSelDlg.Free;
      end;
    end;
end;

{ Fill DevList with list of device layers }
procedure TApxCustomComPort.GetDeviceLayerList(const DevList : TStringList);
begin
  DevList.Assign(GetDeviceLayers);
end;

{ Physically open the COM port }
procedure TApxCustomComPort.InitPort;
var
  I, Res : Integer;
begin
  { Validate the comport }
  ValidateComport;

  { Create the dispatcher }
  if GetDeviceLayers.Find(FDeviceLayer, I) then
    FDispatcher :=
      TApxDeviceLayerInfo(DeviceLayers.Objects[I]).DispatcherClass.Create(Self)
  else
    raise Exception.Create('Unknown Dispatcher Class');

  try
    Res := Dispatcher.InitPort(FDeviceName);

    { Remap access denied and file not found errors }
    if (Res = ecFileNotFound) or (Res = ecPathNotFound) then
      Res := ecBadId;

    if (Res = ecOk) then begin
      { Handle preset properties }
      PortState := psOpen;
      UpdateHandlerFlag;
      SetUseEventWord(FUseEventWord);
      SetThreadBoost(FThreadBoost);
      FOpen := True;

      { Prepare for triggers }
      RegisterComPort(True);

      { Add pending triggers }
      if CopyTriggers then begin
        CopyTriggers := False;
        FDispatcher.RestoreTriggers(SaveTriggerBuffer);
      end;

      { Send OnPortEvent }
      PortOpen;
    end else if (Res < ecOk) and not (csLoading in Self.ComponentState) then
      raise EPortError.Create (Res, False)
  except
    FOpen := False;
    PortState := psClosed;
    FDispatcher.Free;
    FDispatcher := nil;
    raise;
  end;
end;

{ Physically close the COM port }
procedure TApxCustomComPort.DonePort;
begin
  FOpen := False;
  if (PortState = psOpen) then begin

    { Port is shutting down }
    PortState := psShuttingDown;

    { Send OnPortClose event }
    PortClose;

    { Save triggers in case this port is reopened }
    Dispatcher.SaveTriggers(SaveTriggerBuffer);
    CopyTriggers := True;

    { Close the port and clear ComTable }
    Dispatcher.DonePort;

    if Dispatcher.EventBusy then begin
      AxPostMessage(Handle, APX_CLOSEPENDING, nil, 0, 0);
      SafeYield;
    end else begin
      { Get rid of the trigger handler }
      RegisterComPort(False);
      FDispatcher.Free;
      FDispatcher := nil;
      PortState := psClosed;
    end;
  end;
end;

{ Assign values of Source to self }
procedure TApxCustomComPort.Assign(Source: TPersistent);
var
  SourcePort : TApxCustomComPort absolute Source;
  I : Word;
  UL : PUserListEntry;
begin
  if Source is TApxCustomComPort then begin
    { Discard existing userlist }
    if UserList.Count > 0 then
      for I := UserList.Count-1 downto 0 do begin
        UL := UserList.Items[I];
        UserList.Remove(UL);
        Dispose(UL);
      end;
    UserList.Free;

    { Copy Source's userlist }
    UserList := TList.Create;
    if SourcePort.UserList.Count > 0 then
      for I := 0 to SourcePort.UserList.Count-1 do begin
        New(UL);
        Move(SourcePort.UserList.Items[I]^, UL^,
             SizeOf(TUserListEntry));
        UserList.Add(UL);
      end;

    { Copy triggers from Source }
    if (SourcePort.PortState = psOpen) then begin
      SourcePort.Dispatcher.SaveTriggers(SaveTriggerBuffer);
      CopyTriggers := True;
    end;

    { Copy all other fields }
    FOpen            := False;
    FDeviceLayer     := SourcePort.FDeviceLayer;

    Baud             := SourcePort.Baud;
    Parity           := SourcePort.Parity;
    Databits         := SourcePort.Databits;
    Stopbits         := SourcePort.Stopbits;
    FAutoOpen        := SourcePort.FAutoOpen;
    FPromptForPort   := SourcePort.FPromptForPort;
    RS485Mode        := SourcePort.RS485Mode;
    FThreadBoost     := SourcePort.FThreadBoost;

    BufferFull       := SourcePort.BufferFull;
    BufferResume     := SourcePort.BufferResume;
    DTR              := SourcePort.DTR;
    RTS              := SourcePort.RTS;
    InSize           := SourcePort.InSize;
    OutSize          := SourcePort.OutSize;
    HWFlowControl    := SourcePort.HWFlowControl;
    SWFlowControl    := SourcePort.SWFlowControl;
    XOffChar         := SourcePort.XOffChar;
    XOnChar          := SourcePort.XOnChar;
    BaseAddress      := SourcePort.BaseAddress;
    UseEventWord     := SourcePort.UseEventWord;


    { Must go through write method to ensure flag gets updated }
    OnTrigger        := SourcePort.FOnTrigger;
    OnTriggerAvail   := SourcePort.FOnTriggerAvail;
    OnTriggerData    := SourcePort.FOnTriggerData;
    OnTriggerStatus  := SourcePort.FOnTriggerStatus;
    OnTriggerTimer   := SourcePort.FOnTriggerTimer;
    FOnPortOpen      := SourcePort.FOnPortOpen;
    FOnPortClose     := SourcePort.FOnPortClose;
  end;
end;

{ Register a user of this COM port }
procedure TApxCustomComPort.RegisterUser(const H : QObjectH);
var
  UL : PUserListEntry;
begin
  New(UL);
  with UL^ do begin
    Handle := H;
    OpenClose := nil;
  end;
  UserList.Add(UL);
end;

{ Register a user of this COM port }
procedure TApxCustomComPort.RegisterUserCallback(CallBack : TPortCallback);
var
  UL : PUserListEntry;
begin
  New(UL);
  with UL^ do begin
    Handle := nil;
    OpenClose := Callback;
  end;
  UserList.Add(UL);
end;

{ Deregister a user of this COM port }
procedure TApxCustomComPort.DeregisterUser(const H : QObjectH);
var
  UL : PUserListEntry;
  I : Word;
begin
  if UserList.Count > 0 then begin
    for I := UserList.Count-1 downto 0 do begin
      UL := UserList.Items[I];
      with UL^ do begin
        if Handle = H then begin
          UserList.Remove(UL);
          Dispose(UL);
        end;
      end;
    end;
  end;
end;

{ Deregister a user of this COM port }
procedure TApxCustomComPort.DeregisterUserCallback(CallBack : TPortCallback);
var
  UL : PUserListEntry;
  I : Word;
begin
  if UserList.Count > 0 then begin
    for I := UserList.Count-1 downto 0 do begin
      UL := UserList.Items[I];
      with UL^ do begin
        if @CallBack = @OpenClose then begin
          UserList.Remove(UL);
          Dispose(UL);
        end;
      end;
    end;
  end;
end;

{ Process communications receive events, but not triggers }
procedure TApxCustomComPort.ProcessCommunications;
var
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then
    Exit;
  ErrorCode := ValidDispatcher.ProcessCommunications;
  if (ErrorCode < ecOk) and
      not (csLoading in Self.ComponentState) then
    raise EPortError.Create (ErrorCode, False)
end;

{ Flush the input buffer }
procedure TApxCustomComPort.FlushInBuffer;
var
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then
    Exit;
  ErrorCode := ValidDispatcher.FlushBuffer(cbInput);
  if (ErrorCode < ecOk) and
      not (csLoading in Self.ComponentState) then
    raise EPortError.Create (ErrorCode, False)
end;

{ Flush the output buffer }
procedure TApxCustomComPort.FlushOutBuffer;
var
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then
    Exit;
  ErrorCode := ValidDispatcher.FlushBuffer(cbOutput);
  if (ErrorCode < ecOk) and
      not (csLoading in Self.ComponentState) then
    raise EPortError.Create (ErrorCode, False)
end;

{ Adds a string to the debug log }
procedure TApxCustomComPort.AddStringToLog(S : string);
begin
  if (PortState = psShuttingDown) then Exit;
  if Assigned(FDebugLog) then FDebugLog.WriteDebugString(S);
end;

{ Ensure port is opened after loading }
procedure TApxCustomComPort.ForcePortOpen;
begin
  if AutoOpen then
    ForceOpen := True;
end;

{ Send a line break of mSecs duration }
procedure TApxCustomComPort.SendBreak(mSecs : DWORD);
begin
  if (PortState = psShuttingDown) then Exit;
  ValidDispatcher.SendComBreak(mSecs);
end;

{ Add a ShortString data trigger }
function TApxCustomComPort.AddDataTrigger(const Data : ShortString;
  const IgnoreCase : Boolean) : Word;
var
  Len : Word;
  P : array[0..255] of AnsiChar;
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then begin
    Result := 0;
    Exit;
  end;
  Len := Length(Data);
  Move(Data[1], P, Len);

  ErrorCode := ValidDispatcher.AddDataTriggerLen(P, IgnoreCase, Len);
  if (ErrorCode < ecOk) and
      not (csLoading in Self.ComponentState) then
    raise EPortError.Create (ErrorCode, False);
  Result := Word (ErrorCode);
end;

{ Add a timer trigger }
function TApxCustomComPort.AddTimerTrigger : Word;
var
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then
    Result := 0
  else begin
    ErrorCode := ValidDispatcher.AddTimerTrigger;
    if (ErrorCode < ecOk) and
        not (csLoading in Self.ComponentState) then
      raise EPortError.Create (ErrorCode, False);
    Result := Word (ErrorCode);
  end;
end;

{ Add a status trigger of type SType }
function TApxCustomComPort.AddStatusTrigger(const SType : Word) : Word;
var
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then
    Result := 0
  else begin
    ErrorCode := ValidDispatcher.AddStatusTrigger(SType);
    if (ErrorCode < ecOk) and
        not (csLoading in Self.ComponentState) then
      raise EPortError.Create (ErrorCode, False);
    Result := Word (ErrorCode);
  end;
end;

{ Remove trigger with index Index }
procedure TApxCustomComPort.RemoveTrigger(const Handle : Word);
var
  ErrorCode : Integer;
begin
  if (PortState = psOpen) then begin
    ErrorCode := Dispatcher.RemoveTrigger(Handle);
    if (ErrorCode < ecOk) and
        not (csLoading in Self.ComponentState) then
      raise EPortError.Create (ErrorCode, False)
  end;
end;

{ Remove all triggers }
procedure TApxCustomComPort.RemoveAllTriggers;
begin
  if (PortState = psOpen) then begin
    Dispatcher.RemoveAllTriggers;
  end;
end;

{ Set the timer for trigger Index }
procedure TApxCustomComPort.SetTimerTrigger(const Handle : Word;
  const mSecs : DWORD; const Activate : Boolean);
var
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then
    Exit;

  ErrorCode := ValidDispatcher.SetTimerTrigger(Handle, mSecs, Activate);
  if (ErrorCode < ecOk) and
      not (csLoading in Self.ComponentState) then
    raise EPortError.Create (ErrorCode, False)
end;

{ Set status trigger }
procedure TApxCustomComPort.SetStatusTrigger(const Handle : Word;
  const Value : Word; const Activate : Boolean);
var
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then
    Exit;
  ErrorCode := ValidDispatcher.SetStatusTrigger(Handle, Value, Activate);
  if (ErrorCode < ecOk) and
      not (csLoading in Self.ComponentState) then
    raise EPortError.Create (ErrorCode, False)
end;

{I/O}

{ Return the next character in the receive buffer }
function TApxCustomComPort.CharReady : Boolean;
begin
  if (PortState = psShuttingDown) then
    Result := False
  else
    Result := ValidDispatcher.CharReady;
end;

{ Peek at the Count'th character in the buffer (1=next) }
function TApxCustomComPort.PeekChar(const Count : Word) : AnsiChar;
var
  Res : Integer;
  C   : AnsiChar;
begin
  if (PortState = psShuttingDown) then begin
    Res := ecOk;
    C := #0;
  end else
    Res := ValidDispatcher.PeekChar(C, Count);
  if Res = ecOK then
    Result := C
  else begin
    if (Res < ecOk) and
        not (csLoading in Self.ComponentState) then
      raise EPortError.Create (Res, False);
    Result := #0;
  end;
end;

{ Retrieve the next character from the input queue }
function TApxCustomComPort.GetChar : AnsiChar;
var
  Res : Integer;
  C   : AnsiChar;
begin
  if (PortState = psShuttingDown) then begin
    Res := ecOk;
    C := #0;
  end else
    Res := ValidDispatcher.GetChar(C);
  if Res = ecOK then
    Result := C
  else begin
    if (Res < ecOk) and
        not (csLoading in Self.ComponentState) then
      raise EPortError.Create (Res, False);
    Result := #0;
  end;
end;

{ Peek at the next Len characters, but don't remove from buffer }
procedure TApxCustomComPort.PeekBlock(var Block; const Len : Word);
var
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then
    Exit;
  ErrorCode := ValidDispatcher.PeekBlock(PAnsiChar(@Block), Len);
  if (ErrorCode < ecOk) and
      not (csLoading in Self.ComponentState) then
    raise EPortError.Create (ErrorCode, False);
end;

{ Return the next Len characters from the buffer }
procedure TApxCustomComPort.GetBlock(var Block; const Len : Word);
var
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then
    Exit;
  ErrorCode := ValidDispatcher.GetBlock(PAnsiChar(@Block), Len);
  if (ErrorCode < ecOk) and
      not (csLoading in Self.ComponentState) then
    raise EPortError.Create (ErrorCode, False);
end;

{ Add C to the output buffer }
procedure TApxCustomComPort.PutChar(const C : AnsiChar);
var
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then Exit;
  ErrorCode := ValidDispatcher.PutChar(C);
  if (ErrorCode < ecOk) and
      not (csLoading in Self.ComponentState) then
    raise EPortError.Create (ErrorCode, False);
end;

{ Add S to the output buffer }
procedure TApxCustomComPort.PutString(const S : string);
var
  ErrorCode : Integer;
begin
  if (PortState = psShuttingDown) then Exit;

  ErrorCode := ValidDispatcher.PutString(S);
  if (ErrorCode < ecOk) and
      not (csLoading in Self.ComponentState) then
    raise EPortError.Create (ErrorCode, False);
end;

{ Add Block to the output buffer }
function TApxCustomComPort.PutBlock(const Block; const Len : Word) : Integer;
var
  ErrorCode : Integer;
begin
  PutBlock := 0;
  if (PortState = psShuttingDown) then Exit;

  ErrorCode := ValidDispatcher.PutBlock(PAnsiChar(Block), Len);
  if (ErrorCode < ecOk) and
      not (csLoading in Self.ComponentState) then
    raise EPortError.Create (ErrorCode, False);
end;

{ Miscellaneous procedures }

{ Search for a comport in the same form as TComponent }
function SearchComPort(const C : TComponent) : TApxCustomComPort;

  function FindComPort(const C : TComponent) : TApxCustomComPort;
  var
    I  : Integer;
  begin
    Result := nil;
    if not Assigned(C) then
      Exit;

    { Look through all of the owned components }
    for I := 0 to C.ComponentCount-1 do begin
      if C.Components[I] is TApxCustomComPort then begin
        Result := TApxCustomComPort(C.Components[I]);
        Exit;
      end;

      { If this isn't one, see if it owns other components }
      Result := FindComPort(C.Components[I]);
    end;
  end;

begin
  { Search the entire form }
  Result := FindComPort(C);
end;

initialization
  dllCS := TCriticalSection.Create;

finalization
  dllCS.Free;

end.
