(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                  AxProtcl.pas 1.02                    *}
{*********************************************************}

{Global defines potentially affecting this unit}
{$I AxDefine.inc}

{Options required for this unit}
{$I+,G+,X+,F+,V-,Q-,J+}

{$C MOVEABLE,DEMANDLOAD,DISCARDABLE}

{
  Design note:

  To maintain the same user interface as previous version of the TAxxProtocol
  component, this component will create and destroy the TApxxxxxDriver
  (ProtocolDriver) based on the current protocol type.  To set protocol
  specific properties, the ProtocolDriver will be verified that it is of the
  correct type to contain those properties (in addition to verifying the
  CurrentProtocol).
}
unit AxProtcl;
  {-File transfer protocol VCL component}

interface

uses
  Types,
  SysUtils,
  Classes,
  QControls,
  QForms,
  QTypes,
  Qt,
  AxMisc,
  AxPrtDrv,
  AxXModem,
  AxYModem,
  AxZModem,
  AxKermit,       
  AxExcept,
{$IFDEF TRIALRUN}
  {$I TRIAL08.INC}
{$ENDIF}
  AxPort,
  AxString,
  LibC;

type
  {Block check methods - NOTE! this must match OOMISC}
  TApxBlockCheckMethod = (
    bcmNone,
    bcmChecksum,
    bcmChecksum2,
    bcmCrc16,
    bcmCrc32,
    bcmCrcK);

  {Protocol types - NOTE! this must match OOMISC}
  TApxProtocolType = (
    ptNoProtocol,
    ptXmodem,
    ptXmodemCRC,
    ptXmodem1K,
    ptXmodem1KG,
    ptYmodem,
    ptYmodemG,
    ptZmodem,
    ptKermit,
    ptAscii);

  {Zmodem file management options - NOTE! this must match OOMISC}
  TApxZmodemFileOptions = (
    zfoNoOption,           {Place holder}
    zfoWriteNewerLonger,   {Transfer if new, newer or longer}
    zfoWriteCrc,           {Not supported, same as WriteNewer}
    zfoWriteAppend,        {Transfer if new, append if exists}
    zfoWriteClobber,       {Transfer regardless}
    zfoWriteNewer,         {Transfer if new or newer}
    zfoWriteDifferent,     {Transfer if new or diff dates/lens}
    zfoWriteProtect);      {Transfer only if new}

  {Action to take if incoming file exists - NOTE! this must match OOMISC}
  TApxWriteFailAction = (
    wfWriteNone,         {No option set yet}
    wfWriteFail,         {Fail the open attempt}
    wfWriteRename,       {Rename the incoming file}
    wfWriteAnyway,       {Overwrite the existing file}
    wfWriteResume);      {Resume an interrupted receive}

  {ASCII end-of-line translations}
  TApxAsciiEOLTranslation = (
    aetNone,             {No CR/LF translations}
    aetStrip,            {Strip CRs or LFs}
    aetAddCRBefore,      {Add CR before each LF}
    aetAddLFAfter);      {Add LF after each CR}

  {DeleteFailed options for TApxProtocolLog}
  TApxDeleteFailed = (dfNever, dfAlways, dfNonRecoverable);

const
  {Defaults for TApxProtocol properties}
  ApxDefProtocolType         = ptZmodem;
  ApxDefXYmodemBlockWait     = 5;
  ApxDefZmodemOptionOverride = False;
  ApxDefZmodemSkipNoFile     = False;
  ApxDefZmodemFileOption     = zfoWriteNewer;
  ApxDefZmodemRecover        = False;
  ApxDefZmodem8K             = False;
  ApxDefZmodemZRQINITValue   = 0;
  ApxDefZmodemFinishRetry    = 0;
  ApxDefKermitMaxLen         = 80;
  ApxDefKermitMaxWindows     = 0;
  ApxDefKermitSWCTurnDelay   = 0;
  ApxDefKermitTimeoutSecs    = 5;
  ApxDefKermitPadCharacter   = ' ';
  ApxDefKermitPadCount       = 0;
  ApxDefKermitTerminator     = #13;
  ApxDefKermitCtlPrefix      = '#';
  ApxDefKermitHighbitPrefix  = 'Y';
  ApxDefKermitRepeatPrefix   = '~';
  ApxDefAsciiCharDelay       = 0;
  ApxDefAsciiLineDelay       = 0;
  ApxDefAsciiEOLChar         = #13;
  ApxDefAsciiCRTranslation   = aetNone;
  ApxDefAsciiLFTranslation   = aetNone;
  ApxDefAsciiEOFTimeout      = 20000;      {20 seconds}
  ApxDefHonorDirectory       = False;
  ApxDefIncludeDirectory     = False;
  ApxDefRTSLowForWrite       = False;
  ApxDefAbortNoCarrier       = False;
  ApxDefAsciiSuppressCtrlZ   = False;
  ApxDefFinishWait           = 20000;
  ApxDefTurnDelay            = 0;
  ApxDefOverhead             = 0;
  ApxDefWriteFailAction      = wfWriteRename;
  ApxDefStatusInterval       = 1000;
  ApxDefUpcaseFileNames      = True;

  {Defaults for TApxProtocolLog properties}
  awpDefHistoryName       = 'APRO.HIS';
  awpDefDeleteFailed      = dfNonRecoverable;

  MaxKermitLongLen     = 1024;
  MaxKermitWindows     = 27;

  {Status options}
  apFirstCall          = AxMisc.apFirstCall;
  apLastCall           = AxMisc.apLastCall;

  {General protocol status constants}
  psOK                 = AxMisc.psOK;
  psProtocolHandshake  = AxMisc.psProtocolHandshake;
  psInvalidDate        = AxMisc.psInvalidDate;
  psFileRejected       = AxMisc.psFileRejected;
  psFileRenamed        = AxMisc.psFileRenamed;
  psSkipFile           = AxMisc.psSkipFile;
  psFileDoesntExist    = AxMisc.psFileDoesntExist;
  psCantWriteFile      = AxMisc.psCantWriteFile;
  psTimeout            = AxMisc.psTimeout;
  psBlockCheckError    = AxMisc.psBlockCheckError;
  psLongPacket         = AxMisc.psLongPacket;
  psDuplicateBlock     = AxMisc.psDuplicateBlock;
  psProtocolError      = AxMisc.psProtocolError;
  psCancelRequested    = AxMisc.psCancelRequested;
  psEndFile            = AxMisc.psEndFile;
  psResumeBad          = AxMisc.psResumeBad;
  psSequenceError      = AxMisc.psSequenceError;
  psAbortNoCarrier     = AxMisc.psAbortNoCarrier;

  {Specific to certain protocols}
  psGotCrcE            = AxMisc.psGotCrcE;
  psGotCrcG            = AxMisc.psGotCrcG;
  psGotCrcW            = AxMisc.psGotCrcW;
  psGotCrcQ            = AxMisc.psGotCrcQ;
  psTryResume          = AxMisc.psTryResume;
  psHostResume         = AxMisc.psHostResume;
  psWaitAck            = AxMisc.psWaitAck;

  {For specifying log file calls}
  lfReceiveStart       = AxMisc.lfReceiveStart;
  lfReceiveOk          = AxMisc.lfReceiveOk;
  lfReceiveFail        = AxMisc.lfReceiveFail;
  lfReceiveSkip        = AxMisc.lfReceiveSkip;
  lfTransmitStart      = AxMisc.lfTransmitStart;
  lfTransmitOk         = AxMisc.lfTransmitOk;
  lfTransmitFail       = AxMisc.lfTransmitFail;
  lfTransmitSkip       = AxMisc.lfTransmitSkip;

type
  TApxAbstractStatus = class;
  TApxProtocolLog = class;

  {Protocol event handlers}
  TProtocolErrorEvent    = procedure(CP : TObject; ErrorCode : Integer)
                           of object;
  TProtocolFinishEvent   = procedure(CP : TObject; ErrorCode : Integer)
                           of object;
  TProtocolLogEvent      = procedure(CP : TObject; Log : Word)
                           of object;
  TProtocolResumeEvent   = procedure(CP : TObject;
                                     var Resume : TApxWriteFailAction)
                           of object;
  TProtocolStatusEvent   = procedure(CP : TObject; Options : Word)
                           of object;

  TProtocolNextFileEvent = procedure(CP : TObject; var FName : string) of object;
  TProtocolAcceptEvent   = procedure(CP : TObject; var Accept : Boolean;
    var FName : String) of object;

  TProtocolStreamOpenEvent = procedure (    CP      : TObject;
                                        var Stream  : TStream;
                                            Reading : Boolean) of object;
  TProtocolStreamCloseEvent = procedure (    CP     : TObject;
                                         var Stream : TStream) of object;

  ApxHWND = integer; 

  {Protocol component}
  TApxCustomProtocol = class(TApxBaseHandleComponent)
  protected {private}
    {.Z+}
    {Private data}
    NeedBPS           : Boolean;           {True if we don't know BPS yet}
    Force             : Boolean;           {True to force setting options}
    FTransmitting     : Boolean;           {Are we transmitting?}

    {Property data}
    FMsgHandler       : ApxHWND;           {Window handler of MessageHandler}
    FComPort          : TApxCustomComPort; {Comport to use}
    FProtocolType     : TApxProtocolType;  {Current protocol}
    FStatusDisplay    : TApxAbstractStatus;{Built-in status display}
    FProtocolLog      : TApxProtocolLog;   {Built-in protocol logging function}
    FXYmodemBlockWait : Cardinal;          {Inter-block delay }
    FZmodemOptionOverride : Boolean;       {True to override transmitter options}
    FZmodemSkipNoFile : Boolean;           {True to skip new incoming files}
    FZmodemFileOption : TApxZmodemFileOptions;{File mgmt options}
    FZmodemRecover    : Boolean;           {True to enable crash recovery}
    FZmodem8K         : Boolean;           {True to enable 8K zmodem}
    FZmodemZRQINITValue : LongInt;         {Optional ZRQINIT data}
    FZmodemFinishRetry : Cardinal;         {Number of ZFin retries}
    FKermitMaxLen     : Cardinal;          {Max normal packet len}
    FKermitMaxWindows : Cardinal;          {Maximum number of windows}
    FKermitSWCTurnDelay : Cardinal;        {Turn delay when SWC in use}
    FKermitTimeoutSecs : Cardinal;         {Packet timeout in seconds}
    FKermitPadCharacter : Char;            {Pad character}
    FKermitPadCount   : Cardinal;          {Padding count}
    FKermitTerminator : Char;              {Packet terminator character (ASCII)}
    FKermitCtlPrefix  : Char;              {Control char prefix (ASCII value)}
    FKermitHighbitPrefix : Char;           {Hibit prefix (ASCII value)}
    FKermitRepeatPrefix : Char;            {Repeat prefix (ASCII value)}
    FAsciiCharDelay   : Cardinal;          {Inter-character delay}
    FAsciiLineDelay   : Cardinal;          {Inter-line delay}
    FAsciiEOLChar     : Char;              {End-of-line character (ASCII value)}
    FAsciiCRTranslation : TApxAsciiEOLTranslation; {ASCII translate CR}
    FAsciiLFTranslation : TApxAsciiEOLTranslation; {ASCII translate LF}
    FAsciiEOFTimeout  : Cardinal;               {time to assume EOF}

    {Events}
    FOnProtocolAccept   : TProtocolAcceptEvent;
    FOnProtocolError    : TProtocolErrorEvent;
    FOnProtocolFinish   : TProtocolFinishEvent;
    FOnProtocolLog      : TProtocolLogEvent;
    FOnProtocolNextFile : TProtocolNextFileEvent;
    FOnProtocolResume   : TProtocolResumeEvent;
    FOnProtocolStatus   : TProtocolStatusEvent;
    FOnProtocolStreamOpen  : TProtocolStreamOpenEvent;
    FOnProtocolStreamClose : TProtocolStreamCloseEvent;

    procedure CreateWidget; override;
    procedure CheckPort;

  protected
    {Misc protected methods}
    procedure HandleAPXProtocolFinish (var Message : TAxMessage); message APX_PROTOCOLFINISH;
    function EventFilter(Sender: QObjectH; Event: QEventH): Boolean; override;
    procedure Notification(AComponent : TComponent;
                           Operation: TOperation); override;
    procedure TransmitPortMessageHandler (Msg, wParam : Cardinal;
                                          lParam : LongInt);

    procedure ReceivePortMessageHandler (Msg, wParam : Cardinal;
                                        lParam : LongInt);
    procedure RegisterComHook (Transmitting : Boolean);
    procedure DeRegisterComHook;

    {Property methods}
    procedure SetComPort(const NewComPort : TApxCustomComPort);
    procedure SetProtocolType(const NewProtocol : TApxProtocolType);
    function GetDestinationDirectory : String;
    procedure SetDestinationDirectory(const NewDir : String);
    function GetFileMask : TFileName;
    procedure SetFileMask(const NewFileMask : TFileName);
    function GetBatch : Boolean;
    function GetBlockCheckMethod : TApxBlockCheckMethod;
    procedure SetBlockCheckMethod(const NewMethod : TApxBlockCheckMethod);
    function GetHandshakeRetry : Cardinal;
    procedure SetHandshakeRetry(const NewRetry : Cardinal);
    function GetHandshakeWait : Cardinal;
    procedure SetHandshakeWait(const NewWait : Cardinal);
    function GetBlockLength : Cardinal;
    function GetBlockNumber : Cardinal;
    function GetTransmitTimeout : Cardinal;
    procedure SetTransmitTimeout(const NewTimeout : Cardinal);
    function GetFinishWait : Cardinal;
    procedure SetFinishWait(const NewWait : Cardinal);
    function GetActualBPS : LongInt;
    procedure SetActualBPS(const NewBPS : Longint);
    function GetTurnDelay : Cardinal;
    procedure SetTurnDelay(const NewDelay : Cardinal);
    function GetOverhead : Cardinal;
    procedure SetOverhead(const NewOverhead : Cardinal);
    function GetWriteFailAction : TApxWriteFailAction;
    procedure SetWriteFailAction(const NewAction : TApxWriteFailAction);
    function GetProtocolStatus : Cardinal;
    function GetProtocolError : SmallInt;
    function GetFileLength : LongInt;
    function GetFileDate : TDateTime;
    function GetInitialPosition : LongInt;
    function GetStatusInterval : Cardinal;
    procedure SetStatusInterval(const NewInterval : Cardinal);
    procedure SetStatusDisplay(const NewDisplay : TApxAbstractStatus);
    procedure SetProtocolLog(const NewLog : TApxProtocolLog);
    function GetInProgress : Boolean;
    function GetBlockErrors : Cardinal;
    function GetTotalErrors : Cardinal;
    function GetBytesRemaining : LongInt;
    function GetBytesTransferred : LongInt;
    function GetElapsedXfrTime : LongInt;
    function GetFileName : String;
    procedure SetFileName(const NewName : String);
    function GetXYmodemBlockWait : Cardinal;
    procedure SetXYmodemBlockWait(const NewWait : Cardinal);
    function GetZmodemOptionOverride : Boolean;
    procedure SetZmodemOptionOverride(const NewOverride : Boolean);
    function GetZmodemSkipNoFile : Boolean;
    procedure SetZmodemSkipNoFile(const NewSkip : Boolean);
    function GetZmodemFileOption : TApxZmodemFileOptions;
    procedure SetZmodemFileOption(const NewOpt : TApxZmodemFileOptions);
    function GetZmodemRecover : Boolean;
    procedure SetZmodemRecover(const NewRecover : Boolean);
    function GetZmodem8K : Boolean;
    procedure SetZmodem8K(const New8K : Boolean);
    function GetZmodemZRQINITValue : LongInt;
    procedure SetZmodemZRQINITValue(const NewZRQINITValue : LongInt);
    function GetZmodemFinishRetry : Cardinal;
    procedure SetZmodemFinishRetry(const NewRetry : Cardinal);
    function GetKermitMaxLen : Cardinal;
    procedure SetKermitMaxLen(const NewLen : Cardinal);
    function GetKermitMaxWindows : Cardinal;
    procedure SetKermitMaxWindows(const NewMax : Cardinal);
    function GetKermitSWCTurnDelay : Cardinal;
    procedure SetKermitSWCTurnDelay(const NewDelay : Cardinal);
    function GetKermitTimeoutSecs : Cardinal;
    procedure SetKermitTimeoutSecs(const NewTimeout : Cardinal);
    function GetKermitPadCharacter : Char;
    procedure SetKermitPadCharacter(NewChar : Char);
    function GetKermitPadCount : Cardinal;
    procedure SetKermitPadCount(NewCount : Cardinal);
    function GetKermitTerminator : Char;
    procedure SetKermitTerminator(const NewTerminator : Char);
    function GetKermitCtlPrefix : Char;
    procedure SetKermitCtlPrefix(const NewPrefix : Char);
    function GetKermitHighbitPrefix : Char;
    procedure SetKermitHighbitPrefix(const NewPrefix : Char);
    function GetKermitRepeatPrefix : Char;
    procedure SetKermitRepeatPrefix(const NewPrefix : Char);
    function GetKermitWindowsTotal : Cardinal;
    function GetKermitWindowsUsed : Cardinal;
    function GetKermitLongBlocks : Boolean;
    function GetAsciiCharDelay : Cardinal;
    procedure SetAsciiCharDelay(const NewDelay : Cardinal);
    function GetAsciiLineDelay : Cardinal;
    procedure SetAsciiLineDelay(const NewDelay : Cardinal);
    function GetAsciiEOLChar : Char;
    procedure SetAsciiEOLChar(const NewChar : Char);
    function GetAsciiCRTranslation : TApxAsciiEOLTranslation;
    procedure SetAsciiCRTranslation(const NewTrans : TApxAsciiEOLTranslation);
    function GetAsciiLFTranslation : TApxAsciiEOLTranslation;
    procedure SetAsciiLFTranslation(const NewTrans : TApxAsciiEOLTranslation);
    function GetAsciiEOFTimeout : Cardinal;
    procedure SetAsciiEOFTimeout(const NewTimeout : Cardinal);
    function GetHonorDirectory : Boolean;
    procedure SetHonorDirectory(const NewOpt : Boolean);
    function GetIncludeDirectory : Boolean;
    procedure SetIncludeDirectory(const NewOpt : Boolean);
    function GetRTSLowForWrite : Boolean;
    procedure SetRTSLowForWrite(const NewOpt : Boolean);
    function GetAbortNoCarrier : Boolean;
    procedure SetAbortNoCarrier(const NewOpt : Boolean);
    function GetAsciiSuppressCtrlZ : Boolean;
    procedure SetAsciiSuppressCtrlZ(const NewOpt : Boolean);
    function GetUpcaseFileNames : Boolean;
    procedure SetUpcaseFileNames(NewUpcase : Boolean);

    {Protocol event methods}
    procedure apxProtocolAccept(CP : TObject; var Accept : Boolean;
                                var FName : string); virtual;
    procedure apxProtocolError(CP : TObject; ErrorCode : Integer); virtual;
    procedure apxProtocolFinish(CP : TObject; ErrorCode : Integer); virtual;
    procedure apxProtocolLog(CP : TObject; Log : Cardinal); virtual;
    procedure apxProtocolNextFile(CP : TObject; var FName : string); virtual;
    procedure apxProtocolResume(CP : TObject;
                                var Resume : TApxWriteFailAction); virtual;
    procedure apxProtocolStatus(CP : TObject; Options : Cardinal); virtual;
    procedure apxProtocolOpenStream (    CP      : TObject;
                                     var Stream  : TStream;
                                         Reading : Boolean); virtual;
    procedure apxProtocolCloseStream (    CP     : TObject;
                                      var Stream : TStream); virtual;

  public
    ProtocolDriver : TApxBaseProtocolDriver;

    constructor Create(AOwner : TComponent); override;
      {-Create a TApxProtocol component}
    destructor Destroy; override;
      {-Destroy a TApxProtocol component}
    class function GetLogString(const D1, D2, D3 : DWORD) : string; override;

    procedure HandleAPXProtocolError (ErrorCode : SmallInt);
    procedure HandleAPXProtocolStatus (Status : Cardinal);
    procedure HandleAPXProtocolLog (LogItem : Cardinal);
    procedure HandleAPXProtocolResume (var Message : TAxMessage);
    function HandleAPXProtocolAcceptFile (var AcceptFileName : PChar) : Boolean;
    procedure HandleAPXProtocolOpenStream (var Stream  : TStream;
                                                     Reading : Boolean);
    procedure HandleAPXProtocolCloseStream (var Stream : TStream);
    function HandleAPXProtocolNextFile (var FName : string) : Boolean;

    procedure Assign(Source : TPersistent); override;
      {-Assign fields from TApxProtocol object specified by Source}
    {.Z-}
    function EstimateTransferSecs(const Size : Longint) : Longint;
      {-Return the number of seconds to transmit Size bytes}
    function StatusMsg(const Status : Cardinal) : String;
      {-Return a status message for Status}
    procedure StartTransmit;
      {-Start a background transmit session}
    procedure StartReceive;
      {-Start a background receive session}
    procedure CancelProtocol;
      {-Cancel the background protocol session}

    {General properties}
    property ComPort : TApxCustomComPort
      read FComPort write SetComPort;

    {General protocol control properties}
    property ProtocolType : TApxProtocolType
      read FProtocolType write SetProtocolType;
    property DestinationDirectory : String
      read GetDestinationDirectory write SetDestinationDirectory;
    property FileMask : TFileName
      read GetFileMask write SetFileMask;
    property BlockCheckMethod : TApxBlockCheckMethod
      read GetBlockCheckMethod write SetBlockCheckMethod;
    property HandshakeRetry : Cardinal
      read GetHandshakeRetry write SetHandshakeRetry default ApxDefHandshakeRetry;
    property HandshakeWait : Cardinal
      read GetHandshakeWait write SetHandshakeWait default ApxDefHandshakeWait;
    property TransmitTimeout : Cardinal
      read GetTransmitTimeout write SetTransmitTimeout default ApxDefTransTimeout;
    property FinishWait : Cardinal
      read GetFinishWait write SetFinishWait default ApxDefFinishWait;
    property ActualBPS : Longint
      read GetActualBPS write SetActualBPS;
    property TurnDelay : Cardinal
      read GetTurnDelay write SetTurnDelay default ApxDefTurnDelay;
    property Overhead : Cardinal
      read GetOverhead write SetOverhead default ApxDefOverhead;
    property WriteFailAction : TApxWriteFailAction
      read GetWriteFailAction write SetWriteFailAction
      default ApxDefWriteFailAction;

    {Option properties}
    property HonorDirectory : Boolean
      read GetHonorDirectory write SetHonorDirectory
      default ApxDefHonorDirectory;
    property IncludeDirectory : Boolean
      read GetIncludeDirectory write SetIncludeDirectory
      default ApxDefIncludeDirectory;
    property RTSLowForWrite : Boolean
      read GetRTSLowForWrite write SetRTSLowForWrite
      default ApxDefRTSLowForWrite;
    property AbortNoCarrier : Boolean
      read GetAbortNoCarrier write SetAbortNoCarrier
      default ApxDefAbortNoCarrier;
    property AsciiSuppressCtrlZ : Boolean
      read GetAsciiSuppressCtrlZ write SetAsciiSuppressCtrlZ
      default ApxDefAsciiSuppressCtrlZ;
    property UpcaseFileNames : Boolean
      read GetUpcaseFileNames write SetUpcaseFileNames
      default ApxDefUpcaseFileNames;

    {Read only properties}
    property Batch : Boolean
      read GetBatch;
    property BlockLength : Cardinal
      read GetBlockLength;
    property BlockNumber : Cardinal
      read GetBlockNumber;
    property ProtocolStatus : Cardinal
      read GetProtocolStatus;
    property ProtocolError : SmallInt
      read GetProtocolError;
    property FileLength : LongInt
      read GetFileLength;
    property FileDate : TDateTime
      read GetFileDate;
    property InitialPosition : LongInt
      read GetInitialPosition;

    {Status properties}
    property StatusDisplay : TApxAbstractStatus
      read FStatusDisplay write SetStatusDisplay;
    property ProtocolLog : TApxProtocolLog
      read FProtocolLog write SetProtocolLog;
    property StatusInterval : Cardinal
      read GetStatusInterval write SetStatusInterval default ApxDefStatusInterval;
    property InProgress : Boolean
      read GetInProgress;
    property BlockErrors : Cardinal
      read GetBlockErrors;
    property TotalErrors : Cardinal
      read GetTotalErrors;
    property BytesRemaining : LongInt
      read GetBytesRemaining;
    property BytesTransferred : LongInt
      read GetBytesTransferred;
    property ElapsedXfrTime : LongInt
      read GetElapsedXfrTime;
    property FileName : String
      read GetFileName write SetFileName;

    {Xmodem/Ymodem properties}
    property XYmodemBlockWait : Cardinal
      read GetXYmodemBlockWait write SetXYmodemBlockWait
      default ApxDefXYmodemBlockWait;

    {Zmodem properties}
    property ZmodemOptionOverride : Boolean
      read GetZmodemOptionOverride write SetZmodemOptionOverride
      default ApxDefZmodemOptionOverride;
    property ZmodemSkipNoFile : Boolean
      read GetZmodemSkipNoFile write SetZmodemSkipNoFile
      default ApxDefZmodemSkipNoFile;
    property ZmodemFileOption : TApxZmodemFileOptions
      read GetZmodemFileOption write SetZmodemFileOption
      default ApxDefZmodemFileOption;
    property ZmodemRecover : Boolean
      read GetZmodemRecover write SetZmodemRecover default ApxDefZmodemRecover;
    property Zmodem8K : Boolean
      read GetZmodem8K write SetZmodem8K default ApxDefZmodem8K;
    property ZmodemZRQINITValue : LongInt                            
      read GetZmodemZRQINITValue write SetZmodemZRQINITValue
      default ApxDefZmodemZRQINITValue;
    property ZmodemFinishRetry : Cardinal
      read GetZmodemFinishRetry write SetZmodemFinishRetry
      default ApxDefZmodemFinishRetry;

    {Kermit properties}
    property KermitMaxLen : Cardinal
      read GetKermitMaxLen write setKermitMaxLen
      default ApxDefKermitMaxLen;
    property KermitMaxWindows : Cardinal
      read GetKermitMaxWindows write SetKermitMaxWindows
      default ApxDefKermitMaxWindows;
    property KermitSWCTurnDelay : Cardinal
      read GetKermitSWCTurnDelay write SetKermitSWCTurnDelay
      default ApxDefKermitSWCTurnDelay;
    property KermitTimeoutSecs : Cardinal
      read GetKermitTimeoutSecs write SetKermitTimeoutSecs
      default ApxDefKermitTimeoutSecs;
    property KermitPadCharacter : Char
      read GetKermitPadCharacter write SetKermitPadCharacter
      default ApxDefKermitPadCharacter;
    property KermitPadCount : Cardinal
      read GetKermitPadCount write SetKermitPadCount
      default ApxDefKermitPadCount;
    property KermitTerminator : Char
      read GetKermitTerminator write SetKermitTerminator
      default ApxDefKermitTerminator;
    property KermitCtlPrefix : Char
      read GetKermitCtlPrefix write SetKermitCtlPrefix
      default ApxDefKermitCtlPrefix;
    property KermitHighbitPrefix : Char
      read GetKermitHighbitPrefix write SetKermitHighbitPrefix
      default ApxDefKermitHighbitPrefix;
    property KermitRepeatPrefix : Char
      read GetKermitRepeatPrefix write SetKermitRepeatPrefix
      default ApxDefKermitRepeatPrefix;
    property KermitWindowsTotal : Cardinal
      read GetKermitWindowsTotal;
    property KermitWindowsUsed : Cardinal
      read GetKermitWindowsUsed;
    property KermitLongBlocks : Boolean
      read GetKermitLongBlocks;
    property AsciiCharDelay : Cardinal
      read GetAsciiCharDelay write SetAsciiCharDelay
      default ApxDefAsciiCharDelay;
    property AsciiLineDelay : Cardinal
      read GetAsciiLineDelay write SetAsciiLineDelay
      default ApxDefAsciiLineDelay;
    property AsciiEOLChar : Char
      read GetAsciiEOLChar write SetAsciiEOLChar
      default ApxDefAsciiEOLChar;
    property AsciiCRTranslation : TApxAsciiEOLTranslation
      read GetAsciiCRTranslation write SetAsciiCRTranslation
      default ApxDefAsciiCRTranslation;
    property AsciiLFTranslation : TApxAsciiEOLTranslation
      read GetAsciiLFTranslation write SetAsciiLFTranslation
      default ApxDefAsciiLFTranslation;
    property AsciiEOFTimeout : Cardinal
      read GetAsciiEOFTimeout write SetAsciiEOFTimeout
      default ApxDefAsciiEOFTimeout;

    {Protocol events}
    property OnProtocolAccept : TProtocolAcceptEvent
      read FOnProtocolAccept write FOnProtocolAccept;
    property OnProtocolError : TProtocolErrorEvent
      read FOnProtocolError write FOnProtocolError;
    property OnProtocolFinish : TProtocolFinishEvent
      read FOnProtocolFinish write FOnProtocolFinish;
    property OnProtocolLog : TProtocolLogEvent
      read FOnProtocolLog write FOnProtocolLog;
    property OnProtocolNextFile : TProtocolNextFileEvent
      read FOnProtocolNextFile write FOnProtocolNextFile;
    property OnProtocolResume : TProtocolResumeEvent
      read FOnProtocolResume write FOnProtocolResume;
    property OnProtocolStatus : TProtocolStatusEvent
      read FOnProtocolStatus write FOnProtocolStatus;
    property OnProtocolStreamClose : TProtocolStreamCloseEvent
             read FOnProtocolStreamClose write FOnProtocolStreamClose;
    property OnProtocolStreamOpen : TProtocolStreamOpenEvent
             read FOnProtocolStreamOpen write FOnProtocolStreamOpen;
  end;

  {Abstract protocol status class}
  TApxAbstractStatus = class(TApxBaseComponent)
  protected {private}
    {.Z+}
    FDisplay         : TForm;
    FPosition        : TPosition;
    FCtl3D           : Boolean;
    FVisible         : Boolean;
    FCaption         : TCaption;

    procedure Notification(AComponent : TComponent;
                           Operation: TOperation); override;

  protected
    FProtocol        : TApxCustomProtocol;

    procedure SetPosition(const NewPosition : TPosition);
    procedure SetCtl3D(const NewCtl3D : Boolean);
    procedure SetVisible(const NewVisible : Boolean);
    procedure SetCaption(const NewCaption : TCaption);               
    procedure GetProperties;
    procedure Show;

  public
    constructor Create(AOwner : TComponent); override;
      {-Create a TApxAbstractStatus component}
    destructor Destroy; override;
      {-Destroy a TApxAbstractStatus component}
    {.Z-}
    procedure UpdateDisplay(First, Last : Boolean); virtual; abstract;
      {-Update the status display with current data}
    procedure CreateDisplay; dynamic; abstract;
      {-Create the status display}
    procedure DestroyDisplay; dynamic; abstract;
      {-Destroy the status display}

    property Display : TForm
      read FDisplay write FDisplay;

  published
    property Position : TPosition
      read FPosition write SetPosition;
    property Ctl3D : Boolean
      read FCtl3D write SetCtl3D;
    property Visible : Boolean
      read FVisible write SetVisible;
    property Protocol : TApxCustomProtocol
      read FProtocol write FProtocol;
    property Caption : TCaption
      read FCaption write SetCaption;                                 
  end;

  {Builtin log procedure}
  TApxProtocolLog = class(TApxBaseComponent)
  protected {private}
    {.Z+}
    FDeleteFailed  : TApxDeleteFailed;
    FHistoryName   : String;
    FProtocol      : TApxCustomProtocol;

    procedure Notification(AComponent : TComponent;
                           Operation: TOperation); override;

  public
    constructor Create(AOwner : TComponent); override;
      {-Create a TApxProtocolLog component}
    {.Z-}
    procedure UpdateLog(const Log : Cardinal); virtual;
      {-Add a log entry}

  published
    property Protocol : TApxCustomProtocol
      read FProtocol write FProtocol;
    property DeleteFailed : TApxDeleteFailed
      read FDeleteFailed write FDeleteFailed default awpDefDeleteFailed;
    property HistoryName : String
      read FHistoryName write FHistoryName;
  end;

  {Protocol component}
  TApxProtocol = class(TApxCustomProtocol)
  published
    property ComPort;
    property ProtocolType;
    property DestinationDirectory;
    property FileMask;
    property BlockCheckMethod;
    property HandshakeRetry;
    property HandshakeWait;
    property TransmitTimeout;
    property FinishWait;
    property TurnDelay;
    property Overhead;
    property WriteFailAction;
    property HonorDirectory;
    property IncludeDirectory;
    property RTSLowForWrite;
    property AbortNoCarrier;
    property AsciiSuppressCtrlZ;
    property StatusDisplay;
    property ProtocolLog;
    property StatusInterval;
    property FileName;
    property XYmodemBlockWait;
    property ZmodemOptionOverride;
    property ZmodemSkipNoFile;
    property ZmodemFileOption;
    property ZmodemRecover;
    property Zmodem8K;
    property ZmodemFinishRetry;
    property KermitMaxLen;
    property KermitMaxWindows;
    property KermitSWCTurnDelay;
    property KermitTimeoutSecs;
    property KermitPadCharacter;
    property KermitPadCount;
    property KermitTerminator;
    property KermitCtlPrefix;
    property KermitHighbitPrefix;
    property KermitRepeatPrefix;
    property AsciiCharDelay;
    property AsciiLineDelay;
    property AsciiEOLChar;
    property AsciiCRTranslation;
    property AsciiLFTranslation;
    property AsciiEOFTimeout;
    property UpcaseFileNames;
    property OnProtocolAccept;
    property OnProtocolError;
    property OnProtocolFinish;
    property OnProtocolLog;
    property OnProtocolNextFile;
    property OnProtocolResume;
    property OnProtocolStatus;
  end;

  {.Z+}
  {A list of active TApxProtocol objects}
  PProtocolWindowNode = ^TProtocolWindowNode;
  TProtocolWindowNode = record
    pwWindow   : ApxHWND;
    pwProtocol : TApxCustomProtocol;
  end;

  {Miscellaneous functions}
  function CheckNameString(const Check : TApxBlockCheckMethod) : String;
  function FormatMinSec(const TotalSecs : LongInt) : String;
  {.Z-}
  function ProtocolName(const ProtocolType : TApxProtocolType) : String;

  {Component registration procedure}

implementation

{$IFDEF TRIALRUN}
  {$I TRIAL03.INC}
  {$I TRIAL02.INC}
  {$I TRIAL01.INC}
{$ENDIF}

const
  FileSkipMask = $80;   {Skip file if dest doesn't exist}
  FileRecover  = $03;   {Resume interrupted file transfer}

var
  ProtList : TList;

{General purpose routines}

  function LeftPad(const S : String; Len : Byte) : String;
    {-Return a string left-padded to length len}
  var
    o : String;
    SLen : Byte;
  begin
    SLen := Length(S);
    if SLen >= Len then
      LeftPad := S
    else if SLen < 255 then begin
      SetLength(o, Len);
      Move(S[1], o[Succ(Cardinal(Len))-SLen], SLen);
      FillChar(o[1], Len-SLen, ' ');
      LeftPad := o;
    end;
  end;

  function SearchStatusDisplay(const C : TComponent) : TApxAbstractStatus;
    {-Search for a status display in the same form as TComponent}

    function FindStatusDisplay(const C : TComponent) : TApxAbstractStatus;
    var
      I  : Integer;
    begin
      Result := nil;
      if not Assigned(C) then
        Exit;

      {Look through all of the owned components}
      for I := 0 to C.ComponentCount-1 do begin
        if C.Components[I] is TApxAbstractStatus then begin
          {...and it's not assigned}
          if not Assigned(TApxAbstractStatus(C.Components[I]).FProtocol) then begin
            Result := TApxAbstractStatus(C.Components[I]);
            Exit;
          end;
        end;

        {If this isn't one, see if it owns other components}
        Result := FindStatusDisplay(C.Components[I]);
      end;
    end;

  begin
    {Search the entire form}
    Result := FindStatusDisplay(C);
  end;

  function SearchProtocolLog(const C : TComponent) : TApxProtocolLog;
    {-Search for a protocol log in the same form as TComponent}

    function FindProtocolLog(const C : TComponent) : TApxProtocolLog;
    var
      I  : Integer;
    begin
      Result := nil;
      if not Assigned(C) then
        Exit;

      {Look through all of the owned components}
      for I := 0 to C.ComponentCount-1 do begin
        if C.Components[I] is TApxProtocolLog then begin
          {...and it's not assigned}
          if not Assigned(TApxProtocolLog(C.Components[I]).FProtocol) then begin
            Result := TApxProtocolLog(C.Components[I]);
            Exit;
          end;
        end;

        {If this isn't one, see if it owns other components}
        Result := FindProtocolLog(C.Components[I]);
      end;
    end;

  begin
    {Search the entire form}
    Result := FindProtocolLog(C);
  end;

{Message handler window}

  function FindProtocol(Handle : ApxHWND) : TApxCustomProtocol;
    {-Return protocol object for this window handle}
  var
    I : Integer;
  begin
    for I := 0 to ProtList.Count-1 do begin
      with PProtocolWindowNode(ProtList.Items[I])^ do begin
        if pwWindow = Handle then begin
          Result := pwProtocol;
          Exit;
        end;
      end;
    end;
    Result := nil;
  end;

{TApxProtocol}

  procedure TApxCustomProtocol.HandleAPXProtocolFinish (var Message : TAxMessage);
  begin
    DeRegisterComHook;
    apxProtocolFinish(Self, SmallInt (Message.WParam));
  end;

  procedure TApxCustomProtocol.HandleAPXProtocolError (ErrorCode : SmallInt);
  begin
    apxProtocolError(Self, ErrorCode);
  end;

  procedure TApxCustomProtocol.HandleAPXProtocolStatus (Status : Cardinal);
  begin
    apxProtocolStatus(Self, Status);
  end;

  function TApxCustomProtocol.HandleAPXProtocolNextFile (var FName : string) : Boolean;
  begin
    apxProtocolNextFile(Self, FName);
    if FName = '' then
      Result := False
    else
      Result := True;
  end;

  procedure TApxCustomProtocol.HandleAPXProtocolLog (LogItem : Cardinal);
  begin
    apxProtocolLog(Self, LogItem);
  end;

  procedure TApxCustomProtocol.HandleAPXProtocolResume (var Message : TAxMessage);
  var
    Temp : TApxWriteFailAction;
  begin
    Temp := TApxWriteFailAction(Message.wParam);
    apxProtocolResume(Self, Temp);
  end;

  function TApxCustomProtocol.HandleAPXProtocolAcceptFile (var AcceptFileName : PChar) : Boolean;
  var
    Accept : Boolean;
    FName : string;
  begin
    FName := StrPas(AcceptFileName);
    apxProtocolAccept(Self, Accept, FName);
    if Accept then begin
      if FileName <> '' then
        StrPCopy(AcceptFileName, FName);
      Result := True;
    end else begin
      Result := False;
    end;
  end;

  procedure TApxCustomProtocol.HandleAPXProtocolOpenStream (var Stream  : TStream;
                                                     Reading : Boolean);
  begin
    apxProtocolOpenStream (Self, Stream, Reading);
  end;

  procedure TApxCustomProtocol.HandleAPXProtocolCloseStream (var Stream : TStream);
  begin
    apxProtocolCloseStream (Self, Stream);
  end;

  procedure TApxCustomProtocol.CreateWidget;
  begin
    FHandle := QObject_create(nil, nil);
  end;


  function TApxCustomProtocol.EventFilter(Sender : QObjectH; Event : QEventH) : Boolean;

    procedure DoCustomDispatchEvent(Event : QCustomEventH; FreeData : Boolean);
    var
      DataReference : PAxMessage;
      Data          : TAxMessage;
    begin
      DataReference := QCustomEvent_data(Event);

      { Local copy of data }
      Data := DataReference^;

      if FreeData then
        Dispose(DataReference);

      try
        Self.Dispatch(Data);
      except
        raise; 
      end;

      if not FreeData then
        DataReference^.Result := Data.Result;
    end;

  begin

    Result := True;
    case QEvent_type(Event) of

      QEventType_AxPostDispatchMessage:
        begin
          DoCustomDispatchEvent(QCustomEventH(Event), True);
          Result := True;
        end;

{      QEventType_AxPostDispatchMessage :
        begin
          DoCustomDispatchEvent(QCustomEventH(Event), True);
          Result := True;
        end; }

      else inherited EventFilter(Sender, Event);
    end;
  end;


  procedure TApxCustomProtocol.CheckPort;
    {-Set port's comhandle or raise exception}
  begin
    {Make sure comport is open, pass handle to protocol}
    if Assigned (FComPort) then
      ProtocolDriver.ComPort := FComPort
    else
      raise EPortNotAssigned.Create(ecPortNotAssigned, False);
  end;

  procedure TApxCustomProtocol.Notification(AComponent : TComponent;
                                            Operation : TOperation);
  begin
    inherited Notification(AComponent, Operation);

    if Operation = opRemove then begin
      {Owned components going away}
      if AComponent = FComPort then
        ComPort := nil;
      if AComponent = FStatusDisplay then
        StatusDisplay := nil;
      if AComponent = FProtocolLog then
        ProtocolLog := nil;
    end else if Operation = opInsert then begin
      {Check for new comport}
      if AComponent is TApxCustomComPort then
        if not Assigned(FComPort) then
          ComPort := TApxCustomComPort(AComponent);

      {Check for new status component}
      if AComponent is TApxAbstractStatus then begin
        if not Assigned(FStatusDisplay) then
          if not Assigned(TApxAbstractStatus(AComponent).FProtocol) then
            StatusDisplay := TApxAbstractStatus(AComponent);
      end;

      {Check for new protocol log component}
      if AComponent is TApxProtocolLog then begin
        if not Assigned(FProtocolLog) then begin
          if not Assigned(TApxProtocolLog(AComponent).FProtocol) then begin
            ProtocolLog := TApxProtocolLog(AComponent);
            ProtocolLog.FProtocol := Self;
          end;
        end;
      end;
    end;
  end;

  procedure TApxCustomProtocol.SetComPort(const NewComPort : TApxCustomComPort);
    {-Set a new comport}
  begin
    if NewComPort <> FComPort then begin
      FComPort := NewComPort;
      if Assigned(FComPort) then
        ProtocolDriver.ComPort := ComPort
      else
        ProtocolDriver.ComPort := nil;
    end;
  end;

  procedure TApxCustomProtocol.SetProtocolType(
                               const NewProtocol : TApxProtocolType);
    {-Set a new protocol type}
  var
    Status : Integer;
    NewProtocolDriver : TApxBaseProtocolDriver;
  begin
    if (NewProtocol <> FProtocolType) or
       (csLoading in ComponentState) then begin
      {Dispose of current protocol, if necessary}
      case NewProtocol of
        ptXmodem, ptXmodemCRC, ptXmodem1K, ptXmodem1KG :
          NewProtocolDriver := TApxXModemDriver.Create (Self);
          {nothing else to do}
        ptYModem, ptYmodemG :
          begin
            if Assigned (ProtocolDriver) then
              ProtocolDriver.DonePart;
            NewProtocolDriver := TApxYModemDriver.Create (Self);
          end;
        ptZmodem :
          begin
            if Assigned (ProtocolDriver) then
              ProtocolDriver.DonePart;
            NewProtocolDriver := TApxZModemDriver.Create (Self);
          end;
        ptKermit :
          begin
            if Assigned (ProtocolDriver) then
              ProtocolDriver.DonePart;
            NewProtocolDriver := TApxKermitDriver.Create (Self);
          end;
        ptASCII :
          begin
            if Assigned (ProtocolDriver) then
              ProtocolDriver.DonePart;
            NewProtocolDriver := TApxASCIIDriver.Create (Self);
          end;
        else
          { Create ASCII prot by default }
          NewProtocolDriver := TApxASCIIDriver.Create (Self);
      end;

      {Init new protocol}
      if Assigned (ProtocoLDriver) then begin
        NewProtocolDriver.Assign (ProtocolDriver);
        ProtocolDriver.Free;
      end;
      ProtocolDriver := NewProtocolDriver;
      FProtocolType := NewProtocol;

      Status := ecOk;
      Force := True;
      case NewProtocol of
        ptXmodem, ptXmodemCRC, ptXmodem1K, ptXmodem1KG :
          begin
            ProtocolDriver.Reinit; 
            if ProtocolDriver is TApxXModemDriver then
              (ProtocolDriver as TApxXModemDriver).InitData (
                                                   NewProtocol <> ptXmodem,
                                                   NewProtocol = ptXmodem1K,
                                                   NewProtocol = ptXmodem1KG);
            SetXYmodemBlockWait(FXYmodemBlockWait);
          end;

        ptYmodem, ptYModemG :
          begin
            Status := ProtocolDriver.Reinit;
            if ProtocolDriver is TApxYModemDriver then
              (ProtocolDriver as TApxYModemDriver).InitData (
                                                   True,         
                                                   True,
                                                   NewProtocol = ptYmodemG);
            if Status = ecOk then
              SetXYmodemBlockWait(FXYmodemBlockWait);
          end;

        ptZmodem :
          begin
            Status := ProtocolDriver.Reinit;
            if Status = ecOk then begin
              SetZmodemOptionOverride(FZmodemOptionOverride);
              SetZmodemSkipNoFile(FZmodemSkipNoFile);
              SetZmodemFileOption(FZmodemFileOption);
              SetZmodemRecover(FZmodemRecover);
              SetZmodem8K(FZmodem8K);
              SetZmodemFinishRetry(FZmodemFinishRetry);
            end;
          end;

        ptKermit :
          begin
            Status := ProtocolDriver.Reinit;
            if Status = ecOk then begin
              SetKermitMaxLen(FKermitMaxLen);
              SetKermitMaxWindows(FKermitMaxWindows);
              SetKermitSWCTurnDelay(FKermitSWCTurnDelay);
              SetKermitTimeoutSecs(FKermitTimeoutSecs);
              SetKermitPadCharacter(FKermitPadCharacter);
              SetKermitPadCount(FKermitPadCount);
              SetKermitTerminator(FKermitTerminator);
              SetKermitCtlPrefix(FKermitCtlPrefix);
              SetKermitHighbitPrefix(FKermitHighbitPrefix);
              SetKermitRepeatPrefix(FKermitRepeatPrefix);
            end;
          end;

        ptASCII :
          begin
            Status := ProtocolDriver.Reinit;
            if Status = ecOk then begin
              SetAsciiCharDelay(FAsciiCharDelay);
              SetAsciiLineDelay(FAsciiLineDelay);
              SetAsciiEOLChar(FAsciiEOLChar);
              SetAsciiCRTranslation(FAsciiCRTranslation);
              SetAsciiLFTranslation(FAsciiLFTranslation);
              SetAsciiEOFTimeout(FAsciiEOFTimeout);
            end;
          end;
     end;

     {Note new protocol type}
     if Status = ecOk then
       FProtocolType := NewProtocol
     else if (Status < ecOk) and
              not (csLoading in Self.ComponentState) then
      raise EProtocol.Create (Status, False);
    end;
  end;

  function TApxCustomProtocol.GetDestinationDirectory : String;
    {-Return the destination directory}
  begin
    Result := StrPas(ProtocolDriver.DestDir); 
  end;

  procedure TApxCustomProtocol.SetDestinationDirectory(const NewDir : String);
    {-Set a new destination directory}
  begin
    with ProtocolDriver do
      StrPCopy(DestDir, NewDir);
  end;

  function TApxCustomProtocol.GetFileMask : TFileName;
    {-Return the current file mask}
  begin
    Result := StrPas(ProtocolDriver.SearchMask);
  end;

  procedure TApxCustomProtocol.SetFileMask(const NewFileMask : TFileName);
    {-Set a new file mask}
  var
    S : TFileName;
  begin
    with ProtocolDriver do
      if Length(NewFileMask) > 255 then begin
        S := NewFileMask;
        SetLength(S, 255);
        StrPCopy(SearchMask, S);
      end else
        StrPCopy(SearchMask, NewFileMask);
  end;

  function TApxCustomProtocol.GetBatch : Boolean;
    {-Return True if the current protocol supports batch transfers}
  begin
    Result := ProtocolDriver.BatchProtocol;
  end;

  function TApxCustomProtocol.GetBlockCheckMethod : TApxBlockCheckMethod;
    {-Return the current block check method}
  begin
    Result := TApxBlockCheckMethod(ProtocolDriver.CheckType);
  end;

  function TApxCustomProtocol.GetProtocolStatus : Cardinal;
    {-Return the current protocol status}
  begin
    Result := ProtocolDriver.ProtocolStatus;
  end;

  function TApxCustomProtocol.GetProtocolError : SmallInt;
    {-Return the current protocol error}
  begin
    Result := ProtocolDriver.ProtocolError;
  end;

  function TApxCustomProtocol.GetFileLength : LongInt;
    {-Return the file length}
  begin
    Result := ProtocolDriver.SrcFileLen;
  end;

  function TApxCustomProtocol.GetFileDate : TDateTime;
    {-Return the file timestamp}
  begin
    Result := FileDateToDateTime(ProtocolDriver.SrcFileDate);
  end;

  function TApxCustomProtocol.GetInitialPosition : LongInt;
    {-Return the initial file position}
  begin
    Result := ProtocolDriver.InitFilePos;
  end;

  function TApxCustomProtocol.GetStatusInterval : Cardinal;
    {-Return the current status update interval}
  begin
    Result := ProtocolDriver.StatusInterval;
  end;

  procedure TApxCustomProtocol.SetStatusInterval(const NewInterval : Cardinal);
    {-Set a new update status interval}
  begin
    with ProtocolDriver do
      if NewInterval <> StatusInterval then 
        StatusInterval := NewInterval;
  end;

  procedure TApxCustomProtocol.SetStatusDisplay(
                               const NewDisplay : TApxAbstractStatus);
    {-Set a new status display}
  begin
    if NewDisplay <> FStatusDisplay then begin
      FStatusDisplay := NewDisplay;
      if Assigned(FStatusDisplay) then
        FStatusDisplay.FProtocol := Self;
    end;
  end;

  procedure TApxCustomProtocol.SetProtocolLog(const NewLog : TApxProtocolLog);
    {-Set a new protocol log}
  begin
    if NewLog <> FProtocolLog then begin
      FProtocolLog := NewLog;
      if Assigned(FProtocolLog) then
        FProtocolLog.FProtocol := Self;
    end;
  end;

  function TApxCustomProtocol.GetInProgress : Boolean;
    {-Return True if protocol is in progress}
  begin
    Result := ProtocolDriver.InProgress <> 0;
  end;

  function TApxCustomProtocol.GetBlockErrors : Cardinal;
    {-Return the number of block errors}
  begin
    Result := ProtocolDriver.BlockErrors;
  end;

  function TApxCustomProtocol.GetTotalErrors : Cardinal;
    {-Return the number of total errors}
  begin
    Result := ProtocolDriver.TotalErrors;
  end;

  function TApxCustomProtocol.GetBytesRemaining : LongInt;
    {-Return the number of bytes remaining to be transferred}
  begin
    Result := ProtocolDriver.apGetBytesRemaining;
  end;

  function TApxCustomProtocol.GetBytesTransferred : LongInt;
    {-Return the number of bytes transferred so far}
  begin
    Result := ProtocolDriver.apGetBytesTransferred;
  end;

  function TApxCustomProtocol.GetElapsedXfrTime : LongInt;
    {-Return the ticks elapsed for this transfer}
  begin
    Result := ProtocolDriver.ElapsedXfrTime;
  end;

  function TApxCustomProtocol.GetFileName : String;
    {-Return the current file name}
  begin
    Result := StrPas(ProtocolDriver.Pathname);
  end;

  procedure TApxCustomProtocol.SetFileName(const NewName : String);
    {-Set/change the incoming file name}
  var
    P : array[0..255] of Char;
    S : String;
  begin
    {Allow changes only when WorkFile is *not* open}
    case TFileRec(ProtocolDriver.WorkFile).Mode of
      fmInput, fmOutput, fmInOut : ;
      else begin
        if Length(NewName) > 255 then begin
          S := NewName;
          SetLength(S, 255);
          StrPCopy(P, S);
        end else
          StrPCopy(P, NewName);
        ProtocolDriver.apSetReceiveFileName(P);
      end;
    end;
  end;

  function TApxCustomProtocol.GetXYmodemBlockWait : Cardinal;
    {-Return the X/Ymodem block wait value}
  begin
    case TApxProtocolType(ProtocolDriver.CurProtocol) of
      ptXmodem..ptYmodemG :
        begin
          if ProtocolDriver is TApxXModemDriver then
            FXYmodemBlockWait := (ProtocolDriver as TApxXModemDriver).BlockWait div 1000;
          Result := FXYmodemBlockWait;
        end;
      else
        Result := FXYmodemBlockWait;
    end;
  end;

  procedure TApxCustomProtocol.SetXYmodemBlockWait(const NewWait : Cardinal);
    {-Set a new X/Ymodem block wait}
  begin
    if (NewWait <> FXYmodemBlockWait * 1000) or Force then begin
      FXYmodemBlockWait := NewWait;
      case TApxProtocolType(ProtocolDriver.CurProtocol) of
        ptXmodem..ptYmodemG :
          if ProtocolDriver is TApxXModemDriver then
          (ProtocolDriver as TApxXModemDriver).BlockWait := NewWait * 1000;
      end;
    end;
  end;

  function TApxCustomProtocol.GetZmodemOptionOverride : Boolean;
    {-Return the zmodem override option}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
       (ProtocolDriver is TApxZModemDriver) then
      FZmodemOptionOverride := (ProtocolDriver as TApxZModemDriver).FileMgmtOverride;
    Result := FZmodemOptionOverride;
  end;

  procedure TApxCustomProtocol.SetZmodemOptionOverride(
                               const NewOverride : Boolean);
    {-Enable/disable the zmodem option override option}
  begin
    if (NewOverride <> FZmodemOptionOverride) or Force then begin
      FZmodemOptionOverride := NewOverride;
      if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
         (ProtocolDriver is TApxZModemDriver) then
        (ProtocolDriver as TApxZModemDriver).FileMgmtOverride := NewOverride;
    end;
  end;

  function TApxCustomProtocol.GetZmodemSkipNoFile : Boolean;
    {-Return the zmodem skip no file option}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
       (ProtocolDriver is TApxZModemDriver) then
      with ProtocolDriver as TApxZModemDriver do
        FZmodemSkipNoFile := (FileMgmtOpts and FileSkipMask) <> 0;
    Result := FZmodemSkipNoFile;
  end;

  procedure TApxCustomProtocol.SetZmodemSkipNoFile(const NewSkip : Boolean);
    {-Enable/disable the skipnofile option}
  begin
    if (NewSkip <> FZmodemSkipNoFile) or Force then begin
      FZmodemSkipNoFile := NewSkip;
      if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
         (ProtocolDriver is TApxZModemDriver) then
        with ProtocolDriver as TApxZModemDriver do
          if NewSkip then
            FileMgmtOpts := FileMgmtOpts or FileSkipMask
          else
            FileMgmtOpts := FileMgmtOpts and not FileSkipMask;
    end;
  end;

  function TApxCustomProtocol.GetZmodemFileOption : TApxZmodemFileOptions;
    {-Return the zmodem file managment option}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
       (ProtocolDriver is TApxZModemDriver) then
      FZmodemFileOption :=
        TApxZmodemFileOptions((ProtocolDriver as TApxZModemDriver).FileMgmtOpts and not FileSkipMask);
    Result := FZmodemFileOption;
  end;

  procedure TApxCustomProtocol.SetZmodemFileOption(
                               const NewOpt : TApxZmodemFileOptions);
    {-Set new file management options}
  var
    OldSkip : Boolean;
  begin
    if (NewOpt <> FZmodemFileOption) or Force then begin

      {Disallow zfoWriteCrc, it's not supported yet}
      if NewOpt = zfoWriteCrc then
        Exit;

      FZmodemFileOption := NewOpt;
      if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
         (ProtocolDriver is TApxZModemDriver) then begin
        with ProtocolDriver as TApxZModemDriver do begin
          OldSkip := ZmodemSkipNoFile;
          FileMgmtOpts := Ord(NewOpt);
          if OldSkip then
            FileMgmtOpts := FileMgmtOpts or FileSkipMask;
        end;
      end;
    end;
  end;

  function TApxCustomProtocol.GetZmodemRecover : Boolean;
    {-Return the recovery option}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
       (ProtocolDriver is TApxZModemDriver) then
      FZmodemRecover := (ProtocolDriver as TApxZModemDriver).ReceiverRecover;
    Result := FZmodemRecover;
  end;

  procedure TApxCustomProtocol.SetZmodemRecover(const NewRecover : Boolean);
    {-Enable/disable Zmodem crash recovery}
  begin
    if (NewRecover <> FZmodemRecover) or Force then begin
      FZmodemRecover := NewRecover;
      if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
         (ProtocolDriver is TApxZModemDriver) then
        (ProtocolDriver as TApxZModemDriver).ReceiverRecover := NewRecover;
    end;
  end;

  function TApxCustomProtocol.GetZmodem8K : Boolean;
    {-Return the state Zmodem's 8K mode}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
       (ProtocolDriver is TApxZModemDriver) then
      FZmodem8K := (ProtocolDriver as TApxZModemDriver).Use8KBlocks;
    Result := FZmodem8K;
  end;

  procedure TApxCustomProtocol.SetZmodem8K(const New8K : Boolean);
    {-Enable/disable 8K blocks}
  var
    ErrorCode : Integer;
  begin
    if (New8K <> FZmodem8K) or Force then begin
      FZmodem8K := New8K;
      if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
         (ProtocolDriver is TApxZModemDriver) then begin
        ErrorCode := (ProtocolDriver as TApxZModemDriver).SetBigSubpacketOption(New8K); 
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetZmodemZRQINITValue : LongInt;       
    {-Return the Zmodem's ZRQINIT value}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
       (ProtocolDriver is TApxZModemDriver) then
      FZmodemZRQINITValue := (ProtocolDriver as TApxZModemDriver).ZRQINITValue;
    Result := FZmodemZRQINITValue;
  end;

  procedure TApxCustomProtocol.SetZmodemZRQINITValue(const NewZRQINITValue : LongInt);
    {-Set the Zmodem's ZRQINIT value}
  begin
    if (NewZRQINITValue <> FZmodemZRQINITValue) or Force then begin
      FZmodemZRQINITValue := NewZRQINITValue;
      if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
         (ProtocolDriver is TApxZModemDriver) then
        (ProtocolDriver as TApxZModemDriver).ZRQINITValue := NewZRQINITValue;
    end;
  end;

  function TApxCustomProtocol.GetZmodemFinishRetry : Cardinal;
    {-Return the Zmodem finish retry count}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
       (ProtocolDriver is TApxZModemDriver) then
      FZmodemFinishRetry := (ProtocolDriver as TApxZModemDriver).FinishRetry;
    Result := FZmodemFinishRetry;
  end;

  procedure TApxCustomProtocol.SetZmodemFinishRetry(const NewRetry : Cardinal);
    {-Enable/disable 8K blocks}
  var
    ErrorCode : Integer;
  begin
    if (NewRetry <> FZmodemFinishRetry) or Force then begin
      FZmodemFinishRetry := NewRetry;
      if (ProtocolDriver.CurProtocol = Ord(ptZmodem)) and
         (ProtocolDriver is TApxZModemDriver) then begin
        ErrorCode := (ProtocolDriver as TApxZModemDriver).SetZmodemFinishWait(
                              (ProtocolDriver as TApxZModemDriver).FinishWait,
                              FZmodemFinishRetry);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end
    end;
  end;

  function TApxCustomProtocol.GetKermitMaxLen : Cardinal;
    {-Return the max packet len (normal)}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then
      with ProtocolDriver as TApxKermitDriver do begin   

        FKermitMaxLen := KermitOptions.MaxLongPacketLen;
        if FKermitMaxLen = 0 then
          FKermitMaxLen := KermitOptions.MaxPacketLen;
      end;
    Result := FKermitMaxLen;
  end;

  procedure TApxCustomProtocol.SetKermitMaxLen(const NewLen : Cardinal);
    {-Set a new max len}
  var
    ErrorCode : Integer;
  begin
    if (NewLen <> FKermitMaxLen) or Force then begin
      if NewLen <= 94 then begin
        FKermitMaxLen := NewLen;
        if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
           (ProtocolDriver is TApxKermitDriver) then
          with ProtocolDriver as TApxKermitDriver do begin
            ErrorCode := SetMaxPacketLen(NewLen);
            if (ErrorCode < ecOk) and
              not (csLoading in Self.ComponentState) then
              raise EProtocol.Create (ErrorCode, False);
            ErrorCode := SetMaxLongPacketLen(0);
            if (ErrorCode < ecOk) and
                not (csLoading in Self.ComponentState) then
              raise EProtocol.Create (ErrorCode, False);
          end;
      end else begin
        if NewLen > MaxKermitLongLen then
          FKermitMaxLen := MaxKermitLongLen
        else
          FKermitMaxLen := NewLen;
        if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
           (ProtocolDriver is TApxKermitDriver) then begin
          ErrorCode := (ProtocolDriver as TApxKermitDriver).SetMaxLongPacketLen(NewLen);
          if (ErrorCode < ecOk) and
              not (csLoading in Self.ComponentState) then
            raise EProtocol.Create (ErrorCode, False);
        end;
      end;
    end;
  end;

  function TApxCustomProtocol.GetKermitMaxWindows : Cardinal;
    {-Return the maximum number of windows}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then
      FKermitMaxWindows := (ProtocolDriver as TApxKermitDriver).KermitOptions.WindowSize;
    Result := FKermitMaxWindows;
  end;

  procedure TApxCustomProtocol.SetKermitMaxWindows(const NewMax : Cardinal);
    {-Set new max windows}
  var
    ErrorCode : Integer;
  begin
    if (NewMax <> FKermitMaxWindows) or Force then begin
      if NewMax > MaxKermitWindows then
        FKermitMaxWindows := MaxKermitWindows
      else
        FKermitMaxWindows := NewMax;

      {If not really using windows then disable SWC}
      if (NewMax = 0) or (NewMax = 1) then
        FKermitMaxWindows := 0;

      if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
         (ProtocolDriver is TApxKermitDriver) then begin
        ErrorCode := (ProtocolDriver as TApxKermitDriver).SetMaxWindows(FKermitMaxWindows);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetKermitSWCTurnDelay : Cardinal;
    {-Return the turn delay for when sliding windows are in use}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then
      FKermitSWCTurnDelay := (ProtocolDriver as TApxKermitDriver).SWCTurnDelay;
    Result := FKermitSWCTurnDelay;
  end;

  procedure TApxCustomProtocol.SetKermitSWCTurnDelay(const NewDelay : Cardinal);
    {-Set new turn delay value for sliding windows}
  var
    ErrorCode : Integer;
  begin
    if (NewDelay <> FKermitSWCTurnDelay) or Force then begin
      FKermitSWCTurnDelay := NewDelay;
      if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
         (ProtocolDriver is TApxKermitDriver) then begin
        ErrorCode := (ProtocolDriver as TApxKermitDriver).SetSWCTurnDelay(NewDelay);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetKermitTimeoutSecs : Cardinal;
    {-Return the packet timeout, in seconds}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then
      FKermitTimeoutSecs := (ProtocolDriver as TApxKermitDriver).KermitOptions.MaxTimeout;
    Result := FKermitTimeoutSecs;
  end;

  procedure TApxCustomProtocol.SetKermitTimeoutSecs(const NewTimeout : Cardinal);
    {-Set a new timeout value}
  var
    ErrorCode : Integer;
  begin
    if (NewTimeout <> FKermitTimeoutSecs) or Force then begin
      FKermitTimeoutSecs := NewTimeout;
      if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
         (ProtocolDriver is TApxKermitDriver) then begin
        ErrorCode := (ProtocolDriver as TApxKermitDriver).SetMaxTimeoutSecs(NewTimeout);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetKermitPadCharacter : Char;
    {-Return the new pad character}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then begin
      FKermitPadCharacter := (ProtocolDriver as TApxKermitDriver).KermitOptions.PadChar;
    end;
    Result := FKermitPadCharacter;
  end;

  procedure TApxCustomProtocol.SetKermitPadCharacter(NewChar : Char);
    {-Set a new pad character}
  var
    ErrorCode : Integer;
  begin
    if (NewChar <> FKermitPadCharacter) or Force then begin
      FKermitPadCharacter := NewChar;
      if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
         (ProtocolDriver is TApxKermitDriver) then begin
        ErrorCode := (ProtocolDriver as TApxKermitDriver).SetPacketPadding(FKermitPadCharacter,
                                        FKermitPadCount);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetKermitPadCount : Cardinal;
    {-Return the pad count}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then
      FKermitPadCount := (ProtocolDriver as TApxKermitDriver).KermitOptions.PadCount;
    Result := FKermitPadCount;
  end;

  procedure TApxCustomProtocol.SetKermitPadCount(NewCount : Cardinal);
    {-Set a new pad count}
  var
    ErrorCode : Integer;
  begin
    if (NewCount <> FKermitPadCount) or Force then begin
      FKermitPadCount := NewCount;
      if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
         (ProtocolDriver is TApxKermitDriver) then begin
        ErrorCode := (ProtocolDriver as TApxKermitDriver).SetPacketPadding(FKermitPadCharacter,
                                        FKermitPadCount);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetKermitTerminator : Char;
    {-Return the kermit terminator}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then
      FKermitTerminator := (ProtocolDriver as TApxKermitDriver).KermitOptions.Terminator;
    Result := FKermitTerminator;
  end;

  procedure TApxCustomProtocol.SetKermitTerminator(const NewTerminator : Char);
    {-Set new terminator}
  var
    ErrorCode : Integer;
  begin
    if (NewTerminator <> FKermitTerminator) or Force then begin
      FKermitTerminator := NewTerminator;
      if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
         (ProtocolDriver is TApxKermitDriver) then begin
        ErrorCode := (ProtocolDriver as TApxKermitDriver).SetTerminator(NewTerminator);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetKermitCtlPrefix : Char;
    {-Return the control char prefix}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then
      FKermitCtlPrefix := (ProtocolDriver as TApxKermitDriver).KermitOptions.CtlPrefix;
    Result := FKermitCtlPrefix;
  end;

  procedure TApxCustomProtocol.SetKermitCtlPrefix(const NewPrefix : Char);
    {-Set new ctrl char prefix}
  var
    ErrorCode : Integer;
  begin
    if (NewPrefix <> FKermitCtlPrefix) or Force then begin
      FKermitCtlPrefix := NewPrefix;
      if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
         (ProtocolDriver is TApxKermitDriver) then begin
        ErrorCode := (ProtocolDriver as TApxKermitDriver).SetCtlPrefix(NewPrefix);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetKermitHighbitPrefix : Char;
    {-Return the highbit prefix}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then
      FKermitHighbitPrefix := (ProtocolDriver as TApxKermitDriver).KermitOptions.HibitPrefix;
    Result := FKermitHighbitPrefix;
  end;

  procedure TApxCustomProtocol.SetKermitHighbitPrefix(const NewPrefix : Char);
    {-Set new highbit prefix}
  var
    ErrorCode : Integer;
  begin
    if (NewPrefix <> FKermitHighbitPrefix) or Force then begin
      FKermitHighbitPrefix := NewPrefix;
      if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
         (ProtocolDriver is TApxKermitDriver) then begin
        ErrorCode := (ProtocolDriver as TApxKermitDriver).SetHibitPrefix(NewPrefix); 
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetKermitRepeatPrefix : Char;
    {-Return the repeat prefix}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then
      FKermitRepeatPrefix := (ProtocolDriver as TApxKermitDriver).KermitOptions.RepeatPrefix;
    Result := FKermitRepeatPrefix;
  end;

  procedure TApxCustomProtocol.SetKermitRepeatPrefix(const NewPrefix : Char);
    {-Set a new repeat prefix}
  var
    ErrorCode : Integer;
  begin
    if (NewPrefix <> FKermitRepeatPrefix) or Force then begin
      FKermitRepeatPrefix := NewPrefix;
      if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
         (ProtocolDriver is TApxKermitDriver) then begin
        ErrorCode := (ProtocolDriver as TApxKermitDriver).SetRepeatPrefix(NewPrefix);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetKermitWindowsTotal : Cardinal;
    {-Return the total number of windows negotiated}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then
      Result := (ProtocolDriver as TApxKermitDriver).GetSWCSize
    else
      Result := KermitMaxWindows;
  end;

  function TApxCustomProtocol.GetKermitWindowsUsed : Cardinal;
    {-Return the total number of windows filled with data}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then
      Result := (ProtocolDriver as TApxKermitDriver).WindowsUsed
    else
      Result := 0;
  end;

  function TApxCustomProtocol.GetKermitLongBlocks : Boolean;
    {-Return True if long blocks are requested or negotiated}
  var
    Dummy : Cardinal;
    InUse : Boolean;
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptKermit)) and
       (ProtocolDriver is TApxKermitDriver) then begin
      (ProtocolDriver as TApxKermitDriver).GetLPStatus(InUse, Dummy);
      Result := InUse;
    end else
      Result := False;
  end;

  function TApxCustomProtocol.GetAsciiCharDelay : Cardinal;
    {-Return the inter-char delay}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
       (ProtocolDriver is TApxASCIIDriver) then
      FAsciiCharDelay := (ProtocolDriver as TApxASCIIDriver).InterCharDelay;
    Result := FAsciiCharDelay;
  end;

  procedure TApxCustomProtocol.SetAsciiCharDelay(const NewDelay : Cardinal);
    {-Set a new inter-char delay}
  var
    ErrorCode : Integer;
  begin
    if (NewDelay <> FAsciiCharDelay) or Force then begin
      FAsciiCharDelay := NewDelay;
      if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
         (ProtocolDriver is TApxASCIIDriver) then begin
        ErrorCode := (ProtocolDriver as TApxASCIIDriver).SetDelays(FAsciiCharDelay, FAsciiLineDelay); 
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetUpcaseFileNames : Boolean;
    {-Return the UpcaseFileNames value}
  begin
    Result := ProtocolDriver.UpcaseFileNames;
  end;

  procedure TApxCustomProtocol.SetUpcaseFileNames(NewUpcase : Boolean);
    {-Set a new UpcaseFileNames value}
  begin
    ProtocolDriver.UpcaseFileNames := NewUpcase;
  end;

  function TApxCustomProtocol.GetAsciiLineDelay : Cardinal;
    {-Return the inter-line delay}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
       (ProtocolDriver is TApxASCIIDriver) then
      FAsciiLineDelay := (ProtocolDriver as TApxASCIIDriver).InterLineDelay;
    Result := FAsciiLineDelay;
  end;

  procedure TApxCustomProtocol.SetAsciiLineDelay(const NewDelay : Cardinal);
    {-Set a new inter-line delay}
  var
    ErrorCode : Integer;
  begin
    if (NewDelay <> FAsciiLineDelay) or Force then begin
      FAsciiLineDelay := NewDelay;
      if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
         (ProtocolDriver is TApxASCIIDriver) then begin
        ErrorCode := (ProtocolDriver as TApxASCIIDriver).SetDelays(FAsciiCharDelay, FAsciiLineDelay);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetAsciiEOLChar : Char;
    {-Return the EOL character}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
       (ProtocolDriver is TApxASCIIDriver) then
      FAsciiEOLChar := (ProtocolDriver as TApxASCIIDriver).EOLChar;
    Result := FAsciiEOLChar;
  end;

  procedure TApxCustomProtocol.SetAsciiEOLChar(const NewChar : Char);
    {-Set new EOL char}
  var
    ErrorCode : Integer;
  begin
    if (NewChar <> FAsciiEOLChar) or Force then begin
      FAsciiEOLChar := NewChar;
      if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
         (ProtocolDriver is TApxASCIIDriver) then begin
        ErrorCode := (ProtocolDriver as TApxASCIIDriver).SetEOLChar(FAsciiEOLChar);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetAsciiCRTranslation : TApxAsciiEOLTranslation;
    {-Return the CR translation option}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
       (ProtocolDriver is TApxASCIIDriver) then
      FAsciiCRTranslation := TApxAsciiEOLTranslation((ProtocolDriver as TApxASCIIDriver).CRTransMode);
    Result := FAsciiCRTranslation;
  end;

  procedure TApxCustomProtocol.SetAsciiCRTranslation(
                            const NewTrans : TApxAsciiEOLTranslation);
    {-Set a new CR translation}
  var
    ErrorCode : Integer;
  begin
    if (NewTrans <> FAsciiCRTranslation) or Force then begin
      FAsciiCRTranslation := NewTrans;
      if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
         (ProtocolDriver is TApxASCIIDriver) then begin
        ErrorCode := (ProtocolDriver as TApxASCIIDriver).SetEOLTranslation(Ord(FAsciiCRTranslation),
                                         Ord(FAsciiLFTranslation)); 
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetAsciiLFTranslation : TApxAsciiEOLTranslation;
    {-Return LF translation option}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
       (ProtocolDriver is TApxASCIIDriver) then
      FAsciiLFTranslation := TApxAsciiEOLTranslation((ProtocolDriver as TApxASCIIDriver).LFTransMode);
    Result := FAsciiLFTranslation;
  end;

  procedure TApxCustomProtocol.SetAsciiLFTranslation(
                            const NewTrans : TApxAsciiEOLTranslation);
    {-Set a new LF translation}
  var
    ErrorCode : Integer;
  begin
    if (NewTrans <> FAsciiLFTranslation) or Force then begin
      FAsciiLFTranslation := NewTrans;
      if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
         (ProtocolDriver is TApxASCIIDriver) then begin
        ErrorCode := (ProtocolDriver as TApxASCIIDriver).SetEOLTranslation(Ord(FAsciiCRTranslation),
                                         Ord(FAsciiLFTranslation)); 
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetAsciiEOFTimeout : Cardinal;
    {-Return the EOF timeout value, in time}
  begin
    if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
       (ProtocolDriver is TApxASCIIDriver) then
      FAsciiEOFTimeout := (ProtocolDriver as TApxASCIIDriver).RcvTimeout;
    Result := FAsciiEOFTimeout;
  end;

  procedure TApxCustomProtocol.SetAsciiEOFTimeout(const NewTimeout : Cardinal);
    {-Set a new ascii timeout value}
  var
    ErrorCode : Integer;
  begin
    if (NewTimeout <> FAsciiEOFTimeout) or Force then begin
      FAsciiEOFTimeout := NewTimeout;
      if (ProtocolDriver.CurProtocol = Ord(ptAscii)) and
         (ProtocolDriver is TApxASCIIDriver) then begin
        ErrorCode := (ProtocolDriver as TApxASCIIDriver).SetEOFTimeout(FAsciiEOFTimeout);
        if (ErrorCode < ecOk) and
            not (csLoading in Self.ComponentState) then
          raise EProtocol.Create (ErrorCode, False);
      end;
    end;
  end;

  function TApxCustomProtocol.GetHonorDirectory : Boolean;
    {-Return the honor directory option}
  begin
    Result := ProtocolDriver.Flags and apHonorDirectory <> 0;
  end;

  procedure TApxCustomProtocol.SetHonorDirectory(const NewOpt : Boolean);
    {-Set the honordirectory option}
  begin
    with ProtocolDriver do
      if NewOpt <> (Flags and apHonorDirectory <> 0) then
        if NewOpt then
          Flags := Flags or apHonorDirectory
        else
          Flags := Flags and not apHonorDirectory;
  end;

  function TApxCustomProtocol.GetIncludeDirectory : Boolean;
    {-Return the includedirectory option}
  begin
    Result := ProtocolDriver.Flags and apIncludeDirectory <> 0;
  end;

  procedure TApxCustomProtocol.SetIncludeDirectory(const NewOpt : Boolean);
    {-Set the include directory option}
  begin
    with ProtocolDriver do
      if NewOpt <> (Flags and apIncludeDirectory <> 0) then
        if NewOpt then
          Flags := Flags or apIncludeDirectory
        else
          Flags := Flags and not apIncludeDirectory;
  end;

  function TApxCustomProtocol.GetRTSLowForWrite : Boolean;
    {-Return the RTSLowForWrite option}
  begin
    Result := ProtocolDriver.Flags and apRTSLowForWrite <> 0;
  end;

  procedure TApxCustomProtocol.SetRTSLowForWrite(const NewOpt : Boolean);
    {-Set the RTSLowForWrite option}
  begin
    with ProtocolDriver do
      if NewOpt <> (Flags and apRTSLowForWrite <> 0) then
        if NewOpt then
          Flags := Flags or apRTSLowForWrite
        else
          Flags := Flags and not apRTSLowForWrite;
  end;

  function TApxCustomProtocol.GetAbortNoCarrier : Boolean;
    {-Return the AbortNoCarrier option}
  begin
    Result := ProtocolDriver.Flags and apAbortNoCarrier <> 0;
  end;

  procedure TApxCustomProtocol.SetAbortNoCarrier(const NewOpt : Boolean);
    {-Set the AbortNoCarrier option}
  begin
    with ProtocolDriver do
      if NewOpt <> (Flags and apAbortNoCarrier <> 0) then
        if NewOpt then
          Flags := Flags or apAbortNoCarrier
        else
          Flags := Flags and not apAbortNoCarrier;
  end;

  function TApxCustomProtocol.GetAsciiSuppressCtrlZ : Boolean;
    {-Return the AscciSuppressCtrlZ option}
  begin
    Result := ProtocolDriver.Flags and apAsciiSuppressCtrlZ <> 0;
  end;

  procedure TApxCustomProtocol.SetAsciiSuppressCtrlZ(const NewOpt : Boolean);
    {-Set the AsciiSuppressCtrlZ option}
  begin
    with ProtocolDriver do
      if NewOpt <> (Flags and apAsciiSuppressCtrlZ <> 0) then
        if NewOpt then
          Flags := Flags or apAsciiSuppressCtrlZ
        else
          Flags := Flags and not apAsciiSuppressCtrlZ;
  end;

  procedure TApxCustomProtocol.SetBlockCheckMethod(
                               const NewMethod : TApxBlockCheckMethod);
    {-Set a new block check method}
  begin
    with ProtocolDriver do begin
      if (NewMethod <> TApxBlockCheckMethod(CheckType)) and
         (InProgress = 0) then begin
        case TApxProtocolType(ProtocolDriver.CurProtocol) of
           ptNoProtocol,
           ptXmodem1K, ptXmodem1KG,
           ptYmodem, ptYmodemG,
           ptAscii :
             {Don't change} ;
           ptXmodem :
             {Allow only switch to CRC}
             if NewMethod = bcmCrc16 then
               ProtocolType := ptXmodemCRC;
           ptXmodemCRC :
             {Allow only switch to checksum}
             if NewMethod = bcmChecksum then
               ProtocolType := ptXmodem;
           ptZmodem :
             {Allow only CRC16 and CRC32}
             case NewMethod of
               bcmCrc16, bcmCrc32 :
                 CheckType := Cardinal(NewMethod);
             end;
           ptKermit :
             {Allow only Kermit types}
             case NewMethod of
               bcmChecksum, bcmChecksum2, bcmCrcK :
                 CheckType := Cardinal(NewMethod);
             end;
        end;
      end;
    end;
  end;

  function TApxCustomProtocol.GetHandshakeRetry : Cardinal;
    {-Return the handshake retry count}
  begin
    Result := ProtocolDriver.HandshakeRetry;
  end;

  procedure TApxCustomProtocol.SetHandshakeRetry(const NewRetry : Cardinal);
    {-Set a new handshake retry value}
  begin
    with ProtocolDriver do
      if NewRetry <> HandshakeRetry then
        HandshakeRetry := NewRetry
  end;

  function TApxCustomProtocol.GetHandshakeWait : Cardinal;
    {-Return the handshake wait}
  begin
    Result := ProtocolDriver.HandshakeWait;
  end;

  procedure TApxCustomProtocol.SetHandshakeWait(const NewWait : Cardinal);
    {-Set the handshake wait}
  begin
    with ProtocolDriver do
      if NewWait <> HandshakeWait then
        HandshakeWait := NewWait;
  end;

  function TApxCustomProtocol.GetBlockLength : Cardinal;
    {-Return the current block length}
  begin
    if ProtocolDriver.InProgress <> 0 then begin
      Result := ProtocolDriver.LastBlockSize;
      if Result = 0 then
        Result := ProtocolDriver.BlockLen;
    end else                                                        
      Result := ProtocolDriver.BlockLen;
  end;

  function TApxCustomProtocol.GetBlockNumber : Cardinal;
    {-Return the current block number}
  begin
    Result := ProtocolDriver.BlockNum;
  end;

  function TApxCustomProtocol.GetTransmitTimeout : Cardinal;
    {-Return the current transmit timeout}
  begin
    Result := ProtocolDriver.TransTimeout;
  end;

  procedure TApxCustomProtocol.SetTransmitTimeout(const NewTimeout : Cardinal);
    {-Set a new transmit timeout}
  begin
    with ProtocolDriver do
      if NewTimeout <> TransTimeout then
        TransTimeout := NewTimeout;
  end;

  function TApxCustomProtocol.GetFinishWait : Cardinal;
    {-Return finish wait}
  begin
    Result := ProtocolDriver.FinishWait;
  end;

  procedure TApxCustomProtocol.SetFinishWait(const NewWait : Cardinal);
    {-Set new finish wait}
  begin
    with ProtocolDriver do
      if NewWait <> FinishWait then
        FinishWait := NewWait;
  end;

  function TApxCustomProtocol.GetActualBPS : LongInt;
    {-Return actual CPS for protocol}
  begin
    Result := LongInt(ProtocolDriver.ActCPS) * 10;                         
  end;

  procedure TApxCustomProtocol.SetActualBPS(const NewBPS : LongInt);
    {-Set the actual CPS for the protocol}
  begin
    with ProtocolDriver do begin
      NeedBPS := False;
      ProtocolDriver.apSetActualBPS (NewBPS); 
    end;
  end;

  function TApxCustomProtocol.GetTurnDelay : Cardinal;
    {-Return the turnaround delay}
  begin
    Result := ProtocolDriver.TurnDelay;
  end;

  procedure TApxCustomProtocol.SetTurnDelay(const NewDelay : Cardinal);
    {-Set the turnaround delay}
  begin
    with ProtocolDriver do
      if NewDelay <> TurnDelay then
        TurnDelay := NewDelay;
  end;

  function TApxCustomProtocol.GetOverhead : Cardinal;
    {-Return number of overhead bytes}
  begin
    Result := ProtocolDriver.Overhead;
  end;

  procedure TApxCustomProtocol.SetOverhead(const NewOverhead : Cardinal);
    {-Set new number of overhead bytes}
  begin
    with ProtocolDriver do
      if NewOverhead <> Overhead then
        Overhead := NewOverhead;
  end;

  function TApxCustomProtocol.GetWriteFailAction : TApxWriteFailAction;
    {-Return writefail action}
  begin
    Result := TApxWriteFailAction(ProtocolDriver.WriteFailOpt);
  end;

  procedure TApxCustomProtocol.SetWriteFailAction(
                               const NewAction : TApxWriteFailAction);
    {-Set a new write fail action}
  begin
    with ProtocolDriver do
      if Ord(NewAction) <> WriteFailOpt then
        WriteFailOpt := Ord(NewAction);
  end;

  procedure TApxCustomProtocol.apxProtocolAccept(CP : TObject;
                                                 var Accept : Boolean;
                                                 var FName : string);
    {-Internal event handling}
  begin
    Accept := True;
    if Assigned(FOnProtocolAccept) then
      FOnProtocolAccept(CP, Accept, FName);
  end;

  procedure TApxCustomProtocol.apxProtocolError(CP : TObject;
                                                ErrorCode : Integer);
    {-Internal event handling}
  begin
    if Assigned(FOnProtocolError) then
      FOnProtocolError(CP, ErrorCode);
  end;

  procedure TApxCustomProtocol.apxProtocolFinish(CP : TObject;
                                                 ErrorCode : Integer);
      {-Internal event handling}
  begin
    if Assigned(FOnProtocolFinish) then
      FOnProtocolFinish(CP, ErrorCode);
  end;

  procedure TApxCustomProtocol.apxProtocolLog (CP : TObject; Log : Cardinal);
    {-Internal event handling}
  begin
    {If LogProc is assigned then call it}
    if Assigned (FProtocolLog) then
      FProtocolLog.UpdateLog (Log);

    {If event handler is assigned then call it}
    if Assigned (FOnProtocolLog) then
      FOnProtocolLog (CP, Log);
  end;

  procedure TApxCustomProtocol.apxProtocolNextFile (CP : TObject;
                                                    var FName : string);
    {-Internal event handling}
  var
    P : array[0..255] of Char;
  begin
    if Assigned (FOnProtocolNextFile) then
      FOnProtocolNextFile (CP, FName)
    else begin
      FillChar (P, SizeOf (P), 0);
      if ProtocolDriver.apNextFileMask (P) then
        FName := StrPas (P)
      else
        FName := '';
    end;
  end;

  procedure TApxCustomProtocol.apxProtocolResume (CP : TObject;
                                                  var Resume : TApxWriteFailAction);
    {-Internal event handling}
  begin
    if Assigned (FOnProtocolResume) then
      FOnProtocolResume (CP, Resume)
  end;

  procedure TApxCustomProtocol.apxProtocolStatus(CP : TObject; Options : Cardinal);
    {-Internal event handling}
  begin
    {Automatically hand off to status display, if one is attached}
    if Assigned(FStatusDisplay) then
      StatusDisplay.UpdateDisplay(Options and apFirstCall <> 0,
                                  Options and apLastCall <> 0);

    {Call user's event handler}
    if Assigned(FOnProtocolStatus) then
      FOnProtocolStatus(CP, Options);
  end;

  procedure TApxCustomProtocol.apxProtocolOpenStream (    CP      : TObject;
                                                      var Stream  : TStream;
                                                          Reading : Boolean);
  begin
    if Assigned (FOnProtocolStreamOpen) then
      FOnProtocolStreamOpen (CP, Stream, Reading);
  end;

  procedure TApxCustomProtocol.apxProtocolCloseStream (    CP     : TObject;
                                                       var Stream : TStream);
  begin
    if Assigned (FOnProtocolStreamClose) then
      FOnProtocolStreamClose (CP, Stream);    
  end;

  constructor TApxCustomProtocol.Create(AOwner : TComponent);
    {-Create the object instance}
  var
    ErrorCode : Integer;
{$IFDEF TRIALRUN}
  {$I TRIAL06.INC}
{$ENDIF}
  begin
{$IFDEF TRIALRUN}
  TC;
{$ENDIF}
    inherited Create(AOwner);

    {Inits}

    { Create ASCII prot by default }
    
    ProtocolDriver := TApxASCIIDriver.Create (Self);

    Force                 := False;
    FComPort              := nil;
    FProtocolType         := ApxDefProtocolType;
    FStatusDisplay        := nil;
    FProtocolLog          := nil;
    FXYmodemBlockWait     := ApxDefXYmodemBlockWait;
    FZmodemFileOption     := ApxDefZmodemFileOption;
    FZmodemOptionOverride := ApxDefZmodemOptionOverride;
    FZmodemSkipNoFile     := ApxDefZmodemSkipNoFile;
    FZmodemRecover        := ApxDefZmodemRecover;
    FZmodem8K             := ApxDefZmodem8k;
    FZmodemZRQINITValue   := ApxDefZmodemZRQINITValue;
    FKermitMaxLen         := ApxDefKermitMaxLen;
    FKermitMaxWindows     := ApxDefKermitMaxWindows;
    FKermitSWCTurnDelay   := ApxDefKermitSWCTurnDelay;
    FKermitTimeoutSecs    := ApxDefKermitTimeoutSecs;
    FKermitTerminator     := ApxDefKermitTerminator;
    FKermitCtlPrefix      := ApxDefKermitCtlPrefix;
    FKermitHighbitPrefix  := ApxDefKermitHighbitPrefix;
    FKermitRepeatPrefix   := ApxDefKermitRepeatPrefix;
    FKermitPadCharacter   := ApxDefKermitPadCharacter;
    FKermitPadCount       := ApxDefKermitPadCount;
    FAsciiCharDelay       := ApxDefAsciiCharDelay;
    FAsciiLineDelay       := ApxDefAsciiLineDelay;
    FAsciiEOLChar         := ApxDefAsciiEOLChar;
    FAsciiCRTranslation   := ApxDefAsciiCRTranslation;
    FAsciiLFTranslation   := ApxDefAsciiLFTranslation;
    FAsciiEOFTimeout      := ApxDefAsciiEOFTimeout;

    {Create the protocol element}
    NeedBPS := True;

    ErrorCode := ProtocolDriver.apInitProtocolData (nil, DefProtocolOptions); 
    if (ErrorCode < ecOk) and
        not (csLoading in Self.ComponentState) then
      raise EProtocol.Create (ErrorCode, False);

    {PData inits}
    HandshakeWait      := ApxDefHandshakeWait;
    HandshakeRetry     := ApxDefHandshakeRetry;
    TransmitTimeout    := ApxDefTransTimeout;
    FinishWait         := ApxDefFinishWait;
    TurnDelay          := ApxDefTurnDelay;
    Overhead           := ApxDefOverhead;
    HonorDirectory     := ApxDefHonorDirectory;
    IncludeDirectory   := ApxDefIncludeDirectory;
    RTSLowForWrite     := ApxDefRTSLowForWrite;
    AbortNoCarrier     := ApxDefAbortNoCarrier;
    AsciiSuppressCtrlZ := ApxDefAsciiSuppressCtrlZ;
    UpcaseFileNames    := ApxDefUpcaseFileNames;

    {Option inits}
    with ProtocolDriver do begin
      Flags := 0;
      if ApxDefHonorDirectory then
        Flags := Flags or apHonorDirectory;
      if ApxDefIncludeDirectory then
        Flags := Flags or apIncludeDirectory;
      if ApxDefRTSLowForWrite then
        Flags := Flags or apRTSLowForWrite;
      if ApxDefAbortNoCarrier then
        Flags := Flags or apAbortNoCarrier;
      if ApxDefAsciiSuppressCtrlZ then
        Flags := Flags or apAsciiSuppressCtrlZ;
    end;

    {Search for comport}
    FComPort := SearchComPort(Owner);

    {Search for protocol status display}
    StatusDisplay := SearchStatusDisplay(Owner);

    {Search for protocol log}
    FProtocolLog := SearchProtocolLog(Owner);
    if Assigned(FProtocolLog) then
      ProtocolLog.FProtocol := Self;

    {Create the message handler instance...}
    if not (csDesigning in ComponentState) then begin
      FProtocolType := ptNoProtocol;
    end else begin
      FProtocolType := ptNoProtocol;
      SetProtocolType(ApxDefProtocolType);
    end;
  end;

  destructor TApxCustomProtocol.Destroy;
    {-Destroy the object instance}
  var
    I : Cardinal;
    P : PProtocolWindowNode;
  begin
    {Get rid of msg handler window and node}
    if not (csDesigning in ComponentState) then
      with ProtList do
        if Count > 0 then
          for I := 0 to Count-1 do begin
            P := PProtocolWindowNode(Items[I]);
            if P^.pwProtocol = Self then begin
              Remove(Items[I]);
              Dispose(P);
              break;
            end;
          end;

    ProtocolType := ptNoProtocol;
    ProtocolDriver.apDoneProtocol;

    inherited Destroy;
  end;

  class function TApxCustomProtocol.GetLogString(const D1, D2, D3: DWORD): string;
  begin
    if (D1 > 100000) then
      Result := 'Protocol Error: ' + MessageNumberToString (D1 - 100000) + ' :: ' +
                MessageNumberToString (D2 + AxdstNone) + ' :: ' +
                MessageNumberToString (D3) + ' (' + IntToStr (D3) + ')'
    else
      Result := 'Protocol: ' + MessageNumberToString (D1) + ' :: State = ' +
                MessageNumberToString (D2 + AxdstNone);
  end;

  procedure TApxCustomProtocol.Assign(Source : TPersistent);
    {-Assign Source to Self}
  var
    SrcType : TApxProtocolType;

  begin
    if Source is TApxCustomProtocol then begin
      {Note the current protocol type, then force deallocation of
       internal pointers}
      SrcType := TApxCustomProtocol(Source).ProtocolType;
      TApxCustomProtocol(Source).ProtocolType := ptNoProtocol;

      {Force the destination to noprotocol as well}
      ProtocolType := ptNoProtocol;

      {Free the existing critsection pointer}
      DeleteCriticalSection (ProtocolDriver.FProtSection);

      {Overwrite the copied critsection pointer with our own}
      InitializeCriticalSection (ProtocolDriver.FProtSection);

      NeedBPS                := TApxCustomProtocol(Source).NeedBPS;
      Force                  := TApxCustomProtocol(Source).Force;
      FMsgHandler            := TApxCustomProtocol(Source).FMsgHandler;
      FComPort               := TApxCustomProtocol(Source).FComPort;
      FStatusDisplay         := TApxCustomProtocol(Source).FStatusDisplay;
      FProtocolLog           := TApxCustomProtocol(Source).FProtocolLog;
      FXYmodemBlockWait      := TApxCustomProtocol(Source).FXYmodemBlockWait;
      FZmodemOptionOverride  := TApxCustomProtocol(Source).FZmodemOptionOverride;
      FZmodemSkipNoFile      := TApxCustomProtocol(Source).FZmodemSkipNoFile;
      FZmodemFileOption      := TApxCustomProtocol(Source).FZmodemFileOption;
      FZmodemRecover         := TApxCustomProtocol(Source).FZmodemRecover;
      FZmodem8K              := TApxCustomProtocol(Source).FZmodem8K;
      FZmodemZRQINITValue    := TApxCustomProtocol(Source).FZmodemZRQINITValue;
      FKermitMaxLen          := TApxCustomProtocol(Source).FKermitMaxLen;
      FKermitMaxWindows      := TApxCustomProtocol(Source).FKermitMaxWindows;
      FKermitSWCTurnDelay    := TApxCustomProtocol(Source).FKermitSWCTurnDelay;
      FKermitTimeoutSecs     := TApxCustomProtocol(Source).FKermitTimeoutSecs;
      FKermitTerminator      := TApxCustomProtocol(Source).FKermitTerminator;
      FKermitCtlPrefix       := TApxCustomProtocol(Source).FKermitCtlPrefix;
      FKermitHighbitPrefix   := TApxCustomProtocol(Source).FKermitHighbitPrefix;
      FKermitRepeatPrefix    := TApxCustomProtocol(Source).FKermitRepeatPrefix;
      FAsciiCharDelay        := TApxCustomProtocol(Source).FAsciiCharDelay;
      FAsciiLineDelay        := TApxCustomProtocol(Source).FAsciiLineDelay;
      FAsciiEOLChar          := TApxCustomProtocol(Source).FAsciiEOLChar;
      FAsciiCRTranslation    := TApxCustomProtocol(Source).FAsciiCRTranslation;
      FAsciiLFTranslation    := TApxCustomProtocol(Source).FAsciiLFTranslation;
      FAsciiEOFTimeout       := TApxCustomProtocol(Source).FAsciiEOFTimeout;
      FOnProtocolAccept      := TApxCustomProtocol(Source).FOnProtocolAccept;
      FOnProtocolError       := TApxCustomProtocol(Source).FOnProtocolError;
      FOnProtocolFinish      := TApxCustomProtocol(Source).FOnProtocolFinish;
      FOnProtocolLog         := TApxCustomProtocol(Source).FOnProtocolLog;
      FOnProtocolNextFile    := TApxCustomProtocol(Source).FOnProtocolNextFile;
      FOnProtocolResume      := TApxCustomProtocol(Source).FOnProtocolResume;
      FOnProtocolStatus      := TApxCustomProtocol(Source).FOnProtocolStatus;
      FOnProtocolStreamOpen  := TApxCustomProtocol(Source).FOnProtocolStreamOpen;
      FOnProtocolStreamClose := TApxCustomProtocol(Source).FOnProtocolStreamClose;

      {Set both protocols back to the src protocol type}

      ProtocolType           := SrcType;
      TApxCustomProtocol(Source).ProtocolType := SrcType;
    end;
  end;

  function TApxCustomProtocol.EstimateTransferSecs(const Size : Longint) : Longint;
    {-Returns time to send Size bytess}
  begin
    Result := ProtocolDriver.apEstimateTransferSecs(Size);
  end;

  function TApxCustomProtocol.StatusMsg(const Status : Cardinal) : String;
    {-Return a resource string for Status}
  var
    P : array[0..MaxMessageLen] of Char;
  begin
    ProtocolDriver.apStatusMsg(P, Status);
    Result := StrPas(P);
  end;

  procedure TApxCustomProtocol.TransmitPortMessageHandler (Msg, wParam : Cardinal;
                      lParam : LongInt);
  begin
    AxPostMessage (ProtocolDriver.Handle,
                  (Msg or APXTRANSMITMESSAGEMASK), nil, wParam, lParam);
  end;

  procedure TApxCustomProtocol.ReceivePortMessageHandler (Msg, wParam : Cardinal;
                      lParam : LongInt);
  begin
    AxPostMessage (ProtocolDriver.Handle, Msg, nil, wParam, lParam);
  end;

  procedure TApxCustomProtocol.RegisterComHook (Transmitting : Boolean);
  begin
    if not Assigned (FComPort) then
      raise EProtocol.Create (AxecPortNotAssigned, False);
    if not Assigned (ProtocolDriver) then
      raise EProtocol.CreateUnknown ('Protocol driver not assigned', $0); 

    if Transmitting then
      FComPort.Dispatcher.RegisterEventTriggerHandler(TransmitPortMessageHandler)
    else
      FComPort.Dispatcher.RegisterEventTriggerHandler(ReceivePortMessageHandler);
  end;

  procedure TApxCustomProtocol.DeRegisterComHook;
  begin
    if not Assigned (FComPort) then
      raise EProtocol.Create (AxecPortNotAssigned, False);
    FComPort.Dispatcher.DeregisterEventTriggerHandler (TransmitPortMessageHandler);
    FComPort.Dispatcher.DeregisterEventTriggerHandler (ReceivePortMessageHandler);
  end;

  procedure TApxCustomProtocol.StartTransmit;
    { -Start a transmit protocol session }
  begin
    FTransmitting := True;
    with ProtocolDriver do begin
      {Exit quitely if no protocol selected}
      if FProtocolType = ptNoProtocol then
        Exit;

      { Make sure comport is open, pass handle to protocol }
      CheckPort;
      if Assigned(FComPort) then
        ProtocolDriver.apSetProtocolPort(FComPort)
      else
        raise EPortNotAssigned.Create(ecPortNotAssigned, False);

      {Set the protocol's aActCPS field if it isn't already set}
      if NeedBPS then with FComPort do begin
        ActualBPS := Baud;
        NeedBPS := False;
      end;

      { Start the protocol }
      { What is really happening here is subtle.

        First call the apStartProtocol method of the protocol driver (the
        driver class reflects what protocol is selected).  apStartProtocol
        will call either the Transmit or Receive methods.  These methods are
        nothing more than message handlers containing the protocol state
        machine.

        The first time the Transmit or Receive methods are called (via
        apStartProtocol) the Message type is 0.  This forces the state machine
        to initialize.

        Once the method completes, RegisterComHook is called to register an
        event trigger handler with the dispatcher.  The handler for this will
        post a message to either the Transmit or Receive methods of the
        protocol driver.  The message type is provided from the dispatcher.
        The protocol state machine encompassed in Transmit or Receive may do
        some odd things or take a while to exit.  Directly calling it from the
        com hook can cause thread deadlocks.

        To let the component know what is going on, the ProtocolDriver will
        either post messages back to the TApxCustomProtocol or directly call
        methods in the component (it depends on the situation).

        At the end of the protocol, the com hook is deregistered.
      }
      if (FProtocolType >= ptNoProtocol) and
         (FProtocolType <= ptAscii) then begin
        ProtocolDriver.apStartProtocol(Ord(FProtocolType), True);
        RegisterComHook (True);
      end;
    end;
  end;

  procedure TApxCustomProtocol.StartReceive;
    { -Start a protocol receive session }
  begin
    FTransmitting := False;
    with ProtocolDriver do begin
      { Exit quietly if no protocol selected }
      if FProtocolType = ptNoProtocol then
        Exit;

      { Make sure comport is open, pass handle to protocol }
      if Assigned(FComPort) then
        ProtocolDriver.apSetProtocolPort(FComPort)
      else
        raise EPortNotAssigned.Create(ecPortNotAssigned, False);

      { Set the protocol's aActCPS field if it isn't already set }
      if NeedBPS then with FComPort do begin
        ActualBPS := Baud;
        NeedBPS := False;
      end;

      { Start the protocol }
      { Look in StartTransmit for details on what is going on here }
      if (FProtocolType >= ptNoProtocol) and
         (FProtocolType <= ptAscii) then begin 
        ProtocolDriver.apStartProtocol(Ord(FProtocolType), False);
        RegisterComHook (False);
      end;
    end;
  end;

  procedure TApxCustomProtocol.CancelProtocol;
    {-Sends apx_ProtocolCancel message to protocol function}
  begin
    if FTransmitting then
      ProtocolDriver.Transmit (APX_PROTOCOLCANCEL, 0, 0)
    else
      ProtocolDriver.Receive (APX_PROTOCOLCANCEL, 0, 0);
  end;

{TApxAbstractStatus}

  procedure TApxAbstractStatus.Notification(AComponent : TComponent;
                                            Operation: TOperation);
  begin
    inherited Notification(AComponent, Operation);

    if Operation = opRemove then begin
      if AComponent = FProtocol then
        FProtocol := nil;
    end;
  end;

  procedure TApxAbstractStatus.SetPosition(const NewPosition : TPosition);
  begin
    if NewPosition <> FPosition then begin
      FPosition := NewPosition;
      if Assigned(FDisplay) then
        FDisplay.Position := NewPosition;
    end;
  end;

  procedure TApxAbstractStatus.SetCtl3D(const NewCtl3D : Boolean);
  begin
    if NewCtl3D <> FCtl3D then begin
      FCtl3D := NewCtl3D;
    end;
  end;

  procedure TApxAbstractStatus.SetVisible(const NewVisible : Boolean);
  begin
    if NewVisible <> FVisible then begin
      FVisible := NewVisible;
      if Assigned(FDisplay) then
        FDisplay.Visible := NewVisible;
    end;
  end;

  procedure TApxAbstractStatus.SetCaption(const NewCaption : TCaption);
  begin
    if NewCaption <> FCaption then begin
      FCaption := NewCaption;
      if Assigned(FDisplay) then
        FDisplay.Caption := NewCaption;
    end;
  end;                                                                   

  procedure TApxAbstractStatus.GetProperties;
  begin
    if Assigned(FDisplay) then begin
      Position := FDisplay.Position;
      Visible  := FDisplay.Visible;
      Caption  := FDisplay.Caption;                                 
    end;
  end;

  constructor TApxAbstractStatus.Create(AOwner : TComponent);
  begin
    inherited Create(AOwner);
    CreateDisplay;
    GetProperties;
    Caption := 'Protocol Status';                                  
  end;

  destructor TApxAbstractStatus.Destroy;
  begin
    DestroyDisplay;
    inherited Destroy;
  end;

  procedure TApxAbstractStatus.Show;
  begin
    if Assigned(FDisplay) then
      FDisplay.Show;
  end;

{TApxProtocolLog}

  procedure TApxProtocolLog.Notification(AComponent : TComponent;
                                         Operation: TOperation);
  begin
    inherited Notification(AComponent, Operation);

    if Operation = opRemove then begin
      {Owned components going away}
      if AComponent = FProtocol then
        FProtocol := nil;
    end;
  end;

  constructor TApxProtocolLog.Create(AOwner : TComponent);
  begin
    inherited Create(AOwner);

    {Inits}
    HistoryName := awpDefHistoryName;
    DeleteFailed := awpDefDeleteFailed;
  end;

  procedure TApxProtocolLog.UpdateLog(const Log : Cardinal);
    {-Update the standard log}
  var
    CPS : Cardinal;
    HisFile : TextFile;

    function DirectionString : String;
    begin
      case Log of
        lfReceiveStart, lfReceiveOk, lfReceiveSkip, lfReceiveFail :
          DirectionString := ' receive ';
        lfTransmitStart, lfTransmitOk, lfTransmitFail, lfTransmitSkip :
          DirectionString := ' transmit ';
      end;
    end;

  begin
    {Exit if no name specified}
    if FHistoryName = '' then
      Exit;

    {Create or open the history file}
    try
      AssignFile(HisFile, FHistoryName);
      Append(HisFile);
    except
      on E : EInOutError do
        if E.ErrorCode = 2 then
          {File not found, open as new}
          Rewrite(HisFile)
        else
          {Unexpected error, forward the exception}
          raise;
    end;

    {Write the log entry}
    with Protocol, ProtocolDriver do begin
      case Log of
        lfReceiveStart,
        lfTransmitStart :
          WriteLn(HisFile, ProtocolName(ProtocolType),
                           DirectionString,
                           'started on  ',
                           DateTimeToStr(Now), ' : ',
                           FileName);

        lfReceiveOk,
        lfTransmitOK :
          begin
            WriteLn(HisFile, ProtocolName(ProtocolType),
                             DirectionString,
                             'finished OK ',
                             DateTimeToStr(Now), ' : ',
                             FileName);
            if ElapsedXfrTime < 1000 then   
              CPS := FileLength
            else
              CPS := (FileLength div (ElapsedXfrTime div 1000));
            WriteLn(HisFile, '   Elapsed time: ',
                             FormatMinSec (ElapsedXfrTime div 1000),
                          '   CPS: ', CPS,
                          '   Size: ', FileLength, ^M^J);
          end;

        lfReceiveSkip :
          WriteLn(HisFile, ProtocolName(ProtocolType),
                           ' receive skipped ', FileName, ' ',
                           StatusMsg(ProtocolStatus), ^M^J);

        lfReceiveFail :
          begin
            WriteLn(HisFile, ProtocolName(ProtocolType),
                             ' receive failed ', FileName, ' ',
                             StatusMsg(ProtocolStatus), ^M^J);

            case FDeleteFailed of
              dfNever  :
                {Leave the partial file intact} ;
              dfAlways :
                if ProtocolError <> ecCantWriteFile then
                  DeleteFile(FileName);
              dfNonRecoverable :
                if (ProtocolType <> ptZmodem) and
                   (ProtocolError <> ecCantWriteFile) then
                  DeleteFile(FileName);
            end;
          end;

        lfTransmitFail :
          WriteLn(HisFile, ProtocolName(ProtocolType),
                           ' transmit failed ', FileName, ' ',
                           StatusMsg(ProtocolStatus), ^M^J);
        lfTransmitSkip :
          WriteLn(HisFile, ProtocolName(ProtocolType),
                           ' transmit skipped ', FileName, ' ',
                           StatusMsg(ProtocolStatus), ^M^J);
      end;
    end;

    Close(HisFile);
    if IOResult <> 0 then ;
  end;

{Miscellaneous functions}

  function ProtocolName(const ProtocolType : TApxProtocolType) : String;
    {-Return a string of the protocol type}
  begin
    if (ProtocolType >= ptNoProtocol) and (ProtocolType <= ptAscii) then 
      Result := StrPas(ProtocolString[Ord(ProtocolType)])
    else
      Result := StrPas(ProtocolString[0]);
  end;

  function CheckNameString(const Check : TApxBlockCheckMethod) : String;
    {-Return a string of the block check type}
  begin
    case Check of
      bcmChecksum   : Result := bcsChecksum1;
      bcmChecksum2  : Result := bcsChecksum2;
      bcmCrc16      : Result := bcsCrc16;
      bcmCrc32      : Result := bcsCrc32;
      bcmCrcK       : Result := bcsCrcK;
      else            Result := bcsNone;
    end;
  end;

  function FormatMinSec(const TotalSecs : LongInt) : String;
    {-Format TotalSecs as minutes:seconds, leftpadded to 6}
  var
    Min, Sec : LongInt;
    S : String;
  begin
    Min := TotalSecs div 60;
    Sec := TotalSecs mod 60;
    Str(Sec:2, S);
    if S[1] = ' ' then
      S[1] := '0';
    FormatMinSec := LeftPad(IntToStr(Min) + ':' + S, 6);
  end;

initialization
  ProtList := TList.Create;

finalization
  ProtList.Free;
end.

