(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                  AxPrtDrv.pas 1.02                    *}
{*********************************************************}

{Global defines potentially affecting this unit}
{$I AxDefine.inc}

unit AxPrtDrv;

{$I-}

interface

uses
  Types,
  Classes,
  SysUtils,
  LibC,
  AxSystem,
  AxMisc,
  AxPort,
  AxString,
  AxExcept,
  Qt;

const                               

  APXTRANSMITMESSAGEMASK = Cardinal ($40000000);

  ApxFileBufferSize        = 8192;
  ApxDefHandshakeWait      = 10000;
  ApxDefHandshakeRetry     = 10;
  ApxDefTransTimeout       = 30000;
  ApxMaxBlockSize          = 1024;
  ApxBlockFillChar : Char  = ^Z;
  ApxTelixDelay            = 500;

  ApxMaxWindowSlots        = 27;

  AoxDefStatusInterval     = 1000;
  {Compile-time options}
  ApxDefAbsStatusInterval  = 1000;   { 1 second between status updates }

var
  { UnixDaysBase gets "overwritten" in the this unit's init code for Kylix }
  UnixDaysBase : LongInt = 719163; { Days between 1/1/1 and 1/1/1970 }

const
  SecsPerDay = 86400;              { Number of seconds in one day }

const
  ProtocolDataPtr = 1;

  Crc32Table : array[0..255] of DWORD = (                            
  $00000000, $77073096, $ee0e612c, $990951ba, $076dc419, $706af48f, $e963a535,
  $9e6495a3, $0edb8832, $79dcb8a4, $e0d5e91e, $97d2d988, $09b64c2b, $7eb17cbd,
  $e7b82d07, $90bf1d91, $1db71064, $6ab020f2, $f3b97148, $84be41de, $1adad47d,
  $6ddde4eb, $f4d4b551, $83d385c7, $136c9856, $646ba8c0, $fd62f97a, $8a65c9ec,
  $14015c4f, $63066cd9, $fa0f3d63, $8d080df5, $3b6e20c8, $4c69105e, $d56041e4,
  $a2677172, $3c03e4d1, $4b04d447, $d20d85fd, $a50ab56b, $35b5a8fa, $42b2986c,
  $dbbbc9d6, $acbcf940, $32d86ce3, $45df5c75, $dcd60dcf, $abd13d59, $26d930ac,
  $51de003a, $c8d75180, $bfd06116, $21b4f4b5, $56b3c423, $cfba9599, $b8bda50f,
  $2802b89e, $5f058808, $c60cd9b2, $b10be924, $2f6f7c87, $58684c11, $c1611dab,
  $b6662d3d, $76dc4190, $01db7106, $98d220bc, $efd5102a, $71b18589, $06b6b51f,
  $9fbfe4a5, $e8b8d433, $7807c9a2, $0f00f934, $9609a88e, $e10e9818, $7f6a0dbb,
  $086d3d2d, $91646c97, $e6635c01, $6b6b51f4, $1c6c6162, $856530d8, $f262004e,
  $6c0695ed, $1b01a57b, $8208f4c1, $f50fc457, $65b0d9c6, $12b7e950, $8bbeb8ea,
  $fcb9887c, $62dd1ddf, $15da2d49, $8cd37cf3, $fbd44c65, $4db26158, $3ab551ce,
  $a3bc0074, $d4bb30e2, $4adfa541, $3dd895d7, $a4d1c46d, $d3d6f4fb, $4369e96a,
  $346ed9fc, $ad678846, $da60b8d0, $44042d73, $33031de5, $aa0a4c5f, $dd0d7cc9,
  $5005713c, $270241aa, $be0b1010, $c90c2086, $5768b525, $206f85b3, $b966d409,
  $ce61e49f, $5edef90e, $29d9c998, $b0d09822, $c7d7a8b4, $59b33d17, $2eb40d81,
  $b7bd5c3b, $c0ba6cad, $edb88320, $9abfb3b6, $03b6e20c, $74b1d29a, $ead54739,
  $9dd277af, $04db2615, $73dc1683, $e3630b12, $94643b84, $0d6d6a3e, $7a6a5aa8,
  $e40ecf0b, $9309ff9d, $0a00ae27, $7d079eb1, $f00f9344, $8708a3d2, $1e01f268,
  $6906c2fe, $f762575d, $806567cb, $196c3671, $6e6b06e7, $fed41b76, $89d32be0,
  $10da7a5a, $67dd4acc, $f9b9df6f, $8ebeeff9, $17b7be43, $60b08ed5, $d6d6a3e8,
  $a1d1937e, $38d8c2c4, $4fdff252, $d1bb67f1, $a6bc5767, $3fb506dd, $48b2364b,
  $d80d2bda, $af0a1b4c, $36034af6, $41047a60, $df60efc3, $a867df55, $316e8eef,
  $4669be79, $cb61b38c, $bc66831a, $256fd2a0, $5268e236, $cc0c7795, $bb0b4703,
  $220216b9, $5505262f, $c5ba3bbe, $b2bd0b28, $2bb45a92, $5cb36a04, $c2d7ffa7,
  $b5d0cf31, $2cd99e8b, $5bdeae1d, $9b64c2b0, $ec63f226, $756aa39c, $026d930a,
  $9c0906a9, $eb0e363f, $72076785, $05005713, $95bf4a82, $e2b87a14, $7bb12bae,
  $0cb61b38, $92d28e9b, $e5d5be0d, $7cdcefb7, $0bdbdf21, $86d3d2d4, $f1d4e242,
  $68ddb3f8, $1fda836e, $81be16cd, $f6b9265b, $6fb077e1, $18b74777, $88085ae6,
  $ff0f6a70, $66063bca, $11010b5c, $8f659eff, $f862ae69, $616bffd3, $166ccf45,
  $a00ae278, $d70dd2ee, $4e048354, $3903b3c2, $a7672661, $d06016f7, $4969474d,
  $3e6e77db, $aed16a4a, $d9d65adc, $40df0b66, $37d83bf0, $a9bcae53, $debb9ec5,
  $47b2cf7f, $30b5ffe9, $bdbdf21c, $cabac28a, $53b39330, $24b4a3a6, $bad03605,
  $cdd70693, $54de5729, $23d967bf, $b3667a2e, $c4614ab8, $5d681b02, $2a6f2b94,
  $b40bbe37, $c30c8ea1, $5a05df1b, $2d02ef8d
  );

const
  CrcTable: array[0..255] of Cardinal = (
    $0000,  $1021,  $2042,  $3063,  $4084,  $50a5,  $60c6,  $70e7,
    $8108,  $9129,  $a14a,  $b16b,  $c18c,  $d1ad,  $e1ce,  $f1ef,
    $1231,  $0210,  $3273,  $2252,  $52b5,  $4294,  $72f7,  $62d6,
    $9339,  $8318,  $b37b,  $a35a,  $d3bd,  $c39c,  $f3ff,  $e3de,
    $2462,  $3443,  $0420,  $1401,  $64e6,  $74c7,  $44a4,  $5485,
    $a56a,  $b54b,  $8528,  $9509,  $e5ee,  $f5cf,  $c5ac,  $d58d,
    $3653,  $2672,  $1611,  $0630,  $76d7,  $66f6,  $5695,  $46b4,
    $b75b,  $a77a,  $9719,  $8738,  $f7df,  $e7fe,  $d79d,  $c7bc,
    $48c4,  $58e5,  $6886,  $78a7,  $0840,  $1861,  $2802,  $3823,
    $c9cc,  $d9ed,  $e98e,  $f9af,  $8948,  $9969,  $a90a,  $b92b,
    $5af5,  $4ad4,  $7ab7,  $6a96,  $1a71,  $0a50,  $3a33,  $2a12,
    $dbfd,  $cbdc,  $fbbf,  $eb9e,  $9b79,  $8b58,  $bb3b,  $ab1a,
    $6ca6,  $7c87,  $4ce4,  $5cc5,  $2c22,  $3c03,  $0c60,  $1c41,
    $edae,  $fd8f,  $cdec,  $ddcd,  $ad2a,  $bd0b,  $8d68,  $9d49,
    $7e97,  $6eb6,  $5ed5,  $4ef4,  $3e13,  $2e32,  $1e51,  $0e70,
    $ff9f,  $efbe,  $dfdd,  $cffc,  $bf1b,  $af3a,  $9f59,  $8f78,
    $9188,  $81a9,  $b1ca,  $a1eb,  $d10c,  $c12d,  $f14e,  $e16f,
    $1080,  $00a1,  $30c2,  $20e3,  $5004,  $4025,  $7046,  $6067,
    $83b9,  $9398,  $a3fb,  $b3da,  $c33d,  $d31c,  $e37f,  $f35e,
    $02b1,  $1290,  $22f3,  $32d2,  $4235,  $5214,  $6277,  $7256,
    $b5ea,  $a5cb,  $95a8,  $8589,  $f56e,  $e54f,  $d52c,  $c50d,
    $34e2,  $24c3,  $14a0,  $0481,  $7466,  $6447,  $5424,  $4405,
    $a7db,  $b7fa,  $8799,  $97b8,  $e75f,  $f77e,  $c71d,  $d73c,
    $26d3,  $36f2,  $0691,  $16b0,  $6657,  $7676,  $4615,  $5634,
    $d94c,  $c96d,  $f90e,  $e92f,  $99c8,  $89e9,  $b98a,  $a9ab,
    $5844,  $4865,  $7806,  $6827,  $18c0,  $08e1,  $3882,  $28a3,
    $cb7d,  $db5c,  $eb3f,  $fb1e,  $8bf9,  $9bd8,  $abbb,  $bb9a,
    $4a75,  $5a54,  $6a37,  $7a16,  $0af1,  $1ad0,  $2ab3,  $3a92,
    $fd2e,  $ed0f,  $dd6c,  $cd4d,  $bdaa,  $ad8b,  $9de8,  $8dc9,
    $7c26,  $6c07,  $5c64,  $4c45,  $3ca2,  $2c83,  $1ce0,  $0cc1,
    $ef1f,  $ff3e,  $cf5d,  $df7c,  $af9b,  $bfba,  $8fd9,  $9ff8,
    $6e17,  $7e36,  $4e55,  $5e74,  $2e93,  $3eb2,  $0ed1,  $1ef0
  );
  
type
  TApxProcessBlockStart    = (pbsNone, pbs128, pbs1024, pbsCancel, pbsEOT);

  PApxDataBlock            = ^TApxDataBlock;
  TApxDataBlock            = array [1..ApxMaxBlockSize] of Char;

  PApxWorkBlock            = ^TApxWorkBlock;
  TApxWorkBlock            = array [1..2*ApxMaxBlockSize] of Char;

  TApxPosFlags             = array [0..3] of Byte;

  PApxFileBuffer           = ^TApxFileBuffer;
  TApxFileBuffer           = array [0..ApxFileBufferSize - 1] of Byte;

type

  TApxBaseProtocolDriver = class (TApxBaseHandleComponent)
    private
    protected

      { Trigger Handles }

      FStatusTrigger      : Integer;
      FTimeoutTrigger     : Integer;
      FOutBuffFreeTrigger : Integer;
      FOutBuffUsedTrigger : Integer;
      FNoCarrierTrigger   : Integer;

      FComPort            : TApxCustomComport;         
      FBatchProtocol      : Boolean;
      FFilesSent          : Boolean;
      FAbortFlag          : Boolean;
      FTimerStarted       : Boolean;

      FCurProtocol        : Integer;
      FCheckType          : Cardinal;
      FHandshakeRetry     : Cardinal;
      FHandshakeWait      : Cardinal;
      FHandshakeAttempt   : Cardinal;
      FBlockLen           : Cardinal;
      FBlockNum           : Cardinal;
      FFlags              : Cardinal;
      FTransTimeout       : Cardinal;
      FFinishWait         : Cardinal;
      FRcvTimeout         : Cardinal;
      FProtocolStatus     : Cardinal;
      FLastBlockSize      : Cardinal;
      FProtocolError      : Integer;
      FSrcFileLen         : LongInt;
      FSrcFileDate        : Longint;
      FBlockCheck         : DWORD;
      FInitFilePos        : LongInt;
      FReplyTimer         : EventTimer;
      FDataBlock          : PApxDataBlock;

      { Status }
      
      FStatusTimer        : EventTimer;      {How often to show status}
      FForceStatus        : Boolean;
      FTimerPending       : Boolean;
      FInProgress         : Cardinal;
      FBlockErrors        : Cardinal;
      FTotalErrors        : Cardinal;
      FActCPS             : Cardinal;
      FOverhead           : Cardinal;
      FTurnDelay          : Cardinal;
      FStatusInterval     : Cardinal;
      FSaveStatus         : Cardinal;
      FSaveError          : Integer;
      FBytesRemaining     : LongInt;
      FBytesTransferred   : LongInt;
      FElapsedXfrTime     : LongInt;
      FStatusTime         : EventTimer;
      FTimer              : EventTimer;

      { File Buffer Managment }

      FEndPending         : Boolean;
      FFileOpen           : Boolean;
      FNoMoreData         : Boolean;
      FLastBlock          : Boolean;
      FBlkIndex           : Cardinal;
      FWriteFailOpt       : Integer;
      FStartOfs           : LongInt;
      FEndOfs             : LongInt;
      FLastOfs            : LongInt;
      FFileOfs            : LongInt;
      FEndOfDataOfs       : LongInt;
      FFileBuffer         : PApxFileBuffer;             
      FSaveMode           : Cardinal;

      { For getting the next file to transmit }

      FUpcaseFileNames    : Boolean;
      FFindingFirst       : Boolean;
      FFileListIndex      : Cardinal;
      FPathName           : TPathCharArray;
      FSearchMask         : TPathCharArray;
      FFileList           : PFileList;
      FCurRec             : TSearchRec;
      FFFOpen             : Boolean;

      { I doubt this is needed.  The struct has been modified. }

      FUseStreams         : Boolean;                  

      { Large structures }

      FWorkFile           : File;
      FWorkStream         : TStream;
      FDestDir            : TDirCharArray;            

    protected
      function EventFilter(Sender : QObjectH; Event : QEventH) : Boolean; override;
      procedure CreateWidget; override;

      procedure SetTimer (V : EventTimer);
      procedure SetPathName (V : TPathCharArray);

      {Constructors/destructors}

      {Data conversion routines}

      function  apTrimZeros                   (S: string): String;
      function  apOctalStr                    (L : LongInt) : String;
      function  apOctalStr2Long               (S : String) : LongInt;
      function  apPackToYMTimeStamp           (RawTime : LongInt) : LongInt;
      function  apCurrentTimeStamp : LongInt;
      function  apYMTimeStampToPack           (YMTime : LongInt) : LongInt;

      function  apUpdateCrc32                 (CurByte : Byte;
                                               CurCrc  : LongInt) : LongInt;

      {Next file}
      procedure apSetFileList                 (FL : PFileList);
      function  apMakeFileList                (var FL   : PFileList;
                                                   Size : Cardinal) : Integer;
      procedure apDisposeFileList             (FL   : PFileList;
                                               Size : Cardinal);
      function  apAddFileToList(FL : PFileList; PName : PChar) : Integer;
      function  apNextFileList(FName : PChar) : Boolean;

      {Protocol options}
      procedure apSetFileMask(NewMask : PChar);
      procedure apSetDestinationDirectory(Dir : PChar);
      procedure apSetHandshakeWait(NewHandshake, NewRetry : Cardinal);
      procedure apSetOverwriteOption(Opt : Cardinal);
      procedure apSetEfficiencyParms(BlockOverhead, TurnAroundDelay : Cardinal);
      procedure apSetStatusInterval(NewInterval : Cardinal);
      procedure apOptionsOn(OptionFlags : Cardinal);
      procedure apOptionsOff(OptionFlags : Cardinal);
      function  apOptionsAreOn(OptionFlags : Cardinal): Boolean;

      {Status methods}
      function  apSupportsBatch : Boolean;
      function  apGetInitialFilePos : LongInt;
      procedure apGetProtocolInfo(var Info : TProtocolInfo);

      {Finish control}
      procedure apSignalFinish (Transmitting : Boolean);

      {Default file handling procedures}
      procedure aapPrepareReading;
      procedure aapFinishReading;
      function aapReadProtocolBlock(var Block : TApxDataBlock;
                              var BlockSize : Cardinal) : Boolean;
      procedure aapPrepareWriting;
      procedure aapFinishWriting;
      function aapWriteProtocolBlock(var Block : TApxDataBlock;
                               BlockSize : Cardinal) : Boolean;

      {Internal routines}
      procedure apStopProtocol (Transmitting : Boolean);
      function apCrc32OfFile(FName : PChar; Len : Longint) : LongInt;
      procedure apResetStatus;
      procedure apShowFirstStatus;
      procedure apShowLastStatus;

      {Default hooks, send messages to parent window}
      procedure apMsgStatus (Options : Cardinal);
      function apMsgNextFile(var FName : TPathCharArray) : Boolean;
      procedure apMsgLog(Log : Cardinal);
      function apMsgAcceptFile(FName : PChar) : Boolean;
      function apMsgOpenStream (var Stream  : TStream;
                                    Reading : Boolean) : Boolean;
      function apMsgCloseStream (var Stream : TStream) : Boolean;

      {All errors routed through here}
      procedure apProtocolError(ErrorCode : Integer);

      {General}
      function apUpdateChecksum(CurByte : Byte; CheckSum : Cardinal) : Cardinal;
      function apUpdateCrc(CurByte : Byte; CurCrc : Cardinal) : Cardinal;
      function apUpdateCrcKermit(CurByte : Byte; CurCrc : Cardinal) : Cardinal;

      procedure apSetProtocolMsgBase(NewBase : Cardinal);

      { Basic I/O stuff }
      
      procedure ResetWorkFile;
      procedure RewriteWorkFile;
      procedure CloseWorkFile;
      function WorkFileGetDate : Integer;
      procedure WorkFileSetDate (SrcFileDate : Integer);
      function WorkFileSize : Integer;
      procedure SeekWorkFile (Position : Longint);
      procedure BlockReadWorkFile (var Buf;
                                       Count             : Integer;
                                   var AmountTransferred : Integer);
      procedure BlockWriteWorkFile (var Buf;
                                        Count             : Integer;
                                    var AmountTransferred : Integer);
      procedure EraseWorkFile;
      procedure TruncateWorkFile;

    public
      { The TApxCustomProtocol requires direct access to this variable }
      FProtSection        : TRTLCriticalSection;

      constructor Create (AOwner : TComponent); override;

      procedure ShowStatus (Options : Cardinal); virtual;
      procedure LogFile (LogFileStatus : Cardinal); virtual;
      function NextFile (var FName : TPathCharArray) : Boolean; virtual;
      function AcceptFile (FName : PChar) : Boolean; virtual;
      procedure PrepareReading; virtual;
      function ReadProtocolBlock (var Block     : TApxDataBlock;
                                  var BlockSize : Cardinal) : Boolean; virtual;
      procedure FinishReading; virtual;
      procedure PrepareWriting; virtual;
      function WriteProtocolBlock (var Block     : TApxDataBlock;
                                       BlockSize : Cardinal) : Boolean; virtual;
      procedure FinishWriting; virtual;


      function Init (Options: Cardinal) : Integer; virtual; abstract;
      procedure Done; virtual; abstract;
      function Reinit : Integer; virtual; abstract;
      procedure DonePart; virtual; abstract;

      procedure PrepareTransmit; virtual; abstract;
      procedure PrepareReceive; virtual; abstract;

      procedure Transmit (Msg, wParam : Cardinal; lParam : LongInt); virtual; abstract;
      procedure Receive (Msg, wParam : Cardinal; lParam : LongInt); virtual; abstract;

      function  apGetBytesTransferred : LongInt;
      function  apGetBytesRemaining : LongInt;
      procedure apSetReceiveFilename(FName : PChar);
      procedure apSetActualBPS(BPS : LongInt);
      function  apNextFileMask(FName : PChar) : Boolean;
      function  apInitProtocolData            (ComPort : TApxCustomComPort;
                                               Options : Cardinal) : Integer;
      procedure apDoneProtocol;
      function  apEstimateTransferSecs (Size : LongInt) : LongInt;
      function apStatusMsg(P : PChar; Status : Cardinal) : PChar;
      procedure apSetProtocolPort(H : TApxCustomComPort);
      procedure apStartProtocol(Protocol     : Byte;
                                Transmitting : Boolean);
      procedure Assign (Source : TPersistent); override;

      class procedure apRegisterProtocolClass;

      property DataBlock          : PApxDataBlock    
               read FDataBlock write FDataBlock;
      property FileBuffer         : PApxFileBuffer
               read FFileBuffer write FFileBuffer;
      property PathName           : TPathCharArray
               read FPathName write SetPathName;
      property SearchMask         : TPathCharArray
               read FSearchMask write FSearchMask;
      property FileList           : PFileList
               read FFileList write FFileList;
      property DestDir            : TDirCharArray
               read FDestDir write FDestDir;

    published

      property StatusTrigger      : Integer
               read FStatusTrigger write FStatusTrigger;
      property TimeoutTrigger     : Integer
               read FTimeoutTrigger write FTimeoutTrigger;
      property OutBuffFreeTrigger : Integer
               read FOutBuffFreeTrigger write FOutBuffFreeTrigger;
      property OutBuffUsedTrigger : Integer
               read FOutBuffUsedTrigger write FOutBuffUsedTrigger;
      property NoCarrierTrigger   : Integer
               read FNoCarrierTrigger write FNoCarrierTrigger;
      property ComPort            : TApxCustomComport
                read FComPort write FComPort; 
      property BatchProtocol      : Boolean
               read FBatchProtocol write FBatchProtocol;
      property FilesSent          : Boolean
               read FFilesSent write FFilesSent;
      property AbortFlag          : Boolean
               read FAbortFlag write FAbortFlag;
      property TimerStarted       : Boolean
               read FTimerStarted write FTimerStarted;
      property CurProtocol        : Integer
               read FCurProtocol write FCurProtocol;
      property CheckType          : Cardinal
               read FCheckType write FCheckType;
      property HandshakeRetry     : Cardinal
               read FHandshakeRetry write FHandshakeRetry;
      property HandshakeWait      : Cardinal
               read FHandshakeWait write FHandshakeWait;
      property HandshakeAttempt   : Cardinal
               read FHandshakeAttempt write FHandshakeAttempt;
      property BlockLen           : Cardinal
               read FBlockLen write FBlockLen;
      property BlockNum           : Cardinal
               read FBlockNum write FBlockNum;
      property Flags              : Cardinal
               read FFlags write FFlags;
      property TransTimeout       : Cardinal
               read FTransTimeout write FTransTimeout;
      property FinishWait         : Cardinal
               read FFinishWait write FFinishWait;
      property RcvTimeout         : Cardinal
               read FRcvTimeout write FRcvTimeout;
      property ProtocolStatus     : Cardinal
               read FProtocolStatus write FProtocolStatus;
      property LastBlockSize      : Cardinal
               read FLastBlockSize write FLastBlockSize;
      property ProtocolError      : Integer
               read FProtocolError write FProtocolError;
      property SrcFileLen         : LongInt
               read FSrcFileLen write FSrcFileLen;
      property SrcFileDate        : Longint
               read FSrcFileDate write FSrcFileDate;
      property BlockCheck         : DWORD
               read FBlockCheck write FBlockCheck;
      property InitFilePos        : LongInt
               read FInitFilePos write FInitFilePos;
      property ReplyTimer         : EventTimer
               read FReplyTimer write FReplyTimer;
      property ForceStatus        : Boolean
               read FForceStatus write FForceStatus;
      property TimerPending       : Boolean
               read FTimerPending write FTimerPending;
      property InProgress         : Cardinal
               read FinProgress write FInProgress;
      property BlockErrors        : Cardinal
               read FBlockErrors write FBlockErrors;
      property TotalErrors        : Cardinal
               read FTotalErrors write FTotalErrors;
      property ActCPS             : Cardinal
               read FActCPS write FActCPS;
      property Overhead           : Cardinal
               read FOverhead write FOverhead;
      property TurnDelay          : Cardinal
               read FTurnDelay write FTurnDelay;
      property StatusInterval     : Cardinal
               read FStatusInterval write FStatusInterval;
      property SaveStatus         : Cardinal
               read FSaveStatus write FSaveStatus;
      property SaveError          : Integer
               read FSaveError write FSaveError;
      property BytesRemaining     : LongInt
               read FBytesRemaining write FBytesRemaining;
      property BytesTransferred   : LongInt
               read FBytesTransferred write FBytesTransferred;
      property ElapsedXfrTime     : LongInt
               read FElapsedXfrTime write FElapsedXfrTime;
      property StatusTime         : EventTimer
               read FStatusTime write FStatusTime;
      property Timer              : EventTimer
               read FTimer write SetTimer;
      property EndPending         : Boolean
               read FEndPending write FEndPending;
      property FileOpen           : Boolean
               read FFileOpen write FFileOpen;
      property NoMoreData         : Boolean
               read FNoMoreData write FNoMoreData;
      property LastBlock          : Boolean
               read FLastBlock write FLastBlock;
      property BlkIndex           : Cardinal
               read FBlkIndex write FBlkIndex;
      property WriteFailOpt       : Integer
               read FWriteFailOpt write FWriteFailOpt;
      property StartOfs           : LongInt
               read FStartOfs write FStartOfs;
      property EndOfs             : LongInt
               read FEndOfs write FEndOfs;
      property LastOfs            : LongInt
               read FLastOfs write FLastOfs;
      property FileOfs            : LongInt
               read FFileOfs write FFileOfs;
      property EndOfDataOfs       : LongInt
               read FEndOfDataOfs write FEndOfDataOfs;
      property SaveMode           : Cardinal
               read FSaveMode write FSaveMode;
      property UpcaseFileNames    : Boolean
               read FUpcaseFileNames write FUpcaseFileNames;
      property FindingFirst       : Boolean
               read FFindingFirst write FFindingFirst;
      property FileListIndex      : Cardinal
               read FFileListIndex write FFileListIndex;
      property CurRec             : TSearchRec
               read FCurRec write FCurRec;
      property FFOpen             : Boolean
               read FFFopen write FFFOpen;
      property WorkFile           : File                     
               read FWorkFile write FWorkFile;
      property WorkStream         : TStream
               read FWorkStream write FWorkStream;
      property StatusTimer        : EventTimer
               read FStatusTimer write FStatusTimer;
      property UseStreams         : Boolean
               read FUseStreams write FUseStreams;

  end;

{ --------------------------------------------------------------------------- }
{ ASCII Protocol                                                              }

type
  TApxASCIIState = ( taInitial,
                     taGetBlock,
                     taWaitFreeSpace,
                     taSendBlock,
                     taSendDelay,
                     taFinishDrain,
                     taFinished,
                     taDone,

                     raInitial,
                     raCollectBlock,
                     raProcessBlock,
                     raFinished,
                     raDone );

const

  ApxLogASCIIState : array[TApxASCIIState] of TDispatchSubType = (
     dsttaInitial, dsttaGetBlock, dsttaWaitFreeSpace, dsttaSendBlock,
     dsttaSendDelay, dsttaFinishDrain, dsttaFinished, dsttaDone,

     dstraInitial, dstraCollectBlock, dstraProcessBlock, dstraFinished,
     dstraDone);

  {Compile time constants}
  ApxAsciiDefInterCharDelay = 0;   {Default is zero ms delay between chars}
  ApxAsciiDefInterLineDelay = 0;   {Default is zero ms delay between lines}
  ApxAsciiDefEOLChar = cCR;        {Default EOL char is carriage return}
  ApxAsciiDefRcvTimeout = 20000;     {Default time to assume end of receive, 20 sec}
  ApxAsciiDefBlockLen = 60;        {Default block length (assume avg of 60)}
                           {Note: must be less than SizeOf(TDataBlock)-1}
  ApxAsciiDefCRTranslate = atNone; {Default CR Translation is none}
  ApxAsciiDefLFTranslate = atNone; {Default LF Translation is none}
  ApxAsciiDefMaxAccumDelay = 250;  {Max accum milliseconds to delay in one call}

  ApxProtocolDataTrigger = 0;

type
  TApxASCIIDriver = class (TApxBaseProtocolDriver)
    private

      { ASCII Specific Fields }

      FCtrlZEncountered : Boolean;
      FInterCharDelay   : Cardinal;
      FInterLineDelay   : Cardinal;
      FInterCharTime    : Cardinal;
      FInterLineTime    : Cardinal;
      FMaxAccumDelay    : Cardinal;
      FSendIndex        : Cardinal;
      FCRTransMode      : Cardinal;
      FLFTransMode      : Cardinal;
      FEOLChar          : Char;
      FASCIIState       : TApxASCIIState;

    protected
      procedure InitData;

    public
      function SetDelays(CharDelay, LineDelay : Cardinal) : Integer;
      function SetEOLChar(C : Char) : Integer;
      function GetLineNumber : LongInt;
      function SetEOLTranslation(CR, LF : Cardinal) : Integer;
      function SetEOFTimeout(NewTimeout : LongInt) : Integer;

      procedure Cancel;

      function SendBlockPart(var Block : TApxDataBlock) : Boolean;
      function CollectBlock(var Block : TApxDataBlock) : Boolean;
      procedure ReceiveBlock(var Block : TApxDataBlock;
                                     var BlockSize : Cardinal; var HandShake : Char);


      function Init (Options: Cardinal) : Integer; override;
      procedure Done; override;
      function Reinit : Integer; override;
      procedure DonePart; override;

      procedure PrepareTransmit; override;
      procedure PrepareReceive; override;

      procedure Transmit (Msg, wParam : Cardinal; lParam : LongInt); override;
      procedure Receive (Msg, wParam : Cardinal; lParam : LongInt); override;

      procedure Assign (Source : TPersistent); override;
      
    published
      property InterCharDelay   : Cardinal
               read FInterCharDelay write FInterCharDelay;
      property InterLineDelay   : Cardinal
               read FInterLineDelay write FInterLineDelay;
      property CRTransMode      : Cardinal
               read FCRTransMode write FCRTransMode;
      property LFTransMode      : Cardinal
               read FLFTransMode write FLFTransMode;
      property EOLChar          : Char
               read FEOLChar write FEOLChar;


  end;

{ --------------------------------------------------------------------------- }
  
implementation

uses
  AxProtcl;

const
  ApxSeparator = ';';
  ApxEndOfListMark = #0;


constructor TApxBaseProtocolDriver.Create (AOwner : TComponent);
begin
  inherited Create (AOwner);
end;

function TApxBaseProtocolDriver.EventFilter(Sender : QObjectH; Event : QEventH) : Boolean;

  procedure DoCustomDispatchEvent(Event : QCustomEventH; FreeData : Boolean);
  var
    DataReference : PAxMessage;
    Data          : TAxMessage;
  begin
    DataReference := QCustomEvent_data(Event);

    { Local copy of data }
    Data := DataReference^;

    if FreeData then
      Dispose(DataReference);

    try
    
      { The only options are to pass the message to Transmit or
        Recieve.  There is no reason to dispatch the message back to the
        TApxBaseProtocolDriver }
        
      if FInProgress <> 0 then begin
        if (Data.Msg and APXTRANSMITMESSAGEMASK) <> 0 then
          Transmit (Data.Msg and (not APXTRANSMITMESSAGEMASK),
                    Data.wParam, Data.lParam)
        else
          Receive (Data.Msg, Data.wParam, Data.lParam);
      end;
    except
      raise; 
    end;

    if not FreeData then
      DataReference^.Result := Data.Result;
  end;

begin

  case QEvent_type(Event) of

    QEventType_AxSendDispatchMessage :
      begin
        DoCustomDispatchEvent(QCustomEventH(Event), False);
        Result := True;
      end;

    QEventType_AxPostDispatchMessage :
      begin
        DoCustomDispatchEvent(QCustomEventH(Event), True);
        Result := True;
      end;

    else Result := inherited EventFilter(Sender, Event);
  end;
end;


procedure TApxBaseProtocolDriver.CreateWidget;
begin
  FHandle := QObject_create(nil, nil);
end;

procedure TApxBaseProtocolDriver.SetTimer (V : EventTimer);
begin
  FTimer.StartMS := V.StartMS;
  FTimer.ExpireMS := V.ExpireMS;
end;

procedure TApxBaseProtocolDriver.SetPathName (V : TPathCharArray);
begin
  FPathName := V;
end;

function TApxBaseProtocolDriver.apInitProtocolData(ComPort : TApxCustomComPort;
                                                   Options : Cardinal) : Integer;
{-Allocates and initializes a protocol control block with options}
var
  Baud     : LongInt;
{$IFDEF TRIALRUN}
  {$I TRIAL04.INC}
{$ENDIF}
begin
{$IFDEF TRIALRUN}
  TC;
{$ENDIF}

  {Initialize the protocol fields}
  {Make a critical section object}
  InitializeCriticalSection (FProtSection);

  {Init values other than zero}
  FComPort         := ComPort;
  FUpcaseFileNames := False;
  FCurProtocol     := NoProtocol;  
  FHandshakeWait   := ApxDefHandshakeWait;
  FHandshakeRetry  := ApxDefHandshakeRetry;
  FBlockLen        := 128;
  FWriteFailOpt    := wfcWriteRename;
  FTransTimeout    := ApxDefTransTimeout;
  FStatusInterval  := ApxDefAbsStatusInterval;
  FFlags           := Options;
  if (ComPort <> nil) and (ComPort.Open) then begin
    Baud := ComPort.Baud;
    FActCPS := Baud div 10;
  end else
    FActCPS := 0;
  StrPCopy(FSearchMask, '*');

  apInitProtocolData := ecOk;
end;

procedure TApxBaseProtocolDriver.apDoneProtocol;
{-Destroys a protocol}
begin
  { reserved }
end;


procedure TApxBaseProtocolDriver.apSetFileList(FL : PFileList);
{-Set FL as the file list}
begin
  FFileList := FL;
end;

function TApxBaseProtocolDriver.apMakeFileList(var FL   : PFileList;
                                                   Size : Cardinal) : Integer;
{-Allocates a new file list of Size bytes}
type
  PWord = ^Cardinal;
begin
  FL := AllocMem(Size);
  PWord(FL)^ := Size;
  apMakeFileList := ecOk;
end;

procedure TApxBaseProtocolDriver.apDisposeFileList(FL   : PFileList;
                                                   Size : Cardinal);
{-Disposes of file list FL}
type
  PWord = ^Cardinal;
begin
  FreeMem(FL, PWord(FL)^);
end;

function TApxBaseProtocolDriver.apAddFileToList(FL    : PFileList;
                                                PName : PChar) : Integer;
{-Adds pathname PName to file list FL}
type
  PWord = ^Cardinal;
var
  MaxList : Cardinal;
  NewP    : PChar;
begin
  {First Cardinal of list is size}
  MaxList := PWord(FL)^;
  if MaxList <= 3 then begin
    apAddFileToList := ecBadArgument;
    Exit;
  end;

  {Search for the current end of the list}
  NewP := Pointer(FL);
  Inc(NewP, 2);
  NewP := StrScan(NewP, ApxEndOfListMark);
  if NewP = nil then begin
    apAddFileToList := ecBadFileList;
    Exit;
  end;

  {Enough room?}
  if StrLen(PName)+1 > MaxList then begin
    apAddFileToList := ecOutOfMemory;
    Exit;
  end;

  {Add separator if not the first item in the list}
  if (NewP-2) <> Pointer(FL) then begin
    NewP[0] := ';';
    Inc(NewP);
  end;

  {Add the string}
  StrCopy(NewP, PName);
  apAddFileToList := ecOK;
end;

function TApxBaseProtocolDriver.apNextFileMask (FName : PChar) : Boolean;
{-Built-in function that works with file mask fields}
const
  {$IFDEF LINUX}
  AnyFileButDir = faAnyFile and not faDirectory;
  {$ELSE}
  AnyFileButDir = faAnyFile and not (faDirectory or faVolumeID);
  {$ENDIF}
var
  ErrorCode : Integer;
  PName     : array[0..255] of Char;
begin
  {Check for uninitialized search mask}
  if FSearchMask[0] = #0 then begin
    apProtocolError (ecNoSearchMask);
    apNextFileMask := False;
    Exit;
  end;

  {Search for a matching file}
  if FFindingFirst then begin
    ErrorCode := Abs (FindFirst (FSearchMask, AnyFileButDir, FCurRec));
    FFFOpen := True;
    if (ErrorCode = ENOENT) then begin
      apProtocolError (ecNoMatchingFiles);
      FName[0] := #0;
      apNextFileMask := False;
      FindClose (FCurRec);
      FFFOpen := False;
      Exit;
    end else
      FFindingFirst := False;
  end else
    ErrorCode := Abs (FindNext (FCurRec));

  {Check for errors}
  if ErrorCode <> 0 then begin
    {Failed to find file, return error status}
    if ErrorCode = ENOTDIR then
      apProtocolError(ecDirNotFound)
    else if (ErrorCode <> ENOENT) and (ErrorCode <> 1) then
      apProtocolError(-ErrorCode);
    FName[0] := #0;
    apNextFileMask := False;
    FindClose(FCurRec);
    FFFOpen := False;
  end else begin
    {If search mask contains a path, return that path}
    JustPathNameZ(FName, FSearchMask);
    if FName[0] <> #0 then begin
      ApxIncludeTrailingPathDelimiterZ (FName, FName);
      StrPCopy(PName, FCurRec.Name);
      StrCat(FName, PName);
    end else begin
      {$IFDEF LINUX}
      {$WARN SYMBOL_PLATFORM OFF}
      StrPCopy (PName, FCurRec.PathOnly + FCurRec.Name);
      {$WARN SYMBOL_PLATFORM ON}
      StrCopy (FName, PName);
      {$ELSE}
      StrPCopy (PName, FCurRec.PathOnly + FCurRec.Name);
      {$ENDIF}
    end;
    apNextFileMask := True;
  end;
end;

function TApxBaseProtocolDriver.apNextFileList (FName : PChar) : Boolean;
{-Built-in function that works with a list of files}
type
  PWord = ^Cardinal;
const
    MaxLen = 79;
var
    MaxSize : Cardinal;
    MaxNext : Cardinal;
    I       : Cardinal;
    Len     : Cardinal;
begin
  FProtocolError := ecOK;
  MaxSize := PWord(FFileList)^;
  if MaxSize <= 3 then begin
    apNextFileList := False;
    FName[0] := #0;
    Exit;
  end;

  {Return immediately if no more files}
  if FFileList^[FFileListIndex] = ApxEndOfListMark then begin
    apNextFileList := False;
    FName[0] := #0;
    Exit;
  end;

  {Increment past the last separator}
  if FFileListIndex <> 2 then
    Inc(FFileListIndex);

  {Define how far to look for the next marker}
  if LongInt(FFileListIndex) + MaxLen > Integer(MaxSize) then
    MaxNext := MaxSize
  else
    MaxNext := FFileListIndex + MaxLen;

  {Look for the next marker}
  for I := FFileListIndex to MaxNext do begin
    if (FFileList^[I] = ApxSeparator) or
       (FFileList^[I] = ApxEndOfListMark) then begin
      {Extract the pathname}
      Len := I - FFileListIndex;
      Move(FFileList^[FFileListIndex], FName[0], Len);
      FName[Len] := #0;
      apNextFileList := True;
      Inc(FFileListIndex, Len);
      Exit;
    end;
  end;

  {Bad format list (no separator) -- show error}
  apProtocolError(ecBadFileList);
  apNextFileList := False;
  FName[0] := #0;
end;

function TApxBaseProtocolDriver.apGetBytesTransferred : LongInt;
{-Returns bytes already transferred}
var
  TotalOverhead : Cardinal;
  OutBuff       : Cardinal;
  BT            : LongInt;
begin
  OutBuff := 0;
  if FComPort = nil then begin
    Result := 0;
    exit;
  end else
    OutBuff := FComPort.OutBuffUsed;
    
  if OutBuff >= FBlockLen then begin
    {Upload in progress, subtract outbuff from bytestransferred}
    if FBlockLen <> 0 then
      TotalOverhead := FOverhead * (OutBuff div FBlockLen)
    else
      TotalOverhead := 0;
    BT := DWORD(FBytesTransferred) - (OutBuff - TotalOverhead);
    if BT > 0 then
      apGetBytesTransferred := BT
    else
      apGetBytesTransferred := 0;
  end else
    apGetBytesTransferred := FBytesTransferred;
end;

function TApxBaseProtocolDriver.apGetBytesRemaining : LongInt;
{-Return bytes not yet transferred}
var
    BR : Longint;
begin
  BR := FSrcFileLen - apGetBytesTransferred;
  if BR < 0 then
    BR := 0;
  apGetBytesRemaining := BR;
end;

function TApxBaseProtocolDriver.apSupportsBatch : Boolean;
{-Returns True if this protocol supports batch file transfers}
begin
  apSupportsBatch := FBatchProtocol;
end;

function TApxBaseProtocolDriver.apGetInitialFilePos : LongInt;
  {-Returns the file position at the start of resumed file transfer}
begin
  apGetInitialFilePos := FInitFilePos;
end;

function TApxBaseProtocolDriver.apEstimateTransferSecs(Size : LongInt) : LongInt;
{-Return estimated seconds to transfer Size bytes}
var
    Efficiency   : LongInt;
    EffectiveCPS : LongInt;
begin
  if Size = 0 then
    apEstimateTransferSecs := 0
  else begin
    {Calculate efficiency of this protocol}
    Efficiency := (Integer(FBlockLen) * LongInt(100)) div
                   Longint(FBlockLen + FOverHead +
                   (DWORD(FTurnDelay * FActCPS) div 1000));
    EffectiveCPS := (FActCPS * DWORD(Efficiency)) div 100;

    {Calculate remaining seconds}
    if EffectiveCPS > 0 then
      apEstimateTransferSecs := Size div EffectiveCPS
    else
      apEstimateTransferSecs := 0;
  end;
end;

procedure TApxBaseProtocolDriver.apGetProtocolInfo(var Info : TProtocolInfo);
{-Returns a protocol information block}
begin
  with Info do begin
    piStatus           := FProtocolStatus;
    piError            := FProtocolError;
    piProtocolType     := FCurProtocol;
    StrLCopy(piFileName, FPathName, SizeOf(piFileName));
    piFileSize         := FSrcFileLen;
    piBytesTransferred := apGetBytesTransferred;
    piBytesRemaining   := apGetBytesRemaining;
    piInitFilePos      := FInitFilePos;
    piElapsedXfrTime   := FElapsedXfrTime;
    piBlockErrors      := FBlockErrors;
    piTotalErrors      := FTotalErrors;
    piBlockSize        := FBlockLen;
    if FBlockLen <> 0 then
      piBlockNum := piBytesTransferred div Integer(FBlockLen)
    else
      piBlockNum := 0;
    piBlockCheck       := FCheckType;
    piFlags            := FFlags;
  end;
end;

procedure TApxBaseProtocolDriver.apSetFileMask(NewMask : PChar);
{-Set the search mask}
begin
  StrLCopy(FSearchMask, NewMask, SizeOf(FSearchMask));
end;

procedure TApxBaseProtocolDriver.apSetReceiveFilename(FName : PChar);
{-Set or change the incoming file name}
var
    Temp : TCharArray;
begin
  if StrScan(FName, '/') = nil then begin
    {Set aPathname to DestDir path + FName}
    StrLCopy (FPathname,
              ApxIncludeTrailingPathDelimiterZ (Temp, FDestDir),
              SizeOf(FPathname));
    StrLCat(FPathname, FName, SizeOf(FPathname));
  end else
    {Set aPathname directly to FName}
    StrLCopy(FPathName, FName, SizeOf(FPathname));
end;

procedure TApxBaseProtocolDriver.apSetDestinationDirectory(Dir : PChar);
{-Set the directory used to hold received files}
begin
  StrLCopy(FDestDir, Dir, SizeOf(FDestDir));
end;

procedure TApxBaseProtocolDriver.apSetHandshakeWait(NewHandshake, NewRetry : Cardinal);
{-Set the wait time and retry count for the initial handshake}
begin
  if NewHandshake <> 0 then
    FHandshakeWait := NewHandshake;
  if NewRetry <> 0 then
    FHandshakeRetry := NewRetry;
end;

procedure TApxBaseProtocolDriver.apSetEfficiencyParms(
                                 BlockOverhead, TurnAroundDelay : Cardinal);
{-Sets efficiency parameters for EstimateTransferSecs}
begin
  FOverhead := BlockOverhead;
  FTurnDelay := TurnAroundDelay;
end;

procedure TApxBaseProtocolDriver.apSetProtocolPort(H : TApxCustomComPort);
{-Set H as the port object for this protocol}
begin
  FComPort := H;
end;

procedure TApxBaseProtocolDriver.apSetOverwriteOption(Opt : Cardinal);
{-Set option for what to do when the destination file already exists}
begin
  if Opt <= wfcWriteResume then
    FWriteFailOpt := Opt;
end;

procedure TApxBaseProtocolDriver.apSetActualBPS(BPS : LongInt);
{-Sets actual BPS rate (only needed if modem differs from port)}
var
  Bits     : Word;
begin
  if (FComPort = nil) or not FComPort.Open then
    Bits := 10
  else begin
    case FComPort.DataBits of
      dbFive  : Bits := 7;
      dbSix   : Bits := 8;
      dbSeven : Bits := 9;
      dbEight : Bits := 10;
    else
      Bits := 10;
    end;
  end;
  if FComPort.Parity <> pNone then
    Inc(Bits);
  FActCPS := BPS div Bits;
end;

procedure TApxBaseProtocolDriver.apSetStatusInterval(NewInterval : Cardinal);
{-Set new status update interval to NewInterval time}
begin
  FStatusInterval := NewInterval;
end;

procedure TApxBaseProtocolDriver.apOptionsOn(OptionFlags : Cardinal);
{-Activate multiple options}
begin
  FFlags := FFlags or (OptionFlags and not BadProtocolOptions);
end;

procedure TApxBaseProtocolDriver.apOptionsOff(OptionFlags : Cardinal);
{-Deactivate multiple options}
begin
  FFlags := FFlags and not (OptionFlags and not BadProtocolOptions);
end;

function TApxBaseProtocolDriver.apOptionsAreOn(OptionFlags : Cardinal) : Boolean;
{-Return True if all bits in OptionsFlags are on}
begin
  apOptionsAreOn := FFlags and OptionFlags = OptionFlags;
end;

procedure TApxBaseProtocolDriver.apStartProtocol (Protocol     : Byte;
                                                  Transmitting : Boolean);
{-Setup standard protocol triggers}
var
  lParam : LongInt;
begin
  {Note the protocol}
  FCurProtocol := Protocol;

  {Next file stuff}
  FFilesSent := False;
  FFindingFirst := True;
  FFileListIndex := 2;

  {Set up standard triggers}
  FTimeoutTrigger := FComPort.AddTimerTrigger;
  FStatusTrigger := FComPort.AddTimerTrigger;
  FOutBuffFreeTrigger := FComPort.AddStatusTrigger(stOutBuffFree);
  FOutBuffUsedTrigger := FComPort.AddStatusTrigger(stOutBuffUsed);
  FNoCarrierTrigger := FComPort.AddStatusTrigger(stModem);

  {All set?}
  if (FTimeoutTrigger < 0) or
     (FStatusTrigger < 0) or (FOutBuffFreeTrigger < 0) or
     (FOutBuffUsedTrigger < 0) or (FNoCarrierTrigger < 0) then begin
    {Send error message and give up}
    FProtocolError := ecNoMoreTriggers;
    apSignalFinish (Transmitting);
    Exit;
  end;

  with FComPort.Dispatcher do begin

    {Prepare protocol}
    if Transmitting then
      PrepareTransmit
    else
      PrepareReceive;
      
    if FProtocolError = ecOK then begin
      {Call the notification function directly the first time}

      LH (lParam).H := 0;
      LH(lParam).L := 0;
      
      if Transmitting then
        Transmit (0, 0, lParam)
      else
        Receive (0, 0, lParam);

      if FProtocolError <> ecOk then
        exit;

      {Activate status timer now}
      FComPort.SetTimerTrigger(FStatusTrigger, FStatusInterval, True);

      {Set DCD trigger if necessary}
      if FlagIsSet(FFlags, apAbortNoCarrier) then begin
        if CheckDCD then
          {Set modem status trigger to look for carrier loss}
          SetStatusTrigger(FNoCarrierTrigger, msDCDDelta, True)
        else begin
          {Carrier not present now, abort}
          FProtocolError := ecAbortNoCarrier;
          apSignalFinish (Transmitting);
          Exit;
        end;
      end;

    end else
      {Couldn't get started, finish now}
      apSignalFinish (Transmitting);
  end;
end;

procedure TApxBaseProtocolDriver.apStopProtocol (Transmitting : Boolean);
{-Stop the protocol}

  procedure RemoveIt(Trig : Integer);
  begin
    if Trig > 0 then
      FComPort.RemoveTrigger(Trig);
  end;

begin
  {Remove the protocol triggers}
  if FTimeoutTrigger <> 0 then
    RemoveIt(FTimeoutTrigger);
  FTimeoutTrigger := 0;
  if FStatusTrigger <> 0 then
    RemoveIt(FStatusTrigger);
  FStatusTrigger := 0;
  if FOutBuffFreeTrigger <> 0 then
    RemoveIt(FOutBuffFreeTrigger);
  FOutBuffFreeTrigger := 0;
  if FOutBuffUsedTrigger <> 0 then
    RemoveIt(FOutBuffUsedTrigger);
  FOutBuffUsedTrigger := 0;
  if FNoCarrierTrigger <> 0 then
    RemoveIt(FNoCarrierTrigger);
  FNoCarrierTrigger := 0;

  {Remove our trigger handler}

  if (FComPort <> nil) and FComPort.Open then begin
    if Transmitting then
      FComPort.Dispatcher.DeregisterEventTriggerHandler(Transmit)
    else
      FComPort.Dispatcher.DeregisterEventTriggerHandler(Receive);
  end;

  {Close findfirst, if it's still open}
  if FFFOpen then begin
    FFFOpen := False;
    FindClose(FCurRec);
  end;
  {Close the file, if it's still open}
  if FFileOpen then begin
    CloseWorkFile;
    FFileOpen := False;
  end;
end;

{Internal routines}

procedure TApxBaseProtocolDriver.apResetStatus;
  {-Reset status vars}
begin
  if FInProgress = 0 then begin
    FSrcFileLen := 0;
    FBytesRemaining := 0;
  end;
  FBytesTransferred := 0;
  FBlockNum := 0;
  FElapsedXfrTime := 0;
  FBlockErrors := 0;
  FTotalErrors := 0;
  FProtocolStatus := psOK;
  FProtocolError := ecOK;
end;

procedure TApxBaseProtocolDriver.apShowFirstStatus;
{-Show (possible) first status}
const
  Option : array[Boolean] of Cardinal = ($00, $01);
begin
  ShowStatus (Option[FInProgress = 0]); 
  Inc(FInProgress);
end;

procedure TApxBaseProtocolDriver.apShowLastStatus;
{-Reset field and show last status}
const
  Option : array[Boolean] of Cardinal = ($00, $02);
begin
  if FInProgress <> 0 then begin
    Dec(FInProgress);
    ShowStatus (Option[FInProgress = 0]);
  end;
end;

procedure TApxBaseProtocolDriver.apSignalFinish (Transmitting : Boolean);
{-Send finish message to parent window}
var
  DT: Cardinal; 
  ErrMsg: String;
begin
  apStopProtocol (Transmitting);

  {Flag some final status codes as error codes}
  if FProtocolError = ecOk then begin
    case FProtocolStatus of
      psCancelRequested : FProtocolError := ecCancelRequested;
      psTimeout         : FProtocolError := ecTimeout;
      psProtocolError   : FProtocolError := ecProtocolError;
      psSequenceError   : FProtocolError := ecSequenceError;
      psFileRejected    : FProtocolError := ecFileRejected;
      psCantWriteFile   : FProtocolError := ecCantWriteFile;
      psAbortNoCarrier  : FProtocolError := ecAbortNoCarrier;
    end;
  end;
  case FCurProtocol of
    Xmodem,
    XmodemCRC,
    Xmodem1K,
    Xmodem1KG   : DT := AxdtXModem;
    Ymodem,
    YmodemG     : DT := AxdtYModem;
    Zmodem      : DT := AxdtZModem;
    Kermit      : DT := AxdtKermit;
    Ascii       : DT := AxdtAscii;
    else          DT := AxdtNone;
  end; 
  ErrMsg := 'ErrorCode:' + IntToStr(FProtocolError);

  FComPort.DebugLog.AddDebugEntry (TApxCustomProtocol, Cardinal (DT) + 100000,
                                   Cardinal (dstStatus), FProtocolError);

  { Post message to end the protocol }
  
  if Owner is TApxBaseHandleComponent then
    AxPostMessage ((Owner as TApxBaseHandleComponent).Handle,
                   apx_ProtocolFinish, nil, FProtocolError, 0);
end;

procedure TApxBaseProtocolDriver.aapPrepareReading;
{-Prepare to send protocol blocks (usually opens a file)}
var
  Res : Cardinal;
begin
  FProtocolError := ecOK;

  {If file is already open then leave without doing anything}
  if FFileOpen then
    Exit;

  {Report notfound error for empty filename}
  if FPathName[0] = #0 then begin
    apProtocolError(ecFileNotFound);
    Exit;
  end;

  {Allocate a file buffer}
  FFileBuffer := AllocMem(ApxFileBufferSize);

  {Open up the previously specified file}
  FSaveMode := FileMode;
  FileMode := fmOpenRead or fmShareDenyWrite;
  ResetWorkFile;
  FileMode := FSaveMode;
  Res := IOResult;
  if Res <> 0 then begin
    apProtocolError(-Res);
    FreeMem(FFileBuffer, ApxFileBufferSize);
    Exit;
  end;

  {Show file name and size}
  FSrcFileLen := WorkFileSize;
  if IOResult <> 0 then
    FSrcFileLen := 0;
  FBytesRemaining := FSrcFileLen;
  ShowStatus(0);

  {Note file date/time stamp (for those protocols that care)}
  FSrcFileDate := FileGetDate (TFileRec (FWorkFile).Handle);

  {Initialize the file buffering variables}
  FFileOfs := 0;
  FStartOfs := 0;
  FEndOfs := 0;
  FLastOfs := 0;
  FEndPending := False;
  FFileOpen := True;
end;

procedure TApxBaseProtocolDriver.aapFinishReading;
{-Clean up after reading protocol blocks}
begin
  if FFileOpen then begin
    CloseWorkFile;
    if IOResult <> 0 then ;
    FreeMem(FFileBuffer, ApxFileBufferSize);
    FFileOpen := False;
  end;
end;

function TApxBaseProtocolDriver.aapReadProtocolBlock (
                                var Block : TApxDataBlock;
                                var BlockSize : Cardinal) : Boolean;
{-Return with a block to transmit (True to quit)}
var
    BytesRead   : Integer;
    BytesToMove : Integer;
    BytesToRead : Integer;
    Res         : Cardinal;
begin
  if FFileOfs >= FSrcFileLen then begin
    BlockSize := 0;
    aapReadProtocolBlock := True;
    Exit;
  end;

  {Check for a request to start further along in the file (recovering)}
  if FFileOfs > FEndOfs then
    {Skipping blocks - force a read}
    FEndOfs := FFileOfs;

  {Check for a request to retransmit an old block}
  if FFileOfs < FLastOfs then
    {Retransmit - reset end-of-buffer to force a read}
    FEndOfs := FFileOfs;

  if (FFileOfs + Integer(BlockSize)) > FEndOfs then begin
    {Buffer needs to be updated, first shift end section to beginning}
    BytesToMove := FEndOfs - FFileOfs;
    if BytesToMove > 0 then
      Move(FFileBuffer^[FFileOfs - FStartOfs], FFileBuffer^, BytesToMove);

    {Fill end section from file}
    BytesToRead := ApxFileBufferSize - BytesToMove;
    SeekWorkFile (FEndOfs);
    BlockReadWorkFile (FFileBuffer^[BytesToMove], BytesToRead, BytesRead);
    Res := IOResult;
    if (Res <> 0) then begin
      {Exit on error}
      apProtocolError(-Res);
      aapReadProtocolBlock := True;
      BlockSize := 0;
      Exit;
    end else begin
      {Set buffering variables}
      FStartOfs := FFileOfs;
      FEndOfs := FFileOfs + ApxFileBufferSize;
    end;

    {Prepare for the end of the file}
    if BytesRead < BytesToRead then begin
      FEndOfDataOfs := BytesToMove + BytesRead;
      FillChar (FFileBuffer^[FEndofDataOfs], ApxFileBufferSize - FEndOfDataOfs,
                ApxBlockFillChar);
      Inc(FEndOfDataOfs, FStartOfs);
      FEndPending := True;
    end else
      FEndPending := False;
  end;

  {Return the requested block}
  Move(FFileBuffer^[(FFileOfs - FStartOfs)], Block, BlockSize);
  aapReadProtocolBlock := False;
  FLastOfs := FFileOfs;

  {If it's the last block then say so}
  if FEndPending and ((FFileOfs + Integer(BlockSize)) >= FEndOfDataOfs) then begin
    aapReadProtocolBlock := True;
    BlockSize := FEndOfDataOfs - FFileOfs;
  end;
end;

procedure TApxBaseProtocolDriver.aapPrepareWriting;
{-Prepare to save protocol blocks (usually opens a file)}
var
  Res  : Cardinal;
  S    : string[fsPathName];
  Dir  : string[fsDirectory];
  Name : string[fsName];
label
  ExitPoint; 
begin
  {Allocate a file buffer}
  FFileBuffer := AllocMem(ApxFileBufferSize);

  {Does the file exist already?}
  FSaveMode := FileMode;
  FileMode := 0; 
  AssignFile (FWorkFile, FPathName); 
  Reset(FWorkFile, 1);
  
  FileMode := FSaveMode;
  Res := IOResult;

  {Exit on errors other than FileNotFound}
  if (Res <> 0) and (Res <> 2) then begin
    apProtocolError(-Res);
    goto ExitPoint; 
  end;

  {Exit if file exists and option is WriteFail}
  if (Res = 0) and (FWriteFailOpt = wfcWriteFail) then begin
    FProtocolStatus := psCantWriteFile;
    FForceStatus := True;
    goto ExitPoint;
  end;

  CloseWorkFile;
  if IOResult = 0 then ;

  {Change the file name if it already exists and the option is WriteRename}
  if (Res = 0) and (FWriteFailOpt = wfcWriteRename) then begin
    S := StrPas(FPathName);
    Dir := ExtractFilePath(S);
    Name := ExtractFileName(S);
    Name[1] := '$';
    S := Dir + Name;
    StrPCopy(FPathName, S);
  end;

  {Give status a chance to show that the file was renamed}
  ShowStatus(0);

  {Ok to rewrite file now}
  RewriteWorkFile;
  Res := IOResult;
  if Res <> 0 then begin
    apProtocolError(-Res);
    goto ExitPoint; 
  end;

  {Initialized the buffer management vars}
  FStartOfs := 0;
  FLastOfs := 0;
  FEndOfs := FStartOfs + ApxFileBufferSize;
  FFileOpen := True;
  Exit;

ExitPoint:
  CloseWorkFile;
  if IOResult <> 0 then ;
  FreeMem(FFileBuffer, ApxFileBufferSize);
end;

procedure TApxBaseProtocolDriver.aapFinishWriting;
{-Cleans up after saving all protocol blocks}
var
  Res          : Cardinal;
  BytesToWrite : Integer;
  BytesWritten : Integer;
begin
  if FFileOpen then begin
    {Error or end-of-protocol, commit buffer and cleanup}
    BytesToWrite := FFileOfs - FStartOfs;
    BlockWriteWorkFile (FFileBuffer^, BytesToWrite, BytesWritten);
    Res := IOResult;
    if Res <> 0 then
      apProtocolError(-Res)
    else if BytesToWrite <> BytesWritten then
      apProtocolError(ecDiskFull);

    {Get file size and time for those protocols that don't know}
    FSrcFileLen := WorkFileSize;
    FSrcFileDate := WorkFileGetDate;

    CloseWorkFile;
    Res := IOResult;
    if Res <> 0 then
      apProtocolError(-Res);
    FreeMem(FFileBuffer, ApxFileBufferSize);
    FFileOpen := False;
  end;
end;

function TApxBaseProtocolDriver.aapWriteProtocolBlock(
                                 var Block : TApxDataBlock;
                                 BlockSize : Cardinal) : Boolean;
{-Write a protocol block (return True to quit)}
var
  Res          : Cardinal;
  BytesToWrite : Integer;
  BytesWritten : Integer;

  procedure BlockWriteRTS;
  {-Set RTS before BlockWrite}
  begin
    {Lower RTS if requested}
    if FlagIsSet(FFlags, apRTSLowForWrite) then
      if (FComPort <> nil) and FComPort.Open then
        FComPort.RTS := False;

    BlockWriteWorkFile (FFileBuffer^, BytesToWrite, BytesWritten);

    {Raise RTS if it was lowered}
    if FlagIsSet(FFlags, apRTSLowForWrite) then
      if (FComPort <> nil) and FComPort.Open then
        FComPort.RTS := True;
  end;

begin
  FProtocolError := ecOK;
  aapWriteProtocolBlock := True;

  if not FFileOpen then begin
    apProtocolError(ecNotOpen);
    Exit;
  end;

  if FFileOfs < FLastOfs then
    {This is a retransmitted block}
    if FFileOfs > FStartOfs then begin
      {aFileBuffer has some good data, commit that data now}
      SeekWorkFile (FStartOfs);
      BytesToWrite := FFileOfs - FStartOfs;
      BlockWriteRTS;
      Res := IOResult;
      if (Res <> 0) then begin
        apProtocolError(-Res);
        Exit;
      end;
      if (BytesToWrite <> BytesWritten) then begin
        apProtocolError(ecDiskFull);
        Exit;
      end;
    end else begin
      {Block is before data in buffer, discard data in buffer}
      FStartOfs := FFileOfs;
      FEndOfs := FStartOfs + ApxFileBufferSize;
      {Position file just past last good data}
      SeekWorkFile (FFileOfs);
      Res := IOResult;
      if Res <> 0 then begin
        apProtocolError(-Res);
        Exit;
      end;
    end;

  {Will this block fit in the buffer?}
  if (FFileOfs + Integer(BlockSize)) > FEndOfs then begin
    {Block won't fit, commit current buffer to disk}
    BytesToWrite := FFileOfs - FStartOfs;
    BlockWriteRTS;
    Res := IOResult;
    if (Res <> 0) then begin
      apProtocolError(-Res);
      Exit;
    end;
    if (BytesToWrite <> BytesWritten) then begin
      apProtocolError(ecDiskFull);
      Exit;
    end;

    {Reset the buffer management vars}
    FStartOfs := FFileOfs;
    FEndOfs   := FStartOfs + ApxFileBufferSize;
    FLastOfs  := FFileOfs;
  end;

  {Add this block to the buffer}
  Move(Block, FFileBuffer^[FFileOfs - FStartOfs], BlockSize);
  Inc(FLastOfs, BlockSize);
  aapWriteProtocolBlock := False;
end;

procedure TApxBaseProtocolDriver.apProtocolError(ErrorCode : Integer);
{-Sends message and sets aProtocolError}
begin
  (Owner as TApxCustomProtocol).HandleAPXProtocolError (ErrorCode);
  FProtocolError := ErrorCode;
end;

function TApxBaseProtocolDriver.apTrimZeros (S: string): String;
var
  I, J : Integer;
begin
  I := Length(S);
  while (I > 0) and (S[I] <= ' ') do
    Dec(I);
  J := 1;
  while (J < I) and ((S[J] <= ' ') or (S[J] = '0')) do
    Inc(J);
  Result := Copy(S, J, (I-J)+1);
end;

function TApxBaseProtocolDriver.apOctalStr(L : LongInt) : String;
{-Convert L to octal base string}
const
  Digits : array[0..7] of Char = '01234567';
var
  I : Cardinal;
begin
  SetLength(Result, 12);
  for I := 0 to 11 do begin
    apOctalStr[12-I] := Digits[L and 7];
    L := L shr 3;
  end;
end;

function TApxBaseProtocolDriver.apOctalStr2Long(S : String) : LongInt;
{-Convert S from an octal string to a longint}
const
  HiMag = 11;
  Magnitude : array[1..HiMag] of LongInt = (1, 8, 64, 512, 4096,
      32768, 262144, 2097152, 16777216, 134217728, 1073741824);
var
  Len  : Byte;
  I    : Integer;
  J    : Integer;
  Part : LongInt;
  Res  : LongInt;
begin
  {Assume failure}
  apOctalStr2Long := 0;

  {Remove leading blanks and zeros}
  S := apTrimZeros(S);
  Len := Length(S);

  {Return 0 for invalid strings}
  if Len > HiMag then
    Exit;

  {Convert it}
  Res := 0;
  J := 1;
  for I := Len downto 1 do begin
    if (S[I] < '0') or (S[I] > '7') then
      Exit;
    Part := Byte(S[I]) - $30;
    Res := Res + Part * Magnitude[J];
    Inc(J);
  end;
  apOctalStr2Long := Res
end;

function TApxBaseProtocolDriver.apPackToYMTimeStamp(RawTime : LongInt) : LongInt;
{-Return date/time stamp as seconds since 1/1/1970 00:00 GMT}
var
  Days  : LongInt;
  Secs  : LongInt;
  DT    : TDateTime;
begin
  try
    {Get file date as Delphi-style date/time}
    DT := FileDateToDateTime(RawTime);

    {Calculate number of seconds since 1/1/1970}
    Days := Trunc(DT) - UnixDaysBase;
    Secs := Round(Frac(DT) * SecsPerDay);
    Result := (Days * SecsPerDay) + Secs;
  except
    Result := 0;
  end;
end;

function TApxBaseProtocolDriver.apYMTimeStampToPack(YMTime : LongInt) : LongInt;
{-Return a file time stamp in packed format from a Ymodem time stamp}
var
  DT : TDateTime;
begin
  try
    {Convert to Delphi style date, add in unix base}
    DT := YMTime / SecsPerDay;
    DT := DT + UnixDaysBase;

    {Return as packed}
    Result := DateTimeToFileDate(DT);
  except
    Result := 0
  end;
end;

function TApxBaseProtocolDriver.apCurrentTimeStamp : LongInt;
{-Return a Ymodem format file time stamp of the current date/time}
begin
  Result := apPackToYMTimeStamp(DateTimeToFileDate(Now));
end;

function TApxBaseProtocolDriver.apCrc32OfFile(FName : PChar; Len : Longint) : LongInt;
{-Returns Crc32 of FName}
const
  BufSize = 8192;
type
  BufArray = array[1..BufSize] of Byte;
var
  I         : Cardinal;
  BytesRead : Integer;
  Res       : Cardinal;
  FileLoc   : LongInt;
  Buffer    : ^BufArray;
  F         : File;   

begin
  FBlockCheck := 0;

  {If Len is zero then check the entire file}
  if Len = 0 then
    Len := MaxLongint;

  {Get a buffer}
  Buffer := AllocMem(BufSize);

  try

    {Open the file}
    FSaveMode := FileMode;
    FileMode := fmOpenRead or fmShareDenyWrite;
    AssignFile (F, FName); 
    Reset(F, 1);
    FileMode := FSaveMode;
    Res := IOResult;
    if Res <> 0 then
      apProtocolError(-Res)
    else begin

      {Initialize Crc}
      FBlockCheck := $FFFFFFFF;

      {Start at beginning, loop thru file calculating Crc32}
      FileLoc := 0;
      repeat
        BlockRead(F , Buffer^, BufSize, BytesRead);
        Res := IOResult;
        if Res = 0 then begin
          if Len <> MaxLongint then begin
            Inc(FileLoc, BytesRead);
            if FileLoc > Len then
              BytesRead := BytesRead - (FileLoc - Len);
          end;
          for I := 1 to BytesRead do
            FBlockCheck := apUpdateCrc32(Byte(Buffer^[I]), FBlockCheck)
        end;
      until (BytesRead = 0) or (Res <> 0) or (FileLoc >= Len);

      CloseFile (F);
      if IOResult = 0 then ;
    end;

  finally
    apCrc32OfFile := FBlockCheck;
    FreeMem(Buffer, BufSize);
  end;
end;

procedure TApxBaseProtocolDriver.apMsgStatus(Options : Cardinal);
{-Send an apx_ProtocolStatus message to the protocol window}
begin
  (Owner as TApxCustomProtocol).HandleAPXProtocolStatus (Options);
end;

function TApxBaseProtocolDriver.apMsgNextFile(var FName : TPathCharArray) : Boolean;
{-Virtual method for calling NextFile procedure}
var
  NextFileName : string;
  Len : Integer;
begin
  NextFileName := string (FName);
  Result := (Owner as TApxCustomProtocol).HandleAPXProtocolNextFile (NextFileName);
  Len := Length (NextFileName);
  if Len > 254 then
    raise EProtocol.CreateUnknown ('File Name is too long', 0);
  StrPCopy (PChar (@FName), NextFileName);
  FName[Len] := #00;
end;

procedure TApxBaseProtocolDriver.apMsgLog(Log : Cardinal);
{-Send an apx_ProtocolLog message to the protocol window}
begin
  (Owner as TApxCustomProtocol).HandleAPXProtocolLog (Log);
end;

function TApxBaseProtocolDriver.apMsgAcceptFile(FName : PChar) : Boolean;
{-Send apx_ProtocolAcceptFile message to TProtocolWindow}
begin
  Result := (Owner as TApxCustomProtocol).HandleAPXProtocolAcceptFile (FName);
end;

function TApxBaseProtocolDriver.apMsgOpenStream (var Stream  : TStream;
                                                     Reading : Boolean) : Boolean;
begin
  (Owner as TApxCustomProtocol).HandleAPXProtocolOpenStream (Stream, Reading);
  Result := True; 
end;

function TApxBaseProtocolDriver.apMsgCloseStream (var Stream : TStream) : Boolean;
begin
  (Owner as TApxCustomProtocol).HandleAPXProtocolCloseStream (Stream);
  Result := True; 
end;

function TApxBaseProtocolDriver.apUpdateChecksum(CurByte : Byte; CheckSum : Cardinal) : Cardinal;
{-Returns an updated checksum}
begin
  apUpdateCheckSum := CheckSum + CurByte;
end;

function TApxBaseProtocolDriver.apUpdateCrc(CurByte : Byte; CurCrc : Cardinal) : Cardinal;
{-Returns an updated CRC16}
begin
  Result := (CrcTable[((CurCrc shr 8) and 255)] xor
              (CurCrc shl 8) xor CurByte) and $FFFF;
end;

function TApxBaseProtocolDriver.apUpdateCrcKermit(CurByte : Byte; CurCrc : Cardinal) : Cardinal;
{-Returns an updated Crc16 (kermit style)}
var
    I    : Integer;
    Temp : Cardinal;
begin
  for I := 0 to 7 do begin
    Temp := CurCrc xor CurByte;
    CurCrc := CurCrc shr 1;
    if Odd(Temp) then
      CurCrc := CurCrc xor $8408;
    CurByte := CurByte shr 1;
  end;
  Result := CurCrc;
end;

function TApxBaseProtocolDriver.apStatusMsg(P : PChar; Status : Cardinal) : PChar;
{-Return an appropriate error message from the stringtable}
begin
  case Status of
    psOK..psHostResume :
      StrPCopy(P, MessageNumberToString (Status)); 
    else
      P[0] := #0;
  end;
  Result := P;
end;

class procedure TApxBaseProtocolDriver.apRegisterProtocolClass;
begin
  { nothing }
end;

procedure TApxBaseProtocolDriver.apSetProtocolMsgBase(NewBase : Cardinal);
{-Set new base for protocol string table}
begin
  {nothing}
end;

function TApxBaseProtocolDriver.apUpdateCrc32(CurByte : Byte; CurCrc : LongInt) : LongInt;
{-Return the updated 32bit CRC}
{-Normally a good candidate for basm, but Delphi32's code
  generation couldn't be beat on this one!}
begin
  apUpdateCrc32 := Crc32Table[Byte(CurCrc xor CurByte)] xor
                     DWORD((CurCrc shr 8) and $00FFFFFF);
end;

procedure InitializeUnit;
var
  TmpDateSeparator : string[1];
  TmpDateFormat    : string[15];
  TmpDateTime      : TDateTime;
begin
  {Set Unix days base}
  TmpDateFormat       := ShortDateFormat;
  SetLength (TmpDateSeparator,1);
  TmpDateSeparator[1] := DateSeparator;
  DateSeparator       := '/';
  ShortDateFormat     := 'mm/dd/yyyy';
  TmpDateTime         := StrToDateTime ('01/01/1970');
  UnixDaysBase        := Trunc (TmpDateTime); 
  DateSeparator       := TmpDateSeparator[1];
  ShortDateFormat     := TmpDateFormat;

  {Register protocol window class}
  TApxBaseProtocolDriver.apRegisterProtocolClass;
end;

procedure TApxBaseProtocolDriver.ShowStatus (Options : Cardinal);
begin
  apMsgStatus (Options);
end;

procedure TApxBaseProtocolDriver.LogFile (LogFileStatus : Cardinal);
begin
  apMsgLog (LogFileStatus);
end;

function TApxBaseProtocolDriver.NextFile (var FName : TPathCharArray) : Boolean;
begin
  Result := apMsgNextFile (FName);
end;

function TApxBaseProtocolDriver.AcceptFile (FName : PChar) : Boolean;
begin
  Result := apMsgAcceptFile (FName);
end;

procedure TApxBaseProtocolDriver.PrepareReading;
begin
  aapPrepareReading;
end;

function TApxBaseProtocolDriver.ReadProtocolBlock (var Block     : TApxDataBlock;
                                                   var BlockSize : Cardinal) : Boolean;
begin
  Result := aapReadProtocolBLock (Block, BlockSize);
end;

procedure TApxBaseProtocolDriver.FinishReading;
begin
  aapFinishReading;
end;

procedure TApxBaseProtocolDriver.PrepareWriting;
begin
  aapPrepareWriting;
end;

function TApxBaseProtocolDriver.WriteProtocolBlock (var Block     : TApxDataBlock;
                                                        BlockSize : Cardinal) : Boolean;
begin
  Result := aapWriteProtocolBlock (Block, BlockSize);
end;

procedure TApxBaseProtocolDriver.FinishWriting;
begin
  aapFinishWriting;
end;

procedure TApxBaseProtocolDriver.Assign (Source : TPersistent);
begin
  if (Source is TApxBaseProtocolDriver) then
    with Source as TApxBaseProtocolDriver do begin
      Self.StatusTrigger       := StatusTrigger;
      Self.FTimeoutTrigger     := TimeoutTrigger;
      Self.FOutBuffFreeTrigger := OutBuffFreeTrigger;
      Self.FOutBuffUsedTrigger := OutBuffUsedTrigger;
      Self.FNoCarrierTrigger   := NoCarrierTrigger;
      if (Source as TApxBaseProtocolDriver).FComPort <> nil then begin
        if Assigned (Self.FComPort) then
          Self.FComPort.Assign ((Source as TApxBaseProtocolDriver).FComPort) 
      end else
        Self.FComPort := nil;
      Self.FBatchProtocol        := BatchProtocol;
      Self.FFilesSent            := FilesSent;
      Self.FAbortFlag            := AbortFlag;
      Self.FTimerStarted         := TimerStarted;
      Self.FCurProtocol          := CurProtocol;
      Self.FCheckType            := CheckType;
      Self.FHandshakeRetry       := HandshakeRetry;
      Self.FHandshakeWait        := HandshakeWait;
      Self.FHandshakeAttempt     := HandshakeAttempt;
      Self.FBlockLen             := BlockLen;
      Self.FBlockNum             := BlockNum;
      Self.FFlags                := Flags;
      Self.FTransTimeout         := TransTimeout;
      Self.FFinishWait           := FInishWait;
      Self.FRcvTimeout           := RcvTimeout;
      Self.FProtocolStatus       := ProtocolStatus;
      Self.FLastBlockSize        := LastBlockSize;
      Self.FProtocolError        := ProtocolError;
      Self.FSrcFileLen           := SrcFileLen;
      Self.FSrcFileDate          := SrcFileDate;
      Self.FBlockCheck           := BlockCheck;
      Self.FInitFilePos          := InitFilePos;
      Self.FReplyTimer.StartMS   := ReplyTimer.StartMS;
      Self.FReplyTimer.ExpireMS  := ReplyTimer.ExpireMS;
      Self.FDataBlock            := nil;
      Self.FStatusTimer.StartMS  := StatusTimer.StartMS;
      Self.FStatusTimer.ExpireMS := StatusTimer.ExpireMS;
      Self.FForceStatus          := ForceStatus;
      Self.FTimerPending         := TimerPending;
      Self.FInProgress           := InProgress;
      Self.FBlockErrors          := BlockErrors;
      Self.FTotalErrors          := TotalErrors;
      Self.FActCPS               := ActCPS;
      Self.FOverhead             := Overhead;
      Self.FTurnDelay            := TurnDelay;
      Self.FStatusInterval       := StatusInterval;
      Self.FSaveStatus           := SaveStatus;
      Self.FSaveError            := SaveError;
      Self.FBytesRemaining       := BytesRemaining;
      Self.FBytesTransferred     := BytesTransferred;
      Self.FElapsedXfrTime       := ElapsedXfrTime;
      Self.FStatusTime.StartMS   := StatusTime.StartMS;
      Self.FStatusTime.ExpireMS  := StatusTime.ExpireMS;
      Self.FTimer.StartMS        := FTimer.StartMS;
      Self.FTimer.ExpireMS       := FTimer.ExpireMS;
      Self.FEndPending           := EndPending;
      Self.FFileOpen             := FileOpen;
      Self.FNoMoreData           := NoMoreData;
      Self.FLastBlock            := LastBlock;
      Self.FBlkIndex             := BlkIndex;
      Self.FWriteFailOpt         := WriteFailOpt;
      Self.FStartOfs             := StartOfs;
      Self.FEndOfs               := EndOfs;
      Self.FLastOfs              := LastOfs;
      Self.FFileOfs              := FileOfs;
      Self.FEndOfDataOfs         := EndOfDataOfs;
      Self.FSaveMode             := SaveMode;
      Self.FUpcaseFileNames      := UpcaseFileNames;
      Self.FFindingFirst         := FindingFirst;
      Self.FFileListIndex        := FileListIndex;
      Self.FPathName             := PathName;
      Self.FSearchMask           := SearchMask;
      Self.FFFOpen               := FFopen;
      Self.FDestDir              := DestDir;
      Self.FUseStreams           := UseStreams;
    end
  else
    inherited Assign (Source);
end;

procedure TApxBaseProtocolDriver.CloseWorkFile;
begin
  if UseStreams then begin
    apMsgCloseStream (FWorkStream);
  end else
    CloseFile (FWorkFile);
end;

procedure TApxBaseProtocolDriver.ResetWorkFile;
begin
  if UseStreams then begin
    apMsgOpenStream (FWorkStream, True);
  end else begin
    AssignFile (FWorkFile, FPathName);   
    Reset(FWorkFile, 1);
  end;
end;

procedure TApxBaseProtocolDriver.RewriteWorkFile;
begin
  if UseStreams then begin
    apMsgOpenStream (FWorkStream, False);
  end else begin
    AssignFile (FWorkFile, FPathname); 
    Rewrite(FWorkFile, 1);
  end;
end;

function TApxBaseProtocolDriver.WorkFileSize : Integer;
begin
  if UseStreams then 
    Result := FWorkStream.Size
  else
    Result := FileSize (FWorkFile);
end;

function TApxBaseProtocolDriver.WorkFileGetDate : Integer;
begin
  if UseStreams then 
    Result := 0 
  else
    Result := FileGetDate (TFileRec (FWorkFile).Handle);
end;

procedure TApxBaseProtocolDriver.WorkFileSetDate (SrcFileDate : Integer);
begin
  if UseStreams then begin
    { nothing yet }
  end else
    FileSetDate (TFileRec (WorkFile).Name, SrcFileDate);   
end;

procedure TApxBaseProtocolDriver.SeekWorkFile (Position : Longint);
begin
  if UseStreams then 
    FWorkStream.Seek (Position, 0) 
  else
    Seek (FWorkFile, Position);
end;

procedure TApxBaseProtocolDriver.BlockReadWorkFile (var Buf;
                                                        Count : Integer;
                                                    var AmountTransferred : Integer);
begin
  if UseStreams then 
    AmountTransferred := FWorkStream.Read(Buf, Count)
  else
    BlockRead (FWorkFile, Buf, Count, AmountTransferred);
end;

procedure TApxBaseProtocolDriver.BlockWriteWorkFile (var Buf;
                                                        Count : Integer;
                                                    var AmountTransferred : Integer);
begin
  if UseStreams then
    AmountTransferred := FWorkStream.Write (Buf, Count)
  else
    BlockWrite (FWorkFile, Buf, Count, AmountTransferred);
end;

procedure TApxBaseProtocolDriver.EraseWorkFile;
begin
  if UseStreams then
    FWorkStream.Size := 0
  else
    Erase (WorkFile);
end;

procedure TApxBaseProtocolDriver.TruncateWorkFile;
begin
  if UseStreams then
    FWorkStream.Size := FWorkStream.Position
  else
    Truncate (FWorkFile);
end;


{ --------------------------------------------------------------------------- }



{Ascii protocol}

procedure TApxASCIIDriver.InitData;
{-Init data}
{$IFDEF TRIALRUN}
  {$I TRIAL04.INC}
{$ENDIF}
begin
{$IFDEF TRIALRUN}
  TC;
{$ENDIF}
  {Init Ascii data}
  FInterCharDelay   := ApxAsciiDefInterCharDelay;
  FInterLineDelay   := ApxAsciiDefInterLineDelay;
  FEOLChar          := ApxAsciiDefEOLChar;
  FCtrlZEncountered := False;
  FRcvTimeout       := ApxAsciiDefRcvTimeout;
  FMaxAccumDelay    := ApxAsciiDefMaxAccumDelay;
  FBlockLen         := ApxAsciiDefBlockLen;
  CheckType        := bcNone;
  CurProtocol      := Ascii;
  FCRTransMode      := ApxAsciiDefCRTranslate;
  FLFTransMode      := ApxAsciiDefLFTranslate;

end;

function TApxASCIIDriver.Init(Options : Cardinal) : Integer;
{-Allocates and initializes a protocol control block with Options}
begin
  {Check for adequate output buffer size}
  if ComPort.OutBuffUsed + ComPort.OutBuffFree < 1024 then begin
    Init := ecOutputBufferTooSmall;
    Exit;
  end;

  {Allocate block for protocol}
  if apInitProtocolData(ComPort, Options) <> ecOk then begin
    Init := ecOutOfMemory;
    Exit;
  end;

  {Get a protocol DataBlock}
  DataBlock := AllocMem(SizeOf(TApxDataBlock));

  {Can't fail after this}
  Init := ecOK;

  InitData;
end;

function TApxASCIIDriver.Reinit : Integer;
{-Allocates and initializes a protocol control block with Options}
begin
  DataBlock := AllocMem (SizeOf (TApxDataBlock));

  {Can't fail after this}
  Reinit := ecOK;
  InitData;
end;

procedure TApxASCIIDriver.DonePart;
{-Disposes of P}
begin
  FreeMem (DataBlock, SizeOf (TApxDataBlock));
end;

procedure TApxASCIIDriver.Done;
{-Disposes of P}
begin
  DonePart;
  apDoneProtocol;
end;

function TApxASCIIDriver.SetDelays(CharDelay, LineDelay : Cardinal) : Integer;
{-Set the delay (in ms) between each character and each line}
begin
  if FCurProtocol <> Ascii then
    SetDelays := ecBadProtocolFunction
  else begin
    SetDelays := ecOK;
    FInterCharDelay := CharDelay;
    FInterLineDelay := LineDelay;
    FInterCharTime := FInterCharDelay;
    FInterLineTime := FInterLineDelay;
  end;
end;

function TApxASCIIDriver.SetEOLChar(C : Char) : Integer;
{-Set the character used to mark the end of line}
begin
  if CurProtocol <> Ascii then
    SetEOLChar := ecBadProtocolFunction
  else begin
    SetEOLChar := ecOK;
    FEOLChar := C;
  end;
end;

function TApxASCIIDriver.GetLineNumber : LongInt;
{-Return the current line number}
begin
  if CurProtocol <> Ascii then
    GetLineNumber := 0
  else
    GetLineNumber := FBlockNum;
end;

function TApxASCIIDriver.SetEOLTranslation(CR, LF : Cardinal) : Integer;
{-Set the translation modes for CR/LF translations}
begin
  if CurProtocol <> Ascii then
    SetEOLTranslation := ecBadProtocolFunction
  else begin
    SetEOLTranslation := ecOK;
    FCRTransMode := CR;
    FLFTransMode := LF;
  end;
end;

function TApxASCIIDriver.SetEOFTimeout(NewTimeout : LongInt) : Integer;
{-Set the EOF timeout}
begin
  if CurProtocol <> Ascii then
    SetEOFTimeout := ecBadProtocolFunction
  else begin
    SetEOFTimeout := ecOK;
    FRcvTimeout := NewTimeout;
  end;
end;

procedure TApxASCIIDriver.Cancel;
{-Cancel the Ascii protocol}
begin
  {Flush anything that might be left in the output buffer}
  ComPort.FlushOutBuffer;
end;

function TApxASCIIDriver.SendBlockPart(var Block : TApxDataBlock) : Boolean;
{-Send part of the block, return True when finished}
var
  AccumDelay : Cardinal;
  C : Char;
  Finished : Boolean;

  procedure SendChar(C : Char);
      {-Send current character and increment count}
  begin
    {Send the character}
    ComPort.PutChar(C);
    Inc(FBytesTransferred);
    Dec(FBytesRemaining);
  end;

begin
  {Assume not finished}
  SendBlockPart := False;

  {Send as much data as we can}
  AccumDelay := 0;
  Finished := FSendIndex >= FLastBlockSize;
  while not Finished do begin
    {Get next character to send}
    Inc(FSendIndex);
    C := Block[FSendIndex];

    {Check character before sending}
    case C of

      ^M :
        if FCRTransMode = atAddLFAfter then begin
          ComPort.PutString(^M^J);
          Inc(FBytesTransferred, 2);
          Dec(FBytesRemaining, 2);
        end else if FCRTransMode <> atStrip then
          SendChar(^M);

      ^J :
        if FLFTransMode = atAddCRBefore then begin
          ComPort.PutString(^M^J);
          Inc(FBytesTransferred, 2);
          Dec(FBytesRemaining, 2);
        end else if FLFTransMode <> atStrip then
          SendChar(^J);

      else begin
        if C = atEOFMarker then begin
          if FlagIsSet(FFlags, apAsciiSuppressCtrlZ) then begin
            SendBlockPart := True;
            Exit;
          end;
        end else
          SendChar(C);
        end;                                                         
      end;

    {Check for interline delay}
    if (C = FEOLChar) and (FInterLineDelay > 0) then begin
      if FInterLineDelay + AccumDelay < FMaxAccumDelay then begin
        DelayMS(FInterLineDelay, False);
        Inc(AccumDelay, FInterLineDelay);
      end else begin
        ComPort.SetTimerTrigger(FTimeoutTrigger, FInterLineTime, True);
        FAsciiState := taSendDelay;
        Exit;
      end;
    end;

    {Check for interchar delay}
    if FInterCharDelay > 0 then begin
      if FInterCharDelay + AccumDelay < FMaxAccumDelay then begin
        DelayMS(FInterCharDelay, False);
        Inc(AccumDelay, FInterCharDelay);
      end else begin
        ComPort.SetTimerTrigger(FTimeoutTrigger, FInterCharTime, True);
        FAsciiState := taSendDelay;
        Exit;
      end;
    end;

    {Set Finished flag}
    Finished := (FSendIndex >= FLastBlockSize) or
                (AccumDelay > FMaxAccumDelay);
  end;

  {End of block if we get here}
  SendBlockPart := True;
end;

function TApxASCIIDriver.CollectBlock(var Block : TApxDataBlock) : Boolean;
{-Collect received data into aDataBlock, return True for full block}
{-Note: may go one past BlockLen}
var
  C : Char;
  GotEOFMarker : Boolean;

  procedure AddChar(C : Char);
  {-Add C to buffer}
  begin
    Inc(FBlkIndex);
    Block[FBlkIndex] := C;
  end;

begin
  CollectBlock := False;
  GotEOFMarker := False;
  while FComPort.CharReady and (FBlkIndex < FBlockLen) do begin

    {Start the protocol timer if first time thru}
    if FTimerPending then begin
      FTimerPending := False;
      NewTimer(FTimer, 0);
    end;

    {Get the char}
    FComPort.ValidDispatcher.GetChar(C);

    {Character translations}
    case C of
      ^M :
        if FCRTransMode = atAddLFAfter then begin
          AddChar(^M);
          AddChar(^J);
        end else if FCRTransMode <> atStrip then
          AddChar(^M);

      ^J :
        if FLFTransMode = atAddCRBefore then begin
          AddChar(^M);
          AddChar(^J);
        end else if FCRTransMode <> atStrip then
          AddChar(^J);

      else begin
        if C = atEOFMarker then begin
          GotEOFMarker := True;
          if not FlagIsSet(FFlags, apAsciiSuppressCtrlZ) then
            AddChar(atEOFMarker);
          end else
            AddChar(C);
        end;
      end;
    CollectBlock := (FBlkIndex >= FBlockLen) or (GotEOFMarker); 
  end;
end;

procedure TApxASCIIDriver.ReceiveBlock(var Block : TApxDataBlock;
                           var BlockSize : Cardinal; var HandShake : Char);
{-Receive block into Buffer}
var
    I : Cardinal;
begin
  {Check for ^Z}
  I := 1;
  while (I < BlockSize) and not FCtrlZEncountered do begin
    if Block[I] = atEOFMarker then begin
      BlockSize := I;
      FCtrlZEncountered := True;
    end else
      Inc(I);
  end;

  {Update data areas and show status}
  Inc(FBytesTransferred, BlockSize);
  FElapsedXfrTime := ElapsedTime(FTimer);
end;

procedure TApxASCIIDriver.PrepareTransmit;
{-Prepare for transmitting ASCII}
begin
  FFindingFirst := True;
  FFileListIndex := 0;
  apResetStatus;
  apShowFirstStatus;
  if not NextFile (FPathname) then begin
    apShowLastStatus;
    Exit;
  end;

  FCtrlZEncountered := False;
  FBlockNum := 0;
  FForceStatus := True;
  FAsciiState := taInitial;
  FProtocolError := ecOK;
  FNoMoreData := False;
end;

procedure TApxASCIIDriver.Transmit(Msg, wParam : Cardinal;
                     lParam : LongInt);
{-Performs one increment of an ASCII transmit}
var
  TriggerID    : Cardinal absolute wParam;
  Finished     : Boolean;
  StatusTimeMS : Cardinal;
begin
  {Function result is always zero unless the protocol is over}

  EnterCriticalSection(FProtSection); 

  {Exit if protocol was cancelled while waiting for crit section}
  if FAsciiState = taDone then begin
      LeaveCriticalSection(FProtSection); 
    Exit;
  end;

  {Force TriggerID for TriggerAvail messages}
  if Msg = apx_TriggerAvail then
    TriggerID := ApxProtocolDataTrigger;

  repeat

    FComPort.DebugLog.AddDebugEntry (TApxCustomProtocol,   
                                    Cardinal (AxdtAscii),
                                    Cardinal (ApxLogAsciiState[FAsciiState]),
                                    0);

    {Check for user or remote abort}
    if (Integer(TriggerID) = FNoCarrierTrigger) or
       (Msg = apx_ProtocolCancel) then begin
      if Integer(TriggerID) = FNoCarrierTrigger then
        FProtocolStatus := psAbortNoCarrier
      else
        FProtocolStatus := psCancelRequested;
      Cancel;
      FAsciiState := taFinished;
      FForceStatus := False;
    end;

    if (Integer(TriggerID) = FStatusTrigger) or FForceStatus then begin
      FElapsedXfrTime := ElapsedTime(FTimer);
      if FComPort.Dispatcher.TimerTimeRemaining (FStatusTrigger, 
                                  StatusTimeMS) <> 0 then
        StatusTimeMS := 0;
      if LongInt (StatusTimeMS) <= 0 then begin
        ShowStatus(0);
        FComPort.Dispatcher.SetTimerTrigger(FStatusTrigger, FStatusInterval, True);
        FForceStatus := False;
      end;
    end;

    {Process current state}
    case FAsciiState of
      taInitial :
        begin
          {Pathname must already be set before we get here}
          if FUpcaseFileNames then
            AnsiStrUpper(FPathname);

          {Show file name to user logging routine}
          LogFile(lfTransmitStart);

          {Go prepare for reading protocol blocks}
          PrepareReading;
          if FProtocolError = ecOK then begin
            FAsciiState := taGetBlock;
            FComPort.Dispatcher.SetTimerTrigger(FTimeoutTrigger, FTransTimeout, True);
          end else
            FAsciiState := taFinished;
          NewTimer(FTimer, 50);
        end;

      taGetBlock :
        begin
          FLastBlockSize := FBlockLen;
          FNoMoreData := ReadProtocolBlock(FDataBlock^, FLastBlockSize);
          if (FProtocolError = ecOK) and (FLastBlockSize <> 0) then begin
            FAsciiState := taWaitFreespace;
            FSendIndex := 0;
            FComPort.Dispatcher.SetTimerTrigger(FTimeoutTrigger, FTransTimeout, True);
            FComPort.Dispatcher.SetStatusTrigger(FOutBuffFreeTrigger, FBlockLen+1, True);
          end else
            FAsciiState := taFinished;
        end;

      taWaitFreeSpace :
        if Integer(TriggerID) = FOutBuffFreeTrigger then
          FAsciiState := taSendBlock
        else if Integer(TriggerID) = FTimeoutTrigger then
          FAsciiState := taFinished;

      taSendBlock :
        if SendBlockPart(FDataBlock^) then begin
          {Adjust block number and file position}
          Inc(FBlockNum);
          Inc(FFileOfs, FBlockLen);

          {Go get next block to send}
          if FNoMoreData then begin
            FAsciiState := taFinishDrain;
            FComPort.Dispatcher.SetTimerTrigger(FTimeoutTrigger, FTransTimeout, True);
            FComPort.Dispatcher.SetStatusTrigger(FOutBuffUsedTrigger, 0, True);
          end else
            FAsciiState := taGetBlock;

          {Update timer and force status}
          FElapsedXfrTime := ElapsedTime(FTimer);
          FForceStatus := True;
        end;

      taSendDelay :
        if Integer(TriggerID) = FTimeoutTrigger then
          FAsciiState := taSendBlock;

      taFinishDrain :
        if Integer(TriggerID) = FOutBuffUsedTrigger then begin
          FAsciiState := taFinished;
          FProtocolStatus := psEndFile;
        end else if Integer(TriggerID) = FTimeoutTrigger then begin
          apProtocolError(ecTimeout);
          FAsciiState := taFinished;
        end;

      taFinished :
        begin
          if FProtocolError <> ecOK then
            FComPort.FlushInBuffer;  

          {Close the file}
          FinishReading;

          {Show status, user logging, and clean up}
          if FProtocolError = ecOK then
            LogFile(lfTransmitOk)
          else
            LogFile(lfTransmitFail);
          apShowLastStatus;

          {Tell parent we're finished}
          apSignalFinish (True);
          FAsciiState := taDone;
        end;
    end;

    {Should we exit or not}
    case FAsciiState of
      {Stay in state machine}
      taGetBlock,
      taSendBlock,
      taFinished :
        Finished := False;

      {Exit state machine to wait for trigger}
      taFinishDrain,
      taWaitFreeSpace,
      taSendDelay,
      taDone :
        Finished := True;
        
      else
        Finished := False;
    end;
  until Finished;

    LeaveCriticalSection(FProtSection);    
end;

procedure TApxASCIIDriver.PrepareReceive;
{-Prepare to receive by Ascii protocol}
begin
  {Prepare to enter state machine}
  FAsciiState := raInitial;
  FCtrlZEncountered := False;
  apResetStatus;
  apShowFirstStatus;
  FProtocolError := ecOK;
  FForceStatus := False;
end;

procedure TApxASCIIDriver.Receive(Msg, wParam : Cardinal;
                     lParam : LongInt);
{-Performs one increment of an Ascii receive}
label
  ExitPoint;
var
  TriggerID    : Cardinal absolute wParam;
  BlockSize    : Cardinal;
  Finished     : Boolean;
  Handshake    : Char;
  StatusTimeMS : Cardinal;

  procedure Cleanup(DisposeBuffers : Boolean);
  {-Handle error reporting and other cleanup}
  begin
    apShowLastStatus;

    {Tell parent we're finished}
    apSignalFinish (False);
    FAsciiState := raDone;
  end;

begin

  EnterCriticalSection (FProtSection);

      {Exit if protocol was cancelled while waiting for crit section}
  if FAsciiState = raDone then begin
    LeaveCriticalSection (FProtSection);
    Exit;
  end;

      {Force TriggerID for TriggerAvail messages}
  if Msg = apx_TriggerAvail then
    TriggerID := ApxProtocolDataTrigger;

  repeat

    FComPort.DebugLog.AddDebugEntry (TApxCustomProtocol,  
                                     Cardinal (AxdtAscii),
                                     Cardinal (ApxLogAsciiState[FAsciiState]),
                                     0);
      
    {Check for use abort}
    if (Integer(TriggerID) = FNoCarrierTrigger) or
       (Msg = apx_ProtocolCancel) then begin
      if Integer(TriggerID) = FNoCarrierTrigger then
        FProtocolStatus := psAbortNoCarrier
      else
        FProtocolStatus := psCancelRequested;
        Cancel;
        FAsciiState := raFinished;
        FForceStatus := False;
    end;

    {Show status periodically}
    if (Integer(TriggerID) = FStatusTrigger) or FForceStatus then begin
      if FComPort.Dispatcher.TimerTimeRemaining (FStatusTrigger,    
                                         StatusTimeMS) <> 0 then
        StatusTimeMS := 0;
      if LongInt (StatusTimeMS) <= 0 then begin
        ShowStatus(0);
        FComPort.Dispatcher.SetTimerTrigger(FStatusTrigger, FStatusInterval, True);
        FForceStatus := False;
      end;
    end;

        {Process current state}
    case FAsciiState of
      raInitial :
        begin
          {Pathname should already have name of file to receive}
          if FPathname[0] = #0 then begin
            apProtocolError(ecNoFilename);
            Cleanup(True);
            goto ExitPoint; 
          end else if FUpcaseFileNames then
            AnsiStrUpper(FPathname);

          {Send file name to user's LogFile procedure}
          LogFile(lfReceiveStart);

          {Accept this file}
          if not AcceptFile(FPathname) then begin
            Cancel;
            FProtocolStatus := psFileRejected;
            goto ExitPoint; 
          end;

          {Prepare file for writing protocol blocks}
          PrepareWriting;
          if (FProtocolError <> ecOK) or
             (FProtocolStatus = psCantWriteFile) then begin
            if FProtocolStatus = psCantWriteFile then
              FProtocolError := ecCantWriteFile;
            Cancel;
            FAsciiState := raFinished;
            goto ExitPoint; 
          end;

          {Prepare to collect first block}
          FAsciiState := raCollectBlock;
          FFileOfs := 0;
          FBlockNum := 1;
          FBlkIndex := 0;
          FForceStatus := True;

          FComPort.Dispatcher.SetTimerTrigger(FTimeoutTrigger, FRcvTimeout, True);
          FTimerPending := True;
        end;

      raCollectBlock :
        if TriggerID = ApxProtocolDataTrigger then begin
          if CollectBlock(FDataBlock^) then
            FAsciiState := raProcessBlock;
        end else if Integer(TriggerID) = FTimeoutTrigger then begin
          {Timeout out waiting for complete block, assume EOF}
          FCtrlZEncountered := True;
          FAsciiState := raProcessBlock;
        end;

      raProcessBlock :
        begin
          {Go process what's in aDataBlock}
          BlockSize := FBlkIndex;
          ReceiveBlock(FDataBlock^, BlockSize, Handshake);
          WriteProtocolBlock(FDataBlock^, BlockSize);
          if FProtocolError = ecOK then begin
            {Normal received block}
            Inc(FFileOfs, BlockSize);
            FComPort.Dispatcher.SetTimerTrigger(FTimeoutTrigger, FRcvTimeout, True);
            FForceStatus := True;
            FBlkIndex := 0;
            if FCtrlZEncountered then
              FAsciiState := raFinished
            else
              FAsciiState := raCollectBlock;
          end else
            {Error during write, clean up and exit}
            FAsciiState := raFinished;
        end;

      raFinished :
        begin
          FinishWriting;
          LogFile(lfReceiveOk);
          Cleanup(True);
        end;
    end;

ExitPoint:
        {Should we exit or not}
    case FAsciiState of
      {Stay in state machine or exit?}
      raProcessBlock,
      raFinished :
        Finished := False;

      raCollectBlock :
        begin
          Finished := not FComPort.Dispatcher.CharReady;
          TriggerID := ApxProtocolDataTrigger;
        end;

      raDone :
        Finished := True;

      else
        Finished := True;
    end;

  until Finished;

    LeaveCriticalSection(FProtSection);   
end;

procedure TApxASCIIDriver.Assign (Source : TPersistent);
begin
  if (Source is TApxASCIIDriver) then
    with Source as TApxASCIIDriver do begin
      Self.FCtrlZEncountered := FCtrlZEncountered;
      Self.FInterCharDelay   := FInterCharDelay;
      Self.FInterLineDelay   := FInterLineDelay;
      Self.FInterCharTime    := FInterCharTime;
      Self.FInterLineTime    := FInterLineTime;
      Self.FMaxAccumDelay    := FMaxAccumDelay;
      Self.FSendIndex        := FSendIndex;
      Self.FCRTransMode      := FCRTransMode;
      Self.FLFTransMode      := FLFTransMode;
      Self.FEOLChar          := FEOLChar;
      Self.FASCIIState       := FASCIIState;
    end;
  inherited Assign (Source);
end;

initialization

  InitializeUnit;

end.






