(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                 AXSYSTEM.PAS 1.02                     *}
{*********************************************************}

{Global defines potentially affecting this unit}
{$I AxDefine.inc}

{Options required for this unit}
{$X+,B-,I-,T-,J+}

unit AxSystem;
  {-Platform independent system routines}

interface

uses
  {$IFDEF Win32}
  Windows,
  MMSystem,
  {$ENDIF}
  {$IFDEF Linux}
  Libc,
  Types,
  Qt,
  QTypes,
  Classes,
  {$ENDIF}
  SysUtils;

{$IFDEF Win32}
type
  QPCCast = record
    LowPart: DWORD;
    HighPart: DWORD;
  end;

const
  QPFreq : DWORD = 0;
{$ENDIF}

{$IFDEF Linux}
type
  EApxExecError = class (Exception);
{$ENDIF}

  function AxCharLowerBuff(Str : PAnsiChar; Len : DWORD): DWORD;
  {$IFDEF Linux}
  function AxExecAndWait (FileName : PAnsiChar; CommandLine : PAnsiChar;
                          Visibility : Integer; Wait : Boolean) : Integer;
  {$ELSE}
  function AxExecAndWait (FileName : PAnsiChar; CommandLine : PAnsiChar;
                          Visibility : Integer) : Integer;
  {$ENDIF}
  function AxGetOSVersionString : string;
  function AxExtractComNumber(const DeviceName : string) : Word;
  function AxIsDeviceAvailable(const Device : string) : Boolean;
  function AxIsDeviceValid(const Device : string) : Boolean;
  function AxIsPortAvailable(const ComNum : Word) : Boolean;
  function AxIsPortValid(const ComNum : Word) : Boolean;
  function AxMakeComName(const ComNum : Word) : string;
  function AxTimeGetTime : DWORD;

implementation

{$IFDEF WIN32}
function AxCharLowerBuff(Str : PAnsiChar; Len : DWORD) : DWORD;
begin
  Result := CharLowerBuff(Str, Len);
end;
{$ENDIF}

{$IFDEF Linux}
function AxCharLowerBuff(Str : PAnsiChar; Len : DWORD) : DWORD;
var
  Temp : string;
begin
  Result := Len;
  SetString(Temp, Str, Len);
  StrLCopy(Str, PChar(AnsiLowerCase(Temp)), Len);
end;
{$ENDIF}

{$IFDEF WIN32}
{ returns -1 if the Exec failed, otherwise returns the process' exit }
{ code when the process terminates }
function AxExecAndWait(FileName : PAnsiChar; CommandLine : PAnsiChar;
  Visibility : Integer) : Integer;
var
  zAppName : array[0..512] of AnsiChar;
  zCurDir : array[0..255] of AnsiChar;
  WorkDir : ShortString;
  StartupInfo : TStartupInfo;
  ProcessInfo : TProcessInformation;
  Temp : DWORD;
begin
  StrCopy(zAppName, FileName);
  if assigned(CommandLine) then
    StrCat(zAppName, CommandLine);
  GetDir(0, WorkDir);
  StrPCopy(zCurDir, WorkDir);
  FillChar(StartupInfo, Sizeof(StartupInfo),#0);
  StartupInfo.cb := Sizeof(StartupInfo);
  StartupInfo.dwFlags := STARTF_USESHOWWINDOW;
  StartupInfo.wShowWindow := Visibility;
  if not CreateProcess(nil,
      zAppName,              { pointer to command line string }
      nil,                   { pointer to process security attributes }
      nil,                   { pointer to thread security attributes }
      false,                 { handle inheritance flag }
      CREATE_NEW_CONSOLE or  { creation flags }
      NORMAL_PRIORITY_CLASS,
      nil,                   { pointer to new environment block }
      nil,                   { pointer to current directory name }
      StartupInfo,           { pointer to STARTUPINFO }
      ProcessInfo) then      { pointer to PROCESS_INF }
        Result := -1
  else begin
    WaitforSingleObject(ProcessInfo.hProcess,INFINITE);
    GetExitCodeProcess(ProcessInfo.hProcess,Temp);
    Result := Integer(Temp);
  end;
end;
{$ENDIF}

{$IFDEF Linux}
function AxExecAndWait(FileName : PAnsiChar; CommandLine : PAnsiChar;
  Visibility : Integer; Wait : Boolean) : Integer;
const
  MaxArgs = 256;                // Maximum number of arguments that can
                                // be passed on the command line.

var
  Arguments : Array [0..MaxArgs - 1] of PChar;  // Array of arguments

  function GetNextArgument (var StartPos : Integer) : PChar;

  // This gets the next argument from the CommandLine.  This function does
  // not take into account arguments that contain quotes.

  var
    FirstPos : Integer;
  begin
    if not Assigned (CommandLine) then begin
      Result := nil;
      exit;
    end;

    while (CommandLine[StartPos] = ' ') and
          (StartPos <= Integer (StrLen (CommandLine))) do
      Inc (StartPos);
    FirstPos := StartPos;
    while (CommandLine[StartPos] <> ' ') and
          (StartPos <= Integer (StrLen (CommandLine))) do
      Inc (StartPos);
    if Trim (Copy (CommandLine, FirstPos, StartPos - FirstPos)) = '' then begin
      Result := nil;
      exit;
    end;
    Result := StrAlloc (StartPos - FirstPos + 1);
    StrLCopy (Result, CommandLine + FirstPos, StartPos - FirstPos);
    Result[StartPos - FirstPos + 1] := #$00;
  end;

  function GetFirstArgument : PChar;

  // The first argument should always be the name of the calling program.

  begin
    Result := StrAlloc (StrLen (FileName) + 1);
    StrPLCopy (Result, FileName, StrLen (FileName));
    Result [StrLen (FileName)] := #$00;
  end;

  procedure SplitArguments;
  var
    CurrentArgument : integer;
    StringPosition : integer;
  begin
    StringPosition := 0;
    CurrentArgument := 1;

    Arguments[0] := GetFirstArgument;

    repeat
      Arguments [CurrentArgument] := GetNextArgument (StringPosition);
      Inc (CurrentArgument);
      if CurrentArgument > MaxArgs then
        raise EApxExecError.Create ('Too many arguments');
    until Arguments[CurrentArgument-1] = nil;
  end;

  procedure ReleaseArgumentMemory;
  var
    i : integer;
  begin
    i := 0;
    while (Arguments[i] <> nil) do begin
      StrDispose (Arguments[i]);
      Inc (i);
    end;
  end;

  function ForkCommand : Integer;
  var
    pid : pid_t;
    Status : integer;
  begin
    Result := 0;
    pid := fork;
    if pid = -1 then
      raise EApxExecError.Create ('Unable to fork process')
    else if pid = 0 then begin
      if execvp (FileName, @Arguments)  = -1 then
        raise EApxExecError.Create ('Unable to execute process');
    end else begin
      if Wait then begin
        waitpid (pid, @status, WUNTRACED);
        Result := WEXITSTATUS(status);
      end;
    end;
  end;

begin
  Arguments[0] := nil;
  SplitArguments;
  try
    Result := ForkCommand;
  finally
    ReleaseArgumentMemory;
  end;
end;
{$ENDIF}

{$IFDEF Win32}
function AxExtractComNumber(const DeviceName : string) : Word;
begin
  Result := 0;
end;
{$ENDIF}

{$IFDEF Linux}
function AxExtractComNumber(const DeviceName : string) : Word;
const
  StdDevice = '/dev/ttyS';
  NumSet = ['0'..'9'];
var
  I : Integer;
  Temp : string;
begin
  Result := 0;
  if Pos(StdDevice, DeviceName) = 1 then begin
    Temp := Copy(DeviceName, Length(StdDevice)+1, Length(DeviceName));
    for I := 1 to Length(Temp) do
      if not (Temp[I] in NumSet) then Exit;
    Result := StrToInt(Temp)+1;
  end;
end;
{$ENDIF}

{$IFDEF WIN32}
{ Returns string describing OS version }
function AxGetOSVersionString : string;
var
  OSVersion : TOSVersionInfo;
begin
  OSVersion.dwOSVersionInfoSize := SizeOf(OSVersion);
  GetVersionEx(OSVersion);
  case OSVersion.dwPlatformID of
    VER_PLATFORM_WIN32s        : Result := 'Win32s on Windows ';
    VER_PLATFORM_WIN32_WINDOWS : Result := 'Win32 on Windows ';
    VER_PLATFORM_WIN32_NT      : Result := 'Windows NT ';
    else Result := 'Unknown';
  end;
  Result := Result + IntToStr(OSVersion.dwMajorVersion) + '.' +
    IntToStr(OSVersion.dwMinorVersion) + ' ' + StrPas(OSVersion.szCSDVersion);
end;
{$ENDIF}

{$IFDEF Linux}
function AxGetOSVersionString : string;
var
  NR : utsname;
begin
  if uname(NR) = 0 then
    Result := Format('%s %s %s', [NR.sysname, NR.release, NR.machine])
  else
    Result := 'Linux';
end;
{$ENDIF}

{$IFDEF Win32}
function AxIsDeviceAvailable(const Device : string) : Boolean;
var
  R : DWORD;
begin
  R := CreateFile(PAnsiChar(Device), GENERIC_READ or GENERIC_WRITE, 0, nil,
    OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL or FILE_FLAG_OVERLAPPED, 0);
  if R <> INVALID_HANDLE_VALUE then begin
    Result := True;
    CloseHandle(R);
  end else
    Result := False;
end;
{$ENDIF}

{$IFDEF Linux}
function AxIsDeviceAvailable(const Device : string) : Boolean;
begin
  Result := AxIsDeviceValid(Device);
end;
{$ENDIF}

{$IFDEF Win32}
function AxIsDeviceValid(const Device : string) : Boolean;
var
  R : DWORD;
begin
  R := CreateFile(PAnsiChar(Device), GENERIC_READ or GENERIC_WRITE, 0, nil,
    OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL or FILE_FLAG_OVERLAPPED, 0);
  if R <> INVALID_HANDLE_VALUE then begin
    Result := True;
    CloseHandle(R);
  end else
    if (GetLastError = 5) then
      Result := True
    else
      Result := False;
end;
{$ENDIF}

{$IFDEF Linux}
function AxIsDeviceValid(const Device : string) : Boolean;
var
  Handle, Stat : Integer;
begin
  Result := False;
  Handle := Open(PAnsiChar(Device), O_RDWR or O_NOCTTY or O_NDELAY);
  if Handle >= 0 then begin
    Result := (ioctl(Handle, $5415 {TIOCMGET}, @Stat) <> -1);
    __close(Handle);
  end;
end;
{$ENDIF}

function AxIsPortAvailable(const ComNum : Word) : Boolean;
begin
  Result := AxIsDeviceAvailable(AxMakeComName(ComNum));
end;

function AxIsPortValid(const ComNum : Word) : Boolean;
begin
  Result := AxIsDeviceValid(AxMakeComName(ComNum));
end;

{$IFDEF Win32}
{ Return a platform appropriate device name for the COM port number }
function AxMakeComName(const ComNum : Word) : string;
begin
  if ComNum = 0 then
    Result := ''
  else
    Result := Format('\\.\COM%d', [ComNum]);
end;
{$ENDIF}

{$IFDEF Linux}
{ Return a platform appropriate device name for the COM port number }
function AxMakeComName(const ComNum : Word) : string;
begin
  if ComNum = 0 then
    Result := ''
  else
    Result := Format('/dev/ttyS%d', [Pred(ComNum)]);
end;
{$ENDIF}

{$IFDEF WIN32}
{ Centralized timer method -- will use QueryPerformaceCounter }
{ if avail, return is the number of ms since the system started }
function AxTimeGetTime : DWORD;
var
  Count : TLargeInteger;
begin
  if QPFreq <> 0 then begin
    QueryPerformanceCounter(Count);
    asm
      xor edx, edx
      mov eax, QPCCast(Count).HighPart
      div QPFreq
      mov eax, QPCCast(Count).LowPart
      div QPFreq
      mov Result, eax
    end;
  end else
    Result := timeGetTime;
end;
{$ENDIF}

{$IFDEF Linux}
function AxTimeGetTime : DWORD;
var
  TV : TTimeVal;
  TZ : TTimeZone;
begin
  GetTimeOfDay(TV, TZ);
  Result := (TV.tv_sec * 1000) + (TV.tv_usec div 1000);
end;
{$ENDIF}

{$IFDEF WIN32}
var
  Freq : TLargeInteger;
{$ENDIF}

initialization
  {$IFDEF WIN32}
  QueryPerformanceFrequency(Freq);
  { Our assumption that HighPart is zero should be good for }
  { a long time but we'll do this safety check just in case }
  if (QPCCast(Freq).HighPart <> 0) or
     (QPCCast(Freq).LowPart = 0) then Exit;

  QPFreq := QPCCast(Freq).LowPart div 1000;
  if (QPCCast(Freq).LowPart mod 1000) > 500 then Inc(QPFreq);
  {$ENDIF}

end.


