(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                   AxTerm.pas 1.02                     *}
{*********************************************************}

{ Global defines potentially affecting this unit }
{$I AxDefine.inc}
                                                             
{Options required for this unit}
{$Z-}

{$IFOPT R+}
  {$DEFINE UseRangeChecks}
{$ENDIF}

{.$DEFINE TermDebug}
{.$DEFINE DebugGrid}

{$IFOPT D+}
  {$DEFINE CompileDebugCode}
{$ENDIF}

{$R AxTrmVT1.r32}
{$R AxChsVT1.r32}

unit AxTerm;
  { Terminal and Emulator components }

interface

uses
  SysUtils,
  Classes,
  Types,
  Libc,
  Qt,
  QControls,
  QGraphics,
  QForms,
  QStdCtrls,
  QExtCtrls,
  QClipBrd,
  AxMisc,
  AxPort,
  AxExcept,
  AxString;

type
  TApxCaptureMode = (   { terminal data capture modes.. }
          cmOff,        { ..no capturing of data }
          cmOn,         { ..capture new data to new file }
          cmAppend);    { ..capture data, appending to old file }

  TApxTerminalCursorMovedEvent =
     procedure (aSender : TObject; aRow, aCol : integer) of object;

  TApxCursorType = (    { cursor types }
    ctNone,             { ..no cursor is visible }
    ctUnderline,        { ..underline cursor }
    ctBlock);           { ..block cursor }

  TApxVT100LineAttr = ( { special VT100 line attributes }
    ltNormal,           { ..normal, no special attrs }
    ltDblHeightTop,     { ..line is top half of double-height line }
    ltDblHeightBottom,  { ..line is bottom half of double-height line }
    ltDblWidth);        { ..line is double-width }

const
  { Scroll bar constants }
  CYHSCROLL = 15;
  CXVSCROLL = 15;

const
  { Buffer default property values }
  adc_TermBufForeColor = clSilver;
  adc_TermBufBackColor = clBlack;
  adc_TermBufColCount = 80;
  adc_TermBufRowCount = 24;
  adc_TermBufScrollRowCount = 200;
  adc_TermBufUseAbsAddress = True;     { use absolute rows/col values }
  adc_TermBufUseAutoWrap = True;       { chars wrap at end of line }
  adc_TermBufUseAutoWrapDelay = True;  { ..but delay slightly }
  adc_TermBufUseInsertMode = False;    { no insert mode }
  adc_TermBufUseNewLineMode = False;   { LF char is a pure linefeed }
  adc_TermBufUseScrollRegion = False;  { no scroll region used }

const
  { default property values for the terminal }
  adc_TermActive = True;
  adc_TermBackColor = adc_TermBufBackColor;
  adc_TermBlinkTime = 500;
  adc_TermBorderStyle = bsSingle;
  adc_TermCapture = cmOff;
  adc_TermCaptureFile = 'APROTERM.CAP';
  adc_TermColumnCount = adc_TermBufColCount;
  adc_TermCursorType = ctBlock;

  adc_TermFontSize = 15;
  adc_TermFontCharSet = fcsLatin1;
  adc_TermFontPitch = fpVariable;
  adc_TermFontName = 'misc-fixed';
  adc_TermForeColor = adc_TermBufForeColor;

  adc_TermHalfDuplex = False;
  adc_TermHeight = 200;
  adc_TermLazyByteDelay = 200;
  adc_TermLazyTimeDelay = 100;
  adc_TermRowCount = adc_TermBufRowCount;
  adc_TermScrollback = False;
  adc_TermScrollRowCount = adc_TermBufScrollRowCount;
  adc_TermUseLazyDisplay = True;
  adc_TermWantAllKeys = True;
  adc_TermWidth = 300;
  adc_TermLineSpacing = 2;

  { default property values for the VT100 emulator--'power-up' values }
  adc_VT100ANSIMode = True;
  adc_VT100Answerback = 'APROterm';
  adc_VT100AppKeyMode = False;
  adc_VT100AppKeypadMode = False;
  adc_VT100AutoRepeat = True;
  adc_VT100Col132Mode = False;
  adc_VT100G0CharSet = 0;
  adc_VT100G1CharSet = 0;
  adc_VT100GPOMode = False;
  adc_VT100InsertMode = adc_TermBufUseInsertMode;
  adc_VT100Interlace = False;
  adc_VT100NewLineMode = adc_TermBufUseNewLineMode;
  adc_VT100RelOriginMode = not adc_TermBufUseAbsAddress;
  adc_VT100RevScreenMode = False;
  adc_VT100SmoothScrollMode = False;
  adc_VT100WrapAround = adc_TermBufUseAutoWrap;

type
  TApxtWordArray = array [0..MaxInt div sizeof(word) - 1] of word;
  PApxtWordArray = ^TApxtWordArray;
  TApxtLongArray = array [0..MaxInt div sizeof(longint) - 1] of longint;
  PApxtLongArray = ^TApxtLongArray;

  TApxTerminalCharAttr = (  { character attributes }
         tcaBold,           { ..bold }
         tcaUnderline,      { ..underlined }
         tcaStrikethrough,  { ..struck through as if deleted }
         tcaBlink,          { ..blinking }
         tcaReverse,        { ..back/foreground colors reversed }
         tcaInvisible,      { ..invisible }
         tcaSelected);      { ..selected }
  TApxTerminalCharAttrs = set of TApxTerminalCharAttr;

  TApxScrollRowsNotifyEvent =
     procedure (aSender : TObject;
                aCount, aTop, aBottom : integer) of object;

  TApxOnCursorMovedEvent =
     procedure (ASender  : TObject;
                Row, Col : integer) of object;

  TApxTerminalArray = class
    private
      FActColCount : integer;
      FColCount    : integer;
      FDefaultItem : longint;
      FItems       : PAnsiChar;
      FItemSize    : integer;
      FRowCount    : integer;
    protected
      procedure taSetColCount(aNewCount : integer);
      procedure taSetRowCount(aNewCount : integer);

      procedure taClearRows(aBuffer : PAnsiChar;
                            aActColCount : integer;
                            aStartRow, aEndRow : integer);
      procedure taGrowArray(aRowCount,
                            aColCount, aActColCount : integer);
    public
      constructor Create(aItemSize : integer);
      destructor Destroy; override;

      procedure Clear;
        { clear the entire array, filling with DefaultItem }

      procedure ClearItems(aRow : integer;
                           aFromCol, aToCol : integer);
        { clear the row in between the two inclusive columns, filling }
        { with DefaultItem (equivalent to 'erase') }

      procedure DeleteItems(aCount : integer;
                            aRow   : integer;
                            aCol   : integer);
        { delete aCount default items at aRow , aCol; items currently }
        { beyond these positions are pulled left, and their original  }
        { positions filled with default items; this action does not   }
        {  extend beyond the row }


      function GetItemPtr(aRow, aCol : integer) : pointer;
        { return a pointer to the item at aRow, aCol; caller must }
        { properly dereference, and not memory overread }

      procedure InsertItems(aCount : integer;
                            aRow   : integer;
                            aCol   : integer);
        { insert aCount default items at aRow , aCol; items currently }
        { in these positions are pushed right, but not beyond the row }
        { boundary }

      procedure ReplaceItems(aOldItem : pointer;
                             aNewItem : pointer);
        { replace every incidence of aOldItem with aNewItem }

      procedure SetDefaultItem(aDefaultItem : pointer);
        { define the default item to be used to fill new items, eg, }
        { during scrolling, clearing or resizing }

      procedure ScrollRows(aCount : integer;
                           aStartRow, aEndRow : integer);
        { scroll the data by aCount rows, filling new rows with    }
        { DefaultItem; scroll just between aStartRow and aEndRow   }
        { inclusive; if aCount is +ve it means scroll upwards (ie, }
        { the usual sense), if -ve scroll downwards }

      procedure WriteItems(aItems : pointer;
                           aCount : integer;
                           aRow, aCol : integer);
        { write aCount items to aRow, aCol, without wrapping at the }
        { end of the row }

      procedure WriteDupItems(aItem  : pointer;
                              aCount : integer;
                              aRow, aCol : integer);
        { write aCount copies of aItem to aRow, aCol, without wrapping }
        { at the end of the row }

      property ColCount : integer read FColCount write taSetColCount;
      property ItemSize : integer read FItemSize;
      property RowCount : integer read FRowCount write taSetRowCount;
  end;

type
  TApxTerminalBuffer = class
    private
      FAttr         : TApxTerminalCharAttrs; { current attributes }
      FBackColor    : TColor;   { current background color }
      FBeyondMargin : Boolean;  { True if cursor's beyond right margin }
      FCharSet      : byte;     { current charset }
      FColCount     : integer;  { count of columns in both views }
      FCursorCol    : integer;  { current internal cursor col position }
      FCursorMoved  : Boolean;  { True if cursor has moved }
      FCursorRow    : integer;  { current internal cursor row position }
      FDefAnsiChar  : AnsiChar; { default ANSI character }
      FDefAttr      : TApxTerminalCharAttrs; { default attributes }
      FDefCharSet   : byte;     { default charset }
      FDefBackColor : TColor;   { default background color }
      FDefForeColor : TColor;   { default foreground color }
      FDefWideChar  : WideChar; { default wide character }
      FDisplayOriginCol : integer;   { column origin of addressable area }
      FDisplayOriginRow : integer;   { row origin of addressable area }
      FDisplayColCount  : integer;   { column count in addressable area }
      FDisplayRowCount  : integer;   { row count in addressable area }
      FForeColor    : TColor;        { current foreground color }
      FHorzTabStops : PByteArray;    { bitset of horizontal tab stops }
      FInvRectList  : pointer;       { list of invalid rects }
      FOnScrollRows : TApxScrollRowsNotifyEvent;
      FRowCount     : integer;       { count of rows in display view }
      FSRStartRow   : integer;       { start row of scrolling region }
      FSREndRow     : integer;       { end row of scrolling region }
      FSVRowCount   : integer;       { count of rows in scrollback view }
      FUseAbsAddress: Boolean;       { True if absolute values for row/col }
      FUseAutoWrap  : Boolean;       { True if chars wrap to next line }
      FUseAutoWrapDelay: Boolean;    { True if cursor stays at last col }
      FUseInsertMode   : Boolean;    { True if insert rather than replace }
      FUseNewLineMode  : Boolean;    { True if LF means CR+LF }
      FUseScrollRegion : Boolean;    { True if limit to scroll region }
      FUseWideChars : Boolean;       { True if expecting UNICODE chars }
      FVertTabStops : PByteArray;    { bitset of vertical tab stops }

      FCharMatrix   : TApxTerminalArray;       { matrix of chars }
      FCharSetMatrix: TApxTerminalArray;       { matrix of charsets }
      FAttrMatrix   : TApxTerminalArray;       { matrix of attrs }
      FForeColorMatrix : TApxTerminalArray;    { matrix of forecolors }
      FBackColorMatrix : TApxTerminalArray;    { matrix of backcolors }

      FOnCursorMoved : TApxOnCursorMovedEvent; { Cursor moved event}

    protected
      function tbGetCol : integer;
      function tbGetOriginCol : integer;
      function tbGetOriginRow : integer;
      function tbGetRow : integer;

      procedure tbSetBackColor(aValue : TColor);
      procedure tbSetCharSet(aValue : byte);
      procedure tbSetDefAnsiChar(aValue : AnsiChar);
      procedure tbSetDefBackColor(aValue : TColor);
      procedure tbSetDefForeColor(aValue : TColor);
      procedure tbSetForeColor(aValue : TColor);
      procedure tbSetSVRowCount(aNewCount : integer);
      procedure tbSetCol(aCol : integer);
      procedure tbSetColCount(aNewCount : integer);
      procedure tbSetRow(aRow : integer);
      procedure tbSetRowCount(aNewCount : integer);
      procedure tbSetUseScrollRegion(aValue : Boolean);

      procedure tbInvalidateRect(aFromRow, aFromCol, aToRow, aToCol : integer);
      function tbCvtToInternalCol(aCol : integer; aAbsolute : Boolean) : integer;
      function tbCvtToInternalRow(aRow : integer; aAbsolute : Boolean) : integer;
      function tbCvtToExternalCol(aCol : integer; aAbsolute : Boolean) : integer;
      function tbCvtToExternalRow(aRow : integer; aAbsolute : Boolean) : integer;

      function tbAtLastColumn : Boolean;
      procedure tbMoveCursorLeftRight(aDirection : integer;
        aWrap : Boolean; aScroll : Boolean);
      procedure tbMoveCursorUpDown(aDirection : integer; aScroll : Boolean);
      procedure tbReallocBuffers(aNewRowCount : integer; aNewColCount : integer);
      procedure tbScrollRows(aCount, aTop, aBottom : integer);
      procedure tbFireOnCursorMovedEvent;

    public
      constructor Create(aUseWideChars : Boolean);
      destructor Destroy; override;

      {---METHODS---}
      { character attributes }
      procedure GetCharAttrs(var aValue : TApxTerminalCharAttrs);
      procedure GetDefCharAttrs(var aValue : TApxTerminalCharAttrs);
      procedure SetDefCharAttrs(const aValue : TApxTerminalCharAttrs);
      procedure SetCharAttrs(const aValue : TApxTerminalCharAttrs);

      { cursor movement }
      procedure MoveCursorDown(aScroll : Boolean);
      procedure MoveCursorLeft(aWrap : Boolean; aScroll : Boolean);
      procedure MoveCursorRight(aWrap : Boolean; aScroll : Boolean);
      procedure MoveCursorUp(aScroll : Boolean);
      procedure SetCursorPosition(aRow, aCol : integer);

      { insertion/deletion }
      procedure DeleteChars(aCount : integer);
      procedure DeleteLines(aCount : integer);
      procedure InsertChars(aCount : integer);
      procedure InsertLines(aCount : integer);

      { erasing }
      procedure EraseAll;
      procedure EraseChars(aCount : integer);
      procedure EraseFromBOW;
      procedure EraseFromBOL;
      procedure EraseLine;
      procedure EraseScreen;
      procedure EraseToEOL;
      procedure EraseToEOW;

      { horizontal tab stop control }
      procedure SetHorzTabStop;
      procedure ClearHorzTabStop;
      procedure ClearAllHorzTabStops;
      procedure DoHorzTab;
      procedure DoBackHorzTab;

      { vertical tab stop control }
      procedure SetVertTabStop;
      procedure ClearVertTabStop;
      procedure ClearAllVertTabStops;
      procedure DoVertTab;
      procedure DoBackVertTab;

      { scrolling regions }
      procedure SetScrollRegion(aTopRow, aBottomRow : integer);

      { write character/string }
      procedure WriteChar(aCh : char);
      procedure WriteString(const aSt : string);

      { miscellaneous special processing }
      procedure DoBackspace;
      procedure DoCarriageReturn;
      procedure DoLineFeed;
      procedure Reset;

      { get buffer information }
      function GetLineCharPtr(aRow : integer) : pointer;
      function GetLineAttrPtr(aRow : integer) : pointer;
      function GetLineForeColorPtr(aRow : integer) : pointer;
      function GetLineBackColorPtr(aRow : integer) : pointer;
      function GetLineCharSetPtr(aRow : integer) : pointer;

      { getting information about changes }
      function HasCursorMoved : Boolean;
      function HasDisplayChanged : Boolean;
      function GetInvalidRect(var aRect : TRect) : Boolean;

      {---PROPERTIES---}
      { color, charsets }
      property BackColor : TColor read FBackColor write tbSetBackColor;
      property CharSet : byte read FCharSet write tbSetCharSet;
      property DefAnsiChar : AnsiChar read FDefAnsiChar write tbSetDefAnsiChar;
      property DefBackColor : TColor read FDefBackColor write tbSetDefBackColor;

      property DefCharSet : byte read FDefCharSet write FDefCharSet;
      property DefForeColor : TColor read FDefForeColor write tbSetDefForeColor;

      property ForeColor : TColor read FForeColor write tbSetForeColor;

      { scrollback view extent }
      property SVRowCount : integer read FSVRowCount write tbSetSVRowCount;

      { display view properties }
      property Col : integer read tbGetCol write tbSetCol;
      property ColCount : integer read FColCount write tbSetColCount;
      property OriginCol : integer read tbGetOriginCol;
      property OriginRow : integer read tbGetOriginRow;
      property Row : integer read tbGetRow write tbSetRow;
      property RowCount : integer read FRowCount write tbSetRowCount;

      property UseAbsAddress : Boolean
        read FUseAbsAddress write FUseAbsAddress;
      property UseAutoWrap : Boolean
        read FUseAutoWrap write FUseAutoWrap;
      property UseAutoWrapDelay : Boolean
        read FUseAutoWrapDelay write FUseAutoWrapDelay;
      property UseInsertMode : Boolean
        read FUseInsertMode write FUseInsertMode;
      property UseNewLineMode : Boolean
        read FUseNewLineMode write FUseNewLineMode;
      property UseScrollRegion : Boolean
        read FUseScrollRegion write tbSetUseScrollRegion;
      property UseWideChars : Boolean read FUseWideChars;

      property OnScrollRows : TApxScrollRowsNotifyEvent
        read FOnScrollRows write FOnScrollRows;

      { OnCursorMoved }
      { This property is used to notify the TApxTerminal component when }
      { the cursor moves.  It should not be used for your own purposes  }
      { as that may cause unexpected behaviour in the terminal }

      property OnCursorMoved : TApxOnCursorMovedEvent
        read FOnCursorMoved write FOnCursorMoved;
  end;

type
  PadKeyString = ^TApxKeyString;
  TApxKeyString = string[63];

const
  DefaultFontName : string[9] = '<Default>';


type
  TApxKeyboardMapping = class
    private
      FTable : TList;
      FCount : integer;
    protected
      function kbmFindPrim(const aKey : TApxKeyString;
        var aInx : integer; var aNode : pointer) : Boolean;
    public
      constructor Create;
      destructor Destroy; override;

      function Add(const aKey : TApxKeyString;
        const aValue : TApxKeyString) : Boolean;
      procedure Clear;
      function Get(const aKey : TApxKeyString) : TApxKeyString;

      procedure LoadFromFile(const aFileName : string);
      procedure LoadFromRes(aInstance : THandle;
        const aResName  : string);
      procedure StoreToBinFile(const aFileName : string);

      {$IFDEF CompileDebugCode}
      procedure DebugPrint(const aFileName : string);
      {$ENDIF}

      property Count : integer read FCount;
  end;

type
  TApxCharSetMapping = class
  private
    FTable     : TList;
    FCharQueue : pointer;
    FCount     : integer;
    FScript    : pointer;
    FScriptEnd : pointer;
    FScriptFreeList : pointer;
  protected
    procedure csmAddScriptNode(aFont : PadKeyString);
    function csmFindPrim(const aCharSet : TApxKeyString;
      aChar : AnsiChar; var aInx : integer; var aNode : pointer) : Boolean;
    procedure csmFreeScript;
  public
    constructor Create;
    destructor Destroy; override;
    function Add(const aCharSet : TApxKeyString; aFromCh : AnsiChar;
      aToCh : AnsiChar; const aFont : TApxKeyString; aGlyph : AnsiChar) : Boolean;
    procedure Clear;
    procedure GetFontNames(aList : TStrings);
    procedure GenerateDrawScript(const aCharSet : TApxKeyString;
      aText : PAnsiChar);
    function GetNextDrawCommand(var aFont : TApxKeyString;
      aText : PAnsiChar) : Boolean;
    procedure LoadFromFile(const aFileName : string);
    procedure LoadFromRes(aInstance : THandle; const aResName : string);
    procedure StoreToBinFile(const aFileName : string);

    {$IFDEF CompileDebugCode}
    procedure DebugPrint(const aFileName : string);
    {$ENDIF}

    property Count : integer read FCount;
  end;

type
  TApxParserCmdType = ( { Parser return command types... }
    pctNone,            {..no command, unknown command or char ignored}
    pctChar,            {..single displayable character}
    pct8bitChar,        {..single character with bit 7 set}
    pctPending,         {..command being built up}
    pctComplete);       {..a complete command is ready}

  TApxVTParserState = (  { VT Parser states... }
    psIdle,              {..nothing happening}
    psGotEscape,         {..received escape char}
    psParsingANSI,       {..parsing an ESC[ sequence}
    psParsingHash,       {..parsing an ESC# sequence}
    psParsingLeftParen,  {..parsing an ESC( sequence}
    psParsingRightParen, {..parsing an ESC) sequence}
    psParsingCharSet,    {..parsing ESC *, +, -, ., / sequence}
    psGotCommand,        {..received complete command}
    psGotInterCommand,   {..received complete intermediary command}
    psParsingCUP52);     {..received VT52 position cursor command}

type
  PAdIntegerArray = ^TApxIntegerArray;
  TApxIntegerArray = array [0..pred(MaxInt div sizeof(integer))] of integer;

type
  TApxTerminalParser = class
  { the ancestor parser class }
  private
    FArgCount    : integer;
    FCommand     : byte;
    FUseWideChar : Boolean;
  protected
    function tpGetArgument(aInx : integer) : integer; virtual;
    function tpGetSequence : string; virtual;
  public
    constructor Create(aUseWideChar : Boolean);
    destructor Destroy; override;

    function ProcessChar(aCh : AnsiChar) : TApxParserCmdType; virtual;
    function ProcessWideChar(aCh : WideChar) :TApxParserCmdType; virtual;

    procedure Clear; virtual;

    property Argument [aInx : integer] : integer
       read tpGetArgument;
    property ArgumentCount : integer read FArgCount;
    property Command : byte read FCommand;
    property Sequence : string read tpGetSequence;
  end;

  TApxVT100Parser = class(TApxTerminalParser)
  {the VT100 terminal parser}
  private
    FArgCountMax : integer;
    FArgs        : PAdIntegerArray;
    FInVT52Mode  : Boolean;
    FSavedSeq    : pointer;
    FSavedState  : TApxVTParserState;
    FSequence    : pointer;
    FState       : TApxVTParserState;
  protected
    function tpGetArgument(aInx : integer) : integer; override;
    function tpGetSequence : string; override;
    function vtpGetArguments : Boolean;
    function vtpParseANSISeq(aCh : char) : TApxParserCmdType;
    function vtpProcessVT52(aCh : char) : TApxParserCmdType;
    function vtpValidateArgsPrim(aMinArgs : integer;
      aMaxArgs : integer; aDefault : integer) : Boolean;
    procedure vtpGrowArgs;
  public
    constructor Create(aUseWideChar : Boolean);
    destructor Destroy; override;
    function ProcessChar(aCh : AnsiChar) : TApxParserCmdType; override;
    function ProcessWideChar(aCh : WideChar) :TApxParserCmdType; override;
    procedure Clear; override;
    property InVT52Mode : Boolean read FInVT52Mode;
  end;

type
  TApxTerminalEmulator = class;

  TApxCustomTerminal = class;

  TApxTerminalScrollBar = class (TScrollBar)
  private
    FTerminal : TApxCustomTerminal;
    FInternalVisible : Boolean;
    FUpdating : Boolean;
    function GetVisible : Boolean;
    procedure SetVisible(Value : Boolean);
  protected
    procedure InitWidget; override;
    procedure WidgetDestroyed; override;
  public
    constructor Create(AOwner : TComponent); override;
    procedure BeginUpdate;
    procedure EndUpdate;
    function CanFocus : Boolean; override;
    property Visible : Boolean read GetVisible write SetVisible;
  end;

  TApxCustomTerminal = class(TApxBaseCustomControl)
  private
    FCaretX           : integer;
    FCaretY           : integer;
    FCaretBlinkOn     : Boolean;
    FActive           : Boolean;
    FBlinkTime        : integer;
    FBlinkTimeCount   : integer;
    FBlinkTextVisible : Boolean;
    FBorderStyle      : TControlBorderStyle;
    FByteQueue        : pointer;
    FCapture          : TApxCaptureMode;
    FCaptureFile      : string;
    FCaptureStream    : TFileStream;
    FCharHeight       : integer;           { height of char cell in pixels }
    FCharWidth        : integer;           { width of char cell in pixels }
    FClientCols       : integer;           { client width in columns, incl part }
    FClientFullCols   : integer;           { client width in *full* columns }
    FClientRows       : integer;           { client height in rows, incl part }
    FClientFullRows   : integer;           { client height in *full* rows }
    FComPort          : TApxCustomComPort;
    FCreatedCaret     : Boolean;
    FCursorType       : TApxCursorType;
    FDefEmulator      : TApxTerminalEmulator; { default = tty }
    FEmulator         : TApxTerminalEmulator;
    FHalfDuplex       : Boolean;
    FHaveSelection    : Boolean;
    FHeartbeat        : TTimer;
    FLazyByteCount    : integer;
    FLazyTimeCount    : integer;
    FLazyByteDelay    : integer;
    FLazyTimeDelay    : integer;
    FLButtonAnchor    : TPoint;
    FLButtonDown      : Boolean;
    FLButtonRect      : TRect;
    FOnCursorMoved    : TApxTerminalCursorMovedEvent;
    FOriginCol        : integer;
    FOriginRow        : integer;
    FScrollback       : Boolean;
    FHScrollBar       : TApxTerminalScrollBar;
    FVScrollBar       : TApxTerminalScrollBar;
    FShowingCaret     : Boolean;
    FTriggerHandlerOn : Boolean;
    FUsedRect         : TRect;
    FUseLazyDisplay   : Boolean;
    FUnusedRightRect  : TRect;
    FUnusedBottomRect : TRect;
    FUseHScrollBar    : Boolean;
    FUseVScrollBar    : Boolean;
    FWantAllKeys      : Boolean;
    FWaitingForComPortOpen : Boolean;
  protected
    { get and set methods for properties }
    function tmGetAttributes(aRow, aCol : integer) : TApxTerminalCharAttrs;
    function tmGetBackColor(aRow, aCol : integer) : TColor;
    function tmGetCharSet(aRow, aCol : integer) : byte;
    function tmGetColumns : integer;
    function tmGetEmulator : TApxTerminalEmulator;
    function tmGetForeColor(aRow, aCol : integer) : TColor;
    function tmGetLine(aRow : integer) : string;
    function tmGetRows : integer;
    function tmGetScrollbackRows : integer;

    procedure tmSetActive(aValue : Boolean);
    procedure tmSetAttributes(aRow, aCol : integer;
                      const aAttr      : TApxTerminalCharAttrs);
    procedure tmSetBackColor(aRow, aCol : integer; aValue : TColor);
    procedure tmSetBlinkTime(aValue : integer);
    procedure tmSetBorderStyle(aBS : TControlBorderStyle);
    procedure tmSetCapture(aValue : TApxCaptureMode);
    procedure tmSetCaptureFile(const aValue : string);
    procedure tmSetCharSet(aRow, aCol : integer; aValue : byte);
    procedure tmSetComPort(aValue : TApxCustomComPort);
    procedure tmSetColumns(aValue : integer);
    procedure tmSetCursorType(aValue : TApxCursorType);
    procedure tmSetForeColor(aRow, aCol : integer; aValue : TColor);
    procedure tmSetEmulator(aValue : TApxTerminalEmulator);
    procedure tmSetLazyByteDelay(aValue : integer);
    procedure tmSetLazyTimeDelay(aValue : integer);
    procedure tmSetLine(aRow : integer; const aValue : string);
    procedure tmSetOriginCol(aValue : integer);
    procedure tmSetOriginRow(aValue : integer);
    procedure tmSetRows(aValue : integer);
    procedure tmSetScrollback(aValue : Boolean);
    procedure tmSetScrollbackRows(aValue : integer);
    procedure tmSetUseLazyDisplay(aValue : Boolean);
    procedure tmSetWantAllKeys(aValue : Boolean);

    { various miscellaneous methods }
    procedure tmDrawDefaultText;
    procedure tmAttachToComPort;
    procedure tmCalcExtent;
    procedure tmDetachFromComPort;
    procedure tmGetFontInfo;
    procedure tmInvalidateRow(aRow : integer);

    { caret methods }
    procedure tmHideCaret (Force : Boolean);
    procedure tmMakeCaret;
    procedure tmPositionCaret;
    procedure tmShowCaret;

    { scrollbar stuff }
    procedure tmInitHScrollBar;
    procedure tmInitVScrollBar;
    procedure tmScrollHorz(aDist : integer);
    procedure tmScrollVert(aDist : integer);
    procedure tmUpdateScrollbars;

    { selection stuff }
    procedure tmGrowSelect(X, Y : integer);
    procedure tmMarkDeselected(aRow : integer; aFromCol, aToCol : integer);
    procedure tmMarkSelected(aRow : integer; aFromCol, aToCol : integer);
    function tmProcessClipboardCopy(Key : Word; Shift : TShiftState) : Boolean;
    procedure tmStartSelect(X, Y : integer);

    { overridden ancestor methods }
    procedure KeyDown(var Key : word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure MouseDown(Button : TMouseButton; Shift : TShiftState;
      X, Y : integer); override;
    procedure MouseMove(Shift : TShiftState; X, Y : integer); override;
    procedure MouseUp(Button : TMouseButton; Shift : TShiftState;
      X, Y : integer); override;
    procedure Notification(AComponent : TComponent;
      Operation  : TOperation); override;

    { message and event handlers }
    procedure ApxPortOpen(var Msg : TAxMessage); message APX_PORTOPEN;
    procedure ApxPortClose(var Msg : TAxMessage); message APX_PORTCLOSE;
    procedure ApxTermStuff(var Msg : TMessage); message APX_TERMSTUFF;

    function EventFilter(Sender: QObjectH; Event: QEventH): Boolean; override;

    procedure FontChanged; override;
    function ApxKeyDown (var KeyCode : Word; var KeyChar : Char;
      var ShiftState : TShiftState) : Boolean;
    procedure tmBeat(Sender: TObject);
    procedure tmTriggerHandler(Msg, wParam : Cardinal; lParam : longint);
    procedure CancelMode;
    procedure HScroll (Sender : TObject;  ScrollCode: TScrollCode; var ScrollPos: Integer);
    procedure VScroll (Sender : TObject;  ScrollCode: TScrollCode; var ScrollPos: Integer);
    procedure BoundsChanged; override;
    procedure FocusIn;
    procedure FocusOut;
    procedure Resize; override;

    { TCustomControl emulation }
    procedure Paint; override;
    function GetClientRect: TRect; override;
    function WidgetFlags: Integer; override;

  public
    constructor Create(aOwner : TComponent); override;
    destructor Destroy; override;

    { overridden ancestor methods }
    procedure CreateWidget; override;
    procedure DestroyWidget; override;

    { terminal methods }
    procedure Clear;
    procedure ClearAll;
    procedure CopyToClipboard;
    procedure HideSelection;
    procedure WriteChar(aCh : AnsiChar);
    procedure WriteString(const aSt : string);

    { public properties }
    property Attributes [aRow, aCol : integer] : TApxTerminalCharAttrs
      read tmGetAttributes write tmSetAttributes;
    property BackColor [aRow, aCol : integer] : TColor
      read tmGetBackColor write tmSetBackColor;
    property CharHeight : integer read FCharHeight;
    property CharSet [aRow, aCol : integer] : byte
      read tmGetCharSet write tmSetCharSet;
    property CharWidth : integer read FCharWidth;
    property ClientCols : integer read FClientCols;
    property ClientRows : integer read FClientRows;
    property ClientOriginCol : integer read FOriginCol write tmSetOriginCol;
    property ClientOriginRow : integer
      read FOriginRow write tmSetOriginRow;
    property ForeColor [aRow, aCol : integer] : TColor
      read tmGetForeColor write tmSetForeColor;
    property HaveSelection : Boolean read FHaveSelection;
    property Line [aRow : integer] : string read tmGetLine write tmSetLine;

    property Emulator : TApxTerminalEmulator
      read tmGetEmulator write tmSetEmulator;

  published
    property Active : Boolean
      read FActive write tmSetActive default adc_TermActive;
    property BlinkTime : integer
      read FBlinkTime write tmSetBlinkTime default adc_TermBlinkTime;
    property BorderStyle : TControlBorderStyle
      read FBorderStyle write tmSetBorderStyle default adc_TermBorderStyle;
    property Capture : TApxCaptureMode
      read FCapture write tmSetCapture default adc_TermCapture;
    property CaptureFile : string
      read FCaptureFile write tmSetCaptureFile;
    property Columns : integer
      read tmGetColumns write tmSetColumns default adc_TermColumnCount;
    property ComPort : TApxCustomComPort
      read FComPort write tmSetComPort;
    property CursorType : TApxCursorType
      read FCursorType write tmSetCursorType default adc_TermCursorType;
    property HalfDuplex : Boolean
      read FHalfDuplex write FHalfDuplex default adc_TermHalfDuplex;
    property LazyByteDelay : integer
      read FLazyByteDelay write tmSetLazyByteDelay default adc_TermLazyByteDelay;
    property LazyTimeDelay : integer
      read FLazyTimeDelay write tmSetLazyTimeDelay default adc_TermLazyTimeDelay;
    property Rows : integer
      read tmGetRows write tmSetRows default adc_TermRowCount;
    property ScrollbackRows : integer
      read tmGetScrollbackRows write tmSetScrollbackRows default adc_TermScrollRowCount;
    property Scrollback : Boolean
      read FScrollback write tmSetScrollback;
    property UseLazyDisplay : Boolean
      read FUseLazyDisplay write tmSetUseLazyDisplay default adc_TermUseLazyDisplay;
    property WantAllKeys : Boolean
      read FWantAllKeys write tmSetWantAllKeys default adc_TermWantAllKeys;
    property OnCursorMoved : TApxTerminalCursorMovedEvent
      read FOnCursorMoved write FOnCursorMoved;

    { publish ancestor's properties }
    property Align;
    property Color;
    property Enabled;
    property Font;
    property Height;
    property ParentColor;
    property ParentFont;
    property TabOrder;
    property TabStop;
    property Visible;
    property Width;

    { publish ancestor's events }
    property OnClick;
    property OnDblClick;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

  TApxTerminalEmulator = class(TApxBaseComponent)
  private
    FTerminalBuffer  : TApxTerminalBuffer;
    FCharSetMapping  : TApxCharSetMapping;
    FIsDefault       : Boolean;
    FKeyboardMapping : TApxKeyboardMapping;
    FParser          : TApxTerminalParser;
    FTerminal        : TApxCustomTerminal;

    { saved cursor details }
    FSavedAttrs : TApxTerminalCharAttrs;
    FSavedBackColor : TColor;
    FSavedCharSet : byte;
    FSavedCol : integer;
    FSavedForeColor : TColor;
    FSavedRow : integer;

  protected
    { property access methods }
    function teGetNeedsUpdate : Boolean; virtual;
    procedure teSetTerminal(aValue : TApxCustomTerminal); virtual;

    { overridden ancestor methods }
    procedure Notification(AComponent : TComponent;
      Operation  : TOperation); override;

    { new virtual methods }
    procedure teClear; virtual;
    procedure teClearAll; virtual;
    procedure teSendChar(aCh : char; aCanEcho : Boolean); virtual;

    { Cursor Movement Methods }
    procedure teHandleCursorMovement(Sender : TObject;
      Row : Integer; Col : Integer);

  public
    constructor Create(aOwner : TComponent); override;
    destructor Destroy; override;

    procedure BlinkPaint(aVisible : Boolean); virtual;
    function HasBlinkingText : Boolean; virtual;
    procedure KeyDown(var Key : word; Shift: TShiftState); virtual;
    procedure KeyPress(var Key : AnsiChar); virtual;
    procedure LazyPaint; virtual;
    procedure Paint; virtual;
    procedure ProcessBlock(aData : pointer; aDataLen : longint); virtual;

    { read-only properties for low-level objects }
    property Buffer : TApxTerminalBuffer read FTerminalBuffer;
    property CharSetMapping : TApxCharSetMapping read FCharSetMapping;
    property KeyboardMapping : TApxKeyboardMapping read FKeyboardMapping;
    property Parser : TApxTerminalParser read FParser;
    property NeedsUpdate : Boolean read teGetNeedsUpdate;
  published
    property Terminal : TApxCustomTerminal read FTerminal write teSetTerminal;
  end;

  TApxTerminal = class(TApxCustomTerminal)
  published
    property Emulator;
  end;

type

  TApxTTYEmulator = class(TApxTerminalEmulator)
  private
    FCellWidths    : PAdIntegerArray;
    FDisplayStr    : PAnsiChar;
    FDisplayStrSize: integer;
    FPaintFreeList : pointer;
    FRefresh       : Boolean;
  protected
    { property accessor methods }
    function teGetNeedsUpdate : Boolean; override;

    { overridden ancestor methods }
    procedure teClear; override;
    procedure teClearAll; override;
    procedure teSetTerminal(aValue : TApxCustomTerminal); override;

    { miscellaneous }
    procedure ttyDrawChars(aRow, aStartCol, aEndCol : integer;
                           aVisible : Boolean);
    procedure ttyProcessCommand(aCh : AnsiChar);

    { paint node methods }
    procedure ttyExecutePaintScript(aRow    : integer;
                                    aScript : pointer);
    procedure ttyFreeAllPaintNodes;
    procedure ttyFreePaintNode(aNode : pointer);
    function ttyNewPaintNode : pointer;

  public
    constructor Create(aOwner : TComponent); override;
    destructor Destroy; override;

    { overridden ancestor methods }
    procedure KeyPress(var Key : AnsiChar); override;
    procedure LazyPaint; override;
    procedure Paint; override;
    procedure ProcessBlock(aData : pointer; aDataLen : longint); override;
  end;

type
  TApxVT100Emulator = class(TApxTerminalEmulator)
  private
    FAnswerback     : string;
    FBlinkers       : pointer;
    FBlinkFreeList  : pointer;
    FCellWidths     : PAdIntegerArray;
    FDisplayStr     : PAnsiChar;
    FDisplayStrSize : integer;
    FDispUpperASCII : Boolean;
    FLEDs           : integer;
    FLineAttrArray  : TObject;
    FPaintFreeList  : pointer;
    FRefresh        : Boolean;
    FSecondaryFont  : TFont;

    { modes }
    FANSIMode         : Boolean;
    FAppKeyMode       : Boolean;
    FAppKeypadMode    : Boolean;
    FAutoRepeat       : Boolean;
    FCol132Mode       : Boolean;
    FGPOMode          : Boolean; { graphics proc.option not supported }
    FInsertMode       : Boolean;
    FInterlace        : Boolean; { interlace chgs are not supported }
    FNewLineMode      : Boolean;
    FRelOriginMode    : Boolean;
    FRevScreenMode    : Boolean;
    FSmoothScrollMode : Boolean; { smooth scrolling is not supported }
    FWrapAround       : Boolean;

    { character sets }
    FUsingG1   : Boolean;
    FG0CharSet : integer;
    FG1CharSet : integer;

  protected
    { property accessor methods }
    function teGetNeedsUpdate : Boolean; override;
    procedure vttSetCol132Mode(aValue : Boolean);
    procedure vttSetRelOriginMode(aValue : Boolean);
    procedure vttSetRevScreenMode(aValue : Boolean);

    { overridden ancestor methods }
    procedure teClear; override;
    procedure teClearAll; override;
    procedure teSetTerminal(aValue : TApxCustomTerminal); override;

    { miscellaneous }
    procedure vttDrawChars(aRow, aStartVal, aEndVal : integer;
      aVisible : Boolean; aCharValues : Boolean);
    procedure vttProcessCommand;
    function vttGenerateDECREPTPARM(aArg : integer) : string;
    procedure vttInvalidateRow(aRow : integer);
    procedure vttProcess8bitChar(aCh : AnsiChar);
    procedure vttScrollRowsHandler(aSender : TObject;
      aCount, aTop, aBottom : integer);
    procedure vttToggleNumLock;

    { blink node methods }
    procedure vttCalcBlinkScript;
    procedure vttClearBlinkScript;
    procedure vttDrawBlinkOffCycle(aRow, aStartCh, aEndCh : integer);
    procedure vttFreeAllBlinkNodes;
    procedure vttFreeBlinkNode(aNode : pointer);
    function vttNewBlinkNode : pointer;

    { paint node methods }
    procedure vttExecutePaintScript(aRow : integer; aScript : pointer);
    procedure vttFreeAllPaintNodes;
    procedure vttFreePaintNode(aNode : pointer);
    function vttNewPaintNode : pointer;

  public
    constructor Create(aOwner : TComponent); override;
    destructor Destroy; override;

    { overridden ancestor methods }
    procedure BlinkPaint(aVisible : Boolean); override;
    function HasBlinkingText : Boolean; override;
    procedure KeyDown(var Key : word; Shift: TShiftState); override;
    procedure KeyPress(var Key : AnsiChar); override;
    procedure LazyPaint; override;
    procedure Paint; override;
    procedure ProcessBlock(aData : pointer; aDataLen : longint); override;

    { modes }
    property ANSIMode : Boolean read FANSIMode write FANSIMode;
    property AppKeyMode : Boolean read FAppKeyMode write FAppKeyMode;
    property AppKeypadMode : Boolean read FAppKeypadMode write FAppKeypadMode;
    property AutoRepeat : Boolean read FAutoRepeat write FAutoRepeat;
    property Col132Mode : Boolean read FCol132Mode write vttSetCol132Mode;
    property GPOMode : Boolean read FGPOMode write FGPOMode;
    property InsertMode : Boolean read FInsertMode write FInsertMode;
    property Interlace : Boolean read FInterlace write FInterlace;
    property NewLineMode : Boolean read FNewLineMode write FNewLineMode;
    property RelOriginMode : Boolean read FRelOriginMode write vttSetRelOriginMode;
    property RevScreenMode : Boolean read FRevScreenMode write vttSetRevScreenMode;
    property SmoothScrollMode : Boolean read FSmoothScrollMode write FSmoothScrollMode;
    property WrapAround : Boolean read FWrapAround write FWrapAround;

    { miscellaneous }
    property LEDs : integer read FLEDs write FLEDs;

  published
    property Answerback : string read FAnswerback write FAnswerback;
    property DisplayUpperASCII : Boolean read FDispUpperASCII write FDispUpperASCII;
  end;

implementation

const
  BeatInterval = 100;

type
  PBlinkNode = ^TBlinkNode;
  TBlinkNode = packed record
    bnNext    : PBlinkNode;
    bnRow     : integer;
    bnStartCh : integer;
    bnEndCh   : integer;
  end;

type
  PPaintNode = ^TPaintNode;
  TPaintNode = packed record
    pnNext  : PPaintNode;
    pnStart : integer;               { start column of range }
    pnEnd   : integer;               { end column of range }
    pnFore  : TColor;                { foreground color for range }
    pnBack  : TColor;                { background color for range }
    pnAttr  : TApxTerminalCharAttrs; { attributes for range }
    pnCSet  : byte;                  { charset for range }
  end;

const
  VT52DeviceAttrs  = #27'/Z';        { "VT100 acting as VT52" }
  VT100DeviceAttrs = #27'[?1;0c';    { "Base VT100, no options" }
  VT100StatusRpt   = #27'[0n';       { "terminal OK" }
  VT100CursorPos   = #27'[%d;%dR';   { "cursor is at row;col" }
  VT100ReportParm  = #27'[%d;%d;%d;%d;%d;%d;%dx';
                                     { report terminal parameters }

const
  VT100CharSetNames : array [0..4] of string =
                      ('VT100-USASCII',           { charset 0 }
                       'VT100-UK',                { charset 1 }
                       'VT100-linedraw',          { charset 2 }
                       'VT100-ROM1',              { charset 3 }
                       'VT100-ROM2');             { charset 4 }

{===Terminal/Emulator links==========================================}
type
  PTermEmuLink = ^TTermEmuLink;
  TTermEmuLink = record
    telNext : PTermEmuLink;
    telTerm : TApxCustomTerminal;
    telEmu  : TApxTerminalEmulator;
  end;
{--------}
var
  TermEmuLink : PTermEmuLink;
  TermEmuLinkFreeList : PTermEmuLink;
{--------}

{--------}
procedure AddTermEmuLink(aTerminal : TApxCustomTerminal;
                         aEmulator : TApxTerminalEmulator);
var
  Node : PTermEmuLink;
begin
  { if the link already exists, exit }
  Node := TermEmuLink;
  while (Node <> nil) do begin
    if (Node^.telTerm = aTerminal) then
      Exit;
    Node := Node^.telNext;
  end;
  { otherwise, add it }
  if (TermEmuLinkFreeList = nil) then
    New(Node)
  else begin
    Node := TermEmuLinkFreeList;
    TermEmuLinkFreeList := Node^.telNext;
  end;
  Node^.telTerm := aTerminal;
  Node^.telEmu := aEmulator;
  Node^.telNext := TermEmuLink;
  TermEmuLink := Node;
  { now update each object to point to the other }
  aTerminal.Emulator := aEmulator;
  aEmulator.Terminal := aTerminal;
end;
{--------}
procedure RemoveTermEmuLink(aTerminal : TApxCustomTerminal;
                            aNotify   : Boolean);
var
  Dad, Node : PTermEmuLink;
  Emulator  : TApxTerminalEmulator;
begin
  { remove the link }
  Emulator := nil;
  Dad := nil;
  Node := TermEmuLink;
  while (Node <> nil) do begin
    if (Node^.telTerm = aTerminal) then begin
      if (Dad = nil) then
        TermEmuLink := Node^.telNext
      else
        Dad^.telNext := Node^.telNext;
      Emulator := Node^.telEmu;
      Node^.telNext := TermEmuLinkFreeList;
      TermEmuLinkFreeList := Node;
      Break;
    end;
    Dad := Node;
    Node := Node^.telNext;
  end;
  { now update each object to point to nil instead of each other }
  if aNotify and (Emulator <> nil) then begin
    aTerminal.Emulator := nil;
    Emulator.Terminal := nil;
  end;
end;
{====================================================================}

function GetRValue (Colour : Integer) : Integer;
begin
  Result := Colour and $ff;
end;

function GetGValue (Colour : Integer) : Integer;
begin
  Result := (Colour shr 8) and $ff;
end;

function GetBValue (Colour : Integer) : Integer;
begin
  Result := (Colour shr 16) and $ff;
end;

function RGB (R, G, B : Integer) : Integer;
begin
  Result := R or (G shl 8) or (B shl 16);
end;


{===Byte queue==========================================================}
{ Notes: this byte queue was published in Algorithms Alfresco in The    }
{        Delphi Magazine, December 1988. It is reproduced and used here }
{        with permission.                                               }
{        Copyright (c) Julian M Bucknall, 1998. All Rights Reserved.    }
type
  TaaByteQueue = class
    private
      bqHead      : PChar;
      bqTail      : PChar;
      bqQMidPoint : PChar;
      bqQueue     : PChar;
      bqQueueEnd  : PChar;
    protected
      function getCapacity : integer;
      function getCount : integer;
      procedure setCapacity(aValue : integer);
    public
      constructor Create;
      destructor Destroy; override;

      procedure AdvanceAfterPut(aDataLen : integer);
      procedure Clear;
      (***** not used yet
      procedure Get(var aData; aDataLen : integer);
      function IsEmpty : Boolean;
      *****)
      procedure Put(const aData; aDataLen : integer);
      function PutPeek(aDataLen : integer) : pointer;
      function Peek(aDataLen : integer) : pointer;
      procedure Remove(aDataLen : integer);                           

      property Capacity : integer
         read getCapacity write setCapacity;
      property Count : integer
         read getCount;
  end;
{--------}
procedure NotEnoughDataError(aAvail, aReq : integer);
begin
  raise EApxTerminal.CreateFormatted (AxecTrmEmuNotEnoughData,
                                     [aAvail, aReq], False);
end;
{--------}
constructor TaaByteQueue.Create;
begin
  inherited Create;
  Capacity := 64;
end;
{--------}
destructor TaaByteQueue.Destroy;
begin
  if Assigned(bqQueue) then
    FreeMem(bqQueue, bqQueueEnd - bqQueue);
  inherited Destroy;
end;
{--------}
procedure TaaByteQueue.AdvanceAfterPut(aDataLen : integer);
begin
  inc(bqTail, aDataLen);
end;
{--------}
procedure TaaByteQueue.Clear;
begin
  bqHead := bqQueue;
  bqTail := bqQueue;
end;
{--------}
(***** not used yet
procedure TaaByteQueue.Get(var aData; aDataLen : integer);
var
  ByteCount : integer;
begin
  {check for enough data}
  if (aDataLen > Count) then
    NotEnoughDataError(Count, aDataLen);
  {move the data}
  Move(bqHead^, aData, aDataLen);
  inc(bqHead, aDataLen);
  {if we've emptied the queue, move the head/tail pointers back}
  ByteCount := Count;
  if (ByteCount = 0) then begin
    bqHead := bqQueue;
    bqTail := bqQueue;
  end
  {if the head of the queue has moved into the overflow zone, move the
   data back, and reset the head/tail pointers}
  else if (bqHead >= bqQMidPoint) then begin
    Move(bqHead^, bqQueue^, ByteCount);
    bqHead := bqQueue;
    bqTail := bqHead + ByteCount;
  end;
end;
*****)
{--------}
function TaaByteQueue.getCapacity : integer;
begin
  Result := (bqQueueEnd - bqQueue) div 2;
end;
{--------}
function TaaByteQueue.getCount : integer;
begin
  Result := bqTail - bqHead;
end;
{--------}
(***** not used yet
function TaaByteQueue.IsEmpty : Boolean;
begin
  Result := bqHead = bqTail;
end;
*****)
{--------}
procedure TaaByteQueue.Put(const aData; aDataLen : integer);
var
  ByteCount : integer;
begin
  { if required, grow the queue by at least doubling its size }
  ByteCount := Count;
  while (ByteCount + aDataLen > Capacity) do
    Capacity := Capacity * 2;
  { we now have enough room, so add the new data }
  Move(aData, bqTail^, aDataLen);
  inc(bqTail, aDataLen);
end;
{--------}
function TaaByteQueue.PutPeek(aDataLen : integer) : pointer;
var
  ByteCount : integer;
begin
  { if required, grow the queue by at least doubling its size }
  ByteCount := Count;
  while (ByteCount + aDataLen > Capacity) do
    Capacity := Capacity * 2;
  { just return the tail pointer }
  Result := bqTail;
end;
{--------}
function TaaByteQueue.Peek(aDataLen : integer) : pointer;
begin
  { check for enough data }
  if (aDataLen > Count) then
    NotEnoughDataError(Count, aDataLen);
  { just return the head pointer }
  Result := bqHead;
end;
{--------}
procedure TaaByteQueue.Remove(aDataLen : integer);
begin
  { check for enough data }
  if (aDataLen > Count) then
    NotEnoughDataError(Count, aDataLen);
  { move the remaining data to the head }
  Move((bqHead+aDataLen)^, bqHead^, Count - aDataLen);
  bqTail := bqTail - aDataLen;
end;
{--------}
procedure TaaByteQueue.setCapacity(aValue : integer);
var
  ByteCount : integer;
  NewQueue  : PChar;
begin
  { don't allow data to be lost }
  ByteCount := Count;
  if (aValue < ByteCount) then
    aValue := ByteCount;
  { round the requested capacity to nearest 64 bytes }
  aValue := (aValue + 63) and $7FFFFFC0;
  { get a new buffer }
  GetMem(NewQueue, aValue * 2);
  { if we have data to transfer from the old buffer, do so }
  if (ByteCount <> 0) then
    Move(bqHead^, NewQueue^, ByteCount);
  { destroy the old buffer }
  if (bqQueue <> nil) then
    FreeMem(bqQueue, bqQueueEnd - bqQueue);
  { set the head/tail and other pointers }
  bqQueue := NewQueue;
  bqHead := NewQueue;
  bqTail := NewQueue + ByteCount;
  bqQueueEnd := NewQueue + (aValue * 2);
  bqQMidPoint := NewQueue + aValue;
end;
{====================================================================}

{===VT100 line attributes array======================================}
type
  TApxLineAttrArray = class
    private
      FArray          : PAnsiChar;
      FEmulator       : TApxVT100Emulator;
      FTotalLineCount : integer;
    protected
      function laaGetAttr(aInx : integer) : TApxVT100LineAttr;
      procedure laaSetAttr(aInx   : integer;
                           aValue : TApxVT100LineAttr);

      procedure laaResize;
    public
      constructor Create(aEmulator : TApxVT100Emulator);
      destructor Destroy; override;

      procedure Scroll(aCount, aTop, aBottom : integer);

      property Attr [aInx : integer] : TApxVT100LineAttr
                  read laaGetAttr write laaSetAttr;
  end;
{--------}
constructor TApxLineAttrArray.Create(aEmulator : TApxVT100Emulator);
begin
  inherited Create;
  FEmulator := aEmulator;
  FTotalLineCount := aEmulator.Buffer.SVRowCount;
  FArray := AllocMem(FTotalLineCount * sizeof(TApxVT100LineAttr));
end;
{--------}
destructor TApxLineAttrArray.Destroy;
begin
  if (FArray <> nil) then
    FreeMem(FArray, FTotalLineCount * sizeof(TApxVT100LineAttr));
  inherited Destroy;
end;
{--------}
function TApxLineAttrArray.laaGetAttr(aInx : integer) : TApxVT100LineAttr;
begin
  if (FTotalLineCount <> FEmulator.Buffer.SVRowCount) then
    laaResize;
  aInx := aInx + FTotalLineCount - FEmulator.Buffer.RowCount - 1;
  if (aInx < 0) or (aInx >= FTotalLineCount) then
    Result := ltNormal
  else
    Result := TApxVT100LineAttr(FArray[aInx]);
end;
{--------}
procedure TApxLineAttrArray.laaResize;
var
  NewArray      : PAnsiChar;
  NewTotalCount : integer;
begin
  { allocate the new array}
  NewTotalCount := FEmulator.Buffer.SVRowCount;
  NewArray := AllocMem(NewTotalCount * sizeof(TApxVT100LineAttr));
  { copy over the old array from the end, *not* the beginning }
  if (NewTotalCount < FTotalLineCount) then
    Move(FArray[FTotalLineCount - NewTotalCount], NewArray[0],
         NewTotalCount)
  else { FTotalLineCount <= NewTotalCount }
    Move(FArray[0], NewArray[FTotalLineCount - NewTotalCount],
         FTotalLineCount);
  { free the old array, save the new one }
  FreeMem(FArray, FTotalLineCount * sizeof(TApxVT100LineAttr));
  FArray := NewArray;
  FTotalLineCount := NewTotalCount;
end;
{--------}
procedure TApxLineAttrArray.laaSetAttr(aInx   : integer;
                                      aValue : TApxVT100LineAttr);
begin
  if (FTotalLineCount <> FEmulator.Buffer.SVRowCount) then
    laaResize;
  aInx := aInx + FTotalLineCount - FEmulator.Buffer.RowCount - 1;
  if (0 <= aInx) and (aInx < FTotalLineCount) then
    FArray[aInx] := AnsiChar(aValue);
end;
{--------}
procedure TApxLineAttrArray.Scroll(aCount, aTop, aBottom : integer);
var
  i     : integer;
  ToInx : integer;
begin
  aTop := aTop + FTotalLineCount - FEmulator.Buffer.RowCount - 1;
  aBottom := aBottom + FTotalLineCount - FEmulator.Buffer.RowCount - 1;
  if (aCount > 0) then { scroll upwards } begin
    ToInx := aTop;
    for i := (aTop + aCount) to aBottom do begin
      FArray[ToInx] := FArray[i];
      inc(ToInx);
    end;
    for i := ToInx to aBottom do
      FArray[i] := #0;
  end
  else if (aCount < 0) then { scroll downwards } begin
    ToInx := aBottom;
    for i := (aBottom + aCount) downto aTop do begin
      FArray[ToInx] := FArray[i];
      dec(ToInx);
    end;
    for i := ToInx downto aTop do
      FArray[i] := #0;
  end;
end;
{====================================================================}

{===Helper routines==================================================}
function MinI(A, B : integer) : integer;
begin
  if (A < B) then
    Result := A
  else
    Result := B;
end;
{--------}
function MaxI(A, B : integer) : integer;
begin
  if (A > B) then
    Result := A
  else
    Result := B;
end;
{--------}
function BoundI(aLow, X, aHigh : integer) : integer;
begin
  if (aLow < aHigh) then
    if (X <= aLow) then
      Result := aLow
    else if (X >= aHigh) then
      Result := aHigh
    else
      Result := X
  else { aHigh is less than aLow }
    if (X <= aHigh) then
      Result := aHigh
    else if (X >= aLow) then
      Result := aLow
    else
      Result := X;
end;
{====================================================================}

{===TApxTerminalArray=================================================}
constructor TApxTerminalArray.Create(aItemSize : integer);
begin
  inherited Create;
  {save a valid item size}
  case aItemSize of
    1, 2, 4 : FItemSize := aItemSize;
  else
    FItemSize := 1;
  end;{case}
  {set the actual column count to -1, which means 'uncalculated'}
  FActColCount := -1;
end;
{--------}
destructor TApxTerminalArray.Destroy;
begin
  if (FItems <> nil) then begin
    FreeMem(FItems, RowCount * FActColCount * ItemSize);
  end;
  inherited Destroy;
end;
{--------}
procedure TApxTerminalArray.Clear;
begin
  if (FItems <> nil) then
    taClearRows(FItems, FActColCount, 0, pred(RowCount));
end;
{--------}
procedure TApxTerminalArray.ClearItems(aRow : integer;
                                      aFromCol, aToCol : integer);
var
  Walker    : PAnsiChar;
  Value     : longint;
  i         : integer;
begin
  Walker := @FItems[((aRow * FActColCount) + aFromCol) * ItemSize];
  case ItemSize of
    1 : FillChar(Walker^, succ(aToCol - aFromCol), byte(FDefaultItem));
    2 : begin
          Value := word(FDefaultItem);
          for i := 0 to (aToCol - aFromCol) do begin
            PWord(Walker)^ := Value;
            inc(Walker, 2);
          end;
        end;
    4 : begin
          Value := FDefaultItem;
          for i := 0 to (aToCol - aFromCol) do begin
            PLongint(Walker)^ := Value;
            inc(Walker, 4);
          end;
        end;
  end;{case}
end;
{--------}
procedure TApxTerminalArray.DeleteItems(aCount : integer;
                                       aRow   : integer;
                                       aCol   : integer);
var
  ItemCount : integer;
  Distance  : integer;
  FromPtr   : PAnsiChar;
  ToPtr     : PAnsiChar;
  Value     : longint;
  i         : integer;
begin
  {$IFDEF UseRangeChecks}
  { Range check aRow and aCol }
  if (aRow < 0) or (aRow >= RowCount) or
     (aCol < 0) or (aCol >= ColCount) then
    raise EApxTerminal.CreateFormatted (AxecTrmBufDelItemsRangeErr,
                                       [aRow, aCol], False);
  {$ENDIF}
  Distance := ColCount - aCol;
  if (Distance > aCount) then
    Distance := aCount;
  ItemCount := ColCount - aCol - Distance;
  case ItemSize of
    1 : begin
          ToPtr := @FItems[(aRow * FActColCount) + aCol];
          if (ItemCount > 0) then begin
            FromPtr := ToPtr + Distance;
            Move(FromPtr^, ToPtr^, ItemCount);
          end;
          ToPtr := ToPtr + ItemCount;
          FillChar(ToPtr^, Distance, byte(FDefaultItem));
        end;
    2 : begin
          ToPtr := @FItems[((aRow * FActColCount) + aCol) * 2];
          if (ItemCount > 0) then begin
            FromPtr := ToPtr + (Distance * 2);
            Move(FromPtr^, ToPtr^, ItemCount * 2);
          end;
          ToPtr := ToPtr + (ItemCount * 2);
          Value := word(FDefaultItem);
          for i := 0 to pred(Distance) do begin
            PWord(ToPtr)^ := Value;
            inc(ToPtr, 2);
          end;
        end;
    4 : begin
          ToPtr := @FItems[((aRow * FActColCount) + aCol) * 4];
          if (ItemCount > 0) then begin
            FromPtr := ToPtr + (Distance * 4);
            Move(FromPtr^, ToPtr^, ItemCount * 4);
          end;
          ToPtr := ToPtr + (ItemCount * 4);
          Value := FDefaultItem;
          for i := 0 to pred(Distance) do begin
            PLongint(ToPtr)^ := Value;
            inc(ToPtr, 4);
          end;
        end;
  end; { case }
end;
{--------}
function TApxTerminalArray.GetItemPtr(aRow, aCol : integer) : pointer;
begin
  {$IFDEF UseRangeChecks}
  { Range check aRow and aCol }
  if (aRow < 0) or (aRow >= RowCount) or
     (aCol < 0) or (aCol >= ColCount) then
    raise EApxTerminal.CreateFormatted (AxecTrmBufGetItemPtrRangeErr,
                                       [aRow, aCol], False);
  {$ENDIF}
  if FItems = nil then
    Result := nil
  else begin
    case ItemSize of
      1 : Result := @FItems[(aRow * FActColCount) + aCol];
      2 : Result := @FItems[(aRow * FActColCount * 2) + (aCol * 2)];
      4 : Result := @FItems[(aRow * FActColCount * 4) + (aCol * 4)];
    else
      raise EApxTerminal.Create (AxecTrmBufGetItemPtrSizeErr, False);
      Result := nil;
    end; { case }
  end;
end;
{--------}
procedure TApxTerminalArray.InsertItems(aCount : integer;
                                       aRow   : integer;
                                       aCol   : integer);
var
  ItemCount : integer;
  Distance  : integer;
  FromPtr   : PAnsiChar;
  ToPtr     : PAnsiChar;
  Value     : longint;
  i         : integer;
begin
  {$IFDEF UseRangeChecks}
  { Range check aRow and aCol }
  if (aRow < 0) or (aRow >= RowCount) or
     (aCol < 0) or (aCol >= ColCount) then
    raise EApxTerminal.CreateFormatted (AxecTrmBufInsertItemRangeErr,
                                       [aRow, aCol], False);
  {$ENDIF}
  Distance := ColCount - aCol;
  if (Distance > aCount) then
    Distance := aCount;
  ItemCount := ColCount - aCol - Distance;
  case ItemSize of
    1 : begin
          FromPtr := @FItems[(aRow * FActColCount) + aCol];
          if (ItemCount > 0) then begin
            ToPtr := FromPtr + Distance;
            Move(FromPtr^, ToPtr^, ItemCount);
          end;
          FillChar(FromPtr^, Distance, byte(FDefaultItem));
        end;
    2 : begin
          FromPtr := @FItems[((aRow * FActColCount) + aCol) * 2];
          if (ItemCount > 0) then begin
            ToPtr := FromPtr + (Distance * 2);
            Move(FromPtr^, ToPtr^, ItemCount * 2);
          end;
          Value := word(FDefaultItem);
          for i := 0 to pred(Distance) do begin
            PWord(FromPtr)^ := Value;
            inc(FromPtr, 2);
          end;
        end;
    4 : begin
          FromPtr := @FItems[((aRow * FActColCount) + aCol) * 4];
          if (ItemCount > 0) then begin
            ToPtr := FromPtr + (Distance * 4);
            Move(ToPtr^, FromPtr^, ItemCount * 4);
          end;
          Value := FDefaultItem;
          for i := 0 to pred(Distance) do begin
            PLongint(FromPtr)^ := Value;
            inc(FromPtr, 4);
          end;
        end;
  end; { case }
end;
{--------}
procedure TApxTerminalArray.ReplaceItems(aOldItem : pointer;
                                        aNewItem : pointer);

var
  Walker    : PAnsiChar;
  OldValue  : longint;
  NewValue  : longint;
  Row       : integer;
  i         : integer;
begin
  case ItemSize of
    1 : begin
          OldValue := PByte(aOldItem)^;
          NewValue := PByte(aNewItem)^;
        end;
    2 : begin
          OldValue := PWord(aOldItem)^;
          NewValue := PWord(aNewItem)^;
        end;
    4 : begin
          OldValue := PLongint(aOldItem)^;
          NewValue := PLongint(aNewItem)^;
        end;
  else
    { dummy statements that will never get executed, however they fool }
    { the warning analyzer in the compiler }
    OldValue := 0;
    NewValue := 0;
  end;{case}
  for Row := 0 to pred(RowCount) do begin
    Walker := @FItems[(Row * FActColCount) * ItemSize];
    case ItemSize of
      1 : for i := 0 to pred(ColCount) do begin
            if (PByte(Walker)^ = OldValue) then
               PByte(Walker)^ := NewValue;
            inc(Walker);
          end;
      2 : for i := 0 to pred(ColCount) do begin
            if (PWord(Walker)^ = OldValue) then
               PWord(Walker)^ := NewValue;
            inc(Walker, 2);
          end;
      4 : for i := 0 to pred(ColCount) do begin
            if (PLongint(Walker)^ = OldValue) then
               PLongint(Walker)^ := NewValue;
            inc(Walker, 4);
          end;
    end;{case}
  end;
end;
{--------}
procedure TApxTerminalArray.ScrollRows(aCount : integer;
                                      aStartRow, aEndRow : integer);
var
  ThisRow : integer;
  FromPtr : PAnsiChar;
  ToPtr   : PAnsiChar;
  i       : integer;
begin
  {$IFDEF UseRangeChecks}
  { Range check aStartRow and aEndRow }
  if (aStartRow < 0) or (aStartRow >= RowCount) or
     (aEndRow < 0) or (aEndRow >= RowCount) then
    raise EApxTerminal.CreateFormatted (AxecTrmBufScrollRowRangeErr,
                                       [aStartRow, aEndRow], False);
  {$ENDIF}
  if (FItems <> nil) and (aCount <> 0) then begin
    { make sure the end row is larger than the start row }
    if (aEndRow < aStartRow) then begin
      ThisRow := aEndRow;
      aEndRow := aStartRow;
      aStartRow := ThisRow;
    end;
    { split the code depending on whether we are scrolling upwards, }
    { aCount is +ve, or downwards, aCount is -ve }
    if (aCount > 0) then { scroll upwards } begin
      { if the number of rows to scroll is greater than the difference  }
      { between the start and end rows, all we need to do is blank out  }
      { all the rows between start and end inclusive, otherwise we have }
      { some scrolling to do }
      ThisRow := aStartRow;
      if (aCount <= (aEndRow - aStartRow)) then begin
        ToPtr := @FItems[ThisRow * FActColCount * ItemSize];
        FromPtr := @FItems[(ThisRow + aCount) * FActColCount * ItemSize];
        for i := 0 to (aEndRow - aStartRow - aCount) do begin
          Move(FromPtr^, ToPtr^, ColCount * ItemSize);
          inc(FromPtr, FActColCount * ItemSize);
          inc(ToPtr, FActColCount * ItemSize);
          inc(ThisRow);
        end;
      end;
      { now blank out everything from ThisRow to aEndRow }
      taClearRows(FItems, FActColCount, ThisRow, aEndRow);
    end
    else { scroll downwards } begin
      { if the number of rows to scroll is greater than the difference  }
      { between the start and end rows, all we need to do is blank out  }
      { all the rows between start and end inclusive, otherwise we have }
      { some scrolling to do }
      aCount := -aCount;
      ThisRow := aEndRow;
      if (aCount <= (aEndRow - aStartRow)) then begin
        ToPtr := @FItems[ThisRow * FActColCount * ItemSize];
        FromPtr := @FItems[(ThisRow - aCount) * FActColCount * ItemSize];
        for i := 0 to (aEndRow - aStartRow - aCount) do begin
          Move(FromPtr^, ToPtr^, ColCount * ItemSize);
          dec(FromPtr, FActColCount * ItemSize);
          dec(ToPtr, FActColCount * ItemSize);
          dec(ThisRow);
        end;
      end;
      { now blank out everything from aStartRow to ThisRow }
      taClearRows(FItems, FActColCount, aStartRow, ThisRow);
    end;
  end;
end;
{--------}
procedure TApxTerminalArray.SetDefaultItem(aDefaultItem : pointer);
begin
  case ItemSize of
    1 : FDefaultItem := PByte(aDefaultItem)^;
    2 : FDefaultItem := PWord(aDefaultItem)^;
    4 : FDefaultItem := PLongint(aDefaultItem)^;
  end;
end;
{--------}
procedure TApxTerminalArray.taClearRows(aBuffer : PAnsiChar;
                                       aActColCount : integer;
                                       aStartRow, aEndRow : integer);
var
  Walker     : PAnsiChar;
  Value      : longint;
  DWORDCount : integer;
  i          : integer;
begin
  Walker := @aBuffer[aStartRow * aActColCount * ItemSize];
  if (ItemSize = 1) then
    FillChar(Walker^,
             succ(aEndRow - aStartRow) * aActColCount,
             byte(FDefaultItem))
  else begin
    if (ItemSize = 2) then begin
      Value := (FDefaultItem shl 16) + word(FDefaultItem);
      DWORDCount := (succ(aEndRow - aStartRow) * aActColCount) div 2;
    end
    else begin
      Value := FDefaultItem;
      DWORDCount := succ(aEndRow - aStartRow) * aActColCount;
    end;
    for i := 0 to pred(DWORDCount) do begin
      PLongint(Walker)^ := Value;
      inc(Walker, 4);
    end;
  end;
end;
{--------}
procedure TApxTerminalArray.taGrowArray(aRowCount,
                                       aColCount,
                                       aActColCount : integer);
var
  NewArray : PAnsiChar;
  RowSize  : integer;
  NumRows  : integer;
  FromPtr  : PAnsiChar;
  ToPtr    : PAnsiChar;
  i        : integer;
begin
  { make sure we have the new actual column count: this is the external }
  { column count rounded up so that the actual length of a row in bytes }
  { is a multiple of four--this makes fills and moves much faster }
  if (aActColCount = -1) then begin
    case ItemSize of
      1 : aActColCount := ((aColCount + 3) div 4) * 4;
      2 : aActColCount := ((aColCount + 1) div 2) * 2;
      4 : aActColCount := aColCount;
    end;{ case }
  end;
  { nothing to do if either the row or actual column count is zero }
  if (aRowCount = 0) or (aActColCount = 0) then
    Exit;
  { equally obvious, nothing to do if neither the row and actual column }
  { count have changed }
  if (aRowCount = RowCount) and (aActColCount = FActColCount) then
    Exit;
  { at this point we must allocate another array }
  GetMem(NewArray, aRowCount * aActColCount * ItemSize);
  { blank it all out using the current default item }
  taClearRows(NewArray, aActColCount, 0, pred(aRowCount));
  { if the old array existed, transfer over the data, row by row, }
  { starting at the bottom }
  if (FItems <> nil) then begin
    { calculate the number of bytes to copy per row }
    if (ColCount < aColCount) then
      RowSize := ColCount * ItemSize
    else
      RowSize := aColCount * ItemSize;
    { calculate the number of rows to copy }
    if (RowCount < aRowCount) then
      NumRows := RowCount
    else
      NumRows := aRowCount;
    { copy the rows }
    FromPtr := @FItems[RowCount * FActColCount * ItemSize];
    ToPtr := @NewArray[aRowCount * aActColCount * ItemSize];
    for i := pred(RowCount) downto (RowCount - NumRows) do begin
      dec(FromPtr, FActColCount * ItemSize);
      dec(ToPtr, aActColCount * ItemSize);
      Move(FromPtr^, ToPtr^, RowSize);
    end;
    { dispose of the old array }
    FreeMem(FItems, RowCount * FActColCount * ItemSize);
  end;
  { save the new array }
  FItems := NewArray;
  FActColCount := aActColCount;
end;
{--------}
procedure TApxTerminalArray.taSetColCount(aNewCount : integer);
begin
  {$IFDEF UseRangeChecks}
  { Range check aNewCount }
  if (aNewCount < 0) then
    raise EApxTerminal.CreateFormatted (AxecTrmBufColCountTooLow1,
                                       [aNewCount], False);
  {$ENDIF}
  if (aNewCount <> ColCount) then begin
    taGrowArray(RowCount, aNewCount, -1);
    FColCount := aNewCount;
  end;
end;
{--------}
procedure TApxTerminalArray.taSetRowCount(aNewCount : integer);
begin
  {$IFDEF UseRangeChecks}
  {Range check aNewCount}
  if (aNewCount < 0) then
    raise EApxTerminal.CreateFormatted (AxecTrmBufColCountTooLow2,
                                       [aNewCount], False);
  {$ENDIF}
  if (aNewCount <> RowCount) then begin
    taGrowArray(aNewCount, ColCount, FActColCount);
    FRowCount := aNewCount;
  end;
end;
{--------}
procedure TApxTerminalArray.WriteDupItems(aItem  : pointer;
                                         aCount : integer;
                                         aRow, aCol : integer);
var
  Walker    : PAnsiChar;
  Value     : longint;
  i         : integer;
  ItemCount : integer;
begin
  {$IFDEF UseRangeChecks}
  { Range check aRow and aCol }
  if (aRow < 0) or (aRow >= RowCount) or
     (aCol < 0) or (aCol >= ColCount) then
    raise EApxTerminal.CreateFormatted (AxecTrmBufWriteDupItemsRange,
                                       [aRow, aCol], False);
  {$ENDIF}
  if (FItems <> nil) then begin
    ItemCount := ColCount - aCol;
    if (ItemCount > aCount) then
      ItemCount := aCount;
    case ItemSize of
      1 : FillChar(FItems[(aRow * FActColCount) + aCol],
                   ItemCount, PByte(aItem)^);
      2 : begin
            Walker := @FItems[((aRow * FActColCount) + aCol) * 2];
            Value := PWord(aItem)^;
            for i := 0 to pred(ItemCount) do begin
              PWord(Walker)^ := Value;
              inc(Walker, 2);
            end;
          end;
      4 : begin
            Walker := @FItems[((aRow * FActColCount) + aCol) * 4];
            Value := PLongint(aItem)^;
            for i := 0 to pred(ItemCount) do begin
              PLongint(Walker)^ := Value;
              inc(Walker, 4);
            end;
          end;
    end;{case}
  end;
end;
{--------}
procedure TApxTerminalArray.WriteItems(aItems : pointer;
                                      aCount : integer;
                                      aRow, aCol : integer);
var
  ItemCount : integer;
begin
  {$IFDEF UseRangeChecks}
  {Range check aRow and aCol}
  if (aRow < 0) or (aRow >= RowCount) or
     (aCol < 0) or (aCol >= ColCount) then
    raise EApxTerminal.CreateFormatted (AxecTrmBufWriteItemsRangeErr,
                                       [aRow, aCol], False);
  {$ENDIF}
  if (FItems <> nil) then begin
    ItemCount := ColCount - aCol;
    if (ItemCount > aCount) then
      ItemCount := aCount;
    case ItemSize of
      1 : Move(aItems^,
               FItems[(aRow * FActColCount) + aCol],
               ItemCount);
      2 : Move(aItems^,
               FItems[(aRow * FActColCount * 2) + (aCol * 2)],
               ItemCount * 2);
      4 : Move(aItems^,
               FItems[(aRow * FActColCount * 4) + (aCol * 4)],
               ItemCount * 4);
    end;{case}
  end;
end;
{====================================================================}


{===Bitset routines==================================================}
procedure ApxTClearAllBits(aBitset : PByteArray; aBitCount : integer);
begin
  FillChar(aBitset^, (aBitCount+7) shr 3, 0);
end;
{--------}
procedure ApxTClearBit(aBitset : PByteArray; aBit : integer);
var
  BS : PAnsiChar; 
  P  : PAnsiChar;
  M  : byte;
begin
  BS := PAnsiChar (aBitSet);
  P := BS + (aBit shr 3);
  M := 1 shl (byte(aBit) and 7);
  P^ := char(byte(P^) and not M);
end;
{--------}
function ApxTIsBitSet(aBitset : PByteArray; aBit : integer) : Boolean;
var
  BS : PAnsiChar; 
  P  : PAnsiChar;
  M  : byte;
begin
  BS := PAnsiChar (aBitSet); 
  P := BS + (aBit shr 3);
  M := 1 shl (byte(aBit) and 7);
  Result := (byte(P^) and M) <> 0;
end;
{--------}
function ApxTReallocBitset(aBitset      : PByteArray;
                          aOldBitCount : integer;
                          aNewBitCount : integer) : PByteArray;
var
  XferBitCount : integer;
begin
  if (aNewBitCount = 0) then
    Result := nil
  else begin
    Result := AllocMem(aNewBitCount);
    if (aBitset <> nil) then begin
      if (aOldBitCount < aNewBitCount) then
        XferBitCount := aOldBitCount
      else
        XferBitCount := aNewBitCount;
      Move(aBitset^, Result^, (XferBitCount+7) shr 3);
    end;
  end;
  FreeMem(aBitset, (aOldBitCount+7) shr 3);
end;
{--------}
(***** not used yet
procedure ApxTSetAllBits(aBitset : PByteArray; aBitCount : integer);
begin
  FillChar(aBitset^, (aBitCount+7) shr 3, $FF);
end;
*****)
{--------}
procedure ApxTSetBit(aBitset : PByteArray; aBit : integer);
var
  BS : PAnsiChar;
  P  : PAnsiChar;
  M  : byte;
begin
  BS := PAnsiChar (aBitSet); 
  P := BS + (aBit shr 3);
  M := 1 shl (byte(aBit) and 7);
  P^ := char(byte(P^) or M);
end;
{====================================================================}


{===Invalid rectangle routines==========================================}
const
  RectsPerPage = 200;
type
  PInvRect = ^TInvRect;
  TInvRect = packed record
    irNext : PInvRect;
    irRect : TRect;
  end;
  PInvRectPage = ^TInvRectPage;
  TInvRectPage = packed record
    irpNext  : PInvRectPage;
    irpRects : array [0.. pred(RectsPerPage)] of TInvRect;
  end;
var
  InvRectFreeList : PInvRect;
  InvRectPageList : PInvRectPage;
{--------}
procedure ApxTFreeInvRect(P : PInvRect);
begin
  { push rect onto free list }
  P^.irNext := InvRectFreeList;
  InvRectFreeList := P;
end;
{--------}
procedure ApxTAllocInvRectPage;
var
  NewPage : PInvRectPage;
  i       : integer;
begin
  { alloc new page and add it to front of page list }
  New(NewPage);
  NewPage^.irpNext := InvRectPageList;
  InvRectPageList := NewPage;
  { add all rects on this page to free list }
  for i := 0 to pred(RectsPerPage) do
    ApxTFreeInvRect(@NewPage^.irpRects[i]);
end;
{--------}
function ApxTAllocInvRect : PInvRect;
begin
  { pop top rect from free list; if none, add a whole page's worth }
  if (InvRectFreeList = nil) then
    ApxTAllocInvRectPage;
  Result := InvRectFreeList;
  InvRectFreeList := Result^.irNext;
end;
{--------}
procedure ApxTFreeInvRectPages;
var
  Temp : PInvRectPage;
begin
  { dispose of all rect pages }
  while (InvRectPageList <> nil) do begin
    Temp := InvRectPageList;
    InvRectPageList := Temp^.irpNext;
    Dispose(Temp);
  end;
  { since all rects have now gone, force the rect free list to nil }
  InvRectFreeList := nil;
end;
{--------}
procedure ApxTApxdInvalidRect(var aInvRectList : PInvRect;
                       const aRect        : TRect);
var
  NewRect : PInvRect;
begin
  NewRect := ApxTAllocInvRect;
  NewRect^.irNext := aInvRectList;
  aInvRectList := NewRect;
  NewRect^.irRect := aRect;
end;
{--------}
function ApxTRemoveInvalidRect(var aInvRectList : PInvRect;
                           var aRect        : TRect) : Boolean;
var
  TopRect : PInvRect;
begin
  if (aInvRectList = nil) then
    Result := False
  else begin
    Result := True;
    TopRect := aInvRectList;
    aInvRectList := TopRect^.irNext;
    aRect := TopRect^.irRect;
    ApxTFreeInvRect(TopRect);
  end;
end;
{--------}
function ApxTPeekInvalidRect(aInvRectList : PInvRect;
                     var aRect        : TRect) : Boolean;
begin
  if (aInvRectList = nil) then
    Result := False
  else begin
    Result := True;
    aRect := aInvRectList^.irRect;
  end;
end;
{--------}
procedure ApxTMergeInvalidRects(aInvRectList : PInvRect);
var
  Temp    : PInvRect;
  Walker  : PInvRect;
  MinRect : TRect;
begin
  if (aInvRectList = nil) then
    Exit;
  { performs a simple merge of all the invalid rects in the list by     }
  { working out the rect that just covers them all; free the rects from }
  { the list after we read them--leaving the first for our use }
  MinRect := aInvRectList^.irRect;
  Walker := aInvRectList^.irNext;
  while (Walker <> nil) do begin
    with Walker^.irRect do begin
      if Left < MinRect.Left then
        MinRect.Left := Left;
      if Top < MinRect.Top then
        MinRect.Top := Top;
      if Right > MinRect.Right then
        MinRect.Right := Right;
      if Bottom > MinRect.Bottom then
        MinRect.Bottom := Bottom;
    end;
    Temp := Walker;
    Walker := Walker^.irNext;
    ApxTFreeInvRect(Temp);
  end;
  { MinRect now contains the smallest rectangle that covers all invalid }
  { rects in the list; set this minimum invalid rect into the first     }
  { (and only) item in the list }
  aInvRectList^.irNext := nil;
  aInvRectList^.irRect := MinRect;
end;
{====================================================================}


{===TApxTerminalBuffer================================================}
constructor TApxTerminalBuffer.Create(aUseWideChars : Boolean);
var
  i : integer;
begin
  inherited Create;

  { set the values of the properties that define defaults }
  FDefBackColor := adc_TermBufBackColor;
  FDefForeColor := adc_TermBufForeColor;
  FDefAnsiChar := ' ';
  FDefWideChar := ' ';
  FDefCharSet := 0;
  FDefAttr := [];

  { set the 'power-up' values }
  FBackColor := adc_TermBufBackColor;
  FForeColor := adc_TermBufForeColor;
  UseAbsAddress := adc_TermBufUseAbsAddress;
  UseAutoWrap := adc_TermBufUseAutoWrap;
  UseAutoWrapDelay := adc_TermBufUseAutoWrapDelay;
  UseInsertMode := adc_TermBufUseInsertMode;
  UseNewLineMode := adc_TermBufUseNewLineMode;
  UseScrollRegion := adc_TermBufUseScrollRegion;

  { set up all the matrices to hold the displayed data }
  { ..character matrix }
  if aUseWideChars then begin
    FCharMatrix := TApxTerminalArray.Create(sizeof(WideChar));
    FCharMatrix.SetDefaultItem(@FDefWideChar);
  end
  else
  begin
    FCharMatrix := TApxTerminalArray.Create(sizeof(AnsiChar));
    FCharMatrix.SetDefaultItem(@FDefAnsiChar);
  end;
  FCharMatrix.ColCount := adc_TermBufColCount;
  FCharMatrix.RowCount := adc_TermBufScrollRowCount;

  { ..character set matrix }
  FCharSetMatrix := TApxTerminalArray.Create(sizeof(byte));
  FCharSetMatrix.SetDefaultItem(@FDefCharSet);
  FCharSetMatrix.ColCount := adc_TermBufColCount;
  FCharSetMatrix.RowCount := adc_TermBufScrollRowCount;

  { ..character attributes matrix }
  FAttrMatrix := TApxTerminalArray.Create(sizeof(TApxTerminalCharAttrs));
  FAttrMatrix.SetDefaultItem(@FDefAttr);
  FAttrMatrix.ColCount := adc_TermBufColCount;
  FAttrMatrix.RowCount := adc_TermBufScrollRowCount;

  { ..character foreground color matrix }
  FForeColorMatrix := TApxTerminalArray.Create(sizeof(TColor));
  FForeColorMatrix.SetDefaultItem(@FDefForeColor);
  FForeColorMatrix.ColCount := adc_TermBufColCount;
  FForeColorMatrix.RowCount := adc_TermBufScrollRowCount;

  { ..character background color matrix }
  FBackColorMatrix := TApxTerminalArray.Create(sizeof(TColor));
  FBackColorMatrix.SetDefaultItem(@FDefBackColor);
  FBackColorMatrix.ColCount := adc_TermBufColCount;
  FBackColorMatrix.RowCount := adc_TermBufScrollRowCount;

  { initialize the terminal dimensions }
  FUseWideChars := aUseWideChars;
  SVRowCount := adc_TermBufScrollRowCount;
  ColCount := adc_TermBufColCount;
  RowCount := adc_TermBufRowCount;

  ClearAllHorzTabStops;
  ClearAllVertTabStops;
  i := Col;
  while (i < ColCount) do begin
    Col := i;
    SetHorzTabStop;
    inc(i, 8);
  end;
  Row := 1;
  Col := 1;

  { set up the current cursor position }
  FCursorRow := SVRowCount - RowCount;
  FDisplayOriginRow := FCursorRow;
  FCursorCol := 0;

  { add the whole screen as an invalid rect }
  FCursorMoved := True;
  tbInvalidateRect(FCursorRow, 0, pred(SVRowCount), pred(ColCount));
  tbFireOnCursorMovedEvent;
end;
{--------}
destructor TApxTerminalBuffer.Destroy;
var
  OurRect : TRect;
begin
  { remove all of the invalid rects and discard them }
  while ApxTRemoveInvalidRect(PInvRect(FInvRectList), OurRect) do { nothing };
  { free the tab stops }
  ApxTReallocBitset(FVertTabStops, RowCount, 0);
  ApxTReallocBitset(FHorzTabStops, ColCount, 0);
  { free all arrays }
  FBackColorMatrix.Free;
  FForeColorMatrix.Free;
  FAttrMatrix.Free;
  FCharSetMatrix.Free;
  FCharMatrix.Free;
  inherited Destroy;
end;
{--------}
procedure TApxTerminalBuffer.ClearAllHorzTabStops;
begin
  if (ColCount <> 0) then
    ApxTClearAllBits(FHorzTabStops, ColCount);
end;
{--------}
procedure TApxTerminalBuffer.ClearAllVertTabStops;
begin
  if (RowCount <> 0) then
    ApxTClearAllBits(FVertTabStops, RowCount);
end;
{--------}
procedure TApxTerminalBuffer.ClearHorzTabStop;
begin
  if (ColCount <> 0) then
    ApxTClearBit(FHorzTabStops, FCursorCol);
end;
{--------}
procedure TApxTerminalBuffer.ClearVertTabStop;
begin
  if (RowCount <> 0) then
    ApxTClearBit(FVertTabStops, FCursorRow);
end;
{--------}
procedure TApxTerminalBuffer.DeleteChars(aCount : integer);
var
  CharCount : integer;
begin
  FBeyondMargin := False;
  {$IFDEF UseRangeCheck}
  if (aCount <= 0) then
    raise EApxTerminal.Create (AxecTrmBufDeleteCharsTooLow, False);
  {$ENDIF}
  { the actual number of characters to delete is constrained by the }
  { current display region }
  CharCount := FDisplayOriginCol + FDisplayColCount - FCursorCol;
  if (CharCount > aCount) then
    CharCount := aCount;
  if (CharCount > 0) then begin
    FCharMatrix.DeleteItems(CharCount, FCursorRow, FCursorCol);
    FCharSetMatrix.DeleteItems(CharCount, FCursorRow, FCursorCol);
    FAttrMatrix.DeleteItems(CharCount, FCursorRow, FCursorCol);
    FForeColorMatrix.DeleteItems(CharCount, FCursorRow, FCursorCol);
    FBackColorMatrix.DeleteItems(CharCount, FCursorRow, FCursorCol);
    { the cursor does not move }
    tbInvalidateRect(FCursorRow, FCursorCol, FCursorRow, FCursorCol+CharCount-1);
  end;
end;
{--------}
procedure TApxTerminalBuffer.DeleteLines(aCount : integer);
var
  MaxRow : integer;
begin
  FBeyondMargin := False;
  { deleting lines is equivalent to a scroll up to the current cursor }
  { position; we take account of any scroll region, of course }
  {$IFDEF UseRangeCheck}
  if (aCount <= 0) then
    raise EApxTerminal.Create (AxecTrmBufDeleteLinesTooLow, False);
  {$ENDIF}
  MaxRow := FDisplayOriginRow + FDisplayRowCount - 1;
  tbScrollRows(aCount, FCursorRow, MaxRow);
  { the cursor does not move }
  tbInvalidateRect(FCursorRow, FDisplayOriginCol,
                   MaxRow, FDisplayOriginCol+FDisplayColCount-1);
end;
{--------}
procedure TApxTerminalBuffer.DoBackHorzTab;
var
  NewCol : integer;
begin
  if (ColCount > 0) then begin
    NewCol := FCursorCol;
    while (NewCol > FDisplayOriginCol) do begin
      dec(NewCol);
      if ApxTIsBitSet(FHorzTabStops, NewCol) then begin
        FCursorMoved := FCursorMoved or (FCursorCol <> NewCol);
        FCursorCol := NewCol;
        tbFireOnCursorMovedEvent;
        Exit;
      end;
    end;
    FCursorMoved := FCursorMoved or (FCursorCol <> FDisplayOriginCol);
    FCursorCol := FDisplayOriginCol;
    tbFireOnCursorMovedEvent;
  end;
end;
{--------}
procedure TApxTerminalBuffer.DoBackVertTab;
var
  NewRow : integer;
begin
  FBeyondMargin := False;
  if (RowCount > 0) then begin
    NewRow := FCursorRow;
    while (NewRow > FDisplayOriginRow) do begin
      dec(NewRow);
      if ApxTIsBitSet(FVertTabStops, NewRow) then begin
        FCursorMoved := FCursorMoved or (FCursorRow <> NewRow);
        FCursorRow := NewRow;
        tbFireOnCursorMovedEvent;
        Exit;
      end;
    end;
    FCursorMoved := FCursorMoved or (FCursorRow <> FDisplayOriginRow);
    FCursorRow := FDisplayOriginRow;
    tbFireOnCursorMovedEvent;
  end;
end;
{--------}
procedure TApxTerminalBuffer.DoBackspace;
begin
  FBeyondMargin := False;
  if (FCursorCol > FDisplayOriginCol) then begin
    FCursorMoved := True;
    dec(FCursorCol);
    tbFireOnCursorMovedEvent;
  end;
end;
{--------}
procedure TApxTerminalBuffer.DoCarriageReturn;
begin
  FBeyondMargin := False;
  FCursorMoved := FCursorMoved or (FCursorCol <> FDisplayOriginCol);
  FCursorCol := FDisplayOriginCol;
  tbFireOnCursorMovedEvent;
end;
{--------}
procedure TApxTerminalBuffer.DoHorzTab;
var
  NewCol : integer;
  MaxCol : integer;
begin
  FBeyondMargin := False;
  if (ColCount > 0) then begin
    NewCol := FCursorCol;
    MaxCol := FDisplayOriginCol + FDisplayColCount - 1;
    while (NewCol < MaxCol) do begin
      inc(NewCol);
      if ApxTIsBitSet(FHorzTabStops, NewCol) then begin
        FCursorMoved := FCursorMoved or (FCursorCol <> NewCol);
        FCursorCol := NewCol;
        tbFireOnCursorMovedEvent;
        Exit;
      end;
    end;
    FCursorMoved := FCursorMoved or (FCursorCol <> MaxCol);
    FCursorCol := MaxCol;
    tbFireOnCursorMovedEvent;
  end;
end;
{--------}
procedure TApxTerminalBuffer.DoLineFeed;
begin
  FBeyondMargin := False;
  if UseNewLineMode then
    DoCarriageReturn;
  MoveCursorDown(True);
end;
{--------}
procedure TApxTerminalBuffer.DoVertTab;
var
  NewRow : integer;
  MaxRow : integer;
begin
  FBeyondMargin := False;
  if (RowCount > 0) then begin
    NewRow := FCursorRow;
    MaxRow := FDisplayOriginRow + FDisplayRowCount - 1;
    while (NewRow < MaxRow) do begin
      inc(NewRow);
      if ApxTIsBitSet(FVertTabStops, NewRow) then begin
        FCursorMoved := FCursorMoved or (FCursorRow <> NewRow);
        FCursorRow := NewRow;
        tbFireOnCursorMovedEvent;
        Exit;
      end;
    end;
    FCursorMoved := FCursorMoved or (FCursorRow <> MaxRow);
    FCursorRow := MaxRow;
    tbFireOnCursorMovedEvent;
  end;
end;
{--------}
procedure TApxTerminalBuffer.EraseAll;
begin
  { WARNING: this DOES NOT use the scroll region, if defined, it blanks }
  {          out everything in the scrollback buffer }
  FBeyondMargin := False;
  FCharMatrix.Clear;
  FCharSetMatrix.Clear;
  FAttrMatrix.Clear;
  FForeColorMatrix.Clear;
  FBackColorMatrix.Clear;
  Row := 1;
  Col := 1;
  tbInvalidateRect(FCursorRow, 0,
                   pred(SVRowCount), pred(ColCount));
end;
{--------}
procedure TApxTerminalBuffer.EraseChars(aCount : integer);
var
  CharCount : integer;
  ToColNum  : integer;
  CurRow    : integer;
  CurCol    : integer;
  MaxCol    : integer;
  MaxRow    : integer;
begin
  { WARNING: this uses the scroll region, if defined }

  FBeyondMargin := False;
  {$IFDEF UseRangeChecks}
  if (aCount <= 0) then
    raise EApxTerminal.Create (AxecTrmBufEraseCharsCount, False);
  {$ENDIF}
  { this is complicated by the need to erase chars on individual lines separately }
  CurRow := FCursorRow;
  CurCol := FCursorCol;
  MaxCol := FDisplayOriginCol + FDisplayColCount - 1;
  MaxRow := FDisplayOriginRow + FDisplayRowCount - 1;
  while (aCount > 0) and (CurRow <= MaxRow) do begin
    { calculate the number of characters to erase in this row }
    CharCount := MaxCol - CurCol + 1;
    if (CharCount > aCount) then
      CharCount := aCount;
    { calculate the final column number }
    ToColNum := CurCol + CharCount - 1;
    { erase }
    FCharMatrix.ClearItems(CurRow, CurCol, ToColNum);
    FCharSetMatrix.ClearItems(CurRow, CurCol, ToColNum);
    FAttrMatrix.ClearItems(CurRow, CurCol, ToColNum);
    FForeColorMatrix.ClearItems(CurRow, CurCol, ToColNum);
    FBackColorMatrix.ClearItems(CurRow, CurCol, ToColNum);
    tbInvalidateRect(CurRow, CurCol, CurRow, ToColNum);
    { set up for the next loop }
    dec(aCount, CharCount);
    inc(CurRow);
    CurCol := FDisplayOriginCol;
  end;
end;
{--------}
procedure TApxTerminalBuffer.EraseFromBOL;
begin
  { WARNING: this uses the scroll region, if defined }

  FBeyondMargin := False;
  { set all characters from the beginning of the line, up to and }
  { including the cursor position, to blanks }
  FCharMatrix.ClearItems(FCursorRow, FDisplayOriginCol, FCursorCol);
  FCharSetMatrix.ClearItems(FCursorRow, FDisplayOriginCol, FCursorCol);
  FAttrMatrix.ClearItems(FCursorRow, FDisplayOriginCol, FCursorCol);
  FForeColorMatrix.ClearItems(FCursorRow, FDisplayOriginCol, FCursorCol);
  FBackColorMatrix.ClearItems(FCursorRow, FDisplayOriginCol, FCursorCol);
  tbInvalidateRect(FCursorRow, FDisplayOriginCol,
                   FCursorRow, FCursorCol);
end;
{--------}
procedure TApxTerminalBuffer.EraseFromBOW;
var
  DisplayStartRow : integer;
begin
  { WARNING: this DOES NOT use the scroll region, if defined, it blanks }
  {          out everything on the window up to and including the       }
  {          cursor position }

  FBeyondMargin := False;
  { set all characters from the beginning of the line, up to and }
  { including the cursor position, to blanks }
  FCharMatrix.ClearItems(FCursorRow, 0, FCursorCol);
  FCharSetMatrix.ClearItems(FCursorRow, 0, FCursorCol);
  FAttrMatrix.ClearItems(FCursorRow, 0, FCursorCol);
  FForeColorMatrix.ClearItems(FCursorRow, 0, FCursorCol);
  FBackColorMatrix.ClearItems(FCursorRow, 0, FCursorCol);
  tbInvalidateRect(FCursorRow, 0,
                   FCursorRow, FCursorCol);
  { now erase all previous lines, by scrolling them out of existence }
  DisplayStartRow := SVRowCount - RowCount;
  if (FCursorRow > DisplayStartRow) then begin
    tbScrollRows(FCursorRow - DisplayStartRow,
                 DisplayStartRow, pred(FCursorRow));
    tbInvalidateRect(DisplayStartRow, 0,
                     pred(FCursorRow), pred(ColCount));
  end;
end;
{--------}
procedure TApxTerminalBuffer.EraseLine;
var
  MaxCol : integer;
begin
  {WARNING: this uses the scroll region, if defined}

  FBeyondMargin := False;
  { set all characters from the beginning to the end of the line to blanks }
  MaxCol := FDisplayOriginCol + FDisplayColCount - 1;
  FCharMatrix.ClearItems(FCursorRow, FDisplayOriginCol, MaxCol);
  FCharSetMatrix.ClearItems(FCursorRow, FDisplayOriginCol, MaxCol);
  FAttrMatrix.ClearItems(FCursorRow, FDisplayOriginCol, MaxCol);
  FForeColorMatrix.ClearItems(FCursorRow, FDisplayOriginCol, MaxCol);
  FBackColorMatrix.ClearItems(FCursorRow, FDisplayOriginCol, MaxCol);
  tbInvalidateRect(FCursorRow, FDisplayOriginCol,
                   FCursorRow, MaxCol);
end;
{--------}
procedure TApxTerminalBuffer.EraseScreen;
begin
  { WARNING: this DOES NOT use the scroll region, if defined, it blanks }
  {          out everything on the window }

  FBeyondMargin := False;
  { scroll the entire scrollback view by RowCount lines: this will have }
  { the effect of clearing the display view and of setting up the       }
  { scrollback buffer with the previous screen }
  tbScrollRows(RowCount, 0, pred(SVRowCount));
  tbInvalidateRect(SVRowCount - RowCount, 0,
                   pred(SVRowCount), pred(ColCount));
end;
{--------}
procedure TApxTerminalBuffer.EraseToEOL;
var
  MaxCol : integer;
begin
  { WARNING: this uses the scroll region, if defined }

  FBeyondMargin := False;
  { set all characters from and including the cursor position to the }
  { end of the line to blanks }
  MaxCol := FDisplayOriginCol + FDisplayColCount - 1;
  FCharMatrix.ClearItems(FCursorRow, FCursorCol, MaxCol);
  FCharSetMatrix.ClearItems(FCursorRow, FCursorCol, MaxCol);
  FAttrMatrix.ClearItems(FCursorRow, FCursorCol, MaxCol);
  FForeColorMatrix.ClearItems(FCursorRow, FCursorCol, MaxCol);
  FBackColorMatrix.ClearItems(FCursorRow, FCursorCol, MaxCol);
  tbInvalidateRect(FCursorRow, FCursorCol,
                   FCursorRow, MaxCol);
end;
{--------}
procedure TApxTerminalBuffer.EraseToEOW;
begin
  { WARNING: this DOES NOT use the scroll region, if defined, it blanks }
  {          out everything from and including the cursor position up   }
  {          to the end of the screen }

  FBeyondMargin := False;
  { set all characters from and including the cursor position to the }
  { end of the line to blanks }
  FCharMatrix.ClearItems(FCursorRow, FCursorCol, pred(ColCount));
  FCharSetMatrix.ClearItems(FCursorRow, FCursorCol, pred(ColCount));
  FAttrMatrix.ClearItems(FCursorRow, FCursorCol, pred(ColCount));
  FForeColorMatrix.ClearItems(FCursorRow, FCursorCol, pred(ColCount));
  FBackColorMatrix.ClearItems(FCursorRow, FCursorCol, pred(ColCount));
  tbInvalidateRect(FCursorRow, FCursorCol,
                   FCursorRow, pred(ColCount));
  { now erase all succeeding lines, by scrolling them out of existence }
  if (FCursorRow < pred(SVRowCount)) then begin
    tbScrollRows(pred(SVRowCount) - FCursorRow,
                 succ(FCursorRow), pred(SVRowCount));
    tbInvalidateRect(succ(FCursorRow), 0,
                     pred(SVRowCount), pred(ColCount));
  end;
end;
{--------}
procedure TApxTerminalBuffer.GetCharAttrs(var aValue : TApxTerminalCharAttrs);
begin
  aValue := FAttr;
end;
{--------}
procedure TApxTerminalBuffer.GetDefCharAttrs(var aValue : TApxTerminalCharAttrs);
begin
  aValue := FDefAttr;
end;
{--------}
function TApxTerminalBuffer.GetInvalidRect(var aRect : TRect) : Boolean;
begin
  if (FInvRectList = nil) then
    Result := False
  else begin
    { if there is more than one invalid rect, merge them all into one }
    if (PInvRect(FInvRectList)^.irNext <> nil) then
      ApxTMergeInvalidRects(FInvRectList);
    { return the first invalid rect }
    Result := ApxTRemoveInvalidRect(PInvRect(FInvRectList), aRect);
  end;
end;
{--------}
function TApxTerminalBuffer.GetLineAttrPtr(aRow : integer) : pointer;
var
  OurRow : integer;
begin
  { normalize the row number to our internal system }
  OurRow := tbCvtToInternalRow(aRow, True);
  {$IFDEF UseRangeChecks}
  if (OurRow < 0) or
     (OurRow >= FSVRowCount) then
    raise EApxTerminal.Create (AxecTrmBufGetLineAttPtrRange, False);
  {$ENDIF}
  { return the pointer into the matrix }
  Result := FAttrMatrix.GetItemPtr(OurRow, FDisplayOriginCol);
end;
{--------}
function TApxTerminalBuffer.GetLineBackColorPtr(aRow : integer) : pointer;
var
  OurRow : integer;
begin
  { normalize the row number to our internal system }
  OurRow := tbCvtToInternalRow(aRow, True);
  {$IFDEF UseRangeChecks}
  if (OurRow < 0) or
     (OurRow >= FSVRowCount) then
    raise EApxTerminal.Create (AxecTrmBufGetBackColorRng, False);
  {$ENDIF}
  { return the pointer into the matrix }
  Result := FBackColorMatrix.GetItemPtr(OurRow, FDisplayOriginCol);
end;
{--------}
function TApxTerminalBuffer.GetLineCharPtr(aRow : integer) : pointer;
var
  OurRow : integer;
begin
  { normalize the row number to our internal system }
  OurRow := tbCvtToInternalRow(aRow, True);
  {$IFDEF UseRangeChecks}
  if (OurRow < 0) or
     (OurRow >= FSVRowCount) then
    raise EApxTerminal.Create (AxecTrmBufGetLineCharPtrRange, False);
  {$ENDIF}
  { return the pointer into the matrix }
  Result := FCharMatrix.GetItemPtr(OurRow, FDisplayOriginCol)
end;
{--------}
function TApxTerminalBuffer.GetLineCharSetPtr(aRow : integer) : pointer;
var
  OurRow : integer;
begin
  { normalize the row number to our internal system }
  OurRow := tbCvtToInternalRow(aRow, True);
  {$IFDEF UseRangeChecks}
  if (OurRow < 0) or
     (OurRow >= FSVRowCount) then
    raise EApxTerminal.Create (AxecTrmBufGetLineCharSetRange, False);
  {$ENDIF}
  { return the pointer into the matrix }
  Result := FCharSetMatrix.GetItemPtr(OurRow, FDisplayOriginCol)
end;
{--------}
function TApxTerminalBuffer.GetLineForeColorPtr(aRow : integer) : pointer;
var
  OurRow : integer;
begin
  { normalize the row number to our internal system }
  OurRow := tbCvtToInternalRow(aRow, True);
  {$IFDEF UseRangeChecks}
  if (OurRow < 0) or
     (OurRow >= FSVRowCount) then
    raise EApxTerminal.Create (AxecTrmBufGetLineForeColorRng, False);
  {$ENDIF}
  { return the pointer into the matrix }
  Result := FForeColorMatrix.GetItemPtr(OurRow, FDisplayOriginCol)
end;
{--------}
function TApxTerminalBuffer.HasCursorMoved : Boolean;
begin
  { return whether the cursor has moved since the last time this }
  { function was called; reset the internal variable }
  Result := FCursorMoved;
  FCursorMoved := False;
end;
{--------}
function TApxTerminalBuffer.HasDisplayChanged : Boolean;
var
  DummyRect : TRect;
begin
  Result := ApxTPeekInvalidRect(PInvRect(FInvRectList), DummyRect);
end;
{--------}
procedure TApxTerminalBuffer.InsertChars(aCount : integer);
var
  CharCount : integer;
begin
  FBeyondMargin := False;
  {$IFDEF UseRangeCheck}
  if (aCount <= 0) then
    raise EApxTerminal.Create (AxecTrmBufInsertCharsCount, False);
  {$ENDIF}
  { the actual number of characters to delete is constrained by the }
  { current display region }
  CharCount := FDisplayOriginCol + FDisplayColCount - FCursorCol;
  if (CharCount > aCount) then
    CharCount := aCount;
  if (CharCount > 0) then begin
    FCharMatrix.InsertItems(CharCount, FCursorRow, FCursorCol);
    FCharSetMatrix.InsertItems(CharCount, FCursorRow, FCursorCol);
    FAttrMatrix.InsertItems(CharCount, FCursorRow, FCursorCol);
    FForeColorMatrix.InsertItems(CharCount, FCursorRow, FCursorCol);
    FBackColorMatrix.InsertItems(CharCount, FCursorRow, FCursorCol);
    tbInvalidateRect(FCursorRow, FCursorCol,
                     FCursorRow, pred(FCursorCol + CharCount));
  end;
end;
{--------}
procedure TApxTerminalBuffer.InsertLines(aCount : integer);
var
  MaxRow : integer;
begin
  FBeyondMargin := False;
  { inserting lines is equivalent to a scroll down from the current }
  { cursor position; we take account of any scroll region, of course }
  {$IFDEF UseRangeCheck}
  if (aCount <= 0) then
    raise EApxTerminal.Create (AxecTrmBufInsertLinesCount, False);
  {$ENDIF}
  MaxRow := FDisplayOriginRow + FDisplayRowCount - 1;
  tbScrollRows(-aCount, FCursorRow, MaxRow);
  { the cursor position doesn't change as a result of inserting rows }
  tbInvalidateRect(FCursorRow, FDisplayOriginCol,
                   MaxRow, pred(FDisplayOriginCol + FDisplayColCount));
end;
{--------}
procedure TApxTerminalBuffer.MoveCursorDown(aScroll : Boolean);
begin
  FBeyondMargin := False;
  tbMoveCursorUpDown(1, aScroll);
end;
{--------}
procedure TApxTerminalBuffer.MoveCursorLeft(aWrap   : Boolean;
                                           aScroll : Boolean);
begin
  FBeyondMargin := False;
  tbMoveCursorLeftRight(-1, aWrap, aScroll);
end;
{--------}
procedure TApxTerminalBuffer.MoveCursorRight(aWrap   : Boolean;
                                            aScroll : Boolean);
begin
  FBeyondMargin := False;
  tbMoveCursorLeftRight(1, aWrap, aScroll);
end;
{--------}
procedure TApxTerminalBuffer.MoveCursorUp(aScroll : Boolean);
begin
  FBeyondMargin := False;
  tbMoveCursorUpDown(-1, aScroll);
end;
{--------}
procedure TApxTerminalBuffer.Reset;
var
  i : integer;
begin
  { set the attributes, the colors, and the charset }
  FAttr := FDefAttr;
  FForeColor := FDefForeColor;
  FBackColor := FDefBackColor;
  FCharSet := FDefCharSet;

  { set the various matrices to their 'power-up' values }
  if UseWideChars then begin
    FCharMatrix.SetDefaultItem(@FDefWideChar);
  end
  else
  begin
    FCharMatrix.SetDefaultItem(@FDefAnsiChar);
  end;
  FCharSetMatrix.SetDefaultItem(@FDefCharSet);
  FAttrMatrix.SetDefaultItem(@FDefAttr);
  FForeColorMatrix.SetDefaultItem(@FDefForeColor);
  FBackColorMatrix.SetDefaultItem(@FDefBackColor);

  { clear the matrices }
  FCharMatrix.Clear;
  FCharSetMatrix.Clear;
  FAttrMatrix.Clear;
  FForeColorMatrix.Clear;
  FBackColorMatrix.Clear;

  { clear all tab stops, set horizontal ones to every 8 chars }
  ClearAllHorzTabStops;
  ClearAllVertTabStops;
  i := 1;
  while (i < ColCount) do begin
    Col := i;
    SetHorzTabStop;
    inc(i, 8);
  end;

  { set the buffer modes }
  UseAbsAddress := adc_TermBufUseAbsAddress;
  UseAutoWrap := adc_TermBufUseAutoWrap;
  UseAutoWrapDelay := adc_TermBufUseAutoWrapDelay;
  UseInsertMode := adc_TermBufUseInsertMode;
  UseNewLineMode := adc_TermBufUseNewLineMode;
  UseScrollRegion := adc_TermBufUseScrollRegion;

  { reset the cursor position }
  FBeyondMargin := False;
  Row := 1;
  Col := 1;
end;
{--------}
procedure TApxTerminalBuffer.SetCharAttrs(const aValue : TApxTerminalCharAttrs);
begin
  FAttr := aValue;
  FAttrMatrix.SetDefaultItem(@FAttr);
end;
{--------}
procedure TApxTerminalBuffer.SetDefCharAttrs(const aValue : TApxTerminalCharAttrs);
begin
  FDefAttr := aValue;
end;
{--------}
procedure TApxTerminalBuffer.SetCursorPosition(aRow, aCol : integer);
begin
  FBeyondMargin := False;
  Row := aRow;
  Col := aCol;
end;
{--------}
procedure TApxTerminalBuffer.SetScrollRegion(aTopRow, aBottomRow : integer);
var
  Temp : integer;
begin
  FBeyondMargin := False;
  { if the top row is greater than the bottom row, they're out of }
  { order, so switch 'em round }
  if (aTopRow > aBottomRow) then begin
    Temp := aTopRow;
    aTopRow := aBottomRow;
    aBottomRow := Temp;
  end;
  {$IFDEF UseRangeChecks}
  if ((aBottomRow - aTopRow) < 1) or
     (aTopRow < 1) or (aTopRow > RowCount) or
     (aBottomRow < 1) or (aBottomRow > RowCount) then
    raise EApxTerminal.Create (AxecTrmBufScrollRgnRowNum, False);
  {$ENDIF}
  FSRStartRow := tbCvtToInternalRow(aTopRow, True);
  FSREndRow := tbCvtToInternalRow(aBottomRow, True);
  { force the scroll region to be used }
  if UseScrollRegion then
    UseScrollRegion := False;
  if (aTopRow <> 1) or (aBottomRow <> RowCount) then
    UseScrollRegion := True;
end;
{--------}
procedure TApxTerminalBuffer.SetHorzTabStop;
begin
  if (ColCount <> 0) then
    ApxTSetBit(FHorzTabStops, FCursorCol);
end;
{--------}
procedure TApxTerminalBuffer.SetVertTabStop;
begin
  if (RowCount <> 0) then
    ApxTSetBit(FVertTabStops, FCursorRow);
end;
{--------}
function TApxTerminalBuffer.tbAtLastColumn : Boolean;
var
  MaxCol : integer;
begin
  MaxCol := FDisplayOriginCol + FDisplayColCount - 1;
  Result := FCursorCol = MaxCol;
end;
{--------}
function TApxTerminalBuffer.tbCvtToExternalCol(aCol : integer;
                                              aAbsolute : Boolean) : integer;
begin
  { aCol is an internal reference (ie. zero-based and absolute), we     }
  { need the external value (ie, one-based and relative to the start of }
  { the addressable area) }
  if aAbsolute then
    Result := aCol + 1
  else
    Result := aCol - FDisplayOriginCol + 1;
end;
{--------}
function TApxTerminalBuffer.tbCvtToExternalRow(aRow : integer;
                                              aAbsolute : Boolean) : integer;
begin
  { aRow is an internal reference (ie. zero-based and absolute), we     }
  { need the external value (ie, one-based and relative to the start of }
  { the addressable area) }
  if aAbsolute then
    Result := aRow - (SVRowCount - RowCount) + 1
  else
    Result := aRow - FDisplayOriginRow + 1;
end;
{--------}
function TApxTerminalBuffer.tbCvtToInternalCol(aCol : integer;
                                              aAbsolute : Boolean) : integer;
begin
  { aCol is an external reference (ie, one-based and relative to the }
  { start of the addressable area), we need the internal value (ie.  }
  { zero-based and absolute) }
  if aAbsolute then
    Result := aCol - 1
  else
    Result := aCol - 1 + FDisplayOriginCol;
end;
{--------}
function TApxTerminalBuffer.tbCvtToInternalRow(aRow : integer;
                                              aAbsolute : Boolean) : integer;
begin
  { aRow is an external reference (ie, one-based and relative to the }
  { start of the addressable area), we need the internal value (ie.  }
  { zero-based and absolute) }
  if aAbsolute then
    Result := aRow - 1 + (SVRowCount - RowCount)
  else
    Result := aRow - 1 + FDisplayOriginRow;
end;
{--------}
procedure TApxTerminalBuffer.tbFireOnCursorMovedEvent;
begin
  if (assigned (FOnCursorMoved)) and (FCursorMoved) then
    FOnCursorMoved (Self, Row, Col);
end;
{--------}
function TApxTerminalBuffer.tbGetCol : integer;
begin
  Result := tbCvtToExternalCol(FCursorCol, UseAbsAddress);
end;
{--------}
function TApxTerminalBuffer.tbGetOriginCol : integer;
begin
  Result := tbCvtToExternalCol(FDisplayOriginCol, True);
end;
{--------}
function TApxTerminalBuffer.tbGetOriginRow : integer;
begin
  Result := tbCvtToExternalRow(FDisplayOriginRow, True);
end;
{--------}
function TApxTerminalBuffer.tbGetRow : integer;
begin
  Result := tbCvtToExternalRow(FCursorRow, UseAbsAddress);
end;
{--------}
procedure TApxTerminalBuffer.tbInvalidateRect(aFromRow, aFromCol,
                                             aToRow, aToCol : integer);
var
  OurRect : TRect;
begin
  { convert the row and column values to external values }
  OurRect.Left := aFromCol + 1;
  OurRect.Top := aFromRow + RowCount - SVRowCount + 1;
  OurRect.Right := aToCol + 1;
  OurRect.Bottom := aToRow + RowCount - SVRowCount + 1;
  ApxTApxdInvalidRect(PInvRect(FInvRectList), OurRect);
end;
{--------}
procedure TApxTerminalBuffer.tbMoveCursorLeftRight(aDirection : integer;
                                                  aWrap      : Boolean;
                                                  aScroll    : Boolean);
var
  MaxCol   : integer;
  MaxRow   : integer;
  StartRow : integer;
begin
  { if wrap is off, we just advance or retard the cursor position by    }
  { one without extending beyond the left or right margin; scrolling is }
  { not possible in this case }
  MaxCol := FDisplayOriginCol + FDisplayColCount - 1;
  if not aWrap then begin
    if (aDirection < 0) then begin
      if (FCursorCol > FDisplayOriginCol) then begin
        FCursorMoved := True;
        dec(FCursorCol);
        tbFireOnCursorMovedEvent;
      end;
    end
    else begin
      if (FCursorCol < MaxCol) then begin
        FCursorMoved := True;
        inc(FCursorCol);
        tbFireOnCursorMovedEvent;                                                    
      end;
    end;
    Exit;
  end;
  MaxRow := FDisplayOriginRow + FDisplayRowCount - 1;
  { otherwise it's a wrap, as it were }
  { there are several cases here      }
  {  1. if the new cursor position is within the same line, just move  }
  {     it as above                                                    }
  {  2. if the current cursor position is 0 and we're moving left...   }
  {     a. if we're not on the top display row, move the cursor to     }
  {        the last column of the previous row                         }
  {     b. if aScroll is False and we're on the top row, leave         }
  {        the cursor where it is                                      }
  {     c. if aScroll is True and we're on the top row, scroll the     }
  {        display down and move the cursor to the last column of the  }
  {        new top row                                                 }
  {  3. if the current cursor position is on the last column and we're }
  {     moving right...                                                }
  {     a. if we're not on the bottom display row, move the cursor to  }
  {        the first column of the next row                            }
  {     b. if aScroll is False and we're on the bottom row, leave the  }
  {        cursor where it is                                          }
  {     c. if aScroll is True and we're on the bottom, scroll the      }
  {        display up and move the cursor to the first column of the   }
  {        new bottom row }
  if (aDirection < 0) then begin { moving left }
    if (FCursorCol > FDisplayOriginCol) then begin
      FCursorMoved := True;
      dec(FCursorCol);
      tbFireOnCursorMovedEvent;
    end
    else if (FCursorRow > FDisplayOriginRow) then begin
      FCursorMoved := True;
      FCursorCol := MaxCol;
      dec(FCursorRow);
      tbFireOnCursorMovedEvent;
    end
    else if aScroll then begin
      tbScrollRows(-1, FDisplayOriginRow, MaxRow);
      tbInvalidateRect(FDisplayOriginRow, 0, MaxRow, pred(ColCount));
      FCursorMoved := True;
      FCursorCol := MaxCol;
      tbFireOnCursorMovedEvent;                                   
    end;
  end
  else { Direction > 0 } begin { moving right }
    if (FCursorCol < MaxCol) then begin
      FCursorMoved := True;
      inc(FCursorCol);
      tbFireOnCursorMovedEvent;
    end
    else if (FCursorRow < MaxRow) then begin
      FCursorMoved := True;
      FCursorCol := FDisplayOriginCol;
      inc(FCursorRow);
      tbFireOnCursorMovedEvent;
    end
    else if aScroll then begin
      if UseScrollRegion then
        StartRow := FDisplayOriginRow
      else
        StartRow := 0;
      tbScrollRows(1, StartRow, MaxRow);
      tbInvalidateRect(FDisplayOriginRow, 0, MaxRow, pred(ColCount));
      FCursorMoved := True;
      FCursorCol := FDisplayOriginCol;
      tbFireOnCursorMovedEvent;
    end;
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbMoveCursorUpDown(aDirection : integer;
                                               aScroll    : Boolean);
var
  MaxRow   : integer;
  StartRow : integer;
begin
  MaxRow := FDisplayOriginRow + FDisplayRowCount - 1;
  if (aDirection < 0) then begin
    if (FCursorRow > FDisplayOriginRow) then begin
      FCursorMoved := True;
      dec(FCursorRow);
      tbFireOnCursorMovedEvent;                                   
    end
    else if aScroll then begin
      tbScrollRows(-1, FDisplayOriginRow, MaxRow);
      tbInvalidateRect(FDisplayOriginRow, 0, MaxRow, pred(ColCount));
    end;
  end
  else { Direction > 0 } begin
    if (FCursorRow < MaxRow) then begin
      FCursorMoved := True;
      inc(FCursorRow);
      tbFireOnCursorMovedEvent;                                  
    end
    else if aScroll then begin
      if UseScrollRegion then
        StartRow := FDisplayOriginRow
      else
        StartRow := 0;
      tbScrollRows(1, StartRow, MaxRow);
      tbInvalidateRect(FDisplayOriginRow, 0, MaxRow, pred(ColCount));
      
    end;
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbReallocBuffers(aNewRowCount : integer;
                                             aNewColCount : integer);
begin
  {check for changes in row count}
  if (aNewRowCount <> SVRowCount) then begin
    FCharMatrix.RowCount := aNewRowCount;
    FCharSetMatrix.RowCount := aNewRowCount;
    FAttrMatrix.RowCount := aNewRowCount;
    FForeColorMatrix.RowCount := aNewRowCount;
    FBackColorMatrix.RowCount := aNewRowCount;
    FSVRowCount := aNewRowCount;
  end
  { otherwise it's a change in column count }
  else begin
    FCharMatrix.ColCount := aNewColCount;
    FCharSetMatrix.ColCount := aNewColCount;
    FAttrMatrix.ColCount := aNewColCount;
    FForeColorMatrix.ColCount := aNewColCount;
    FBackColorMatrix.ColCount := aNewColCount;
    FColCount := aNewColCount;
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbScrollRows(aCount, aTop, aBottom : integer);
begin
  FCharMatrix.ScrollRows(aCount, aTop, aBottom);
  FCharSetMatrix.ScrollRows(aCount, aTop, aBottom);
  FAttrMatrix.ScrollRows(aCount, aTop, aBottom);
  FForeColorMatrix.ScrollRows(aCount, aTop, aBottom);
  FBackColorMatrix.ScrollRows(aCount, aTop, aBottom);
  if Assigned(FOnScrollRows) then
    FOnScrollRows(Self, aCount,
                  aTop + 1 - (SVRowCount - RowCount),
                  aBottom + 1 - (SVRowCount - RowCount));
end;
{--------}
procedure TApxTerminalBuffer.tbSetBackColor(aValue : TColor);
begin
  if (aValue <> BackColor) then begin
    FBackColor := aValue;
    FBackColorMatrix.SetDefaultItem(@FBackColor);
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbSetCharSet(aValue : byte);
begin
  if (aValue <> CharSet) then begin
    FCharSet := aValue;
    FCharSetMatrix.SetDefaultItem(@FCharSet);
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbSetCol(aCol : integer);
var
  OurCol : integer;
begin
  FBeyondMargin := False;
  if (aCol <> Col) then begin
    OurCol := tbCvtToInternalCol(aCol, UseAbsAddress);
    if (OurCol < 0) then
      OurCol := 0
    else if (OurCol >= ColCount) then
      OurCol := pred(ColCount);
    FCursorMoved := True;
    FCursorCol := OurCol;
    tbFireOnCursorMovedEvent;
  end;
end;


{--------}
procedure TApxTerminalBuffer.tbSetColCount(aNewCount : integer);
begin
  { only do something if the value changes }
  if (aNewCount <> ColCount) then begin
    { check the new value is sensible }
    if (aNewCount < 2) then
      raise EApxTerminal.Create (AxecTrmBufColCountTooSmall, False);
    { reallocate the tab positions bitset }
    FHorzTabStops :=
       ApxTReallocBitset(FHorzTabStops, ColCount, aNewCount);
    { if the number of scrollback rows is zero, just make a note of the }
    { new value: there won't have been any allocations yet }
    if (SVRowCount = 0) then begin
      FColCount := aNewCount;
      FDisplayColCount  := aNewCount;
    end
    { otherwise we can allocate new buffers and transfer over the old }
    else begin
      tbReallocBuffers(SVRowCount, aNewCount);
      tbInvalidateRect(SVRowCount - RowCount, 0,
                       pred(SVRowCount), pred(aNewCount));
      if UseScrollRegion then
        UseScrollRegion := False
      else begin
        FCursorRow := SVRowCount - RowCount;
        FCursorCol := 0;
        FDisplayColCount  := aNewCount;
      end;
    end;
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbSetDefAnsiChar(aValue : AnsiChar);
begin
  if (aValue <> DefAnsiChar) then begin
    FDefAnsiChar := aValue;
    FCharMatrix.SetDefaultItem(@FDefAnsiChar);
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbSetDefBackColor(aValue : TColor);

begin
  if (FDefBackColor <> aValue) then begin
    FBackColorMatrix.ReplaceItems(@FDefBackColor, @aValue);
    FDefBackColor := aValue;
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbSetDefForeColor(aValue : TColor);

begin
  if (FDefForeColor <> aValue) then begin
    FForeColorMatrix.ReplaceItems(@FDefForeColor, @aValue);
    FDefForeColor := aValue;
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbSetForeColor(aValue : TColor);
begin
  if (aValue <> ForeColor) then begin
    FForeColor := aValue;
    FForeColorMatrix.SetDefaultItem(@FForeColor);
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbSetRow(aRow : integer);
var
  OurRow : integer;
begin
  FBeyondMargin := False;
  if (aRow <> Row) then begin
    OurRow := tbCvtToInternalRow(aRow, UseAbsAddress);
    if (OurRow < 0) then
      OurRow := 0
    else if (OurRow >= SVRowCount) then
      OurRow := pred(SVRowCount);
    FCursorMoved := True;
    FCursorRow := OurRow;
      tbFireOnCursorMovedEvent;
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbSetRowCount(aNewCount : integer);
begin
  { only do something if the value changes, and changes to something }
  { not greater than the scrollback view count }
  if (aNewCount <> RowCount) and
     (aNewCount <= SVRowCount) then begin
    { check the new value is sensible }
    if (aNewCount < 2) then
      raise EApxTerminal.Create (AxecTrmBufSetRowCountTooSmall, False);
    { changing the row count resets the scroll region }
    if UseScrollRegion then
      UseScrollRegion := False;
    { reallocate the tab positions bitset }
    FVertTabStops :=
       ApxTReallocBitset(FVertTabStops, RowCount, aNewCount);
    { set the display origin and the cursor position }
    FDisplayOriginRow := FSVRowCount - aNewCount;
    if (FCursorRow < FDisplayOriginRow) then
      FCursorRow := FDisplayOriginRow;
    { save the new value }
    FRowCount := aNewCount;
    FDisplayRowCount := aNewCount;
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbSetSVRowCount(aNewCount : integer);
begin
  { only do something if the value changes }
  if (aNewCount <> SVRowCount) then begin
    { check the new value is sensible }
    if (aNewCount < 2) then
      raise EApxTerminal.Create (AxecTrmBufSetSVRowCountToSmall, False);
    { if the new scrollback view count is less than the display view }
    { count, reduce the display view count to match }
    if (aNewCount < RowCount) then begin
      FRowCount := aNewCount;
      FDisplayRowCount := aNewCount;
    end;
    { set the display origin and the cursor position }
    FDisplayOriginRow := aNewCount - RowCount;
    FCursorRow := FCursorRow - SVRowCount + aNewCount;
    { if the number of columns is zero, just make a note of the new }
    { value: there won't have been any allocations yet }
    if (ColCount = 0) then
      FSVRowCount := aNewCount
    { otherwise we can allocate new buffers and transfer over the old }
    else begin
      tbReallocBuffers(aNewCount, ColCount);
      tbInvalidateRect(SVRowCount - RowCount, 0,
                       pred(SVRowCount), pred(aNewCount));
    end;
  end;
end;
{--------}
procedure TApxTerminalBuffer.tbSetUseScrollRegion(aValue : Boolean);
begin
  if (aValue <> UseScrollRegion) then begin
    { calculate the limits beyond which the cursor cannot move }
    if aValue { limit to scroll region } then begin
      FDisplayOriginCol := 0;
      FDisplayOriginRow := FSRStartRow;
      FDisplayColCount  := ColCount;
      FDisplayRowCount  := FSREndRow - FSRStartRow + 1;
    end
    else { limit to full display area } begin
      FDisplayOriginCol := 0;
      FDisplayOriginRow := SVRowCount - RowCount;
      FDisplayColCount  := ColCount;
      FDisplayRowCount  := RowCount;
    end;
    { rest the cursor to the top left corner of the allowed region }
    FCursorMoved := True;
    FCursorCol := FDisplayOriginCol;
    FCursorRow := FDisplayOriginRow;
    tbFireOnCursorMovedEvent;
    { save the property value }
    FUseScrollRegion := aValue;
  end;
end;
{--------}
procedure TApxTerminalBuffer.WriteChar(aCh : char);
begin
  { this is performed as }
  {  - write the character to the current cursor                       }
  {  - write the current attributes, colors and charset to the current }
  {    cursor                                                          }
  {  - advance the cursor                                              }
  { the latter operation may not do anything if UseAutoWrap is off and }
  { the cursor is at the end of the line, otherwise, if it's on, a     }
  { scroll will occur }
  if FBeyondMargin then begin
    MoveCursorRight(UseAutoWrap, False);
    FBeyondMargin := False;
  end;
  if UseInsertMode then begin
    FCharMatrix.InsertItems(1, FCursorRow, FCursorCol);
    FCharSetMatrix.InsertItems(1, FCursorRow, FCursorCol);
    FAttrMatrix.InsertItems(1, FCursorRow, FCursorCol);
    FForeColorMatrix.InsertItems(1, FCursorRow, FCursorCol);
    FBackColorMatrix.InsertItems(1, FCursorRow, FCursorCol);
  end;
  FCharMatrix.WriteItems(@aCh, 1, FCursorRow, FCursorCol);
  FCharSetMatrix.WriteItems(@FCharSet, 1, FCursorRow, FCursorCol);
  FAttrMatrix.WriteItems(@FAttr, 1, FCursorRow, FCursorCol);
  FForeColorMatrix.WriteItems(@FForeColor, 1, FCursorRow, FCursorCol);
  FBackColorMatrix.WriteItems(@FBackColor, 1, FCursorRow, FCursorCol);
  tbInvalidateRect(FCursorRow, FCursorCol,
                   FCursorRow, FCursorCol);
  if (not UseAutoWrapDelay) or (not tbAtLastColumn) then
    MoveCursorRight(UseAutoWrap, False)
  else
    FBeyondMargin := True;
end;
{--------}
procedure TApxTerminalBuffer.WriteString(const aSt : string);
var
  i : integer;
begin
  for i := 1 to length(aSt) do begin
    WriteChar(aSt[i]);
  end;
end;
{====================================================================}

const
  { The hash table sizes: these are prime numbers that suit these }
  { particular implementations }
  KBHashTableSize = 57;    { keyboard mapping hash table size }
  CSHashTableSize = 199;   { charset mapping hash table size }

  OurSignature : longint = $33544841;
    {Note: $33544841 = AHT3 = APRO Hash Table, version 3}


type
  PKBHashNode = ^TKBHashNode;   { hash table node for keyboard maps }
  TKBHashNode = packed record
    kbnNext  : PKBHashNode;
    kbnKey   : PadKeyString;
    kbnValue : PadKeyString;
  end;

type
  PCSHashNode = ^TCSHashNode;   { hash table node for charset maps }
  TCSHashNode = packed record
    csnNext    : PCSHashNode;
    csnCharSet : PadKeyString;
    csnFont    : PadKeyString;
    csnChar    : AnsiChar;
    csnGlyph   : AnsiChar;
  end;

  PScriptNode = ^TScriptNode;
  TScriptNode = packed record
    snNext : PScriptNode;
    snFont : PadKeyString;
    snText : PAnsiChar;
  end;



{===TCharQueue=======================================================}
const
  CharQueueDelta = 32;
type
  TCharQueue = class
    private
      FSize : longint;
      FLen  : longint;
      FText : PAnsiChar;
    protected
      function cqGetDupText : PAnsiChar;
    public
      constructor Create;
      destructor Destroy; override;

      procedure Add(aCh : AnsiChar);
      procedure Clear;

      property DupText : PAnsiChar read cqGetDupText;
  end;
{--------}
constructor TCharQueue.Create;
begin
  inherited Create;
  { allocate a starter character queue }
  GetMem(FText, CharQueueDelta);
  FSize := CharQueueDelta;
  FText[0] := #0;
end;
{--------}
destructor TCharQueue.Destroy;
begin
  if (FText <> nil) then
    FreeMem(FText, FSize);
  inherited Destroy;
end;
{--------}
procedure TCharQueue.Add(aCh : AnsiChar);
var
  NewQ : PAnsiChar;
begin
  if (FLen = FSize-1) then begin
    GetMem(NewQ, FSize + CharQueueDelta);
    StrCopy(NewQ, FText);
    FreeMem(FText, FSize);
    inc(FSize, CharQueueDelta);
    FText := NewQ;
  end;
  FText[FLen] := aCh;
  inc(FLen);
  FText[FLen] := #0;
end;
{--------}
procedure TCharQueue.Clear;
begin
  FLen := 0;
  FText[0] := #0;
end;
{--------}
function TCharQueue.cqGetDupText : PAnsiChar;
begin
  GetMem(Result, FLen+1);
  StrCopy(Result, FText);
end;
{====================================================================}

{===Helper routines==================================================}
{ Note: The ELF hash functions are described in "Practical Algorithms }
{       For Programmers" by Andrew Binstock and John Rex, Addison     }
{       Wesley, with modifications in Dr Dobbs Journal, April 1996.   }
{       They're modified to suit this implementation.                 }
function HashELF(const S : TApxKeyString) : longint;
var
  G : longint;
  i : integer;
begin
  Result := 0;
  for i := 1 to length(S) do begin
    Result := (Result shl 4) + ord(S[i]);
    G := Result and longint($F0000000);
    if (G <> 0) then
      Result := Result xor (G shr 24);
    Result := Result and (not G);
  end;
end;
{--------}
function HashELFPlusChar(const S : TApxKeyString;
                               C : AnsiChar) : longint;
var
  G : longint;
  i : integer;
begin
  Result := ord(C);
  G := Result and longint($F0000000);
  if (G <> 0) then
    Result := Result xor (G shr 24);
  Result := Result and (not G);
  for i := 1 to length(S) do begin
    Result := (Result shl 4) + ord(S[i]);
    G := Result and longint($F0000000);
    if (G <> 0) then
      Result := Result xor (G shr 24);
    Result := Result and (not G);
  end;
end;
{--------}
function AllocKeyString(const aSt : TApxKeyString) : PadKeyString;
begin
  GetMem(Result, succ(length(aSt)));
  Result^ := aSt;
end;
{--------}
procedure FreeKeyString(aKS : PadKeyString);
begin
  if (aKS <> nil) then
    FreeMem(aKS, succ(length(aKS^)));
end;
{--------}
function ProcessCharSetLine(const aLine : ShortString;
                              var aCharSet : TApxKeyString;
                              var aFromCh  : AnsiChar;
                              var aToCh    : AnsiChar;
                              var aFontName: TApxKeyString;
                              var aGlyph   : AnsiChar) : Boolean;
var
  InWord    : Boolean;
  CharInx   : integer;
  StartCh   : integer;
  QuoteMark : AnsiChar;
  WordStart : array [0..4] of integer;
  WordEnd   : array [0..4] of integer;
  WordCount : integer;
  WordLen   : integer;
  Chars     : array [0..4] of AnsiChar;
  i         : integer;
  AsciiCh   : integer;
  ec        : integer;
  TestSt    : string[3];
begin
  { assumption: the line has had trailing spaces stripped, the line is }
  { not the empty string, the line starts with a ' ' character         }

  { assume we'll fail to parse the line properly }
  Result := False;

  { extract out the 5 words; if there are not at least 5 words, exit }
  QuoteMark := ' '; { needed to fool the compiler }
  StartCh := 0;     { needed to fool the compiler }
  InWord := False;
  WordCount := 0;
  CharInx := 1;
  while CharInx <= length(aLine) do begin
    if InWord then begin
      if (QuoteMark = ' ') then begin
        if (aLine[CharInx] = ' ') then begin
          InWord := False;
          WordStart[WordCount] := StartCh;
          WordEnd[WordCount] := pred(CharInx);
          inc(WordCount);
          if (WordCount = 5) then
            Break;
        end
      end
      else { the quotemark is active } begin
        if (aLine[CharInx] = QuoteMark) then
          QuoteMark := ' ';
      end;
    end
    else { not in a word } begin
      if (aLine[CharInx] <> ' ') then begin
        InWord := True;
        StartCh := CharInx;
        QuoteMark := aLine[CharInx];
        if (QuoteMark <> '''') and (QuoteMark <> '"') then
          QuoteMark := ' ';
      end;
    end;
    inc(CharInx);
  end;
  { when we reach this point we know where the last word ended }
  if InWord then begin
    if (QuoteMark <> ' ') then
      Exit; { the last word had no close quote }
    WordStart[WordCount] := StartCh;
    WordEnd[WordCount] := pred(CharInx);
    inc(WordCount);
  end;
  if (WordCount <> 5) then
    Exit;
  { fix the quoted strings }
  for i := 0 to 4 do begin
    if (aLine[WordStart[i]] = '''') or
       (aLine[WordStart[i]] = '"') then begin
      inc(WordStart[i]);
      dec(WordEnd[i]);
      if (WordEnd[i] < WordStart[i]) then
        Exit; { the word was either '' or "" }
    end;
  end;
  { we now know where each word can be found; the only special words }
  { are words 1, 2, and 4 which must be single characters, or ASCII  }
  { values of the form \xnn }
  for i := 1 to 4 do
    if (i <> 3) then begin
      WordLen := succ(WordEnd[i] - WordStart[i]);
      if (WordLen = 1) then
        Chars[i] := aLine[WordStart[i]]
      else if (WordLen = 4) then begin
        CharInx := WordStart[i];
        if (aLine[CharInx] <> '\') or
           (aLine[CharInx+1] <> 'x') then
          Exit;
        TestSt := Copy(aLine, CharInx+1, 3);
        TestSt[1]:= '$';
        Val(TestSt, AsciiCh, ec);
        if (ec <> 0) then
          Exit;
        Chars[i] := chr(AsciiCh);
      end
      else
        Exit; { unknown format }
    end;
  { return values }
  aFromCh := Chars[1];
  aToCh := Chars[2];
  aGlyph := Chars[4];
  aCharSet := Copy(aLine, WordStart[0], succ(WordEnd[0] - WordStart[0]));
  aFontName := Copy(aLine, WordStart[3], succ(WordEnd[3] - WordStart[3]));
  Result := True;
end;
{====================================================================}


{===TApxKeyboardMapping==================================================}
constructor TApxKeyboardMapping.Create;
begin
  inherited Create;
  FTable := TList.Create;
  FTable.Count := KBHashTableSize;
end;
{--------}
destructor TApxKeyboardMapping.Destroy;
begin
  if (FTable <> nil) then begin
    Clear;
    FTable.Destroy;
  end;
  inherited Destroy;
end;
{--------}
function TApxKeyboardMapping.Add(const aKey   : TApxKeyString;
                             const aValue : TApxKeyString) : Boolean;
var
  Inx  : integer;
  Node : PKBHashNode;
begin
  if kbmFindPrim(aKey, Inx, pointer(Node)) then
    Result := False
  else begin
    Result := True;
    New(Node);
    Node^.kbnNext := FTable[Inx];
    Node^.kbnKey := AllocKeyString(aKey);
    Node^.kbnValue := AllocKeyString(aValue);
    FTable[Inx] := Node;
    inc(FCount);
  end;
end;
{--------}
procedure TApxKeyboardMapping.Clear;
var
  i    : integer;
  Node : PKBHashNode;
  Temp : PKBHashNode;
begin
  for i := 0 to pred(KBHashTableSize) do begin
    Node := FTable[i];
    while (Node <> nil) do begin
      Temp := Node;
      Node := Node^.kbnNext;
      FreeKeyString(Temp^.kbnKey);
      FreeKeyString(Temp^.kbnValue);
      Dispose(Temp);
    end;
    FTable[i] := nil;
  end;
  FCount := 0;
end;
{--------}
{$IFDEF CompileDebugCode}
procedure TApxKeyboardMapping.DebugPrint(const aFileName : string);
var
  F    : text;
  i    : integer;
  Node : PKBHashNode;
begin
  System.Assign(F, aFileName);
  System.Rewrite(F);

  for i := 0 to pred(KBHashTableSize) do begin
    writeln(F, '---', i, '---');
    Node := FTable[i];
    while (Node <> nil) do begin
      writeln(F, Node^.kbnKey^:20, Node^.kbnValue^:20);
      Node := Node^.kbnNext;
    end;
  end;

  writeln(F);
  writeln(F, 'Count: ', Count, ' (mean: ', Count/CSHashTableSize:5:3, ')');

  System.Close(F);
end;
{$ENDIF}
{--------}
function TApxKeyboardMapping.Get(const aKey : TApxKeyString) : TApxKeyString;
var
  Inx  : integer;
  Node : PKBHashNode;
begin
  if kbmFindPrim(aKey, Inx, pointer(Node)) then
    Result := Node^.kbnValue^
  else
    Result := '';
end;
{--------}
function TApxKeyboardMapping.kbmFindPrim(const aKey  : TApxKeyString;
                                          var aInx  : integer;
                                          var aNode : pointer) : Boolean;
var
  Node : PKBHashNode;
begin
  { assume we won't find aKey }
  Result := False;
  aNode := nil;
  { calculate the index, ie hash, of the key }
  aInx := HashELF(aKey) mod KBHashTableSize;
  { traverse the linked list at this entry, looking for the key in each }
  { node we encounter--a case-sensitive comparison }
  Node := FTable[aInx];
  while (Node <> nil) do begin
    if (aKey = Node^.kbnKey^) then begin
      Result := True;
      aNode := Node;
      Exit;
    end;
    Node := Node^.kbnNext;
  end;
end;
{--------}
procedure TApxKeyboardMapping.LoadFromFile(const aFileName : string);
var
  Lines     : TStringList;
  ActualLen : integer;
  i         : integer;
  LineInx   : integer;
  Word1Start: integer;
  Word1End  : integer;
  Word2Start: integer;
  Word2End  : integer;
  LookingForStart : Boolean;
  Line      : string[255];
begin
  { clear the hash table, ready for loading }
  Clear;
  { create the stringlist to hold the keymap script }
  Lines := TStringList.Create;
  try
    { load the keymap script }
    Lines.LoadFromFile(aFileName);
    for LineInx := 0 to pred(Lines.Count) do begin
      { get this line }
      Line := Lines[LineInx];
      { remove trailing spaces }
      ActualLen := length(Line);
      for i := ActualLen downto 1 do
        if (Line[i] = ' ') then
          dec(ActualLen)
        else
          Break;
      Line[0] := chr(ActualLen);
      { only process detail lines }
      if (Line <> '') and (Line[1] <> '*') then begin
        { identify the first 'word' }
        Word1Start := 0;
        Word1End := 0;
        LookingForStart := True;
        for i := 1 to ActualLen do begin
          if LookingForStart then begin
            if (Line[i] <> ' ') then begin
              Word1Start := i;
              LookingForStart := False;
            end;
          end
          else { looking for end } begin
            if (Line[i] = ' ') then begin
              Word1End := i - 1;
              Break;
            end;
          end;
        end;
        { if we've set Word1End then there are at least two words in  }
        { the line, otherwise there was only one word (which we shall }
        { ignore) }
        if (Word1End <> 0) then begin
          { identify the second 'word' }
          Word2Start := 0;
          Word2End := 0;
          LookingForStart := True;
          for i := succ(Word1End) to ActualLen do begin
            if LookingForStart then begin
              if (Line[i] <> ' ') then begin
                Word2Start := i;
                LookingForStart := False;
              end;
            end
            else { looking for end } begin
              if (Line[i] = ' ') then begin
                Word2End := i - 1;
                Break;
              end;
            end;
          end;
          if (Word2End = 0) then
            Word2End := ActualLen;
          { add the key and value to the hash table }
          Add(System.Copy(Line, Word1Start, succ(Word1End-Word1Start)),
              System.Copy(Line, Word2Start, succ(Word2End-Word2Start)));
        end;
      end;
    end;
  finally
    Lines.Free;
  end;
end;
{--------}
procedure TApxKeyboardMapping.LoadFromRes(aInstance : THandle;
                                const aResName  : string);
var
  MS        : TMemoryStream;
  ResInfo   : TResourceHandle;
  ResHandle : THandle;
  ResNameZ  : PAnsiChar;
  Res       : PByteArray; 
  i         : integer;
  Sig       : longint;
  ResCount  : longint;
  BytesRead : longint;
  Key       : TApxKeyString;
  Value     : TApxKeyString;
begin
  { Note: this code has been written to work with all versions of     }
  { Delphi, both 16-bit and 32-bit. Hence it does not make use of any }
  { of the features available in later compilers, ie, typecasting a   }
  { string to a PAnsiChar, or TResourceStream) }

  { clear the hash table, ready for loading }
  Clear;
  { get the resource info handle }
  GetMem (ResNameZ, succ (length (aResName)));
  try
    StrPCopy(ResNameZ, aResName);
    ResInfo := FindResource (Cardinal (aInstance), ResNameZ, RT_RCDATA);
  finally
    FreeMem(ResNameZ, succ (length (aResName)));
  end;
  if (ResInfo = 0) then
    Exit;
  { load and lock the resource }
  ResHandle := System.LoadResource (Cardinal (aInstance), ResInfo);
  Res := LockResource (ResHandle);
  if (ResHandle = 0) then
    Exit;
  try
    { create a memory stream }
    MS := TMemoryStream.Create;
    try
      { copy the resource to our memory stream }
      MS.Write(Res^, SizeOfResource (aInstance, ResInfo));
      MS.Position := 0;
      { read the header signature, get out if it's not ours }
      BytesRead := MS.Read (Sig, sizeof (Sig));
      if (BytesRead <> sizeof (Sig)) or (Sig <> OurSignature) then
        Exit;
      { read the count of key/value string pairs in the resource }
      MS.Read (ResCount, sizeof (ResCount));
      { read that number of key/value string pairs and add them to the hash table }
      for i := 0 to pred (ResCount) do begin
        MS.Read (Key[0], 1);
        MS.Read (Key[1], ord (Key[0]));
        MS.Read (Value[0], 1);
        MS.Read (Value[1], ord(Value[0]));
        Add (Key, Value);
      end;
      { read the footer signature, clear and get out if it's not ours }
      BytesRead := MS.Read (Sig, sizeof(Sig));
      if (BytesRead <> sizeof (Sig)) or (Sig <> OurSignature) then begin
        Clear;
        Exit;
      end;
    finally
      MS.Free;
    end;
  finally
    FreeResource(ResHandle);
  end;
end;
{--------}
procedure TApxKeyboardMapping.StoreToBinFile(const aFileName : string);
var
  aFS  : TFileStream;
  i    : integer;
  Node : PKBHashNode;
begin
  { create a file stream }
  aFS := TFileStream.Create(aFileName, fmCreate);
  try
    { write our signature as header }
    aFS.Write(OurSignature, sizeof(OurSignature));
    { write the number of key/value string pairs }
    aFS.Write(FCount, sizeof(FCount));
    { write all the key/value string pairs }
    for i := 0 to pred(KBHashTableSize) do begin
      Node := FTable[i];
      while (Node <> nil) do begin
        aFS.Write(Node^.kbnKey^, succ(length(Node^.kbnKey^)));
        aFS.Write(Node^.kbnValue^, succ(length(Node^.kbnValue^)));
        Node := Node^.kbnNext;
      end;
    end;
    { write our signature as footer as a check }
    aFS.Write(OurSignature, sizeof(OurSignature));
  finally
    aFS.Free;
  end;
end;
{====================================================================}


{===TApxCharSetMapping================================================}
constructor TApxCharSetMapping.Create;
begin
  inherited Create;
  FTable := TList.Create;
  FTable.Count := CSHashTableSize;
  FCharQueue := pointer(TCharQueue.Create);
end;
{--------}
destructor TApxCharSetMapping.Destroy;
var
  Temp, Walker : PScriptNode;
begin
  { free the hash table }
  if (FTable <> nil) then begin
    Clear;
    FTable.Destroy;
  end;
  { free the character queue }
  TCharQueue(FCharQueue).Free;
  { free the script node freelist }
  Walker := FScriptFreeList;
  while (Walker <> nil) do begin
    Temp := Walker;
    Walker := Walker^.snNext;
    Dispose(Temp);
  end;
  inherited Destroy;
end;
{--------}
function TApxCharSetMapping.Add(const aCharSet : TApxKeyString;
                                     aFromCh  : AnsiChar;
                                     aToCh    : AnsiChar;
                               const aFont    : TApxKeyString;
                                     aGlyph   : AnsiChar) : Boolean;
var
  Inx  : integer;
  Node : PCSHashNode;
  Ch   : AnsiChar;
  Glyph: AnsiChar;
begin
  { we must do this in two stages: first, determine that we can add }
  { *all* the character mappings; second, do so }

  { stage one: check no mapping currently exists }
  Result := False;
  for Ch := aFromCh to aToCh do begin
    if csmFindPrim(aCharSet, Ch, Inx, pointer(Node)) then
      Exit;
  end;
  { stage two: add all charset/char mappings }
  Result := True;
  Glyph := aGlyph;
  for Ch := aFromCh to aToCh do begin
    Inx := HashELFPlusChar(aCharSet, Ch) mod CSHashTableSize;
    New(Node);
    Node^.csnNext := FTable[Inx];
    Node^.csnCharSet := AllocKeyString(aCharSet);
    Node^.csnFont := AllocKeyString(aFont);
    Node^.csnChar := Ch;
    Node^.csnGlyph := Glyph;
    FTable[Inx] := Node;
    inc(FCount);
    inc(Glyph);
  end;
end;
{--------}
procedure TApxCharSetMapping.Clear;
var
  i    : integer;
  Node : PCSHashNode;
  Temp : PCSHashNode;
begin
  { free the script: in a moment there's going to be no mapping }
  csmFreeScript;
  { clear out the hash table }
  for i := 0 to pred(CSHashTableSize) do begin
    Node := FTable[i];
    while (Node <> nil) do begin
      Temp := Node;
      Node := Node^.csnNext;
      FreeKeyString(Temp^.csnCharSet);
      FreeKeyString(Temp^.csnFont);
      Dispose(Temp);
    end;
    FTable[i] := nil;
  end;
  FCount := 0;
end;
{--------}
procedure TApxCharSetMapping.csmAddScriptNode(aFont : PadKeyString);
var
  Node : PScriptNode;
begin
  { allocate and set up the new node }
  if (FScriptFreeList = nil) then
    New(Node)
  else begin
    Node := FScriptFreeList;
    FScriptFreeList := Node^.snNext;
  end;
  Node^.snNext := nil;
  Node^.snFont := aFont;
  Node^.snText := TCharQueue(FCharQueue).DupText;
  { add the node to the script }
  if (FScript <> nil) then
    PScriptNode(FScriptEnd)^.snNext := Node
  else
    FScript := Node;
  { update the tail pointer }
  FScriptEnd := Node;
end;
{--------}
function TApxCharSetMapping.csmFindPrim(const aCharSet : TApxKeyString;
                                             aChar    : AnsiChar;
                                         var aInx     : integer;
                                         var aNode    : pointer) : Boolean;
var
  Node : PCSHashNode;
begin
  { assume we won't find aCharSet/aChar }
  Result := False;
  aNode := nil;
  { calculate the index, ie hash, of the charset/char }
  aInx := HashELFPlusChar(aCharSet, aChar) mod CSHashTableSize;
  { traverse the linked list at this entry, looking for the character   }
  { in each node we encounter--a case-sensitive comparison--if we get a }
  { match, compare the character set name as well, again case-          }
  { insensitive }
  Node := FTable[aInx];
  while (Node <> nil) do begin
    if (aChar = Node^.csnChar) then begin
      if (aCharSet = Node^.csnCharSet^) then begin
        Result := True;
        aNode := Node;
        Exit;
      end;
    end;
    Node := Node^.csnNext;
  end;
end;
{--------}
procedure TApxCharSetMapping.csmFreeScript;
var
  Walker, Temp : PScriptNode;
begin
  Walker := FScript;
  FScript := nil;
  while (Walker <> nil) do begin
    Temp := Walker;
    Walker := Walker^.snNext;
    FreeMem(Temp^.snText, StrLen(Temp^.snText));
    { NOTE: we do NOT free the font name: it's a copy of an allocated }
    { string in the mapping hash table }
    Temp^.snNext := FScriptFreeList;
    FScriptFreeList := Temp;
  end;
end;
{--------}
{$IFDEF CompileDebugCode}
procedure TApxCharSetMapping.DebugPrint(const aFileName : string);
var
  F    : text;
  i    : integer;
  Node : PCSHashNode;
begin
  System.Assign(F, aFileName);
  System.Rewrite(F);

  for i := 0 to pred(CSHashTableSize) do begin
    writeln(F, '---', i, '---');
    Node := FTable[i];
    while (Node <> nil) do begin
      writeln(F, Node^.csnCharSet^:20,
                 ord(Node^.csnChar):4,
                 Node^.csnFont^:20,
                 ord(Node^.csnGlyph):4);
      Node := Node^.csnNext;
    end;
  end;

  writeln(F);
  writeln(F, 'Count: ', Count, ' (mean: ', Count/CSHashTableSize:5:3, ')');

  System.Close(F);
end;
{$ENDIF}
{--------}
procedure TApxCharSetMapping.GenerateDrawScript(const aCharSet : TApxKeyString;
                                                      aText    : PAnsiChar);
var
  i    : integer;
  Inx  : integer;
  TextLen  : integer;
  Node     : PCSHashNode;
  Ch       : AnsiChar;
  CurFont  : PadKeyString;
  ThisFont : PadKeyString;
  ThisChar : AnsiChar;
begin
  {nothing to do if the string is empty}
  TextLen := StrLen(aText);
  if (TextLen = 0) then
    Exit;
  {destroy any current script}
  csmFreeScript;
  TCharQueue(FCharQueue).Clear;
  {we don't yet have a font name}
  CurFont := nil;
  {read the text, char by char}
  for i := 0 to pred(TextLen) do begin
    {look up this charset/char in the hash table}
    Ch := aText[i];
    if csmFindPrim(aCharSet, Ch, Inx, pointer(Node)) then begin
      {found it, use the named font and glyph}
      ThisFont := Node^.csnFont;
      ThisChar := Node^.csnGlyph;
    end
    else begin
      {if not found, use the default font and glyph}
      ThisFont := @DefaultFontName;
      ThisChar := Ch;
    end;
    { if the font has changed, create a script node for the previous font }
    if (CurFont = nil) then
      CurFont := ThisFont;
    if (CurFont^ <> ThisFont^) then begin
      csmAddScriptNode(CurFont);
      CurFont := ThisFont; 
      TCharQueue(FCharQueue).Clear;
    end;
    {add this character to the current string}
    TCharQueue(FCharQueue).Add(ThisChar);
  end;
  {add the final script node to finish off the string}
  csmAddScriptNode(CurFont);
  TCharQueue(FCharQueue).Clear;
end;
{--------}
procedure TApxCharSetMapping.GetFontNames(aList : TStrings);
var
  i    : integer;
  Node : PCSHashNode;
  PrevFont : string;
begin
  aList.Clear;
  PrevFont := '';
  for i := 0 to pred(CSHashTableSize) do begin
    Node := FTable[i];
    while (Node <> nil) do begin
      if (CompareText(Node^.csnFont^, PrevFont) <> 0) then begin
        PrevFont := Node^.csnFont^;
        if (aList.IndexOf(PrevFont) = -1) then
          aList.Add(PrevFont);
      end;
      Node := Node^.csnNext;
    end;
  end;
end;
{--------}
function TApxCharSetMapping.GetNextDrawCommand(var aFont : TApxKeyString;
                                                  aText : PAnsiChar) : Boolean;
var
  Temp : PScriptNode;
begin
  {start off with the obvious case: there's no script}
  if (FScript = nil) then begin
    Result := False;
    Exit;
  end;
  {we'll definitely return something}
  Result := True;
  {return the data from the top node}
  aFont := PScriptNode(FScript)^.snFont^;
  StrCopy(aText, PScriptNode(FScript)^.snText);
  {unlink the top node}
  Temp := PScriptNode(FScript);
  FScript := Temp^.snNext;
  {free the unlinked top node}
  FreeMem(Temp^.snText, StrLen(Temp^.snText));
  {NOTE: we do NOT free the font name: it's a copy of an allocated
   string in the mapping hash table}
  Temp^.snNext := FScriptFreeList;
  FScriptFreeList := Temp;
end;
{--------}
procedure TApxCharSetMapping.LoadFromFile(const aFileName : string);
var
  Lines     : TStringList;
  ActualLen : integer;
  i         : integer;
  LineInx   : integer;
  Line      : string[255];
  CharSet   : TApxKeyString;
  FontName  : TApxKeyString;
  FromCh    : AnsiChar;
  ToCh      : AnsiChar;
  Glyph     : AnsiChar;
begin
  {clear the hash table, ready for loading}
  Clear;
  {create the stringlist to hold the mapping script}
  Lines := TStringList.Create;
  try
    {load the mapping script}
    Lines.LoadFromFile(aFileName);
    for LineInx := 0 to pred(Lines.Count) do begin
      {get this line}
      Line := Lines[LineInx];
      {remove trailing spaces}
      ActualLen := length(Line);
      for i := ActualLen downto 1 do
        if (Line[i] = ' ') then
          dec(ActualLen)
        else
          Break;
      Line[0] := chr(ActualLen);
      {only process detail lines}
      if (Line <> '') and (Line[1] = ' ') then begin
        if ProcessCharSetLine(Line, CharSet, FromCh, ToCh, FontName, Glyph) then
          Add(CharSet, FromCh, ToCh, FontName, Glyph);
      end;
    end;
  finally
    Lines.Free;
  end;
end;
{--------}
procedure TApxCharSetMapping.LoadFromRes(aInstance : THandle;
                                  const aResName  : string);
var
  MS        : TMemoryStream;
  ResInfo   : THandle;
  ResHandle : THandle;
  ResNameZ  : PAnsiChar;
  Res       : PByteArray;
  i         : integer;
  Sig       : longint;
  ResCount  : longint;
  BytesRead : longint;
  CharSet   : TApxKeyString;
  Font      : TApxKeyString;
  Ch        : AnsiChar;
  Glyph     : AnsiChar;
begin
  {Note: this code has been written to work with all versions of
   Delphi, both 16-bit and 32-bit. Hence it does not make use of any
   of the features available in later compilers, ie, typecasting a
   string to a PChar, or TResourceStream)

  {clear the hash table, ready for loading}
  Clear;
  {get the resource info handle}
  GetMem(ResNameZ, succ(length(aResName))); 
  try
    StrPCopy(ResNameZ, aResName);
    ResInfo := FindResource(aInstance, ResNameZ, RT_RCDATA);
  finally
    FreeMem(ResNameZ, succ(length(aResName)));
  end;
  if (ResInfo = 0) then
    Exit;
  {load and lock the resource}
  ResHandle := LoadResource(aInstance, ResInfo);
  if (ResHandle = 0) then
    Exit;
  Res := LockResource (ResHandle);
  try
    {create a memory stream}
    MS := TMemoryStream.Create;
    try
      {copy the resource to our memory stream}
      MS.Write(Res^, SizeOfResource(aInstance, ResInfo));  
      MS.Position := 0;
      {read the header signature, get out if it's not ours}
      BytesRead := MS.Read(Sig, sizeof(Sig));
      if (BytesRead <> sizeof(Sig)) or (Sig <> OurSignature) then
        Exit;
      {read the count of mappings in the resource}
      MS.Read(ResCount, sizeof(ResCount));
      {read that number of mappings and add them to the hash table}
      for i := 0 to pred(ResCount) do begin
        MS.Read(CharSet[0], 1);
        MS.Read(CharSet[1], ord(CharSet[0]));
        MS.Read(Font[0], 1);
        MS.Read(Font[1], ord(Font[0]));
        MS.Read(Ch, 1);
        MS.Read(Glyph, 1);
        Add(CharSet, Ch, Ch, Font, Glyph);
      end;
      {read the footer signature, clear and get out if it's not ours}
      BytesRead := MS.Read(Sig, sizeof(Sig));
      if (BytesRead <> sizeof(Sig)) or (Sig <> OurSignature) then begin
        Clear;
        Exit;
      end;
    finally
      MS.Free;
    end;
  finally
    FreeResource(ResHandle); 
  end; 
end;
{--------}
procedure TApxCharSetMapping.StoreToBinFile(const aFileName : string);
var
  aFS  : TFileStream;
  i    : integer;
  Node : PCSHashNode;
begin
  {create a file stream}
  aFS := TFileStream.Create(aFileName, fmCreate);
  try
    {write our signature as header}
    aFS.Write(OurSignature, sizeof(OurSignature));
    {write the number of mappings}
    aFS.Write(FCount, sizeof(FCount));
    {write all the mappings}
    for i := 0 to pred(CSHashTableSize) do begin
      Node := FTable[i];
      while (Node <> nil) do begin
        aFS.Write(Node^.csnCharSet^, succ(length(Node^.csnCharSet^)));
        aFS.Write(Node^.csnFont^, succ(length(Node^.csnFont^)));
        aFS.Write(Node^.csnChar, sizeof(AnsiChar));
        aFS.Write(Node^.csnGlyph, sizeof(AnsiChar));
        Node := Node^.csnNext;
      end;
    end;
    {write our signature as footer as a further check on reading}
    aFS.Write(OurSignature, sizeof(OurSignature));
  finally
    aFS.Free;
  end;
end;
{====================================================================}
{===TApxTerminalParser================================================}
constructor TApxTerminalParser.Create(aUseWideChar : Boolean);
begin
  inherited Create;
  FUseWideChar := aUseWideChar;
  FCommand := eNone;
end;
{--------}
destructor TApxTerminalParser.Destroy;
begin
  inherited Destroy;
end;
{--------}
procedure TApxTerminalParser.Clear;
begin
  {do nothing at this level}
end;
{--------}
function TApxTerminalParser.ProcessChar(aCh : AnsiChar) : TApxParserCmdType;
begin
  Result := pctNone;
end;
{--------}
function TApxTerminalParser.ProcessWideChar(aCh : WideChar) : TApxParserCmdType;
begin
  Result := pctNone;
end;
{--------}
function TApxTerminalParser.tpGetArgument(aInx : integer) : integer;
begin
  Result := 0;
end;
{--------}
function TApxTerminalParser.tpGetSequence : string;
begin
  Result := '';
end;
{====================================================================}


{====================================================================}
type
  PSeq = ^TSeq;
  TSeq = packed record
    sSize : longint;
    sLen  : longint;
    sText : array [1..10000] of AnsiChar;
  end;
{--------}
function ReAllocSeq(aSeq : PSeq; aSize : longint) : PSeq;
var
  NewSeq : PSeq;
begin
  if (aSize = 0) then
    NewSeq := nil
  else begin
    GetMem(NewSeq, 2*sizeof(longint) + aSize);
    NewSeq^.sSize := aSize;
    NewSeq^.sLen := 0;
    if (aSeq <> nil) then begin
      Move(aSeq^.sText, NewSeq^.sText, aSeq^.sLen);
      NewSeq^.sLen := aSeq^.sLen;
    end;
  end;
  if (aSeq <> nil) then
    FreeMem(aSeq, 2*sizeof(longint) + aSeq^.sSize);
  Result := NewSeq;
end;
{--------}
procedure AddCharToSeq(var aSeq : PSeq; aCh : AnsiChar);
begin
  if (aSeq = nil) then
    aSeq := ReAllocSeq(aSeq, 64)
  else if (aSeq^.sSize = aSeq^.sLen) then
    aSeq := ReAllocSeq(aSeq, aSeq^.sSize + 64);
  inc(aSeq^.sLen);
  aSeq^.sText[aSeq^.sLen] := aCh;
end;
{--------}
procedure AssignSeqToChar(var aSeq : PSeq; aCh : AnsiChar);
begin
  if (aSeq <> nil) then
    aSeq^.sLen := 0;
  AddCharToSeq(aSeq, aCh);
end;
{--------}
procedure CopySeq(aFromSeq : PSeq; var aToSeq : PSeq);
begin
  if (aFromSeq = nil) then begin
    if (aToSeq <> nil) then
      aToSeq^.sLen := 0;
  end
  else begin
    if (aToSeq = nil) or
       (aToSeq^.sSize < aFromSeq^.sLen) then
      aToSeq := ReAllocSeq(aToSeq, aFromSeq^.sLen);
    if (aToSeq <> nil) then begin
      aToSeq^.sLen := aFromSeq^.sLen;
      Move(aFromSeq^.sText, aToSeq^.sText, aFromSeq^.sLen);
    end;
  end;
end;
{--------}
procedure DelCharFromSeq(aSeq : PSeq);
begin
  if (aSeq <> nil) and (aSeq^.sLen > 0) then
    dec(aSeq^.sLen);
end;
{--------}
procedure ClearSeq(aSeq : PSeq);
begin
  if (aSeq <> nil) then
    aSeq^.sLen := 0;
end;
{--------}
function GetSeqLength(aSeq : PSeq) : integer;
begin
  Result := aSeq^.sLen;
end;
{--------}
function GetStringFromSeq(aSeq : PSeq) : string;
begin
  Result := '';
  if (aSeq <> nil) and (aSeq^.sLen > 0) then begin
    SetLength(Result, aSeq^.sLen);
    Move(aSeq^.sText, Result[1], aSeq^.sLen);
  end;
end;
{====================================================================}

const
  DECSCLseq : string[6] = ^['[61"p';

{===TApxVT100Parser===================================================}
constructor TApxVT100Parser.Create(aUseWideChar : Boolean);
begin
  inherited Create(aUseWideChar);
  FArgCount := 0;
  vtpGrowArgs;
  FInVT52Mode := False;
end;
{--------}
destructor TApxVT100Parser.Destroy;
begin
  FSequence := ReAllocSeq(FSequence, 0);
  FSavedSeq := ReAllocSeq(FSavedSeq, 0);
  if (FArgs <> nil) then begin
    FreeMem(FArgs, sizeof(integer) * FArgCountMax);
    FArgs := nil;
    FArgCountMax := 0;
  end;
  inherited Destroy;
end;
{--------}
procedure TApxVT100Parser.Clear;
begin
  ClearSeq(FSequence);
  FCommand := eNone;
  if (FArgCount <> 0) then begin
    FillChar(FArgs^, sizeof(integer) * FArgCount, 0);
    FArgCount := 0;
  end;
end;
{--------}
function TApxVT100Parser.ProcessChar(aCh : AnsiChar) : TApxParserCmdType;
begin
  {if the current state is psGotCommand, the previous character
   managed to complete a command. Before comtinuing we should clear
   all traces of the previous command and sequence}
  if (FState = psGotCommand) then begin
    FArgCount := 0;
    ClearSeq(FSequence);
    FCommand := eNone;
    FState := psIdle;
  end;

  {if the current state is psGotInterCommand, the previous character
   was non-displayable and a command; restore the previously saved
   state}
  if (FState = psGotInterCommand) then begin
    FArgCount := 0;
    FCommand := eNone;
    CopySeq(FSavedSeq, PSeq(FSequence));
    FState := FSavedState;
  end;

  {assume that the result is going to be that we are building up a
   command escape sequence}
  Result := pctPending;

  {add the character to the sequence string we're building up,
   although we may delete it later}
  AddCharToSeq(PSeq(FSequence), aCh);

  {if the character is non-displayable, process it immediately, even
   if we're in the middle of parsing some other command}
  if (aCh < ' ') then begin
    FSavedState := FState;
    DelCharFromSeq(FSequence);
    CopySeq(FSequence, PSeq(FSavedSeq));
    FState := psGotInterCommand;
    Result := pctComplete;
    case aCh of
      cENQ : begin {enquiry request}
               AssignSeqToChar(PSeq(FSequence), cENQ);
               FCommand := eENQ;
             end;
      cBel : begin {sound bell}
               AssignSeqToChar(PSeq(FSequence), cBel);
               FCommand := eBel;
             end;
      cBS  : begin {backspace}
               AssignSeqToChar(PSeq(FSequence), cBS);
               FCommand := eBS;
             end;
      cTab : begin {horizontal tab}
               AssignSeqToChar(PSeq(FSequence), cTab);
               FCommand := eCHT;
               FArgCount := 1;
               FArgs^[0] := 1; {ie a single tab}
             end;
      cLF  : begin
               AssignSeqToChar(PSeq(FSequence), cLF);
               FCommand := eLF;
             end;
      cVT  : begin
               AssignSeqToChar(PSeq(FSequence), cVT);
               FCommand := eCVT;
               FArgCount := 1;
               FArgs^[0] := 1; {ie a single tab}
             end;
      cFF  : begin {formfeed, equals clear screen}
               AssignSeqToChar(PSeq(FSequence), cFF);
               FCommand := eED;
               FArgCount := 1;
               FArgs^[0] := 2; {ie <esc>[2J}
             end;
      cCR  : begin {carriage return}
               AssignSeqToChar(PSeq(FSequence), cCR);
               FCommand := eCR;
             end;
      cSO  : begin {shift out character set, ie use G0}
               AssignSeqToChar(PSeq(FSequence), cSO);
               FCommand := eSO;
             end;
      cSI  : begin {shift in character set, ie use G1}
               AssignSeqToChar(PSeq(FSequence), cSI);
               FCommand := eSI;
             end;
      cCan,
      cSub : begin {abandon current escape sequence}
               Result := pctNone;
             end;
      cEsc : begin {start a new escape sequence}
               {abandon whatever escape sequence we're in}
               AssignSeqToChar(PSeq(FSequence), cEsc);
               FArgCount := 0;
               FState := psGotEscape;
               Result := pctPending;
             end;
    else
      {otherwise ignore the non-displayable char}
      DelCharFromSeq(FSequence);
      Result := pctNone;
    end;{case}
  end
  {otherwise parse the character}
  else begin
    case FState of
      psIdle :
        begin
          if (aCh < #127) then begin
            FState := psGotCommand;
            FCommand := eChar;
            Result := pctChar;
          end
          else {full 8-bit char} begin
            FState := psGotCommand;
            FCommand := eChar;
            Result := pct8bitChar;
          end;
        end;
      psGotEscape :
        if InVT52Mode then begin
          Result := vtpProcessVT52(aCh);
        end
        else {in VT100 mode} begin
          case aCh of
            '[' : FState := psParsingANSI;
            '(' : FState := psParsingLeftParen;
            ')' : FState := psParsingRightParen;
            '#' : FState := psParsingHash;
            '*', '+', '-', '.', '/' : FState := psParsingCharSet;
          else {it's a two character esc. seq.}
            FState := psGotCommand;
            Result := pctComplete;
            case aCh of
              '1' : begin {set graphics processor option on}
                      { NOT SUPPORTED }
                      FCommand := eNone;
                    end;
              '2' : begin {set graphics processor option off}
                      { NOT SUPPORTED }
                      FCommand := eNone;
                    end;
              '7' : begin {save cursor pos}
                      FCommand := eSaveCursorPos;
                    end;
              '8' : begin {restore cursor pos}
                      FCommand := eRestoreCursorPos;
                    end;
              '<' : begin {switch to ANSI--ie, do nothing}
                      FCommand := eNone;
                      Result := pctNone;
                    end;
              '=' : begin {set application keypad mode}
                      FCommand := eSM;
                      FArgCount := 2;
                      FArgs^[0] := -2;
                      FArgs^[1] := 999; {special APRO code!}
                    end;
              '>' : begin {set numeric keypad mode}
                      FCommand := eRM;
                      FArgCount := 2;
                      FArgs^[0] := -2;
                      FArgs^[1] := 999; {special APRO code!}
                    end;
              'D' : begin {index = cursor down + scroll}
                      FCommand := eIND2;
                    end;
              'E' : begin {next line}
                      FCommand := eNEL;
                    end;
              'H' : begin {set horx tab stop}
                      FCommand := eHTS;
                    end;
              'M' : begin {reverse index = cursor up + scroll}
                      FCommand := eRI;
                    end;

              'Z' : begin {device attributes}
                      FCommand := eDA;
                      FArgCount := 1;
                      FArgs^[0] := 0; {stands for VT100}
                    end;
              'c' : begin
                      FCommand := eRIS;
                    end;
            else
              {ignore the char & seq.--it's not one we know}
              Result := pctNone;
            end;{case}
          end;{case}
        end;
      psParsingANSI :
        begin
          if (#$40 <= aCh) and (aCh < #$7F) then begin
            {the command is now complete-see if we know about it}
            FState := psGotCommand;
            Result := vtpParseANSISeq(aCh);
          end;
          {otherwise, the next character has already been added to
           the sequence string, so there's nothing extra to do}
        end;
      psParsingLeftParen :
        begin
          if ('0' <= aCh) and (aCh <= '~') then begin
            {the command is complete}
            if (GetSeqLength(FSequence) = 3) then begin
              FState := psGotCommand;
              Result := pctComplete;
              FCommand := eDECSCS;
              FArgCount := 2;
              FArgs^[0] := 0; {0 = set G0 charset}
              case aCh of
                'A' : FArgs^[1] := ord('A');
                'B' : FArgs^[1] := ord('B');
                '0' : FArgs^[1] := 0;
                '1' : FArgs^[1] := 1;
                '2' : FArgs^[1] := 2;
              else
                {ignore the char & seq.--it's not one we know}
                FState := psGotCommand;
                Result := pctNone;
                FCommand := eNone;
                FArgCount := 0;
              end;{case}
            end
            else {sequence is too long} begin
              FState := psGotCommand;
              Result := pctNone;
              FCommand := eNone;
              FArgCount := 0;
            end;
          end;
        end;
      psParsingRightParen :
        begin
          if ('0' <= aCh) and (aCh <= '~') then begin
            {the command is complete}
            if (GetSeqLength(FSequence) = 3) then begin
              FState := psGotCommand;
              Result := pctComplete;
              FCommand := eDECSCS;
              FArgCount := 2;
              FArgs^[0] := 1; {0 = set G1 charset}
              case aCh of
                'A' : FArgs^[1] := ord('A');
                'B' : FArgs^[1] := ord('B');
                '0' : FArgs^[1] := 0;
                '1' : FArgs^[1] := 1;
                '2' : FArgs^[1] := 2;
              else
                {ignore the char & seq.--it's not one we know}
                FState := psGotCommand;
                Result := pctNone;
                FCommand := eNone;
                FArgCount := 0;
              end;{case}
            end
            else {sequence is too long} begin
              FState := psGotCommand;
              Result := pctNone;
              FCommand := eNone;
              FArgCount := 0;
            end;
          end;
        end;
      psParsingCharSet :
        begin
          {these are the VT200+ "switch charset" sequences: we ignore
           them after finding the first char in range $30..$7E}
          if ('0' <= aCh) and (aCh <= '~') then begin
            FState := psGotCommand;
            Result := pctNone;
            FCommand := eNone;
            FArgCount := 0;
          end;
        end;
      psParsingHash :
        begin
          FState := psGotCommand;
          Result := pctComplete;
          case aCh of
            '3' : begin
                    FCommand := eDECDHL;
                    FArgCount := 1;
                    FArgs^[0] := 0; {0 = top half}
                  end;
            '4' : begin
                    FCommand := eDECDHL;
                    FArgCount := 1;
                    FArgs^[0] := 1; {1 = bottom half}
                  end;
            '5' : begin
                    FCommand := eDECSWL;
                  end;
            '6' : begin
                    FCommand := eDECDWL;
                  end;
            '8' : begin
                    FCommand := eDECALN;
                  end;
          else
            {ignore the char & seq.--it's not one we know}
            FState := psGotCommand;
            Result := pctNone;
          end;{case}
        end;
      psParsingCUP52 :
        begin
          if (FArgCount = 0) then begin
            FArgs^[0] := ord(aCh) - $1F;
            inc(FArgCount);
          end
          else begin
            FState := psGotCommand;
            FCommand := eCUP;
            FArgs^[1] := ord(aCh) - $1F;
            inc(FArgCount);
            Result := pctComplete;
          end;
        end;
    else
      {invalid state?}
    end;{case}
  end;
end;
{--------}
function TApxVT100Parser.ProcessWideChar(aCh : WideChar) :TApxParserCmdType;
begin
  Result := pctNone;
end;
{--------}
function TApxVT100Parser.tpGetArgument(aInx : integer) : integer;
begin
  if (aInx < 0) or (aInx >= FArgCount) then
    Result := 0
  else
    Result := FArgs^[aInx];
end;
{--------}
function TApxVT100Parser.tpGetSequence : string;
begin
  if (FCommand <> eNone) then
    Result := GetStringFromSeq(FSequence)
  else
    Result := '';
end;
{--------}
function TApxVT100Parser.vtpGetArguments : Boolean;
var
  ChInx   : integer;
  StartInx: integer;
  Ch      : char;
  ec      : integer;
  TempStr : string[255];
begin
  {for this parser, we assume
     1. arguments consist of numeric digits only
     2. arguments are separated by ';'
     3. the first argument can be ? (DEC VT100 special)
     4. argument parsing stops at the first character #$20 - #$2F, or
        #$40 - #$7E}

  {assume the sequence is badly formed}
  Result := False;

  {first check for the third character being ?}
  if (PSeq(FSequence)^.sText[3] = '?') then begin
    FArgCount := 1;
    FArgs^[0] := -2;
    StartInx := 4;
  end
  else
    StartInx := 3;

  {scan the rest of the characters until we reach a char in the range
   $20-$2F, or $40-$7E; look out for numeric digits and semi-colons}
  TempStr := '';
  for ChInx := StartInx to PSeq(FSequence)^.sLen do begin
    Ch := PSeq(FSequence)^.sText[ChInx];
    if ((#$20 <= Ch) and (Ch <= #$2F)) or
       ((#$40 <= Ch) and (Ch <= #$7E)) then
      Break;
    if (Ch = ';') then begin
      if (FArgCountMax = FArgCount) then
        vtpGrowArgs;
      if (TempStr = '') then begin
        FArgs^[FArgCount] := -1;
        inc(FArgCount);
      end
      else begin
        Val(TempStr, FArgs^[FArgCount], ec);
        if (ec <> 0) then
          Exit;
        inc(FArgCount);
        TempStr := '';
      end;
    end
    else if ('0' <= Ch) and (Ch <= '9') then begin
      TempStr := TempStr + Ch;
    end
    else {bad character}
      Exit;
  end;

  {convert the final argument}
  if (FArgCountMax = FArgCount) then
    vtpGrowArgs;
  if (TempStr = '') then begin
    FArgs^[FArgCount] := -1;
    inc(FArgCount);
  end
  else begin
    Val(TempStr, FArgs^[FArgCount], ec);
    if (ec <> 0) then
      Exit;
    inc(FArgCount);
  end;

  {if we got here, everything was all right}
  Result := True;
end;
{--------}
procedure TApxVT100Parser.vtpGrowArgs;
var
  NewMax   : integer;
  NewArray : PAdIntegerArray;
begin
  {use a simple increase-by-half algorithm}
  if (FArgCountMax = 0) then
    NewMax := 16
  else
    NewMax := (FArgCountMax * 3) div 2;
  {alloc the new array, zeroed}
  NewArray := AllocMem(sizeof(integer) * NewMax);
  {if there's any data in the old array copy it over, delete it}
  if (FArgs <> nil) then begin
    Move(FArgs^, NewArray^, sizeof(integer) * FArgCount);
    FreeMem(FArgs, sizeof(integer) * FArgCountMax);
  end;
  {remember the new details}
  FArgs := NewArray;
  FArgCountMax := NewMax;
end;
{--------}
function TApxVT100Parser.vtpParseANSISeq(aCh : char) : TApxParserCmdType;
begin
  {when this method is called FSequence has the full escape sequence,
   and FArgCount, FArgs, FCommand have to be set; for convenience aCh
   is the final character in FSequence--the command identifier--and
   FSequence must have at least three characters in it}

  {assume the sequence is invalid}
  Result := pctNone;

  {special case: DECSCL}
  if (GetStringFromSeq(FSequence) = DECSCLseq) then begin
    FCommand := eRIS;
    Result := pctComplete;
  end;

  {split out the arguments in the sequence, build up the FArgs array;
   note that an arg of -1 means 'default', and -2 means ? (a special
   DEConly parameter)}
  if not vtpGetArguments then
    Exit;

  {identify the command character}
  case aCh of
    '@' : begin {insert character--VT102}
            FCommand := eICH;
            {should only have one parameter, default of 1}
            if not vtpValidateArgsPrim(1, 1, 1) then Exit;
          end;
    'A' : begin {Cursor up}
            FCommand := eCUU;
            {should only have one parameter, default of 1}
            if not vtpValidateArgsPrim(1, 1, 1) then Exit;
          end;
    'B' : begin {Cursor down}
            FCommand := eCUD;
            {should only have one parameter, default of 1}
            if not vtpValidateArgsPrim(1, 1, 1) then Exit;
          end;
    'C' : begin {Cursor right}
            FCommand := eCUF;
            {should only have one parameter, default of 1}
            if not vtpValidateArgsPrim(1, 1, 1) then Exit;
          end;
    'D' : begin {cursor left}
            FCommand := eCUB;
            {should only have one parameter, default of 1}
            if not vtpValidateArgsPrim(1, 1, 1) then Exit;
          end;
    'H' : begin {cursor position}
            FCommand := eCUP;
            {should have two parameters, both default of 1}
            if not vtpValidateArgsPrim(2, 2, 1) then Exit;
          end;
    'J' : begin {Erase in display}
            FCommand := eED;
            {should only have one parameter, default of 0}
            if not vtpValidateArgsPrim(1, 1, 0) then Exit;
          end;
    'K' : begin {Erase in line}
            FCommand := eEL;
            {should only have one parameter, default of 0}
            if not vtpValidateArgsPrim(1, 1, 0) then Exit;
          end;
    'L' : begin {Insert line--VT102}
            FCommand := eIL;
            {should only have one parameter, default of 1}
            if not vtpValidateArgsPrim(1, 1, 1) then Exit;
          end;
    'M' : begin {Delete line--VT102}
            FCommand := eDL;
            {should only have one parameter, default of 1}
            if not vtpValidateArgsPrim(1, 1, 1) then Exit;
          end;
    'P' : begin {delete character--VT102}
            FCommand := eDCH;
            {should only have one parameter, default of 1}
            if not vtpValidateArgsPrim(1, 1, 1) then Exit;
          end;
    'X' : begin {erase character--VT102}
            FCommand := eECH;
            {should only have one parameter, default of 1}
            if not vtpValidateArgsPrim(1, 1, 1) then Exit;
          end;
    'c' : begin {Device attributes}
            FCommand := eDA;
            {should only have one parameter, default of 0}
            if not vtpValidateArgsPrim(1, 1, 0) then Exit;
          end;
    'f' : begin {cursor position}
            FCommand := eCUP;
            {should have two parameters, both default of 1}
            if not vtpValidateArgsPrim(2, 2, 1) then Exit;
          end;
    'g' : begin {clear horizontal tabs}
            FCommand := eTBC;
            {should only have one parameter, default of 0}
            if not vtpValidateArgsPrim(1, 1, 0) then Exit;
          end;
    'h' : begin {set mode}
            FCommand := eSM;
            {should have one parameter, or 2 if the first is ?, no
             defaults}
           end;
    'l' : begin {reset mode}
            FCommand := eRM;
            {should have one parameter, or 2 if the first is ?, no
             defaults}
            {we have to try and spot one command in particular: the
             switch to VT52 mode}
            if (FArgCount = 2) and
               (FArgs^[0] = -2) and (FArgs^[1] = 2) then
              FInVT52Mode := True; 
          end;
    'm' : begin
            FCommand := eSGR;
            {should have at least one parameter, default of 0 for all
             parameters}
            if not vtpValidateArgsPrim(1, 30000, 0) then Exit;
          end;
    'n' : begin {Device status report}
            FCommand := eDSR;
            {should only have one parameter, no default}
            if not vtpValidateArgsPrim(1, 1, -1) then Exit;
          end;
    'q' : begin {DEC PRIVATE-set/clear LEDs}
            FCommand := eDECLL;
            {should have at least one parameter, default of 0 for all
             parameters}
            if not vtpValidateArgsPrim(1, 30000, 0) then Exit;
          end;
    'r' : begin {DEC PRIVATE-set top/bottom margins}
            FCommand := eDECSTBM;
            {should have two parameters, first default of 1, second
             default unknowable by this class}
          end;
    's' : begin {save cursor pos - ANSI.SYS escape sequence}
            FCommand := eSaveCursorPos;
          end;
    'u' : begin {restore cursor pos - ANSI.SYS escape sequence}
            FCommand := eRestoreCursorPos;
          end;
    'x' : begin {DEC PRIVATE-request terminal parameters}
            FCommand := eDECREQTPARM;
            {should only have one parameter, no default}
            if not vtpValidateArgsPrim(1, 1, -1) then Exit;
          end;
    'y' : begin {DEC PRIVATE-invoke confidence test}
            FCommand := eDECTST;
            {should have two parameters, no default for first, second
             default to 0}
          end;
  else {the command letter is unknown}
    Exit;
  end;{case}

  {if we get here the sequence is valid and we've patched up the
   arguments list and count}
  Result := pctComplete;
end;
{--------}
function TApxVT100Parser.vtpProcessVT52(aCh : char) : TApxParserCmdType;
begin
  FState := psGotCommand;
  Result := pctComplete;
  case aCh of
    '<' : begin { switch to ANSI mode }
            FCommand := eSM;
            FArgCount := 2;  { pretend it's Esc[?2h }
            FArgs^[0] := -2;
            FArgs^[1] := 2;
            FInVT52Mode := False;
          end;
    '=' : begin { enter alternate keypad mode }
            FCommand := eSM;
            FArgCount := 2;
            FArgs^[0] := -2;
            FArgs^[1] := 999; { special APRO code! }
          end;
    '>' : begin { leave alternate keypad mode }
            FCommand := eRM;
            FArgCount := 2;
            FArgs^[0] := -2;
            FArgs^[1] := 999; { special APRO code! }
          end;
    'A' : begin { cursor up }
            FCommand := eCUU;
            FArgCount := 1;
            FArgs^[0] := 1;
          end;
    'B' : begin { cursor down }
            FCommand := eCUD;
            FArgCount := 1;
            FArgs^[0] := 1;
          end;
    'C' : begin { cursor right }
            FCommand := eCUF;
            FArgCount := 1;
            FArgs^[0] := 1;
          end;
    'D' : begin { cursor left }
            FCommand := eCUB;
            FArgCount := 1;
            FArgs^[0] := 1;
          end;
    'F' : begin { switch to graphics characters }
            FCommand := eSO;
          end;
    'G' : begin { switch to ASCII characters }
            FCommand := eSI;
          end;
    'H' : begin { move cursor home }
            FCommand := eCUP;
            FArgCount := 2;
            FArgs^[0] := 1;
            FArgs^[1] := 1;
          end;
    'I' : begin { reverse index = cursor up + scroll }
            FCommand := eRI;
          end;
    'J' : begin { erase to end of screen }
            FCommand := eED;
            FArgCount := 1;
            FArgs^[0] := 0; { ie <esc>[0J }
          end;
    'K' : begin { erase to end of line }
            FCommand := eEL;
            FArgCount := 1;
            FArgs^[0] := 0; { ie <esc>[0K }
          end;
    'Y' : begin { position cursor }
            FState := psParsingCUP52;
            FCommand := eCUP;
            Result := pctPending;
          end;
    'Z' : begin { device attributes, identify }
            FCommand := eDA;
            FArgCount := 1;
            FArgs^[0] := 52; { ie VT52 emulation }
          end;
  else
    Result := pctNone;
  end;{case}
end;
{--------}
function TApxVT100Parser.vtpValidateArgsPrim(aMinArgs : integer;
                                            aMaxArgs : integer;
                                            aDefault : integer) : Boolean;
var
  i : integer;
begin
  Result := False;
  { if we have too many arguments, something's obviously wrong }
  if (FArgCount > aMaxArgs) then
    Exit;
  { if we have too few, make the missing ones the default }
  while (FArgCount < aMinArgs) do begin
    if (FArgCountMax = FArgCount) then
      vtpGrowArgs;
    FArgs^[FArgCount] := aDefault;
    inc(FArgCount);
  end;
  { convert any -1 arguments to the default }
  for i := 0 to pred(FArgCount) do
    if (FArgs^[i] = -1) then
      FArgs^[i] := aDefault;
  { and we're done }
  Result := True;
end;
{=====================================================================}

{===TApxTerminalScrollBar=============================================}
constructor TApxTerminalScrollBar.Create (AOwner : TComponent);
begin
  inherited Create (AOwner);

  FTerminal := AOwner as TApxCustomTerminal;
  Track := False;
end;

procedure TApxTerminalScrollBar.BeginUpdate;
begin
  FInternalVisible := Visible;
  FUpdating := True;
end;

procedure TApxTerminalScrollBar.EndUpdate;
begin
  FUpdating := False;
  Visible := FInternalVisible;
end;

function TApxTerminalScrollBar.CanFocus : Boolean;
begin
  Result := False;
end;

procedure TApxTerminalScrollBar.InitWidget;
begin
  inherited InitWidget;
  QWidget_setFocusPolicy (FHandle, QWidgetFocusPolicy_NoFocus);
  Visible := False;
end;

procedure TApxTerminalScrollBar.WidgetDestroyed;
begin
  inherited WidgetDestroyed;
end;

function TApxTerminalScrollBar.GetVisible : Boolean;
begin
  if FUpdating then
    Result := FInternalVisible
  else
    Result := inherited Visible;
end;

procedure TApxTerminalScrollBar.SetVisible (Value : Boolean);
begin
  if FUpdating then
    FInternalVisible := Value
  else
  begin
    inherited Visible := Value;
    if Value then
      QWidget_show(Handle)
    else
      QWidget_hide(Handle);
  end;
end;

{===TApxTerminal======================================================}
constructor TApxCustomTerminal.Create(aOwner : TComponent);
{$IFDEF TRIALRUN}
  {$I TRIAL04.INC}
{$ENDIF}
begin
{$IFDEF TRIALRUN}
  TC;
{$ENDIF}
  inherited Create(aOwner);

  { set up the default values of the terminal's properties }
  Active := adc_TermActive; { the set access method must be called }
  FBlinkTime := adc_TermBlinkTime;
  FBorderStyle := adc_TermBorderStyle;
  FCapture := adc_TermCapture;
  FCaptureFile := adc_TermCaptureFile;
  FCursorType := adc_TermCursorType;
  FHalfDuplex := adc_TermHalfDuplex;
  FLazyByteDelay := adc_TermLazyByteDelay;
  FLazyTimeDelay := adc_TermLazyTimeDelay;
  FUseLazyDisplay := adc_TermUseLazyDisplay;
  FWantAllKeys := adc_TermWantAllKeys;

  { create the default emulator }
  FDefEmulator := TApxTTYEmulator.Create (nil);
  FDefEmulator.FIsDefault := True;
  FDefEmulator.Terminal := Self;
  if (FEmulator =  nil) then
    FEmulator := FDefEmulator;

  { set up heartbeat timer }
  FHeartbeat := TTimer.Create(Self);
  FHeartbeat.Interval := BeatInterval;
  FHeartbeat.OnTimer := tmBeat;
  FHeartbeat.Enabled := False;

  { create a byte queue to receive data }
  FByteQueue := TaaByteQueue.Create;

  { ScrollBar Stuff }
  FHScrollBar := TApxTerminalScrollBar.Create (Self);
  FHScrollBar.Kind := sbHorizontal;
  FHScrollBar.Height := CYHSCROLL;
  FHScrollBar.OnScroll := HScroll;

  FVScrollBar := TApxTerminalScrollBar.Create (Self); 
  FVScrollBar.Kind := sbVertical;
  FVScrollBar.Width := CXVSCROLL;
  FVScrollBar.OnScroll := VScroll;

  FHScrollBar.Parent := Self; 
  FHScrollBar.Color := clScrollBar;
  FVScrollBar.Parent := Self;
  FVScrollBar.Color := clScrollBar;

  { make sure the origin is at (1,1) }
  FOriginCol := 1;
  FOriginRow := 1;

  Width := adc_TermWidth;
  Height := adc_TermHeight;

  { fonts - note.  Scroll bars must be created first }
  Font.Name := adc_TermFontName;
  Font.Size := adc_TermFontSize;
  Font.CharSet := adc_TermFontCharSet;
  Font.Pitch := adc_TermFontPitch;
  Font.Color := adc_TermForeColor;
  Color := adc_TermBackColor;

  FCaretX := 0;
  FCaretY := 0;

  SetBounds(Left, Top, Width, Height); 
end;
{--------}
destructor TApxCustomTerminal.Destroy;
begin
  Emulator := nil;
  FEmulator := nil;
  FDefEmulator.Free;
  FDefEmulator := nil;
  FHeartbeat.Free;
  RemoveTermEmuLink(Self, False);
  TaaByteQueue(FByteQueue).Free;
  inherited Destroy;
end;

{--------}
procedure TApxCustomTerminal.ApxPortOpen(var Msg : TAxMessage);
begin
  { note: if FWaitingForComPortOpen is True, Active is also True }
  if FWaitingForComPortOpen then
    tmAttachToComPort;
  Msg.Result := 1;
end;
{--------}
procedure TApxCustomTerminal.ApxPortClose(var Msg : TAxMessage);
begin
  tmDetachFromComPort;
  Msg.Result := 1;
end;
{--------}
procedure TApxCustomTerminal.ApxTermStuff;
var
  DataLen  : integer;
  DataPtr  : pointer;
  PaintRect : TRect;
begin
  { this message is sent by the tmTriggerAvail, and the WriteChar    }
  { WriteString methods; note that if there are two or more          }
  { APX_TERMSTUFF messages in the message queue, the first one to be }
  { accepted will clear the byte queue, so the others need to take   }
  { account of the fact that there may be no data }

  DataLen := TaaByteQueue(FByteQueue).Count;
  if (DataLen > 0) then begin
    DataPtr := TaaByteQueue(FByteQueue).Peek(DataLen);

    { if we are capturing, save the data to file }
    if (Capture = cmOn) then begin
      if (FCaptureStream <> nil) then
        FCaptureStream.Write(DataPtr^, DataLen);
    end;

    { otherwise get the emulator to process the block }
    FEmulator.ProcessBlock(DataPtr, DataLen);
    if (not UseLazyDisplay) and FEmulator.NeedsUpdate then begin
      {hide the caret}
      tmHideCaret (False);
      try
        PaintRect.Left := 0;
        PaintRect.Top := 0;
        PaintRect.Right := ClientCols * CharWidth;
        PaintRect.Bottom := ClientRows * CharHeight;
        Canvas.SetClipRect(PaintRect);
        FEmulator.LazyPaint;
      finally
        tmShowCaret;
      end;
    end;

    {increment the byte count for the lazy painting}
    inc(FLazyByteCount, DataLen);

    {now we've processed all the data, clear the byte queue}
    if (DataLen <> TaaByteQueue(FByteQueue).Count) then
      TaaByteQueue(FByteQueue).Remove(DataLen)
    else
      TaaByteQueue(FByteQueue).Clear;
  end;
end;

function TApxCustomTerminal.EventFilter(Sender : QObjectH; Event : QEventH) : Boolean;
var
  KeyCode: Word;
  KeyChar: Char;
  ShiftState: TShiftState;

  procedure DoCustomDispatchEvent(Event : QCustomEventH; FreeData : Boolean);
  var
    DataReference : PAxMessage;
    Data          : TAxMessage;
  begin
    DataReference := QCustomEvent_data(Event);

    { Local copy of data }
    Data := DataReference^;

    if FreeData then
      Dispose(DataReference);

    try
      Self.Dispatch(Data);
    except
      raise; { Do not swallow exceptions by default }
    end;

    if not FreeData then
      DataReference^.Result := Data.Result;
  end;

begin
  Result := False;

  case QEvent_type(Event) of

    QEventType_AxSendDispatchMessage :
      begin
        DoCustomDispatchEvent(QCustomEventH(Event), False);
        Result := True;
        Exit;
      end;

    QEventType_AxPostDispatchMessage :
      begin
        DoCustomDispatchEvent(QCustomEventH(Event), True);
        Result := True;
        Exit
      end;

    QEventType_KeyPress :
      begin
        if not (csDesigning in ComponentState) then begin
          KeyCode := QKeyEvent_key(QKeyEventH(Event));
          ShiftState := ButtonStateToShiftState(QKeyEvent_state(QKeyEventH(Event)));
          KeyChar := Char(QKeyEvent_ascii(QKeyEventH(Event)));
          Result := ApxKeyDown(KeyCode, KeyChar, ShiftState);
        end
      end;

    QEventType_FocusIn : FocusIn;

    QEventType_FocusOut : FocusOut;

  end;
  if not Result then Result := inherited EventFilter(Sender, Event);
end;

{--------}
procedure TApxCustomTerminal.Clear;
begin
  FEmulator.teClear;
end;
{--------}
procedure TApxCustomTerminal.ClearAll;
begin
  FEmulator.teClearAll;
end;
{--------}
procedure TApxCustomTerminal.FontChanged;
begin
  { make sure our ancestors do their stuff }
  inherited;
  { if we have no handle, there's nothing we can do }
  if not HandleAllocated then Exit;
  { otherwise, work out the character size }
  tmGetFontInfo;
end;
{--------}
function TApxCustomTerminal.ApxKeyDown(var KeyCode : Word;
  var KeyChar : Char; var ShiftState : TShiftState) : Boolean;
begin
  Result := False;
  { if the left button is down (we're selecting) and the key is Escape, }
  { cancel the mode }
  if FLButtonDown and (KeyCode = Key_Escape) then begin
    CancelMode;
    Exit;
  end;

  { if we have a selection and the key is Ctrl+C or Ctrl+Insert, copy }
  { the selection to the clipboard }
  if tmProcessClipboardCopy(KeyCode, ShiftState) then Exit;

  { ignore all shift and supershift keys }
  if (KeyCode = Key_Shift) or (KeyCode = Key_Control) or
    (KeyCode = Key_Alt) then Exit;

  { pass the key, etc, onto the emulator }
  FEmulator.KeyDown(KeyCode, ShiftState);

  { if the emulator successfully processed the key, let the caller }
  { know that it's been processed }
  if (KeyCode = 0) then
    Result := True;
end;
{--------}
procedure TApxCustomTerminal.CopyToClipboard;
var
  i         : integer;
  CharCount : integer;
  StartCol  : integer;
  ColCount  : integer;
  CurInx    : integer;
  RowText   : PAnsiChar;
  TextPtr   : PAnsiChar;
begin
  { calculate the amount of text in the selection; this equals the }
  { number of characters for each line, plus CR/LF per line }
  ColCount := FEmulator.Buffer.ColCount;
  CharCount := 0;
  StartCol := FLButtonRect.Left;
  for i := FLButtonRect.Top to pred(FLButtonRect.Bottom) do begin
    CharCount := CharCount + (ColCount - StartCol + 1) + 2;
    StartCol := 1;
  end;
  CharCount := CharCount + FLButtonRect.Right + 2;
  { allocate enough memory to store this set of characters, plus the }
  { terminating null }
  GetMem(TextPtr, CharCount + 1);
  { copy all the selected characters }
  CurInx := 0;
  StartCol := FLButtonRect.Left;
  for i := FLButtonRect.Top to pred(FLButtonRect.Bottom) do begin
    RowText := FEmulator.Buffer.GetLineCharPtr(i);
    Move(RowText[StartCol - 1], TextPtr[CurInx],
         ColCount - StartCol + 1);
    Inc(CurInx, ColCount - StartCol + 1);
    TextPtr[CurInx] := ^M;
    TextPtr[CurInx+1] := ^J;
    Inc(CurInx, 2);
    StartCol := 1;
  end;
  RowText := FEmulator.Buffer.GetLineCharPtr(FLButtonRect.Bottom);
  Move(RowText[StartCol - 1], TextPtr[CurInx],
       FLButtonRect.Right - StartCol + 1);
  Inc(CurInx, FLButtonRect.Right - StartCol + 1);
  {$IFDEF MSWINDOWS}
  TextPtr[CurInx] := ^M;
  Inc(CurInx);
  {$ENDIF}
  TextPtr[CurInx] := ^J;
  Inc(CurInx);
  TextPtr[CurInx] := #0;

  Clipboard.AsText := TextPtr;
end;
{--------}
procedure TApxCustomTerminal.CreateWidget;
  {------}
  function FindComPort(aForm : TWidgetControl) : TApxCustomComPort;
    { -Search for an existing TComPort }
  var
    i : Integer;
  begin
    for i := 0 to pred(aForm.ComponentCount) do
      if aForm.Components[i] is TApxCustomComPort then begin
        Result := TApxCustomComPort(aForm.Components[i]);
        Exit;
      end;
    Result := nil;
  end;
  {------}
var
  ParentForm : TWidgetControl;
begin
  { call ancestor method to create handle and hooks }
  inherited CreateWidget;
  { Hopefully, a handle was created... }
  if HandleAllocated then begin
    { if we don't have a com port, go find one }
    if (ComPort = nil) then begin
      ParentForm := GetParentForm(Self);
      if (ParentForm <> nil) then
        FComPort := FindComPort(ParentForm);
    end;
    { check to see if we're attached to a comport, if so, register }
    if (ComPort <> nil) and
       not (csDesigning in ComponentState) then begin
      ComPort.RegisterUser(Handle);
      if Active then
        tmAttachToComPort;
    end;
    { start the heartbeat }
    FHeartbeat.Enabled := True;
    if (csDesigning in ComponentState) then
      tmDrawDefaultText;
  end;
end;
{--------}
procedure TApxCustomTerminal.DestroyWidget;
begin
  if HandleAllocated then begin
    { check to see if we're attached to a comport, if so, deregister }
    if (ComPort <> nil) and
       not (csDesigning in ComponentState) then begin
      tmDetachFromComPort;
      ComPort.DeregisterUser(Handle);
    end;
    { stop the heartbeat }
    FHeartbeat.Enabled := False;
  end;
  inherited DestroyWidget;
end;
{--------}
procedure TApxCustomTerminal.HideSelection;
var
  i        : integer;
  ColCount : integer;
begin
  { clear the current selection }
  ColCount := FEmulator.Buffer.ColCount;
  tmMarkDeselected(FLButtonRect.Top, FLButtonRect.Left, ColCount);
  for i := succ(FLButtonRect.Top) to pred(FLButtonRect.Bottom) do
    tmMarkDeselected(i, 1, ColCount);
  tmMarkDeselected(FLButtonRect.Bottom, 1, FLButtonRect.Right);
  { initialize the selection variables }
  FLButtonRect := Rect (0, 0, 0, 0);
  FLButtonAnchor.X := 0;
  FLButtonAnchor.Y := 0;
  FHaveSelection := False;
end;
{--------}
procedure TApxCustomTerminal.KeyDown(var Key : word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if (not WantAllKeys) and (Key <> 0) then begin
    FEmulator.KeyDown(Key, Shift);
  end;
end;
{--------}
procedure TApxCustomTerminal.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if (Key <> #0) then
    FEmulator.KeyPress(Key);
end;
{--------}
procedure TApxCustomTerminal.MouseDown(Button : TMouseButton;
                                      Shift  : TShiftState;
                                      X, Y   : integer);
begin
  inherited MouseDown(Button, Shift, X, Y);
  if (Button = mbLeft) then begin
    if not Focused and CanFocus then
      SetFocus;
    FLButtonDown := True;
    { force mouse position into bounds }
    X := BoundI(0, X, pred(ClientCols * CharWidth));
    Y := BoundI(0, Y, pred(ClientRows * CharHeight));
    { start the selection }
    tmStartSelect(X, Y);
  end;
end;
{--------}
procedure TApxCustomTerminal.MouseMove(Shift : TShiftState; X, Y : integer);
begin
  inherited MouseMove(Shift, X, Y);
  if FLButtonDown then begin
    { force mouse position into bounds }
    X := BoundI(0, X, pred(ClientCols * CharWidth));
    Y := BoundI(0, Y, pred(ClientRows * CharHeight));
    { update the selection }
    tmGrowSelect(X, Y);
  end;
end;
{--------}
procedure TApxCustomTerminal.MouseUp(Button : TMouseButton;
                              Shift : TShiftState; X, Y : integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
  if FLButtonDown and (Button = mbLeft) then begin
    FLButtonDown := False;
  end;
end;
{--------}
procedure TApxCustomTerminal.Notification(AComponent : TComponent;
                                         Operation  : TOperation);
begin
  if (Operation = opRemove) then begin
    if (AComponent = Emulator) then
      Emulator := nil
    else if (AComponent = ComPort) then begin
      Active := False;
      ComPort := nil;
    end;
  end
  else {Operation = opInsert} begin
    if (AComponent is TApxTerminalEmulator) then begin
      if (Emulator = nil) then
        Emulator := TApxTerminalEmulator(AComponent);
    end
    else if (AComponent is TApxCustomComPort) then begin
      if (ComPort = nil) then
        ComPort := TApxCustomComPort(AComponent);
    end;
  end;
  inherited Notification(AComponent, Operation);
end;
{--------}
function TermIntersectRect(var lprcDst : TRect;
                         const lprcSrc1, lprcSrc2: TRect) : Boolean;
begin
  Result := IntersectRect(lprcDst, lprcSrc1, lprcSrc2);
end;
{--------}
procedure TApxCustomTerminal.Paint;
var
  DirtyRect : TRect;
  PaintRect : TRect;
  {$IFDEF DebugGrid}
  OldPenColor : TColor;
  OldFontSize : Integer;
  OldPenStyle : TPenStyle;
  i : Integer;
  {$ENDIF}

  procedure DrawScrollBarsCorner;  
  var
    ARect : TRect;
  begin
    if FHScrollBar.Visible and FVScrollBar.Visible then
    begin
      ARect := Rect(0, 0, FVScrollBar.Width, FHScrollBar.Height);
      OffsetRect(ARect, FVScrollBar.Left, FHScrollBar.Top);
      Canvas.Brush.Color := clScrollBar;
      Canvas.Brush.Style := bsSolid;
      Canvas.FillRect(ARect);
    end;
  end;

begin
  { hide the caret }
  tmHideCaret (False);
  try
    { get the current clip rect--ie, the rect that needs painting }
    DirtyRect := Canvas.ClipRect;

    { paint any unused bits of that rect }
    if TermIntersectRect(PaintRect, DirtyRect, FUnusedRightRect) then begin
      Canvas.Brush.Color := Color;
      Canvas.FillRect(PaintRect);
    end;
    if TermIntersectRect(PaintRect, DirtyRect, FUnusedBottomRect) then begin
      Canvas.Brush.Color := Color;
      Canvas.FillRect(PaintRect);
    end;
    { find out if there is anything else to paint }
    if TermIntersectRect(PaintRect, DirtyRect, FUsedRect) then begin
      { create a clipping region and select it into the canvas, so that }
      { the emulator only gets to know about bits it can paint }
      { tell the emulator to paint what it has }
      PaintRect.Left := 0;
      PaintRect.Top := 0;
      PaintRect.Right := ClientCols * CharWidth;
      PaintRect.Bottom := ClientRows * CharHeight;

      { Adjust for scroll bars }
      if (FVScrollBar.Visible) and
         (PaintRect.Right > Width - FVScrollBar.Width) then
        Dec (PaintRect.Right, CharWidth);
      if (FHScrollBar.Visible) and
         (PaintRect.Bottom > Height - FHScrollBar.Height) then
        Dec (PaintRect.Bottom, CharHeight);


      Canvas.SetClipRect (PaintRect);
      FEmulator.Paint
    end;

    {$IFDEF DebugGrid}
    OldPenColor := Canvas.Pen.Color;
    OldPenStyle := Canvas.Pen.Style;
    Canvas.Pen.Color := clGreen;
    Canvas.Pen.Style := psDot;
    for i := 1 to Columns do begin
      Canvas.MoveTo (i * CharWidth, 0);
      Canvas.LineTo (i * CharWidth, CharHeight * Rows);
    end;
    for i := 1 to Rows do begin
      Canvas.MoveTo (0, i * CharHeight);
      Canvas.LineTo (Columns * CharWidth, CharHeight * i);
    end;
    Canvas.Pen.Color := clRed;
    OldFontSize := Canvas.Font.Size;
    Canvas.Font.Size := 8;
    for i := 1 to Columns do
      Canvas.TextOut (i * CharWidth - Canvas.TextWidth (IntToStr(i)), CharHeight * Rows, IntToStr(i));
    for i := 1 to Rows do
      Canvas.TextOut (Columns * CharWidth + 2, CharHeight * (i - 1), IntToStr(i));
    Canvas.Pen.Color := OldPenColor;
    Canvas.Pen.Style := OldPenStyle;
    Canvas.Font.Size := OldFontSize;
    {$ENDIF}

    { Reset clipping rectangle so the corner can be drawn }
    PaintRect.Left := 0;
    PaintRect.Top := 0;
    PaintRect.Right := Width;
    PaintRect.Bottom := Height;
    Canvas.SetClipRect (PaintRect); 
    DrawScrollBarsCorner; 
  finally
    { show the caret again }
    tmShowCaret;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmAttachToComPort;
begin
  { Assumptions: not designtime                  }
  {              ComPort <> nil                  }
  {              Active = True                   }
  {              FWaitingForComPort = don't care }
  if (not FTriggerHandlerOn) and
     (ComPort.AutoOpen or ComPort.Open) then begin
    FWaitingForComPortOpen := False;
    if not ComPort.Open then
      ComPort.Open := True;
    ComPort.Dispatcher.RegisterEventTriggerHandler(tmTriggerHandler);
    FTriggerHandlerOn := True;
  end
  else
    FWaitingForComPortOpen := True;
end;
{--------}
procedure TApxCustomTerminal.tmBeat(Sender: TObject);
var
  MousePt  : TPoint;
  PaintRect : TRect;
  ScrollPos : integer;
begin
  { if the left mouse button is down we may need to scroll to help the }
  { user select text }
  if FLButtonDown then begin
    GetCursorPos(MousePt);
    MousePt := ScreenToClient(MousePt);
    if FUseHScrollBar then
      if (MousePt.X < 0) then begin
        ScrollPos := FHScrollBar.Position;
        FHScrollBar.Scroll (scLineUp, ScrollPos);
      end
      else if (MousePt.X >= ClientWidth) then begin
        ScrollPos := FHScrollBar.Position;
        FHScrollBar.Scroll (scLineDown, ScrollPos);
      end;
    if (MousePt.Y < 0) then begin
      ScrollPos := FVScrollBar.Position;
      FVScrollBar.Scroll (scLineUp, ScrollPos);
    end
    else if (MousePt.Y >= ClientHeight) then begin
      ScrollPos := FVScrollBar.Position;
      FVScrollBar.Scroll (scLineDown, ScrollPos);
    end;
    MouseMove([ssLeft], MousePt.X, MousePt.Y);
  end;
  { only blink if the blink time is not zero }
  if (BlinkTime > 0) then begin
    inc(FBlinkTimeCount, BeatInterval);
    if (FBlinkTimeCount > BlinkTime) then begin
      FBlinkTextVisible := not FBlinkTextVisible;
      if FEmulator.HasBlinkingText then begin
        tmHideCaret (False);
        try
          PaintRect.Left := 0;
          PaintRect.Top := 0;
          PaintRect.Right := ClientCols * CharWidth;
          PaintRect.Bottom := ClientRows * CharHeight;
          Canvas.SetClipRect (PaintRect);
          FEmulator.BlinkPaint(FBlinkTextVisible);
        finally
          tmShowCaret;
        end;
      end;
      FBlinkTimeCount := 0;
      FCaretBlinkOn := not FCaretBlinkOn;
      tmPositionCaret;
    end;
  end;

  { only use the lazy display option if requested to }
  if UseLazyDisplay then begin
    { check for a lazy display time or byte count }
    inc(FLazyTimeCount, BeatInterval);
    if FEmulator.NeedsUpdate and
       ((FLazyTimeCount > FLazyTimeDelay) or
        (FLazyByteCount > FLazyByteDelay)) then begin
      tmHideCaret (False);
      try
        PaintRect.Left := 0;
        PaintRect.Top := 0;
        PaintRect.Right := ClientCols * CharWidth;
        PaintRect.Bottom := ClientRows * CharHeight;
        Canvas.SetClipRect (PaintRect);
        FEmulator.LazyPaint;
      finally
        tmShowCaret;
      end;
      FLazyTimeCount := 0;
      FLazyByteCount := 0;
    end;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmCalcExtent;
var
  WorkColCount : integer;
  WorkRowCount : integer;
begin
  { ASSUMPTION: there is a handle allocated, but an emulator may not be }
  {             attached. Since there is a handle, we've already        }
  {             calculated the character width and height.              }

  { first calculate the max number of columns and rows in the client area }
  if (FEmulator = nil) then begin
    FClientCols := 0;
    FClientRows := 0;
    Exit;
  end;

  WorkColCount := Columns;
  WorkRowCount := Rows;

  FClientCols := (ClientWidth + pred(FCharWidth)) div FCharWidth;
  FClientCols := MinI(FClientCols, WorkColCount - ClientOriginCol + 1);
  FClientFullCols := ClientWidth div FCharWidth;
  FClientFullCols := MinI(FClientFullCols, WorkColCount - ClientOriginCol + 1);

  FClientRows := (ClientHeight + pred(FCharHeight)) div FCharHeight;
  FClientRows := MinI(FClientRows, WorkRowCount - ClientOriginRow + 1);
  FClientFullRows := ClientHeight div FCharHeight;
  FClientFullRows := MinI(FClientFullRows, WorkRowCount - ClientOriginRow + 1);

  { if there is a handle for the terminal, we have to worry about }
  { whether the scrollbars are visible or not }
  if HandleAllocated then begin
    { if the width is too small, force a horizontal scroll bar; if too }
    { large, force the horizontal scroll bar off }
    if (FClientFullCols < WorkColCount) then begin
      if not FUseHScrollBar then begin
        FUseHScrollBar := True;
        Exit;
      end
    end
    else begin
      if FUseHScrollBar then begin
        FUseHScrollBar := False;
        Exit;
      end
    end;
    { if the height is too small, force a vertical scroll bar; if too }
    { large, force the vertical scroll bar off }
    if (FClientFullRows < WorkRowCount) then begin
      if not FUseVScrollBar then begin
        FUseVScrollBar := True;
        { note: if we have scrollback rows we already have a vertical scrollbar }
        if (ScrollbackRows <= Rows) then begin
          Exit;
        end
      end
    end
    else begin
      if FUseVScrollBar then begin
        FUseVScrollBar := False;
        Exit;
      end
    end;
  end;

  { now calculate the used and unused area rects }
  FUsedRect := Rect(0, 0, ClientWidth, ClientHeight);
  FUnusedRightRect := FUsedRect;
  FUnusedBottomRect := FUsedRect;

  FUsedRect.Right := MinI(ClientCols * CharWidth, ClientWidth);
  FUsedRect.Bottom := MinI(ClientRows * CharHeight, ClientHeight);

  FUnusedRightRect.Left := FUsedRect.Right;
  FUnusedRightRect.Bottom := FUsedRect.Bottom;
  FUnusedBottomRect.Top := FUsedRect.Bottom;

  { if we are using a horizontal scroll bar, set the range }
  if FUseHScrollBar then begin
    tmInitHScrollBar;
  end;

  { if we are using a vertical scroll bar, set the range }
  if FUseVScrollBar or (ScrollbackRows > Rows) then begin
    tmInitVScrollBar;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmDetachFromComPort;
begin
  { Assumptions: ComPort <> nil }
  if FTriggerHandlerOn then begin
    ComPort.Dispatcher.DeregisterEventTriggerHandler(tmTriggerHandler);
    FTriggerHandlerOn := False;
  end;
  if Active then
    FWaitingForComPortOpen := True;
end;
{--------}
procedure TApxCustomTerminal.tmDrawDefaultText;
const
  FormatMost = 'Design mode; line %.2d'^M^J;
  FormatLast = 'Design mode; line %.2d'^M;
  VT100Most = #27'[0;%dmDesign mode; line %.2d'^M^J;
  VT100Last = #27'[0;%dmDesign mode; line %.2d'^M;
var
  i  : integer;
  S  : string;
  VTColor : integer;
begin
  { ASSUMPTION: Handle is allocated }

  if (FByteQueue = nil) or
     (FEmulator.Buffer = nil) then
    Exit;

  for i := 1 to ScrollbackRows do
    WriteString(^M^J);

  if (FEmulator is TApxVT100Emulator) then begin
    VTColor := 31;
    for i := 1 to pred(Rows) do begin
      S := Format(VT100Most, [VTColor, i]);
      WriteString(S);
      inc(VTColor);
      if (VTColor = 38) then
        VTColor := 31;
    end;
    S := Format(VT100Last, [VTColor, Rows]);
    WriteString(S);
  end
  else begin
    for i := 1 to pred(Rows) do begin
      S := Format(FormatMost, [i]);
      WriteString(S);
    end;
    S := Format(FormatLast, [Rows]);
    WriteString(S);
  end;
  Invalidate;
end;
{--------}
function TApxCustomTerminal.tmGetAttributes(aRow, aCol : integer) : TApxTerminalCharAttrs;
var
  AttrArray : PByteArray;
begin
  if (FEmulator.Buffer <> nil) then begin
    AttrArray := FEmulator.Buffer.GetLineAttrPtr(aRow);
    Result := TApxTerminalCharAttrs(AttrArray^[pred(aCol)]);
  end
  else
    Result := [];
end;
{--------}
function TApxCustomTerminal.tmGetBackColor(aRow, aCol : integer) : TColor;
var
  ColorArray : PApxtLongArray;
begin
  if (FEmulator.Buffer <> nil) then begin
    ColorArray := FEmulator.Buffer.GetLineBackColorPtr(aRow);
    Result := TColor(ColorArray^[pred(aCol)]);
  end
  else
    Result := adc_TermBackColor;
end;
{--------}
function TApxCustomTerminal.tmGetCharSet(aRow, aCol : integer) : byte;
var
  CharSetArray : PByteArray;
begin
  if (FEmulator.Buffer <> nil) then begin
    CharSetArray := FEmulator.Buffer.GetLineCharSetPtr(aRow);
    Result := CharSetArray^[pred(aCol)];
  end
  else
    Result := 0;
end;
{--------}
function TApxCustomTerminal.tmGetColumns : integer;
begin
  if (FEmulator.Buffer <> nil) then
    Result := FEmulator.Buffer.ColCount
  else
    Result := 0;
end;
{--------}
function TApxCustomTerminal.tmGetEmulator : TApxTerminalEmulator;
begin
  if (FEmulator = FDefEmulator) then
    Result := nil
  else
    Result := FEmulator;
end;
{--------}
procedure TApxCustomTerminal.tmGetFontInfo;
var
  i               : integer;
  Metrics         : QFontMetricsH;
  SizeChanged     : Boolean;
  NewValue        : integer;
  TempFont        : TFont;
  FontList        : TStringList;
  {------}
  procedure UpdateCharSizes;
  begin
    with Metrics do begin
      { calculate the height }
      NewValue := QFontMetrics_lineSpacing(Metrics);
      if (NewValue > FCharHeight) then begin
        SizeChanged := True;
        FCharHeight := NewValue;
      end;
      { calculate the width }
      NewValue := QFontMetrics_maxWidth(Metrics);
      if (NewValue > FCharWidth) then begin
        SizeChanged := True;
        FCharWidth := NewValue;
      end;
    end;
  end;
  {------}
begin
  if (Assigned (FEmulator)) and (FEmulator.CharSetMapping <> nil) then begin
    FontList := TStringList.Create;
    try
      FEmulator.CharSetMapping.GetFontNames(FontList);
    except
      FontList.Free;
      raise;
    end;
  end
  else
    FontList := nil;

  { get a DC, in order that we can get the font metrics for the current }
  { font; release the DC afterwards }

  TempFont := nil;
  FCharWidth := 0;
  FCharHeight := 0;
  try
    { obtain the character cell height and width from the metrics }
    SizeChanged := False;
    if (FontList = nil) then begin
      Metrics := QFontMetrics_create (Font.Handle);
      try
        UpdateCharSizes;
      finally
        QFontMetrics_destroy (Metrics);
      end;
    end
    else begin
      TempFont := TFont.Create;
      TempFont.Assign(Font);
      TempFont.CharSet := DEFAULT_CHARSET;
      for i := 0 to pred(FontList.Count) do begin
        if (FontList[i] = DefaultFontName) then
          TempFont.Name := Font.Name
        else
          TempFont.Name := FontList[i];
        Metrics := QFontMetrics_create (TempFont.Handle);
        try
          UpdateCharSizes;
        finally
          QFontMetrics_destroy (Metrics);
        end;
      end;
    end;
    { if either the width or height changed, invalidate the display }
    if HandleAllocated then begin
      if SizeChanged then begin
        tmCalcExtent;
        Invalidate;
      end;
    end;
  finally
    FontList.Free;
    TempFont.Free;
  end;
end;
{--------}
function TApxCustomTerminal.tmGetForeColor(aRow, aCol : integer) : TColor;
var
  ColorArray : PApxtLongArray;
begin
  if (FEmulator.Buffer <> nil) then begin
    ColorArray := FEmulator.Buffer.GetLineForeColorPtr(aRow);
    Result := TColor(ColorArray^[pred(aCol)]);
  end
  else
    Result := clWhite;
end;
{--------}
function TApxCustomTerminal.tmGetLine(aRow : integer) : string;
var
  CharArray : PAnsiChar;
begin
  if (FEmulator.Buffer <> nil) then begin
    CharArray := FEmulator.Buffer.GetLineCharPtr(aRow);
    SetLength(Result, FEmulator.Buffer.ColCount);
    Move(CharArray^, Result[1], FEmulator.Buffer.ColCount);
  end
  else
    Result := '';
end;
{--------}
function TApxCustomTerminal.tmGetRows : integer;
begin
  if (FEmulator.Buffer <> nil) then
    Result := FEmulator.Buffer.RowCount
  else
    Result := 0;
end;
{--------}
function TApxCustomTerminal.tmGetScrollbackRows : integer;
begin
  if (FEmulator.Buffer <> nil) then
    Result := FEmulator.Buffer.SVRowCount
  else
    Result := 0;
end;
{--------}
procedure TApxCustomTerminal.tmGrowSelect(X, Y : integer);
var
  NewRow   : integer;
  NewCol   : integer;
  StartCol : integer;
  ColCount : integer;
  NewRgn   : TRect;
  i        : integer;
begin
  { if this gets called, we have a selection }
  FHaveSelection := True;

  { get the column count as a local variable: we'll be using it a lot }
  ColCount := FEmulator.Buffer.ColCount;

  { X and Y are mouse coordinates on the terminal display; convert to a }
  { row/col position }
  NewCol := (X div CharWidth) + ClientOriginCol;
  NewCol := MinI(MaxI(NewCol, 1), ColCount);
  NewRow := (Y div CharHeight) + ClientOriginRow;
  with FEmulator.Buffer do
    NewRow := MinI(MaxI(NewRow, RowCount-SVRowCount), RowCount);

  { generate the new selected region }
  if (NewRow < FLButtonAnchor.Y) then begin
    NewRgn.Top := NewRow;
    NewRgn.Left := NewCol;
    NewRgn.Bottom := FLButtonAnchor.Y;
    NewRgn.Right := FLButtonAnchor.X;
  end
  else if (NewRow > FLButtonAnchor.Y) then begin
    NewRgn.Top := FLButtonAnchor.Y;
    NewRgn.Left := FLButtonAnchor.X;
    NewRgn.Bottom := NewRow;
    NewRgn.Right := NewCol;
  end
  else { NewRow and the anchor row are the same } begin
    NewRgn.Top := NewRow;
    NewRgn.Bottom := NewRow;
    if (NewCol <= FLButtonAnchor.X) then begin
      NewRgn.Left := NewCol;
      NewRgn.Right := FLButtonAnchor.X;
    end
    else begin
      NewRgn.Left := FLButtonAnchor.X;
      NewRgn.Right := NewCol;
    end;
  end;

  { check to see how the current selection grew (we need to mark the }
  { new bits as selected) or shrank (the new bits need to be         }
  { deselected) }
  if (NewRgn.Top = NewRgn.Bottom) then begin
    StartCol := FLButtonRect.Left;
    for i := FLButtonRect.Top to pred(FLButtonRect.Bottom) do begin
      tmMarkDeselected(i, StartCol, ColCount);
      StartCol := 1;
    end;
    tmMarkDeselected(FLButtonRect.Bottom, 1, FLButtonRect.Right);
    tmMarkSelected(NewRgn.Top, NewRgn.Left, NewRgn.Right);
  end
  else begin
    if (NewRgn.Top < FLButtonRect.Top) then begin
      { the selection grew upwards }
      StartCol := NewRgn.Left;
      for i := NewRgn.Top to pred(FLButtonRect.Top) do begin
        tmMarkSelected(i, StartCol, ColCount);
        StartCol := 1;
      end;
      tmMarkSelected(FLButtonRect.Top, 1, FLButtonRect.Left);
    end
    else if (NewRgn.Top = FLButtonRect.Top) then begin
      { the selection might have changed on the top row }
      if (NewRgn.Left < FLButtonRect.Left) then
        tmMarkSelected(NewRgn.Top, NewRgn.Left, FLButtonRect.Left)
      else if (NewRgn.Left > FLButtonRect.Left) then
        tmMarkDeselected(NewRgn.Top, FLButtonRect.Left, NewRgn.Left);
    end
    else { new top > old top } begin
      StartCol := FLButtonRect.Left;
      for i := FLButtonRect.Top to pred(NewRgn.Top) do begin
        tmMarkDeselected(i, StartCol, ColCount);
        StartCol := 1;
      end;
      tmMarkDeselected(NewRgn.Top, 1, NewRgn.Left);
    end;
    if (NewRgn.Bottom > FLButtonRect.Bottom) then begin
      { the selection grew downwards }
      StartCol := FLButtonRect.Right;
      tmMarkSelected(FLButtonRect.Bottom, FLButtonRect.Right, ColCount);
      for i := FLButtonRect.Bottom to pred(NewRgn.Bottom) do begin
        tmMarkSelected(i, StartCol, ColCount);
        StartCol := 1;
      end;
      tmMarkSelected(NewRgn.Bottom, 1, NewRgn.Right);
    end
    else if (NewRgn.Bottom = FLButtonRect.Bottom) then begin
      { the selection might have changed on the bottom row }
      if (NewRgn.Right > FLButtonRect.Right) then
        tmMarkSelected(NewRgn.Bottom, FLButtonRect.Right, NewRgn.Right)
      else if (NewRgn.Right < FLButtonRect.Right) then
        tmMarkDeselected(NewRgn.Bottom, NewRgn.Right, FLButtonRect.Right);
    end
    else { new bottom < old bottom } begin
      StartCol := NewRgn.Right;
      for i := NewRgn.Bottom to pred(FLButtonRect.Bottom) do begin
        tmMarkDeselected(i, StartCol, ColCount);
        StartCol := 1;
      end;
      tmMarkDeselected(FLButtonRect.Bottom, 1, FLButtonRect.Right);
    end;
  end;
  FLButtonRect := NewRgn;
end;
{--------}
procedure TApxCustomTerminal.tmHideCaret (Force : Boolean);
begin
  if FShowingCaret then begin
    if Force then begin
      if (CursorType = ctUnderline) then
        QWidget_update (Handle, FCaretX, FCaretY, FCaretX+CharWidth, FCaretY + 2)
      else if (CursorType = ctBlock) then
        QWidget_update (Handle, FCaretX, FCaretY, FCaretX+CharWidth, FCaretY + CharHeight);
    end;
    FShowingCaret := False;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmInitHScrollBar;
begin
  { Assumptions: FUseHScrollBar = True     }
  {              FClientFullCols < Columns }
  if not Assigned (FHScrollBar) then
    Exit;
  with FHScrollBar do begin
    Min := 0;
    Max := Columns - ClientCols + 1; 
    Position := ClientOriginCol - 1;
    Visible := True;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmInitVScrollBar;
begin
  { Assumptions: FUseVScrollBar = True AND FClientFullRows < Rows }
  {              OR ScrollbackRows > Rows }
  if not Assigned (FVScrollBar) then
    Exit;
  if FUseVScrollBar and (not Scrollback) then begin
    with FVScrollBar do begin
      Min := 0;
      Max := Rows - ClientRows + 1;
      Position := ClientOriginRow - 1;
      Visible := True;
    end;
  end
  else begin
    with FVScrollBar do begin
      Min := Rows - ScrollBackRows;
      Max := Rows - 1;
      Position := ClientOriginRow - 1;
      Visible := True;
    end;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmMakeCaret;
begin
  FCreatedCaret := True;
end;
{--------}
procedure TApxCustomTerminal.tmInvalidateRow(aRow : integer);
var
  InvRect : TRect;
begin
  if HandleAllocated and (ClientOriginRow <= aRow) and
     (aRow < ClientOriginRow + ClientRows) then begin
    InvRect.Left := 0;
    InvRect.Top := (aRow - ClientOriginRow) * CharHeight;
    InvRect.Right := ClientWidth;
    InvRect.Bottom := InvRect.Top + CharHeight;
    {$IFDEF LINUX}
    InvalidateRect (InvRect, False);
    {$ELSE}
    InvalidateRect (Handle, @InvRect, False);
    {$ENDIF}
  end;
end;
{--------}
procedure TApxCustomTerminal.tmMarkDeselected(aRow : integer;
                                             aFromCol, aToCol : integer);
var
  AttrArray : PByteArray;
  i         : integer;
  CharAttrs : TApxTerminalCharAttrs;
begin
  { get the attributes for this line }
  AttrArray := FEmulator.Buffer.GetLineAttrPtr(aRow);
  for i := pred(aFromCol) to pred(aToCol) do begin
    CharAttrs := TApxTerminalCharAttrs(AttrArray^[i]);
    Exclude(CharAttrs, tcaSelected);
    AttrArray^[i] := byte(CharAttrs);
  end;
  tmInvalidateRow(aRow);
end;
{--------}
procedure TApxCustomTerminal.tmMarkSelected(aRow : integer;
                                           aFromCol, aToCol : integer);
var
  AttrArray : PByteArray;
  i         : integer;
  CharAttrs : TApxTerminalCharAttrs;
begin
  {get the attributes for this line}
  AttrArray := FEmulator.Buffer.GetLineAttrPtr(aRow);
  for i := pred(aFromCol) to pred(aToCol) do begin
    CharAttrs := TApxTerminalCharAttrs(AttrArray^[i]);
    Include(CharAttrs, tcaSelected);
    AttrArray^[i] := byte(CharAttrs);
  end;
  tmInvalidateRow(aRow);
end;
{--------}
procedure TApxCustomTerminal.tmPositionCaret;
var
  X, Y : integer;
begin
  with FEmulator.Buffer do begin
    if UseAbsAddress then begin
      X := (Col - FOriginCol) * CharWidth;
      Y := (Row - FOriginRow {+ 1} ) * CharHeight;
    end
    else begin
      X := (Col + OriginCol {- 1} - FOriginCol) * CharWidth;
      Y := (Row + OriginRow {- 1} - FOriginRow) * CharHeight;
    end;
  end;
  if (CursorType = ctUnderline) then
    inc(Y, CharHeight - 2);
  if (0 <= X) and (X < ClientWidth) and
     (0 <= Y) and (Y < ClientHeight) then begin
    if FCaretBlinkOn then begin 
      Canvas.Pen.Color := adc_TermForeColor;
      Canvas.Brush.Color := adc_TermForeColor;
    end else begin
      Canvas.Pen.Color := adc_TermBackColor;
      Canvas.Brush.Color := adc_TermBackColor;
    end;
    if (FCaretX <> X) or (FCaretY <> Y) then begin
      if (CursorType = ctUnderline) then
        QWidget_update (Handle, FCaretX, FCaretY, FCaretX+CharWidth, FCaretY + 2)
      else if (CursorType = ctBlock) then
        QWidget_update (Handle, FCaretX, FCaretY, FCaretX+CharWidth, FCaretY + CharHeight);
      FCaretX := X;
      FCaretY := Y;                                                   
    end;
    Canvas.Pen.Width := 1;
    if (CursorType = ctUnderline) then
      Canvas.Rectangle (X, Y, X + CharWidth, Y + 2)
    else if (CursorType = ctBlock) then begin
      Canvas.Rectangle (X, Y, X + CharWidth, Y + CharHeight);
      { Draw text here.... }
    end;
    FShowingCaret := True;
  end;
end;
{--------}
function TApxCustomTerminal.tmProcessClipboardCopy(Key : Word;
  Shift : TShiftState) : Boolean;
begin
  { this method is called from the main keydown methods to check for   }
  { Ctrl+C or Ctrl+Insert, so that the selected text can get copied to }
  { the clipboard }
  Result := False;
  { check whether we have a selection }
  if not FHaveSelection then Exit;
  { check whether the key is a C or an Insert }
  if (Key <> Key_C) and (Key <> Key_Insert) then Exit;
  { see if the Ctrl key is down and the shift and Alt keys are not }
  if (not (ssCtrl in Shift)) or (ssShift in Shift) or (ssAlt in Shift) then
    Exit;
  { we have the required keystroke and a selection: copy to clipboard }
  Result := True;
  CopyToClipboard;
end;
{--------}
procedure TApxCustomTerminal.tmScrollHorz(aDist : integer);
begin
  { aDist is +ve if we are scrolling to the right (ie the current       }
  { window contents are moved to a position aDist pixels to the right), }
  { and is -ve for a leftward scroll }

  { However, since the terminal handles the scrollback buffer and the   }
  { scrolling by using the ClientOriginRow and ClientOriginColumn,      }
  { there is no need to actually scroll the component.  The component   }
  { should have been Invalidated when it was scrolled.  The repainting  }
  { will perform the scroll. }

  Self.SetFocus;
end;
{--------}
procedure TApxCustomTerminal.tmScrollVert(aDist : integer);
begin
  { aDist is +ve if we are scrolling downwards (ie the current window   }
  { contents are moved to a position aDist pixels down), and is -ve for }
  { a upward scroll }

  { However, since the terminal handles the scrollback buffer and the   }
  { scrolling by using the ClientOriginRow and ClientOriginColumn,      }
  { there is no need to actually scroll the component.  The component   }
  { should have been Invalidated when it was scrolled.  The repainting  }
  { will perform the scroll. }

  Self.SetFocus;
end;
{--------}
procedure TApxCustomTerminal.tmSetActive(aValue : Boolean);
begin
  if (aValue <> Active) then begin
    FWaitingForComPortOpen := False;
    FActive := aValue;
    { if we have a comport then either attach to or detach from it }
    if (ComPort <> nil) then begin
      if not (csDesigning in ComponentState) then
        if Active then
          tmAttachToComPort
        else
          tmDetachFromComPort;
    end
    else if Active then
      FWaitingForComPortOpen := True;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetAttributes(aRow, aCol : integer;
  const aAttr : TApxTerminalCharAttrs);
var
  AttrArray : PByteArray;
begin
  if (FEmulator.Buffer <> nil) then begin
    AttrArray := FEmulator.Buffer.GetLineAttrPtr(aRow);
    TApxTerminalCharAttrs(AttrArray^[pred(aCol)]) := aAttr;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetBackColor(aRow, aCol : integer;
  aValue : TColor);
var
  ColorArray : PApxtLongArray;
begin
  if (FEmulator.Buffer <> nil) then begin
    ColorArray := FEmulator.Buffer.GetLineBackColorPtr(aRow);
    TColor(ColorArray^[pred(aCol)]) := aValue;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetBlinkTime(aValue : integer);
begin
  if (aValue <> BlinkTime) then begin
    if (aValue <= 0) then
      aValue := 0
    else if (aValue <= 250) then
      aValue := 250;
    FBlinkTime := aValue;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetBorderStyle(aBS : TControlBorderStyle);
begin
  if (aBS <> BorderStyle) then begin
    FBorderStyle := aBS;
    RecreateWidget;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetCapture(aValue : TApxCaptureMode);
begin
  if (aValue <> Capture) then begin
    if (aValue = cmAppend) and (Capture = cmOn) then
      Exit;
    { if we're capturing now, close the stream }
    if (Capture = cmOn) then begin
      FCaptureStream.Free;
      FCaptureStream := nil;
    end;
    { save the new value }
    FCapture := aValue;
    { if we should now be capturing, open the stream }
    if (Capture <> cmOff) then begin
      if (CaptureFile = '') then
        FCaptureFile := adc_TermCaptureFile;
      if not (csDesigning in ComponentState) then begin
        if (Capture = cmOn) then
          FCaptureStream := TFileStream.Create(CaptureFile, fmCreate)
        else if (Capture = cmAppend) then begin
          FCaptureStream := TFileStream.Create(CaptureFile, fmOpenReadWrite);
          FCaptureStream.Seek(0, soFromEnd);
          FCapture := cmOn;
        end;
      end;
    end;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetCaptureFile(const aValue : string);
var
  WasCapturing : Boolean;
begin
  if (aValue <> CaptureFile) then begin
    { first turn off capturing if needed }
    if (Capture = cmOff) then
      WasCapturing := False
    else begin
      WasCapturing := True;
      Capture := cmOff;
    end;
    { save the new filename }
    FCaptureFile := aValue;
    { now turn capturing back on }
    if WasCapturing then
      Capture := cmOn;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetCharSet(aRow, aCol : integer; aValue : byte);
var
  CharSetArray : PByteArray;
begin
  if (FEmulator.Buffer <> nil) then begin
    CharSetArray := FEmulator.Buffer.GetLineCharSetPtr(aRow);
    CharSetArray^[pred(aCol)] := aValue;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetColumns(aValue : integer);
var
  OldValue : integer;
begin
  if (FEmulator.Buffer <> nil) and
     (aValue > 0) then begin
    OldValue := FEmulator.Buffer.ColCount;
    if (OldValue <> aValue) then begin
      FEmulator.Buffer.ColCount := aValue;
      FOriginCol := 1;
      FOriginRow := 1;
      tmCalcExtent;
      Invalidate;
    end;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetComPort(aValue : TApxCustomComPort);
begin
  if (aValue <> ComPort) then begin
    { if we don't yet have a handle, just save the new value }
    if not HandleAllocated then begin
      if (FComPort <> nil) and (FComPort.MasterTerminal = Self) then
        FComPort.MasterTerminal := nil;
      FComPort := aValue;
      if (FComPort <> nil) and (FComPort.MasterTerminal = nil) then
        FComPort.MasterTerminal := Self;
    end
    { otherwise we need to do some comport registration }
    else begin
      { detach/deregister from the old comport }
      if (ComPort <> nil) and
         not (csDesigning in ComponentState) then begin
        tmDetachFromComPort;
        ComPort.DeregisterUser(Handle);
        if (FComPort.MasterTerminal = Self) then
          FComPort.MasterTerminal := nil;
      end;
      { save the new value }
      FComPort := aValue;
      { register/attach with the new comport }
      if (ComPort <> nil) and
        not (csDesigning in ComponentState) then begin
          ComPort.RegisterUser(Handle);
        if (FComPort.MasterTerminal = nil) then
          FComPort.MasterTerminal := Self;
        if Active then
          tmAttachToComPort;
      end;
    end;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetCursorType(aValue : TApxCursorType);
begin
  if (aValue <> CursorType) then begin
    tmHideCaret (True);
    FCursorType := aValue;
    tmMakeCaret;
    tmShowCaret;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetEmulator(aValue : TApxTerminalEmulator);
var
  OldRowCount : integer;
  OldColCount : integer;
  OldSVRowCount : integer;
begin
  if (aValue <> Emulator) then begin
    if HandleAllocated then
      FHeartbeat.Enabled := False;
    { if we were attached to an emulator, remove its link }
    OldSVRowCount := FEmulator.Buffer.SVRowCount;
    OldRowCount := FEmulator.Buffer.RowCount;
    OldColCount := FEmulator.Buffer.ColCount;
    if (Emulator <> nil) then begin
      FEmulator := nil; { to stop recursion with the next call }
      RemoveTermEmuLink(Self, True);
    end;
    { attach ourselves to the new emulator }
    if (aValue = nil) then
      FEmulator := FDefEmulator
    else begin
      FEmulator := aValue;
      AddTermEmuLink(Self, Emulator);
    end;
    {set the new emulator up}
    if (OldSVRowCount > 0) then begin
      FEmulator.Buffer.SVRowCount := OldSVRowCount;
      FEmulator.Buffer.RowCount := OldRowCount;
      FEmulator.Buffer.ColCount := OldColCount;
    end;
    if HandleAllocated then begin
      FHeartbeat.Enabled := True;
      if (csDesigning in ComponentState) then
        tmDrawDefaultText;
    end;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetForeColor(aRow, aCol : integer;
                                     aValue     : TColor);
var
  ColorArray : PApxtLongArray;
begin
  if (FEmulator.Buffer <> nil) then begin
    ColorArray := FEmulator.Buffer.GetLineForeColorPtr(aRow);
    TColor(ColorArray^[pred(aCol)]) := aValue;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetLazyByteDelay(aValue : integer);
begin
  if (0 < aValue) and (aValue <= 1024) and
     (aValue <> LazyByteDelay) then begin
    FLazyByteDelay := aValue;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetLazyTimeDelay(aValue : integer);
begin
  if (0 < aValue) and (aValue <= 1000) and
     (aValue <> LazyTimeDelay) then begin
    FLazyTimeDelay := aValue;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetLine(aRow : integer; const aValue : string);
var
  CharArray : PAnsiChar;
  StLen     : integer;
  i         : integer;
begin
  if (FEmulator.Buffer <> nil) then begin
    CharArray := FEmulator.Buffer.GetLineCharPtr(aRow);
    StLen := length(aValue);
    if (StLen > FEmulator.Buffer.ColCount) then
      Move(aValue[1], CharArray[0], FEmulator.Buffer.ColCount)
    else begin
      Move(aValue[1], CharArray[0], StLen);
      for i := StLen to pred(FEmulator.Buffer.ColCount) do
        CharArray[i] := ' ';
    end;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetOriginCol(aValue : integer);
var
  MaxOriginCol : integer;
  OldOrigin    : integer;
begin
  if (aValue <> ClientOriginCol) then begin
    { work out the maximum value }
    MaxOriginCol := Columns - FClientFullCols + 1;
    if (MaxOriginCol < 1) then
      MaxOriginCol := 1;
    { save the old value }
    OldOrigin := FOriginCol;
    { set the new value }
    if (aValue < 1) then
      FOriginCol := 1
    else if (aValue > MaxOriginCol) then
      FOriginCol := MaxOriginCol
    else
      FOriginCol := aValue;
    {scroll the window, set the scrollbar position}
    if (OldOrigin <> ClientOriginCol) then begin
      tmScrollHorz((OldOrigin - ClientOriginCol) * CharWidth);
      tmCalcExtent;
      with FHScrollBar do
        Position := ClientOriginCol - 1;
    end;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetOriginRow(aValue : integer);
var
  MinOriginRow : integer;
  MaxOriginRow : integer;
  OldOrigin    : integer;
begin
  if (aValue <> ClientOriginRow) then begin
    { work out the minimum value }
    if FUseVScrollBar and (not Scrollback) then
      MinOriginRow := 1
    else
      MinOriginRow := Rows - ScrollbackRows + 1;
    { work out the maximum value }
    MaxOriginRow := Rows - FClientFullRows + 1;
    if (MaxOriginRow < 1) then
      MaxOriginRow := 1;
    { save the old value }
    OldOrigin := FOriginRow;
    { set the new value }
    if (aValue < MinOriginRow) then
      FOriginRow := MinOriginRow
    else if (aValue > MaxOriginRow) then
      FOriginRow := MaxOriginRow
    else
      FOriginRow := aValue;
    {scroll the window, set the scrollbar position}
    if (OldOrigin <> ClientOriginRow) then begin
      tmScrollVert((OldOrigin - ClientOriginRow) * CharHeight);
      tmCalcExtent;
      FVScrollBar.Position := ClientOriginRow - 1;
      FVScrollBar.Visible := True;
      with FHScrollBar do
        Position := ClientOriginRow - 1;
    end;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetRows(aValue : integer);
var
  OldValue : integer;
begin
  if (FEmulator.Buffer <> nil) and
     (aValue > 0) then begin
    OldValue := FEmulator.Buffer.RowCount;
    if (OldValue <> aValue) then begin
      FEmulator.Buffer.RowCount := aValue;
      tmCalcExtent;
      Invalidate;
    end;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetScrollback(aValue : Boolean);
begin
  if (aValue <> Scrollback) then begin
    FScrollback := aValue;
    tmInitVScrollBar;
  end;
end;
{--------}
procedure TApxCustomTerminal.tmSetScrollbackRows(aValue : integer);
begin
  if (FEmulator.Buffer <> nil) and (aValue > 0) then
    FEmulator.Buffer.SVRowCount := aValue;
end;
{--------}
procedure TApxCustomTerminal.tmSetUseLazyDisplay(aValue : Boolean);
begin
  FUseLazyDisplay := aValue;
end;
{--------}
procedure TApxCustomTerminal.tmSetWantAllKeys(aValue : Boolean);
begin
  if (aValue <> WantAllKeys) then begin
    FWantAllKeys := aValue;
    if FWantAllKeys then
      InputKeys := [ikAll, ikArrows, ikChars, ikReturns, ikTabs, ikEdit, ikNav, ikEsc]
    else
      InputKeys := [ikArrows, ikChars, ikReturns, ikTabs, ikEdit, ikNav, ikEsc];
  end;
end;
{--------}
procedure TApxCustomTerminal.tmShowCaret;
begin
  if (not FShowingCaret) and FCreatedCaret and
     (not (csDesigning in ComponentState)) and
     Focused then 
    tmPositionCaret;
end;
{--------}
procedure TApxCustomTerminal.tmStartSelect(X, Y : integer);
begin
  { clear the current selection }
  if (FLButtonRect.Left > 0) and (FLButtonRect.Top > 0) and
     (FLButtonRect.Bottom > 0) and (FLButtonRect.Right > 0) then
    HideSelection;
  { X and Y are mouse coordinates on the terminal display; convert to a }
  { row/col position }
  FLButtonAnchor.X := (X div CharWidth) + ClientOriginCol;
  FLButtonAnchor.Y := (Y div CharHeight) + ClientOriginRow;
  { reset the current selection }
  FLButtonRect.Left := FLButtonAnchor.X;
  FLButtonRect.Top := FLButtonAnchor.Y;
  FLButtonRect.Right := FLButtonAnchor.X;
  FLButtonRect.Bottom := FLButtonAnchor.Y;
end;
{--------}
procedure TApxCustomTerminal.tmTriggerHandler(Msg, wParam : Cardinal;
                                       lParam : longint);
var
  Buffer : pointer;
begin
  if (Msg = APX_TRIGGERAVAIL) then begin
    { we get all the data available and buffer it elsewhere }
    Buffer := TaaByteQueue(FByteQueue).PutPeek(wParam);
    ComPort.Dispatcher.GetBlock(Buffer, wParam);
    TaaByteQueue(FByteQueue).AdvanceAfterPut(wParam);
    { tell ourselves that we have more data }
    AxPostMessage(Handle, APX_TERMSTUFF, nil, 0, 0);
  end;
end;
{--------}
procedure TApxCustomTerminal.tmUpdateScrollBars;
begin
  FHScrollBar.BeginUpdate;
  FVScrollBar.BeginUpdate;
  try
    FHScrollBar.Visible := FUseHScrollBar;
    FVScrollBar.Visible := FUseVScrollBar;
    FHScrollBar.Top := Height - FHScrollBar.Height - 0 { BorderSize  bs = 2 };
    FHScrollBar.Left := 0 {BorderSize};
    FHScrollBar.Width := Width - 2 * 0 {BorderSize};
    if FVScrollBar.Visible then
      FHScrollBar.Width := FHScrollBar.Width - FVScrollBar.Width;
    FVScrollBar.Top := 0 {BorderSize};
    FVScrollBar.Left := Width - FVScrollBar.Width - 0 {BorderSize};
    FVScrollBar.Height := Height - 2 * 0 {BorderSize};
    if FHScrollBar.Visible then
      FVScrollBar.Height := FVScrollBar.Height - FHScrollBar.Height;
  finally
    FHScrollBar.EndUpdate;
    FVScrollBar.EndUpdate;
  end;
end;
{--------}
procedure TApxCustomTerminal.CancelMode;
begin
  HideSelection;
  FLButtonDown := False;
end;
{--------}
procedure TApxCustomTerminal.HScroll (Sender : TObject;  ScrollCode: TScrollCode;
  var ScrollPos: Integer);
var
  PageSize : integer;
  NewPos   : integer;
  MaxOriginCol : integer;
begin
  NewPos := 0;
  PageSize := FClientFullCols;
  MaxOriginCol := Columns - PageSize + 1;
  case ScrollCode of
    scLineUp   : NewPos := ClientOriginCol - 1;
    scLineDown : NewPos := ClientOriginCol + 1;
    scPageUp   : NewPos := ClientOriginCol - PageSize;
    scPageDown : NewPos := ClientOriginCol + PageSize;
    scPosition : NewPos := ScrollPos + 1;
    scTrack    : NewPos := ScrollPos + 1;
  else
    Exit; { ignore it }
  end;
  if (NewPos < 1) then
    NewPos := 1
  else if (NewPos > MaxOriginCol) then
    NewPos := MaxOriginCol;
  if (NewPos <> ClientOriginCol) then
    ClientOriginCol := NewPos;

  { Invalidating the component will cause it to repaint.  Scrolling will be }
  { handled when the component repaints }
  tmUpdateScrollBars;
  Invalidate;
end;
{--------}
procedure TApxCustomTerminal.BoundsChanged;
begin
  inherited;
  tmUpdateScrollBars;
end;
{--------}
procedure TApxCustomTerminal.FocusOut;
begin
  tmHideCaret (False);
end;
{--------}
procedure TApxCustomTerminal.FocusIn;
begin
  if not FCreatedCaret then
    tmHideCaret (False);
    tmMakeCaret;
  if not FShowingCaret then
    tmShowCaret;
end;
{--------}
function TApxCustomTerminal.WidgetFlags: Integer;
begin
  Result := inherited WidgetFlags or
            Integer(WidgetFlags_WResizeNoErase) or
            Integer(WidgetFlags_WRepaintNoErase);
end;
{--------}
function TApxCustomTerminal.GetClientRect: TRect;
begin
  Result := Inherited GetClientRect;
  { Dec (Result.Right, 2 * BorderSize); }
  { Dec (Result.Bottom, 2 * BorderSize); }
  if FVScrollBar.Visible then
    Dec (Result.Right, FVScrollBar.Width);
  if FHScrollBar.Visible then
    Dec (Result.Bottom, FHScrollBar.Height);
  AdjustForClient (Result);
end;
{--------}
procedure TApxCustomTerminal.Resize;
begin
  inherited Resize;
  if (CharWidth > 0) then begin
    { calculate the new values for the extent of the window }
    tmCalcExtent;
    { if the current origin column is not 1 and we can show more of the }
    { terminal if it were less, then do so }
    if (ClientOriginCol > 1) then begin
      if ((ClientCols * CharWidth) < ClientWidth) then
        ClientOriginCol := Columns - (ClientWidth div CharWidth) + 1;
    end;
    { if the current origin row is not 1 and we can show more of the }
    { terminal if it were less, then do so }
    if (ClientOriginRow > 1) then begin
      if ((ClientRows * CharHeight) < ClientHeight) then
        ClientOriginRow := Rows - (ClientHeight div CharHeight) + 1;
    end;
  end;
  tmUpdateScrollBars; 
  
  { Invalidate the control.  Repainting will handle any scroll bar issues }
  Invalidate;         
end;

{--------}
procedure TApxCustomTerminal.VScroll (Sender : TObject;  ScrollCode: TScrollCode;
  var ScrollPos: Integer);
var
  PageSize : integer;
  NewPos   : integer;
  MinOriginRow : integer;
  MaxOriginRow : integer;
begin
  NewPos := 0;
  { if we've a vertical scrollbar because the window is smaller than }
  { the terminal screen...}
  if FUseVScrollBar then begin
    { if we're not currently going through the scrollback buffer and  }
    { the current origin row is 1 and the user clicks the up arrow on }
    { the scrollbar, auto switch into scrollback mode }
    if (not Scrollback) then begin
      if (ScrollCode = scLineUp) and (ClientOriginRow = 1) then
        Scrollback := True;
    end
    { if we're currently going through the scrollback buffer and the }
    { current origin row is greater than 1, auto switch into         }
    { non-scrollback mode }
    else { Scrollback is True } begin
      if (ClientOriginRow > 1) then
        Scrollback := False;
    end;
  end;

  PageSize := FClientFullRows;
  if FUseVScrollBar and (not Scrollback) then
    MinOriginRow := 1
  else
    MinOriginRow := Rows - ScrollbackRows - 1;
  MaxOriginRow := Rows - PageSize + 1;
  if (MaxOriginRow < 1) then
    MaxOriginRow := 1;
  case ScrollCode of
    scLineUp   : NewPos := ClientOriginRow - 1;
    scLineDown : NewPos := ClientOriginRow + 1;
    scPageUp   : NewPos := ClientOriginRow - PageSize;
    scPageDown : NewPos := ClientOriginRow + PageSize;
    scPosition : NewPos := ScrollPos + 1;
    scTrack    : NewPos := ScrollPos + 1;
  else
    Exit; { if anything else, ignore it }
  end;
  if (NewPos < MinOriginRow) then
    NewPos := MinOriginRow
  else if (NewPos > MaxOriginRow) then
    NewPos := MaxOriginRow;
  if (NewPos <> ClientOriginRow) then
    ClientOriginRow := NewPos;

  { Invalidating the component will cause it to repaint.  Scrolling will be }
  { handled when the component repaints }
  tmUpdateScrollBars;
  Invalidate; 
end;
{--------}
procedure TApxCustomTerminal.WriteChar (aCh : AnsiChar);
begin
  { stuff the data into the byte queue }
  TaaByteQueue (FByteQueue).Put (aCh, sizeof (AnsiChar));
  { tell ourselves that we have more data }
  AxPostMessage(Handle, APX_TERMSTUFF, nil, 0, 0);
end;
{--------}
procedure TApxCustomTerminal.WriteString(const aSt : string);
begin
  { stuff the data into the byte queue }
  TaaByteQueue(FByteQueue).Put(aSt[1], length(aSt));
  { tell ourselves that we have more data }
  AxPostMessage(Handle, APX_TERMSTUFF, nil, 0, 0);
end;
{====================================================================}


{===TApxTerminalEmulator==============================================}
constructor TApxTerminalEmulator.Create(aOwner : TComponent);
begin
  inherited Create(aOwner);
end;
{--------}
destructor TApxTerminalEmulator.Destroy;
begin
  if not FIsDefault then
    Terminal := nil;
  inherited Destroy;
end;
{--------}
procedure TApxTerminalEmulator.BlinkPaint(aVisible : Boolean);
begin
  {do nothing}
end;
{--------}
function TApxTerminalEmulator.HasBlinkingText : Boolean;
begin
  Result := False;
end;
{--------}
procedure TApxTerminalEmulator.KeyDown(var Key : word; Shift: TShiftState);
begin
  {do nothing}
end;
{--------}
procedure TApxTerminalEmulator.KeyPress(var Key : AnsiChar);
begin
  {do nothing}
end;
{--------}
procedure TApxTerminalEmulator.LazyPaint;
begin
  {do nothing}
end;
{--------}
procedure TApxTerminalEmulator.Notification(AComponent : TComponent;
                                           Operation  : TOperation);
begin
  if (Operation = opRemove) and (AComponent = Terminal) then begin
    FTerminal := nil;
  end;
  if (Operation = opInsert) and
     (AComponent is TApxTerminal) and
     (Terminal = nil) then begin
    Terminal := TApxTerminal(AComponent);
  end;
  inherited Notification(AComponent, Operation);
end;
{--------}
procedure TApxTerminalEmulator.Paint;
begin
  {do nothing}
end;
{--------}
procedure TApxTerminalEmulator.ProcessBlock(aData : pointer; aDataLen : longint);
begin
  {do nothing}
end;
{--------}
procedure TApxTerminalEmulator.teClear;
begin
  {do nothing};
end;
{--------}
procedure TApxTerminalEmulator.teClearAll;
begin
  {do nothing};
end;
{--------}
function TApxTerminalEmulator.teGetNeedsUpdate : Boolean;
begin
  Result := Buffer.HasDisplayChanged or Buffer.HasCursorMoved;
end;
{--------}
procedure TApxTerminalEmulator.teHandleCursorMovement (Sender : TObject;
                                                      Row    : Integer;
                                                      Col    : Integer);
{ HandleCursorMovement

  Method is called by the Terminal Buffer's OnCursorMoved event}
begin
  if assigned (FTerminal) then begin
    if assigned (FTerminal.OnCursorMoved) then
      FTerminal.OnCursorMoved (Sender, Row, Col);
  end;
end;                                                                                                                                        

{--------}
procedure TApxTerminalEmulator.teSendChar(aCh : char; aCanEcho : Boolean);
begin
  {Assumptions: aCh is a printable character or is one of the standard
                non-printable characters}
  if (Terminal <> nil) then begin
    if aCanEcho and Terminal.HalfDuplex then
      ProcessBlock(@aCh, 1);
    if (Terminal.ComPort <> nil) and Terminal.ComPort.Open then
      Terminal.ComPort.PutChar(aCh);
  end;
end;
{--------}
procedure TApxTerminalEmulator.teSetTerminal(aValue : TApxCustomTerminal);
var
  OldTerminal : TApxCustomTerminal;
begin
  if (aValue <> Terminal) then begin
    {if we were attached to a terminal, remove its link}
    if (Terminal <> nil) and (not FIsDefault) then begin
      OldTerminal := Terminal;
      FTerminal := nil; {to stop recursion in the following call}
      RemoveTermEmuLink(OldTerminal, True);
    end;
    FTerminal := aValue;
    {attach ourselves to the new terminal}
    if (Terminal <> nil) and (not FIsDefault) then begin
      if (Terminal.Emulator <> nil) and (Terminal.Emulator <> Self) then
        RemoveTermEmuLink(Terminal, True);
      AddTermEmuLink(Terminal, Self);
    end;
  end;
end;
{====================================================================}


{====================================================================}
constructor TApxTTYEmulator.Create(aOwner : TComponent);
begin
  {Note: the buffer *must* be created before the ancestor can perform
         initialization. The reason is that at design time dropping an
         emulator on the form will cause a series of Notification
         calls to take place. This in turn could cause a terminal's
         tmSetEmulator method to be called, which would then set up
         some default text in the emulator's buffer.}

  {create the buffer}
  FTerminalBuffer := TApxTerminalBuffer.Create(False);

  {now let the ancestor do his stuff}
  inherited Create(aOwner);

  {set up the terminal buffer and ourselves}
  FDisplayStrSize := 256; {enough for standard terminals}
  GetMem(FDisplayStr, FDisplayStrSize);
  GetMem(FCellWidths, 255 * sizeof(integer));
  FillChar(FCellWidths^, 255 * sizeof(integer), 0);
  FTerminalBuffer.OnCursorMoved := teHandleCursorMovement;
end;
{--------}
destructor TApxTTYEmulator.Destroy;
begin
  {free the paint node free list}
  ttyFreeAllPaintNodes;
  {free the cell widths array}
  if (FCellWidths <> nil) then
    FreeMem(FCellWidths, 255 * sizeof(integer));           
  {free the display string}
  if (FDisplayStr <> nil) then
    FreeMem(FDisplayStr, FDisplayStrSize);
  {free the internal objects}
  FTerminalBuffer.Free;
  inherited Destroy;
end;
{--------}
procedure TApxTTYEmulator.KeyPress(var Key : AnsiChar);
begin
  {send it to the host}
  if (Key = ^H) then
    Key := #127;
  teSendChar(Key, True);
end;
{--------}
procedure TApxTTYEmulator.LazyPaint;
var
  DirtyRect : TRect;
  Row       : integer;
begin
  {the LazyPaint method is called by the terminal, if and only if
   either of the lazy timers expired; in other words, either a certain
   amount of data has been received by the terminal, or a certain time
   has elapsed since the last update}

  if (Terminal.Color <> Buffer.BackColor) or
     (Terminal.Font.Color <> Buffer.ForeColor) then begin
    Buffer.DefBackColor := Terminal.Color;
    Buffer.DefForeColor := Terminal.Font.Color;
    Buffer.BackColor := Terminal.Color;
    Buffer.ForeColor := Terminal.Font.Color;
    FRefresh := True;
  end;                                                     

  { if we have to refresh the display, do it, and throw away all the 'dirty'
    data }

  if FRefresh then begin
    FRefresh := False;
    for Row := 1 to Buffer.RowCount do
      ttyDrawChars(Row, 1, Buffer.ColCount, True);

    { if it is necessary to force an update, uncomment the following lines
      of code }

    { while Buffer.GetInvalidRect(DirtyRect) do
      QWidget_update (Terminal.Handle,
                     (DirtyRect.Left - 1) * Terminal.CharWidth,
                     (DirtyRect.Top - 1) * Terminal.CharHeight,
                     (DirtyRect.Right - DirtyRect.Left + 1) * Terminal.CharWidth,
                     (DirtyRect.Bottom - DirtyRect.Top + 1) * Terminal.CharHeight); }
  end
  else begin
    { using the 'dirty' data in the buffer, draw the required characters }
    while Buffer.GetInvalidRect(DirtyRect) do begin
      for Row := DirtyRect.Top to DirtyRect.Bottom do 
        ttyDrawChars(Row, DirtyRect.Left, DirtyRect.Right, True);
      QWidget_update (Terminal.Handle,
                      (DirtyRect.Left - 1) * Terminal.CharWidth,
                      (DirtyRect.Top - 1) * Terminal.CharHeight,
                      (DirtyRect.Right - DirtyRect.Left + 1) * Terminal.CharWidth,
                      (DirtyRect.Bottom - DirtyRect.Top + 1) * Terminal.CharHeight); 
    end;
  end;
end;
{--------}
procedure TApxTTYEmulator.Paint;
var
  DirtyRect : TRect;
  Row       : integer;
begin
  { the Paint method is called by the terminal, if and only if the }
  { terminal component received a WM_PAINT message }

  if (Terminal.Color <> Buffer.BackColor) or
     (Terminal.Font.Color <> Buffer.ForeColor) then begin
    Buffer.DefBackColor := Terminal.Color;
    Buffer.DefForeColor := Terminal.Font.Color;
    Buffer.BackColor := Terminal.Color;
    Buffer.ForeColor := Terminal.Font.Color;
    FRefresh := True;
  end;

  {repaint the clip region}
  DirtyRect := Terminal.Canvas.ClipRect;
  with Terminal do begin
    DirtyRect.Left := (DirtyRect.Left div CharWidth) + ClientOriginCol;
    DirtyRect.Right := (pred(DirtyRect.Right) div CharWidth) + ClientOriginCol;
    DirtyRect.Top := (DirtyRect.Top div CharHeight) + ClientOriginRow;
    DirtyRect.Bottom := (pred(DirtyRect.Bottom) div CharHeight) + ClientOriginRow;
  end;
  for Row := DirtyRect.Top to DirtyRect.Bottom do
    ttyDrawChars(Row, DirtyRect.Left, DirtyRect.Right, True);
end;
{--------}
procedure TApxTTYEmulator.ProcessBlock(aData : pointer; aDataLen : longint);
var
  DataAsChar : PAnsiChar;
  i          : integer;
  Ch         : AnsiChar;
begin
  DataAsChar := PAnsiChar (aData);
  for i := 0 to pred (aDataLen) do begin 
    Ch := DataAsChar[i];
    if (Ch < ' ') then
      ttyProcessCommand (Ch)
    else
      Buffer.WriteChar (DataAsChar[i]);
  end;
end;
{--------}
procedure TApxTTYEmulator.teClear;
begin
  Buffer.EraseScreen;
end;
{--------}
procedure TApxTTYEmulator.teClearAll;
begin
  Buffer.EraseAll;
end;
{--------}
function TApxTTYEmulator.teGetNeedsUpdate : Boolean;
begin
  Result := Buffer.HasDisplayChanged or FRefresh or
            Buffer.HasCursorMoved;
end;
{--------}
procedure TApxTTYEmulator.teSetTerminal(aValue : TApxCustomTerminal);
begin
  inherited teSetTerminal(aValue);
end;
{--------}
procedure TApxTTYEmulator.ttyDrawChars(aRow, aStartCol, aEndCol : integer;
  aVisible : Boolean);
var
  ColNum      : integer;
  StartColNum : integer;
  BackColor   : TColor;
  ForeColor   : TColor;
  Attr        : TApxTerminalCharAttrs;
  ForeColors  : PApxtLongArray;
  BackColors  : PApxtLongArray;
  Attrs       : PByteArray;
  Script      : PPaintNode;
  PaintNode   : PPaintNode;
begin
  { ASSUMPTION: aStartCol <= aEndCol }

  { avoid any drawing if the row simply is not visible }
  if (aRow < Terminal.ClientOriginRow) or
     (aRow >= Terminal.ClientOriginRow + Terminal.ClientRows) then
    Exit;
  { same if the range of columns is not visible }
  if (aEndCol < Terminal.ClientOriginCol) or
     (aStartCol >= Terminal.ClientOriginCol + Terminal.ClientCols) then
    Exit;

  { if this point is reached, we have to paint *something* }

  { force the parameter values in range }
  if (aStartCol < Terminal.ClientOriginCol) then
    aStartCol := Terminal.ClientOriginCol;
  if (aEndCol > Terminal.ClientOriginCol + Terminal.ClientCols) then
    aEndCol := Terminal.ClientOriginCol + Terminal.ClientCols;

  { this is the main processing: in general we'll be displaying text in }
  { this section; what will happen here, is that we first generate a    }
  { script of drawing commands (which background color, which text      }
  { color, which attributes, which text), and then we execute the       }
  { script }

  { get the pointers to the colors, the attributes }
  BackColors := Buffer.GetLineBackColorPtr(aRow);
  ForeColors := Buffer.GetLineForeColorPtr(aRow);
  Attrs := Buffer.GetLineAttrPtr(aRow);

  { get the initial values for the display variables }
  BackColor := BackColors^[aStartCol-1];
  ForeColor := ForeColors^[aStartCol-1];
  Attr := TApxTerminalCharAttrs(Attrs^[aStartCol-1]);

  { make a note of the start column }
  StartColNum := aStartCol;
  ColNum := aStartCol;

  { build the script as a stack of paint commands }
  Script := nil;
  while (ColNum < aEndCol) do begin
    { look at the next column }
    inc(ColNum);
    { if any info has changed... }
    if (ForeColor <> ForeColors^[ColNum-1]) or
       (BackColor <> BackColors^[ColNum-1]) or
       (Attr <> TApxTerminalCharAttrs(Attrs^[ColNum-1])) then begin
      { get a new node, initialize it }
      PaintNode := ttyNewPaintNode;
      PaintNode^.pnStart := StartColNum;
      PaintNode^.pnEnd := pred(ColNum);
      PaintNode^.pnFore := ForeColor;
      PaintNode^.pnBack := BackColor;
      PaintNode^.pnAttr := Attr;
      PaintNode^.pnCSet := 0;
      { add it to the script }
      PaintNode^.pnNext := Script;
      Script := PaintNode;
      { save the new values of the variables }
      ForeColor := ForeColors^[ColNum-1];
      BackColor := BackColors^[ColNum-1];
      Attr := TApxTerminalCharAttrs(Attrs^[ColNum-1]);
      StartColNum := ColNum;
    end;
  end;
  { create the final paint command }
  PaintNode := ttyNewPaintNode;
  PaintNode^.pnStart := StartColNum;
  PaintNode^.pnEnd := aEndCol;
  PaintNode^.pnFore := ForeColor;
  PaintNode^.pnBack := BackColor;
  PaintNode^.pnAttr := Attr;
  PaintNode^.pnCSet := 0;
  { add it to the script }
  PaintNode^.pnNext := Script;
  Script := PaintNode;

  { now execute the paint script }
  ttyExecutePaintScript(aRow, Script);
end;
{--------}
procedure TApxTTYEmulator.ttyExecutePaintScript(aRow : integer; aScript : pointer);
var
  Canvas       : TCanvas;
  Font         : TFont;
  Walker, Temp : PPaintNode;
  ForeColor    : TColor;
  BackColor    : TColor;
  CharWidth    : integer;
  CharHeight   : integer;
  OriginCol    : integer;
  TextStrLen   : integer;
  TextChars    : PAnsiChar;
  WorkRect     : TRect;
  Y            : integer;
  R, G, B      : integer;
  Reversed     : Boolean;
  I, L         : Integer;
begin
  { get some values as local variables }
  Canvas := Terminal.Canvas;
  Font := Terminal.Font;
  TextChars := Buffer.GetLineCharPtr(aRow);
  CharHeight := Terminal.CharHeight;
  WorkRect.Top := (aRow - Terminal.ClientOriginRow) * CharHeight;
  WorkRect.Bottom := WorkRect.Top + CharHeight;
  CharWidth := Terminal.CharWidth;
  OriginCol := Terminal.ClientOriginCol;
  { set the cell widths }
  for Y := 0 to pred(Buffer.ColCount) do
    FCellWidths^[Y] := CharWidth;
  { process the script }
  Walker := PPaintNode(aScript);
  while (Walker <> nil) do begin
    { check for reverse }
    Reversed := (tcaReverse in Walker^.pnAttr) xor
                (tcaSelected in Walker^.pnAttr);
    if Reversed then begin
      ForeColor := Walker^.pnBack;
      BackColor := Walker^.pnFore;
    end
    else begin
      ForeColor := Walker^.pnFore;
      BackColor := Walker^.pnBack;
    end;
    { check for invisible }
    if (tcaInvisible in Walker^.pnAttr) then
      ForeColor := BackColor
    { check for bold }
    else if (tcaBold in Walker^.pnAttr) then begin
      if Reversed then begin
        R := MinI (GetRValue (BackColor) + $80, $FF);
        G := MinI (GetGValue (BackColor) + $80, $FF);
        B := MinI (GetBValue (BackColor) + $80, $FF);
        BackColor := RGB (R, G, B);
      end
      else begin
        R := MinI (GetRValue (ForeColor) + $80, $FF);
        G := MinI (GetGValue (ForeColor) + $80, $FF);
        B := MinI (GetBValue (ForeColor) + $80, $FF);
        ForeColor := RGB (R, G, B);
      end;
    end;

    { get the length of the text to display }
    TextStrLen := succ(Walker^.pnEnd - Walker^.pnStart);

    { move the required text to our display string }
    Move(TextChars[Walker^.pnStart - 1], FDisplayStr^, TextStrLen);
    FDisplayStr[TextStrLen] := #0;

    { set the correct background }
    Canvas.Brush.Color := BackColor;

    { calculate the left and right values for the rect }
    WorkRect.Left := (Walker^.pnStart - OriginCol) * CharWidth;
    WorkRect.Right := WorkRect.Left + (TextStrLen * CharWidth);

    { display the bit o'text }
    Canvas.Font := Font;
    Canvas.Font.Color := ForeColor;

    Canvas.FillRect(Rect(WorkRect.Left, WorkRect.Top,
      WorkRect.Right, WorkRect.Bottom));

    { Work with CLX here -- have to paint a char at a time to get them }
    { in their cells in some cases }
    L := Length(FDisplayStr);
    for i := 1 to L do
      if FDisplayStr[i -1] <> ' ' then
        Canvas.TextOut(WorkRect.Left + (i - 1) * CharWidth,
          WorkRect.Top - adc_TermLineSpacing, FDisplayStr[i - 1]);

    {finally, draw the underline and/or strike through}
    Canvas.Pen.Color := ForeColor;
    if (tcaUnderline in Walker^.pnAttr) then begin
      Y := WorkRect.Bottom - 1 - CharHeight;
      Canvas.MoveTo(WorkRect.Left, Y);
      Canvas.LineTo(WorkRect.Right, Y);
    end;
    if (tcaStrikeThrough in Walker^.pnAttr) then begin
      Y := WorkRect.Top + (WorkRect.Bottom - WorkRect.Top) div 2 - CharHeight;
      Canvas.MoveTo(WorkRect.Left, Y);
      Canvas.LineTo(WorkRect.Right, Y);
    end;

    { walk to the next paint node, free this one }
    Temp := Walker;
    Walker := Walker^.pnNext;
    ttyFreePaintNode(Temp);
  end;
end;
{--------}
procedure TApxTTYEmulator.ttyFreeAllPaintNodes;
var
  Walker, Temp : PPaintNode;
begin
  Walker := FPaintFreeList;
  while (Walker <> nil) do begin
    Temp := Walker;
    Walker := Walker^.pnNext;
    Dispose(Temp);
  end;
  FPaintFreeList := nil;
end;
{--------}
procedure TApxTTYEmulator.ttyFreePaintNode(aNode : pointer);
begin
  PPaintNode(aNode)^.pnNext := FPaintFreeList;
  FPaintFreeList := aNode;
end;
{--------}
function TApxTTYEmulator.ttyNewPaintNode : pointer;
begin
  if (FPaintFreeList = nil) then
    New(PPaintNode(Result))
  else begin
    Result := FPaintFreeList;
    FPaintFreeList := PPaintNode(Result)^.pnNext;
  end;
end;
{--------}
procedure TApxTTYEmulator.ttyProcessCommand(aCh : AnsiChar);
begin
  { Assumption: aCh is less than space, i.e., is one of the }
  { unprintable characters }
  case aCh of
    ^G : { bell }
      begin
        {$IFDEF LINUX}
        Beep;
        {$ENDIF}
        {$IFDEF MSWINDOWS}
        MessageBeep(MB_ICONASTERISK);
        {$ENDIF}
      end;
    ^H : { backspace }
      begin
        Buffer.DoBackspace;
      end;
    ^I : { tab }
      begin
        Buffer.DoHorzTab;
      end;
    ^J : { linefeed }
      begin
        Buffer.DoLineFeed;
      end;
    ^M : { carriage return }
      begin
        Buffer.DoCarriageReturn;
      end;
  end;{ case }
end;
{====================================================================}


{====================================================================}
constructor TApxVT100Emulator.Create(aOwner : TComponent);
begin
  { Note: the buffer *must* be created before the ancestor can perform  }
  {       initialization. The reason is that at design time dropping an }
  {       emulator on the form will cause a series of Notification      }
  {       calls to take place. This in turn could cause a terminal's    }
  {       tmSetEmulator method to be called, which would then set up    }
  {       some default text in the emulator's buffer. }

  { create the buffer and parser and keyboard mapper }
  FTerminalBuffer := TApxTerminalBuffer.Create(False);
  FParser := TApxVT100Parser.Create(False);
  FKeyboardMapping := TApxKeyboardMapping.Create;
  FCharSetMapping := TApxCharSetMapping.Create;

  { now let the ancestor do his stuff }
  inherited Create(aOwner);

  { set up our default values }
  ANSIMode := adc_VT100ANSIMode;
  AppKeyMode := adc_VT100AppKeyMode;
  AppKeypadMode := adc_VT100AppKeypadMode;
  AutoRepeat := adc_VT100AutoRepeat;
  FCol132Mode := adc_VT100Col132Mode;
  GPOMode := adc_VT100GPOMode;
  InsertMode := adc_VT100InsertMode;
  Interlace := adc_VT100Interlace;
  NewLineMode := adc_VT100NewLineMode;
  FRelOriginMode := adc_VT100RelOriginMode;
  FRevScreenMode := adc_VT100RevScreenMode;
  SmoothScrollMode := adc_VT100SmoothScrollMode;
  WrapAround := adc_VT100WrapAround;

  FAnswerback := adc_VT100Answerback;
  FG0CharSet := adc_VT100G0CharSet;
  FG1CharSet := adc_VT100G1CharSet;

  FDisplayStrSize := 133; { enough for both 80- and 132-column mode }
  GetMem(FDisplayStr, FDisplayStrSize);

  { make sure we're told about scrolling rows so we can track double }
  { height and double width lines }
  FLineAttrArray := TApxLineAttrArray.Create(Self);
  FTerminalBuffer.OnScrollRows := vttScrollRowsHandler;

  { make sure the mappers are initialized }
  FKeyboardMapping.LoadFromRes(hInstance, 'ADVT100KeyMap');
  FCharSetMapping.LoadFromRes(hInstance, 'ADVT100CharSet');

  { initialize the secondary font--a cache to help charset switches }
  FSecondaryFont := TFont.Create;
  if (Terminal <> nil) then
    FSecondaryFont.Assign(Terminal.Font);
  GetMem(FCellWidths, 132 * sizeof(integer));
  FillChar(FCellWidths^, 132 * sizeof(integer), 0);
  FTerminalBuffer.OnCursorMoved := teHandleCursorMovement;
end;
{--------}
destructor TApxVT100Emulator.Destroy;
begin
  { clear the current blink script, free the blink node free list }
  vttClearBlinkScript;
  vttFreeAllBlinkNodes;
  { free the paint node free list }
  vttFreeAllPaintNodes;
  FreeMem(FCellWidths, 132 * sizeof(integer));
  { free the display string }
  if (FDisplayStr <> nil) then
    FreeMem(FDisplayStr, FDisplayStrSize);
  { free the internal objects }
  FSecondaryFont.Free;
  FLineAttrArray.Free;
  FKeyboardMapping.Free;
  FCharSetMapping.Free;
  FParser.Free;
  FTerminalBuffer.Free;
  inherited Destroy;
end;
{--------}
procedure TApxVT100Emulator.BlinkPaint(aVisible : Boolean);
var
  Walker : PBlinkNode;
begin
  { the BlinkPaint method is called by the terminal, if and only if    }
  { the blink timer expired; in other words, the blinking text must be }
  { displayed in the opposite sense, on or off }

  { read all the nodes in the blink linked list, redraw them }
  Walker := PBlinkNode(FBlinkers);
  while (Walker <> nil) do
    with Walker^ do begin
      vttDrawChars(bnRow, bnStartCh, bnEndCh, aVisible, True);
      Walker := bnNext;
    end;
end;
{--------}
function TApxVT100Emulator.HasBlinkingText : Boolean;
begin
  Result := FBlinkers <> nil;
end;
{--------}
procedure TApxVT100Emulator.KeyDown(var Key : word; Shift: TShiftState);
  {------}
  function CvtHexChar(const S : string; aInx : integer) : AnsiChar;
  var
    Hex : string[3];
    Value : integer;
    ec    : integer;
  begin
    Hex := '$  ';
    Hex[2] := S[aInx];
    Hex[3] := S[aInx+1];
    Val(Hex, Value, ec);
    if (ec = 0) then
      Result := chr(Value)
    else
      Result := '?';
  end;
  {------}
var
  VKKey     : string;
  VTKey     : string;
  VTKeyString : string;
  i         : integer;
  EscChar   : Boolean;
  HexChar1  : Boolean;
  HexChar2  : Boolean;
  IsEscSeq  : Boolean;
  IsRepeat  : Boolean;
  IsNumLock : Boolean;
  Ch        : AnsiChar;
begin
  { check for a repeated key and AutoRepeat is off }
  IsRepeat := (Key and $8000) <> 0;
  if IsRepeat then begin
    if not AutoRepeat then begin
      Key := 0;
      Exit;
    end;
    Key := Key and $7FFF;
  end;
  { make a note in case we hit the numlock key }
  IsNumLock := Key = Key_Numlock;
  { we need to convert this keystroke into a VT100 string to pass back }
  { to the server; this is a three step process... }
  { first, convert the keystroke to its name }
  VKKey := Format('\x%.2x', [Key]);
  VKKey := KeyboardMapping.Get(VKKey);
  { if we can continue, add the shift state in the order shift, ctrl, then alt }
  if (VKKey = '') then
    Exit;
  if ssAlt in Shift then
    VKKey := 'alt+' + VKKey;
  if ssCtrl in Shift then
    VKKey := 'ctrl+' + VKKey;
  if ssShift in Shift then
    VKKey := 'shift+' + VKKey;
  { now convert this into a DEC VT100 keyname }
  VTKey := KeyboardMapping.Get(VKKey);
  { if we managed to convert it to a VT100 key, modify the name to }
  { include the application/cursor key mode }
  if (VTKey = '') then
    Exit;
  { if this is a cursor key the name ends in 'c', replace it with 0, 1, }
  { or 2, depending on whether we're in VT52 mode, ANSI mode with }
  { cursor key mode reset, or ANSI mode with cursor key mode set }
  if (VTKey[length(VTKey)] = 'c') then begin
    if not ANSIMode then
      VTKey[length(VTKey)] := '0'
    else if not AppKeyMode then
      VTKey[length(VTKey)] := '1'
    else
      VTKey[length(VTKey)] := '2';
  end
  { if this is a Keypad key the name ends in 'k', replace it with 3, 4, }
  { 5, or 6, depending on whether we're in VT52 numeric mode, VT52 }
  { application mode, ANSI numeric mode, or ANSI application mode }
  else if (VTKey[length(VTKey)] = 'k') then begin
    if not ANSIMode then
      if not AppKeypadMode then
        VTKey[length(VTKey)] := '3'
      else
        VTKey[length(VTKey)] := '4'
    else
      if not AppKeypadMode then
        VTKey[length(VTKey)] := '5'
      else
        VTKey[length(VTKey)] := '6';
  end;
  { now get the string we need to send to the host for this key }
  VTKeyString := KeyboardMapping.Get(VTKey);
  if (VTKeyString <> '') then begin
    IsEscSeq := False;
    EscChar := False;
    HexChar1 := False;
    HexChar2 := False;
    { interpret the key string }
    for i := 1 to length(VTKeyString) do begin
      { get the next character }
      Ch := VTKeyString[i];
      { if the previous character was a '\' then we're reading an }
      { escaped character, either of the form \e or \xhh }
      if EscChar then begin
        if (Ch = 'e') then begin
          teSendChar(#27, False);
          IsEscSeq := True;
        end
        else if (Ch = 'x') then begin
          HexChar1 := True;
          Ch := CvtHexChar(VTKeyString, i+1);
          if (Ch = #13) then begin
            teSendChar(Ch, True);
            if NewLineMode then
              teSendChar(#10, True);
          end
          else if (Ch = #27) then begin
            teSendChar(#27, False);
            IsEscSeq := True;
          end
          else
            teSendChar(Ch, not IsEscSeq);
        end
        else { it's not \e or \xhh } begin
          teSendChar('\', True);
          teSendChar(Ch, True);
        end;
        EscChar := False;
      end
      { if the previous character was the x in \xhh, ignore this one: }
      { we've already interpreted it with the \x }
      else if HexChar1 then begin
        HexChar1 := False;
        HexChar2 := True;
      end
      { if the previous character was the first h in \xhh, ignore this }
      { one: we've already interpreted it with the \x }
      else if HexChar2 then begin
        HexChar2 := False;
      end
      { if this character is a '\' we're starting an escaped character }
      else if (Ch = '\') then
        EscChar := True
      { otherwise there's nothing special about this character }
      else
        teSendChar(Ch, not IsEscSeq);
    end;
    { we've now fully processed the key, so make sure it doesn't come }
    { back to bite us }
    Key := 0;
    { if this was the NumLock key, we'd better set it on again }
    if IsNumLock then begin
      vttToggleNumLock;
    end;
  end;
end;
{--------}
procedure TApxVT100Emulator.KeyPress(var Key : AnsiChar);
begin
  { on a key press, send the key to the host; for a CR character we   }
  { either send CR (NewLineMode is OFF) or a CR/LF (NewLineMode is on }
  if (Key = #13) then begin
    teSendChar(Key, True);
    if NewLineMode then
      teSendChar(#10, True);
  end
  else
    teSendChar(Key, True);
end;
{--------}
procedure TApxVT100Emulator.LazyPaint;
var
  DirtyRect : TRect;
  Row       : integer;
begin
  { the LazyPaint method is called by the terminal, if and only if      }
  { either of the lazy timers expired; in other words, either a certain }
  { amount of data has been received by the terminal, or a certain time }
  { has elapsed since the last update }

  { if we have to refresh the display, do it, and throw away all the }
  { 'dirty' data }
  if FRefresh then begin
    FRefresh := False;
    for Row := 1 to Buffer.RowCount do
      vttDrawChars(Row, 1, Buffer.ColCount, True, False);

    { If it is neccessary to force an update, uncomment the following line }

    { while Buffer.GetInvalidRect(DirtyRect) do 
      QWidget_update (Terminal.Handle,
                      (DirtyRect.Left - 1) * Terminal.CharWidth,
                      (DirtyRect.Top - 1) * Terminal.CharHeight,
                      (DirtyRect.Right - DirtyRect.Left + 1) * Terminal.CharWidth,
                      (DirtyRect.Bottom - DirtyRect.Top + 1) * Terminal.CharHeight); }
  end
  else begin
    { using the 'dirty' data in the buffer, draw the required characters }
    while Buffer.GetInvalidRect(DirtyRect) do begin
      for Row := DirtyRect.Top to DirtyRect.Bottom do
        vttDrawChars(Row, DirtyRect.Left, DirtyRect.Right, True, False);
      QWidget_update (Terminal.Handle,
                      (DirtyRect.Left - 1) * Terminal.CharWidth,
                      (DirtyRect.Top - 1) * Terminal.CharHeight,
                      (DirtyRect.Right - DirtyRect.Left + 1) * Terminal.CharWidth,
                      (DirtyRect.Bottom - DirtyRect.Top + 1) * Terminal.CharHeight);
    end;
  end;
end;
{--------}
procedure TApxVT100Emulator.Paint;
var
  DirtyRect : TRect;
  Row       : integer;
begin
  { the Paint method is called by the terminal, if and only if the }
  { terminal component received a WM_PAINT message }

  { repaint the clip region }
  DirtyRect := Terminal.Canvas.ClipRect;
  with Terminal do begin
    DirtyRect.Left :=
       (DirtyRect.Left div CharWidth) + ClientOriginCol;
    DirtyRect.Right :=
       (pred(DirtyRect.Right) div CharWidth) + ClientOriginCol;
    DirtyRect.Top :=
       (DirtyRect.Top div CharHeight) + ClientOriginRow;
    DirtyRect.Bottom :=
       (pred(DirtyRect.Bottom) div CharHeight) + ClientOriginRow;
  end;
  for Row := DirtyRect.Top to DirtyRect.Bottom do
    vttDrawChars(Row, DirtyRect.Left, DirtyRect.Right, True, False);
end;
{--------}
procedure TApxVT100Emulator.ProcessBlock(aData : pointer; aDataLen : longint);
var
  DataAsChar : PChar;
  i          : integer;
begin
  DataAsChar := PAnsiChar (aData);
  for i := 0 to pred(aDataLen) do begin
    case Parser.ProcessChar(DataAsChar[i]) of
      pctNone :
        { ignore it };
      pctChar :
        begin
          Buffer.WriteChar(DataAsChar[i]);
        end;
      pct8BitChar:
        begin
          if DisplayUpperASCII then
            Buffer.WriteChar(DataAsChar[i])
          else
            vttProcess8bitChar(DataAsChar[i]);
        end;
      pctPending :
        { nothing to do yet };
      pctComplete :
        vttProcessCommand;
    end;
  end;
  vttCalcBlinkScript;
end;
{--------}
procedure TApxVT100Emulator.teClear;
begin
  Buffer.EraseScreen;
end;
{--------}
procedure TApxVT100Emulator.teClearAll;
begin
  Buffer.EraseAll;
end;
{--------}
function TApxVT100Emulator.teGetNeedsUpdate : Boolean;
begin
  Result := Buffer.HasDisplayChanged or FRefresh or                  
            Buffer.HasCursorMoved;                                     
end;
{--------}
procedure TApxVT100Emulator.teSetTerminal(aValue : TApxCustomTerminal);
begin
  inherited teSetTerminal(aValue);
  if (aValue <> nil) and (aValue.Font <> nil) and
     (FSecondaryFont <> nil) then
    FSecondaryFont.Assign(aValue.Font);
end;

{--------}
procedure TApxVT100Emulator.vttCalcBlinkScript;
var
  RowInx       : integer;
  ColInx       : integer;
  StartColInx  : integer;
  Temp         : PBlinkNode;
  Attrs        : TApxTerminalCharAttrs;
  AttrArray    : PByteArray;
  InBlinkRegion: Boolean;
begin
  if Buffer.HasDisplayChanged then begin
    { clear the current script of blink regions }
    vttClearBlinkScript;
    { search for the current blink regions in the buffer }
    InBlinkRegion := False;
    StartColInx := 0;
    with Buffer do
      for RowInx := Terminal.ClientOriginRow to
                    pred(Terminal.ClientOriginRow + RowCount) do begin
        AttrArray := GetLineAttrPtr(RowInx);
        for ColInx := 0 to pred(ColCount) do begin
          Attrs := TApxTerminalCharAttrs(AttrArray^[ColInx]);
          if InBlinkRegion then begin
            if not (tcaBlink in Attrs) then begin
              InBlinkRegion := False;
              Temp := vttNewBlinkNode;
              Temp^.bnRow := RowInx;
              Temp^.bnStartCh := succ(StartColInx);
              Temp^.bnEndCh := ColInx;
              Temp^.bnNext := PBlinkNode(FBlinkers);
              FBlinkers := Temp;
            end;
          end
          else { not tracking blink region } begin
            if (tcaBlink in Attrs) then begin
              InBlinkRegion := True;
              StartColInx := ColInx;
            end;
          end;
        end;
      end;

    { close off the final blink region if there is one }
    if InBlinkRegion then begin
      Temp := vttNewBlinkNode;
      Temp^.bnRow := Buffer.RowCount;
      Temp^.bnStartCh := succ(StartColInx);
      Temp^.bnEndCh := Buffer.ColCount;
      Temp^.bnNext := PBlinkNode(FBlinkers);
      FBlinkers := Temp;
    end;
  end;
end;
{--------}
procedure TApxVT100Emulator.vttClearBlinkScript;
var
  Walker, Temp : PBlinkNode;
begin
  Walker := PBlinkNode(FBlinkers);
  while (Walker <> nil) do begin
    Temp := Walker;
    Walker := Walker^.bnNext;
    vttFreeBlinkNode(Temp);
  end;
  FBlinkers := nil;
end;
{--------}
procedure TApxVT100Emulator.vttDrawBlinkOffCycle(
                                    aRow, aStartCh, aEndCh : integer);
var
  ChNum      : integer;
  BackColor  : TColor;
  BackColors : PApxtLongArray;
  WorkRect   : TRect;
  CharWidth  : integer;
  OriginCh   : integer;
begin
  { get the character width }
  if (TApxLineAttrArray(FLineAttrArray).Attr[aRow] = ltNormal) then begin
    CharWidth := Terminal.CharWidth;
    OriginCh := Terminal.ClientOriginCol;
  end
  else begin
    CharWidth := Terminal.CharWidth * 2;
    OriginCh := ((Terminal.ClientOriginCol - 1) div 2) + 1;
  end;

  { get the pointers to the fore/background colors }
  if RevScreenMode then
    BackColors := Buffer.GetLineForeColorPtr(aRow)
  else
    BackColors := Buffer.GetLineBackColorPtr(aRow);

  { just paint the background }
  with Terminal.Canvas do begin
    { we have to display the background color in separate steps since }
    { different characters across the display may have different      }
    { background colors; set to the loop variables }
    BackColor := BackColors^[aStartCh-1];
    ChNum := aStartCh;
    WorkRect :=
       Rect((aStartCh - OriginCh) * CharWidth,
            (aRow - Terminal.ClientOriginRow) * Terminal.CharHeight,
            0, { to be set in the loop }
            (aRow - Terminal.ClientOriginRow + 1) * Terminal.CharHeight);
    while (ChNum < aEndCh) do begin
      inc(ChNum);
      { if the background color changes, draw the rect in the old }
      { color, and then save the new color and start point }
      if (BackColor <> BackColors^[ChNum-1]) then begin
        WorkRect.Right := (ChNum - OriginCh) * CharWidth;
        Brush.Color := BackColor;
        FillRect(WorkRect);
        BackColor := BackColors^[ChNum-1];
        WorkRect.Left := WorkRect.Right;
      end;
    end;
    { finish off the last rect in the final color }
    WorkRect.Right := (aEndCh - OriginCh + 1) * CharWidth;
    Brush.Color := BackColor;
    FillRect(WorkRect);
  end;
end;
{--------}
procedure TApxVT100Emulator.vttDrawChars(aRow, aStartVal, aEndVal : integer;
                                        aVisible : Boolean;
                                        aCharValues : Boolean);
var
  ChNum     : integer;
  StartChNum : integer;
  StartChar  : integer;
  EndChar    : integer;
  BackColor  : TColor;
  ForeColor  : TColor;
  Attr       : TApxTerminalCharAttrs;
  CharSet    : byte;
  ForeColors : PApxtLongArray;
  BackColors : PApxtLongArray;
  Attrs      : PByteArray;
  CharSets   : PByteArray;
  Script     : PPaintNode;
  PaintNode  : PPaintNode;
  LineAttr   : TApxVT100LineAttr;
begin
  {ASSUMPTION: aStartVal <= aEndVal}

  { Notes: The terminal display can be viewed in two ways: either a    }
  {        certain number of characters across, or a certain number of }
  {        columns across. When each character is exactly one column   }
  {        (or cell) in width it doesn't matter whether we talk about  }
  {        characters or columns. If the characters are double width,  }
  {        it *does* matter, since 1 character unit is then equivalent }
  {        to two column units. aCharValues True means that aStartVal  }
  {        and aEndVal are character values; False means that they are }
  {        column values. In the latter case, if the row consists of   }
  {        double-width characters then we'll have to convert the      }
  {        column values to character values. }

  { avoid any drawing if the row simply is not visible }
  if (aRow < Terminal.ClientOriginRow) or
     (aRow >= Terminal.ClientOriginRow + Terminal.ClientRows) then
    Exit;

  { get the line attribute }
  LineAttr := TApxLineAttrArray(FlineAttrArray).Attr[aRow];

  { if the start/end parameters are column values, convert to character values }
  if not aCharValues then begin
    if (LineAttr <> ltNormal) then begin
      aStartVal := ((aStartVal - 1) div 2) + 1;
      aEndVal := ((aEndVal - 1) div 2) + 1;
    end;
  end;

  { check to see whether the range of characters is visible, force }
  { values into visible range }
  if (LineAttr = ltNormal) then begin
    StartChar := Terminal.ClientOriginCol;
    EndChar := Terminal.ClientOriginCol + Terminal.ClientCols;
  end
  else {double-width} begin
    StartChar := ((Terminal.ClientOriginCol - 1) div 2) + 1;
    EndChar := ((Terminal.ClientOriginCol + Terminal.ClientCols - 1)
               div 2) + 1;
  end;
  if (aEndVal < StartChar) or (aStartVal >= EndChar) then
    Exit;
  aStartVal := MaxI(aStartVal, StartChar);
  aEndVal := MinI(aEndVal, EndChar);

  { if this point is reached, we have to paint *something* }

  { there will be calls to this method that are just drawing blinking }
  { characters for the 'off' cycle (aVisible = False), so make this a }
  { special case }
  if not aVisible then begin
    vttDrawBlinkOffCycle(aRow, aStartVal, aEndVal);
    Exit;
  end;

  { if this row is the bottom half of a double-height row we don't have }
  { to display any text since it is displayed with the top half }
  if (LineAttr = ltDblHeightBottom) then
    dec(aRow);

  { this is the main processing: in general we'll be displaying text in }
  { this section; what will happen here, is that we first generate a    }
  { script of drawing commands (which background color, which text      }
  { color, which attributes, which character set, which text), and then }
  { we execute the script }

  { get the pointers to the colors, the attributes, the charsets }
  BackColors := Buffer.GetLineBackColorPtr(aRow);
  ForeColors := Buffer.GetLineForeColorPtr(aRow);
  Attrs := Buffer.GetLineAttrPtr(aRow);
  CharSets := Buffer.GetLineCharSetPtr(aRow);

  { get the initial values for the display variables }
  BackColor := BackColors^[aStartVal-1];
  ForeColor := ForeColors^[aStartVal-1];
  Attr := TApxTerminalCharAttrs(Attrs^[aStartVal-1]);
  CharSet := CharSets^[aStartVal-1];

  { make a note of the start char number }
  StartChNum := aStartVal;
  ChNum := aStartVal;

  { build the script as a stack of paint commands }
  Script := nil;
  while (ChNum < aEndVal) do begin
    { look at the next char number }
    inc(ChNum);
    { if any info has changed... }
    if (ForeColor <> ForeColors^[ChNum-1]) or
       (BackColor <> BackColors^[ChNum-1]) or
       (Attr <> TApxTerminalCharAttrs(Attrs^[ChNum-1])) or
       (CharSet <> CharSets^[ChNum-1]) then begin
      { get a new node, initialize it }
      PaintNode := vttNewPaintNode;
      PaintNode^.pnStart := StartChNum;
      PaintNode^.pnEnd := pred(ChNum);
      PaintNode^.pnFore := ForeColor;
      PaintNode^.pnBack := BackColor;
      PaintNode^.pnAttr := Attr;
      PaintNode^.pnCSet := CharSet;
      { add it to the script }
      PaintNode^.pnNext := Script;
      Script := PaintNode;
      { save the new values of the variables }
      ForeColor := ForeColors^[ChNum-1];
      BackColor := BackColors^[ChNum-1];
      Attr := TApxTerminalCharAttrs(Attrs^[ChNum-1]);
      CharSet := CharSets^[ChNum-1];
      StartChNum := ChNum;
    end;
  end;
  { create the final paint command }
  PaintNode := vttNewPaintNode;
  PaintNode^.pnStart := StartChNum;
  PaintNode^.pnEnd := aEndVal;
  PaintNode^.pnFore := ForeColor;
  PaintNode^.pnBack := BackColor;
  PaintNode^.pnAttr := Attr;
  PaintNode^.pnCSet := CharSet;
  { add it to the script }
  PaintNode^.pnNext := Script;
  Script := PaintNode;

  { now execute the paint script }
  vttExecutePaintScript(aRow, Script);
end;
{--------}
procedure TApxVT100Emulator.vttExecutePaintScript(aRow    : integer;
                                                 aScript : pointer);
var
  Canvas       : TCanvas;
  Font         : TFont;
  Walker, Temp : PPaintNode;
  ForeColor    : TColor;
  BackColor    : TColor;
  CharWidth    : integer;
  CharHeight   : integer;
  OriginCol    : integer;
  OffsetCol    : integer;
  TextStrLen   : integer;
  TextChars    : PAnsiChar;
  PartTextLen  : integer;
  TextCol      : integer;
  FontName     : TApxKeyString;
  WorkRect     : TRect;
  Y            : integer;
  R, G, B      : integer;
  Reversed     : Boolean;
  DblHeight    : Boolean;
  L, i         : Integer;


begin
  case TApxLineAttrArray (FLineAttrArray).Attr[aRow] of
    ltNormal          : DblHeight := False;
    ltDblHeightTop    : DblHeight := True;
    ltDblHeightBottom : DblHeight := True;
    ltDblWidth        : DblHeight := False;
  else
    DblHeight := False;
  end;
  { get some values as local variables }
  Canvas := Terminal.Canvas;
  Font := Terminal.Font;
  TextChars := Buffer.GetLineCharPtr (aRow);
  CharHeight := Terminal.CharHeight;
  WorkRect.Top := (aRow - Terminal.ClientOriginRow) * CharHeight; 
  if DblHeight then
    WorkRect.Bottom := WorkRect.Top + (CharHeight * 2)
  else
    WorkRect.Bottom := WorkRect.Top + CharHeight;
  if (TApxLineAttrArray (FLineAttrArray).Attr[aRow] = ltNormal) then begin
    CharWidth := Terminal.CharWidth;
    OriginCol := Terminal.ClientOriginCol;
    OffsetCol := 0;
  end
  else begin
    CharWidth := Terminal.CharWidth * 2;
    OriginCol := ((Terminal.ClientOriginCol - 1) div 2){ + 1}; 
    if Odd (Terminal.ClientOriginCol) then
      OffsetCol := 0
    else
      OffsetCol := -Terminal.CharWidth;
  end;
  { set the cell widths }
  for Y := 0 to pred (Buffer.ColCount) do
    FCellWidths^[Y] := CharWidth;
  { process the script }
  Walker := PPaintNode (aScript);
  while (Walker <> nil) do begin
    { check for reverse }
    Reversed := RevScreenMode xor
                (tcaReverse in Walker^.pnAttr) xor
                (tcaSelected in Walker^.pnAttr);
    if Reversed then begin
      ForeColor := Walker^.pnBack;
      BackColor := Walker^.pnFore;
    end
    else begin
      ForeColor := Walker^.pnFore;
      BackColor := Walker^.pnBack;
    end;
    { check for invisible }
    if (tcaInvisible in Walker^.pnAttr) then
      ForeColor := BackColor
    { check for bold }
    else if (tcaBold in Walker^.pnAttr) then begin
      if Reversed then begin
        R := MinI (GetRValue (BackColor) + $80, $FF);
        G := MinI (GetGValue (BackColor) + $80, $FF);
        B := MinI (GetBValue (BackColor) + $80, $FF);
        BackColor := RGB (R, G, B);
      end
      else begin
        R := MinI (GetRValue (ForeColor) + $80, $FF);
        G := MinI (GetGValue (ForeColor) + $80, $FF);
        B := MinI (GetBValue (ForeColor) + $80, $FF);
        ForeColor := RGB (R, G, B);
      end;
    end;

    { get the length of the text to display }
    TextStrLen := succ (Walker^.pnEnd - Walker^.pnStart);

    { move the required text to our display string }
    Move (TextChars[Walker^.pnStart - 1], FDisplayStr^, TextStrLen);
    FDisplayStr[TextStrLen] := #0;

    { set the correct background }
    Canvas.Brush.Color := BackColor;

    { ask the charset mapping to create a draw script for this text }
    FCharSetMapping.GenerateDrawScript (VT100CharSetNames[Walker^.pnCSet],
                                        FDisplayStr);

    { while the charset mapping passes us back draw commands, draw }
    TextCol := Walker^.pnStart - OriginCol;
    while FCharSetMapping.GetNextDrawCommand (FontName, FDisplayStr) do begin
      { calculate the length of this bit o' text }
      PartTextLen := StrLen (FDisplayStr);

      { calculate the left and right values for the rect }
      WorkRect.Left := (TextCol * CharWidth) + OffsetCol;
      WorkRect.Right := WorkRect.Left + (PartTextLen * CharWidth);

      { display the bit o'text }
      if (FontName = DefaultFontName) then
        Canvas.Font := Font
      else begin
        FSecondaryFont.Name := FontName;
        Canvas.Font := FSecondaryFont;
      end;
      Canvas.Font.Color := ForeColor;
      Canvas.Font.CharSet := DEFAULT_CHARSET;
      if DblHeight then
        Canvas.Font.Size := Canvas.Font.Size * 2;

     Canvas.FillRect(Rect(WorkRect.Left, WorkRect.Top,
       WorkRect.Right, WorkRect.Bottom));

     { With CLX, you need to paint each character individually to ensure }
     { each one gets in the correct cell }
     L := Length(FDisplayStr);
     for i := 1 to L do
       if FDisplayStr [i - 1] <> ' ' then
         Canvas.TextOut(WorkRect.Left + (i - 1) * CharWidth,
           WorkRect.Top - adc_TermLineSpacing, FDisplayStr[i - 1]);

      { finally, draw the underline and/or strike through }
      Canvas.Pen.Color := ForeColor;
      if (tcaUnderline in Walker^.pnAttr) then begin
        Y := WorkRect.Bottom - 1;
        Canvas.MoveTo(WorkRect.Left, Y);
        Canvas.LineTo(WorkRect.Right, Y);
      end;
      if (tcaStrikeThrough in Walker^.pnAttr) then begin
        Y := WorkRect.Top + (WorkRect.Bottom - WorkRect.Top) div 2;
        Canvas.MoveTo(WorkRect.Left, Y);
        Canvas.LineTo(WorkRect.Right, Y);
      end;

      { advance past this bit o'text }
      inc(TextCol, PartTextLen);
    end;

    { walk to the next paint node, free this one }
    Temp := Walker;
    Walker := Walker^.pnNext;
    vttFreePaintNode(Temp);
  end;
end;
{--------}
procedure TApxVT100Emulator.vttFreeAllBlinkNodes;
var
  Walker, Temp : PBlinkNode;
begin
  Walker := FBlinkFreeList;
  while (Walker <> nil) do begin
    Temp := Walker;
    Walker := Walker^.bnNext;
    Dispose(Temp);
  end;
  FBlinkFreeList := nil;
end;
{--------}
procedure TApxVT100Emulator.vttFreeAllPaintNodes;
var
  Walker, Temp : PPaintNode;
begin
  Walker := FPaintFreeList;
  while (Walker <> nil) do begin
    Temp := Walker;
    Walker := Walker^.pnNext;
    Dispose(Temp);
  end;
  FPaintFreeList := nil;
end;
{--------}
procedure TApxVT100Emulator.vttFreeBlinkNode(aNode : pointer);
begin
  PBlinkNode(aNode)^.bnNext := FBlinkFreeList;
  FBlinkFreeList := aNode;
end;
{--------}
procedure TApxVT100Emulator.vttFreePaintNode(aNode : pointer);
begin
  PPaintNode(aNode)^.pnNext := FPaintFreeList;
  FPaintFreeList := aNode;
end;
{--------}
function TApxVT100Emulator.vttGenerateDECREPTPARM(aArg : integer) : string;
var
  par   : integer;
  nbits : integer;
  xspd  : integer;
  rspd  : integer;
begin
  { calculate the parity }
  case Terminal.ComPort.Parity of
    pOdd  : par := 4;
    pEven : par := 5;
  else
    par := 1;
  end;{case}
  { calculate the number of bits }
  if (Terminal.ComPort.DataBits = dbEight) then
    nbits := 1
  else
    nbits := 2;
  { calculate the speed }
  case Terminal.ComPort.Baud of
     150 : xspd := 32;
     300 : xspd := 48;
     600 : xspd := 56;
    1200 : xspd := 64;
    2400 : xspd := 88;
    4800 : xspd := 104;
    9600 : xspd := 112;
  else
    xspd := 120;
  end;{case}
  rspd := xspd;
  { calculate the reply string }
  Result := Format(VT100ReportParm,
                   [aArg+2, par, nbits, xspd, rspd, 1, 0]);
end;
{--------}
procedure TApxVT100Emulator.vttInvalidateRow(aRow : integer);
var
  InvRect : TRect;
begin
  if Terminal.HandleAllocated and
     (Terminal.ClientOriginRow <= aRow) and
     (aRow < Terminal.ClientOriginRow + Terminal.ClientRows) then begin
    InvRect.Left := 0;
    InvRect.Top := (aRow - Terminal.ClientOriginRow) * Terminal.CharHeight;
    InvRect.Right := Terminal.ClientWidth;
    InvRect.Bottom := InvRect.Top + Terminal.CharHeight;
    {$IFDEF LINUX}
    Terminal.InvalidateRect (InvRect, False);
    {$ELSE}
    InvalidateRect (Terminal.Handle, @InvRect, False);
    {$ENDIF}
  end;
end;
{--------}
function TApxVT100Emulator.vttNewBlinkNode : pointer;
begin
  if (FBlinkFreeList = nil) then
    New(PBlinkNode(Result))
  else begin
    Result := FBlinkFreeList;
    FBlinkFreeList := PBlinkNode(Result)^.bnNext;
  end;
end;
{--------}
function TApxVT100Emulator.vttNewPaintNode : pointer;
begin
  if (FPaintFreeList = nil) then
    New(PPaintNode(Result))
  else begin
    Result := FPaintFreeList;
    FPaintFreeList := PPaintNode(Result)^.pnNext;
  end;
end;
{--------}
procedure TApxVT100Emulator.vttProcess8bitChar(aCh : AnsiChar);
const
  FirstChar = #176;
  LastChar = #218;
  CvtChars : array [FirstChar..LastChar] of AnsiChar =
             'aaaxuuukkuxkjjjkmvwtqnttmlvwtqnvvwwmmllnnjl';
var
  OldCharSet : byte;
begin
  { convert the line draw characters from the OEM character set to the }
  { VT100 linedraw charset }
  if (FirstChar <= aCh) and (aCh <= LastChar) then begin
    OldCharSet := Buffer.CharSet;
    Buffer.CharSet := 2;
    Buffer.WriteChar(CvtChars[aCh]);
    Buffer.CharSet := OldCharSet;
  end;
end;
{--------}
procedure TApxVT100Emulator.vttProcessCommand;
var
  i   : integer;
  Arg : integer;
  Arg2: integer;
  OldDefChar : AnsiChar;
  Attrs      : TApxTerminalCharAttrs;
begin
  {assumption: this method is called immediately Parser.ProcessChar }
  {             returns pctComplete. Hence Parser.ArgumentCount,    }
  {             Parser.Argument, Parser.Command, Parser.Sequence    }
  {             are all set properly }

  if (Terminal.ComPort <> nil) and Terminal.ComPort.Open then
  Terminal.ComPort.AddStringToLog(Parser.Sequence);

  { WARNING: this case statement is in numeric order! }
  case Parser.Command of
    eGotoXY {or eCUP, eHVP} :
      begin
        if (Parser.ArgumentCount = 2) then
          Buffer.SetCursorPosition(Parser.Argument[0], Parser.Argument[1]);
      end;
    eUp {or eCUU} :
      begin
        if (Parser.ArgumentCount = 1) then begin
          Arg := Parser.Argument[0];
          if (Arg = 0) then
            Arg := 1;
          for i := 1 to Arg do
            Buffer.MoveCursorUp(False);
        end;
      end;
    eDown {or eCUD, eIND, eVPR} :
      begin
        if (Parser.ArgumentCount = 1) then begin
          Arg := Parser.Argument[0];
          if (Arg = 0) then
            Arg := 1;
          for i := 1 to Arg do
            Buffer.MoveCursorDown(False);
        end;
      end;
    eRight {or eCUF, eHPR} :
      begin
        if (Parser.ArgumentCount = 1) then begin
          Arg := Parser.Argument[0];
          if (Arg = 0) then
            Arg := 1;
          for i := 1 to Arg do
            Buffer.MoveCursorRight(False, False);
        end;
      end;
    eLeft {or eCUB} :
      begin
        if (Parser.ArgumentCount = 1) then begin
          Arg := Parser.Argument[0];
          if (Arg = 0) then
            Arg := 1;
          for i := 1 to Arg do
            Buffer.MoveCursorLeft(False, False);
        end;
      end;
    eSetMode {or eSM} :
      begin
        if (Parser.ArgumentCount = 1) then begin
          case Parser.Argument[0] of
             4 : begin
                   InsertMode := True;
                   Buffer.UseInsertMode := True;
                 end;
            20 : begin
                   NewLineMode := True;
                   Buffer.UseNewLineMode := True;
                 end;
          end;{case}
        end
        else if (Parser.ArgumentCount = 2) and
                (Parser.Argument[0] = -2) then begin
          case Parser.Argument[1] of
            1 : AppKeyMode := True;
            2 : ANSIMode := True;
            3 : begin
                  Col132Mode := True;
                end;
            4 : SmoothScrollMode := True;
            5 : begin
                  RevScreenMode := True;
                end;
            6 : begin
                  RelOriginMode := True;
                end;
            7 : begin
                  WrapAround := True;
                  Buffer.UseAutoWrap := True;
                end;
            8 : AutoRepeat := True;
            9 : Interlace := True;
            999 {apecial APRO value} : AppKeypadMode := True;
          end;{case}
        end;
      end;
    eSetAttribute {or eSGR} :
      begin
        for i := 0 to pred(Parser.ArgumentCount) do begin
          case Parser.Argument[i] of
             0 : begin
                   Buffer.SetCharAttrs([]);
                   Buffer.ForeColor := Buffer.DefForeColor;
                   Buffer.BackColor := Buffer.DefBackColor;
                 end;
             1 : begin
                   Buffer.GetCharAttrs(Attrs);
                   Include(Attrs, tcaBold);
                   Buffer.SetCharAttrs(Attrs);
                 end;
             4 : begin
                   Buffer.GetCharAttrs(Attrs);
                   Include(Attrs, tcaUnderline);
                   Buffer.SetCharAttrs(Attrs);
                 end;
             5 : begin
                   Buffer.GetCharAttrs(Attrs);
                   Include(Attrs, tcaBlink);
                   Buffer.SetCharAttrs(Attrs);
                 end;
             7 : begin
                   Buffer.GetCharAttrs(Attrs);
                   Include(Attrs, tcaReverse);
                   Buffer.SetCharAttrs(Attrs);
                 end;
             8 : begin
                   Buffer.GetCharAttrs(Attrs);
                   Include(Attrs, tcaInvisible);
                   Buffer.SetCharAttrs(Attrs);
                 end;
            22 : begin
                   Buffer.GetCharAttrs(Attrs);
                   Exclude(Attrs, tcaBold);
                   Buffer.SetCharAttrs(Attrs);
                 end;
            24 : begin
                   Buffer.GetCharAttrs(Attrs);
                   Exclude(Attrs, tcaUnderline);
                   Buffer.SetCharAttrs(Attrs);
                 end;
            25 : begin
                   Buffer.GetCharAttrs(Attrs);
                   Exclude(Attrs, tcaBlink);
                   Buffer.SetCharAttrs(Attrs);
                 end;
            27 : begin
                   Buffer.GetCharAttrs(Attrs);
                   Exclude(Attrs, tcaReverse);
                   Buffer.SetCharAttrs(Attrs);
                 end;
            28 : begin
                   Buffer.GetCharAttrs(Attrs);
                   Exclude(Attrs, tcaInvisible);
                   Buffer.SetCharAttrs(Attrs);
                 end;
            30 : Buffer.ForeColor := clBlack;
            31 : Buffer.ForeColor := clMaroon;
            32 : Buffer.ForeColor := clGreen;
            33 : Buffer.ForeColor := clOlive;
            34 : Buffer.ForeColor := clNavy;
            35 : Buffer.ForeColor := clPurple;
            36 : Buffer.ForeColor := clTeal;
            37 : Buffer.ForeColor := clSilver;
            40 : Buffer.BackColor := clBlack;
            41 : Buffer.BackColor := clMaroon;
            42 : Buffer.BackColor := clGreen;
            43 : Buffer.BackColor := clOlive;
            44 : Buffer.BackColor := clNavy;
            45 : Buffer.BackColor := clPurple;
            46 : Buffer.BackColor := clTeal;
            47 : Buffer.BackColor := clSilver;
          end;{case}
        end;
      end;
    eSaveCursorPos :
      begin
        Buffer.GetCharAttrs(FSavedAttrs);
        FSavedBackColor := Buffer.BackColor;
        FSavedCharSet := Buffer.CharSet;
        FSavedCol := Buffer.Col;
        FSavedForeColor := Buffer.ForeColor;
        FSavedRow := Buffer.Row;
      end;
    eRestoreCursorPos :
      begin
        Buffer.SetCharAttrs(FSavedAttrs);
        Buffer.BackColor := FSavedBackColor;
        Buffer.CharSet := FSavedCharSet;
        Buffer.Col := FSavedCol;
        Buffer.ForeColor := FSavedForeColor;
        Buffer.Row := FSavedRow;
      end;
    eDeviceStatusReport {or eDSR} :
      begin
        { we *only* issue a reply if we are the master terminal,   }
        { otherwise if there were several terminals on a form, all }
        { attached to the same comport, the host would get several }
        { responses }
        if (Terminal.ComPort.MasterTerminal = Terminal) then begin
          if (Parser.ArgumentCount = 1) then
            if (Parser.Argument[0] = 5) then
              Terminal.ComPort.PutString(VT100StatusRpt)
            else if (Parser.Argument[0] = 6) then
              Terminal.ComPort.PutString(
                 Format(VT100CursorPos, [Buffer.Row, Buffer.Col]));
        end;
      end;
    eCHT :
      begin
        if (Parser.ArgumentCount = 1) then
          for i := 1 to Parser.Argument[0] do
            Buffer.DoHorzTab;
      end;
    eCVT :
      begin
        if (Parser.ArgumentCount = 1) then
          for i := 1 to Parser.Argument[0] do
            Buffer.DoVertTab;
      end;
    eDA :
      begin
        { we *only* issue a reply if we are the master terminal,   }
        { otherwise if there were several terminals on a form, all }
        { attached to the same comport, the host would get several }
        { responses }
        if (Terminal.ComPort.MasterTerminal = Terminal) then begin
          if (Parser.ArgumentCount = 1) then
            if (Parser.Argument[0] = 0) then
              Terminal.ComPort.PutString(VT100DeviceAttrs)
            else if (Parser.Argument[0] = 52) then
              Terminal.ComPort.PutString(VT52DeviceAttrs);
        end;
      end;
    eDCH :
      begin
        if (Parser.ArgumentCount > 0) then
          Buffer.DeleteChars(Parser.Argument[0]);
      end;
    eDL :
      begin
        if (Parser.ArgumentCount > 0) then
          Buffer.DeleteLines(Parser.Argument[0]);
      end;
    eECH :
      begin
        if (Parser.ArgumentCount > 0) then
          Buffer.EraseChars(Parser.Argument[0]);
      end;
    eED :
      begin
        if (Parser.ArgumentCount = 1) then
          case Parser.Argument[0] of
            0 : Buffer.EraseToEOW;
            1 : Buffer.EraseFromBOW;
            2 : Buffer.EraseScreen;
          end;
      end;
    eEL :
      begin
        if (Parser.ArgumentCount = 1) then
          case Parser.Argument[0] of
            0 : Buffer.EraseToEOL;
            1 : Buffer.EraseFromBOL;
            2 : Buffer.EraseLine;
          end;
      end;
    eHTS :
      begin
        Buffer.SetHorzTabStop;
      end;
    eICH :
      begin
        if (Parser.ArgumentCount > 0) then
          Buffer.InsertChars(Parser.Argument[0]);
      end;
    eIL :
      begin
        if (Parser.ArgumentCount > 0) then
          Buffer.InsertLines(Parser.Argument[0]);
      end;
    eNEL :
      begin
        Buffer.DoLineFeed;
        if not NewLineMode then
          Buffer.DoCarriageReturn;
      end;
    eRI :
      begin
        Buffer.MoveCursorUp(True);
      end;
    eRIS :
      begin
        Buffer.Reset; { also clears the screen }

        ANSIMode := adc_VT100ANSIMode;
        AppKeyMode := adc_VT100AppKeyMode;
        AppKeypadMode := adc_VT100AppKeypadMode;
        AutoRepeat := adc_VT100AutoRepeat;
        Col132Mode := adc_VT100Col132Mode;
        GPOMode := adc_VT100GPOMode;
        InsertMode := adc_VT100InsertMode;
        Interlace := adc_VT100Interlace;
        NewLineMode := adc_VT100NewLineMode;
        RelOriginMode := adc_VT100RelOriginMode;
        RevScreenMode := adc_VT100RevScreenMode;
        SmoothScrollMode := adc_VT100SmoothScrollMode;
        WrapAround := adc_VT100WrapAround;

        FUsingG1 := False;
        FG0CharSet := adc_VT100G0CharSet;
        FG1CharSet := adc_VT100G1CharSet;
      end;
    eRM :
      begin
        if (Parser.ArgumentCount = 1) then begin
          case Parser.Argument[0] of
             4 : begin
                   InsertMode := False;
                   Buffer.UseInsertMode := False;
                 end;
            20 : begin
                   NewLineMode := False;
                   Buffer.UseNewLineMode := False;
                 end;
          end;{case}
        end
        else if (Parser.ArgumentCount = 2) and
                (Parser.Argument[0] = -2) then begin
          case Parser.Argument[1] of
            1 : AppKeyMode := False;
            2 : ANSIMode := False;
            3 : begin
                  Col132Mode := False;
                end;
            4 : SmoothScrollMode := False;
            5 : begin
                  RevScreenMode := False;
                end;
            6 : begin
                  RelOriginMode := False;
                end;
            7 : begin
                  WrapAround := False;
                  Buffer.UseAutoWrap := False;
                end;
            8 : AutoRepeat := False;
            9 : Interlace := False;
            999 {secial APRO value} : AppKeypadMode := False;
          end;{case}
        end;
      end;
    eTBC :
      begin
        if (Parser.Argument[0] = 0) then
          Buffer.ClearHorzTabStop
        else if (Parser.Argument[0] = 3) then
          Buffer.ClearAllHorzTabStops;
      end;
    eDECSTBM :
      begin
        Arg := Parser.Argument[0];
        if (Arg = -1) or (Arg = 0) then
          Arg := 1;
        Arg2 := Parser.Argument[1];
        if (Arg2 = -1) or (Arg2 = 0) then
          Arg2 := Buffer.RowCount;
        Buffer.SetScrollRegion(Arg, Arg2);
      end;
    eENQ :
      begin
        { we *only* issue a reply if we are the master terminal,   }
        { otherwise if there were several terminals on a form, all }
        { attached to the same comport, the host would get several }
        { responses }
        if (Terminal.ComPort.MasterTerminal = Terminal) then begin
          Terminal.ComPort.PutString(Answerback);
        end;
      end;
    eBel :
      begin
        {$IFDEF LINUX}
        Beep;
        {$ELSE}
        MessageBeep(MB_ICONASTERISK);
        {$ENDIF}
      end;
    eBS :
      begin
        Buffer.DoBackspace;
      end;
    eLF :
      begin
        Buffer.DoLineFeed;
      end;
    eCR :
      begin
        Buffer.DoCarriageReturn;
      end;
    eSO :
      begin
        if not ANSIMode then
          Buffer.CharSet := 2
        else
          Buffer.Charset := FG1CharSet;
        FUsingG1 := True;
      end;
    eSI :
      begin
        if not ANSIMode then
          Buffer.CharSet := 0
        else
          Buffer.Charset := FG0CharSet;
        FUsingG1 := False;
      end;
    eIND2 :
      begin
        Buffer.MoveCursorDown(True);
      end;
    eDECALN :
      begin
        { scroll buffer up, fill screen with character E's }
        OldDefChar := Buffer.DefAnsiChar;
        Buffer.DefAnsiChar := 'E';
        Buffer.EraseScreen;
        Buffer.DefAnsiChar := OldDefChar;
      end;
    eDECDHL :
      begin
        if (Parser.ArgumentCount = 1) then begin
          with TApxLineAttrArray(FLineAttrArray) do
            if (Parser.Argument[0] = 0) then
              Attr[Buffer.Row] := ltDblHeightTop
            else
              Attr[Buffer.Row] := ltDblHeightBottom;
          vttInvalidateRow(Buffer.Row);
        end;
      end;
    eDECDWL :
      begin
        with TApxLineAttrArray(FLineAttrArray) do
          Attr[Buffer.Row] := ltDblWidth;
        vttInvalidateRow(Buffer.Row);
      end;
    eDECLL :
      begin
        for i := 0 to pred(Parser.ArgumentCount) do
          case Parser.Argument[i] of
            0 : LEDs := 0;
            1 : LEDs := LEDs or $1;
            2 : LEDs := LEDs or $2;
            3 : LEDs := LEDs or $4;
            4 : LEDs := LEDs or $8;
          end;{case}
      end;
    eDECREQTPARM :
      begin
        { we *only* issue a reply if we are the master terminal,   }
        { otherwise if there were several terminals on a form, all }
        { attached to the same comport, the host would get several }
        { responses }
        if (Terminal.ComPort.MasterTerminal = Terminal) then begin
          Arg := 1;
          if (Parser.ArgumentCount = 1) then begin
            Arg := Parser.Argument[0];
            if (Arg < 0) then
              Arg := 0
            else if (Arg > 1) then
              Arg := 1;
          end;
          Terminal.ComPort.PutString(vttGenerateDECREPTPARM(Arg));
        end;
      end;
    eDECSWL :
      begin
        with TApxLineAttrArray(FLineAttrArray) do
          Attr[Buffer.Row] := ltNormal;
        vttInvalidateRow(Buffer.Row);
      end;
    eDECTST :
      begin
        {}
      end;
    eDECSCS :
      begin
        if Parser.ArgumentCount = 2 then
          case Parser.Argument[0] of
            0 :
              begin
                case Parser.Argument[1] of
                  0        : FG0CharSet := 2;
                  1        : FG0CharSet := 3;
                  2        : FG0CharSet := 4;
                  ord('A') : FG0CharSet := 1;
                  ord('B') : FG0CharSet := 0;
                end;
                if not FUsingG1 then                       
                  Buffer.Charset := FG0CharSet;             
              end;
            1 :
              begin
                case Parser.Argument[1] of
                  0        : FG1CharSet := 2;
                  1        : FG1CharSet := 3;
                  2        : FG1CharSet := 4;
                  ord('A') : FG1CharSet := 1;
                  ord('B') : FG1CharSet := 0;
                end;
                if FUsingG1 then                            
                  Buffer.Charset := FG1CharSet;
              end;
          end{case}
      end;
  end;{case}
end;
{--------}
procedure TApxVT100Emulator.vttScrollRowsHandler(aSender : TObject;
                                     aCount, aTop, aBottom : integer);
begin
  TApxLineAttrArray(FLIneAttrArray).Scroll(aCount, aTop, aBottom);
end;
{--------}
procedure TApxVT100Emulator.vttSetCol132Mode(aValue : Boolean);
begin
  if (aValue <> Col132Mode) then begin
    FCol132Mode := aValue;
    if Col132Mode then
      Terminal.Columns := 132
    else
      Terminal.Columns := 80;
  end;
end;
{--------}
procedure TApxVT100Emulator.vttSetRelOriginMode(aValue : Boolean);
begin
  if (aValue <> RelOriginMode) then begin
    FRelOriginMode := aValue;
    Buffer.UseAbsAddress := not aValue;
  end;
end;
{--------}
procedure TApxVT100Emulator.vttSetRevScreenMode(aValue : Boolean);
begin
  if (aValue <> RevScreenMode) then begin
    FRevScreenMode := aValue;
    FRefresh := True;
  end;
end;
{--------}
procedure TApxVT100Emulator.vttToggleNumLock;
{
  There does not appear to be a reliable way to get or set the num lock flag
  inside of XWindows.
}
{ var
  NumLockState : TKeyboardState; }
begin
{  GetKeyboardState(NumLockState);
  if ((NumLockState[VK_NUMLOCK] and $01) = 0) then begin
    NumLockState[VK_NUMLOCK] := NumLockState[VK_NUMLOCK] or $01;
  end
  else begin
    NumLockState[VK_NUMLOCK] := NumLockState[VK_NUMLOCK] and $FE;
  end;
  SetKeyboardState(NumLockState); }
end;
{====================================================================}

{===Initialization/finalization======================================}
procedure ADTrmEmuDone;
var
  Node, Temp : PTermEmuLink;
begin
  {dispose of active links}
  Node := TermEmuLink;
  while (Node <> nil) do begin
    Temp := Node;
    Node := Node^.telNext;
    Dispose(Temp);
  end;
  {dispose of inactive links}
  Node := TermEmuLinkFreeList;
  while (Node <> nil) do begin
    Temp := Node;
    Node := Node^.telNext;
    Dispose(Temp);
  end;
end;

initialization
  TermEmuLink := nil;
  TermEmuLinkFreeList := nil;
  InvRectFreeList := nil;
  InvRectPageList := nil;

finalization
  ADTrmEmuDone;
  ApxTFreeInvRectPages;
  
end.
