(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                  AxXModem.pas 1.02                    *}
{*********************************************************}

{Global defines potentially affecting this unit}
{$I AxDefine.inc}

{Options required for this unit}
{$I-,B-,F+,A-,X+}

unit AxXModem;
  {-Provides Xmodem/Crc/1K receive and transmit functions}

interface

uses
  LibC,
  SysUtils,
  Classes,
  AxMisc,
  AxPort,
  AxPrtDrv,
  AxString;

type
  {Xmodem protocol states}
  TApxXmodemState = (
    {Transmit states}
    txInitial,              {Open file, log it, etc.}
    txHandshake,            {Waiting for handshake}
    txGetBlock,             {Get the next block to transmit}
    txWaitFreeSpace,        {Wait until outbuffer has enough freespace}
    txSendBlock,            {Send next protocol block}
    txDraining,             {Waiting for protocol block to drain}
    txReplyPending,         {Waiting for reply to last block}
    txEndDrain,             {Wait for output buffer to drain before EOT}
    txFirstEndOfTransmit,   {Send first EOT}
    txRestEndOfTransmit,    {Send subseqent EOTs}
    txEotReply,             {Waiting for EOT reply}
    txFinished,             {Close file, log it, etc.}
    txDone,                 {Signal end of protocol}

    {Receive states}
    rxInitial,              {Initialize vars, get buffers, etc.}
    rxWaitForHSReply,       {Waiting for 1st reply to handshake}
    rxWaitForBlockStart,    {Wait for block start}
    rxCollectBlock,         {Collect data}
    rxProcessBlock,         {Process block}
    rxFinishedSkip,         {Close file, log as skip}
    rxFinished,             {Close file, log as good/bad}
    rxDone);                {Signal end of protocol}

type
  TApxXModemDriver = class (TApxBaseProtocolDriver)

    private
    protected
      {General}
      FCRCMode         : Boolean;
      F1KMode          : Boolean;         {True for XModem1K}
      FGMode           : Boolean;         {True for YmodemG}
      FMaxBlockErrors  : Cardinal;        {Max number of allowed block errors}
      FBlockWait       : Cardinal;        {Wait seconds between blocks}
      FEotCheckCount   : Cardinal;        {Number of Eot retries required}
      FStartChar       : Char;            {Block start character}

      {Temp vars that state machine requires to be static}
      FHandshake       : Char;            {Handshake character}
      FNaksReceived    : Cardinal;        {Count naks received}
      FEotCounter      : Cardinal;        {Counts received EOTs}
      FCanCounter      : Cardinal;        {Counts received cCans}
      FOverheadLen     : Cardinal;        {Number of overhead bytes per block}
      FNoBaseCritSections : Boolean;      {Critical section may be handled by
                                           child class}

      {State information}
      FXmodemState     : TApxXmodemState;    {Current state of Xmodem}

      function IsXYProtocol (Protocol : Byte) : Boolean;
      function IsXProtocol (Protocol : Byte) : Boolean;
      function GetProtocolType (CRC, OneK, G, Y : Boolean) : Cardinal;
      procedure TransmitEot (First : Boolean);
      function ProcessEotReply : Boolean;

      {Internal (but used by AWYMODEM)}
      function PrepHandshake : Boolean;
      function ProcessHandshake : Boolean;
      procedure TransmitBlock (var Block : TApxDataBlock;
                          BLen : Cardinal; BType : Char);
      procedure ReceiveBlock (var Block : TApxDataBlock;
                         var BlockSize : Cardinal; var HandShake : Char);
      function ProcessBlockReply : Boolean;
      function CollectBlock (var Block : TApxDataBlock) : Boolean;
      function GetHandshakeChar : Char;
      procedure SendHandshakeChar (Handshake : Char);
      function CheckForBlockStart (var C : Char) : Boolean;
      function ProcessBlockStart (C : Char) : TApxProcessBlockStart;
    
    public

      {Constructors/destructors}
      constructor Create (AOwner : TComponent); override;

      function Init (Options : Cardinal) : Integer; override;
      procedure InitData (UseCRC, Use1K, UseGMode : Boolean); virtual;
      procedure Done; override;
      procedure DonePart; override;

      function Reinit : Integer; override; 

      {Options}
      function SetCRCMode (Enable : Boolean) : Integer;
      function Set1KMode (Enable : Boolean) : Integer;
      function SetGMode (Enable : Boolean) : Integer;
      function SetBlockWait (NewBlockWait : Cardinal) : Integer;
      function SetXmodemFinishWait (NewFinishWait : Cardinal) : Integer;

      {Control}
      procedure PrepareTransmit; override;
      procedure PrepareReceive; override;
      function TransmitPrim (Msg, wParam : Cardinal; lParam : LongInt) : LongInt;
      procedure Transmit (Msg, wParam : Cardinal; lParam : LongInt); override;
      function ReceivePrim (Msg, wParam : Cardinal; lParam : LongInt) : LongInt;
      procedure Receive (Msg, wParam : Cardinal; lParam : LongInt); override;

      procedure Cancel;
      procedure Assign (Source : TPersistent); override;

    published
      property CRCMode         : Boolean
               read FCRCMode write FCRCMode;
      property X1KMode         : Boolean
               read F1KMode write F1KMode;
      property GMode           : Boolean
               read FGMode write FGMode;
      property MaxBlockErrors  : Cardinal
               read FMaxBlockErrors write FMaxBlockErrors;
      property BlockWait       : Cardinal
               read FBlockWait write FBlockWait;
      property EotCheckCount   : Cardinal
               read FEotCheckCount write FEotCheckCount;
      property StartChar       : Char
               read FStartChar write FStartChar;
  end;

const
  {Compile-time constants}
  DrainWait = 60000;              {OutBuf drain time before error (60 sec)}
  XmodemOverhead = 5;            {Overhead bytes for each block}
  XmodemTurnDelay = 1000;        {MSec turnaround delay for each block}

  {Mode request characters}
  GReq   = 'G';
  CrcReq = 'C';
  ChkReq = cNak;

implementation

uses
  AxProtcl;
  
{$IFDEF TRIALRUN}
  {$I TRIAL07.INC}
  {$I TRIAL03.INC}
  {$I TRIAL01.INC}
{$ENDIF}

const
  {Compile-time constants}
  DefBlockWait = 5000;             {Normal between-block wait time (5 sec)}
  MaxCrcTry = 3;                 {Max tries for Crc before trying checksum}
  DefMaxBlockErrors = 5;         {Default maximum acceptable errors per block}
  aDataTrigger = 0;

const
  LogXModemState : array[TApxXmodemState] of TDispatchSubType = (
     dsttxInitial, dsttxHandshake, dsttxGetBlock, dsttxWaitFreeSpace,
     dsttxSendBlock, dsttxDraining, dsttxReplyPending,
     dsttxEndDrain, dsttxFirstEndOfTransmit, dsttxRestEndOfTransmit,
     dsttxEotReply, dsttxFinished, dsttxDone,
     dstrxInitial, dstrxWaitForHSReply, dstrxWaitForBlockStart,
     dstrxCollectBlock, dstrxProcessBlock,  dstrxFinishedSkip,
     dstrxFinished, dstrxDone);

constructor TApxXModemDriver.Create (AOwner : TComponent);
begin
  inherited Create (AOwner);

  FNoBaseCritSections := False;
end;


function TApxXModemDriver.IsXYProtocol (Protocol : Byte) : Boolean;
{-Return True if this is an Xmodem or Ymodem protocol}
begin
  case Protocol of
    Xmodem, XmodemCRC, Xmodem1K, Xmodem1KG,
    Ymodem, YmodemG :
      IsXYProtocol := True;
  else
    IsXYProtocol := False;
  end;
end;

function TApxXModemDriver.IsXProtocol (Protocol : Byte) : Boolean;
{-Return True if this is an Xmodem protocol}
begin
  case Protocol of
    Xmodem, XmodemCRC, Xmodem1K, Xmodem1KG :
      IsXProtocol := True;
  else
    IsXProtocol := False;
  end;
end;

function TApxXModemDriver.GetProtocolType (CRC, OneK, G, Y : Boolean) : Cardinal;
{-Return the protocol type}
const
  KType : array[Boolean] of Cardinal = (Xmodem1K, Ymodem);
  GType : array[Boolean] of Cardinal = (Xmodem1KG, YmodemG);
begin
  if not CRC then
    GetProtocolType := Xmodem
  else if not OneK then
    GetProtocolType := XmodemCRC
  else if not G then
    GetProtocolType := KType[Y]
  else
    GetProtocolType := GType[Y];
end;

procedure TApxXModemDriver.InitData (UseCRC, Use1K, UseGMode : Boolean);
{-Allocates and initializes a protocol control block with options}
{$IFDEF TRIALRUN}
  {$I TRIAL04.INC}
{$ENDIF}
begin
{$IFDEF TRIALRUN}
  TC;
{$ENDIF}
  {Set modes...}
  CurProtocol := Xmodem;
  SetCRCMode (UseCRC);
  Set1KMode (Use1K);
  SetGMode (UseGMode);

  {Miscellaneous inits}
  FEotCheckCount := 1;
  FBlockWait := DefBlockWait;
  FMaxBlockErrors := DefMaxBlockErrors;
  Overhead := XmodemOverhead;
  TurnDelay := XmodemTurnDelay;
  FinishWait := 0;
end;

function TApxXModemDriver.Init (Options : Cardinal) : Integer;
{-Allocates and initializes a protocol control block with options}
var
  OutSize : Cardinal;
begin
  {Check for adequate output buffer size}
  OutSize := ComPort.OutSize; 
  if (OutSize < (1024 + XmodemOverhead)) then begin
    Init := ecOutputBufferTooSmall;
    Exit;
  end;

  {Init standard data}
  if apInitProtocolData (ComPort, Options) <> ecOk then begin
    Init := ecOutOfMemory;
    Exit;
  end;

    {Can't fail after this}
  Init := ecOK;
  InitData (FCRCMode, F1KMode, FGMode);
end;

function TApxXModemDriver.Reinit : Integer;
{-Allocates and initializes a protocol control block with options}
begin
  InitData (FCRCMode, F1KMode, FGMode);
  Result := 0;
end;

procedure TApxXModemDriver.Done;
{-Disposes of P}
begin
  apDoneProtocol;
end;

procedure TApxXModemDriver.DonePart;
{-Disposes of P}
begin
  { do nothing }
end;

function TApxXModemDriver.SetCRCMode (Enable : Boolean) : Integer;
{-Enable/disable CRC mode}
var
    Y : Boolean;
begin
  {Check protocol type}
  Y := False;
  case CurProtocol of
    Xmodem, XmodemCRC   :
      ;
    Xmodem1K, Xmodem1KG :
      Enable := True;
    Ymodem, YmodemG     :
      begin
        Y := True;
        Enable := True;
      end;
    else begin
      SetCRCMode := ecBadProtocolFunction;
      Exit;
    end;
  end;

  {Ok now}
  SetCRCMode := ecOK;

  {Set check type}
  FCRCMode := Enable;
  if FCRCMode then
    CheckType := bcCrc16
  else
    CheckType := bcChecksum1;

  {Set the protocol type}
  CurProtocol := GetProtocolType (FCRCMode, F1KMode, FGMode, Y);
end;

function TApxXModemDriver.Set1KMode (Enable : Boolean) : Integer;
{-Enable/disable Xmodem1K}
var
  Y : Boolean;
begin
  {Check the protocol type}
  Y := False;
  case CurProtocol of
    Xmodem, Xmodem1K, Xmodem1KG, XmodemCRC :
      Y := False;
    Ymodem, YmodemG :
      Y := True;
    else begin
      Set1KMode := ecBadProtocolFunction;
      Exit;
    end;
  end;

  {Ok now}
  Set1KMode := ecOK;

  {Turn 1K mode on or off}
  F1KMode := Enable;
  if F1KMode then begin
    BlockLen := 1024;
    FStartChar := cStx;
    FCRCMode := True;
  end else begin
    BlockLen := 128;
    FStartChar := cSoh;
  end;

  {Set the protocol type}
  CurProtocol := GetProtocolType (FCRCMode, F1KMode, FGMode, Y);
end;

function TApxXModemDriver.SetGMode (Enable : Boolean) : Integer;
{-Enable/disable streaming}
var
  Y : Boolean;
begin
  Y := False;
  {Check the protocol type}
  case CurProtocol of
    Xmodem, Xmodem1K, Xmodem1KG, XmodemCRC :
      Y := False;
    Ymodem, YmodemG :
      Y := True;
    else begin
      SetGMode := ecBadProtocolFunction;
      Exit;
    end;
  end;

  {Ok now}
  SetGMode := ecOK;

  {Turn G mode on or off}
  FGMode := Enable;
  if FGMode then begin
    {Force 1K mode if entering G mode}
    Set1KMode (True);
    TurnDelay := 0;
    FEotCheckCount := 0;
  end else begin
    TurnDelay := XmodemTurnDelay;
    FEotCheckCount := 1;
    FMaxBlockErrors := DefMaxBlockErrors;
  end;

  {Set the protocol type}
  CurProtocol := GetProtocolType (FCRCMode, F1KMode, FGMode, Y);
end;

function TApxXModemDriver.SetBlockWait (NewBlockWait : Cardinal) : Integer;
{-Set inter-block wait time}
begin
  if not IsXYProtocol (CurProtocol) then
    SetBlockWait := ecBadProtocolFunction
  else begin
    SetBlockWait := ecOK;
    FBlockWait := NewBlockWait;
  end;
end;

function TApxXModemDriver.SetXmodemFinishWait (NewFinishWait : Cardinal) : Integer;
{-Set additional finish wait (time to wait for EOT response)}
begin
  if IsXYProtocol(CurProtocol) then
    SetXmodemFinishWait := ecBadProtocolFunction
  else begin
    SetXmodemFinishWait := ecOK;
    FinishWait := NewFinishWait;
  end;
end;

function TApxXModemDriver.PrepHandshake : Boolean;
{-Prepare to wait for a handshake char, return False if too many errors}
begin
  Inc (FHandshakeAttempt);
  if HandshakeAttempt > HandshakeRetry then begin
    PrepHandshake := False;
    apProtocolError(ecTimeout);
  end else begin
    ComPort.SetTimerTrigger (TimeoutTrigger, HandshakeWait, True);
    PrepHandshake := True;
    if HandshakeAttempt <> 1 then begin
      Inc (FBlockErrors);
      Inc (FTotalErrors);
      ForceStatus := True;
    end;
  end;
end;

procedure TApxXModemDriver.Cancel;
{-Sends cancel request to remote}
const
  CanStr : array[0..6] of Char = cCan+cCan+cCan+cBS+cBS+cBS;
begin
  {Flush anything that might be left in the output buffer}
  ComPort.FlushOutBuffer;

  {Cancel with three CANCEL chars}
  ComPort.PutBlock (CanStr, StrLen(CanStr));
  ForceStatus := True;
end;

function TApxXModemDriver.GetHandshakeChar : Char;
{-Returns proper handshake character}
begin
  if FGMode then
    GetHandshakeChar := GReq
  else if FCRCMode then
    GetHandshakeChar := CrcReq
  else
    GetHandshakeChar := ChkReq;
end;

function TApxXModemDriver.ProcessHandshake : Boolean;
{-Process handshake, return true if OK}
var
  C : Char;
begin
  {If we get here we know a character is waiting}
  ComPort.Dispatcher.GetChar(C);
  ProtocolStatus := psOK;
  case C of
    cCan : {Remote requested a cancel}
      begin
        ProtocolStatus := psCancelRequested;
        ForceStatus := True;
      end;
    ChkReq : {Set checksum mode}
      begin
        CheckType := bcChecksum1;
        FCRCMode := False;
      end;
    CrcReq : {Set CRC mode}
      begin
        CheckType := bcCrc16;
        FCRCMode := True;
      end;
    GReq : {Set G mode (streaming mode)}
      begin
        CheckType := bcCrc16;
        FCRCMode := True;
        FGMode := True;
      end;
    else begin {Unexpected character}
      ProtocolStatus := psProtocolError;
      ForceStatus := True;
    end;
  end;

  {Update the protocol type}
  if ProtocolStatus = psOK then
    CurProtocol := GetProtocolType (FCRCMode, F1KMode, FGMode,
                                        not IsXProtocol (CurProtocol));

  ProcessHandshake := ProtocolStatus = psOK;
end;

function TApxXModemDriver.ProcessBlockReply : Boolean;
{-Process reply to last block; return True for ack}
var
  C : Char;
begin
  {Handle GMode (all blocks are assumed to succeed)}
  if FGMode then begin
    ProtocolStatus := psOK;
    Inc (FBytesTransferred, BlockLen);
    Dec (FBytesRemaining, BlockLen);
    if BytesRemaining < 0 then
      BytesRemaining := 0;
    Inc (FBlockNum);
    Inc (FFileOfs, BlockLen);

    {Check for cancel from remote}
    if ComPort.CharReady then begin
      ComPort.Dispatcher.GetChar(C);
      if (C = cCan) or (C = cNak) then begin
        ProtocolStatus := psCancelRequested;
        ForceStatus := True;
      end;
    end;
    ProcessBlockReply := ProtocolStatus = psOK;
  end else begin
    {Get the reply}
    ComPort.Dispatcher.GetChar(C);

    {Process the reply}
    case C of
      cAck : {Block was acknowledged}
        begin
          ProtocolStatus := psOK;
          Inc (FBytesTransferred, BlockLen);
          Dec (FBytesRemaining, BlockLen);
          if BytesRemaining < 0 then
            BytesRemaining := 0;
          Inc (FBlockNum);
          Inc (FFileOfs, BlockLen);
        end;
      cCan : {Cancel}
        begin
          ProtocolStatus := psCancelRequested;
          ForceStatus := True;
        end;
      else {Nak or unexpected character}
        Inc(FBlockErrors);
        Inc(FTotalErrors);
        if C = cNak then
          ProtocolStatus := psBlockCheckError
        else
          ProtocolStatus := psProtocolError;
        ForceStatus := True;
    end;
    ProcessBlockReply := ProtocolStatus = psOK;
  end;
end;

procedure TApxXModemDriver.TransmitBlock (var Block : TApxDataBlock;
                            BLen : Cardinal; BType : Char);
{-Transmits one data block}
var
  I : Integer;
begin
  if BlockErrors > FMaxBlockErrors then
    {Too many errors}
    if F1KMode and (BlockLen = 1024) then begin
      {1KMode - reduce the block size and try again}
      BlockLen := 128;
      FStartChar := cSoh;
      BlockErrors := 0;
    end else begin
      {Std Xmodem - have to cancel}
      Cancel;
      apProtocolError(ecTooManyErrors);
      Exit;
    end;

  {Send the StartBlock char, the block sequence and its compliment}
  with ComPort do begin
    PutChar (FStartChar);
    PutChar (Char (Lo (BlockNum)));
    PutChar (Char (not Lo (BlockNum)));
  end;
  {Init the aBlockCheck value}
  BlockCheck := 0;

  {Send the data on its way}
  ComPort.PutBlock (Block, BlockLen);

  {Calculate the check character}
  if FCRCMode then
    for  I := 1 to BlockLen do
      BlockCheck := apUpdateCrc (Byte (Block[I]), BlockCheck)
  else
    for I := 1 to BlockLen do
      BlockCheck := apUpdateCheckSum (Byte (Block[I]), BlockCheck);

  {Send the check character}
  if FCRCMode then begin
    BlockCheck := apUpdateCrc (0, BlockCheck);
    BlockCheck := apUpdateCrc (0, BlockCheck);
    ComPort.PutChar (Char (Hi (BlockCheck)));
    ComPort.PutChar (Char (Lo (BlockCheck)));
  end else
    ComPort.PutChar (Char (BlockCheck));
end;

procedure TApxXModemDriver.TransmitEot (First : Boolean);
{-Transmit an Xmodem EOT (end of transfer)}
begin
  ProtocolStatus := psOK;
  if First then begin
    BlockErrors := 0;
    FNaksReceived := 0;
  end;

  {Ensure no stale ACKs are in the Rx buffer}
  ComPort.FlushInBuffer;
  {Send the Eot char}
  ComPort.PutChar (cEot);
end;

function TApxXModemDriver.ProcessEotReply : Boolean;
{-Get a response to an EOT, return True for ack or cancel}
var
    C : Char;
begin
  {Get the response}
  ComPort.Dispatcher.GetChar(C);
  case C of
    cAck : {Receiver acknowledged Eot, this transfer is over}
      begin
        ProcessEotReply := True;
        ProtocolStatus := psEndFile;
      end;
    cCan : {Receiver asked to cancel, this transfer is over}
      begin
        ProcessEotReply := True;
        ProtocolStatus := psCancelRequested;
        ForceStatus := True;
      end;
    cNak : {Some Xmodems always NAK the first 1 or 2 EOTs}
                {So, don't count them as errors till we get 3 }
      begin
        ProcessEotReply := False;
        Inc (FNaksReceived);
        If FNaksReceived >= 3 then begin
          Cancel;
          apProtocolError(ecTooManyErrors);
        end;
      end;
    else {Unexpected character received}
      ProcessEotReply := False;
      Inc (FBlockErrors);
      Inc (FTotalErrors);
      ProtocolStatus := psProtocolError;
  end
end;

procedure TApxXModemDriver.SendHandshakeChar (Handshake : Char);
{-Send the current handshake char}
begin
  {If in Gmode, filter out all standard Acks}
  if not FGmode or (Handshake <> cAck) then
    ComPort.PutChar(Handshake);
end;

function TApxXModemDriver.CheckForBlockStart (var C : Char) : Boolean;
{-Scan input buffer for start char, return True if found}
begin
  ProtocolStatus := psOK;
  CheckForBlockStart := False;

  {Ready to scan...}
  BlockErrors := 0;
  while ComPort.CharReady do begin

    {Check the next character}
    ComPort.Dispatcher.GetChar(C);
    case C of
      cSoh, cStx, cEot, cCan :
        begin
          CheckForBlockStart := True;
          Exit;
        end;
      else begin
        ProtocolStatus := psProtocolError;
        ForceStatus := True;
        FEotCounter := 0;
        FCanCounter := 0;
      end;
    end;
  end;
end;

function TApxXModemDriver.ProcessBlockStart (C : Char) : TApxProcessBlockStart;
{-Standard action for block start characters}
begin
  case C of
    cSoh :
      begin
        ProcessBlockStart := pbs128;
        BlockLen := 128;
        BlkIndex := 0;
      end;
    cStx :
      begin
        ProcessBlockStart := pbs1024;
        BlockLen := 1024;
        BlkIndex := 0;
      end;
    cCan :
      begin
        FEotCounter := 0;
        Inc (FCanCounter);
        if FCanCounter > 2 then begin
          ProcessBlockStart := pbsCancel;
          Cancel;
        end else
          ProcessBlockStart :=  pbsNone;
      end;
    cEot :
      begin
         FCanCounter := 0;
         Inc (FEotCounter);
         if FEotCounter = 1 then begin
           ProcessBlockStart := pbsNone;
           ComPort.PutChar(cNak);
           ComPort.SetTimerTrigger(TimeoutTrigger, FBlockWait, True);
         end else begin
           ProcessBlockStart := pbsEOT;
           ProtocolStatus := psEndFile;
           ComPort.PutChar(cAck);
         end;
      end;
    else
      ProcessBlockStart := pbsNone;
  end;
end;

function TApxXModemDriver.CollectBlock(var Block : TApxDataBlock) : Boolean;
{-Collect received data into DataBlock, return True for full block}
var
  TotalLen : Cardinal;
  C : Char;
begin
  FHandshake := cNak;
  TotalLen := BlockLen + FOverheadLen;
  while ComPort.CharReady and (BlkIndex < TotalLen) do begin
    ComPort.Dispatcher.GetChar(C);
    Inc(FBlkIndex);
    Block[BlkIndex] := C;
  end;
  CollectBlock := BlkIndex = TotalLen;
end;

procedure TApxXModemDriver.ReceiveBlock(var Block : TApxDataBlock;
                           var BlockSize : Cardinal; var HandShake : Char);
{-Process the data already in Block}
type
  LHW = record
    L, H : Char;
  end;
var
  R1, R2 : Byte;
  I      : Cardinal;
  Check  : Word;
begin
  {Get and compare block sequence numbers}
  R1 := Byte(Block[1]);
  R2 := Byte(Block[2]);
  if (not R1) <> R2 then begin
    Inc(FBlockErrors);
    Inc(FTotalErrors);
    Cancel;
    ProtocolStatus := psSequenceError;
    apProtocolError(ecSequenceError);
    Exit;
  end;

  {Calculate the block check value}
  BlockCheck := 0;
  if FCRCMode then
    for I := 3 to BlockLen+2 do
      BlockCheck := apUpdateCrc(Byte(Block[I]), BlockCheck)
  else
    for I := 3 to BlockLen+2 do
      BlockCheck := apUpdateCheckSum(Byte(Block[I]), BlockCheck);

  {Check the block-check character}
  if FCRCMode then begin
    BlockCheck := apUpdateCrc (0, BlockCheck);
    BlockCheck := apUpdateCrc (0, BlockCheck);
    LHW (Check).H := Block[BlockLen+3];
    LHW (Check).L := Block[BlockLen+4];
  end else begin
    Check := Byte (Block[BlockLen+3]);
    BlockCheck := BlockCheck and $FF;
  end;

  if Check <> BlockCheck then begin
    {Block check error}
    Inc (FBlockErrors);
    Inc (FTotalErrors);
    ComPort.FlushInBuffer;
    ProtocolStatus := psBlockCheckError;
    Exit;
  end;

  {Check the block sequence for missing or duplicate blocks}
  if (BlockNum <> 0) and (R1 = Lo(BlockNum-1)) then begin
    {This is a duplicate block}
    HandShake := cAck;
    BlockErrors := 0;
    ProtocolStatus := psDuplicateBlock;
    Exit;
  end;
  if R1 <> Lo(BlockNum) then begin
    {Its a sequence error}
    Cancel;
    ProtocolStatus := psSequenceError;
    apProtocolError(ecSequenceError);
    Exit;
  end;

  {Block is ok}
  Handshake := cAck;

  {Update status fields for the next call to the user status routine}
  Inc(FBlockNum);
  Inc(FBytesTransferred, BlockLen);
  Dec(FBytesRemaining, BlockLen);
  if BytesRemaining < 0 then
    BytesRemaining := 0;
  BlockErrors := 0;
  ProtocolError := ecOK;
  ProtocolStatus := psOK;
  ForceStatus := True;
  BlockSize := BlockLen;
end;

procedure TApxXModemDriver.PrepareTransmit;
{-Prepare for transmitting Xmodem}
begin
  {Inits}
  apResetStatus;
  apShowFirstStatus;

  {Get the file to transmit}
  if not NextFile(FPathname) then begin
    {aProtocolError already set}
    apShowLastStatus;
    Exit;
  end;

  {Other inits}
  TimerStarted := False;
  ForceStatus := True;
  FXmodemState := txInitial;
  DataBlock := nil;

  {Discard any unread data}
  ComPort.FlushInBuffer;
end;

function TApxXModemDriver.TransmitPrim(Msg, wParam : Cardinal;
                      lParam : LongInt) : LongInt;
{-Perform one increment of an Xmodem transmit}
var
  TriggerID     : Cardinal absolute wParam;
  Wait          : Cardinal;
  BufSize       : Cardinal;
  Finished      : Boolean;
  C             : Char;
  StatusTimeMS  : Cardinal;

  procedure PrepSendBlock;
  {-Prepare to (re)send the current block}
  begin
    ProtocolError := ecOK;
    {Don't waste time if the buffer space is available}
    if (ComPort.OutBuffFree >= (BlockLen + XmodemOverhead)) then
      FXmodemState := txSendBlock
    else begin
      FXmodemState := txWaitFreespace;
      ComPort.SetTimerTrigger(TimeoutTrigger, TransTimeout, True);
      ComPort.SetStatusTrigger (OutBuffFreeTrigger,
                                BlockLen + XmodemOverhead, True);
    end;
  end;

begin
  with FComPort do begin 
    {Function result is always zero unless the protocol is over}
    Result := 0;

    if not FNoBaseCritSections then
      EnterCriticalSection (FProtSection);

  {Exit if protocol was cancelled while waiting for crit section}
    if FXmodemState = txDone then begin
      if not FNoBaseCritSections then
        LeaveCriticalSection (FProtSection);  
      Result := 1;
      Exit;
    end;

    {If it's a TriggerAvail message then force the TriggerID}
    if Msg = apx_TriggerAvail then
      TriggerID := ApxProtocolDataTrigger;

    repeat

      ComPort.DebugLog.AddDebugEntry (TApxCustomProtocol,
                                    Cardinal (AxdtXModem),
                                    Cardinal (LogXModemState[FXmodemState]),
                                    0);

      {Check for user or remote abort}
      if ((Integer(TriggerID) = NoCarrierTrigger) and
          not ComPort.Dispatcher.CheckDCD) or
          (Msg = apx_ProtocolCancel) then begin
        if Msg = apx_ProtocolCancel then begin
          Cancel;
          ProtocolStatus := psCancelRequested;
        end else
          ProtocolStatus := psAbortNoCarrier;
        FXmodemState := txFinished;
        ForceStatus := True;
      end;

      {Show status periodically}
      if (Integer(TriggerID) = StatusTrigger) or ForceStatus then begin
        if TimerStarted then
          ElapsedXfrTime := ElapsedTime (Timer);
        if Dispatcher.TimerTimeRemaining (StatusTrigger, StatusTimeMS) <> 0 then
          StatusTimeMS := 0;
        if LongInt (StatusTimeMS) <= 0 then begin
          ShowStatus(0);
          ComPort.SetTimerTrigger (StatusTrigger, StatusInterval, True);
          ForceStatus := False;
        end;
      end;

      {Process current state}
      case FXmodemState of
        txInitial :
          begin
            {Get a protocol DataBlock}
            DataBlock := AllocMem(SizeOf(TApxDataBlock));

            {Upcase the pathname}
            if UpcaseFileNames then
              AnsiStrUpper(Pathname);

            {Show file name to user logging routine}
            LogFile(lfTransmitStart);

            {Show handshaking in progress}
            ProtocolStatus := psProtocolHandshake;
            ForceStatus := True;

            {Prepare to read protocol blocks}
            PrepareReading;
            if ProtocolError = ecOK then begin
              {Set the first block number}
              BlockNum := 1;

              {Check for handshake character}
              FXmodemState := txHandshake;
              HandshakeAttempt := 0;
              if not PrepHandshake then
                FXmodemState := txFinished;
            end else
              FXmodemState := txFinished;
          end;

        txHandshake :
          if TriggerID = ApxProtocolDataTrigger then begin
            if ProcessHandshake then begin
              {Start protocol timer now}
              NewTimer (FTimer, 50);
              TimerStarted := True;
              FXmodemState := txGetBlock;
              FileOfs := 0;
              BlockErrors := 0;
              TotalErrors := 0;
              if FGMode then
                FMaxBlockErrors := 0;
              ProtocolStatus := psOK;
            end else begin
              if ProtocolStatus = psCancelRequested then
                FXmodemState := txFinished
              else if not PrepHandshake then
                FXmodemState := txFinished
            end;
          end else if Integer(TriggerID) = TimeoutTrigger then
            if not PrepHandshake then
              FXmodemState := txFinished;

        txGetBlock :
          begin
            LastBlockSize := BlockLen;
            BlockErrors := 0;
            NoMoreData := ReadProtocolBlock(DataBlock^, FLastBlockSize);
            PrepSendBlock;
          end;

        txWaitFreeSpace :
          if Integer(TriggerID) = OutBuffFreeTrigger then
            {Got enough free space, go send the block}
            FXmodemState := txSendBlock
          else if Integer(TriggerID) = TimeoutTrigger then begin
            {Never got adequate free space, can't continue}
            apProtocolError (ecTimeout);
            FXmodemState := txFinished;
          end else if (TriggerID = ApxProtocolDataTrigger) and FGMode then
            {In G mode, cancels could show up here}
            while ComPort.CharReady do begin
              ComPort.Dispatcher.GetChar(C);
              if (C = cCan) then begin
                ProtocolStatus := psCancelRequested;
                ForceStatus := True;
                FXmodemState := txFinished;
                break;
              end;
            end;

        txSendBlock :
          if LastBlockSize <= 0 then
            {Don't send empty blocks}
            FXmodemState := txFirstEndOfTransmit
          else begin
            {If no errors, then send this block to the remote}
            if ProtocolError = ecOK then begin
              TransmitBlock(DataBlock^, BlockLen, ' ');

              {If TransmitBlock failed, go clean up}
              if ProtocolError <> ecOK then begin
                FlushOutBuffer;
                FXmodemState := txFinished;
              end else

                {Prepare to handle reply}
                if FGMode then begin
                  {Process possible reply}
                  if ProcessBlockReply then begin
                    {No reply, continue as though ack was received}
                    if NoMoreData then begin
                      {Finished, wait for buffer to drain}
                      FXmodemState := txEndDrain;
                      if FinishWait = 0 then begin
                        {Calculate finish drain time}
                        BufSize := InBuffUsed + InBuffFree;
                        if ActCPS = 0 then
                          Wait := 2 * (FBlockWait + ((BufSize) * 182) div 10)
                        else
                          Wait := 2 * (FBlockWait +
                                      ((BufSize div ActCPS) * 182) div 10);
                      end else
                        {Use user-specified finish drain time}
                        Wait := FinishWait;
                      SetTimerTrigger(TimeoutTrigger, Wait, True);
                      SetStatusTrigger(OutBuffUsedTrigger, 0, True);
                    end else
                      FXmodemState := txGetBlock;
                  end else begin
                    {Got CAN or NAK, cancel the protocol}
                    FlushOutBuffer;
                    FXmodemState := txFinished;
                  end;
                end else begin
                  {Wait for output buffer to drain}
                  FXmodemState := txDraining;
                  SetTimerTrigger(TimeoutTrigger, DrainWait, True);
                  SetStatusTrigger (OutBuffUsedTrigger, 0, True);
                end;

                {Force a status update}
                ForceStatus := True;
              end else begin
                {Disk read error, have to give up}
                Cancel;
                FXmodemState := txFinished;
              end;
          end;

        txDraining :
          if (Integer(TriggerID) = OutBuffUsedTrigger) or
               (TriggerID = ApxProtocolDataTrigger) or
               (Integer(TriggerID) = TimeoutTrigger) then begin
            FXmodemState := txReplyPending;
            SetTimerTrigger(TimeoutTrigger, FBlockWait, True);
          end;

        txReplyPending :
          if TriggerID = ApxProtocolDataTrigger then begin
            if ProcessBlockReply then begin
                {Got reply, go send next block}
              if NoMoreData then
                FXmodemState := txFirstEndofTransmit
              else
                FXmodemState := txGetBlock;
            end else if ProtocolStatus = psCancelRequested then begin
              {Got two cancels, we're finished}
              FlushOutBuffer;  
              FXmodemState := txFinished;
            end else
              {Got junk or Nak for a response, go send block again}
              PrepSendBlock;
          end else if Integer(TriggerID) = TimeoutTrigger then
            {Got timeout, try to send block again}
            PrepSendBlock;

        txEndDrain:
          if (Integer(TriggerID) = OutBuffUsedTrigger) or
               (Integer(TriggerID) = TimeoutTrigger) then
            FXmodemState := txFirstEndOfTransmit;

        txFirstEndOfTransmit :
          begin
            TransmitEot(True);
            SetTimerTrigger(TimeoutTrigger, TransTimeout, True);
            FXmodemState := txEotReply;
          end;

        txRestEndOfTransmit :
          begin
            TransmitEot(False);
            SetTimerTrigger(TimeoutTrigger, FBlockWait, True);
            if BlockErrors <= FMaxBlockErrors then begin
              FXmodemState := txEotReply;
            end else begin
              apProtocolError(ecTooManyErrors);
              FXmodemState := txFinished;
            end;
          end;

        txEotReply :
          if TriggerID = ApxProtocolDataTrigger then
            if ProcessEotReply then
              FXmodemState := txFinished
            else
              FXmodemState := txRestEndOfTransmit
          else if Integer(TriggerID) = TimeoutTrigger then
            FXmodemState := txRestEndOfTransmit;

        txFinished :
          begin
            if (ProtocolStatus <> psEndFile) or
                 (ProtocolError <> ecOK) then
              FlushInBuffer;  

            {Close the file}
            FinishReading;

            {Show status, user logging}
            if (ProtocolStatus = psEndFile) then
              LogFile(lfTransmitOk)
            else
              LogFile(lfTransmitFail);
            {apShowLastStatus(P);}

            {Clean up}
            FreeMem(DataBlock, SizeOf(TApxDataBlock));
            FXmodemState := txDone;

            if Msg <> apx_FromYmodem then begin
              {Say we're finished}
              apShowLastStatus;
              apSignalFinish (True);
            end else
              ShowStatus(0);

            {Tell caller we're finished}
            Result := 1;
          end;
      end;

      {Should we exit or not}
      case FXmodemState of
        {Stay in state machine}
        txGetBlock,
        txSendBlock,
        txFirstEndOfTransmit,
        txRestEndOfTransmit,
        txFinished:
          Finished := False;

        {Stay in state machine if data available}
        txEotReply,
        txDraining,
        txReplyPending,
        txHandshake:
          Finished := not CharReady;

        {Finished with state machine}
        txWaitFreeSpace,
        txEndDrain,
        txDone:
          Finished := True
        else
          Finished := True;
      end;

      {Force data trigger if staying in state machine}
      TriggerID := ApxProtocolDataTrigger;
    until Finished;
  end;

  if not FNoBaseCritSections then
    LeaveCriticalSection(FProtSection);
end;

procedure TApxXModemDriver.Transmit (Msg, wParam : Cardinal; lParam : LongInt);
begin
  TransmitPrim(Msg, wParam, lParam);
end;

procedure TApxXModemDriver.PrepareReceive;
{-Starts Xmodem receive protocol}
begin
  {Prepare state machine, show first status}
  FXmodemState := rxInitial;
  DataBlock := nil;
  apResetStatus;
  apShowFirstStatus;
  ForceStatus := False;
  TimerStarted := False;
end;

function TApxXModemDriver.ReceivePrim (Msg, wParam : Cardinal;
                     lParam : LongInt) : LongInt;
{-Performs one increment of an Xmodem receive}
label
  ExitPoint;
var
  TriggerID     : Cardinal absolute wParam;
  DataPtr       : PApxDataBlock;
  Finished      : Boolean;
  C             : Char;
  StatusTimeMS  : Cardinal; 

  procedure Cleanup(DisposeBuffers : Boolean);
  {-Handle error reporting and other cleanup}
  begin
    if DisposeBuffers then
      FreeMem(DataBlock, SizeOf(TApxDataBlock)+XmodemOverhead);

    if Msg <> apx_FromYmodem then begin
      apShowLastStatus;
      apSignalFinish (False);
    end;

    FXmodemState := rxDone;
    Result := 1;
  end;

  function CheckErrors : Boolean;
  {-Increment block errors, return True if too many}
  begin
    Inc(FBlockErrors);
    Inc(FTotalErrors);
    if BlockErrors > FMaxBlockErrors then begin
      CheckErrors := True;
      apProtocolError(ecTooManyErrors);
      ProtocolStatus := psProtocolError;
      ForceStatus := True;
    end else
      CheckErrors := False;
  end;

begin
  if not FNoBaseCritSections then
    EnterCriticalSection(FProtSection);

    {Exit if protocol was cancelled while waiting for crit section}
    if FXmodemState = rxDone then begin
      if not FNoBaseCritSections then
        LeaveCriticalSection(FProtSection); 
      Result := 1;
      Exit;
    end;

    {Set TriggerID directly for TriggerAvail messages}
    if Msg = apx_TriggerAvail then
      TriggerID := aDataTrigger;

    repeat
      {Return 0 unless finished}
      Result := 0;

      ComPort.DebugLog.AddDebugEntry (TApxCustomProtocol,
                                    Cardinal (AxdtXModem),
                                    Cardinal (LogXModemState[FXmodemState]),
                                    0);
        
      {Check for user abort}
      if ((Integer(TriggerID) = NoCarrierTrigger) and             
          not FComPort.Dispatcher.CheckDCD) or
           (Msg = apx_ProtocolCancel) then begin
        if Msg = apx_ProtocolCancel then begin
          Cancel;
          ProtocolStatus := psCancelRequested;
        end else
          ProtocolStatus := psAbortNoCarrier;
        FXmodemState := rxFinished;
        ForceStatus := True;
      end;

      {Show status periodically}
      if (Integer(TriggerID) = StatusTrigger) or ForceStatus then begin
        if TimerStarted then
          ElapsedXfrTime := ElapsedTime (Timer);
        if FComPort.Dispatcher.TimerTimeRemaining (StatusTrigger,  
                                  StatusTimeMS) <> 0 then
          StatusTimeMS := 0;
        if LongInt (StatusTimeMS) <= 0 then begin
          ShowStatus(0);
          FComPort.Dispatcher.SetTimerTrigger(StatusTrigger, StatusInterval, True);
          ForceStatus := False;
        end;
      end;

      {Process current state}
      case FXmodemState of
        rxInitial :
          begin
            {Get a protocol DataBlock}
            DataBlock := AllocMem(SizeOf(TApxDataBlock)+XmodemOverhead);

            {Pathname should already have name of file to receive}
            if Pathname[0] = #0 then begin
              apProtocolError(ecNoFilename);
              Cancel;
              Cleanup(True);
              if not FNoBaseCritSections then
                LeaveCriticalSection(FProtSection);  
              Exit;
            end else if UpcaseFileNames then
              AnsiStrUpper(PathName);

            {Send file name to user's LogFile procedure}
            LogFile(lfReceiveStart);

            {Accept this file}
            if not AcceptFile(PathName) then begin
              Cancel;
              ProtocolStatus := psFileRejected;
              FXmodemState := rxFinishedSkip;
              goto ExitPoint;
            end;

            {Prepare to write file}
            PrepareWriting;
            if (ProtocolError <> ecOK) or
                 (ProtocolStatus = psCantWriteFile) then begin
              if ProtocolStatus = psCantWriteFile then
                ProtocolError := ecCantWriteFile;
              Cancel;
              FXmodemState := rxFinishedSkip;
              goto ExitPoint;
            end;

            {Start sending handshakes}
            FileOfs := 0;
            FXmodemState := rxWaitForHSReply;
            FHandshake := GetHandshakeChar;
            SendHandshakeChar (FHandshake);
            BlockNum := 1;
            FEotCounter := 0;
            FCanCounter := 0;
            FComPort.Dispatcher.SetTimerTrigger (TimeoutTrigger, HandshakeWait, True);

            {Set overhead length based on check type}
            if FCRCMode then
              FOverheadLen := 4
            else
              FOverheadLen := 3;
          end;

        rxWaitForHSReply :
          if TriggerID = ApxProtocolDataTrigger then begin
            FXmodemState := rxWaitForBlockStart;
          end else if Integer(TriggerID) = TimeoutTrigger then begin
            if CheckErrors then
              FXmodemState := rxFinished
            else begin
              if (FHandshake = CrcReq) and
                 (BlockErrors > MaxCrcTry) then begin
                {Step down to Xmodem checksum}
                BlockErrors := 0;
                CheckType := bcChecksum1;
                FHandshake := ChkReq;
                FCRCMode := False;
                Dec(FOverheadLen);
              end;
              FComPort.Dispatcher.PutChar(FHandshake);
              FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
            end;
          end;

        rxWaitForBlockStart :
          if TriggerID = ApxProtocolDataTrigger then begin
            {Check for timer start}
            if not TimerStarted then begin
              NewTimer(FTimer, 50);
              TimerStarted := True;
              if FGMode then
                FMaxBlockErrors := 0;
            end;

            {Process the received character}
            if CheckForBlockStart(C) then begin
              case ProcessBlockStart(C) of
                pbs128, pbs1024 :
                  begin
                    FXmodemState := rxCollectBlock;
                    FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, FBlockWait, True);
                  end;
                pbsCancel, pbsEOT :
                  FXmodemState := rxFinished;
              end;
            end;
          end else if Integer(TriggerID) = TimeoutTrigger then begin
            {Timeout waiting for block start}
            if FEotCounter <> 0 then begin
              {Timeout waiting for second cEot, end normally}
              FComPort.Dispatcher.PutChar(cAck);
              FXmodemState := rxFinished;
              ProtocolStatus := psEndFile;
            end else if CheckErrors or (FCanCounter <> 0) then begin
              {Too many errors, quit the protocol}
              if FCanCounter <> 0 then begin
                ProtocolStatus := psCancelRequested;
                ForceStatus := True;
              end;
              FXmodemState := rxFinished;
            end else begin
              {Simple timeout, resend handshake}
              FXmodemState := rxWaitForHSReply;
              SendHandshakeChar(FHandshake);
              FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, FBlockWait, True);
            end;
          end;

        rxCollectBlock :
          if TriggerID = ApxProtocolDataTrigger then begin
            {Got data, collect into DataBlock}
            if CollectBlock(DataBlock^) then
              FXmodemState := rxProcessBlock;
          end else if Integer(TriggerID) = TimeoutTrigger then begin
            {Timeout out waiting for complete block, send nak}
            FComPort.Dispatcher.PutChar(cNak);
            FXmodemState := rxWaitForBlockStart;
            ProtocolStatus := psTimeout;
            FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, FBlockWait, True);
          end;

        rxProcessBlock :
          begin
            {Go process what's in DataBlock}
            ReceiveBlock(DataBlock^, FLastBlockSize, FHandshake);
            SendHandshakeChar(FHandshake);
            if ProtocolStatus = psOK then begin
              {Got block ok, go write it out (skip blocknum bytes)}
              DataPtr := DataBlock;
              Inc (PByte (DataPtr), 2);
              WriteProtocolBlock(DataPtr^, LastBlockSize);
              if ProtocolError <> ecOK then begin
                {Failed to write the block, cancel protocol}
                Cancel;
                FXmodemState := rxFinished;
              end else begin
                {Normal received block -- keep going}
                Inc(FFileOfs, LastBlockSize);
                FXmodemState := rxWaitForBlockStart;
                FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, FBlockWait, True);
              end;
            end else begin
              if (ProtocolError <> ecOK) or FGMode then begin
                {Fatal error - cancel protocol}
                Cancel;
                FXmodemState := rxFinished;
              end else begin
                {Failed to get block, go try again}
                FXmodemState := rxWaitForHSReply;
                FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
              end;
            end;
          end;

        rxFinishedSkip :
          begin
            FinishWriting;
            LogFile(lfReceiveSkip);
            Cleanup(True);
          end;

        rxFinished :
          begin
            FinishWriting;
            if (ProtocolStatus = psEndFile) then
              LogFile(lfReceiveOk)
            else
              LogFile(lfReceiveFail);
            Cleanup(True);
          end;
      end;

ExitPoint:
      {Should we exit or not}
      case FXmodemState of
        {Stay in state machine}
        rxProcessBlock,
        rxFinishedSkip,
        rxFinished:
          Finished := False;

        {Stay in state machine if data available}
        rxWaitForBlockStart,
        rxCollectBlock:
          begin
            Finished := not FComPort.Dispatcher.CharReady;
            TriggerID := aDataTrigger;
          end;

          {Finished with state machine}
        rxInitial,
        rxWaitForHSReply,
        rxDone:
          Finished := True
          
        else
          Finished := True;
      end;
    until Finished;

  if not FNoBaseCritSections then
    LeaveCriticalSection(FProtSection);    
end;

procedure TApxXModemDriver.Receive (Msg, wParam : Cardinal; lParam : LongInt);
begin
  ReceivePrim (Msg, wParam, lParam);
end;

procedure TApxXModemDriver.Assign (Source : TPersistent);
begin
  inherited Assign (Source);
  if Source is TApxXModemDriver then
    with Source as TApxXModemDriver do begin
      FCRCMode         := FCRCMode;
      F1KMode          := F1KMode;
      FGMode           := FGMode;
      FMaxBlockErrors  := FMaxBlockErrors;
      FBlockWait       := FBlockWait;
      FEotCheckCount   := FEotCheckCount;
      FStartChar       := FStartChar;
      FHandshake       := FHandshake;
      FNaksReceived    := FNaksReceived;
      FEotCounter      := FEotCounter;
      FCanCounter      := FCanCounter;
      FOverheadLen     := FOverheadLen;
      FXmodemState     := FXmodemState;
    end;
end;

end.
