(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                  AxYModem.pas 1.02                    *}
{*********************************************************}

{Global defines potentially affecting this unit}
{$I AxDefine.inc}

{Options required for this unit}
{$I-,B-,F+,A-,X+}

unit AxYModem;
  {-Provides Ymodem/YmodemG recieve and transmit functions}

interface

uses
  Classes,
  SysUtils,
  LibC,
  AxSystem,
  AxMisc,
  AxPrtDrv,
  AxXModem,
  AxPort,
  AxString;

type
  {Ymodem protocol transmit states}
  TApxYmodemState = (
    {Transmit states}
    tyInitial,              {Get next file}
    tyHandshake,            {Waiting for handshake}
    tyGetFileName,          {Get the next file to transmit}
    tySendFileName,         {Format and send file name block}
    tyDraining,             {Waiting for protocol block to drain}
    tyReplyPending,         {Waiting for reply to name block}
    tyPrepXmodem,           {Prepare to enter Xmodem state table}
    tySendXmodem,           {Calling Xmodem state table}
    tyFinished,             {Send EOT block}
    tyFinishDrain,          {Wait for EOT to block to drain}
    tyDone,                 {Signal end of protocol}

    {Receive states}
    ryInitial,              {Initialize vars, get buffers, etc.}
    ryDelay,                {Delay the handshake for Telix}
    ryWaitForHSReply,       {Waiting for 1st reply to handshake}
    ryWaitForBlockStart,    {Wait for block start}
    ryCollectBlock,         {Collect received chars into DataBlock}
    ryProcessBlock,         {Process complete DataBlock}
    ryOpenFile,             {Extract file info}
    ryPrepXmodem,           {Prepare to enter Xmodem state}
    ryReceiveXmodem,        {Calling Xmodem state table}
    ryFinished,             {Clean up}
    ryDone);                {Signal end of protocol}


type
  TApxYModemDriver = class (TApxXModemDriver)
  
    private
    protected
      {Unique Ymodem fields}
      FSaveLen         : LongInt;         {Saved file length}
      FNewDT           : LongInt;         {Date/time stamp}
      FSaveName        : TPathCharArray;  {Saved file name}
      FFileHeader      : PApxDataBlock;      {Needed for file name block}
      FYmodemState     : TApxYmodemState;   {Current Ymodem state}

    public
      {constructors/destructors}
      constructor Create (AOwner : TComponent); override;
      function Init(Options : Cardinal) : Integer; override;
      procedure Done; override;
      procedure InitData(UseCRC, Use1K, UseGMode : Boolean); override;

      function Reinit : Integer; override;
      procedure DonePart; override;
      procedure Assign (Source : TPersistent); override;

      {Control}
      procedure PrepareTransmit; override;
      procedure PrepareReceive; override;
      procedure Transmit(Msg, wParam : Cardinal; lParam : LongInt); override;
      procedure Receive(Msg, wParam : Cardinal; lParam : LongInt); override;
  end;

implementation

uses
  AxProtcl;
  
{$IFDEF TRIALRUN}
  {$I TRIAL07.INC}
  {$I TRIAL03.INC}
  {$I TRIAL01.INC}
{$ENDIF}

const
  aDataTrigger = 0;
  LogYModemState : array [TApxYmodemState] of TDispatchSubType = (        
     dsttyInitial,
     dsttyHandshake,
     dsttyGetFileName,
     dsttySendFileName,
     dsttyDraining,
     dsttyReplyPending,
     dsttyPrepXmodem,
     dsttySendXmodem,
     dsttyFinished,
     dsttyFinishDrain,
     dsttyDone,
     dstryInitial,
     dstryDelay,
     dstryWaitForHSReply,
     dstryWaitForBlockStart,
     dstryCollectBlock,
     dstryProcessBlock,
     dstryOpenFile,
     dstryPrepXmodem,
     dstryReceiveXmodem,
     dstryFinished,
     dstryDone);

constructor TApxYModemDriver.Create (AOwner : TComponent);
begin
  inherited Create (AOwner);
  FNoBaseCritSections := True;
end;

procedure TApxYModemDriver.InitData (UseCRC, Use1K, UseGMode : Boolean);
{-Allocates and initializes a protocol control block with options}
begin
  {Set modes}

  CurProtocol := Ymodem;
  SetCRCMode (True);
  Set1KMode (Use1K);
  SetGMode (UseGMode);

  {Other inits}

  FBatchProtocol := True;

  {Don't ask for any EOT retries}

  FEotCheckCount := 0;
end;

function TApxYModemDriver.Init (Options : Cardinal) : Integer;
{-Allocates and initializes a protocol control block with options}
var
    {InSize, }OutSize : Cardinal;
begin
  {Check for adequate output buffer size}
  OutSize := ComPort.OutSize;
  if OutSize < (1024+XmodemOverhead) then begin
    Init := ecOutputBufferTooSmall;
    Exit;
  end;

  {Init standard data}
  if apInitProtocolData(ComPort, Options) <> ecOk then begin
    Init := ecOutOfMemory;
    Exit;
  end;

  {Allocate the name block buffer}
  FFileHeader := AllocMem (SizeOf (TApxDataBlock) + XmodemOverhead);

  {Can't fail after this}
  Init := ecOK;

  {Init the protocol data}
  InitData (FCRCMode, F1KMode, FGMode);
end;

function TApxYModemDriver.Reinit : Integer;
{-Allocates and initializes a protocol control block with options}
begin
  {Allocate the name block buffer}
  FFileHeader := AllocMem (SizeOf (TApxDataBlock) + XmodemOverhead);

  {Can't fail after this}
  Reinit := ecOK;

  {Init the data}
  InitData (FCRCMode, F1KMode, FGMode);
end;

procedure TApxYModemDriver.Done;
{-Destroy Ymodem object}
{$IFDEF TRIALRUN}
  {$I TRIAL04.INC}
{$ENDIF}
begin
  if assigned (FFileHeader) then begin 
    FreeMem(FFileHeader, SizeOf(TApxDataBlock) + XmodemOverhead);
    apDoneProtocol;
  end;
{$IFDEF TRIALRUN}
  TC;
{$ENDIF}
end;

procedure TApxYModemDriver.DonePart;
{-Destroy Ymodem object}
begin
  if assigned (FFileHeader) then 
    FreeMem(FFileHeader, SizeOf(TApxDataBlock) + XmodemOverhead);
end;

procedure TApxYModemDriver.PrepareTransmit;
{-Prepare to transmit a Ymodem batch}
begin
  {Reset status vars}
  apResetStatus;
  ProtocolStatus := psProtocolHandshake;
  apShowFirstStatus;
  ForceStatus := False;
  TimerStarted := False;

  {Set first state}
  FYmodemState := tyInitial;

  {Flush trigger buffer}
  ComPort.FlushInBuffer;
end;

procedure TApxYModemDriver.Transmit(Msg, wParam : Cardinal;
                      lParam : LongInt);
{-Perform one increment of Ymodem batch transmit}
label
  ExitPoint;
var
  TriggerID        : Cardinal absolute wParam;
  XState           : Cardinal;
  Finished         : Boolean;
  StatusTimeMS     : Cardinal; 
  ExitStateMachine : Boolean;
  I                : Integer;
  Len              : Byte;
  S2               : string[13];
  S1               : TPathCharArray;
  S                : string[fsPathname];
  Name             : string[fsName];

  function CheckErrors : Boolean;
  {-Increment block errors, return True if too many}
  begin
    Inc (FBlockErrors);
    Inc (FTotalErrors);
    if BlockErrors > FMaxBlockErrors then begin
      CheckErrors := True;
      apProtocolError (ecTooManyErrors);
      ProtocolStatus := psProtocolError;
    end else
      CheckErrors := False;
  end;

begin
  EnterCriticalSection(FProtSection);

  {Exit if protocol was cancelled while waiting for crit section}
  if FYmodemState = ryDone then begin
    LeaveCriticalSection(FProtSection);
    Exit;
  end;

  {Set TriggerID directly for TriggerAvail messages}
  if Msg = apx_TriggerAvail then
    TriggerID := aDataTrigger;

  repeat
    ComPort.DebugLog.AddDebugEntry (TApxCustomProtocol,
                                    Cardinal (AxdtYModem),
                                    Cardinal (LogYModemState[FYmodemState]),
                                    0);

    {Check for user or remote abort}
    if (Integer(TriggerID) = NoCarrierTrigger) or
       (Msg = apx_ProtocolCancel) then begin
      if Msg = apx_ProtocolCancel then begin
        Cancel;
        ProtocolStatus := psCancelRequested;
      end else
        ProtocolStatus := psAbortNoCarrier;
      FYmodemState := tyFinished;
      ForceStatus := False;
      LogFile(lfTransmitFail);
    end;

    {Show status periodically}
    if FYmodemState <> tySendXmodem then begin
      if (Integer(TriggerID) = FStatusTrigger) or ForceStatus then begin
        if TimerStarted then
          ElapsedXfrTime := ElapsedTime(Timer);
        if FComPort.Dispatcher.TimerTimeRemaining(StatusTrigger,    
                                    StatusTimeMS) <> 0 then
          StatusTimeMS := 0;
        if LongInt (StatusTimeMS) <= 0 then begin
          ShowStatus(0);
          FComPort.Dispatcher.SetTimerTrigger(StatusTrigger, StatusInterval, True);
          ForceStatus := False;
        end;
      end;
    end;

    ExitStateMachine := True;

    {Process current state}
    case FYmodemState of
      tyInitial :
        begin
          {Check for handshake character}
          FYmodemState := tyHandshake;
         HandshakeAttempt := 0;
          if not PrepHandshake then
            FYmodemState := tyFinished; 
        end;

      tyHandshake :
        if TriggerID = aDataTrigger  then begin
          if ProcessHandshake then begin
            {Start protocol timer now}
            TimerStarted := True;
            NewTimer(FTimer, 50);
            BlockErrors := 0;
            FYmodemState := tyGetFileName;
            {If GMode don't allow any more errors}
            if FGMode then
              FMaxBlockErrors := 0;
            end else begin
              {Not a valid handshake character, note error}
              if not PrepHandshake then
                FYmodemState := tyFinished;
          end;
        end else if Integer(TriggerID) = TimeoutTrigger then
          {Timeout waiting for handshake character, note error}
          if not PrepHandshake then
            FYmodemState := tyFinished; 

      tyGetFileName :
        if NextFile(FPathName) then begin
          {Open file now to get size and date stamp}
          PrepareReading;

          {Quit if we couldn't open the file}
          if ProtocolError <> ecOk then begin
            FYmodemState := tyFinished;
            goto ExitPoint;
          end;

          {Save the file name and length}
          StrLCopy (FSaveName, PathName, SizeOf (FSaveName));
          FSaveLen := SrcFileLen;

          {Make a Ymodem file header record}
          FillChar(FFileHeader^, SizeOf(FFileHeader^) + XmodemOverhead, 0);

          {Fill in the file name}
          S := StrPas(PathName);
          Name := ExtractFileName(S);
          if FlagIsSet(Flags, apIncludeDirectory) then  
            StrPCopy(S1, S)
          else
            StrPCopy(S1, Name);

          {Change name to lower case, change '\' to '/'}
          Len := StrLen(S1);
          AxCharLowerBuff(S1, Len);
          for I := 0 to Len-1 do begin
            if S1[I] = '\' then
              S1[I] := '/';
          end;
          Move(S1[0], FFileHeader^, Len);

          {Fill in file size}
          Str(SrcFileLen, S2);
          Move(S2[1], FFileHeader^[Len+2], Length(S2));
          Inc(Len, Length(S2));

          {Convert time stamp to Ymodem format and stuff in yFileHeader}
          if SrcFileDate <> 0 then begin
            S2 := ' ' + apOctalStr(apPackToYMTimeStamp(SrcFileDate));
            Move(S2[1], FFileHeader^[Len+2], Length(S2));
            Inc(Len, Length(S2)+2);
          end;

          {Determine block size from the used part of the yFileHeader}
          if Len <= 128 then begin
            BlockLen := 128;
            F1KMode := False;
            FStartChar := cSoh;
          end else begin
            BlockLen := 1024;
            F1KMode := True;
            FStartChar := cStx;
          end;

          {Init status vars for the header transfer}
          SrcFileLen := BlockLen;
          BytesRemaining := BlockLen;
          BytesTransferred := 0;
          ElapsedXfrTime := 0;
          FPathname[0] := #0;

          {Go send the file header}     
          FYmodemState := tySendFileName;
        end else
          FYModemState := tyFinished;

      tySendFileName :
        begin
          {Send the file header}
          BlockNum := 0;
          TransmitBlock (FFileHeader^, BlockLen, ' '); 
          if ProtocolError <> ecOK then begin
            FYmodemState := tyFinished;
            goto ExitPoint;
          end;

          {If we get this far we will eventually need a cleanup block}
          FFilesSent := True;

          {Wait for the buffer to drain} 
          FYmodemState := tyDraining;
          FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, DrainWait, True);
          FComPort.Dispatcher.SetStatusTrigger(OutBuffUsedTrigger, 0, True); 
        end;

      tyDraining :
        if (Integer(TriggerID) = OutBuffUsedTrigger) or
           (Integer(TriggerID) = TimeoutTrigger) then begin
          FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, FBlockWait, True); 
          FYmodemState := tyReplyPending;
        end;  

      tyReplyPending :
        if TriggerID = aDataTrigger then begin
          {Process the header reply}
          if FGMode then
            FYModemState := tyPrepXmodem
          else if ProcessBlockReply then
            FYmodemState := tyPrepXmodem
          else if CheckErrors then
            FYmodemState := tyFinished
          else
            FYmodemState := tySendFilename
        end else if Integer(TriggerID) = TimeoutTrigger then
          {Timeout waiting for header reply}
          if CheckErrors then
            FYmodemState := tyFinished
          else
            FYmodemState := tySendFilename; 

      tyPrepXmodem :
        begin
          {Reset some status vars}
          BytesTransferred := 0;
          ElapsedXfrTime := 0;

          {Restore the pathname and file size}
          if UpcaseFileNames then
            AnsiStrUpper(FSaveName);
          StrLCopy(Pathname, FSaveName, SizeOf(Pathname));
          SrcFileLen := FSaveLen;
          BytesRemaining := FSaveLen;

          {Start transmitting the file with 1K blocks}
          F1KMode := True;
          FStartChar := cStx;
          BlockLen := 1024;
          CheckType := bcCrc16;
          ForceStatus := True; 
          FXmodemState := txInitial;
          FYmodemState := tySendXmodem;
          DataBlock := nil;
          ExitStateMachine := False;
          if FComPort.Dispatcher.CharReady then
            TriggerID := aDataTrigger; 
        end;

      tySendXmodem :
        begin
          {Let the Xmodem state machine handle it}
          XState := TransmitPrim (apx_FromYmodem, TriggerID, lParam);
          if XState = 1 then begin
            if ProtocolError = ecOK then
              FYmodemState := tyInitial
            else
              FYmodemState := tyFinished;
          end;
          ExitStateMachine := True; 
        end;

      tyFinished :
        begin
          FinishReading;
          if FilesSent and (ProtocolStatus <> psCancelRequested) then begin
            {Send an empty header block to indicate end of Batch}
            FillChar(FFileHeader^, 128, 0);
            BlockNum := 0;
            F1KMode := False;
            BlockLen := 128;
            FStartChar := cSoh;
            TransmitBlock(FFileHeader^, BlockLen, ' ');
            FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, FinishWait, True);
            FComPort.Dispatcher.SetStatusTrigger(OutBuffUsedTrigger, 0, True); 
            FYmodemState := tyFinishDrain;
          end else begin
            {Never sent any files, quit without sending empty block}
            apShowLastStatus;
            apSignalFinish (True);
            FYmodemState := tyDone;
          end; 
        end;

      tyFinishDrain :
        if (Integer(TriggerID) = TimeoutTrigger) or
           (Integer(TriggerID) = OutBuffUsedTrigger) then begin
          {We're finished} 
          apShowLastStatus;
          FYmodemState := tyDone;
          apSignalFinish (True);
        end;
    end;

ExitPoint:
    {Set function result}
    case FYmodemState of
      {Leave protocol state machine}
      tyDone,
      tyReplyPending,
      tyDraining,
      tyFinishDrain:
        Finished := True;

      {Stay in protocol state machine}
      tyGetFileName,
      tySendFileName,
      tyFinished:
        Finished := False;

      {Stay in protocol machine if data available}
      tyPrepXmodem,
      tyHandshake:
        Finished := not FComPort.Dispatcher.CharReady;

      {Leave or stay as required}
      tySendXmodem:
        Finished := ExitStateMachine;
        
      else
        Finished := True;
    end;

    {If staying in state machine simulate data received}
    if not Finished then
      TriggerID := aDataTrigger;

  until Finished;

  LeaveCriticalSection(FProtSection);
end;

procedure TApxYModemDriver.PrepareReceive;
{-Prepare for Ymodem receive}
begin
  {Reset status vars}
  apResetStatus;
  ProtocolError := ecOK;
  apShowFirstStatus;
  ForceStatus := False;
  TimerStarted := False;
  FYmodemState := ryInitial;
end;

procedure TApxYModemDriver.Receive(Msg, wParam : Cardinal; lParam : LongInt);
{-Ymodem receive state machine}
label
  ExitPoint;
var
  TriggerID        : Cardinal absolute wParam;
  Code             : Integer;
  Res              : Cardinal;
  XState           : Cardinal;
  BlockSize        : Cardinal;
  BlockPos         : Integer;
  I                : Integer;
  CurSize          : LongInt;
  Finished         : Boolean;
  StatusTimeMS     : Cardinal; 
  ExitStateMachine : Boolean;
  C                : Char;
  S                : String;
  SLen             : Byte;
  S1               : ShortString;
  S1Len            : Byte absolute S1;
  Name             : String[fsName];
  NameExt          : array[0..fsName] of Char;

  function CheckErrors : Boolean;
  {-Increment block errors, return True if too many}
  begin
    Inc(FBlockErrors);
    Inc(FTotalErrors);
    if BlockErrors > FMaxBlockErrors then begin
      CheckErrors := True;
      apProtocolError(ecTooManyErrors);
      ProtocolStatus := psProtocolError;
    end else
      CheckErrors := False;
  end;

begin
  ExitStateMachine := True;

  EnterCriticalSection(FProtSection);

    {Exit if protocol was cancelled while waiting for crit section}
    if FYmodemState = ryDone then begin
      LeaveCriticalSection(FProtSection);  
      Exit;
    end;

    {Force TriggerID for TriggerAvail messages}
    if Msg = apx_TriggerAvail then
      TriggerID := aDataTrigger;

    repeat
      ComPort.DebugLog.AddDebugEntry (TApxCustomProtocol,
                                    Cardinal (AxdtYModem),
                                    Cardinal (LogYModemState[FYmodemState]),
                                    0);
      
      {Check for user abort}
      if (Integer(TriggerID) = NoCarrierTrigger) or
         (Msg = apx_ProtocolCancel) then begin
        if Msg = apx_ProtocolCancel then begin
          Cancel;
          ProtocolStatus := psCancelRequested;
        end else
          ProtocolStatus := psAbortNoCarrier;
        LogFile(lfReceiveFail);
        FYmodemState := ryFinished;
        ForceStatus := False;
      end;

      {Show status periodically}
      if FYmodemState <> ryReceiveXmodem then begin
        if (Integer(TriggerID) = StatusTrigger) or ForceStatus then begin
          if FComPort.Dispatcher.TimerTimeRemaining (StatusTrigger,    
                                    StatusTimeMS) <> 0 then
            StatusTimeMS := 0;
          if LongInt (StatusTimeMS) <= 0 then begin
            ShowStatus(0);
            FComPort.Dispatcher.SetTimerTrigger(StatusTrigger, StatusInterval, True);
            ForceStatus := False;
          end;
        end;
      end;

        {Process current state}
      case FYmodemState of
        ryInitial :
          begin
            {Manually reset status vars before getting a file header}
            SrcFileLen := 0;
            BytesRemaining := 0;
            BytesTransferred := 0;
            ElapsedXfrTime := 0;
            BlockNum := 0;
            FPathname[0] := #0;

            {Get a ymodem header block}
            FillChar (FFileHeader^, SizeOf (FFileHeader^) + XmodemOverhead, 0);
            F1KMode := False;
            CheckType := bcCrc16;
            BlockSize := 128;
            BlockNum := 0;
            FOverheadLen := 4;

            {Testing shows a short delay is required here for Telix}
            FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, ApxTelixDelay, True); 
            FYmodemState := ryDelay;
          end;

        ryDelay :
          if Integer(TriggerID) = TimeoutTrigger then begin
            {Finished with Telix delay, send handshake}
            FHandshake := GetHandshakeChar;
            FComPort.Dispatcher.PutChar(FHandshake);
            FEotCounter := 0;
            FCanCounter := 0;

            {Start waiting for handshake reply} 
            FYmodemState := ryWaitForHSReply;
            FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
          end; 

        ryWaitForHSReply :
          if TriggerID = aDataTrigger then begin
            {Got handshake reply, see if it's a block start} 
            FYmodemState := ryWaitForBlockStart;
            if FGMode then
              FMaxBlockErrors := 0;

            {Force a fresh timer for each file}
            TimerStarted := False;
          end else if Integer(TriggerID) = TimeoutTrigger then begin
            {Timeout waiting for handshake reply, resend or fail}
            if CheckErrors then
              FYmodemState := ryFinished
            else begin
              if BlockErrors > FMaxBlockErrors then
                FHandshake := ChkReq;
              FComPort.Dispatcher.PutChar(FHandshake);
              FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
            end;
          end; 

        ryWaitForBlockStart :
          if TriggerID = aDataTrigger then begin
            {Got data, see if it's a block start character}
            if CheckForBlockStart(C) then begin
              case ProcessBlockStart(C) of
                pbs128, pbs1024 :
                  begin
                    if not TimerStarted then begin
                      TimerStarted := True;
                      NewTimer(FTimer, 50);
                    end;
                    FYmodemState := ryCollectBlock;
                  end;
                pbsCancel, pbsEOT :
                  FYmodemState := ryFinished;
              end;
            end;
          end else if Integer(TriggerID) = TimeoutTrigger then
            {Timeout out waiting for rest of block, quit or resend handshake}
            if CheckErrors then
              FYmodemState := ryFinished
            else
              FYmodemState := ryInitial;

        ryCollectBlock :
          if TriggerID = aDataTrigger then begin  
            {Collect new data into DataBlock}
            if CollectBlock(FFileHeader^) then
              FYmodemState := ryProcessBlock;
          end else if Integer(TriggerID) = TimeoutTrigger then
            {Timeout out collecting initial block, quit or resend handshake}
            if CheckErrors then
              FYmodemState := ryFinished
            else
              FYmodemState := ryInitial;

        ryProcessBlock :
          begin
            {Go process data already in DataBlock}
            ReceiveBlock(FFileHeader^, BlockSize, FHandshake);
            SendHandshakeChar(FHandshake);

            {Extract file info if we got block ok}
            if ProtocolError = ecOK then begin
              {Finished if entire block is null}
              Finished := True;
              I := 3;
              while (I < 120) and Finished do begin
                if FFileHeader^[I] <> #0 then
                  Finished := False;
                Inc(I);
              end;

              {If finished, send last ack and exit}
              if Finished then begin
                FYmodemState := ryFinished;
                goto ExitPoint;
              end;

              SetLength(S, 1024);

              {Extract the file name from the header}
              BlockPos := 3;
              I := 0;
              while (FFileHeader^[BlockPos] <> #0) and
                      (BlockPos < fsPathName+2) do begin
                Inc(I);
                S[I] := FFileHeader^[BlockPos];
                if S[I] = '/' then
                  S[I] := '\';
                Inc(BlockPos);
              end;
              SLen := I;

              if UpcaseFileNames then begin
                SetLength(S, SLen);
                S := AnsiUpperCase(S);
              end;
              StrPCopy(Pathname, S);

              if not FlagIsSet(Flags, apHonorDirectory) then begin
                Name := ExtractFileName(S);
                StrPCopy(NameExt, Name);
                ApxIncludeTrailingPathDelimiterZ (PathName, DestDir); 
                StrLCat(PathName, NameExt, SizeOf (PathName));
              end;

              {Extract the file size}
              I := 1;
              Inc(BlockPos);
              while (FFileHeader^[BlockPos] <> #0) and
                      (FFileHeader^[BlockPos] <> ' ') and
                      (I <= 255) do begin
                S1[I] := FFileHeader^[BlockPos];
                Inc(I);
                Inc(BlockPos);
              end;
              Dec(I);
              S1Len := I;

              if S1Len = 0 then
                SrcFileLen := 0
              else begin
                Val(S1, FSrcFileLen, Code);
                if Code <> 0 then
                  SrcFileLen := 0;
              end;
              BytesRemaining := SrcFileLen;

              {Extract the file date/time stamp}
              I := 1;
              Inc(BlockPos);
              while (FFileHeader^[BlockPos] <> #0) and
                      (FFileHeader^[BlockPos] <> ' ') and
                      (I <= 255) do begin
                S1[I] := FFileHeader^[BlockPos];
                Inc(I);
                Inc(BlockPos);
              end;
              Dec(I);
              S1Len := I;
              if S1Len = 0 then
                FNewDT := 0
              else begin
                FNewDT := apOctalStr2Long(S1);
                if FNewDT = 0 then begin
                  {Invalid char in date/time stamp, show the error and continue}
                  FNewDT := 0;
                  ProtocolStatus := psInvalidDate;
                  ShowStatus(0);
                end;
              end;

              {Manually reset status vars before getting file}
              BytesTransferred := 0;
              ElapsedXfrTime := 0;

              {Receive the file using CRC and 1K blocks}
              F1KMode := True;
              CheckType := bcCrc16;
              BlockLen := 1024;
              FSaveLen := SrcFileLen;

              {Go prep Xmodem}
              FYmodemState := ryPrepXmodem;
            end else
              {Error getting name block...}
              if FGMode then
                {Can't recover when in GMode, go quit}
                FYmodemState := ryFinished
              else begin
                {Nak already sent, go get block again}
                FYmodemState := ryWaitForHSReply;
                FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
              end;
          end;

        ryPrepXmodem :
          begin
            FXmodemState := rxInitial;
            DataBlock := nil;
            apResetStatus;
            ProtocolStatus := psProtocolHandshake;
            FYmodemState := ryReceiveXmodem;
            ExitStateMachine := False;
            SrcFileLen := FSaveLen; 
          end;

        ryReceiveXmodem :
          begin
            ExitStateMachine := True;
            XState := ReceivePrim(apx_FromYmodem, TriggerID, lParam); 

            if XState = 1 then begin
              if ProtocolError = ecOK then begin
                {If this is a file, check for truncation and file date}
                ResetWorkFile;
                if IOResult = 0 then begin
                  {If a new file size was supplied, truncate to that length}
                  if FSaveLen <> 0 then begin

                    {Get the file size of the file (as received)}
                    CurSize := WorkFileSize; 

                    {If the requested file size is within one block, truncate the file}
                    if (CurSize - FSaveLen) < 1024 then begin
                      SeekWorkFile (FSaveLen);
                      TruncateWorkFile;
                      Res := IOResult;
                      if Res <> 0 then begin
                        apProtocolError(Res); 
                        FYmodemState := ryFinished;
                        goto ExitPoint;
                      end;
                    end;
                  end;

                  {If a new date/time stamp was specified, update the file time}
                  if FNewDT <> 0 then begin
                    FNewDT := apYMTimeStampToPack(FNewDT);
                    WorkFileSetDate (FNewDT);
                  end;
                end;
                CloseWorkFile;
                if IOResult <> 0 then ;

                {Go look for another file}
                FYmodemState := ryInitial;
                FComPort.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
                ForceStatus := True;
              end else
                FYmodemState := ryFinished;
            end; 
          end;

        ryFinished :
          begin
            apShowLastStatus;
            apSignalFinish (False);
            FYmodemState := ryDone;
          end;
      end;

ExitPoint:
      {Set function result}
      case FYmodemState of
        {Stay in state machine}
        ryInitial,
        ryOpenFile,
        ryProcessBlock,
        ryFinished,
        ryPrepXmodem:
          Finished := False;

        {Leave only if no data waiting}
        ryWaitForBlockStart,
        ryCollectBlock:
          begin
            Finished := not FComPort.Dispatcher.CharReady;
            TriggerID := aDataTrigger;
          end;

        {Stay or leave as previously specified}
        ryReceiveXmodem:
          Finished := ExitStateMachine;

        {Leave state machine}
        ryDone,
        ryDelay,
        ryWaitForHSReply:
          Finished := True;
          
        else
          Finished := True;

      end;
    until Finished;

    LeaveCriticalSection(FProtSection);
end;

procedure TApxYModemDriver.Assign (Source : TPersistent);
begin
  inherited Assign (Source);
  if Source is TApxYModemDriver then
    with Source as TApxYModemDriver do begin
      Self.FSaveLen         := FSaveLen;
      Self.FNewDT           := FNewDT;
      Self.FSaveName        := FSaveName;
      Self.FFileHeader      := FFileHeader;
      Self.FYmodemState     := FYModemState;
    end;
end;

end.
