(***** BEGIN LICENSE BLOCK *****
 * This product is dual licensed.  Select the license that is most appropriate
 * for your situation.
 *
 * Version: LGPL 2.1
 *
 * The contents of this file are subject to the Lesser GNU Public License Version
 * 2.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.fsf.org/licenses/lgpl.txt
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TurboPower Async Professional
 *
 * The Initial Developer of the Original Code is
 * TurboPower Software
 *
 * Portions created by the Initial Developer are Copyright (C) 1991-2002
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** *)
{*********************************************************}
{*                  AxZModem.pas 1.02                    *}
{*********************************************************}

{Global defines potentially affecting this unit}
{$I AxDefine.inc}

{Options required for this unit}
{$Q-,V-,I-,B-,F+,A-,X+}

unit AxZModem;
  {-Provides Zmodem receive and transmit functions}

interface

uses
  Types,
  Classes,
  SysUtils,
  LibC,
  AxSystem,
  AxMisc,
  AxPrtDrv,
  AxPort,
  AxString;

const
  {Compile-time constants}
  MaxAttentionLen = 32;           {Maximum length of attention string}
  MaxHandshakeWait = 60000;        {Time to wait for first hdr (60 secs)}
  MaxBadBlocks = 20;              {Quit if this many bad blocks}
  DefReceiveTimeout = 20000;        {Default time for received data (20 secs)}
  DrainingStatusInterval  = 1000;   {Default status interval for draining eof}
  DefFinishWaitZM = 20000;          {Wait time for ZFins, 30 secs}
  DefFinishRetry = 3;             {Retry ZFin 3 times}

  {For estimating protocol transfer times}
  ZmodemTurnDelay = 0;            {Millisecond turnaround delay}
  ZmodemOverHead  = 20;           {Default overhead for each data subpacket}

  {For checking max block sizes}
  ZMaxBlock : array[Boolean] of Cardinal = (1024, 8192);
  ZMaxWork  : array[Boolean] of Cardinal = (2048, 16384);

  {Zmodem constants}
  ZPad       = '*';                  {Pad}
  ZDle       = cCan;                 {Data link escape}
  ZBin       = 'A';                  {Binary header using Crc16}
  ZHex       = 'B';                  {Hex header using Crc16}
  ZBin32     = 'C';                  {Binary header using Crc32}

  {Zmodem frame types}
  ZrQinit    = #0;                   {Request init (to receiver)}
  ZrInit     = #1;                   {Init (to sender)}
  ZsInit     = #2;                   {Init (to receiver) (optional)}
  ZAck       = #3;                   {Acknowledge last frame}
  ZFile      = #4;                   {File info frame (to receiver)}
  ZSkip      = #5;                   {Skip to next file (to receiver)}
  ZNak       = #6;                   {Error receiving last data subpacket}
  ZAbort     = #7;                   {Abort protocol}
  ZFin       = #8;                   {Finished protocol}
  ZRpos      = #9;                   {Resume from this file position}
  ZData      = #10;                  {Data subpacket(s) follows}
  ZEof       = #11;                  {End of current file}
  ZFerr      = #12;                  {Error reading or writing file}
  ZCrc       = #13;                  {Request for file CRC (to receiver)}
  ZChallenge = #14;                  {Challenge the sender}
  ZCompl     = #15;                  {Complete}
  ZCan       = #16;                  {Cancel requested (to either)}
  ZFreeCnt   = #17;                  {Request diskfree}
  ZCommand   = #18;                  {Execute this command (to receiver)}

type
  {Main Zmodem state table}
  TApxZmodemState = (
    {Transmit states}
    tzInitial,       {Allocates buffers, sends zrqinit}
    tzHandshake,     {Wait for hdr (zrinit), rsend zrqinit on timout}
    tzGetFile,       {Call NextFile, build ZFile packet}
    tzSendFile,      {Send ZFile packet}
    tzCheckFile,     {Wait for hdr (zrpos), set next state to tzData}
    tzStartData,     {Send ZData and next data subpacket}
    tzEscapeData,    {Check for header, escape next block}
    tzSendData,      {Wait for free space in buffer, send escaped block}
    tzWaitAck,       {Wait for Ack on ZCRCW packets}
    tzSendEof,       {Send eof}
    tzDrainEof,      {Wait for output buffer to drain}
    tzCheckEof,      {Wait for hdr (zrinit)}
    tzSendFinish,    {Send zfin}
    tzCheckFinish,   {Wait for hdr (zfin)}
    tzError,         {Cleanup after errors}
    tzCleanup,       {Release buffers and other cleanup}
    tzDone,          {Signal end of protocol}

    {Receive states}
    rzRqstFile,      {Send zrinit}
    rzDelay,         {Delay handshake for Telix}
    rzWaitFile,      {Waits for hdr (zrqinit, zrfile, zsinit, etc)}
    rzCollectFile,   {Collect file info into work block}
    rzSendInit,      {Extract send init info}
    rzSendBlockPrep, {Discard last two chars of previous hex packet}
    rzSendBlock,     {Collect sendinit block}
    rzSync,          {Send ZrPos with current file position}
    rzStartFile,     {Extract file info, prepare writing, etc., put zrpos}
    rzStartData,     {Wait for hdr (zrdata)}
    rzCollectData,   {Collect data subpacket}
    rzGotData,       {Got dsp, put it}
    rzWaitEof,       {Wait for hdr (zreof)}
    rzEndOfFile,     {Close file, log it, etc}
    rzSendFinish,    {Send ZFin, goto rzWaitOO}
    rzCollectFinish, {Check for OO, goto rzFinish}
    rzError,         {Handle errors while file was open}
    rzWaitCancel,    {Wait for the cancel to leave the outbuffer}
    rzCleanup,       {Clean up buffers, etc.}
    rzDone);         {Signal end of protocol}

type
  TApxHeaderState = (
    hsNone,          {Not currently checking for a header}
    hsGotZPad,       {Got initial or second asterisk}
    hsGotZDle,       {Got ZDle}
    hsGotZBin,       {Got start of binary header}
    hsGotZBin32,     {Got start of binary 32 header}
    hsGotZHex,       {Got start of hex header}
    hsGotHeader);    {Got complete header}

  {Hex header collection states}
  TApxHexHeaderStates = (
    hhFrame,         {Processing frame type char}
    hhPos1,          {Processing 1st position info byte}
    hhPos2,          {Processing 2nd position info byte}
    hhPos3,          {Processing 3rd position info byte}
    hhPos4,          {Processing 4th position info byte}
    hhCrc1,          {Processing 1st CRC byte}
    hhCrc2);         {Processing 2nd CRC byte}

  {Binary header collection states}
  TApxBinaryHeaderStates = (
    bhFrame,         {Processing frame type char}
    bhPos1,          {Processing 1st position info byte}
    bhPos2,          {Processing 2nd position info byte}
    bhPos3,          {Processing 3rd position info byte}
    bhPos4,          {Processing 1th position info byte}
    bhCrc1,          {Processing 1st CRC byte}
    bhCrc2,          {Processing 2nd CRC byte}
    bhCrc3,          {Processing 3rd CRC byte}
    bhCrc4);         {Processing 4th CRC byte}

  {Only two states possible when receiving blocks}
  TApxReceiveBlockStates = (
    rbData,          {Receiving data bytes}
    rbCrc);          {Receiving block check bytes}

type    
  {Describes data area of headers}
  TApxPosFlags = array[0..3] of Byte;
    
type
  TApxZModemDriver = class (TApxBaseProtocolDriver)
  
    private
    protected
      {General}
      FLastFrame       : Char;            {Holds last frame type for status}
      FTerminator      : Char;            {Current block type}
      FHeaderType      : Char;            {Current header type}
      FZmodemState     : TApxZmodemState;    {Current Zmodem state}
      FHeaderState     : TApxHeaderState;    {General header state}
      FHexHdrState     : TApxHexHeaderStates; {Current hex header state}
      FBinHdrState     : TApxBinaryHeaderStates; {Current binary header state}
      FRcvBlockState   : TApxReceiveBlockStates; {Current receive block state}
      FFileMgmtOverride: Boolean;         {True to override senders file mg opts}
      FReceiverRecover : Boolean;         {True to force file recovery}
      FUseCrc32        : Boolean;         {True when using 32bit CRCs}
      FCanCrc32        : Boolean;         {True when Crc32 capable}
      FHexPending      : Boolean;         {True for next char in hex pair}
      FEscapePending   : Boolean;         {True for next char in esc pair}
      FEscapeAll       : Boolean;         {Escaping all ctl chars}
      FControlCharSkip : Boolean;         {True to skip all ctrl chars}
      FWasHex          : Boolean;         {True if processing hex header}
      FDiscardCnt      : Cardinal;        {Characters discarded so far}
      FConvertOpts     : Cardinal;        {File conversion opts rqst by sender}
      FFileMgmtOpts    : Cardinal;        {File mgmt opts rqst by sender}
      FTransportOpts   : Cardinal;        {File transport opts rqst by sender}
      FFinishRetry     : Cardinal;        {Times to resend ZFin}
      FWorkSize        : Cardinal;        {Index into working buffer}
      FCanCount        : Cardinal;        {Track contiguous <cancels>}
      FHexChar         : Cardinal;        {Saved hex value}
      FCrcCnt          : Cardinal;        {Number of CRC bytes rcv'd}
      FOCnt            : Cardinal;        {Number of 'O's rcv'd}
      FLastFileOfs     : LongInt;         {File position reported by remote}
      FAttentionStr    : array[1..MaxAttentionLen] of Byte; {Attn string value}

      {For controlling autoadjustment of block size}
      FUse8KBlocks     : Boolean;         {True when using 8K blocks}
      FTookHit         : Boolean;         {True if we got ZrPos packet}
      FGoodAfterBad    : Cardinal;        {Holds count of good blocks}

      {Working buffers}
      FDataBlockLen    : Cardinal;        {Count of valid data in DataBlock}
      FDataInTransit   : LongInt;         {Amount of unacked data in transit}
      FWorkBlock       : PApxWorkBlock;      {Holds fully escaped data block}

      {Receiving...}
      FRcvFrame        : Char;            {Type of last received frame}
      FRcvHeader       : TApxPosFlags;       {Received header}

      {Transmitting...}
      FRcvBuffLen      : Cardinal;        {Size of receiver's buffer}
      FLastChar        : Char;            {Last character sent}
      FTransHeader     : TApxPosFlags;       {Header to transmit}
      FZRQINITValue    : LongInt;         {Optional ZRQINIT value}

      procedure DeallocBuffers;
      function AllocBuffers : Boolean;
      procedure InitData;
      procedure PutCharEscaped (C : Char);
      procedure UpdateBlockCheck (CurByte: Byte);
      procedure SendBlockCheck;
      function VerifyBlockCheck : Boolean;
      function GotCancel : Boolean;
      function GetCharStripped (var C : Char) : Boolean;
      procedure PutAttentionString;
      procedure PutCharHex (C : Char);
      procedure PutHexHeader(FrameType : Char);
      procedure GetCharEscaped (var C : Char);
      procedure GetCharHex (var C : Char);
      function CollectHexHeader : Boolean;
      function CollectBinaryHeader(Crc32 : Boolean) : Boolean;
      procedure CheckForHeader;
      function BlockError (OkState, ErrorState : TApxZmodemState;
                           MaxErrors : Cardinal) : Boolean;
      function ReceiveBlock (var Block : TApxDataBlock) : Boolean;
      procedure ExtractFileInfo;
      procedure WriteDataBlock;
      procedure PutBinaryHeader(FrameType : Char);
      function EscapeChar (C : Char) : Boolean;
      procedure EscapeBlock (var Block : TApxDataBlock;
                                 BLen : Cardinal);
      procedure TransmitBlock;
      procedure ExtractReceiverInfo;
      procedure InsertFileInfo;
      procedure GotZrPos;
      procedure ProcessHeader;

    public
      {Constructors/destructors}
      function Init(Options : Cardinal) : Integer; override;
      procedure Done; override;

      function Reinit : Integer; override;
      procedure DonePart; override;

      {Options}
      function SetFileMgmtOptions(Override, SkipNoFile : Boolean;
                              FOpt : Byte) : Integer;
      function SetRecoverOption(OnOff : Boolean) : Integer;
      function SetBigSubpacketOption(UseBig : Boolean) : Integer;
      function SetZmodemFinishWait(NewWait : Cardinal;
                               NewRetry : Byte) : Integer;

      {Control}
      procedure PrepareTransmit; override;
      procedure PrepareReceive; override;            
      procedure Transmit(Msg, wParam : Cardinal; lParam : LongInt); override;
      procedure Receive(Msg, wParam : Cardinal; lParam : LongInt); override;

      procedure PrepareWriting; override;
      procedure FinishWriting; override;
      procedure Cancel;
      procedure Assign (Source : TPersistent); override;

    published
      property FileMgmtOverride: Boolean
               read FFileMgmtOverride write FFileMgmtOverride;
      property ReceiverRecover : Boolean
               read FReceiverRecover write FReceiverRecover;
      property UseCrc32        : Boolean
               read FUseCRC32 write FUseCRC32;
      property ConvertOpts     : Cardinal
               read FConvertOpts write FConvertOpts;
      property FileMgmtOpts    : Cardinal
               read FFileMgmtOpts write FFileMgmtOpts;
      property TransportOpts   : Cardinal
               read FTransportOpts write FTransportOpts;
      property FinishRetry     : Cardinal
               read FFinishRetry write FFinishRetry;
      property Use8KBlocks     : Boolean
               read FUse8KBlocks write FUse8KBlocks;  
      property ZRQINITValue    : LongInt
               read FZRQINITValue write FZRQINITValue;         

  end;

implementation

uses
  AxProtcl;

{$IFDEF TRIALRUN}
  {$I TRIAL07.INC}
  {$I TRIAL03.INC}
  {$I TRIAL01.INC}
{$ENDIF}

const
  {For various hex char manipulations}
  HexDigits : array[0..15] of Char = '0123456789abcdef';

  {For initializing block check values}
  CheckInit : array[Boolean] of Integer = (0, -1);

  {For manipulating file management masks}
  FileMgmtMask = $07;              {Isolate file mgmnt values}
  FileSkipMask = $80;              {Skip file if dest doesn't exist}

  {Only supported conversion option}
  FileRecover = $03;               {Resume interrupted file transfer}

  {Data subpacket terminators}
  ZCrcE      = 'h';                {End  - last data subpacket of file}
  ZCrcG      = 'i';                {Go   - no response necessary}
  ZCrcQ      = 'j';                {Ack  - requests ZACK or ZRPOS}
  ZCrcW      = 'k';                {Wait - sender waits for answer}

  {Translate these escaped sequences}
  ZRub0      = 'l';                {Translate to $7F}
  ZRub1      = 'm';                {Translate to $FF}

  {Byte offsets for pos/flag bytes}
  ZF0 = 3;                         {Flag byte 3}
  ZF1 = 2;                         {Flag byte 2}
  ZF2 = 1;                         {Flag byte 1}
  ZF3 = 0;                         {Flag byte 0}
  ZP0 = 0;                         {Position byte 0}
  ZP1 = 1;                         {Position byte 1}
  ZP2 = 2;                         {Position byte 1}
  ZP3 = 3;                         {Position byte 1}

  {Bit masks for ZrInit}
  CanFdx  = $0001;           {Can handle full-duplex}
  CanOvIO = $0002;           {Can do disk and serial I/O overlaps}
  CanBrk  = $0004;           {Can send a break}
  CanCry  = $0008;           {Can encrypt/decrypt, not supported}
  CanLzw  = $0010;           {Can LZ compress, not supported}
  CanFc32 = $0020;           {Can use 32 bit CRC}
  EscAll  = $0040;           {Escapes all control chars, not supported}
  Esc8    = $0080;           {Escapes the 8th bit, not supported}

  {Bit masks for ZsInit}
  TESCtl  = $0040;           {Sender asks for escaped ctl chars, not supported}
  TESC8   = $0080;           {Sender asks for escaped hi bits, not supported}

  {Character constants}
  cDleHi  = Char(Ord(cDle) + $80);
  cXonHi  = Char(Ord(cXon) + $80);
  cXoffHi = Char(Ord(cXoff) + $80);

  aDataTrigger = 0;

  LogZModemState : array[TApxZmodemState] of TDispatchSubType = (        
     dsttzInitial, dsttzHandshake, dsttzGetFile, dsttzSendFile,
     dsttzCheckFile, dsttzStartData, dsttzEscapeData, dsttzSendData,
     dsttzWaitAck, dsttzSendEof, dsttzDrainEof, dsttzCheckEof,
     dsttzSendFinish, dsttzCheckFinish, dsttzError, dsttzCleanup,
     dsttzDone,
     dstrzRqstFile, dstrzDelay, dstrzWaitFile, dstrzCollectFile,
     dstrzSendInit, dstrzSendBlockPrep, dstrzSendBlock, dstrzSync,
     dstrzStartFile, dstrzStartData, dstrzCollectData, dstrzGotData,
     dstrzWaitEof, dstrzEndOfFile, dstrzSendFinish, dstrzCollectFinish,
     dstrzError, dstrzWaitCancel, dstrzCleanup, dstrzDone);


procedure TApxZModemDriver.PrepareWriting;
{-Prepare to save protocol blocks (usually opens a file)}
var
  FileExists     : Boolean;
  FileSkip       : Boolean;
  Result         : Cardinal;
  FileLen        : LongInt;
  FileDate       : LongInt;
  SeekPoint      : LongInt;
  FileStartOfs   : LongInt;
  YMTSrcFileDate : LongInt;
  FileOpt        : Byte;

  procedure ErrorCleanup;
  begin
    CloseWorkFile;
    if IOResult <> 0 then ;
      FreeMem(FileBuffer, ApxFileBufferSize);
  end;

  { Allows a 1 sec fudge to compensate for FAT timestamp rounding }
  function YMStampEqual(YMStamp1, YMStamp2 : LongInt) : Boolean;
  begin
    Result := abs(YMStamp1 - YMStamp2) <= 1;
  end;

  { Allows a 1 sec fudge to compensate for FAT timestamp rounding }
  function YMStampLessOrEqual(YMStamp1, YMStamp2 : LongInt) : Boolean;
  begin
    Result := YMStampEqual(YMStamp1, YMStamp2) or (YMStamp1 < YMStamp2);
  end;

begin
  ProtocolError := ecOK;
  {Allocate a file buffer}
  FileBuffer := AllocMem(ApxFileBufferSize);

  {Set file mgmt options}
  FileSkip := (FFileMgmtOpts and FileSkipMask) <> 0;
  FileOpt := FFileMgmtOpts and FileMgmtMask;

  {Check for a local request for file recovery}
  if FReceiverRecover then
    FConvertOpts := FConvertOpts or FileRecover;

  {Does the file exist already?}
  SaveMode := FileMode;
  FileMode := 0;
  ResetWorkFile;
  Result := IOResult;
  FileMode := SaveMode;

  {Exit on errors other than FileNotFound}
  if (Result <> 0) and (Result <> 2) then begin
    apProtocolError(-Result);
    ErrorCleanup;
    Exit;
  end;

  {Note if file exists, its size and timestamp}
  FileExists := (Result = 0);
  if FileExists then begin
    FileLen := WorkFileSize;
    FileDate := WorkFileGetDate;
    FileDate := apPackToYMTimeStamp(FileDate);
  end else begin
    FileLen := 0;
    FileDate := 0;
  end;
  CloseWorkFile;
  if IOResult = 0 then ;

  {If recovering, skip all file managment checks and go append file}
  if FileExists and
     (SrcFileLen > FileLen) and
     ((FConvertOpts and FileRecover) = FileRecover) then begin
    SeekPoint := FileLen;
    FileStartOfs := FileLen;
    InitFilePos := FileLen;
  end else begin
    {Tell status we're not recovering}
    InitFilePos := 0;

    {Check for skip condition}
    if FileSkip and not FileExists then begin
      ProtocolStatus := psFileDoesntExist;
      ErrorCleanup;
      Exit;
    end;

    {Process the file management options}
    SeekPoint := 0;
    FileStartOfs := 0;
    case FileOpt of
      zfWriteNewerLonger : {Transfer only if new, newer or longer}
        if FileExists then begin
          YMTSrcFileDate := apPackToYMTimeStamp(SrcFileDate);
          if YMStampLessOrEqual(YMTSrcFileDate, FileDate) and
             (SrcFileLen <= FileLen) then begin
            ProtocolStatus := psCantWriteFile;
            ErrorCleanup;
            Exit;
          end;
        end;
      zfWriteAppend :      {Transfer regardless, append if exists}
        if FileExists then
          SeekPoint := FileLen;
      zfWriteClobber :     {Transfer regardless, overwrite} ;
        {Nothing to do, this is the normal behavior}
      zfWriteDifferent :   {Transfer only if new, size diff, or dates diff}
        if FileExists then begin
          YMTSrcFileDate := apPackToYMTimeStamp(SrcFileDate);
          if YMStampEqual(YMTSrcFileDate, FileDate) and
             (SrcFileLen = FileLen) then begin
            ProtocolStatus := psCantWriteFile;
            ErrorCleanup;
            Exit;
          end;
        end;
      zfWriteProtect :     {Transfer only if dest file doesn't exist}
        if FileExists then begin
          ProtocolStatus := psCantWriteFile;
          ErrorCleanup;
          Exit;
        end;
      zfWriteCrc,          {Not supported, treat as WriteNewer}
      zfWriteNewer :       {Transfer only if new or newer}
        if FileExists then begin
          YMTSrcFileDate := apPackToYMTimeStamp(SrcFileDate);
          if YMStampLessOrEqual(YMTSrcFileDate, FileDate) then
          begin
            ProtocolStatus := psCantWriteFile;
            ErrorCleanup;
            Exit;
          end;
        end;
    end;
  end;

  {Rewrite or append to file}
  if SeekPoint = 0 then begin
    RewriteWorkFile;
    {New or overwriting destination file}
  end else begin
    {Appending to file}
    ResetWorkFile;
    SeekWorkFile (SeekPoint); 
  end;
  Result := IOResult;
  if Result <> 0 then begin
    apProtocolError(-Result);
    ErrorCleanup;
    Exit;
  end;

  {Initialized the buffer management vars}
  FileOfs := FileStartOfs;
  StartOfs := FileStartOfs;
  LastOfs := FileStartOfs;
  EndOfs := StartOfs + ApxFileBufferSize;
  FileOpen := True;
end;

procedure TApxZModemDriver.FinishWriting;
{-Cleans up after saving all protocol blocks}
var
  BytesToWrite : Integer;
  BytesWritten : Integer;
  Result       : Cardinal;
begin
  if FileOpen then begin
    {Error or end-of-file, commit buffer}
    BytesToWrite := FileOfs - StartOfs;
    BlockWriteWorkFile (FileBuffer^, BytesToWrite, BytesWritten);
    Result := IOResult;
    if (Result <> 0) then
      apProtocolError(-Result);
    if (BytesToWrite <> BytesWritten) then
      apProtocolError(ecDiskFull);

    {Set the timestamp to that of the source file}
    if ProtocolError = ecOK then begin
      WorkFileSetDate (SrcFileDate);
    end;

    {Clean up}
    CloseWorkFile;
    if IOResult <> 0 then ;
    FreeMem(FileBuffer, ApxFileBufferSize);
    FileOpen := False;
  end;
end;

procedure TApxZModemDriver.DeallocBuffers;
{-Release block and work buffers}
begin
  FreeMem(DataBlock, ZMaxBlock[FUse8KBlocks]);
  FreeMem(FWorkBlock, ZMaxWork[FUse8KBlocks]);
end;

function TApxZModemDriver.AllocBuffers : Boolean;
begin
  DataBlock := nil;
  FWorkBlock := nil;
  DataBlock := AllocMem(ZMaxBlock[FUse8KBlocks]);
  FWorkBlock := AllocMem(ZMaxWork[FUse8KBlocks]);
  AllocBuffers := True;
end;

procedure TApxZModemDriver.InitData;
{-Init the protocol data}
{$IFDEF TRIALRUN}
  {$I TRIAL04.INC}
{$ENDIF}
begin
{$IFDEF TRIALRUN}
  TC;
{$ENDIF}
  {Init this object's fields}
  CurProtocol := Zmodem;
  BatchProtocol := True;
  FileOpen := False;
  FileOfs := 0;
  RcvTimeout := DefReceiveTimeout;
  CheckType := bcCrc32;
  SrcFileDate := 0;
  BlockLen := ZMaxBlock[FUse8KBlocks];
  Overhead := ZmodemOverhead;
  TurnDelay := ZmodemTurnDelay;
  FinishWait := DefFinishWaitZM;
  HandshakeWait := MaxHandshakeWait;
  FillChar(FAttentionStr, MaxAttentionLen, 0);
  FLastFileOfs := 0;
  FUseCrc32 := True;
  FCanCrc32 := True;
  FReceiverRecover := False;
  FFileMgmtOpts := zfWriteNewer;
  FFileMgmtOverride := False;
  FTookHit := False;
  FGoodAfterBad := 0;
  FEscapePending := False;
  FHexPending := False;
  FFinishRetry := DefFinishRetry;
  FEscapeAll := False;
end;

function TApxZModemDriver.Init(Options : Cardinal) : Integer;
{-Allocates and initializes a protocol control block with options}
const
  MinSize : array[Boolean] of Cardinal = (2048+30, 16384+30);
var
  {InSize, } OutSize : Cardinal;
begin
  {Check for adequate output buffer size}
  OutSize := ComPort.OutSize;
  if OutSize < MinSize[FlagIsSet(Options, apZmodem8K)] then begin
    Init := ecOutputBufferTooSmall;
    Exit;
  end;

  {Allocate protocol record, init base data}
  if apInitProtocolData(ComPort, Options) <> ecOk then begin
    Init := ecOutOfMemory;
    Exit;
  end;

  {Allocate data blocks}
  FUse8KBlocks := FlagIsSet(Options, apZmodem8K);
  if not AllocBuffers then begin
    Init := ecOutOfMemory;
    Done;
    Exit;
  end;

  {Can't fail after this}
  Init := ecOK;

  {Init the data}
  InitData;
end;

function TApxZModemDriver.Reinit : Integer;
{-Allocates and init just the Zmodem stuff}
begin
  {Allocate data blocks}
  FUse8KBlocks := False;
  if not AllocBuffers then begin
    Reinit := ecOutOfMemory;
    Done;
    Exit;
  end;

  {Can't fail after this}
  Reinit := ecOK;

  {Init the data}
  InitData;
end;

procedure TApxZModemDriver.Done;
{-Dispose of Zmodem}
begin
  DeallocBuffers;
  apDoneProtocol;
end;

procedure TApxZModemDriver.DonePart;
{-Dispose of just the Zmodem stuff}
begin
  DeallocBuffers;
end;

function TApxZModemDriver.SetFileMgmtOptions(Override, SkipNoFile : Boolean;
                                 FOpt : Byte) : Integer;
{-Set file mgmt options to use when sender doesn't specify}
const
    SkipMask : array[Boolean] of Byte = ($00, $80);
begin
  if CurProtocol <> Zmodem then begin
    SetFileMgmtOptions := ecBadProtocolFunction;
    Exit;
  end;

  SetFileMgmtOptions := ecOK;
  FFileMgmtOverride := Override;
  FFileMgmtOpts := (FOpt and FileMgmtMask) or SkipMask[SkipNoFile];
end;

function TApxZModemDriver.SetRecoverOption(OnOff : Boolean) : Integer;
{-Turn file recovery on (will be ignored if dest file doesn't exist)}
begin
  if CurProtocol <> Zmodem then
    SetRecoverOption := ecBadProtocolFunction
  else begin
    SetRecoverOption := ecOK;
    FReceiverRecover := OnOff;
  end;
end;

function TApxZModemDriver.SetBigSubpacketOption(UseBig : Boolean) : Integer;
{-Turn on/off 8K subpacket support}
begin
  SetBigSubpacketOption := ecOk;
  if CurProtocol <> Zmodem then
    SetBigSubpacketOption := ecBadProtocolFunction
  else if UseBig <> FUse8KBlocks then begin
    {Changing block sizes, get rid of old buffers}
    DeallocBuffers;

    {Set new size and allocate buffers}
    if UseBig then
      Flags := Flags or apZmodem8K
    else
      Flags := Flags and not apZmodem8K;
    FUse8KBlocks := UseBig;
    if not AllocBuffers then begin
      SetBigSubpacketOption := ecOutOfMemory;
      Exit;
    end;
    BlockLen := ZMaxBlock[FUse8KBlocks];
  end;
end;

function TApxZModemDriver.SetZmodemFinishWait(NewWait : Cardinal;
                                 NewRetry : Byte) : Integer;
{-Set new finish wait and retry values}
begin
  if CurProtocol <> Zmodem then
    SetZmodemFinishWait := ecBadProtocolFunction
  else begin
    SetZmodemFinishWait := ecOK;
    if FinishWait <> 0 then
      FinishWait := NewWait;
    FFinishRetry := NewRetry;
  end;
end;

procedure TApxZModemDriver.PutCharEscaped (C : Char);
{-Transmit with C with escaping as required}
var
  C1 : Char;
  C2 : Char;
begin
  {Check for chars to escape}
  if FEscapeAll and ((Byte(C) and $60) = 0) then begin
    {Definitely needs escaping}
    ComPort.PutChar(ZDle);
    FLastChar := Char(Byte(C) xor $40);
  end else if (Byte(C) and $11) = 0 then
    {No escaping, just send it}
    FLastChar := C
  else begin
    {Might need escaping}
    C1 := Char(Byte(C) and $7F);
    C2 := Char(Byte(FLastChar) and $7F);
    case C of
      cXon, cXoff, cDle,        {Escaped control chars}
      cXonHi, cXoffHi, cDleHi,  {Escaped hibit control chars}
      ZDle :                    {Escape the escape char}
        begin
          ComPort.PutChar(ZDle);
          FLastChar := Char(Byte(C) xor $40);
        end;
      else
        if ((C1 = cCR) and (C2 = #$40)) then begin
          ComPort.PutChar(ZDle);
          FLastChar := Char(Byte(C) xor $40);
        end else
          FLastChar := C;
    end;
  end;
  ComPort.PutChar(FLastChar);
end;

procedure TApxZModemDriver.UpdateBlockCheck (CurByte: Byte);
{-Updates the block check character (whatever it is)}
begin
  if FUseCrc32 then
    BlockCheck := apUpdateCrc32(CurByte, BlockCheck)
  else
    BlockCheck := apUpdateCrc(CurByte, BlockCheck);
end;

procedure TApxZModemDriver.SendBlockCheck;
    {-Makes final adjustment and sends the aBlockCheck character}
type
  QB = array[1..4] of char;
var
  I : Byte;
begin
  if FUseCrc32 then begin
    {Complete and send a 32 bit CRC}
    BlockCheck := not BlockCheck;
    for I := 1 to 4 do
      PutCharEscaped (QB (BlockCheck)[I]);
  end else begin
    {Complete and send a 16 bit CRC}
    UpdateBlockCheck(0);
    UpdateBlockCheck(0);
    PutCharEscaped(Char (Hi (BlockCheck)));
    PutCharEscaped(Char (Lo (BlockCheck)));
  end;
end;

function TApxZModemDriver.VerifyBlockCheck : Boolean;
{-checks the block check value}
begin
  {Assume a block check error}
  VerifyBlockCheck := False;

  if FUseCrc32 then begin
    if BlockCheck <> $DEBB20E3 then
      Exit
  end else begin
    UpdateBlockCheck(0);
    UpdateBlockCheck(0);
    if BlockCheck <> 0 then
      Exit;
  end;

  {If we get here, the block check value is ok}
  VerifyBlockCheck := True;
end;

procedure TApxZModemDriver.Cancel;
{-Sends the cancel string}
const
  {Cancel string is 8 CANs followed by 8 Backspaces}
  CancelStr : array[0..16] of Char = #24#24#24#24#24#24#24#24#8#8#8#8#8#8#8#8#0;
var
  TotalOverhead : Cardinal;
  OutBuff : Cardinal;
begin
  {Flush anything that might be left in the output buffer}
  OutBuff := ComPort.OutBuffUsed;
  if OutBuff > BlockLen then begin
    TotalOverhead := Overhead * (OutBuff div BlockLen);
    Dec(FBytesTransferred, Outbuff - TotalOverhead);
  end;
  ComPort.FlushOutBuffer;

  {Send the cancel string}
  ComPort.PutBlock(CancelStr, StrLen(CancelStr));
  ProtocolStatus := psCancelRequested;
  ForceStatus := True;
end;

function TApxZModemDriver.GotCancel : Boolean;
{-Return True if CanCount >= 5}
begin
  Inc(FCanCount);
  if FCanCount >= 5 then begin
    ProtocolStatus := psCancelRequested;
    ForceStatus := True;
    GotCancel := True;
  end else
    GotCancel := False;
end;

function TApxZModemDriver.GetCharStripped (var C : Char) : Boolean;
{-Get next char, strip hibit, discard Xon/Xoff, return False for no char}
begin
  {Get a character, discard Xon and Xoff}
  repeat
    if ComPort.CharReady then
      ComPort.Dispatcher.GetChar(C)
    else begin
      GetCharStripped := False;
      Exit;
    end;
  until (C <> cXon) and (C <> cXoff);

  {Strip the high-order bit}
  C := Char(Ord(C) and Ord(#$7F));

  {Handle cancels}
  if (C = cCan) then begin
    if GotCancel then begin
      GetCharStripped := False;
      Exit
    end;
  end else
    FCanCount := 0;
  GetCharStripped := True;
end;

procedure TApxZModemDriver.PutAttentionString;
{-Puts a string (#221 = Break, #222 = Delay)}
var
  I  : Cardinal;
begin
  I := 1;
  while FAttentionStr[I] <> 0 do begin
    case FAttentionStr[I] of
      $DD : {Remote wants Break as his attention signal}
        ComPort.SendBreak(50);
      $DE : {Remote wants us to pause for one second}
        DelayMS(1000, True);
      else   {Remote wants us to send a normal char}
        ComPort.PutChar(Chr(FAttentionStr[I]));
    end;
    Inc(I);
  end;
end;

procedure TApxZModemDriver.PutCharHex (C : Char);
{-Sends C as two hex ascii digits}
var
  B : Byte absolute C;
begin
  ComPort.PutChar(HexDigits[B shr 4]);
  ComPort.PutChar(HexDigits[B and $0F]);
end;

procedure TApxZModemDriver.PutHexHeader(FrameType : Char);
{-Sends a hex header}
const
  HexHeaderStr : array[0..4] of Char = ZPad+ZPad+ZDle+ZHex;
var
  SaveCrc32 : Boolean;
  Check     : Cardinal;
  I         : Byte;
  C         : Char;
begin
  {Initialize the aBlockCheck value}
  SaveCrc32 := FUseCrc32;
  FUseCrc32 := False;
  BlockCheck := 0;

  {Send the header and the frame type}
  ComPort.PutBlock(HexHeaderStr, SizeOf(HexHeaderStr)-1);
  PutCharHex(FrameType);
  UpdateBlockCheck(Ord(FrameType));

  {Send the position/flag bytes}
  for I := 0 to SizeOf(FTransHeader)-1 do begin
    PutCharHex(Char(FTransHeader[I]));
    UpdateBlockCheck(FTransHeader[I]);
  end;

  {Update Crc16 and send it (hex encoded)}
  UpdateBlockCheck(0);
  UpdateBlockCheck(0);
  Check := Cardinal(BlockCheck);
  PutCharHex(Char(Hi(Check)));
  PutCharHex(Char(Lo(Check)));

  {End with a carriage return, hibit line feed}
  ComPort.PutChar(cCR);
  C := Chr(Ord(cLF) or $80);
  ComPort.PutChar(C);

  {Conditionally send Xon}
  if (FrameType <> ZFin) and (FrameType <> ZAck) then
    ComPort.PutChar(cXon);

  {Note frame type for status}
  FLastFrame := FrameType;

  {Restore crc type}
  FUseCrc32 := SaveCrc32;
end;

procedure TApxZModemDriver.GetCharEscaped (var C : Char);
{-Get a character (handle data link escaping)}
label
  Escape;
begin
  FControlCharSkip := False;

  {Go get escaped char if we already have the escape}
  if FEscapePending then
    goto Escape;

  {Get a character}
  ComPort.Dispatcher.GetChar(C);

  {Process char}
  case C of
    cXon,
    cXoff,
    cXonHi,
    cXoffHi :
      begin
        {unescaped control char, ignore it}
        FControlCharSkip := True;
        Exit;
      end;
  end;

  {If not data link escape or cancel then just return the character}
  if (C <> ZDle) then begin
    FCanCount := 0;
    Exit;
  end else if GotCancel then
    {Got 5 cancels, ZDle's, in a row}
    Exit;

Escape:
  {Need another character, get it or say we're pending}
  if ComPort.CharReady then begin
    FEscapePending := False;
    ComPort.Dispatcher.GetChar(C);

    {If cancelling make sure we get at least 5 of them}
    if (C = cCan) then begin
      GotCancel;
      Exit;
    end else begin
      {Must be an escaped character}
      FCanCount := 0;
      case C of
        ZCrcE : {Last DataSubpacket of file}
          ProtocolStatus := psGotCrcE;
        ZCrcG : {Normal DataSubpacket, no response necessary}
          ProtocolStatus := psGotCrcG;
        ZCrcQ : {ZAck or ZrPos requested}
          ProtocolStatus := psGotCrcQ;
        ZCrcW : {DataSubpacket contains file information}
          ProtocolStatus := psGotCrcW;
        ZRub0 :         {Ascii delete}
          C := #$7F;
        ZRub1 :         {Hibit Ascii delete}
          C := #$FF;
        else            {Normal escaped character}
          C := Char(Ord(C) xor $40)
      end;
    end;
  end else
    FEscapePending := True;
end;

procedure TApxZModemDriver.GetCharHex (var C : Char);
{-Return a character that was transmitted in hex}
label
  Hex;

  function NextHexNibble : Byte;
  {-Gets the next char, returns it as a hex nibble}
  var
    C : Char;
  begin
    {Get the next char, assume it's ascii hex character}
    ComPort.Dispatcher.GetChar(C);

    {Handle cancels}
    if (C = cCan) then begin
      if GotCancel then begin
        NextHexNibble := 0;
        Exit;
      end;
    end else
      FCanCount := 0;

    {Ignore errors, they'll eventually show up as bad blocks}
    NextHexNibble := Pos (C, HexDigits) - 1;
  end;

begin
  if FHexPending then
    goto Hex;
  FHexChar := NextHexNibble shl 4;
Hex:
  if ComPort.CharReady then begin
    FHexPending := False;
    Inc(FHexChar, NextHexNibble);
    C := Chr(FHexChar);
  end else
    FHexPending := True;
end;

function TApxZModemDriver.CollectHexHeader : Boolean;
{-Gets the data and trailing portions of a hex header}
var
  C : Char;
begin
  {Assume the header isn't ready}
  CollectHexHeader := False;

  GetCharHex(C);
  if FHexPending or (ProtocolStatus = psCancelRequested) then
    Exit;

  {Init block check on startup}
  if FHexHdrState = hhFrame then begin
    BlockCheck := 0;
    FUseCrc32 := False;
  end;

  {Always update the block check}
  UpdateBlockCheck (Ord (C));

  {Process this character}
  case FHexHdrState of
    hhFrame :
      FRcvFrame := C;
    hhPos1..hhPos4 :
      FRcvHeader[Ord(FHexHdrState)-1] := Ord(C);
    hhCrc1 :
      {just keep going} ;
    hhCrc2 :
      if not VerifyBlockCheck then begin
        ProtocolStatus := psBlockCheckError;
        Inc(FTotalErrors);
        FHeaderState := hsNone;
      end else begin
        {Say we got a good header}
        CollectHexHeader := True;
      end;
  end;

  {Goto next state}
  if FHexHdrState <> hhCrc2 then
    Inc(FHexHdrState)
  else
    FHexHdrState := hhFrame;
end;

function TApxZModemDriver.CollectBinaryHeader (Crc32 : Boolean) : Boolean;
{-Collects a binary header, returns True when ready}
var
  C : Char;
begin
  {Assume the header isn't ready}
  CollectBinaryHeader := False;

  {Get the waiting character}
  GetCharEscaped(C);
  if FEscapePending or (ProtocolStatus = psCancelRequested) then
    Exit;
  if FControlCharSkip then
    Exit;

  {Init block check on startup}
  if FBinHdrState = bhFrame then begin
    FUseCrc32 := Crc32;
    BlockCheck := CheckInit[FUseCrc32];
  end;

  {Always update the block check}
  UpdateBlockCheck (Ord (C));

  {Process this character}
  case FBinHdrState of
    bhFrame :
      FRcvFrame := C;
    bhPos1..bhPos4 :
      FRcvHeader[Ord(FBinHdrState)-1] := Ord(C);
    bhCrc2 :
      if not FUseCrc32 then begin
        if not VerifyBlockCheck then begin
          ProtocolStatus := psBlockCheckError;
          Inc(FTotalErrors);
          FHeaderState := hsNone;
        end else begin
          {Say we got a good header}
          CollectBinaryHeader := True;
        end;
      end;
    bhCrc4 :
      {Check the Crc value}
      if not VerifyBlockCheck then begin
        ProtocolStatus := psBlockCheckError;
        Inc(FTotalErrors);
        FHeaderState := hsNone;
      end else begin
        {Say we got a good header}
        CollectBinaryHeader := True;
      end;
  end;

  {Go to next state}
  if FBinHdrState <> bhCrc4 then
    Inc(FBinHdrState)
  else
    FBinHdrState := bhFrame;
end;

procedure TApxZModemDriver.CheckForHeader;
{-Samples input stream for start of header}
var
  C : Char;
begin
  {Assume no header ready}
  ProtocolStatus := psNoHeader;

  {Process potential header characters}
  while ComPort.CharReady do begin

    {Only get the next char if we don't know the header type yet}
    case FHeaderState of
      hsNone, hsGotZPad, hsGotZDle :
        if not GetCharStripped(C) then
          Exit;
    end;

    {Try to accumulate the start of a header}
    ProtocolStatus := psNoHeader;
    case FHeaderState of
      hsNone :
        if C = ZPad then
          FHeaderState := hsGotZPad;
      hsGotZPad :
        case C of
          ZPad : ;
          ZDle :
            FHeaderState := hsGotZDle;
          else
            FHeaderState := hsNone;
        end;
      hsGotZDle :
        case C of
          ZBin   :
            begin
              FWasHex := False;
              FHeaderState := hsGotZBin;
              FBinHdrState := bhFrame;
              FEscapePending := False;
              {if zpCollectBinaryHeader(P, False) then}
              {  zHeaderState := hsGotHeader;         }
            end;
          ZBin32 :
            begin
              FWasHex := False;
              FHeaderState := hsGotZBin32;
              FBinHdrState := bhFrame;
              FEscapePending := False;
              {if zpCollectBinaryHeader(P, True) then}
              {  zHeaderState := hsGotHeader;        }
            end;
          ZHex   :
            begin
              FWasHex := True;
              FHeaderState := hsGotZHex;
              FHexHdrState := hhFrame;
              FHexPending := False;
              {if zpCollectHexHeader(P) then}
            end;
          else
            FHeaderState := hsNone;
        end;
      hsGotZBin :
        if CollectBinaryHeader(False) then
          FHeaderState := hsGotHeader;
      hsGotZBin32 :
        if CollectBinaryHeader(True) then
          FHeaderState := hsGotHeader;
      hsGotZHex :
        if CollectHexHeader then
          FHeaderState := hsGotHeader;
    end;

    if (FHeaderState = hsGotHeader) and (FRcvFrame = ZEof) and
       (FLastFrame = ZrPos) then
      FHeaderState := hsNone;

    {If we just got a header, note file pos and frame type}
    if FHeaderState = hsGotHeader then begin
      ProtocolStatus := psGotHeader;
      case FLastFrame of
        ZrPos, ZAck, ZData, ZEof :
          {Header contained a reported file position}
          FLastFileOfs := LongInt(FRcvHeader);
      end;

      {Note frame type for status}
      FLastFrame := FRcvFrame;

      {...and leave}
      Exit;
    end;

    {Also leave if we got any errors or we got a cancel request}
    if (ProtocolError <> ecOK) or
       (ProtocolStatus = psCancelRequested) then
      Exit;
  end;
end;

function TApxZModemDriver.BlockError (OkState, ErrorState : TApxZmodemState;
                        MaxErrors : Cardinal) : Boolean;
{-Handle routine block/timeout errors, return True if error}
begin
  Inc(FBlockErrors);
  Inc(FTotalErrors);
  if BlockErrors > MaxErrors then begin
    BlockError := True;
    Cancel;
    apProtocolError(ecTooManyErrors);
    FZmodemState := ErrorState;
  end else begin
    BlockError := False;
    FZmodemState := OkState;
  end;
end;

function TApxZModemDriver.ReceiveBlock (var Block : TApxDataBlock) : Boolean;
{-Get a binary data subpacket, return True when block complete (or error)}
var
  C : Char;
begin
  {Assume the block isn't ready}
  ReceiveBlock := False;

  while ComPort.CharReady do begin
    {Handle first pass}
    if (FDataBlockLen = 0) and (FRcvBlockState = rbData) then
      BlockCheck := CheckInit[FUseCrc32];

    {Get the waiting character}
    ProtocolStatus := psOK;
    GetCharEscaped(C);
    if FEscapePending or (ProtocolStatus = psCancelRequested) then
      Exit;
    if FControlCharSkip then
      Exit;

    {Always update the block check}
    UpdateBlockCheck(Ord(C));

    case FRcvBlockState of
      rbData :
        case ProtocolStatus of
          psOK :     {Normal character}
            begin
              {Check for a long block}
              Inc(FDataBlockLen);
              if FDataBlockLen > BlockLen then begin
                ProtocolStatus := psLongPacket;
                Inc(FTotalErrors);
                Inc(FBlockErrors);
                ReceiveBlock := True;
                Exit;
              end;

              {Store the character}
              Block[FDataBlockLen] := C;
            end;

          psGotCrcE,
          psGotCrcG,
          psGotCrcQ,
          psGotCrcW : {End of DataSubpacket - get/check CRC}
            begin
              FRcvBlockState := rbCrc;
              FCrcCnt := 0;
              SaveStatus := ProtocolStatus;
            end;
        end;

      rbCrc :
        begin
          Inc(FCrcCnt);
          if (FUseCrc32 and (FCrcCnt = 4)) or
             (not FUseCrc32 and (FCrcCnt = 2)) then begin
            if not VerifyBlockCheck then begin
              Inc(FBlockErrors);
              Inc(FTotalErrors);
              ProtocolStatus := psBlockCheckError;
            end else
              {Show proper status}
              ProtocolStatus := SaveStatus;

            {Say block is ready for processing}
            ReceiveBlock := True;
            Exit;
          end;
        end;
    end;
  end;
end;


procedure TApxZModemDriver.ExtractFileInfo;
{-Extracts file information into fields}
var
  BlockPos  : Cardinal;
  I         : Integer;
  Code      : Integer;
  S         : string;
  SLen      : Byte;
  S1        : ShortString;
  S1Len     : Byte absolute S1;
  Name      : ShortString;
  NameExt   : array[0..255] of Char;
begin
  {Extract the file name from the data block}
  BlockPos := 1;
  SetLength(S, 1024);
  while (DataBlock^[BlockPos] <> #0) and (BlockPos < 255) do begin
    S[BlockPos] := DataBlock^[BlockPos];
    if S[BlockPos] = '/' then
      S[BlockPos] := '\';
    Inc(BlockPos);
  end;
  SLen := BlockPos - 1;
  SetLength(S, SLen);
  if (SLen > 0) and (UpcaseFileNames) then begin
    S := AnsiUpperCase(S);
  end;

  {Set Pathname}
  if Length(S) > 255 then
    SetLength(S, 255);
  StrPCopy(Pathname, S);

  {Should we use its directory or ours?}
  if not FlagIsSet(Flags, apHonorDirectory) then begin
    Name := ExtractFileName(S);
    StrPCopy(NameExt, Name);
    ApxIncludeTrailingPathDelimiterZ (PathName, DestDir);
    StrLCat(PathName, NameExt, SizeOf(PathName));
  end;

  {Extract the file size}
  I := 1;
  Inc(BlockPos);
  while (DataBlock^[BlockPos] <> #0) and
        (DataBlock^[BlockPos] <> ' ') and
        (I <= 255) do begin
    S1[I] := DataBlock^[BlockPos];
    Inc(I); Inc(BlockPos);
  end;
  Dec(I);
  S1Len := I;
  if S1Len = 0 then
    SrcFileLen := 0
  else begin
    Val(S1, FSrcFileLen, Code);
    if Code <> 0 then
      {Invalid date format, just ignore}
      SrcFileLen := 0;
  end;
  BytesRemaining := SrcFileLen;
  BytesTransferred := 0;

  {Extract the file date/time stamp}
  I := 1;
  Inc(BlockPos);
  while (DataBlock^[BlockPos] <> #0) and
        (DataBlock^[BlockPos] <> ' ') and
        (I <= 255) do begin
    S1[I] := DataBlock^[BlockPos];
    Inc(I);
    Inc(BlockPos);
  end;
  Dec(I);
  S1Len := I;
  S1 := apTrimZeros(S1);
  if S1 = '' then
    SrcFileDate := apYMTimeStampToPack(apCurrentTimeStamp)
  else
    SrcFileDate := apYMTimeStampToPack(apOctalStr2Long(S1));
end;

procedure TApxZModemDriver.WriteDataBlock;
{-Call WriteProtocolBlock for the last received DataBlock}
var
  Failed : Boolean;
begin
  {Write this block}
  Failed := WriteProtocolBlock(DataBlock^, FDataBlockLen);

  {Process result}
  if Failed then
    Cancel
  else begin
    Inc(FFileOfs, FDataBlockLen);
    Dec(FBytesRemaining, FDataBlockLen);
    Inc(FBytesTransferred, FDataBlockLen);
  end;
end;

procedure TApxZModemDriver.PrepareReceive;
{-Prepare to receive Zmodem parts}
begin
  {Init the status stuff}
  apResetStatus;
  apShowFirstStatus;
  NewTimer (FStatusTimer, StatusInterval);
  TimerStarted := False;

  {Flush input buffer}
  ComPort.FlushInBuffer;

  {Init state variables}
  FHeaderType := ZrInit;
  FZmodemState := rzRqstFile;
  FHeaderState := hsNone;
  ProtocolError := ecOK;
end;

procedure TApxZModemDriver.Receive(Msg, wParam : Cardinal;
                     lParam : LongInt);
{-Performs one increment of a Zmodem receive}
label
  ExitPoint;
var
  TriggerID    : Cardinal absolute wParam;
  Finished     : Boolean;
  C            : Char;
  StatusTimeMS : Cardinal; 
begin
  {Get the protocol pointer from data pointer 1}
  EnterCriticalSection (FProtSection);

    {Exit if protocol was cancelled while waiting for crit section}
    if FZmodemState = rzDone then begin
        LeaveCriticalSection(FProtSection);
      Exit;
    end;

    {Set Trigger_ID directly for TriggerAvail messages}
    if Msg = apx_TriggerAvail then
      TriggerID := aDataTrigger;

    repeat

      ComPort.DebugLog.AddDebugEntry (TApxCustomProtocol,
                                      Cardinal (AxdtZModem),
                                      Cardinal (LogZModemState[FZmodemState]),
                                      0);

      {Check for user abort}
      if ProtocolStatus <> psCancelRequested then begin
        if (Integer(TriggerID) = NoCarrierTrigger) then begin
          FZmodemState := rzError;
          ProtocolStatus := psAbortNoCarrier;
        end;
        if Msg = apx_ProtocolCancel then begin
          Cancel;
          FZmodemState := rzError;
        end;
      end;

      {Show status at requested intervals and after significant events}
      if ForceStatus or (Integer(TriggerID) = StatusTrigger) then begin
        if TimerStarted then
          ElapsedXfrTime := ElapsedTime(Timer);
        if FComport.Dispatcher.TimerTimeRemaining (StatusTrigger, StatusTimeMS) <> 0 then
          StatusTimeMS := 0;
        if LongInt (StatusTimeMS) <= 0 then begin
          ShowStatus(0);
          FComport.Dispatcher.SetTimerTrigger(StatusTrigger, StatusInterval, True);
          ForceStatus := False;
        end;
        if Integer(TriggerID) = StatusTrigger then begin
            LeaveCriticalSection (FProtSection);  
          Exit;
        end;
      end;

      {Preprocess header requirements}
      case FZmodemState of
        rzWaitFile,
        rzStartData,
        rzWaitEof :
          if TriggerID = aDataTrigger then begin
            {Header might be present, try to get one}
            CheckForHeader;
            if ProtocolStatus = psCancelRequested then
              FZmodemState := rzError;
          end else if Integer(TriggerID) = TimeoutTrigger then
            {Timed out waiting for something, let state machine handle it}
            ProtocolStatus := psTimeout
          else
            {Indicate that we don't have a header}
            ProtocolStatus := psNoHeader;
      end;

      {Main state processor}
      case FZmodemState of
        rzRqstFile :
          begin
            FCanCount := 0;

            {Init pos/flag bytes to zero}
            LongInt(FTransHeader) := 0;

            {Set our receive options}
            FTransHeader[ZF0] := CanFdx or     {Full duplex}
                                 CanOvIO or    {Overlap I/O}
                                 CanFc32 or    {Use Crc32 on frames}
                                 CanBrk;       {Can send break}

            {Testing shows that Telix needs a delay here}
            FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, ApxTelixDelay, True);
            FZmodemState := rzDelay;
          end;

        rzDelay :
          begin
            {Send the header}
            PutHexHeader(FHeaderType);

            FZmodemState := rzWaitFile;
            FHeaderState := hsNone;
            FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
          end;

        rzSendBlockPrep :
          if TriggerID = aDataTrigger then begin
            while FComport.Dispatcher.CharReady and (FDiscardCnt < 2) do begin
              FComport.Dispatcher.GetChar(C);
              Inc(FDiscardCnt);
            end;
            if FDiscardCnt = 2 then
              FZmodemState := rzSendBlock;
          end else if Integer(TriggerID) = TimeoutTrigger then begin
            Inc(FBlockErrors);
            Inc(FTotalErrors);
            if TotalErrors < HandshakeRetry then
              FZmodemState := rzRqstFile
            else
              FZmodemState := rzCleanup;
          end;

        rzSendBlock :
          if TriggerID = aDataTrigger then begin
            {Collect the data subpacket}
            if ReceiveBlock(DataBlock^) then
              if (ProtocolStatus = psBlockCheckError) or
                 (ProtocolStatus = psLongPacket) then
                {Error receiving block, go try again}
                FZmodemState := rzRqstFile
              else
                {Got block OK, go process}
                FZmodemState := rzSendInit
            else if ProtocolStatus = psCancelRequested then
              FZmodemState := rzError;
          end else if Integer(TriggerID) = TimeoutTrigger then begin
            {Timed out waiting for block...}
            Inc(FBlockErrors);
            Inc(FTotalErrors);
            if BlockErrors < HandshakeRetry then begin
              PutHexHeader(ZNak);
              FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
              FZmodemState := rzWaitFile;
              FHeaderState := hsNone;
            end else
              FZmodemState := rzCleanup;
          end;

        rzSendInit :
          begin
            {Save attention string}
            Move(DataBlock^, FAttentionStr, MaxAttentionLen);

            {Turn on escaping if transmitter requests it}
            FEscapeAll := (FRcvHeader[ZF0] and EscAll) = EscAll;

            {Needs an acknowledge}
            PutHexHeader(ZAck);
            {Go wait for ZFile packet}
            FZmodemState := rzWaitFile;
            FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
          end;

        rzWaitFile :
          case ProtocolStatus of
            psGotHeader :
              begin
                case FRcvFrame of
                  ZrQInit : {Go send ZrInit again}
                    FZmodemState := rzRqstFile;
                  ZFile : {Beginning of file transfer attempt}
                    begin
                      {Save conversion and transport options}
                      FConvertOpts := FRcvHeader[ZF0];
                      FTransportOpts := FRcvHeader[ZF2];

                      {Save file mgmt options (if not overridden)}
                      if not FFileMgmtOverride then
                        FFileMgmtOpts := FRcvHeader[ZF1];

                      {Set file mgmt default if none specified}
                      if FFileMgmtOpts = 0 then
                        FFileMgmtOpts := zfWriteProtect;

                      {Start collecting the ZFile's data subpacket}
                      FZmodemState := rzCollectFile;
                      BlockErrors := 0;
                      TotalErrors := 0;
                      FDataBlockLen := 0;
                      FRcvBlockState := rbData;
                      FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
                    end;

                  ZSInit :  {Sender's transmission options}
                    begin
                      {Start collecting ZSInit's data subpacket}
                      BlockErrors := 0;
                      FDataBlockLen := 0;
                      FRcvBlockState := rbData;
                      FComport.Dispatcher.SetTimerTrigger (TimeoutTrigger,
                                       HandshakeWait, True);
                      if FWasHex then begin
                        FZmodemState := rzSendBlockPrep;
                        FDiscardCnt := 0;
                      end else
                        FZmodemState := rzSendBlock;
                    end;

                  ZFreeCnt : {Sender is requesting a count of our freespace}
                    begin
                      PutHexHeader(ZAck);
                    end;

                  ZCommand : {Commands not implemented}
                    begin
                      PutHexHeader(ZNak);
                    end;

                  ZCompl,
                  ZFin:      {Finished}
                    begin
                      FZmodemState := rzSendFinish;
                      BlockErrors := 0;
                    end;
                end;
                FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
              end;
            psNoHeader :
              {Keep waiting for a header} ;
            psBlockCheckError,
            psTimeout :
              BlockError(rzRqstFile, rzCleanup, HandshakeRetry);
          end;

        rzCollectFile :
          if TriggerID = aDataTrigger then begin
            {Collect the data subpacket}
            if ReceiveBlock(DataBlock^) then
              if (ProtocolStatus = psBlockCheckError) or
                 (ProtocolStatus = psLongPacket) then
                {Error getting block, go try again}
                FZmodemState := rzRqstFile
              else
                {Got block OK, go extract file info}
                FZmodemState := rzStartFile
            else if ProtocolStatus = psCancelRequested then
              FZmodemState := rzError;
          end else if Integer(TriggerID) = TimeoutTrigger then begin
            {Timeout collecting block}
            Inc(FBlockErrors);
            Inc(FTotalErrors);
            if BlockErrors < HandshakeRetry then begin
              PutHexHeader(ZNak);
              FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
            end else
              FZmodemState := rzCleanup;
          end;

        rzStartFile :
          begin
            {Got the data subpacket to the ZFile, extract the file info}
            ExtractFileInfo;

            {Call user's LogFile function}
            ProtocolStatus := psOK;
            LogFile(lfReceiveStart);

            {Accept this file}
            if not AcceptFile(PathName) then begin
              FHeaderType := ZSkip;
              ProtocolStatus := psSkipFile;
              LogFile(lfReceiveSkip);
              FZmodemState := rzRqstFile;
              ForceStatus := True;
              goto ExitPoint;
            end;

            {Prepare to write this file}
            PrepareWriting;
            if ProtocolError = ecOK then begin
              case ProtocolStatus of
                psCantWriteFile,
                psFileDoesntExist : {Skip this file}
                  begin
                    FHeaderType := ZSkip;
                    ProtocolStatus := psSkipFile;
                    LogFile(lfReceiveSkip);
                    FZmodemState := rzRqstFile;
                    ForceStatus := True;
                    goto ExitPoint;
                  end;
              end;
            end else begin
              Cancel;
              FZmodemState := rzError;
              goto ExitPoint;
            end;

            {Start protocol timer now}
            NewTimer(FTimer, 1);
            TimerStarted := True;

            {Go send the initial ZrPos}
            FZmodemState := rzSync;
            ForceStatus := True;
          end;

        rzSync :
          begin
            {Incoming data will just get discarded so flush inbuf now}
            FComport.FlushInBuffer;  

            {Insert file size into header and send to remote}
            LongInt(FTransHeader) := FileOfs;
            PutHexHeader(ZrPos);

            {Set status info}
            BytesRemaining := SrcFileLen - FileOfs;
            BytesTransferred := FileOfs;

            FZmodemState := rzStartData;
            FHeaderState := hsNone;
            BlockErrors := 0;
            FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
          end;

        rzStartData :
          case ProtocolStatus of
            psGotHeader :
              case FRcvFrame of
                ZData :  {One or more data subpackets follow}
                  begin
                    if FileOfs <> FLastFileOfs then begin
                      Inc(FBlockErrors);
                      Inc(FTotalErrors);
                      if BlockErrors > MaxBadBlocks then begin
                        Cancel;
                        apProtocolError(ecTooManyErrors);
                        FZmodemState := rzError;
                        goto ExitPoint;
                      end;
                      PutAttentionString;
                      FZmodemState := rzSync;
                    end else begin
                      FZmodemState := rzCollectData;
                      FDataBlockLen := 0;
                      FRcvBlockState := rbData;
                      FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
                    end;
                  end;
                ZNak : {Nak received}
                  begin
                    Inc(FTotalErrors);
                    Inc(FBlockErrors);
                    if BlockErrors > MaxBadBlocks then begin
                      Cancel;
                      apProtocolError(ecTooManyErrors);
                      FZmodemState := rzError;
                    end else
                      {Resend ZrPos}
                      FZmodemState := rzSync;
                  end;
                ZFile : {File frame}
                  {Already got a File frame, just go send ZrPos again}
                  FZmodemState := rzSync;
                ZEof : {End of current file}
                  begin
                    ProtocolStatus := psEndFile;
                    FZmodemState := rzEndOfFile;
                  end;
                else begin
                  {Error during GetHeader}
                  Inc(FTotalErrors);
                  Inc(FBlockErrors);
                  if BlockErrors > MaxBadBlocks then begin
                    Cancel;
                    apProtocolError(ecTooManyErrors);
                    FZmodemState := rzError;
                    goto ExitPoint;
                  end;
                  PutAttentionString;
                  FZmodemState := rzSync;
                end;
              end;
            psNoHeader :
              {Just keep waiting for header} ;
            psBlockCheckError,
            psTimeout :
              BlockError(rzSync, rzError, HandshakeRetry);
          end;

        rzCollectData :
          if TriggerID = aDataTrigger then begin
            FComport.Dispatcher.SetTimerTrigger (TimeoutTrigger, HandshakeWait, True);

            {Collect the data subpacket}
            if ReceiveBlock(DataBlock^) then begin
              {Block is okay -- process it}
              case ProtocolStatus of
                psCancelRequested : {Cancel requested}
                  FZmodemState := rzError;
                psGotCrcW : {Send requests a wait}
                  begin
                    {Write this block}
                    WriteDataBlock;
                    if ProtocolError = ecOK then begin
                      {Acknowledge with the current file position}
                      LongInt(FTransHeader) := FileOfs;
                      PutHexHeader(ZAck);
                      FZmodemState := rzStartData;
                      FHeaderState := hsNone;
                    end else begin
                      Cancel;
                      FZmodemState := rzError;
                    end;
                  end;
                psGotCrcQ : {Ack requested}
                  begin
                    {Write this block}
                    WriteDataBlock;
                    if ProtocolError = ecOK then begin
                      LongInt(FTransHeader) := FileOfs;
                      PutHexHeader(ZAck);
                      {Don't change state - will get next data subpacket}
                    end else begin
                      Cancel;
                      FZmodemState := rzError;
                    end;
                  end;
                psGotCrcG : {Normal subpacket - no response necessary}
                  begin
                    {Write this block}
                    WriteDataBlock;
                    if ProtocolError <> ecOK then begin
                      Cancel;
                      FZmodemState := rzError;
                    end;
                  end;
                psGotCrcE : {Last data subpacket}
                  begin
                    {Write this block}
                    WriteDataBlock;
                    if ProtocolError = ecOK then begin
                      FZmodemState := rzWaitEof;
                      FHeaderState := hsNone;
                      BlockErrors := 0;
                    end else begin
                      Cancel;
                      FZmodemState := rzError;
                    end;
                  end;
                else begin
                  {Error in block}
                  if BlockErrors < MaxBadBlocks then begin
                    PutAttentionString;
                    FZmodemState := rzSync;
                  end else begin
                    Cancel;
                    apProtocolError(ecTooManyErrors);
                    FZmodemState := rzError;
                  end;
                  goto ExitPoint;
                end;
              end;

              {Prepare to collect next block}
              ForceStatus := True;
              FDataBlockLen := 0;
              FRcvBlockState := rbData;
            end else if ProtocolStatus = psCancelRequested then
              FZmodemState := rzError

          end else if Integer(TriggerID) = TimeoutTrigger then begin
            {Timeout collecting datasubpacket}
            Inc(FBlockErrors);
            Inc(FTotalErrors);
            if BlockErrors < MaxBadBlocks then begin
              PutAttentionString;
              FZmodemState := rzSync;
            end else begin
              Cancel;
              FZmodemState := rzError;
            end;
          end;

        rzWaitEof :
          case ProtocolStatus of
            psGotHeader :
              case FRcvFrame of
                ZEof : {End of current file}
                  begin
                    ProtocolStatus := psEndFile;
                    ShowStatus(0);
                    FinishWriting;
                    if ProtocolError = ecOK then
                      LogFile(lfReceiveOk)
                    else
                      LogFile(lfReceiveFail);

                    {Go get the next file}
                    FZmodemState := rzRqstFile;
                  end;
                else begin
                  {Error during GetHeader}
                  Inc(FTotalErrors);
                  Inc(FBlockErrors);
                  if BlockErrors > MaxBadBlocks then begin
                    Cancel;
                    apProtocolError(ecTooManyErrors);
                    FZmodemState := rzError;
                    goto ExitPoint;
                  end;
                  PutAttentionString;
                  FZmodemState := rzSync;
                end;
              end;
            psNoHeader :
              {Just keep collecting rest of header} ;
            psBlockCheckError,
            psTimeout :
              BlockError(rzSync, rzError, HandshakeRetry);
          end;

        rzEndOfFile :
          if FileOfs = FLastFileOfs then begin
            FinishWriting;

            {Send Proper status to user logging routine}
            if ProtocolError = ecOK then
              LogFile(lfReceiveOk)
            else
              LogFile(lfReceiveFail);
            FZmodemState := rzRqstFile;
          end else
            FZmodemState := rzSync;

        rzSendFinish :
          begin
            {Insert file position into header}
            LongInt(FTransHeader) := FileOfs;
            PutHexHeader(ZFin);
            FZmodemState := rzCollectFinish;
            FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, FinishWait, True);
            FOCnt := 0;
          end;

        rzCollectFinish :
          if TriggerID = aDataTrigger then begin
            FComport.Dispatcher.GetChar(C);
            if C = 'O' then begin
              Inc(FOCnt);
              if FOCnt = 2 then
                FZmodemState := rzCleanup;
            end;
          end else if Integer(TriggerID) = TimeoutTrigger then begin
            {Retry 3 times only (same as DSZ)}
            Inc(FBlockErrors);
            Inc(FTotalErrors);
            if BlockErrors < FFinishRetry then
              {Go send ZFin again}
              FZmodemState := rzSendFinish
            else
              {Cleanup anyway}
              FZmodemState := rzCleanup;
          end;

        rzError :
          begin
            if FileOpen then begin
              FinishWriting;
              LogFile(lfReceiveFail);
            end;

            {Wait for cancel to go out}
            if FComport.OutBuffUsed > 0 then begin
              FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, TransTimeout, True);
              FComport.Dispatcher.SetStatusTrigger(OutBuffUsedTrigger, 0, True);
              FZmodemState := rzWaitCancel;
            end else
              FZmodemState := rzCleanup;
          end;

        rzWaitCancel :
          {Cancel went out or we timed out, doesn't matter which}
          FZmodemState := rzCleanup;

        rzCleanup :
          begin
            apShowLastStatus;
            FComport.FlushInBuffer;  
            FZmodemState := rzDone;
            apSignalFinish (False);
          end;
      end;

ExitPoint:
      {Stay in state machine or leave?}
      case FZmodemState of
        {Stay in state machine for these states}
        rzRqstFile,
        rzSendInit,
        rzSync,
        rzStartFile,
        rzGotData,
        rzEndOfFile,
        rzSendFinish,
        rzError,
        rzCleanup:
          Finished := False;

        {Stay in state machine only if more data ready}
        rzSendBlockPrep,
        rzSendBlock,
        rzWaitFile,
        rzWaitEof,
        rzCollectData,
        rzStartData,
        rzCollectFinish,
        rzCollectFile:
          Finished := not FComport.Dispatcher.CharReady;

        {Exit state machine on these states (waiting for trigger hit)}
        rzDelay,
        rzWaitCancel,
        rzDone:
          Finished := True;
          
        else
          Finished := True;
      end;

      {Clear header state if we just processed a header}
      if (ProtocolStatus = psGotHeader) or
         (ProtocolStatus = psNoHeader) then
        ProtocolStatus := psOK;
      if FHeaderState = hsGotHeader then
        FHeaderState := hsNone;

      {If staying in state machine force data ready}
      TriggerID := aDataTrigger;
    until Finished;

  LeaveCriticalSection(FProtSection);  
end;

procedure TApxZModemDriver.PutBinaryHeader(FrameType : Char);
{-Sends a binary header (Crc16 or Crc32)}
var
  I : Integer;
begin
  FUseCrc32 := FCanCrc32;

  {Send '*'<DLE>}
  ComPort.PutChar(ZPad);
  ComPort.PutChar(ZDle);

  {Send frame identifier}
  if FUseCrc32 then begin
    ComPort.PutChar(ZBin32);
    BlockCheck := $FFFFFFFF;
  end else begin
    ComPort.PutChar(ZBin);
    BlockCheck := 0;
  end;

  {Send frame type}
  PutCharEscaped(FrameType);
  UpdateBlockCheck(Ord(FrameType));

  {Put the position/flags data bytes}
  for I := 0 to 3 do begin
    PutCharEscaped(Char(FTransHeader[I]));
    UpdateBlockCheck(Ord(FTransHeader[I]));
  end;

  {Put the Crc bytes}
  SendBlockCheck;

  {Note frame type for status}
  FLastFrame := FrameType;
end;

function TApxZModemDriver.EscapeChar(C : Char) : Boolean;
{-Return True if C needs to be escaped}
var
  C1 : Char;
  C2 : Char;
begin
  {Check for chars to escape}
  if FEscapeAll and ((Byte(C) and $60) = 0) then
    {Definitely needs escaping}
    EscapeChar := True
  else if Ord(C) and $11 = 0 then
    {Definitely does not need escaping}
    EscapeChar := False
  else begin
    {Might need escaping}
    case C of
      cXon, cXoff, cDle,        {Escaped control chars}
      cXonHi, cXoffHi, cDleHi,  {Escaped hibit control chars}
      ZDle :                    {Escape the escape char}
        EscapeChar := True;
      else begin
        C1 := Char(Byte(C) and $7F);
        C2 := Char(Byte(FLastChar) and $7F);
        EscapeChar := ((C1 = cCR) and (C2 = #$40));
      end;
    end;
  end;
end;

procedure TApxZModemDriver.EscapeBlock (var Block : TApxDataBlock;
                                            BLen : Cardinal);
{-Escape data from Block into zWorkBlock}
var
  I : Cardinal;
  C : Char;
begin
  {Initialize aBlockCheck}
  if FCanCrc32 then begin
    FUseCrc32 := True;
    BlockCheck := $FFFFFFFF;
  end else begin
    FUseCrc32 := False;
    BlockCheck := 0;
  end;

  {Escape the data into zWorkBlock}
  FWorkSize := 1;
  for I := 1 to BLen do begin
    {Escape the entire block}
    C := Block[I];
    UpdateBlockCheck(Ord(C));
    if EscapeChar(C) then begin
      {This character needs escaping, stuff a ZDle and escape it}
      FWorkBlock^[FWorkSize] := zDle;
      Inc(FWorkSize);
      C := Char(Ord(C) xor $40);
    end;

    {Stuff the character}
    FWorkBlock^[FWorkSize] := C;
    Inc(FWorkSize);
    FLastChar := C;
  end;
  Dec(FWorkSize);
end;

procedure TApxZModemDriver.TransmitBlock;
{-Transmits one data subpacket from Block}
begin
  if FWorkSize <> 0 then
    ComPort.PutBlock(FWorkBlock^, FWorkSize);
  {Send the frame type}
  UpdateBlockCheck(Byte(FTerminator));
  ComPort.PutChar(ZDle);
  ComPort.PutChar(FTerminator);

  {Send the block check characters}
  SendBlockCheck;

  {Follow CrcW subpackets with an Xon}
  if FTerminator = ZCrcW then
    ComPort.PutChar(cXon);

  {Update status vars}
  LastBlockSize := FDataBlockLen;
  Inc(FFileOfs, FDataBlockLen);
  Inc(FBytesTransferred, FDataBlockLen);
  Dec(FBytesRemaining, FDataBlockLen);
  ForceStatus := True;
end;

procedure TApxZModemDriver.ExtractReceiverInfo;
{-Extract receiver info from last ZrInit header}
const
  Checks : array[Boolean] of Cardinal = (bcCrc16, bcCrc32);
begin
  {Extract info from received ZrInit}
  FRcvBuffLen := FRcvHeader[ZP0] + ((FRcvHeader[ZP1]) shl 8);
  FCanCrc32 := (FRcvHeader[ZF0] and CanFC32) = CanFC32;
  CheckType := Checks[FCanCrc32];
  FEscapeAll := (FRcvHeader[ZF0] and EscAll) = EscAll;
end;

procedure TApxZModemDriver.InsertFileInfo;
{-Build a ZFile data subpacket}
var
    I    : Cardinal;
    Name : string[fsName];
    CA   : TCharArray;
    S    : string[fsPathname];
    Len  : Byte;
begin
  {Make a file header record}
  FillChar(DataBlock^, SizeOf(TApxDataBlock) , 0);

  {Fill in the file name}
  S := StrPas(PathName);
  Name := ExtractFileName(S);
  if FlagIsSet(Flags, apIncludeDirectory) then
    StrPCopy(CA, S)
  else
    StrPCopy(CA, Name);

  {Change name to lower case, change '\' to '/'}
  Len := StrLen(CA);
  AxCharLowerBuff(CA, Len);
  for I := 0 to Len-1 do begin
    if CA[I] = '\' then
      CA[I] := '/';
  end;
  Move(CA[0], DataBlock^, Len);

  {Fill in file size}
  Str(SrcFileLen, S);
  Move(S[1], DataBlock^[Len+2], Length(S));
  Inc(Len, Length(S)+1);

  {Convert time stamp to Ymodem format and stuff in aDataBlock}
  if SrcFileDate <> 0 then begin
    S := ' ' + apOctalStr(apPackToYMTimeStamp(SrcFileDate));
    Move(S[1], DataBlock^[Len+1], Length(S));
    Inc(Len, Length(S)+1);
  end;

  {Save the length of the file info string for the ZFile header}
  FDataBlockLen := Len;

  {Take care of status information}
  BytesRemaining := SrcFileLen;
  BytesTransferred := 0;
  ForceStatus := True;
end;

procedure TApxZModemDriver.PrepareTransmit;
{-Transmit all files that fit the Mask}
const
  MinSize : array[Boolean] of Cardinal = (2048+30, 16384+30);
var
  {InSize, }OutSize : Cardinal;
begin
  {Check buffer sizes (again)}
  OutSize := ComPort.OutSize;
  if OutSize < MinSize[FlagIsSet(Flags, apZmodem8K)] then begin
    ProtocolError := ecOutputBufferTooSmall;
    Exit;
  end;

  {Reset status vars}
  apResetStatus;
  apShowFirstStatus;
  NewTimer(FStatusTimer, StatusInterval);
  TimerStarted := False;

  {State machine inits}
  FHeaderState := hsNone;
  ForceStatus := False;
  FZmodemState := tzInitial;
  ProtocolError := ecOK;
end;

procedure TApxZModemDriver.GotZrPos;
{-Got an unsolicited ZRPOS, must be due to bad block}
begin
  Inc(FBlockErrors);
  Inc(FTotalErrors);
  FileOfs := LongInt(FRcvHeader);
  if FileOfs > SrcFileLen then
    FileOfs := SrcFileLen;
  BytesTransferred := FileOfs;
  BytesRemaining := SrcFileLen - BytesTransferred;
  if BlockLen > 256 then
    BlockLen := BlockLen shr 1;
  LastBlockSize := BlockLen;
  FTookHit := True;
  FGoodAfterBad := 0;
  ComPort.FlushOutBuffer;
  FZmodemState := tzStartData;
end;

procedure TApxZModemDriver.ProcessHeader;
{-Process a header}
begin
  case ProtocolStatus of
    psGotHeader :
      case FRcvFrame of
        ZCan, ZAbort : {Receiver says quit}
          begin
            ProtocolStatus := psCancelRequested;
            ForceStatus := True;
            FZmodemState := tzError;
          end;
        ZAck :
          FZmodemState := tzStartData;
        ZrPos :        {Receiver is sending its desired file position}
          GotZrPos;
        else begin
          {Garbage, send Nak}
          PutBinaryHeader(ZNak);
        end;
      end;
    psBlockCheckError,
    psTimeout :
      BlockError(tzStartData, tzError, MaxBadBlocks);
  end;
end;

procedure TApxZModemDriver.Transmit(Msg, wParam : Cardinal;
                      lParam : LongInt);
{-Performs one increment of a Zmodem transmit}
label
  ExitPoint;
const
  RZcommand : array[0..3] of Char = 'rz'+cCr;
  FreeMargin = 60;
var
  TriggerID     : Cardinal absolute wParam;
  NewInterval   : Cardinal;
  Finished      : Boolean;
  Crc32         : LongInt;
  Secs          : LongInt;
  StatusTimeMS  : Cardinal;
  StartTime     : DWORD;

    { BCB compiler bug workaround }
  procedure CheckFinished;
  begin
    Finished := not FComport.Dispatcher.CharReady;
  end;

begin
  StartTime := AxTimeGetTime;

  EnterCriticalSection(FProtSection);

  {Exit if protocol was cancelled while waiting for crit section}
  if FZmodemState = tzDone then begin
    LeaveCriticalSection(FProtSection);
    Exit;
  end;

  {Force TriggerID for TriggerAvail messages}
  if Msg = apx_TriggerAvail then
    TriggerID := aDataTrigger;

  repeat

    ComPort.DebugLog.AddDebugEntry (TApxCustomProtocol,
                                    Cardinal (AxdtZModem),
                                    Cardinal (LogZModemState[FZmodemState]),
                                    0);

    {Check for user abort (but not twice)}
    if ProtocolStatus <> psCancelRequested then begin
      if (Integer(TriggerID) = NoCarrierTrigger) then begin
        FZmodemState := tzError;
        ProtocolStatus := psAbortNoCarrier;
      end;
      if Msg = apx_ProtocolCancel then begin
        Cancel;
        FZmodemState := tzError;
      end;
    end;

    {Show status at requested intervals and after significant events}
    if ForceStatus or (Integer(TriggerID) = StatusTrigger) then begin
      if TimerStarted then
        ElapsedXfrTime := ElapsedTime(Timer);

      {Use user-specified status interval unless draining eof}
      if FZmodemState = tzDrainEof then
        NewInterval := DrainingStatusInterval
      else
        NewInterval := StatusInterval;

      if FComport.Dispatcher.TimerTimeRemaining(StatusTrigger,
                                  StatusTimeMS) <> 0 then
        StatusTimeMS := 0;
      if LongInt (StatusTimeMS) <= 0 then begin
        ShowStatus(0);
        FComport.Dispatcher.SetTimerTrigger(StatusTrigger, NewInterval, True);
        ForceStatus := False;
      end;

      if Integer(TriggerID) = StatusTrigger then begin
            LeaveCriticalSection(FProtSection);   
        Exit;
      end;
    end;

    {Preprocess header requirements}
    case FZmodemState of
      tzHandshake,
      tzCheckFile,
      tzCheckEOF,
      tzDrainEof,
      tzCheckFinish,
      tzSendData,
      tzWaitAck :
        if TriggerID = aDataTrigger then begin
          {Header might be present, try to get one}
          CheckForHeader;
          if ProtocolStatus = psCancelRequested then
            FZmodemState := tzError;
        end else if Integer(TriggerID) = TimeoutTrigger then
          {Timeout, let state machine handle it}
          ProtocolStatus := psTimeout
        else
          {Indicate no header yet}
          ProtocolStatus := psNoHeader;
    end;

    {Process the current state}
    case FZmodemState of
      tzInitial :
        begin
          FCanCount := 0;

          {Send RZ command (via the attention string)}
          Move(RZcommand, FAttentionStr, SizeOf(RZcommand));
          PutAttentionString;
          FillChar(FAttentionStr, SizeOf(FAttentionStr), 0);

          {Send ZrQinit header (requests receiver's ZrInit)}
          LongInt(FTransHeader) := FZRQINITValue;
          PutHexHeader(ZrQInit);
          BlockErrors := 0;
          TotalErrors := 0;
          FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
          FZmodemState := tzHandshake;
          FHeaderState := hsNone;
        end;

      tzHandshake :
        case ProtocolStatus of
          psGotHeader :
            case FRcvFrame of
              ZrInit :     {Got ZrInit, extract info}
                begin
                  ExtractReceiverInfo;
                  FZmodemState := tzGetFile;
                end;
              ZChallenge : {Receiver is challenging, respond with same number}
                begin
                  FTransHeader := FRcvHeader;
                  PutHexHeader(ZAck);
                end;
              ZCommand :   {Commands not supported}
                PutHexHeader(ZNak);
              ZrQInit :    {Remote is trying to transmit also, do nothing}
                ;
              else         {Unexpected reply, nak it}
                PutHexHeader(ZNak);
            end;
          psNoHeader :
                {Keep waiting for header} ;
          psBlockCheckError,
          psTimeout  : {Send another ZrQinit}
            if not BlockError (tzHandshake,
                               tzError, HandshakeRetry) then begin
              PutHexHeader (ZrQInit);
              FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
            end;
        end;

      tzGetFile :
        begin
          {Get the next file to send}
          if not NextFile(FPathname) then begin
            FZmodemState := tzSendFinish;
            goto ExitPoint;
          end;
          FilesSent := True;

          {Let all hooks see an upper case pathname}
          if UpcaseFileNames then
            AnsiStrUpper(PathName);

          {Zero out status fields in case status msg is waiting}
          BytesTransferred := 0;
          BytesRemaining := 0;

          {Show file name to user logging routine}
          LogFile(lfTransmitStart);

          {Prepare to read file blocks}
          PrepareReading;
          if ProtocolError <> ecOK then begin
            Cancel;
            FZmodemState := tzError;
            goto ExitPoint;
          end;

          {Start protocol timer now}
          NewTimer(FTimer, 1);
          TimerStarted := True;

          {Build the header data area}
          LongInt(FTransHeader) := 0;
          FTransHeader[ZF1] := FFileMgmtOpts;
          if FReceiverRecover then
            FTransHeader[ZF0] := FileRecover;

          {Insert file information into header}
          InsertFileInfo;
          ForceStatus := True;
          FZmodemState := tzSendFile;
        end;

      tzSendFile :
        begin
          {Send the ZFile header and data subpacket with file info}
          PutBinaryHeader(ZFile);
          FTerminator := ZCrcW;
          EscapeBlock(DataBlock^, FDataBlockLen);
          TransmitBlock;

          {Clear status vars that zpTransmitBlock changed}
          BytesTransferred := 0;
          BytesRemaining := 0;

          {Go wait for response}
          BlockErrors := 0;
          FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
          FZmodemState := tzCheckFile;
          FHeaderState := hsNone;
        end;

      tzCheckFile :
        case ProtocolStatus of
          psGotHeader :
            case FRcvFrame of
              ZrInit : {Got an extra ZrInit, ignore it}
                ;
              ZCrc :   {Receiver is asking for Crc32 of the file, send it}
                begin
                  Crc32 := apCrc32OfFile(PathName, 0);
                  if ProtocolError = ecOK then begin
                    LongInt(FTransHeader) := Crc32;
                    PutHexHeader(ZCrc);
                  end else
                    FZmodemState := tzError;
                end;
              ZSkip :  {Receiver wants to skip this file}
                begin
                  ProtocolStatus := psSkipFile;
                  ShowStatus(0);

                  {Close file and log skip}
                  FinishReading;
                  LogFile(lfTransmitSkip);

                  {Go look for another file}
                  FZmodemState := tzGetFile;
                end;
              ZrPos :  {Receiver tells us where to seek in our file}
                begin
                  {Get file offset}
                  FileOfs := LongInt(FRcvHeader);
                  BytesTransferred := FileOfs;
                  InitFilePos := FileOfs;
                  BytesRemaining := SrcFileLen - BytesTransferred;

                  {Go send the data subpackets}
                  FZmodemState := tzStartData;
                end;
            end;
          psNoHeader : {Keep waiting for header}
                ;
          psBlockCheckError,
          psTimeout :  {Timeout waiting for response to ZFile}
            if not BlockError(tzCheckFile,
                              tzError, HandshakeRetry) then begin
              {Resend ZFile}
              PutBinaryHeader(ZFile);

              TransmitBlock;
              FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, HandshakeWait, True);
            end;
        end;

      tzStartData :
        begin
          {Drain trailing chars from inbuffer...}
          FComport.FlushInBuffer;

          {...and kill whatever might still be in the output buffer}
          FComport.FlushOutBuffer; 

          {Get ready}
          FDataInTransit := 0;
          BlockErrors := 0;

          {Send ZData header}
          LongInt(FTransHeader) := FileOfs;
          PutBinaryHeader(ZData);

          FZmodemState := tzEscapeData;
        end;

      tzEscapeData :
        begin
          {Get a block to send}
          if FTookHit then begin
            Inc(FGoodAfterBad);
            if FGoodAfterBad > 4 then begin
              FTookHit := False;
              if BlockLen < ZMaxBlock[FUse8KBlocks] then
                BlockLen := ZMaxBlock[FUse8KBlocks];
              LastBlockSize := BlockLen;
            end;
          end;
          FDataBlockLen := BlockLen;
          LastBlock := ReadProtocolBlock(DataBlock^, FDataBlockLen);
          if ProtocolError <> ecOK then begin
            Cancel;
            FZmodemState := tzError;
            goto ExitPoint;
          end;

          {Show the new data on the way}
          if FRcvBuffLen <> 0 then
            Inc(FDataInTransit, FDataBlockLen);

          {Set the terminator}
          if LastBlock then
            {Tell receiver its the last subpacket}
            FTerminator := ZCrcE
          else if (FRcvBuffLen <> 0) and
                  (FDataInTransit >= Integer(FRcvBuffLen)) then begin
            {Receiver's buffer is nearly full, wait for acknowledge}
            FTerminator := ZCrcW;
          end else
            {Normal data subpacket, no special action}
            FTerminator := ZCrcG;

          {Escape this data into zWorkBlock}
          EscapeBlock(DataBlock^, FDataBlockLen);

          FZmodemState := tzSendData;
          FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, TransTimeout, True);
          FComport.Dispatcher.SetStatusTrigger(OutBuffFreeTrigger, FWorkSize+FreeMargin, True);
          BlockErrors := 0;
        end;

      tzSendData :
        if (Integer(TriggerID) = OutBuffFreeTrigger) then begin 
          TransmitBlock;
          if LastBlock then begin
            FZmodemState := tzSendEof;
            BlockErrors := 0;
          end else if FTerminator = ZCrcW then begin
            FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, TransTimeout, True);
            FZmodemState := tzWaitAck;
          end else
            FZmodemState := tzEscapeData;
          ForceStatus := True;
        end else if TriggerID = aDataTrigger then
          {Got a header from the receiver, process it}
          if ProtocolStatus = psGotHeader then
            ProcessHeader;

      tzWaitAck :
        if TriggerID = aDataTrigger then
          ProcessHeader;

      tzSendEof :
        begin
          {Send the eof}
          LongInt(FTransHeader) := FileOfs;
          PutBinaryHeader(ZEof);
          FZmodemState := tzDrainEof;
          FComport.Dispatcher.SetStatusTrigger(OutBuffUsedTrigger, 0, True);
          FComport.Dispatcher.SetTimerTrigger(StatusTrigger,
                               DrainingStatusInterval, True);

          {Calculate how long it will take to drain the buffer}
          if ActCPS <> 0 then begin
            Secs := (FComPort.OutBuffUsed div ActCPS) * 2;
            if Secs < 10 then
              Secs := 10;
            FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, Secs*1000, True);
          end else
            FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, TransTimeout, True);
        end;

      tzDrainEof :
        if TriggerID = aDataTrigger then begin
          case ProtocolStatus of
            psGotHeader :
              case FRcvFrame of
                ZCan, ZAbort : {Receiver says quit}
                  begin
                    ProtocolStatus := psCancelRequested;
                    ForceStatus := True;
                    FZmodemState := tzError;
                  end;
                ZrPos :        {Receiver is sending its file position}
                  GotZrPos;
                ZAck :         {Response to last CrcW data subpacket}
                  ;
                ZSkip, ZrInit : {Finished with this file}
                  begin
                    {Close file and log success}
                    FinishReading;
                    LogFile(lfTransmitOk);

                    {Go look for another file}
                    FZmodemState := tzGetFile;
                  end;
                else begin
                  {Garbage, send Nak}
                  PutBinaryHeader(ZNak);
                end;
              end;

            psBlockCheckError,
            psTimeout :
               BlockError(tzStartData, tzError, MaxBadBlocks);
          end;

        end else if Integer(TriggerID) = OutBuffUsedTrigger then begin
          FZmodemState := tzCheckEof;
          TriggerID := aDataTrigger;
          FHeaderState := hsNone;
          FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, FinishWait, True);
        end else if Integer(TriggerID) = TimeoutTrigger then begin
          apProtocolError(ecTimeout);
          FZmodemState := tzError;
        end;

      tzCheckEof :
        case ProtocolStatus of
          psGotHeader :
            begin
              case FRcvFrame of
                ZCan, ZAbort : {Receiver says quit}
                  begin
                    ProtocolStatus := psCancelRequested;
                    ForceStatus := True;
                    FZmodemState := tzError;
                  end;
                ZrPos :        {Receiver is sending its desired file position}
                  begin
                    Inc(FBlockErrors);
                    Inc(FTotalErrors);
                    FileOfs := LongInt(FRcvHeader);
                    BytesTransferred := FileOfs;
                    BytesRemaining := SrcFileLen - BytesTransferred;

                    {We got a hit, reduce block size by 1/2}
                    if BlockLen > 256 then
                      BlockLen := BlockLen shr 1;
                    LastBlockSize := BlockLen;
                    FTookHit := True;
                    FGoodAfterBad := 0;
                    FComport.FlushOutBuffer;  
                    FZmodemState := tzStartData;
                  end;
                ZAck :         {Response to last CrcW data subpacket}
                  ;
                ZSkip, ZrInit : {Finished with this file}
                  begin
                    {Close file and log success}
                    FinishReading;
                    LogFile(lfTransmitOk);

                    {Go look for another file}
                    FZmodemState := tzGetFile;
                  end;
                else begin
                  {Garbage, send Nak}
                  PutBinaryHeader(ZNak);
                end;
              end;
            end;
          psNoHeader :
            {Keep waiting for header} ;
          psBlockCheckError,
          psTimeout :
            BlockError(tzSendEof, tzError, MaxBadBlocks);
        end;

      tzSendFinish :
        begin
          LongInt(FTransHeader) := FileOfs;
          PutHexHeader(ZFin);
          FComport.Dispatcher.SetTimerTrigger(TimeoutTrigger, FinishWait, True);
          BlockErrors := 0;
          FZmodemState := tzCheckFinish;
          FHeaderState := hsNone;
        end;

      tzCheckFinish :
        case ProtocolStatus of
          psGotHeader :
            case FRcvFrame of
              ZFin :
                begin
                  FComPort.Dispatcher.PutChar('O');
                  FComPort.Dispatcher.PutChar('O');
                  FZmodemState := tzCleanup;
                end;
              else begin
                ProtocolError := ecOK;
                FZmodemState := tzCleanup;
              end;
            end;
          psNoHeader :
            {Keep waiting for header} ;
          psBlockCheckError,
          psTimeout :
            begin
              {Give up quietly}
              FZmodemState := tzCleanup;
              ProtocolError := ecOK;
            end;
        end;

      tzError :
        begin
          {Cleanup on aborted or canceled protocol}
          if FilesSent then begin
            FinishReading;
            LogFile(lfTransmitFail);
          end;
          FZmodemState := tzCleanup;
          if ProtocolStatus <> psCancelRequested then
            FComport.FlushOutBuffer;
        end;

      tzCleanup:
        begin
          {Flush last few chars from last received header}
          FComport.FlushInBuffer; 
          PutAttentionString;

          apShowLastStatus;
          FZmodemState := tzDone;
          apSignalFinish (True);

          {Restore "no files" error code if we got that error earlier}
          if (ProtocolError = ecOK) and not FilesSent then
            ProtocolError := ecNoMatchingFiles;
        end;
    end;

ExitPoint:
    {Stay in state machine or exit?}
    case FZmodemState of
      {Leave state machine}
      tzHandshake,
      {tzSendData,}
      tzWaitAck,
      tzDrainEof,
      tzDone:
        Finished := True;

      {Stay in state machine if we can send another packet}
      tzSendData:
        begin
          if (FComPort.Dispatcher.OutBuffFree >= FWorkSize+FreeMargin) and
             not FComPort.Dispatcher.CharReady and
             (StartTime = AxTimeGetTime) then begin
            Finished := False;
            TriggerID := OutBuffFreeTrigger;
            FComPort.Dispatcher.SetStatusTrigger (OutBuffFreeTrigger,
                                         FWorkSize + FreeMargin, True);
            Continue;
          end else
            Finished := True;
        end;

      {Stay in state machine only if data available}
      tzCheckFinish,
      tzCheckEof,
      tzCheckFile:
        CheckFinished;

      {Stay in state machine}
      tzInitial,
      tzGetFile,
      tzSendFile,
      tzEscapeData,
      tzStartData,
      tzSendEof,
      tzSendFinish,
      tzError,
      tzCleanup:
        Finished := False;
      else
        Finished := True;
    end;

    {Clear header state if we just processed a header}
    if (ProtocolStatus = psGotHeader) or
       (ProtocolStatus = psNoHeader) then
      ProtocolStatus := psOK;
    if FHeaderState = hsGotHeader then
      FHeaderState := hsNone;

    {If staying in state machine for a check for data}
    TriggerID := aDataTrigger;
  until Finished;

  LeaveCriticalSection(FProtSection);  
end;

procedure TApxZModemDriver.Assign (Source : TPersistent);
begin
  inherited Assign (Source);
  if (Source is TApxZModemDriver) then
    with (Source as TApxZModemDriver) do begin
      Self.FLastFrame       := FLastFrame;
      Self.FTerminator      := FTerminator;
      Self.FHeaderType      := FHeaderType;
      Self.FZmodemState     := FZModemState;
      Self.FHeaderState     := FHeaderState;
      Self.FHexHdrState     := FHexHdrState;
      Self.FBinHdrState     := FBinHdrState;
      Self.FRcvBlockState   := FRcvBlockState;
      Self.FFileMgmtOverride:= FFileMgmtOverride;
      Self.FReceiverRecover := FReceiverRecover;
      Self.FUseCrc32        := FUseCrc32;
      Self.FCanCrc32        := FCanCrc32;
      Self.FHexPending      := FHexPending;
      Self.FEscapePending   := FEscapePending;
      Self.FEscapeAll       := FEscapeAll;
      Self.FControlCharSkip := FControlCharSkip;
      Self.FWasHex          := FWasHex;
      Self.FDiscardCnt      := FDiscardCnt;
      Self.FConvertOpts     := FConvertOpts;
      Self.FFileMgmtOpts    := FFileMgmtOpts;
      Self.FTransportOpts   := FTransportOpts;
      Self.FFinishRetry     := FFinishRetry;
      Self.FWorkSize        := FWorkSize;
      Self.FCanCount        := FCanCount;
      Self.FHexChar         := FHexChar;
      Self.FCrcCnt          := FCrcCnt;
      Self.FOCnt            := FOCnt;
      Self.FLastFileOfs     := FLastFileOfs;
      Self.FUse8KBlocks     := FUse8KBlocks;
      Self.FTookHit         := FTookHit;
      Self.FGoodAfterBad    := FGoodAfterBad;
      Self.FDataBlockLen    := FDataBlockLen;
      Self.FDataInTransit   := FDataInTransit;
      Self.FRcvFrame        := FRcvFrame;
      Self.FRcvHeader       := FRcvHeader;
      Self.FRcvBuffLen      := FRcvBuffLen;
      Self.FLastChar        := FLastChar;
      Self.FZRQINITValue    := FZRQINITValue;
    end;
end;

end.

